// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include "RWStepBasic_RWSiUnitAndSolidAngleUnit.pxx"
#include <StepBasic_DimensionalExponents.hxx>
#include <StepBasic_SiPrefix.hxx>
#include <StepBasic_SiUnitAndSolidAngleUnit.hxx>
#include <StepBasic_SiUnitName.hxx>
#include <StepData_StepReaderData.hxx>
#include <StepData_StepWriter.hxx>

#include "RWStepBasic_RWSiUnitName.pxx"
#include "RWStepBasic_RWSiPrefix.pxx"

RWStepBasic_RWSiUnitAndSolidAngleUnit::RWStepBasic_RWSiUnitAndSolidAngleUnit() = default;

void RWStepBasic_RWSiUnitAndSolidAngleUnit::ReadStep(
  const occ::handle<StepData_StepReaderData>&           data,
  const int                                             num0,
  occ::handle<Interface_Check>&                         ach,
  const occ::handle<StepBasic_SiUnitAndSolidAngleUnit>& ent) const
{
  int num = num0;

  // --- Instance of common supertype NamedUnit ---
  if (!data->CheckNbParams(num, 1, ach, "named_unit"))
    return;

  // --- field : dimensions ---
  // --- This field is redefined ---
  // szv#4:S4163:12Mar99 `bool stat1 =` not needed
  data->CheckDerived(num, 1, "dimensions", ach, false);

  // --- Instance of plex component SiUnit ---
  num = data->NextForComplex(num);
  if (!data->CheckNbParams(num, 2, ach, "si_unit"))
    return;

  // --- field : prefix ---
  StepBasic_SiPrefix aPrefix    = StepBasic_spExa;
  bool               hasAprefix = false;
  if (data->IsParamDefined(num, 1))
  {
    if (data->ParamType(num, 1) == Interface_ParamEnum)
    {
      const char* text = data->ParamCValue(num, 1);
      hasAprefix       = RWStepBasic_RWSiPrefix::ConvertToEnum(text, aPrefix);
      if (!hasAprefix)
      {
        ach->AddFail("Enumeration si_prefix has not an allowed value");
        return;
      }
    }
    else
    {
      ach->AddFail("Parameter #1 (prefix) is not an enumeration");
      return;
    }
  }

  // --- field : name ---
  StepBasic_SiUnitName aName;
  if (data->ParamType(num, 2) == Interface_ParamEnum)
  {
    const char* text = data->ParamCValue(num, 2);
    if (!RWStepBasic_RWSiUnitName::ConvertToEnum(text, aName))
    {
      ach->AddFail("Enumeration si_unit_name has not an allowed value");
      return;
    }
  }
  else
  {
    ach->AddFail("Parameter #2 (name) is not an enumeration");
    return;
  }

  // --- Instance of plex component SolidAngleUnit ---
  num = data->NextForComplex(num);
  if (!data->CheckNbParams(num, 0, ach, "solid_angle_unit"))
    return;

  //--- Initialisation of the red entity ---
  ent->Init(hasAprefix, aPrefix, aName);
}

void RWStepBasic_RWSiUnitAndSolidAngleUnit::WriteStep(
  StepData_StepWriter&                                  SW,
  const occ::handle<StepBasic_SiUnitAndSolidAngleUnit>& ent) const
{
  // --- Instance of common supertype NamedUnit ---
  SW.StartEntity("NAMED_UNIT");

  // --- field : dimensions ---
  // --- redefined field ---
  SW.SendDerived();

  // --- Instance of plex component SiUnit ---
  SW.StartEntity("SI_UNIT");

  // --- field : prefix ---
  bool hasAprefix = ent->HasPrefix();
  if (hasAprefix)
    SW.SendEnum(RWStepBasic_RWSiPrefix::ConvertToString(ent->Prefix()));
  else
    SW.SendUndef();

  // --- field : name ---
  SW.SendEnum(RWStepBasic_RWSiUnitName::ConvertToString(ent->Name()));

  // --- Instance of plex component SolidAngleUnit ---
  SW.StartEntity("SOLID_ANGLE_UNIT");
}
