// Created on: 2007-05-29
// Created by: Vlad Romashko
// Copyright (c) 2007-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <TDataStd_ReferenceArray.hxx>

#include <Standard_Dump.hxx>
#include <Standard_GUID.hxx>
#include <Standard_Type.hxx>
#include <TDF_Attribute.hxx>
#include <TDF_DataSet.hxx>
#include <TDF_Label.hxx>
#include <TDF_RelocationTable.hxx>
#include <TDF_Tool.hxx>

IMPLEMENT_STANDARD_RTTIEXT(TDataStd_ReferenceArray, TDF_Attribute)

//=================================================================================================

const Standard_GUID& TDataStd_ReferenceArray::GetID()
{
  static Standard_GUID TDataStd_ReferenceArrayID("7EE745A6-BB50-446c-BB0B-C195B23AB5CA");
  return TDataStd_ReferenceArrayID;
}

//=======================================================================
// function : SetAttr
// purpose  : Implements Set functionality
//=======================================================================
static occ::handle<TDataStd_ReferenceArray> SetAttr(const TDF_Label&     label,
                                                    const int            lower,
                                                    const int            upper,
                                                    const Standard_GUID& theGuid)
{
  occ::handle<TDataStd_ReferenceArray> A;
  if (!label.FindAttribute(theGuid, A))
  {
    A = new TDataStd_ReferenceArray;
    A->Init(lower, upper);
    A->SetID(theGuid);
    label.AddAttribute(A);
  }
  else if (lower != A->Lower() || upper != A->Upper())
  {
    A->Init(lower, upper);
  }
  return A;
}

//=================================================================================================

TDataStd_ReferenceArray::TDataStd_ReferenceArray()
    : myID(GetID())
{
}

//=================================================================================================

void TDataStd_ReferenceArray::Init(const int lower, const int upper)
{
  Standard_RangeError_Raise_if(upper < lower, "TDataStd_ReferenceArray::Init");
  Backup();
  myArray = new NCollection_HArray1<TDF_Label>(lower, upper);
}

//=================================================================================================

occ::handle<TDataStd_ReferenceArray> TDataStd_ReferenceArray::Set(const TDF_Label& label,
                                                                  const int        lower,
                                                                  const int        upper)
{
  return SetAttr(label, lower, upper, GetID());
}

//=======================================================================
// function : Set
// purpose  : Set user defined attribute with specific ID
//=======================================================================
occ::handle<TDataStd_ReferenceArray> TDataStd_ReferenceArray::Set(const TDF_Label&     label,
                                                                  const Standard_GUID& theGuid,
                                                                  const int            lower,
                                                                  const int            upper)
{
  return SetAttr(label, lower, upper, theGuid);
}

//=================================================================================================

void TDataStd_ReferenceArray::SetValue(const int index, const TDF_Label& value)
{
  if (myArray.IsNull())
    return;
  if (value == myArray->Value(index))
    return;
  Backup();

  myArray->SetValue(index, value);
}

//=================================================================================================

TDF_Label TDataStd_ReferenceArray::Value(const int index) const
{
  return myArray->Value(index);
}

//=================================================================================================

int TDataStd_ReferenceArray::Lower() const
{
  if (myArray.IsNull())
    return 0;
  return myArray->Lower();
}

//=================================================================================================

int TDataStd_ReferenceArray::Upper() const
{
  if (myArray.IsNull())
    return -1;
  return myArray->Upper();
}

//=================================================================================================

int TDataStd_ReferenceArray::Length() const
{
  if (myArray.IsNull())
    return 0;
  return myArray->Length();
}

//=================================================================================================

const occ::handle<NCollection_HArray1<TDF_Label>>& TDataStd_ReferenceArray::InternalArray() const
{
  return myArray;
}

//=================================================================================================

void TDataStd_ReferenceArray::SetInternalArray(
  const occ::handle<NCollection_HArray1<TDF_Label>>& values,
  const bool)
{
  //  myArray = values;
  int  aLower    = values->Lower();
  int  anUpper   = values->Upper();
  bool aDimEqual = false;
  int  i;

#ifdef OCC2932
  if (Lower() == aLower && Upper() == anUpper)
  {
    aDimEqual    = true;
    bool isEqual = true;
    if (isCheckItems)
    {
      for (i = aLower; i <= anUpper; i++)
      {
        if (myArray->Value(i) != values->Value(i))
        {
          isEqual = false;
          break;
        }
      }
      if (isEqual)
        return;
    }
  }
#endif

  Backup();

  if (myArray.IsNull() || !aDimEqual)
    myArray = new NCollection_HArray1<TDF_Label>(aLower, anUpper);

  for (i = aLower; i <= anUpper; i++)
    myArray->SetValue(i, values->Value(i));
}

//=================================================================================================

const Standard_GUID& TDataStd_ReferenceArray::ID() const
{
  return myID;
}

//=================================================================================================

void TDataStd_ReferenceArray::SetID(const Standard_GUID& theGuid)
{
  if (myID == theGuid)
    return;
  Backup();
  myID = theGuid;
}

//=================================================================================================

void TDataStd_ReferenceArray::SetID()
{
  Backup();
  myID = GetID();
}

//=================================================================================================

occ::handle<TDF_Attribute> TDataStd_ReferenceArray::NewEmpty() const
{
  return new TDataStd_ReferenceArray();
}

//=================================================================================================

void TDataStd_ReferenceArray::Restore(const occ::handle<TDF_Attribute>& With)
{
  occ::handle<TDataStd_ReferenceArray> anArray = occ::down_cast<TDataStd_ReferenceArray>(With);
  if (!anArray->myArray.IsNull())
  {
    const NCollection_Array1<TDF_Label>& arr   = anArray->myArray->Array1();
    int                                  lower = arr.Lower(), i = lower, upper = arr.Upper();
    Init(lower, upper);
    for (; i <= upper; i++)
    {
      myArray->SetValue(i, arr.Value(i));
    }
    myID = anArray->ID();
  }
  else
  {
    myArray.Nullify();
  }
}

//=================================================================================================

void TDataStd_ReferenceArray::Paste(const occ::handle<TDF_Attribute>&       Into,
                                    const occ::handle<TDF_RelocationTable>& RT) const
{
  occ::handle<TDataStd_ReferenceArray> anArray = occ::down_cast<TDataStd_ReferenceArray>(Into);
  if (myArray.IsNull())
  {
    anArray->myArray.Nullify();
    return;
  }
  const NCollection_Array1<TDF_Label>& arr   = myArray->Array1();
  int                                  lower = arr.Lower(), i = lower, upper = arr.Upper();
  if (lower != anArray->Lower() || upper != anArray->Upper())
    anArray->Init(lower, upper);
  for (; i <= upper; i++)
  {
    TDF_Label L = arr.Value(i), rL;
    if (!L.IsNull())
    {
      if (!RT->HasRelocation(L, rL))
        rL = L;
      anArray->myArray->SetValue(i, rL);
    }
  }
  anArray->SetID(myID);
}

//=======================================================================
// function : References
// purpose  : Adds the referenced attributes or labels.
//=======================================================================
void TDataStd_ReferenceArray::References(const occ::handle<TDF_DataSet>& aDataSet) const
{
  if (!Label().IsImported() && !myArray.IsNull())
  {
    const NCollection_Array1<TDF_Label>& arr   = myArray->Array1();
    int                                  lower = arr.Lower(), i = lower, upper = arr.Upper();
    for (; i <= upper; i++)
    {
      if (!arr.Value(i).IsNull())
        aDataSet->AddLabel(arr.Value(i));
    }
  }
}

//=================================================================================================

Standard_OStream& TDataStd_ReferenceArray::Dump(Standard_OStream& anOS) const
{
  anOS << "\nReferenceArray: ";
  char sguid[Standard_GUID_SIZE_ALLOC];
  myID.ToCString(sguid);
  anOS << sguid << std::endl;
  return anOS;
}

//=================================================================================================

void TDataStd_ReferenceArray::DumpJson(Standard_OStream& theOStream, int theDepth) const
{
  OCCT_DUMP_TRANSIENT_CLASS_BEGIN(theOStream)

  OCCT_DUMP_BASE_CLASS(theOStream, theDepth, TDF_Attribute)

  if (!myArray.IsNull())
  {
    OCCT_DUMP_FIELD_VALUE_NUMERICAL(theOStream, myArray->Lower())
    OCCT_DUMP_FIELD_VALUE_NUMERICAL(theOStream, myArray->Upper())

    TCollection_AsciiString aLabel;
    for (NCollection_Array1<TDF_Label>::Iterator anArrayIt(myArray->Array1()); anArrayIt.More();
         anArrayIt.Next())
    {
      aLabel.Clear();
      TDF_Tool::Entry(anArrayIt.Value(), aLabel);
      OCCT_DUMP_FIELD_VALUE_STRING(theOStream, aLabel)
    }
  }
}
