/*
 * Copyright (c) 2022 Intel Corporation
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#include <zephyr/ztest.h>
#include <zephyr/crypto/crypto.h>
#include <zephyr/kernel.h>
#include <zephyr/device.h>

#ifdef CONFIG_CRYPTO_MBEDTLS_SHIM
#define CRYPTO_DRV_NAME CONFIG_CRYPTO_MBEDTLS_SHIM_DRV_NAME
#elif DT_HAS_COMPAT_STATUS_OKAY(microchip_sha_g1_crypto)
#define CRYPTO_DEV_COMPAT microchip_sha_g1_crypto
#elif DT_HAS_COMPAT_STATUS_OKAY(renesas_smartbond_crypto)
#define CRYPTO_DEV_COMPAT renesas_smartbond_crypto
#elif DT_HAS_COMPAT_STATUS_OKAY(st_stm32_hash)
#define CRYPTO_DEV_COMPAT st_stm32_hash
#elif DT_HAS_COMPAT_STATUS_OKAY(espressif_esp32_sha)
#define CRYPTO_DEV_COMPAT espressif_esp32_sha
#elif DT_HAS_COMPAT_STATUS_OKAY(nxp_s32_crypto_hse_mu)
#define CRYPTO_DEV_COMPAT nxp_s32_crypto_hse_mu
#else
#error "You need to enable one crypto device"
#endif

/* Following test are part of mbedTLS */

uint8_t test1[] = {};
uint8_t test2[] = {0xbd};
uint8_t test3[] = {0x5f, 0xd4};
uint8_t test4[] = {0xb0, 0xbd, 0x69};
uint8_t test5[] = {0xc9, 0x8c, 0x8e, 0x55};
uint8_t test6[] = {0x81, 0xa7, 0x23, 0xd9, 0x66};
uint8_t test7[] = {
	0x83, 0x90, 0xcf, 0x0b, 0xe0, 0x76, 0x61, 0xcc, 0x76, 0x69, 0xaa, 0xc5,
	0x4c, 0xe0, 0x9a, 0x37, 0x73, 0x3a, 0x62, 0x9d, 0x45, 0xf5, 0xd9, 0x83,
	0xef, 0x20, 0x1f, 0x9b, 0x2d, 0x13, 0x80, 0x0e, 0x55, 0x5d, 0x9b, 0x10,
	0x97, 0xfe, 0xc3, 0xb7, 0x83, 0xd7, 0xa5, 0x0d, 0xcb, 0x5e, 0x2b, 0x64,
	0x4b, 0x96, 0xa1, 0xe9, 0x46, 0x3f, 0x17, 0x7c, 0xf3, 0x49, 0x06, 0xbf,
	0x38, 0x8f, 0x36, 0x6d, 0xb5, 0xc2, 0xde, 0xee, 0x04, 0xa3, 0x0e, 0x28,
	0x3f, 0x76, 0x4a, 0x97, 0xc3, 0xb3, 0x77, 0xa0, 0x34, 0xfe, 0xfc, 0x22,
	0xc2, 0x59, 0x21, 0x4f, 0xaa, 0x99, 0xba, 0xba, 0xff, 0x16, 0x0a, 0xb0,
	0xaa, 0xa7, 0xe2, 0xcc, 0xb0, 0xce, 0x09, 0xc6, 0xb3, 0x2f, 0xe0, 0x8c,
	0xbc, 0x47, 0x46, 0x94, 0x37, 0x5a, 0xba, 0x70, 0x3f, 0xad, 0xbf, 0xa3,
	0x1c, 0xf6, 0x85, 0xb3, 0x0a, 0x11, 0xc5, 0x7f, 0x3c, 0xf4, 0xed, 0xd3,
	0x21, 0xe5, 0x7d, 0x3a, 0xe6, 0xeb, 0xb1, 0x13, 0x3c, 0x82, 0x60, 0xe7,
	0x5b, 0x92, 0x24, 0xfa, 0x47, 0xa2, 0xbb, 0x20, 0x52, 0x49, 0xad, 0xd2,
	0xe2, 0xe6, 0x2f, 0x81, 0x74, 0x91, 0x48, 0x2a, 0xe1, 0x52, 0x32, 0x2b,
	0xe0, 0x90, 0x03, 0x55, 0xcd, 0xcc, 0x8d, 0x42, 0xa9, 0x8f, 0x82, 0xe9,
	0x61, 0xa0, 0xdc, 0x6f, 0x53, 0x7b, 0x7b, 0x41, 0x0e, 0xff, 0x10, 0x5f,
	0x59, 0x67, 0x3b, 0xfb, 0x78, 0x7b, 0xf0, 0x42, 0xaa, 0x07, 0x1f, 0x7a,
	0xf6, 0x8d, 0x94, 0x4d, 0x27, 0x37, 0x1c, 0x64, 0x16, 0x0f, 0xe9, 0x38,
	0x27, 0x72, 0x37, 0x25, 0x16, 0xc2, 0x30, 0xc1, 0xf4, 0x5c, 0x0d, 0x6b,
	0x6c, 0xca, 0x7f, 0x27, 0x4b, 0x39, 0x4d, 0xa9, 0x40, 0x2d, 0x3e, 0xaf,
	0xdf, 0x73, 0x39, 0x94, 0xec, 0x58, 0xab, 0x22, 0xd7, 0x18, 0x29, 0xa9,
	0x83, 0x99, 0x57, 0x4d, 0x4b, 0x59, 0x08, 0xa4, 0x47, 0xa5, 0xa6, 0x81,
	0xcb, 0x0d, 0xd5, 0x0a, 0x31, 0x14, 0x53, 0x11, 0xd9, 0x2c, 0x22, 0xa1,
	0x6d, 0xe1, 0xea, 0xd6, 0x6a, 0x54, 0x99, 0xf2, 0xdc, 0xeb, 0x4c, 0xae,
	0x69, 0x47, 0x72, 0xce, 0x90, 0x76, 0x2e, 0xf8, 0x33, 0x6a, 0xfe, 0xc6,
	0x53, 0xaa, 0x9b, 0x1a, 0x1c, 0x48, 0x20, 0xb2, 0x21, 0x13, 0x6d, 0xfc,
	0xe8, 0x0d, 0xce, 0x2b, 0xa9, 0x20, 0xd8, 0x8a, 0x53, 0x0c, 0x94, 0x10,
	0xd0, 0xa4, 0xe0, 0x35, 0x8a, 0x3a, 0x11, 0x05, 0x2e, 0x58, 0xdd, 0x73,
	0xb0, 0xb1, 0x79, 0xef, 0x8f, 0x56, 0xfe, 0x3b, 0x5a, 0x2d, 0x11, 0x7a,
	0x73, 0xa0, 0xc3, 0x8a, 0x13, 0x92, 0xb6, 0x93, 0x8e, 0x97, 0x82, 0xe0,
	0xd8, 0x64, 0x56, 0xee, 0x48, 0x84, 0xe3, 0xc3, 0x9d, 0x4d, 0x75, 0x81,
	0x3f, 0x13, 0x63, 0x3b, 0xc7, 0x9b, 0xaa, 0x07, 0xc0, 0xd2, 0xd5, 0x55,
	0xaf, 0xbf, 0x20, 0x7f, 0x52, 0xb7, 0xdc, 0xa1, 0x26, 0xd0, 0x15, 0xaa,
	0x2b, 0x98, 0x73, 0xb3, 0xeb, 0x06, 0x5e, 0x90, 0xb9, 0xb0, 0x65, 0xa5,
	0x37, 0x3f, 0xe1, 0xfb, 0x1b, 0x20, 0xd5, 0x94, 0x32, 0x7d, 0x19, 0xfb,
	0xa5, 0x6c, 0xb8, 0x1e, 0x7b, 0x66, 0x96, 0x60, 0x5f, 0xfa, 0x56, 0xeb,
	0xa3, 0xc2, 0x7a, 0x43, 0x86, 0x97, 0xcc, 0x21, 0xb2, 0x01, 0xfd, 0x7e,
	0x09, 0xf1, 0x8d, 0xee, 0xa1, 0xb3, 0xea, 0x2f, 0x0d, 0x1e, 0xdc, 0x02,
	0xdf, 0x0e, 0x20, 0x39, 0x6a, 0x14, 0x54, 0x12, 0xcd, 0x6b, 0x13, 0xc3,
	0x2d, 0x2e, 0x60, 0x56, 0x41, 0xc9, 0x48, 0xb7, 0x14, 0xae, 0xc3, 0x0c,
	0x06, 0x49, 0xdc, 0x44, 0x14, 0x35, 0x11, 0xf3, 0x5a, 0xb0, 0xfd, 0x5d,
	0xd6, 0x4c, 0x34, 0xd0, 0x6f, 0xe8, 0x6f, 0x38, 0x36, 0xdf, 0xe9, 0xed,
	0xeb, 0x7f, 0x08, 0xcf, 0xc3, 0xbd, 0x40, 0x95, 0x68, 0x26, 0x35, 0x62,
	0x42, 0x19, 0x1f, 0x99, 0xf5, 0x34, 0x73, 0xf3, 0x2b, 0x0c, 0xc0, 0xcf,
	0x93, 0x21, 0xd6, 0xc9, 0x2a, 0x11, 0x2e, 0x8d, 0xb9, 0x0b, 0x86, 0xee,
	0x9e, 0x87, 0xcc, 0x32, 0xd0, 0x34, 0x3d, 0xb0, 0x1e, 0x32, 0xce, 0x9e,
	0xb7, 0x82, 0xcb, 0x24, 0xef, 0xbb, 0xbe, 0xb4, 0x40, 0xfe, 0x92, 0x9e,
	0x8f, 0x2b, 0xf8, 0xdf, 0xb1, 0x55, 0x0a, 0x3a, 0x2e, 0x74, 0x2e, 0x8b,
	0x45, 0x5a, 0x3e, 0x57, 0x30, 0xe9, 0xe6, 0xa7, 0xa9, 0x82, 0x4d, 0x17,
	0xac, 0xc0, 0xf7, 0x2a, 0x7f, 0x67, 0xea, 0xe0, 0xf0, 0x97, 0x0f, 0x8b,
	0xde, 0x46, 0xdc, 0xde, 0xfa, 0xed, 0x30, 0x47, 0xcf, 0x80, 0x7e, 0x7f,
	0x00, 0xa4, 0x2e, 0x5f, 0xd1, 0x1d, 0x40, 0xf5, 0xe9, 0x85, 0x33, 0xd7,
	0x57, 0x44, 0x25, 0xb7, 0xd2, 0xbc, 0x3b, 0x38, 0x45, 0xc4, 0x43, 0x00,
	0x8b, 0x58, 0x98, 0x0e, 0x76, 0x8e, 0x46, 0x4e, 0x17, 0xcc, 0x6f, 0x6b,
	0x39, 0x39, 0xee, 0xe5, 0x2f, 0x71, 0x39, 0x63, 0xd0, 0x7d, 0x8c, 0x4a,
	0xbf, 0x02, 0x44, 0x8e, 0xf0, 0xb8, 0x89, 0xc9, 0x67, 0x1e, 0x2f, 0x8a,
	0x43, 0x6d, 0xde, 0xef, 0xfc, 0xca, 0x71, 0x76, 0xe9, 0xbf, 0x9d, 0x10,
	0x05, 0xec, 0xd3, 0x77, 0xf2, 0xfa, 0x67, 0xc2, 0x3e, 0xd1, 0xf1, 0x37,
	0xe6, 0x0b, 0xf4, 0x60, 0x18, 0xa8, 0xbd, 0x61, 0x3d, 0x03, 0x8e, 0x88,
	0x37, 0x04, 0xfc, 0x26, 0xe7, 0x98, 0x96, 0x9d, 0xf3, 0x5e, 0xc7, 0xbb,
	0xc6, 0xa4, 0xfe, 0x46, 0xd8, 0x91, 0x0b, 0xd8, 0x2f, 0xa3, 0xcd, 0xed,
	0x26, 0x5d, 0x0a, 0x3b, 0x6d, 0x39, 0x9e, 0x42, 0x51, 0xe4, 0xd8, 0x23,
	0x3d, 0xaa, 0x21, 0xb5, 0x81, 0x2f, 0xde, 0xd6, 0x53, 0x61, 0x98, 0xff,
	0x13, 0xaa, 0x5a, 0x1c, 0xd4, 0x6a, 0x5b, 0x9a, 0x17, 0xa4, 0xdd, 0xc1,
	0xd9, 0xf8, 0x55, 0x44, 0xd1, 0xd1, 0xcc, 0x16, 0xf3, 0xdf, 0x85, 0x80,
	0x38, 0xc8, 0xe0, 0x71, 0xa1, 0x1a, 0x7e, 0x15, 0x7a, 0x85, 0xa6, 0xa8,
	0xdc, 0x47, 0xe8, 0x8d, 0x75, 0xe7, 0x00, 0x9a, 0x8b, 0x26, 0xfd, 0xb7,
	0x3f, 0x33, 0xa2, 0xa7, 0x0f, 0x1e, 0x0c, 0x25, 0x9f, 0x8f, 0x95, 0x33,
	0xb9, 0xb8, 0xf9, 0xaf, 0x92, 0x88, 0xb7, 0x27, 0x4f, 0x21, 0xba, 0xee,
	0xc7, 0x8d, 0x39, 0x6f, 0x8b, 0xac, 0xdc, 0xc2, 0x24, 0x71, 0x20, 0x7d,
	0x9b, 0x4e, 0xfc, 0xcd, 0x3f, 0xed, 0xc5, 0xc5, 0xa2, 0x21, 0x4f, 0xf5,
	0xe5, 0x1c, 0x55, 0x3f, 0x35, 0xe2, 0x1a, 0xe6, 0x96, 0xfe, 0x51, 0xe8,
	0xdf, 0x73, 0x3a, 0x8e, 0x06, 0xf5, 0x0f, 0x41, 0x9e, 0x59, 0x9e, 0x9f,
	0x9e, 0x4b, 0x37, 0xce, 0x64, 0x3f, 0xc8, 0x10, 0xfa, 0xaa, 0x47, 0x98,
	0x97, 0x71, 0x50, 0x9d, 0x69, 0xa1, 0x10, 0xac, 0x91, 0x62, 0x61, 0x42,
	0x70, 0x26, 0x36, 0x9a, 0x21, 0x26, 0x3a, 0xc4, 0x46, 0x0f, 0xb4, 0xf7,
	0x08, 0xf8, 0xae, 0x28, 0x59, 0x98, 0x56, 0xdb, 0x7c, 0xb6, 0xa4, 0x3a,
	0xc8, 0xe0, 0x3d, 0x64, 0xa9, 0x60, 0x98, 0x07, 0xe7, 0x6c, 0x5f, 0x31,
	0x2b, 0x9d, 0x18, 0x63, 0xbf, 0xa3, 0x04, 0xe8, 0x95, 0x36, 0x47, 0x64,
	0x8b, 0x4f, 0x4a, 0xb0, 0xed, 0x99, 0x5e
};

static const uint8_t sha224_results[7][28] = {
	{0xd1, 0x4a, 0x02, 0x8c, 0x2a, 0x3a, 0x2b, 0xc9, 0x47, 0x61, 0x02, 0xbb, 0x28, 0x82,
	 0x34, 0xc4, 0x15, 0xa2, 0xb0, 0x1f, 0x82, 0x8e, 0xa6, 0x2a, 0xc5, 0xb3, 0xe4, 0x2f},
	{0xb1, 0xe4, 0x6b, 0xb9, 0xef, 0xe4, 0x5a, 0xf5, 0x54, 0x36, 0x34, 0x49, 0xc6, 0x94,
	 0x5a, 0x0d, 0x61, 0x69, 0xfc, 0x3a, 0x5a, 0x39, 0x6a, 0x56, 0xcb, 0x97, 0xcb, 0x57},
	{0xd2, 0xb0, 0x86, 0xdb, 0xee, 0x60, 0x90, 0x7e, 0xd1, 0x09, 0x86, 0x77, 0xfe, 0x04,
	 0x37, 0xb8, 0x14, 0x55, 0x66, 0x87, 0x17, 0x2d, 0x1f, 0x59, 0xcf, 0x38, 0x6c, 0x97},
	{0xbb, 0xac, 0x30, 0x15, 0x10, 0x7f, 0xe0, 0x2b, 0x0c, 0x0b, 0x2b, 0xdc, 0xb2, 0x6f,
	 0x44, 0x19, 0x1b, 0xa8, 0xdb, 0xab, 0x43, 0xf5, 0xc6, 0xbb, 0xaa, 0x0a, 0x13, 0x41},
	{0xc2, 0x63, 0xf2, 0x74, 0x12, 0xc1, 0xfd, 0xfb, 0x29, 0x33, 0xb6, 0x2a, 0xec, 0x9f,
	 0xa2, 0x60, 0x9c, 0x05, 0x7b, 0x91, 0x1e, 0x99, 0x2d, 0x8f, 0xb2, 0xe5, 0xe1, 0x5a},
	{0x95, 0x99, 0xbd, 0x2d, 0xe7, 0x5b, 0xdf, 0xef, 0xe4, 0xff, 0xf3, 0xc3, 0x78, 0x98,
	 0x87, 0xf5, 0xa5, 0x26, 0xd0, 0xea, 0x63, 0xa8, 0xa5, 0xd2, 0x8f, 0xc5, 0x62, 0xd6},
	{0x1f, 0xf8, 0x37, 0xcf, 0xf3, 0x74, 0x77, 0x4d, 0x0c, 0x23, 0x2e, 0x77, 0xb9, 0x64,
	 0xed, 0x29, 0xf5, 0xd1, 0xe7, 0x75, 0xad, 0x1a, 0x4e, 0x76, 0x6f, 0x18, 0x7a, 0x77},
};

uint8_t sha256_results[7][32] = {
	{0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14, 0x9a, 0xfb, 0xf4, 0xc8,
	 0x99, 0x6f, 0xb9, 0x24, 0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
	 0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55},
	{0x68, 0x32, 0x57, 0x20, 0xaa, 0xbd, 0x7c, 0x82, 0xf3, 0x0f, 0x55, 0x4b,
	 0x31, 0x3d, 0x05, 0x70, 0xc9, 0x5a, 0xcc, 0xbb, 0x7d, 0xc4, 0xb5, 0xaa,
	 0xe1, 0x12, 0x04, 0xc0, 0x8f, 0xfe, 0x73, 0x2b},
	{0x7c, 0x4f, 0xbf, 0x48, 0x44, 0x98, 0xd2, 0x1b, 0x48, 0x7b, 0x9d, 0x61,
	 0xde, 0x89, 0x14, 0xb2, 0xea, 0xda, 0xf2, 0x69, 0x87, 0x12, 0x93, 0x6d,
	 0x47, 0xc3, 0xad, 0xa2, 0x55, 0x8f, 0x67, 0x88},
	{0x40, 0x96, 0x80, 0x42, 0x21, 0x09, 0x3d, 0xdc, 0xcf, 0xbf, 0x46, 0x83,
	 0x14, 0x90, 0xea, 0x63, 0xe9, 0xe9, 0x94, 0x14, 0x85, 0x8f, 0x8d, 0x75,
	 0xff, 0x7f, 0x64, 0x2c, 0x7c, 0xa6, 0x18, 0x03},
	{0x7a, 0xbc, 0x22, 0xc0, 0xae, 0x5a, 0xf2, 0x6c, 0xe9, 0x3d, 0xbb, 0x94,
	 0x43, 0x3a, 0x0e, 0x0b, 0x2e, 0x11, 0x9d, 0x01, 0x4f, 0x8e, 0x7f, 0x65,
	 0xbd, 0x56, 0xc6, 0x1c, 0xcc, 0xcd, 0x95, 0x04},
	{0x75, 0x16, 0xfb, 0x8b, 0xb1, 0x13, 0x50, 0xdf, 0x2b, 0xf3, 0x86, 0xbc,
	 0x3c, 0x33, 0xbd, 0x0f, 0x52, 0xcb, 0x4c, 0x67, 0xc6, 0xe4, 0x74, 0x5e,
	 0x04, 0x88, 0xe6, 0x2c, 0x2a, 0xea, 0x26, 0x05},
	{0x41, 0x09, 0xcd, 0xbe, 0xc3, 0x24, 0x0a, 0xd7, 0x4c, 0xc6, 0xc3, 0x7f,
	 0x39, 0x30, 0x0f, 0x70, 0xfe, 0xde, 0x16, 0xe2, 0x1e, 0xfc, 0x77, 0xf7,
	 0x86, 0x59, 0x98, 0x71, 0x4a, 0xad, 0x0b, 0x5e}
};

static const uint8_t sha384_results[7][48] = {
	{0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38, 0x4c, 0xd9, 0x32, 0x7e,
	 0xb1, 0xb1, 0xe3, 0x6a, 0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
	 0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda, 0x27, 0x4e, 0xde, 0xbf,
	 0xe7, 0x6f, 0x65, 0xfb, 0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b},
	{0x43, 0x72, 0xe3, 0x8a, 0x92, 0xa2, 0x8b, 0x5d, 0x2c, 0x39, 0x1e, 0x62,
	 0x45, 0x2a, 0x86, 0xd5, 0x0e, 0x02, 0x67, 0x22, 0x8b, 0xe1, 0x76, 0xc7,
	 0x7d, 0x24, 0x02, 0xef, 0xfe, 0x9f, 0xa5, 0x0d, 0xe4, 0x07, 0xbb, 0xb8,
	 0x51, 0xb3, 0x7d, 0x59, 0x04, 0xab, 0xa2, 0xde, 0xde, 0x74, 0xda, 0x2a},
	{0xe9, 0x32, 0x48, 0x92, 0xec, 0x87, 0xf2, 0x29, 0x75, 0xc1, 0x4e, 0x42,
	 0x0f, 0x3d, 0x77, 0x7d, 0xe5, 0xe9, 0x75, 0xbf, 0x01, 0x86, 0xeb, 0x99,
	 0xdc, 0x44, 0x7c, 0xbe, 0x96, 0xcf, 0x8a, 0x7a, 0xcb, 0xbb, 0x78, 0x31,
	 0xcf, 0xaf, 0xbb, 0xbf, 0x40, 0x4c, 0xdd, 0x5a, 0x45, 0x91, 0x64, 0x75},
	{0x81, 0x3d, 0x8e, 0x6f, 0x29, 0x18, 0x99, 0xb8, 0x49, 0xf1, 0xab, 0x87,
	 0xcb, 0x04, 0x51, 0x48, 0x88, 0xb7, 0x0f, 0x5c, 0x66, 0xbe, 0xc7, 0x96,
	 0x44, 0x59, 0x55, 0xae, 0x83, 0xc3, 0x9e, 0xee, 0xec, 0x62, 0xc2, 0x5c,
	 0x25, 0xe9, 0x42, 0x94, 0x49, 0xd1, 0x33, 0x40, 0xc4, 0x7a, 0x96, 0x38},
	{0x2a, 0x6c, 0xca, 0xf7, 0xc7, 0xa7, 0xde, 0x23, 0x2e, 0xec, 0x95, 0x52,
	 0x1a, 0x64, 0x4e, 0x6d, 0x27, 0x95, 0x6e, 0x32, 0x1c, 0x9f, 0x49, 0x21,
	 0x87, 0x04, 0x14, 0x77, 0x76, 0xbc, 0x52, 0x39, 0xc5, 0x72, 0xaa, 0xe4,
	 0xbf, 0xbb, 0x19, 0x72, 0x49, 0xd7, 0xd7, 0xbd, 0x09, 0xef, 0x12, 0x88},
	{0x7c, 0xab, 0x44, 0x4f, 0xc2, 0x63, 0xf1, 0x68, 0x83, 0x29, 0x6a, 0x84,
	 0xc5, 0xb3, 0x00, 0xc0, 0x56, 0x0f, 0x4c, 0x4f, 0x08, 0x96, 0x95, 0x99,
	 0xef, 0xe0, 0x81, 0x10, 0x81, 0x16, 0xd2, 0xea, 0xed, 0xb3, 0x30, 0xb8,
	 0x06, 0x11, 0x61, 0x9c, 0x12, 0x97, 0x46, 0x05, 0xf5, 0xdf, 0x91, 0x18},
	{0xa1, 0x4c, 0x97, 0x00, 0x2f, 0x08, 0x3a, 0xf7, 0x17, 0x2d, 0x30, 0x99,
	 0x28, 0x3f, 0x36, 0xcf, 0xd1, 0xf8, 0x56, 0xe3, 0x4b, 0x89, 0x63, 0x26,
	 0x39, 0xe5, 0x36, 0xbe, 0xbe, 0xa5, 0x69, 0xa6, 0xac, 0x89, 0x19, 0x91,
	 0x5a, 0xc7, 0x78, 0xfb, 0xaa, 0xa2, 0xe2, 0x0d, 0x6f, 0xfc, 0x28, 0x41},
};

static const uint8_t sha512_results[7][64] = {
	{0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd, 0xf1, 0x54, 0x28, 0x50, 0xd6,
	 0x6d, 0x80, 0x07, 0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc, 0x83, 0xf4,
	 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce, 0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2,
	 0xb0, 0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f, 0x63, 0xb9, 0x31, 0xbd,
	 0x47, 0x41, 0x7a, 0x81, 0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e},
	{0x29, 0x6e, 0x22, 0x67, 0xd7, 0x4c, 0x27, 0x8d, 0xaa, 0xaa, 0x94, 0x0d, 0x17,
	 0xb0, 0xcf, 0xb7, 0x4a, 0x50, 0x83, 0xf8, 0xe0, 0x69, 0x72, 0x6d, 0x8c, 0x84,
	 0x1c, 0xbe, 0x59, 0x6e, 0x04, 0x31, 0xcb, 0x77, 0x41, 0xa5, 0xb5, 0x0f, 0x71,
	 0x66, 0x6c, 0xfd, 0x54, 0xba, 0xcb, 0x7b, 0x00, 0xae, 0xa8, 0x91, 0x49, 0x9c,
	 0xf4, 0xef, 0x6a, 0x03, 0xc8, 0xa8, 0x3f, 0xe3, 0x7c, 0x3f, 0x7b, 0xaf},
	{0x1c, 0xac, 0x66, 0x62, 0x60, 0x2f, 0xcc, 0x5f, 0x6c, 0x69, 0xf4, 0x8a, 0xa1,
	 0x7a, 0x92, 0x47, 0x50, 0x31, 0x36, 0x3d, 0xa5, 0xc7, 0xb7, 0x8e, 0x2c, 0xa3,
	 0xb9, 0xd2, 0x1f, 0x27, 0xd6, 0x72, 0x6c, 0x2c, 0xc7, 0x6c, 0xa5, 0x51, 0x8a,
	 0xda, 0x2c, 0x2f, 0xa7, 0x2e, 0xca, 0xf8, 0x34, 0x3f, 0x0c, 0xe1, 0xc1, 0xe8,
	 0x2f, 0x69, 0x96, 0xad, 0x6e, 0x8b, 0xbc, 0x5c, 0x84, 0xc2, 0x7f, 0xb3},
	{0x74, 0x29, 0x16, 0x51, 0x71, 0x75, 0xab, 0x4c, 0xa7, 0x01, 0x89, 0xc9, 0xae,
	 0x15, 0xca, 0xad, 0x39, 0x59, 0x9b, 0x67, 0x2d, 0xca, 0x75, 0x7b, 0x25, 0x62,
	 0x75, 0x47, 0xfb, 0x84, 0x5b, 0xe0, 0x55, 0x25, 0x23, 0x21, 0x32, 0x67, 0x3f,
	 0x09, 0x10, 0xcc, 0x24, 0x29, 0xf5, 0x0b, 0xe9, 0xc3, 0x86, 0xec, 0x14, 0xe8,
	 0x52, 0xc5, 0xa9, 0x0a, 0x03, 0xc5, 0x63, 0xe1, 0xf6, 0xd7, 0x7b, 0x5c},
	{0x09, 0xa9, 0x85, 0xc1, 0x5f, 0xa5, 0xcc, 0x68, 0xa8, 0x96, 0x8f, 0xfa, 0xea,
	 0xf3, 0xb5, 0xec, 0x23, 0x45, 0x69, 0xad, 0x56, 0x74, 0x50, 0x39, 0x4c, 0x07,
	 0x5b, 0x90, 0x61, 0xc8, 0xf7, 0xdf, 0x58, 0xff, 0x11, 0x29, 0x95, 0x70, 0x18,
	 0x82, 0x16, 0xc6, 0x31, 0xde, 0x60, 0x4d, 0xf0, 0xab, 0x08, 0xcd, 0xd1, 0x93,
	 0x29, 0x1c, 0xe5, 0x76, 0x3a, 0xcd, 0xc4, 0x27, 0xdf, 0x06, 0x04, 0x89},
	{0xf8, 0xbc, 0x6c, 0x96, 0x37, 0x92, 0xe6, 0x3d, 0x84, 0x15, 0x32, 0xee, 0xa5,
	 0x2f, 0x56, 0x70, 0xa4, 0x3e, 0xd0, 0xac, 0x72, 0xde, 0xa9, 0xb8, 0xd9, 0x2b,
	 0x0d, 0xd9, 0x19, 0xc8, 0x1a, 0x28, 0xa0, 0xff, 0xc5, 0x6d, 0xb6, 0x9a, 0xc9,
	 0x12, 0x30, 0x78, 0x2a, 0xe1, 0x2d, 0x35, 0x6e, 0x62, 0xd3, 0x94, 0x4b, 0xa0,
	 0xdb, 0x29, 0x7f, 0xe9, 0xaf, 0x34, 0x85, 0xbc, 0xd6, 0x9e, 0xe4, 0x1c},
	{0xd2, 0x7d, 0xf8, 0xfb, 0x7e, 0xd5, 0xb9, 0x1c, 0xa8, 0x08, 0x33, 0x77, 0xf1,
	 0x3c, 0x6f, 0x59, 0x63, 0xd7, 0x4e, 0x4a, 0xcc, 0x54, 0x24, 0x6b, 0x2e, 0x14,
	 0x96, 0x25, 0xd2, 0x52, 0xeb, 0xf2, 0x66, 0x25, 0x3c, 0xae, 0x62, 0x83, 0x38,
	 0xc1, 0x11, 0x94, 0x16, 0x8b, 0x73, 0x72, 0xf3, 0xf8, 0x61, 0x04, 0x1a, 0x46,
	 0xaf, 0x54, 0x4b, 0x2f, 0xec, 0xdf, 0x76, 0x08, 0x57, 0x71, 0xef, 0x8c},
};

static inline const struct device *get_crypto_dev(void)
{
#ifdef CRYPTO_DRV_NAME
	const struct device *dev = device_get_binding(CRYPTO_DRV_NAME);
#else
	const struct device *dev = DEVICE_DT_GET_ONE(CRYPTO_DEV_COMPAT);
#endif
	return dev;
}

static void run_vector_set(enum hash_algo algo, size_t out_len, const uint8_t *const *inputs,
			   const size_t *in_lens, const uint8_t *const expected[], size_t nvec)
{
	const struct device *dev = get_crypto_dev();

	zassert_true(dev && device_is_ready(dev), "Crypto device is not ready");

	struct hash_ctx ctx = {.flags = CAP_SYNC_OPS | CAP_SEPARATE_IO_BUFS};
	int rc = hash_begin_session(dev, &ctx, algo);

	/* Skip test if algorithm is not supported */
	if (rc == -ENOTSUP) {
		ztest_test_skip();
		return;
	}

	zassert_equal(rc, 0, "begin_session failed");

	for (size_t i = 0; i < nvec; i++) {
		uint8_t out[64] = {0}; /* big enough for SHA-512 */

		struct hash_pkt pkt = {
			/* Safe: hash operations only read from in_buf, never modify it */
			.in_buf = inputs[i],
			.in_len = in_lens[i],
			.out_buf = out,
		};
		rc = hash_compute(&ctx, &pkt);
		zassert_equal(rc, 0, "hash_compute failed @vec %d", (int)i + 1);
		rc = memcmp(out, expected[i], out_len);
		zassert_equal(rc, 0, "digest mismatch @vec %d", (int)i + 1);
	}

	hash_free_session(dev, &ctx);
}

/* Convert our 2D byte arrays to array-of-pointers for helper */
#define PTRS_FROM_2D(name)                                                                         \
	static const uint8_t *name##_ptrs[] = {name[0], name[1], name[2], name[3],                 \
					       name[4], name[5], name[6]}

PTRS_FROM_2D(sha224_results);
PTRS_FROM_2D(sha256_results);
PTRS_FROM_2D(sha384_results);
PTRS_FROM_2D(sha512_results);

/* Inputs & sizes */
static const uint8_t *inputs[] = {test1, test2, test3, test4, test5, test6, test7};
static const size_t in_lens[] = {sizeof(test1), sizeof(test2), sizeof(test3), sizeof(test4),
				 sizeof(test5), sizeof(test6), sizeof(test7)};

ZTEST(crypto_hash, test_sha224)
{
	run_vector_set(CRYPTO_HASH_ALGO_SHA224, 28, inputs, in_lens,
		       (const uint8_t *const *)sha224_results_ptrs, 7);
}

ZTEST(crypto_hash, test_sha256)
{
	run_vector_set(CRYPTO_HASH_ALGO_SHA256, 32, inputs, in_lens,
		       (const uint8_t *const *)sha256_results_ptrs, 7);
}

ZTEST(crypto_hash, test_sha384)
{
	run_vector_set(CRYPTO_HASH_ALGO_SHA384, 48, inputs, in_lens,
		       (const uint8_t *const *)sha384_results_ptrs, 7);
}

ZTEST(crypto_hash, test_sha512)
{
	run_vector_set(CRYPTO_HASH_ALGO_SHA512, 64, inputs, in_lens,
		       (const uint8_t *const *)sha512_results_ptrs, 7);
}

ZTEST_SUITE(crypto_hash, NULL, NULL, NULL, NULL, NULL);
