/*
 * Copyright (c) 2023 Intel Corporation.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#include <errno.h>
#include <zephyr/sys/util.h>

#include <zephyr/debug/coredump.h>
#include "coredump_internal.h"

#include <zephyr/logging/log.h>
#include <zephyr/logging/log_ctrl.h>
#include <adsp_memory.h>
#include <adsp_debug_window.h>

LOG_MODULE_REGISTER(coredump, CONFIG_DEBUG_COREDUMP_LOG_LEVEL);

static int error;
static uint32_t mem_wptr;

#ifdef CONFIG_INTEL_ADSP_DEBUG_SLOT_MANAGER
static void *coredump_slot_addr;
#endif

static void coredump_mem_window_backend_start(void)
{
	/* Reset error & mem write ptr */
	error = 0;
	mem_wptr = 0;
#ifdef CONFIG_INTEL_ADSP_DEBUG_SLOT_MANAGER
	struct adsp_dw_desc slot_desc = { .type = ADSP_DW_SLOT_TELEMETRY, };

	/* Forcibly take debug slot 1 */
	coredump_slot_addr = adsp_dw_seize_slot(1, &slot_desc, NULL);
	if (!coredump_slot_addr) {
		/* Try to get the first slot if slot 1 is not available as fallback */
		coredump_slot_addr = adsp_dw_seize_slot(0, &slot_desc, NULL);
	}
#else
	ADSP_DW->descs[1].type = ADSP_DW_SLOT_TELEMETRY;
#endif

	while (LOG_PROCESS()) {
		;
	}

	LOG_PANIC();
	LOG_ERR(COREDUMP_PREFIX_STR COREDUMP_BEGIN_STR);
}

static void coredump_mem_window_backend_end(void)
{
	if (error != 0) {
		LOG_ERR(COREDUMP_PREFIX_STR COREDUMP_ERROR_STR);
	}

	LOG_ERR(COREDUMP_PREFIX_STR COREDUMP_END_STR);
}

static void coredump_mem_window_backend_buffer_output(uint8_t *buf, size_t buflen)
{
	uint8_t *coredump_data = buf;
	size_t data_left;
#ifdef CONFIG_INTEL_ADSP_DEBUG_SLOT_MANAGER
	uint32_t *mem_window_separator = (uint32_t *)coredump_slot_addr;
	uint8_t *mem_window_sink = (uint8_t *)coredump_slot_addr + 4 + mem_wptr;

	if (!coredump_slot_addr) {
		return;
	}
#else
	uint32_t *mem_window_separator = (uint32_t *)(ADSP_DW->slots[1]);
	uint8_t *mem_window_sink = (uint8_t *)(ADSP_DW->slots[1]) + 4 + mem_wptr;
#endif

	/* Default place for telemetry dump is in memory window. Each data is easily find using
	 * separator. For telemetry that separator is 0x0DEC0DEB.
	 */
	*mem_window_separator = 0x0DEC0DEB;

	/* skip the overflow data. Don't wrap around to keep the most important data
	 * such as registers and call stack in the beginning of mem window.
	 */
	if (mem_wptr >= ADSP_DW_SLOT_SIZE - 4) {
		return;
	}

	if (buf) {
		for (data_left = buflen; data_left > 0; data_left--) {
			*mem_window_sink = *coredump_data;
			mem_window_sink++;
			coredump_data++;
		}

		mem_wptr += buflen;
	} else {
		error = -EINVAL;
	}
}

static int coredump_mem_window_backend_query(enum coredump_query_id query_id,
					     void *arg)
{
	int ret;

	switch (query_id) {
	case COREDUMP_QUERY_GET_ERROR:
		ret = error;
		break;
	default:
		ret = -ENOTSUP;
		break;
	}

	return ret;
}

static int coredump_mem_window_backend_cmd(enum coredump_cmd_id cmd_id,
					   void *arg)
{
	int ret;

	switch (cmd_id) {
	case COREDUMP_CMD_CLEAR_ERROR:
		ret = 0;
		error = 0;
		break;
	default:
		ret = -ENOTSUP;
		break;
	}

	return ret;
}

struct coredump_backend_api coredump_backend_intel_adsp_mem_window = {
	.start = coredump_mem_window_backend_start,
	.end = coredump_mem_window_backend_end,
	.buffer_output = coredump_mem_window_backend_buffer_output,
	.query = coredump_mem_window_backend_query,
	.cmd = coredump_mem_window_backend_cmd,
};
