/*
 * Copyright (c) 2025, FocalTech Systems CO.,Ltd
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef ZEPHYR_INCLUDE_DT_BINDINGS_CLOCK_FOCALTECH_FT9001_CLOCKS_H_
#define ZEPHYR_INCLUDE_DT_BINDINGS_CLOCK_FOCALTECH_FT9001_CLOCKS_H_

/**
 * @defgroup focaltech_clock_macros FocalTech Clock Configuration Macros
 * @brief Macros for encoding clock register and bit information
 * @{
 */

#define FOCALTECH_CLOCK_SHIFT 6U
#define FOCALTECH_CLOCK_MASK  ((1U << FOCALTECH_CLOCK_SHIFT) - 1U)

#define FOCALTECH_CLOCK_CONFIG(reg, bit)                                                           \
	(((FOCALTECH_##reg##_OFFSET) << FOCALTECH_CLOCK_SHIFT) | ((bit) & FOCALTECH_CLOCK_MASK))

/** @} */

/**
 * @defgroup focaltech_clk_regs FocalTech Clock Power Module Registers
 * @brief Clock Power Module Register offsets
 * @{
 */

/** Multi Clock Gate Control Register */
#define FOCALTECH_MULTICGCR_OFFSET 0x3CU
/** System Clock Gate Control Register */
#define FOCALTECH_SYSCGCR_OFFSET   0x40U
/** AHB3 Clock Gate Control Register */
#define FOCALTECH_AHB3CGCR_OFFSET  0x44U
/** Algorithm Clock Gate Control Register */
#define FOCALTECH_ARITHCGCR_OFFSET 0x48U
/** IPS Clock Gate Control Register */
#define FOCALTECH_IPSCGCR_OFFSET   0x4CU
/** EPORT Clock Gate Control Register */
#define FOCALTECH_EPORTCGCR_OFFSET 0x84U

/** @} */

/**
 * @defgroup focaltech_clk_enables FocalTech Clock Enable/Disable Definitions
 * @brief Clock enable/disable definitions for peripherals
 * @{
 */

/**
 * @defgroup focaltech_multicgcr_clocks MULTICGCR Clock Control
 * @brief Multi Clock Gate Control Register peripherals
 * @{
 */

/** OTP AHB clock */
#define FOCALTECH_CLOCK_OTP_AHB   FOCALTECH_CLOCK_CONFIG(MULTICGCR, 0U)
/** MCC clock */
#define FOCALTECH_CLOCK_MCC       FOCALTECH_CLOCK_CONFIG(MULTICGCR, 1U)
/** MCC address clock */
#define FOCALTECH_CLOCK_MCCADR    FOCALTECH_CLOCK_CONFIG(MULTICGCR, 2U)
/** ADC clock */
#define FOCALTECH_CLOCK_ADC       FOCALTECH_CLOCK_CONFIG(MULTICGCR, 3U)
/** MESH clock */
#define FOCALTECH_CLOCK_MESH      FOCALTECH_CLOCK_CONFIG(MULTICGCR, 4U)
/** TC clock */
#define FOCALTECH_CLOCK_TC        FOCALTECH_CLOCK_CONFIG(MULTICGCR, 5U)
/** Clock output */
#define FOCALTECH_CLOCK_CLKOUT    FOCALTECH_CLOCK_CONFIG(MULTICGCR, 6U)
/** Key controller clock */
#define FOCALTECH_CLOCK_KEY_CTRL  FOCALTECH_CLOCK_CONFIG(MULTICGCR, 7U)
/** CCM reset clock */
#define FOCALTECH_CLOCK_CCM_RESET FOCALTECH_CLOCK_CONFIG(MULTICGCR, 8U)
/** OTP IPS clock */
#define FOCALTECH_CLOCK_OTP_IPS   FOCALTECH_CLOCK_CONFIG(MULTICGCR, 9U)
/** CPM IPS clock */
#define FOCALTECH_CLOCK_CPM_IPS   FOCALTECH_CLOCK_CONFIG(MULTICGCR, 10U)
/** Trace clock */
#define FOCALTECH_CLOCK_TRACE     FOCALTECH_CLOCK_CONFIG(MULTICGCR, 11U)
/** I2S master clock */
#define FOCALTECH_CLOCK_I2S_M     FOCALTECH_CLOCK_CONFIG(MULTICGCR, 12U)
/** I2S slave clock */
#define FOCALTECH_CLOCK_I2S_S     FOCALTECH_CLOCK_CONFIG(MULTICGCR, 13U)
/** CRC0 clock */
#define FOCALTECH_CLOCK_CRC0      FOCALTECH_CLOCK_CONFIG(MULTICGCR, 14U)
/** CRC1 clock */
#define FOCALTECH_CLOCK_CRC1      FOCALTECH_CLOCK_CONFIG(MULTICGCR, 15U)

/** @} */

/**
 * @defgroup focaltech_syscgcr_clocks SYSCGCR Clock Control
 * @brief System Clock Gate Control Register peripherals
 * @{
 */

/** SSI slave 1 clock */
#define FOCALTECH_CLOCK_SSISLV1   FOCALTECH_CLOCK_CONFIG(SYSCGCR, 0U)
/** DMAC1 clock */
#define FOCALTECH_CLOCK_DMAC1     FOCALTECH_CLOCK_CONFIG(SYSCGCR, 1U)
/** DMAC2 clock */
#define FOCALTECH_CLOCK_DMAC2     FOCALTECH_CLOCK_CONFIG(SYSCGCR, 2U)
/** AHB2 MUX clock */
#define FOCALTECH_CLOCK_AHB2_MUX  FOCALTECH_CLOCK_CONFIG(SYSCGCR, 3U)
/** PXLP1 AHB clock */
#define FOCALTECH_CLOCK_PXLP1_AHB FOCALTECH_CLOCK_CONFIG(SYSCGCR, 10U)
/** PXLP2 AHB clock */
#define FOCALTECH_CLOCK_PXLP2_AHB FOCALTECH_CLOCK_CONFIG(SYSCGCR, 11U)
/** SRAM data clock */
#define FOCALTECH_CLOCK_SRAMD     FOCALTECH_CLOCK_CONFIG(SYSCGCR, 12U)
/** SRAM0 clock */
#define FOCALTECH_CLOCK_SRAM0     FOCALTECH_CLOCK_CONFIG(SYSCGCR, 13U)
/** SRAM1 clock */
#define FOCALTECH_CLOCK_SRAM1     FOCALTECH_CLOCK_CONFIG(SYSCGCR, 14U)
/** SRAM2 clock */
#define FOCALTECH_CLOCK_SRAM2     FOCALTECH_CLOCK_CONFIG(SYSCGCR, 15U)
/** SRAM3/4 clock */
#define FOCALTECH_CLOCK_SRAM3_4   FOCALTECH_CLOCK_CONFIG(SYSCGCR, 16U)
/** SSI4 clock */
#define FOCALTECH_CLOCK_SSI4      FOCALTECH_CLOCK_CONFIG(SYSCGCR, 17U)
/** SSI5 clock */
#define FOCALTECH_CLOCK_SSI5      FOCALTECH_CLOCK_CONFIG(SYSCGCR, 18U)
/** SSI6 clock */
#define FOCALTECH_CLOCK_SSI6      FOCALTECH_CLOCK_CONFIG(SYSCGCR, 19U)

/** @} */

/**
 * @defgroup focaltech_ahb3cgcr_clocks AHB3CGCR Clock Control
 * @brief AHB3 Clock Gate Control Register peripherals
 * @{
 */

/** ROM clock */
#define FOCALTECH_CLOCK_ROM         FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 0U)
/** ROM OTP MUX clock */
#define FOCALTECH_CLOCK_ROM_OTP_MUX FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 1U)
/** AHB3 MUX clock */
#define FOCALTECH_CLOCK_AHB3_MUX    FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 2U)
/** USB controller clock */
#define FOCALTECH_CLOCK_USBC        FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 3U)
/** PXLP1 APB clock */
#define FOCALTECH_CLOCK_PXLP1_APB   FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 4U)
/** PXLP2 APB clock */
#define FOCALTECH_CLOCK_PXLP2_APB   FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 5U)
/** I2S1 clock */
#define FOCALTECH_CLOCK_I2S1        FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 6U)
/** I2S2 clock */
#define FOCALTECH_CLOCK_I2S2        FOCALTECH_CLOCK_CONFIG(AHB3CGCR, 7U)

/** @} */

/**
 * @defgroup focaltech_arithcgcr_clocks ARITHCGCR Clock Control
 * @brief Algorithm Clock Gate Control Register peripherals
 * @{
 */

/** AES clock */
#define FOCALTECH_CLOCK_AES        FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 1U)
/** SM4 clock */
#define FOCALTECH_CLOCK_SM4        FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 2U)
/** RF AES SM4 clock */
#define FOCALTECH_CLOCK_RF_AES_SM4 FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 4U)
/** Crypto clock */
#define FOCALTECH_CLOCK_CRYPTO     FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 5U)
/** SHA clock */
#define FOCALTECH_CLOCK_SHA        FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 6U)
/** EDMAC0 clock */
#define FOCALTECH_CLOCK_EDMAC0     FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 7U)
/** DES clock */
#define FOCALTECH_CLOCK_DES        FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 8U)
/** ZUC clock */
#define FOCALTECH_CLOCK_ZUC        FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 9U)
/** AHB2MLB clock */
#define FOCALTECH_CLOCK_AHB2MLB    FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 10U)
/** AHB2IPS2 clock */
#define FOCALTECH_CLOCK_AHB2IPS2   FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 11U)
/** TRNG clock */
#define FOCALTECH_CLOCK_TRNG       FOCALTECH_CLOCK_CONFIG(ARITHCGCR, 12U)

/** @} */

/**
 * @defgroup focaltech_ipscgcr_clocks IPSCGCR Clock Control
 * @brief IPS Clock Gate Control Register peripherals
 * @{
 */

/** IO controller clock */
#define FOCALTECH_CLOCK_IO_CTRL FOCALTECH_CLOCK_CONFIG(IPSCGCR, 0U)
/** Watchdog timer clock */
#define FOCALTECH_CLOCK_WDT     FOCALTECH_CLOCK_CONFIG(IPSCGCR, 1U)
/** RTC clock */
#define FOCALTECH_CLOCK_RTC     FOCALTECH_CLOCK_CONFIG(IPSCGCR, 2U)
/** PIT1 clock */
#define FOCALTECH_CLOCK_PIT1    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 3U)
/** PIT2 clock */
#define FOCALTECH_CLOCK_PIT2    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 4U)
/** USI1 clock */
#define FOCALTECH_CLOCK_USI1    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 5U)
/** EDMAC1 clock */
#define FOCALTECH_CLOCK_EDMAC1  FOCALTECH_CLOCK_CONFIG(IPSCGCR, 6U)
/** SPI1 clock */
#define FOCALTECH_CLOCK_SPI1    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 7U)
/** SPI2 clock */
#define FOCALTECH_CLOCK_SPI2    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 8U)
/** SPI3 clock */
#define FOCALTECH_CLOCK_SPI3    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 9U)
/** SCI1 clock */
#define FOCALTECH_CLOCK_SCI1    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 10U)
/** SCI2 clock */
#define FOCALTECH_CLOCK_SCI2    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 11U)
/** USI2 clock */
#define FOCALTECH_CLOCK_USI2    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 12U)
/** CAN1 clock */
#define FOCALTECH_CLOCK_CAN1    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 13U)
/** I2C1 clock */
#define FOCALTECH_CLOCK_I2C1    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 14U)
/** PWM0 clock */
#define FOCALTECH_CLOCK_PWM0    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 15U)
/** I2C2 clock */
#define FOCALTECH_CLOCK_I2C2    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 16U)
/** CAN2 clock */
#define FOCALTECH_CLOCK_CAN2    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 17U)
/** SCI3 clock */
#define FOCALTECH_CLOCK_SCI3    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 18U)
/** QADC clock */
#define FOCALTECH_CLOCK_QADC    FOCALTECH_CLOCK_CONFIG(IPSCGCR, 20U)
/** DAC clock */
#define FOCALTECH_CLOCK_DAC     FOCALTECH_CLOCK_CONFIG(IPSCGCR, 21U)
/** MCC IPS clock */
#define FOCALTECH_CLOCK_MCC_IPS FOCALTECH_CLOCK_CONFIG(IPSCGCR, 22U)
/** PWMT1 clock */
#define FOCALTECH_CLOCK_PWMT1   FOCALTECH_CLOCK_CONFIG(IPSCGCR, 23U)
/** LD clock */
#define FOCALTECH_CLOCK_LD      FOCALTECH_CLOCK_CONFIG(IPSCGCR, 24U)
/** PWMT2 clock */
#define FOCALTECH_CLOCK_PWMT2   FOCALTECH_CLOCK_CONFIG(IPSCGCR, 25U)
/** PGD clock */
#define FOCALTECH_CLOCK_PGD     FOCALTECH_CLOCK_CONFIG(IPSCGCR, 26U)
/** Security detector clock */
#define FOCALTECH_CLOCK_SEC_DET FOCALTECH_CLOCK_CONFIG(IPSCGCR, 27U)
/** PCI clock */
#define FOCALTECH_CLOCK_PCI     FOCALTECH_CLOCK_CONFIG(IPSCGCR, 28U)
/** PMU RTC clock */
#define FOCALTECH_CLOCK_PMURTC  FOCALTECH_CLOCK_CONFIG(IPSCGCR, 29U)
/** AHB2IPS clock */
#define FOCALTECH_CLOCK_AHB2IPS FOCALTECH_CLOCK_CONFIG(IPSCGCR, 30U)
/** PWMT3 clock */
#define FOCALTECH_CLOCK_PWMT3   FOCALTECH_CLOCK_CONFIG(IPSCGCR, 31U)

/** @} */

/**
 * @defgroup focaltech_eportcgcr_clocks EPORTCGCR Clock Control
 * @brief EPORT Clock Gate Control Register peripherals
 * @{
 */

/** EPORT0 clock */
#define FOCALTECH_CLOCK_EPORT0 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 0U)
/** EPORT1 clock */
#define FOCALTECH_CLOCK_EPORT1 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 1U)
/** EPORT2 clock */
#define FOCALTECH_CLOCK_EPORT2 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 2U)
/** EPORT3 clock */
#define FOCALTECH_CLOCK_EPORT3 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 3U)
/** EPORT4 clock */
#define FOCALTECH_CLOCK_EPORT4 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 4U)
/** EPORT5 clock */
#define FOCALTECH_CLOCK_EPORT5 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 5U)
/** EPORT6 clock */
#define FOCALTECH_CLOCK_EPORT6 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 6U)
/** EPORT7 clock */
#define FOCALTECH_CLOCK_EPORT7 FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 7U)
/** ACMP1 clock */
#define FOCALTECH_CLOCK_ACMP1  FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 8U)
/** ACMP2 clock */
#define FOCALTECH_CLOCK_ACMP2  FOCALTECH_CLOCK_CONFIG(EPORTCGCR, 9U)

/** @} */

/** @} */

#endif /* ZEPHYR_INCLUDE_DT_BINDINGS_CLOCK_FOCALTECH_FT9001_CLOCKS_H_ */
