/*
 * Copyright (c) 2016 Cadence Design Systems, Inc.
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @file
 * @brief Xtensa specific kernel interface header
 *
 * This header contains the Xtensa specific kernel interface.  It is
 * included by the kernel interface architecture-abstraction header
 * (include/zephyr/arch/cpu.h).
 */

#ifndef ZEPHYR_INCLUDE_ARCH_XTENSA_ARCH_H_
#define ZEPHYR_INCLUDE_ARCH_XTENSA_ARCH_H_

#include <zephyr/irq.h>

#include <zephyr/devicetree.h>
#if !defined(_ASMLANGUAGE) && !defined(__ASSEMBLER__)
#include <zephyr/types.h>
#include <zephyr/toolchain.h>
#include <zephyr/arch/common/sys_bitops.h>
#include <zephyr/arch/common/sys_io.h>
#include <zephyr/arch/common/ffs.h>
#include <zephyr/sw_isr_table.h>
#include <zephyr/arch/xtensa/syscall.h>
#include <zephyr/arch/xtensa/thread.h>
#include <zephyr/arch/xtensa/irq.h>
#include <xtensa/config/core.h>
#include <zephyr/arch/common/addr_types.h>
#include <zephyr/arch/xtensa/gdbstub.h>
#include <zephyr/debug/sparse.h>
#include <zephyr/arch/xtensa/thread_stack.h>
#include <zephyr/sys/slist.h>

#include <zephyr/drivers/timer/system_timer.h>

#ifdef CONFIG_XTENSA_MMU
#include <zephyr/arch/xtensa/xtensa_mmu.h>
#endif

#ifdef CONFIG_XTENSA_MPU
#include <zephyr/arch/xtensa/mpu.h>
#endif

/**
 * @defgroup xtensa_apis Xtensa APIs
 * @ingroup arch-interface
 * @{
 * @}
 *
 * @defgroup xtensa_internal_apis Xtensa Internal APIs
 * @ingroup xtensa_apis
 * @{
 * @}
 */

#include <zephyr/arch/exception.h>

#ifdef __cplusplus
extern "C" {
#endif

struct arch_mem_domain {
#ifdef CONFIG_XTENSA_MMU
	uint32_t *ptables;
	uint8_t asid;
	bool dirty;

	/* Following are used to program registers when changing page tables. */
	uint32_t reg_asid;
	uint32_t reg_ptevaddr;
	uint32_t reg_ptepin_as;
	uint32_t reg_ptepin_at;
	uint32_t reg_vecpin_as;
	uint32_t reg_vecpin_at;
#endif
#ifdef CONFIG_XTENSA_MPU
	struct xtensa_mpu_map mpu_map;
#endif
	sys_snode_t node;
};

typedef struct arch_mem_domain arch_mem_domain_t;

/**
 * @brief Generate hardware exception.
 *
 * This generates hardware exception which is used by ARCH_EXCEPT().
 *
 * @param reason_p Reason for exception.
 */
void xtensa_arch_except(int reason_p);

/**
 * @brief Generate kernel oops.
 *
 * This generates kernel oops which is used by arch_syscall_oops().
 *
 * @param reason_p Reason for exception.
 * @param ssf Stack pointer.
 */
void xtensa_arch_kernel_oops(int reason_p, void *ssf);

#ifdef CONFIG_USERSPACE

#define ARCH_EXCEPT(reason_p) do { \
	if (k_is_user_context()) { \
		arch_syscall_invoke1(reason_p, \
			K_SYSCALL_XTENSA_USER_FAULT); \
	} else { \
		xtensa_arch_except(reason_p); \
	} \
	CODE_UNREACHABLE; \
} while (false)

#else

#define ARCH_EXCEPT(reason_p) do { \
		xtensa_arch_except(reason_p); \
		CODE_UNREACHABLE; \
	} while (false)

#endif

__syscall void xtensa_user_fault(unsigned int reason);

#include <zephyr/syscalls/arch.h>

/* internal routine documented in C file, needed by IRQ_CONNECT() macro */
void z_irq_priority_set(uint32_t irq, uint32_t prio, uint32_t flags);

#define ARCH_IRQ_CONNECT(irq_p, priority_p, isr_p, isr_param_p, flags_p) \
	{ \
		Z_ISR_DECLARE(irq_p, flags_p, isr_p, isr_param_p); \
	}

/** Implementation of @ref arch_k_cycle_get_32. */
static inline uint32_t arch_k_cycle_get_32(void)
{
	return sys_clock_cycle_get_32();
}

/** Implementation of @ref arch_k_cycle_get_64. */
static inline uint64_t arch_k_cycle_get_64(void)
{
	return sys_clock_cycle_get_64();
}

/** Implementation of @ref arch_nop. */
static ALWAYS_INLINE void arch_nop(void)
{
	__asm__ volatile("nop");
}

/**
 * @brief Lock VECBASE if supported by hardware.
 *
 * The bit 0 of VECBASE acts as a lock bit on hardware supporting
 * this feature. When this bit is set, VECBASE cannot be changed
 * until it is cleared by hardware reset. When the hardware does not
 * support this bit, it is hardwired to 0.
 */
static ALWAYS_INLINE void xtensa_vecbase_lock(void)
{
	int vecbase;

	__asm__ volatile("rsr.vecbase %0" : "=r" (vecbase));
	__asm__ volatile("wsr.vecbase %0; rsync" : : "r" (vecbase | 1));
}

#if defined(CONFIG_XTENSA_MMU) || defined(__DOXYGEN__)
/**
 * @brief Perform additional steps after MMU initialization.
 *
 * This performs additional steps related to memory management
 * after the main MMU initialization code. This needs to defined
 * in the SoC layer. Default is do no nothing.
 *
 * @param is_core0 True if this is called while executing on
 *                 CPU core #0.
 */
void arch_xtensa_mmu_post_init(bool is_core0);
#endif

#ifdef __cplusplus
}
#endif

#endif /* !defined(_ASMLANGUAGE) && !defined(__ASSEMBLER__)  */

#endif /* ZEPHYR_INCLUDE_ARCH_XTENSA_ARCH_H_ */
