/*
 * Copyright (c) 2022 René Beckmann
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#include <zephyr/net/mqtt_sn.h>
#include <zephyr/sys/util.h> /* for ARRAY_SIZE */
#include <zephyr/tc_util.h>
#include <zephyr/ztest.h>

#include <mqtt_sn_msg.h>

#include <zephyr/logging/log.h>
LOG_MODULE_REGISTER(test);

static const struct mqtt_sn_data client_id = MQTT_SN_DATA_STRING_LITERAL("zephyr");
static const struct mqtt_sn_data client2_id = MQTT_SN_DATA_STRING_LITERAL("zephyr2");
static const uint8_t gw_id = 12;
static const struct mqtt_sn_data gw_addr = MQTT_SN_DATA_STRING_LITERAL("gw1");

static uint8_t tx[255];
static uint8_t rx[255];

static struct msg_send_data {
	int called;
	uint8_t msg_data[CONFIG_MQTT_SN_LIB_MAX_PAYLOAD_SIZE];
	size_t msg_sz;
	int ret;
	const void *dest_addr;
	size_t addrlen;
	struct mqtt_sn_client *client;
} msg_send_data;

struct k_sem mqtt_sn_tx_sem;
struct k_sem mqtt_sn_rx_sem;
struct k_sem mqtt_sn_cb_sem;

int mqtt_sn_data_cmp(struct mqtt_sn_data data1, struct mqtt_sn_data data2)
{
	return data1.size == data2.size && strncmp(data1.data, data2.data, data1.size);
}

static int msg_sendto(struct mqtt_sn_client *client, void *buf, size_t sz, const void *dest_addr,
		      size_t addrlen)
{
	zassert_not_null(buf);
	zassert_true(sz <= sizeof(msg_send_data.msg_data),
		     "buffer is bigger than supported by the test: %zu", sz);

	msg_send_data.called++;
	msg_send_data.msg_sz = sz;
	msg_send_data.client = client;
	msg_send_data.dest_addr = dest_addr;
	msg_send_data.addrlen = addrlen;

	memcpy(msg_send_data.msg_data, buf, sz);

	k_sem_give(&mqtt_sn_tx_sem);

	return msg_send_data.ret;
}

static void assert_msg_send(int called, size_t msg_sz, const struct mqtt_sn_data *dest_addr)
{
	zassert_equal(msg_send_data.called, called, "msg_send called %d times instead of %d",
		      msg_send_data.called, called);
	zassert_equal(msg_send_data.msg_sz, msg_sz, "msg_sz is %zu instead of %zu",
		      msg_send_data.msg_sz, msg_sz);
	if (dest_addr != NULL) {
		zassert_equal(mqtt_sn_data_cmp(*dest_addr,
					       *((struct mqtt_sn_data *)msg_send_data.dest_addr)),
			      0, "Addresses incorrect");
	}

	memset(&msg_send_data, 0, sizeof(msg_send_data));
}

static void assert_msg_send_data(int called, const void *msg_data, size_t msg_sz,
				 const struct mqtt_sn_data *dest_addr)
{
	zassert_equal(msg_send_data.called, called, "msg_send called %d times instead of %d",
		      msg_send_data.called, called);
	if (msg_send_data.msg_sz != msg_sz || memcmp(msg_send_data.msg_data, msg_data, msg_sz)) {
		LOG_ERR("Unexpected msg_data");
		LOG_HEXDUMP_ERR(msg_send_data.msg_data, msg_send_data.msg_sz, "actual");
		LOG_HEXDUMP_ERR(msg_data, msg_sz, "expected");
		zassert_true(false);
	}
	if (dest_addr != NULL) {
		zassert_equal(mqtt_sn_data_cmp(*dest_addr,
					       *((struct mqtt_sn_data *)msg_send_data.dest_addr)),
			      0, "Addresses incorrect");
	}

	memset(&msg_send_data, 0, sizeof(msg_send_data));
}

static struct {
	uint8_t data[CONFIG_MQTT_SN_LIB_MAX_PAYLOAD_SIZE];
	struct mqtt_sn_evt last_evt;
	int called;
} evt_cb_data;

static void evt_cb(struct mqtt_sn_client *client, const struct mqtt_sn_evt *evt)
{
	memcpy(&evt_cb_data.last_evt, evt, sizeof(*evt));

	if (evt->type == MQTT_SN_EVT_PUBLISH) {
		zassert_true(evt->param.publish.data.size <= sizeof(evt_cb_data.data));
		memcpy(evt_cb_data.data, evt->param.publish.data.data,
		       evt->param.publish.data.size);
		evt_cb_data.last_evt.param.publish.data.data = evt_cb_data.data;
	}

	evt_cb_data.called++;

	k_sem_give(&mqtt_sn_cb_sem);
}

static bool tp_initialized;
static struct mqtt_sn_transport transport;

static int tp_init(struct mqtt_sn_transport *tp)
{
	tp_initialized = true;
	return 0;
}

static struct {
	const void *data;
	ssize_t sz;
	const void *src_addr;
	size_t addrlen;
} recvfrom_data;

static ssize_t tp_recvfrom(struct mqtt_sn_client *client, void *buffer, size_t length,
			   void *src_addr, size_t *addrlen)
{
	if (recvfrom_data.data && recvfrom_data.sz > 0 && length >= recvfrom_data.sz) {
		memcpy(buffer, recvfrom_data.data, recvfrom_data.sz);
		memcpy(src_addr, recvfrom_data.src_addr, recvfrom_data.addrlen);
		*addrlen = recvfrom_data.addrlen;

		k_sem_give(&mqtt_sn_rx_sem);
	}

	return recvfrom_data.sz;
}

int tp_poll(struct mqtt_sn_client *client)
{
	return recvfrom_data.sz;
}

#define NUM_TEST_CLIENTS 15
static ZTEST_BMEM struct mqtt_sn_client mqtt_clients[NUM_TEST_CLIENTS];
static ZTEST_BMEM struct mqtt_sn_client *mqtt_client;

static void setup(void *f)
{
	ARG_UNUSED(f);
	static ZTEST_BMEM size_t i;

	zassert_true(i < NUM_TEST_CLIENTS, "Too few clients, increase NUM_TEST_CLIENTS");

	mqtt_client = &mqtt_clients[i++];

	transport = (struct mqtt_sn_transport){
		.init = tp_init, .sendto = msg_sendto, .recvfrom = tp_recvfrom, .poll = tp_poll};
	tp_initialized = false;

	memset(&evt_cb_data, 0, sizeof(evt_cb_data));
	memset(&msg_send_data, 0, sizeof(msg_send_data));
	memset(&recvfrom_data, 0, sizeof(recvfrom_data));
	k_sem_init(&mqtt_sn_tx_sem, 0, 1);
	k_sem_init(&mqtt_sn_rx_sem, 0, 1);
	k_sem_init(&mqtt_sn_cb_sem, 0, 1);
}

static void cleanup(void *f)
{
	ARG_UNUSED(f);

	mqtt_sn_client_deinit(mqtt_client);
}

static int input(struct mqtt_sn_client *client, const void *buf, size_t sz,
		 const struct mqtt_sn_data *src_addr)
{
	recvfrom_data.data = buf;
	recvfrom_data.sz = sz;
	recvfrom_data.src_addr = src_addr->data;
	recvfrom_data.addrlen = src_addr->size;

	return mqtt_sn_input(client);
}

static void mqtt_sn_connect_no_will(struct mqtt_sn_client *client)
{
	/* connack with return code accepted */
	static uint8_t connack[] = {3, 0x05, 0x00};
	int err;

	err = mqtt_sn_client_init(client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_true(tp_initialized, "Transport not initialized");

	err = mqtt_sn_add_gw(client, gw_id, gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");
	zassert_false(sys_slist_is_empty(&client->gateway), "GW not saved.");

	err = mqtt_sn_connect(client, false, false);
	zassert_equal(err, 0, "unexpected error %d", err);
	assert_msg_send(1, 12, &gw_addr);
	zassert_equal(client->state, 0, "Wrong state");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");

	err = input(client, connack, sizeof(connack), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(client->state, 1, "Wrong state");
	zassert_equal(evt_cb_data.called, 1, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_CONNECTED, "Wrong event");
}

static ZTEST(mqtt_sn_client, test_mqtt_sn_handle_advertise)
{
	static uint8_t advertise[] = {5, 0x00, 0x0c, 0, 1};
	static uint8_t connack[] = {3, 0x05, 0x00};
	int err;

	err = mqtt_sn_client_init(mqtt_client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);

	err = input(mqtt_client, advertise, sizeof(advertise), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(evt_cb_data.called, 1, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_ADVERTISE, "Wrong event");

	err = input(mqtt_client, advertise, sizeof(advertise), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(sys_slist_len(&mqtt_client->gateway), 1, "Too many Gateways stored.");
	zassert_equal(evt_cb_data.called, 2, "Unexpected event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_ADVERTISE, "Wrong event");

	err = mqtt_sn_connect(mqtt_client, false, false);
	zassert_equal(err, 0, "unexpected error %d", err);
	assert_msg_send(1, 12, &gw_addr);
	zassert_equal(mqtt_client->state, 0, "Wrong state");
	zassert_equal(evt_cb_data.called, 2, "Unexpected event");

	err = input(mqtt_client, connack, sizeof(connack), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(mqtt_client->state, 1, "Wrong state");
	zassert_equal(evt_cb_data.called, 3, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_CONNECTED, "Wrong event");

	err = k_sem_take(&mqtt_sn_cb_sem, K_NO_WAIT);
	err = k_sem_take(&mqtt_sn_cb_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");

	zassert_true(sys_slist_is_empty(&mqtt_client->gateway), "GW not cleared on timeout");
	zassert_equal(evt_cb_data.called, 4, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_DISCONNECTED, "Wrong event");
	zassert_equal(mqtt_client->state, 0, "Wrong state");
}

static ZTEST(mqtt_sn_client, test_mqtt_sn_add_gw)
{
	int err;

	err = mqtt_sn_client_init(mqtt_client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);

	err = mqtt_sn_add_gw(mqtt_client, gw_id, gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");
}

/* Test send SEARCHGW and GW response */
static ZTEST(mqtt_sn_client, test_mqtt_sn_search_gw)
{
	int err;
	static uint8_t gwinfo[3];

	gwinfo[0] = 3;
	gwinfo[1] = 0x02;
	gwinfo[2] = gw_id;

	err = mqtt_sn_client_init(mqtt_client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);

	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	err = mqtt_sn_search(mqtt_client, 1);
	zassert_equal(err, 0, "unexpected error %d", err);

	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");

	assert_msg_send(1, 3, NULL);
	zassert_equal(mqtt_client->state, 0, "Wrong state");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");

	err = input(mqtt_client, gwinfo, sizeof(gwinfo), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_GWINFO, "Wrong event");
}

/* Test send SEARCHGW and peer response */
static ZTEST(mqtt_sn_client, test_mqtt_sn_search_peer)
{
	int err;
	static uint8_t gwinfo[3 + 3];

	gwinfo[0] = 3 + gw_addr.size;
	gwinfo[1] = 0x02;
	gwinfo[2] = gw_id;
	memcpy(&gwinfo[3], gw_addr.data, 3);

	err = mqtt_sn_client_init(mqtt_client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);

	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	err = mqtt_sn_search(mqtt_client, 1);
	zassert_equal(err, 0, "unexpected error %d", err);

	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");

	assert_msg_send(1, 3, NULL);
	zassert_equal(mqtt_client->state, 0, "Wrong state");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");

	err = input(mqtt_client, gwinfo, sizeof(gwinfo), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(evt_cb_data.called, 1, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_GWINFO, "Wrong event");
}

static ZTEST(mqtt_sn_client, test_mqtt_sn_respond_searchgw)
{
	int err;
	static uint8_t searchgw[] = {3, 0x01, 1};

	err = mqtt_sn_client_init(mqtt_client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);

	err = mqtt_sn_add_gw(mqtt_client, gw_id, gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");

	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	err = input(mqtt_client, searchgw, sizeof(searchgw), &client2_id);
	zassert_equal(err, 0, "unexpected error %d", err);

	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");

	zassert_equal(evt_cb_data.called, 1, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_SEARCHGW, "Wrong event");
	assert_msg_send(1, 3 + gw_addr.size, NULL);
}

static ZTEST(mqtt_sn_client, test_mqtt_sn_connect_no_will)
{
	mqtt_sn_connect_no_will(mqtt_client);
}

static ZTEST(mqtt_sn_client, test_mqtt_sn_connect_will)
{
	static uint8_t willtopicreq[] = {2, 0x06};
	static uint8_t willmsgreq[] = {2, 0x08};
	static uint8_t connack[] = {3, 0x05, 0x00};

	int err;

	err = mqtt_sn_client_init(mqtt_client, &client_id, &transport, evt_cb, tx, sizeof(tx), rx,
				  sizeof(rx));
	zassert_equal(err, 0, "unexpected error %d", err);

	err = mqtt_sn_add_gw(mqtt_client, gw_id, gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_false(sys_slist_is_empty(&mqtt_client->gateway), "GW not saved.");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");

	mqtt_client->will_topic = MQTT_SN_DATA_STRING_LITERAL("topic");
	mqtt_client->will_msg = MQTT_SN_DATA_STRING_LITERAL("msg");

	err = mqtt_sn_connect(mqtt_client, true, false);
	zassert_equal(err, 0, "unexpected error %d", err);
	assert_msg_send(1, 12, &gw_addr);
	zassert_equal(mqtt_client->state, 0, "Wrong state");

	err = input(mqtt_client, willtopicreq, sizeof(willtopicreq), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(mqtt_client->state, 0, "Wrong state");
	assert_msg_send(1, 8, &gw_addr);

	err = input(mqtt_client, willmsgreq, sizeof(willmsgreq), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(mqtt_client->state, 0, "Wrong state");
	zassert_equal(evt_cb_data.called, 0, "Unexpected event");
	assert_msg_send(1, 5, &gw_addr);

	err = input(mqtt_client, connack, sizeof(connack), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(mqtt_client->state, 1, "Wrong state");
	zassert_equal(evt_cb_data.called, 1, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_CONNECTED, "Wrong event");
}

/* Test a simple incoming PUBLISH event */
static ZTEST(mqtt_sn_client, test_mqtt_sn_publish_event_qos0)
{
	struct mqtt_sn_data data = MQTT_SN_DATA_STRING_LITERAL("Hello.");
	struct mqtt_sn_data topic = MQTT_SN_DATA_STRING_LITERAL("zephyr");
	uint8_t suback[] = {8, 0x13, 0, 0x1B, 0x1B, 0x00, 0x01, 0};
	static const uint8_t pubmsg[] = {0x0d, 0x0c, 0x01, 0x00, 0x2a, 0x00, 0x00,
					 'H',  'e',  'l',  'l',  'o',  '.'};
	int err;

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	err = mqtt_sn_subscribe(mqtt_client, MQTT_SN_QOS_0, &topic);
	zassert_ok(err, "Unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect a SUBSCRIBE message */
	assert_msg_send(1, 11, &gw_addr);

	err = input(mqtt_client, suback, sizeof(suback), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);

	/* Send PUBLISH */
	err = input(mqtt_client, pubmsg, sizeof(pubmsg), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(evt_cb_data.called, 2, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_PUBLISH, "Wrong event");
	zassert_equal(evt_cb_data.last_evt.param.publish.data.size, data.size,
		      "Unexpected publish data size: %zu",
		      evt_cb_data.last_evt.param.publish.data.size);
	zassert_mem_equal(evt_cb_data.last_evt.param.publish.data.data, data.data, data.size);
}

/* Test a simple incoming PUBLISH event on a predefined topic */
static ZTEST(mqtt_sn_client, test_mqtt_sn_publish_event_qos0_predefined)
{
	struct mqtt_sn_data data = MQTT_SN_DATA_STRING_LITERAL("Hello.");
	struct mqtt_sn_data topic = MQTT_SN_DATA_STRING_LITERAL("zephyr");
	uint8_t suback[] = {8, 0x13, 0, 0x1B, 0x1B, 0x00, 0x01, 0};
	static const uint8_t pubmsg[] = {0x0d, 0x0c, 0x01, 0x00, 0x2a, 0x00, 0x00,
					 'H',  'e',  'l',  'l',  'o',  '.'};
	int err;

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	err = mqtt_sn_predefine_topic(mqtt_client, 42, &topic);
	zassert_equal(err, 0, "unexpected error %d", err);

	err = mqtt_sn_predefine_topic(mqtt_client, 42, &topic);
	zassert_equal(err, -EALREADY, "unexpected error %d", err);

	err = mqtt_sn_subscribe(mqtt_client, MQTT_SN_QOS_0, &topic);
	zassert_ok(err, "Unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect a SUBSCRIBE message */
	assert_msg_send(1, 7, &gw_addr);

	err = input(mqtt_client, suback, sizeof(suback), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);

	/* Send PUBLISH */
	err = input(mqtt_client, pubmsg, sizeof(pubmsg), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	zassert_equal(evt_cb_data.called, 2, "NO event");
	zassert_equal(evt_cb_data.last_evt.type, MQTT_SN_EVT_PUBLISH, "Wrong event");
	zassert_equal(evt_cb_data.last_evt.param.publish.data.size, data.size,
		      "Unexpected publish data size: %zu",
		      evt_cb_data.last_evt.param.publish.data.size);
	zassert_mem_equal(evt_cb_data.last_evt.param.publish.data.data, data.data, data.size);
}

/* Test a simple PUBLISH message */
static ZTEST(mqtt_sn_client, test_mqtt_sn_publish_qos0)
{
	struct mqtt_sn_data data = MQTT_SN_DATA_STRING_LITERAL("Hello, World!");
	struct mqtt_sn_data topic = MQTT_SN_DATA_STRING_LITERAL("zephyr");
	/* registration ack with topic ID 0x1A1B, msg ID 0x0001, return code accepted */
	uint8_t regack[] = {7, 0x0B, 0x1A, 0x1B, 0x00, 0x01, 0};
	int err;

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	err = mqtt_sn_publish(mqtt_client, MQTT_SN_QOS_0, &topic, false, &data);
	zassert_equal(err, 0, "Unexpected error %d", err);

	assert_msg_send(0, 0, NULL);

	/* Expect a REGISTER to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send(1, 12, &gw_addr);

	/* Send REGACK in response */
	err = input(mqtt_client, regack, sizeof(regack), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	assert_msg_send(0, 0, NULL);

	/* Expect PUBLISH to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send(1, 20, &gw_addr);

	/* Expect publishes to be empty - all done */
	zassert_true(sys_slist_is_empty(&mqtt_client->publish), "Publish not empty");

	/* Expect topics not to be empty - should be remembered */
	zassert_false(sys_slist_is_empty(&mqtt_client->topic), "Topic empty");
}

/* Test a simple PUBLISH message on a predefined topic */
static ZTEST(mqtt_sn_client, test_mqtt_sn_publish_qos0_predefined)
{
	struct mqtt_sn_data data = MQTT_SN_DATA_STRING_LITERAL("Hello");
	struct mqtt_sn_data topic = MQTT_SN_DATA_STRING_LITERAL("zephyr");
	static const uint8_t msg_data_pub[] = {0x0c, 0x0c, 0x01, 0x00, 0x2a, 0x00,
					       0x00, 'H',  'e',  'l',  'l',  'o'};
	int err;

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	err = mqtt_sn_predefine_topic(mqtt_client, 42, &topic);
	zassert_equal(err, 0, "unexpected error %d", err);

	err = mqtt_sn_predefine_topic(mqtt_client, 42, &topic);
	zassert_equal(err, -EALREADY, "unexpected error %d", err);

	/* Expect topics not to be empty, because we just predefined one */
	zassert_false(sys_slist_is_empty(&mqtt_client->topic), "Topic empty");

	err = mqtt_sn_publish(mqtt_client, MQTT_SN_QOS_0, &topic, false, &data);
	zassert_equal(err, 0, "Unexpected error %d", err);

	assert_msg_send(0, 0, NULL);

	/* Expect PUBLISH to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send_data(1, msg_data_pub, sizeof(msg_data_pub), &gw_addr);

	/* Expect publishes to be empty - all done */
	zassert_true(sys_slist_is_empty(&mqtt_client->publish), "Publish not empty");

	/* Expect topics not to be empty - should be remembered */
	zassert_false(sys_slist_is_empty(&mqtt_client->topic), "Topic empty");
}

/*
 * Test two PUBLISH messages
 *
 * 6.5 Topic Name Registration Procedure
 * ...
 * At any point in time a client may have only one REGISTER message outstanding, i.e. it has to wait
 * for a REGACK message before it can register another topic name.
 */
static ZTEST(mqtt_sn_client, test_mqtt_sn_wait_regack)
{
	struct mqtt_sn_data data = MQTT_SN_DATA_STRING_LITERAL("Hello, World!");
	struct mqtt_sn_data topic1 = MQTT_SN_DATA_STRING_LITERAL("zephyr1");
	struct mqtt_sn_data topic2 = MQTT_SN_DATA_STRING_LITERAL("zephyr2");
	/* registration ack with topic ID 0x1A1B, msg ID 0x0002, return code accepted */
	uint8_t regack1[] = {7, 0x0B, 0x1A, 0x1B, 0x00, 0x01, 0};
	/* registration ack with topic ID 0x1A1C, msg ID 0x0003, return code accepted */
	uint8_t regack2[] = {7, 0x0B, 0x1A, 0x1C, 0x00, 0x03, 0};
	int err;

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	err = mqtt_sn_publish(mqtt_client, MQTT_SN_QOS_0, &topic1, false, &data);
	zassert_equal(err, 0, "Unexpected error %d", err);
	err = mqtt_sn_publish(mqtt_client, MQTT_SN_QOS_0, &topic2, false, &data);
	zassert_equal(err, 0, "Unexpected error %d", err);

	assert_msg_send(0, 0, NULL);

	/* Expect one and only one REGISTER to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send(1, 13, &gw_addr);

	/* Send REGACK in response */
	err = input(mqtt_client, regack1, sizeof(regack1), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	assert_msg_send(0, 0, NULL);

	/* Expect one more REGISTER and one PUBLISH to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send(2, 20, &gw_addr);

	/* Expect publishes not to be empty - not done yet */
	zassert_false(sys_slist_is_empty(&mqtt_client->publish), "Publish not empty");

	/* Send next REGACK in response */
	err = input(mqtt_client, regack2, sizeof(regack2), &gw_addr);
	zassert_equal(err, 0, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);
	assert_msg_send(0, 0, NULL);

	/* Expect one more PUBLISH to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send(1, 20, &gw_addr);

	/* Expect publishes to be empty - all done */
	zassert_true(sys_slist_is_empty(&mqtt_client->publish), "Publish not empty");

	/* Expect topics not to be empty - should be remembered */
	zassert_false(sys_slist_is_empty(&mqtt_client->topic), "Topic empty");
}

/*
 * Test two SUBSCRIBE / UNSUBSCRIBE messages
 *
 * 6.9 Client’s Topic Subscribe/Un-subscribe Procedure
 * ...
 * As for the REGISTER procedure, a client may have only one SUBSCRIBE or one UNSUBCRIBE transaction
 * open at a time.
 */
static ZTEST(mqtt_sn_client, test_mqtt_sn_wait_suback)
{
	struct mqtt_sn_data topic1 = MQTT_SN_DATA_STRING_LITERAL("zephyr1");
	struct mqtt_sn_data topic2 = MQTT_SN_DATA_STRING_LITERAL("zephyr2");
	uint8_t suback1[] = {8, 0x13, 0, 0x1B, 0x1B, 0x00, 0x01, 0};
	uint8_t suback2[] = {8, 0x13, 0, 0x1B, 0x1C, 0x00, 0x02, 0};
	int err;

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	err = mqtt_sn_subscribe(mqtt_client, MQTT_SN_QOS_0, &topic1);
	zassert_ok(err, "Unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect a SUBSCRIBE message */
	assert_msg_send(1, 12, &gw_addr);

	err = mqtt_sn_subscribe(mqtt_client, MQTT_SN_QOS_0, &topic2);
	zassert_ok(err, "Unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect NO message */
	assert_msg_send(0, 0, NULL);

	err = input(mqtt_client, suback1, sizeof(suback1), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect second SUBSCRIBE message */
	assert_msg_send(1, 12, &gw_addr);

	err = mqtt_sn_unsubscribe(mqtt_client, MQTT_SN_QOS_0, &topic1);
	zassert_ok(err, "Unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect NO message - SUBSCRIBE in progress */
	assert_msg_send(0, 0, NULL);

	/* Unsubscribe while subscribe in progress - should fail */
	err = mqtt_sn_unsubscribe(mqtt_client, MQTT_SN_QOS_0, &topic2);
	zassert_not_ok(err, "Error expected");

	err = input(mqtt_client, suback2, sizeof(suback2), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
	/* Expect UNSUBSCRIBE message */
	assert_msg_send(1, 12, &gw_addr);
}

/*
 * Test updating the will topic after the initial connect.
 */
static ZTEST(mqtt_sn_client, test_mqtt_sn_will_topic_update)
{
	int err;
	static struct mqtt_sn_data topic = MQTT_SN_DATA_STRING_LITERAL("will");
	static const uint8_t msg_data_request[] = {0x07, 0x1a, 0x00, 'w', 'i', 'l', 'l'};
	static const uint8_t msg_data_request_empty[] = {0x02, 0x1a};
	static const uint8_t msg_data_response[] = {0x03, 0x1b, 0x00};

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	mqtt_client->will_topic = topic;
	mqtt_client->will_retain = false;
	mqtt_client->will_qos = 0;

	err = mqtt_sn_update_will_topic(mqtt_client);
	zassert_ok(err, "unexpected error %d", err);

	/* Parallel updates are not supported. */
	err = mqtt_sn_update_will_topic(mqtt_client);
	zassert_equal(err, -EALREADY, "unexpected error %d", err);

	/* Expect WILLTOPICUPD to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send_data(1, msg_data_request, sizeof(msg_data_request), &gw_addr);

	/* Send WILLTOPICRESP in response */
	err = input(mqtt_client, msg_data_response, sizeof(msg_data_response), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));

	/* Request deletion of the will topic */
	mqtt_client->will_topic.size = 0;
	err = mqtt_sn_update_will_topic(mqtt_client);
	zassert_ok(err, "unexpected error %d", err);

	/* Expect WILLTOPICUPD to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send_data(1, msg_data_request_empty, sizeof(msg_data_request_empty), &gw_addr);

	/* Send WILLTOPICRESP in response */
	err = input(mqtt_client, msg_data_response, sizeof(msg_data_response), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
}

/*
 * Test updating the will message after the initial connect.
 */
static ZTEST(mqtt_sn_client, test_mqtt_sn_will_message_update)
{
	int err;
	static struct mqtt_sn_data will_msg = MQTT_SN_DATA_STRING_LITERAL("RIP");
	static const uint8_t msg_data_request[] = {0x05, 0x1c, 'R', 'I', 'P'};
	static const uint8_t msg_data_response[] = {0x03, 0x1d, 0x00};

	mqtt_sn_connect_no_will(mqtt_client);
	err = k_sem_take(&mqtt_sn_tx_sem, K_NO_WAIT);

	mqtt_client->will_msg = will_msg;

	err = mqtt_sn_update_will_message(mqtt_client);
	zassert_ok(err, "unexpected error %d", err);

	/* Parallel updates are not supported. */
	err = mqtt_sn_update_will_message(mqtt_client);
	zassert_equal(err, -EALREADY, "unexpected error %d", err);

	/* Expect WILLMSGUPD to be sent */
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(10));
	zassert_equal(err, 0, "Timed out waiting for callback.");
	assert_msg_send_data(1, msg_data_request, sizeof(msg_data_request), &gw_addr);

	/* Send WILLMSGRESP in response */
	err = input(mqtt_client, msg_data_response, sizeof(msg_data_response), &gw_addr);
	zassert_ok(err, "unexpected error %d", err);
	err = k_sem_take(&mqtt_sn_tx_sem, K_SECONDS(1));
}

ZTEST_SUITE(mqtt_sn_client, NULL, NULL, setup, cleanup, NULL);
