/*
 * Copyright (c) 2021 Microchip Technology Inc. and its subsidiaries.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef _MEC172X_ECIA_H
#define _MEC172X_ECIA_H

#include <stdint.h>
#include <stddef.h>
#include <zephyr/devicetree.h>

#define ECIA_BASE_ADDR			DT_REG_ADDR(DT_NODELABEL(ecia))

#define MCHP_FIRST_GIRQ_NOS		8u
#define MCHP_LAST_GIRQ_NOS		26u

/*
 * GIRQ22 is not connected to the NVIC. GIRQ22 source wake the PLL from
 * deep sleep to provide clocks for a data transfer.
 */
#define MCHP_ECIA_GIRQ_CLK_WAKE_ONLY	22u

#define MCHP_ECIA_AGGR_BITMAP	(BIT(8) | BIT(9) | BIT(10) | BIT(11) | \
				 BIT(12) | BIT(22) | BIT(24) | BIT(25) | \
				 BIT(26))

#define MCHP_ECIA_DIRECT_BITMAP	(BIT(13) | BIT(14) | BIT(15) | BIT(16) | \
				 BIT(17) | BIT(18) | BIT(19) | BIT(20) | \
				 BIT(21) | BIT(23))

#define MCHP_ECIA_ALL_BITMAP	GENMASK(26, 8)

/* MEC172x implements 8 priority levels. ARM NVIC 0 = highest priority */
#define MCHP_NVIC_NUM_PRI_BITS	3u
#define MCHP_NVIC_PRI_LO_VAL	7u

/*
 * ARM Cortex-M4 NVIC registers
 * External sources are grouped by 32-bit registers.
 * MEC172x has 181 external sources requiring 6 32-bit registers.
 */
#define MCHP_NUM_NVIC_REGS	6u
#define MCHP_NVIC_SET_EN_BASE	0xe000e100u
#define MCHP_NVIC_CLR_EN_BASE	0xe000e180u
#define MCHP_NVIC_SET_PEND_BASE 0xe000e200u
#define MCHP_NVIC_CLR_PEND_BASE 0xe000e280u
#define MCHP_NVIC_ACTIVE_BASE	0xe000e800u
#define MCHP_NVIC_PRI_BASE	0xe000e400u

/*
 * ECIA registers
 * Implements 19 GIRQ's. GIRQ's aggregated interrupts source into one
 * set of registers.
 * For historical reason GIRQ's are numbered starting at 8 in the documentation.
 * This numbering only affects the ECIA BLOCK_EN_SET, BLOCK_EN_CLR, and
 * BLOCK_ACTIVE registers: GIRQ8 is bit[8], ..., GIRQ26 is bit[26].
 *
 * Each GIRQ is composed of 5 32-bit registers.
 * +00h = GIRQ08 Source containing RW/1C status bits
 * +04h = Enable Set write 1 to bit(s) to enable the corresponding source(s)
 * +08h = Read-Only Result = Source AND Enable-Set
 * +0Ch = Enable Clear write 1 to bit(s) to disable the corresponding source(s)
 * +14h = Reserved(unused).
 * +18h = GIRQ09 Source
 * ...
 * There are three other registers at offset 0x200, 0x204, and 0x208
 * 0x200: BLOCK_EN_SET bit == 1 connects bit-wise OR of all GIRQn result
 *	to a NVIC GIRQn input.
 *	bit[8]=GIRQ8, bit[9]=GIRQ9, ..., bit[26]=GIRQ26
 * 0x204: BLOCK_EN_CLR bit == 1 disconnects bit-wise OR of GIRQn source
 *	bits from NVIC GIRQn input.
 * 0x208: BLOCK_ACTIVE (read-only)
 *	bit[8]=GIRQ8 has at least one source bit enabled and active.
 *	...
 *	bit[26]=GIRQ26 has at least one source bit enabled and active.
 * The aggregated (bit-wise OR) of GIRQ08, ..., GIRQ26 are connected to NVIC
 * external inputs except for GIRQ22. GIRQ22 is a special peripheral clock
 * wake. If GIRQ22 sources are enabled activity on a source will re-enable the
 * PLL long enough for a data transfer from the external device. If the data
 * transfer requires the CPU to be notified the normal peripheral interrupt
 * will then fire waking the CPU. If no CPU notification was enabled, HW will
 * disable the PLL once the data transfer completes allowing the SoC to
 * re-enter deep sleep.
 * Aggregated GIRQ NVIC mapping:
 * GIRQ08 -> NVIC 0
 * GIRQ09 -> NVIC 1
 * ...
 * GIRQ21 -> NVIC 13
 * GIRQ22 No connection to NVIC
 * GIRQ23 -> NVIC 14
 * ...
 * GIRQ26 -> NVIC 17
 * NVIC 20 and above are direct mode(not aggregated) connections.
 * Result bits in GIRQ's 13 - 21, and 23 can be directly connected to NVIC
 * external inputs 20 and above.
 */
#define MCHP_ECIA_BLK_ENSET_OFS		0x200u
#define MCHP_ECIA_BLK_ENCLR_OFS		0x204u
#define MCHP_ECIA_BLK_ACTIVE_OFS	0x208u

enum MCHP_GIRQ_IDS {
	MCHP_GIRQ08_ID = 8,
	MCHP_GIRQ09_ID,
	MCHP_GIRQ10_ID,
	MCHP_GIRQ11_ID,
	MCHP_GIRQ12_ID,
	MCHP_GIRQ13_ID,
	MCHP_GIRQ14_ID,
	MCHP_GIRQ15_ID,
	MCHP_GIRQ16_ID,
	MCHP_GIRQ17_ID,
	MCHP_GIRQ18_ID,
	MCHP_GIRQ19_ID,
	MCHP_GIRQ20_ID,
	MCHP_GIRQ21_ID,
	MCHP_GIRQ22_ID,
	MCHP_GIRQ23_ID,
	MCHP_GIRQ24_ID,
	MCHP_GIRQ25_ID,
	MCHP_GIRQ26_ID,
	MCHP_GIRQ_ID_MAX,
};

/* zero based index into ECIA_Type GIRQ[] */
enum MCHP_GIRQ_INDEX {
	GIRQ08_IDX = 0,
	GIRQ09_IDX,
	GIRQ10_IDX,
	GIRQ11_IDX,
	GIRQ12_IDX,
	GIRQ13_IDX,
	GIRQ14_IDX,
	GIRQ15_IDX,
	GIRQ16_IDX,
	GIRQ17_IDX,
	GIRQ18_IDX,
	GIRQ19_IDX,
	GIRQ20_IDX,
	GIRQ21_IDX,
	GIRQ22_IDX,
	GIRQ23_IDX,
	GIRQ24_IDX,
	GIRQ25_IDX,
	GIRQ26_IDX,
	GIRQ_MAX_IDX
};

/* GIRQ08 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_GPIO_0140_GIRQ_POS		0
#define MCHP_GPIO_0141_GIRQ_POS		1
#define MCHP_GPIO_0142_GIRQ_POS		2
#define MCHP_GPIO_0143_GIRQ_POS		3
#define MCHP_GPIO_0144_GIRQ_POS		4
#define MCHP_GPIO_0145_GIRQ_POS		5
#define MCHP_GPIO_0146_GIRQ_POS		6
#define MCHP_GPIO_0147_GIRQ_POS		7
#define MCHP_GPIO_0150_GIRQ_POS		8
#define MCHP_GPIO_0151_GIRQ_POS		9
#define MCHP_GPIO_0152_GIRQ_POS		10
#define MCHP_GPIO_0153_GIRQ_POS		11
#define MCHP_GPIO_0154_GIRQ_POS		12
#define MCHP_GPIO_0155_GIRQ_POS		13
#define MCHP_GPIO_0156_GIRQ_POS		14
#define MCHP_GPIO_0157_GIRQ_POS		15
#define MCHP_GPIO_0160_GIRQ_POS		16
#define MCHP_GPIO_0161_GIRQ_POS		17
#define MCHP_GPIO_0162_GIRQ_POS		18
#define MCHP_GPIO_0165_GIRQ_POS		21
#define MCHP_GPIO_0166_GIRQ_POS		22
#define MCHP_GPIO_0170_GIRQ_POS		24
#define MCHP_GPIO_0171_GIRQ_POS		25
#define MCHP_GPIO_0172_GIRQ_POS		26
#define MCHP_GPIO_0173_GIRQ_POS		27
#define MCHP_GPIO_0174_GIRQ_POS		28
#define MCHP_GPIO_0175_GIRQ_POS		29

#define MCHP_GPIO_0140_GIRQ_BIT		BIT(0)
#define MCHP_GPIO_0141_GIRQ_BIT		BIT(1)
#define MCHP_GPIO_0142_GIRQ_BIT		BIT(2)
#define MCHP_GPIO_0143_GIRQ_BIT		BIT(3)
#define MCHP_GPIO_0144_GIRQ_BIT		BIT(4)
#define MCHP_GPIO_0145_GIRQ_BIT		BIT(5)
#define MCHP_GPIO_0146_GIRQ_BIT		BIT(6)
#define MCHP_GPIO_0147_GIRQ_BIT		BIT(7)
#define MCHP_GPIO_0150_GIRQ_BIT		BIT(8)
#define MCHP_GPIO_0151_GIRQ_BIT		BIT(9)
#define MCHP_GPIO_0152_GIRQ_BIT		BIT(10)
#define MCHP_GPIO_0153_GIRQ_BIT		BIT(11)
#define MCHP_GPIO_0154_GIRQ_BIT		BIT(12)
#define MCHP_GPIO_0155_GIRQ_BIT		BIT(13)
#define MCHP_GPIO_0156_GIRQ_BIT		BIT(14)
#define MCHP_GPIO_0157_GIRQ_BIT		BIT(15)
#define MCHP_GPIO_0160_GIRQ_BIT		BIT(16)
#define MCHP_GPIO_0161_GIRQ_BIT		BIT(17)
#define MCHP_GPIO_0162_GIRQ_BIT		BIT(18)
#define MCHP_GPIO_0165_GIRQ_BIT		BIT(21)
#define MCHP_GPIO_0166_GIRQ_BIT		BIT(22)
#define MCHP_GPIO_0170_GIRQ_BIT		BIT(24)
#define MCHP_GPIO_0171_GIRQ_BIT		BIT(25)
#define MCHP_GPIO_0172_GIRQ_BIT		BIT(26)
#define MCHP_GPIO_0173_GIRQ_BIT		BIT(27)
#define MCHP_GPIO_0174_GIRQ_BIT		BIT(28)
#define MCHP_GPIO_0175_GIRQ_BIT		BIT(29)
#define MCHP_GPIO_0140_0176_GIRQ_MASK	0x3f67ffffu

/* GIRQ09 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_GPIO_0100_GIRQ_POS		0
#define MCHP_GPIO_0101_GIRQ_POS		1
#define MCHP_GPIO_0102_GIRQ_POS		2
#define MCHP_GPIO_0103_GIRQ_POS		3
#define MCHP_GPIO_0104_GIRQ_POS		4
#define MCHP_GPIO_0105_GIRQ_POS		5
#define MCHP_GPIO_0106_GIRQ_POS		6
#define MCHP_GPIO_0107_GIRQ_POS		7
#define MCHP_GPIO_0110_GIRQ_POS		8
#define MCHP_GPIO_0111_GIRQ_POS		9
#define MCHP_GPIO_0112_GIRQ_POS		10
#define MCHP_GPIO_0113_GIRQ_POS		11
#define MCHP_GPIO_0114_GIRQ_POS		12
#define MCHP_GPIO_0115_GIRQ_POS		13
#define MCHP_GPIO_0116_GIRQ_POS		14
#define MCHP_GPIO_0117_GIRQ_POS		15
#define MCHP_GPIO_0120_GIRQ_POS		16
#define MCHP_GPIO_0121_GIRQ_POS		17
#define MCHP_GPIO_0122_GIRQ_POS		18
#define MCHP_GPIO_0123_GIRQ_POS		19
#define MCHP_GPIO_0124_GIRQ_POS		20
#define MCHP_GPIO_0125_GIRQ_POS		21
#define MCHP_GPIO_0126_GIRQ_POS		22
#define MCHP_GPIO_0127_GIRQ_POS		23
#define MCHP_GPIO_0130_GIRQ_POS		24
#define MCHP_GPIO_0131_GIRQ_POS		25
#define MCHP_GPIO_0132_GIRQ_POS		26
#define MCHP_GPIO_0133_GIRQ_POS		27
#define MCHP_GPIO_0134_GIRQ_POS		28
#define MCHP_GPIO_0135_GIRQ_POS		29

#define MCHP_GPIO_0100_GIRQ_BIT		BIT(0)
#define MCHP_GPIO_0101_GIRQ_BIT		BIT(1)
#define MCHP_GPIO_0102_GIRQ_BIT		BIT(2)
#define MCHP_GPIO_0103_GIRQ_BIT		BIT(3)
#define MCHP_GPIO_0104_GIRQ_BIT		BIT(4)
#define MCHP_GPIO_0105_GIRQ_BIT		BIT(5)
#define MCHP_GPIO_0106_GIRQ_BIT		BIT(6)
#define MCHP_GPIO_0107_GIRQ_BIT		BIT(7)
#define MCHP_GPIO_0110_GIRQ_BIT		BIT(8)
#define MCHP_GPIO_0111_GIRQ_BIT		BIT(9)
#define MCHP_GPIO_0112_GIRQ_BIT		BIT(10)
#define MCHP_GPIO_0113_GIRQ_BIT		BIT(11)
#define MCHP_GPIO_0114_GIRQ_BIT		BIT(12)
#define MCHP_GPIO_0115_GIRQ_BIT		BIT(13)
#define MCHP_GPIO_0116_GIRQ_BIT		BIT(14)
#define MCHP_GPIO_0117_GIRQ_BIT		BIT(15)
#define MCHP_GPIO_0120_GIRQ_BIT		BIT(16)
#define MCHP_GPIO_0121_GIRQ_BIT		BIT(17)
#define MCHP_GPIO_0122_GIRQ_BIT		BIT(18)
#define MCHP_GPIO_0123_GIRQ_BIT		BIT(19)
#define MCHP_GPIO_0124_GIRQ_BIT		BIT(20)
#define MCHP_GPIO_0125_GIRQ_BIT		BIT(21)
#define MCHP_GPIO_0126_GIRQ_BIT		BIT(22)
#define MCHP_GPIO_0127_GIRQ_BIT		BIT(23)
#define MCHP_GPIO_0130_GIRQ_BIT		BIT(24)
#define MCHP_GPIO_0131_GIRQ_BIT		BIT(25)
#define MCHP_GPIO_0132_GIRQ_BIT		BIT(26)
#define MCHP_GPIO_0133_GIRQ_BIT		BIT(27)
#define MCHP_GPIO_0134_GIRQ_BIT		BIT(28)
#define MCHP_GPIO_0135_GIRQ_BIT		BIT(29)
#define MCHP_GPIO_0100_0136_GIRQ_MASK	0x3fffffffu

/* GIRQ10 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_GPIO_0040_GIRQ_POS		0
#define MCHP_GPIO_0041_GIRQ_POS		1
#define MCHP_GPIO_0042_GIRQ_POS		2
#define MCHP_GPIO_0043_GIRQ_POS		3
#define MCHP_GPIO_0044_GIRQ_POS		4
#define MCHP_GPIO_0045_GIRQ_POS		5
#define MCHP_GPIO_0046_GIRQ_POS		6
#define MCHP_GPIO_0047_GIRQ_POS		7
#define MCHP_GPIO_0050_GIRQ_POS		8
#define MCHP_GPIO_0051_GIRQ_POS		9
#define MCHP_GPIO_0052_GIRQ_POS		10
#define MCHP_GPIO_0053_GIRQ_POS		11
#define MCHP_GPIO_0054_GIRQ_POS		12
#define MCHP_GPIO_0055_GIRQ_POS		13
#define MCHP_GPIO_0056_GIRQ_POS		14
#define MCHP_GPIO_0057_GIRQ_POS		15
#define MCHP_GPIO_0060_GIRQ_POS		16
#define MCHP_GPIO_0061_GIRQ_POS		17
#define MCHP_GPIO_0062_GIRQ_POS		18
#define MCHP_GPIO_0063_GIRQ_POS		19
#define MCHP_GPIO_0064_GIRQ_POS		20
#define MCHP_GPIO_0065_GIRQ_POS		21
#define MCHP_GPIO_0066_GIRQ_POS		22
#define MCHP_GPIO_0067_GIRQ_POS		23
#define MCHP_GPIO_0070_GIRQ_POS		24
#define MCHP_GPIO_0071_GIRQ_POS		25
#define MCHP_GPIO_0072_GIRQ_POS		26
#define MCHP_GPIO_0073_GIRQ_POS		27
#define MCHP_GPIO_0074_GIRQ_POS		28
#define MCHP_GPIO_0075_GIRQ_POS		29
#define MCHP_GPIO_0076_GIRQ_POS		30

#define MCHP_GPIO_0040_GIRQ_BIT		BIT(0)
#define MCHP_GPIO_0041_GIRQ_BIT		BIT(1)
#define MCHP_GPIO_0042_GIRQ_BIT		BIT(2)
#define MCHP_GPIO_0043_GIRQ_BIT		BIT(3)
#define MCHP_GPIO_0044_GIRQ_BIT		BIT(4)
#define MCHP_GPIO_0045_GIRQ_BIT		BIT(5)
#define MCHP_GPIO_0046_GIRQ_BIT		BIT(6)
#define MCHP_GPIO_0047_GIRQ_BIT		BIT(7)
#define MCHP_GPIO_0050_GIRQ_BIT		BIT(8)
#define MCHP_GPIO_0051_GIRQ_BIT		BIT(9)
#define MCHP_GPIO_0052_GIRQ_BIT		BIT(10)
#define MCHP_GPIO_0053_GIRQ_BIT		BIT(11)
#define MCHP_GPIO_0054_GIRQ_BIT		BIT(12)
#define MCHP_GPIO_0055_GIRQ_BIT		BIT(13)
#define MCHP_GPIO_0056_GIRQ_BIT		BIT(14)
#define MCHP_GPIO_0057_GIRQ_BIT		BIT(15)
#define MCHP_GPIO_0060_GIRQ_BIT		BIT(16)
#define MCHP_GPIO_0061_GIRQ_BIT		BIT(17)
#define MCHP_GPIO_0062_GIRQ_BIT		BIT(18)
#define MCHP_GPIO_0063_GIRQ_BIT		BIT(19)
#define MCHP_GPIO_0064_GIRQ_BIT		BIT(20)
#define MCHP_GPIO_0065_GIRQ_BIT		BIT(21)
#define MCHP_GPIO_0066_GIRQ_BIT		BIT(22)
#define MCHP_GPIO_0067_GIRQ_BIT		BIT(23)
#define MCHP_GPIO_0070_GIRQ_BIT		BIT(24)
#define MCHP_GPIO_0071_GIRQ_BIT		BIT(25)
#define MCHP_GPIO_0072_GIRQ_BIT		BIT(26)
#define MCHP_GPIO_0073_GIRQ_BIT		BIT(27)
#define MCHP_GPIO_0074_GIRQ_BIT		BIT(28)
#define MCHP_GPIO_0075_GIRQ_BIT		BIT(29)
#define MCHP_GPIO_0076_GIRQ_BIT		BIT(30)
#define MCHP_GPIO_0040_0076_GIRQ_MASK	0x7fffffffu

/* GIRQ11 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_GPIO_0000_GIRQ_POS		0
#define MCHP_GPIO_0001_GIRQ_POS		1
#define MCHP_GPIO_0002_GIRQ_POS		2
#define MCHP_GPIO_0003_GIRQ_POS		3
#define MCHP_GPIO_0004_GIRQ_POS		4
#define MCHP_GPIO_0005_GIRQ_POS		5
#define MCHP_GPIO_0006_GIRQ_POS		6
#define MCHP_GPIO_0007_GIRQ_POS		7
#define MCHP_GPIO_0010_GIRQ_POS		8
#define MCHP_GPIO_0011_GIRQ_POS		9
#define MCHP_GPIO_0012_GIRQ_POS		10
#define MCHP_GPIO_0013_GIRQ_POS		11
#define MCHP_GPIO_0014_GIRQ_POS		12
#define MCHP_GPIO_0015_GIRQ_POS		13
#define MCHP_GPIO_0016_GIRQ_POS		14
#define MCHP_GPIO_0017_GIRQ_POS		15
#define MCHP_GPIO_0020_GIRQ_POS		16
#define MCHP_GPIO_0021_GIRQ_POS		17
#define MCHP_GPIO_0022_GIRQ_POS		18
#define MCHP_GPIO_0023_GIRQ_POS		19
#define MCHP_GPIO_0024_GIRQ_POS		20
#define MCHP_GPIO_0025_GIRQ_POS		21
#define MCHP_GPIO_0026_GIRQ_POS		22
#define MCHP_GPIO_0027_GIRQ_POS		23
#define MCHP_GPIO_0030_GIRQ_POS		24
#define MCHP_GPIO_0031_GIRQ_POS		25
#define MCHP_GPIO_0032_GIRQ_POS		26
#define MCHP_GPIO_0033_GIRQ_POS		27
#define MCHP_GPIO_0034_GIRQ_POS		28
#define MCHP_GPIO_0035_GIRQ_POS		29
#define MCHP_GPIO_0036_GIRQ_POS		30

#define MCHP_GPIO_0000_GIRQ_BIT		BIT(0)
#define MCHP_GPIO_0001_GIRQ_BIT		BIT(1)
#define MCHP_GPIO_0002_GIRQ_BIT		BIT(2)
#define MCHP_GPIO_0003_GIRQ_BIT		BIT(3)
#define MCHP_GPIO_0004_GIRQ_BIT		BIT(4)
#define MCHP_GPIO_0005_GIRQ_BIT		BIT(5)
#define MCHP_GPIO_0006_GIRQ_BIT		BIT(6)
#define MCHP_GPIO_0007_GIRQ_BIT		BIT(7)
#define MCHP_GPIO_0010_GIRQ_BIT		BIT(8)
#define MCHP_GPIO_0011_GIRQ_BIT		BIT(9)
#define MCHP_GPIO_0012_GIRQ_BIT		BIT(10)
#define MCHP_GPIO_0013_GIRQ_BIT		BIT(11)
#define MCHP_GPIO_0014_GIRQ_BIT		BIT(12)
#define MCHP_GPIO_0015_GIRQ_BIT		BIT(13)
#define MCHP_GPIO_0016_GIRQ_BIT		BIT(14)
#define MCHP_GPIO_0017_GIRQ_BIT		BIT(15)
#define MCHP_GPIO_0020_GIRQ_BIT		BIT(16)
#define MCHP_GPIO_0021_GIRQ_BIT		BIT(17)
#define MCHP_GPIO_0022_GIRQ_BIT		BIT(18)
#define MCHP_GPIO_0023_GIRQ_BIT		BIT(19)
#define MCHP_GPIO_0024_GIRQ_BIT		BIT(20)
#define MCHP_GPIO_0025_GIRQ_BIT		BIT(21)
#define MCHP_GPIO_0026_GIRQ_BIT		BIT(22)
#define MCHP_GPIO_0027_GIRQ_BIT		BIT(23)
#define MCHP_GPIO_0030_GIRQ_BIT		BIT(24)
#define MCHP_GPIO_0031_GIRQ_BIT		BIT(25)
#define MCHP_GPIO_0032_GIRQ_BIT		BIT(26)
#define MCHP_GPIO_0033_GIRQ_BIT		BIT(27)
#define MCHP_GPIO_0034_GIRQ_BIT		BIT(28)
#define MCHP_GPIO_0035_GIRQ_BIT		BIT(29)
#define MCHP_GPIO_0036_GIRQ_BIT		BIT(30)
#define MCHP_GPIO_0000_0036_GIRQ_MASK	0x7fffffffu

/* GIRQ12 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_GPIO_0200_GIRQ_POS		0
#define MCHP_GPIO_0201_GIRQ_POS		1
#define MCHP_GPIO_0202_GIRQ_POS		2
#define MCHP_GPIO_0203_GIRQ_POS		3
#define MCHP_GPIO_0204_GIRQ_POS		4
#define MCHP_GPIO_0205_GIRQ_POS		5
#define MCHP_GPIO_0206_GIRQ_POS		6
#define MCHP_GPIO_0207_GIRQ_POS		7
#define MCHP_GPIO_0210_GIRQ_POS		8
#define MCHP_GPIO_0211_GIRQ_POS		9
#define MCHP_GPIO_0212_GIRQ_POS		10
#define MCHP_GPIO_0213_GIRQ_POS		11
#define MCHP_GPIO_0214_GIRQ_POS		12
#define MCHP_GPIO_0215_GIRQ_POS		13
#define MCHP_GPIO_0216_GIRQ_POS		14
#define MCHP_GPIO_0217_GIRQ_POS		15
#define MCHP_GPIO_0220_GIRQ_POS		16
#define MCHP_GPIO_0221_GIRQ_POS		17
#define MCHP_GPIO_0222_GIRQ_POS		18
#define MCHP_GPIO_0223_GIRQ_POS		19
#define MCHP_GPIO_0224_GIRQ_POS		20
#define MCHP_GPIO_0225_GIRQ_POS		21
#define MCHP_GPIO_0226_GIRQ_POS		22
#define MCHP_GPIO_0227_GIRQ_POS		23
#define MCHP_GPIO_0230_GIRQ_POS		24
#define MCHP_GPIO_0231_GIRQ_POS		25
#define MCHP_GPIO_0232_GIRQ_POS		26
#define MCHP_GPIO_0233_GIRQ_POS		27
#define MCHP_GPIO_0234_GIRQ_POS		28
#define MCHP_GPIO_0235_GIRQ_POS		29
#define MCHP_GPIO_0236_GIRQ_POS		30

#define MCHP_GPIO_0200_GIRQ_BIT		BIT(0)
#define MCHP_GPIO_0201_GIRQ_BIT		BIT(1)
#define MCHP_GPIO_0202_GIRQ_BIT		BIT(2)
#define MCHP_GPIO_0203_GIRQ_BIT		BIT(3)
#define MCHP_GPIO_0204_GIRQ_BIT		BIT(4)
#define MCHP_GPIO_0205_GIRQ_BIT		BIT(5)
#define MCHP_GPIO_0206_GIRQ_BIT		BIT(6)
#define MCHP_GPIO_0207_GIRQ_BIT		BIT(7)
#define MCHP_GPIO_0210_GIRQ_BIT		BIT(8)
#define MCHP_GPIO_0211_GIRQ_BIT		BIT(9)
#define MCHP_GPIO_0212_GIRQ_BIT		BIT(10)
#define MCHP_GPIO_0213_GIRQ_BIT		BIT(11)
#define MCHP_GPIO_0214_GIRQ_BIT		BIT(12)
#define MCHP_GPIO_0215_GIRQ_BIT		BIT(13)
#define MCHP_GPIO_0216_GIRQ_BIT		BIT(14)
#define MCHP_GPIO_0217_GIRQ_BIT		BIT(15)
#define MCHP_GPIO_0220_GIRQ_BIT		BIT(16)
#define MCHP_GPIO_0221_GIRQ_BIT		BIT(17)
#define MCHP_GPIO_0222_GIRQ_BIT		BIT(18)
#define MCHP_GPIO_0223_GIRQ_BIT		BIT(19)
#define MCHP_GPIO_0224_GIRQ_BIT		BIT(20)
#define MCHP_GPIO_0225_GIRQ_BIT		BIT(21)
#define MCHP_GPIO_0226_GIRQ_BIT		BIT(22)
#define MCHP_GPIO_0227_GIRQ_BIT		BIT(23)
#define MCHP_GPIO_0230_GIRQ_BIT		BIT(24)
#define MCHP_GPIO_0231_GIRQ_BIT		BIT(25)
#define MCHP_GPIO_0232_GIRQ_BIT		BIT(26)
#define MCHP_GPIO_0233_GIRQ_BIT		BIT(27)
#define MCHP_GPIO_0234_GIRQ_BIT		BIT(28)
#define MCHP_GPIO_0235_GIRQ_BIT		BIT(29)
#define MCHP_GPIO_0236_GIRQ_BIT		BIT(30)
#define MCHP_GPIO_0200_0236_GIRQ_MASK	0x7fffffffu

/* GIRQ13 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_I2C_SMB_0_GIRQ_POS		0
#define MCHP_I2C_SMB_1_GIRQ_POS		1
#define MCHP_I2C_SMB_2_GIRQ_POS		2
#define MCHP_I2C_SMB_3_GIRQ_POS		3
#define MCHP_I2C_SMB_4_GIRQ_POS		4

#define MCHP_I2C_SMB_0_GIRQ_BIT		BIT(0)
#define MCHP_I2C_SMB_1_GIRQ_BIT		BIT(1)
#define MCHP_I2C_SMB_2_GIRQ_BIT		BIT(2)
#define MCHP_I2C_SMB_3_GIRQ_BIT		BIT(3)
#define MCHP_I2C_SMB_4_GIRQ_BIT		BIT(4)
#define MCHP_SMB_I2C_GIRQ_MASK		0x1fu

#define MCHP_GIRQ13_NVIC_AGGR		5u
#define MCHP_GIRQ13_NVIC_DIRECT(bitpos)	(20u + (uint32_t)(bitpos))

/* GIRQ14 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_DMA_CH00_GIRQ_POS		0
#define MCHP_DMA_CH01_GIRQ_POS		1
#define MCHP_DMA_CH02_GIRQ_POS		2
#define MCHP_DMA_CH03_GIRQ_POS		3
#define MCHP_DMA_CH04_GIRQ_POS		4
#define MCHP_DMA_CH05_GIRQ_POS		5
#define MCHP_DMA_CH06_GIRQ_POS		6
#define MCHP_DMA_CH07_GIRQ_POS		7
#define MCHP_DMA_CH08_GIRQ_POS		8
#define MCHP_DMA_CH09_GIRQ_POS		9
#define MCHP_DMA_CH10_GIRQ_POS		10
#define MCHP_DMA_CH11_GIRQ_POS		11
#define MCHP_DMA_CH12_GIRQ_POS		12
#define MCHP_DMA_CH13_GIRQ_POS		13
#define MCHP_DMA_CH14_GIRQ_POS		14
#define MCHP_DMA_CH15_GIRQ_POS		15

#define MCHP_DMA_CH00_GIRQ_BIT		BIT(0)
#define MCHP_DMA_CH01_GIRQ_BIT		BIT(1)
#define MCHP_DMA_CH02_GIRQ_BIT		BIT(2)
#define MCHP_DMA_CH03_GIRQ_BIT		BIT(3)
#define MCHP_DMA_CH04_GIRQ_BIT		BIT(4)
#define MCHP_DMA_CH05_GIRQ_BIT		BIT(5)
#define MCHP_DMA_CH06_GIRQ_BIT		BIT(6)
#define MCHP_DMA_CH07_GIRQ_BIT		BIT(7)
#define MCHP_DMA_CH08_GIRQ_BIT		BIT(8)
#define MCHP_DMA_CH09_GIRQ_BIT		BIT(9)
#define MCHP_DMA_CH10_GIRQ_BIT		BIT(10)
#define MCHP_DMA_CH11_GIRQ_BIT		BIT(11)
#define MCHP_DMA_CH12_GIRQ_BIT		BIT(12)
#define MCHP_DMA_CH13_GIRQ_BIT		BIT(13)
#define MCHP_DMA_CH14_GIRQ_BIT		BIT(14)
#define MCHP_DMA_CH15_GIRQ_BIT		BIT(15)
#define MCHP_DMA_GIRQ_MASK		0xffffu

#define MCHP_GIRQ14_NVIC_AGGR		6u
#define MCHP_GIRQ14_NVIC_DIRECT(bitpos)	(24u + (uint32_t)(bitpos))

/* GIRQ15 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_UART_0_GIRQ_POS		0
#define MCHP_UART_1_GIRQ_POS		1
#define MCHP_EMI_0_GIRQ_POS		2
#define MCHP_EMI_1_GIRQ_POS		3
#define MCHP_EMI_2_GIRQ_POS		4
#define MCHP_ACPI_EC_0_IBF_GIRQ_POS	5
#define MCHP_ACPI_EC_0_OBE_GIRQ_POS	6
#define MCHP_ACPI_EC_1_IBF_GIRQ_POS	7
#define MCHP_ACPI_EC_1_OBE_GIRQ_POS	8
#define MCHP_ACPI_EC_2_IBF_GIRQ_POS	9
#define MCHP_ACPI_EC_2_OBE_GIRQ_POS	10
#define MCHP_ACPI_EC_3_IBF_GIRQ_POS	11
#define MCHP_ACPI_EC_3_OBE_GIRQ_POS	12
#define MCHP_ACPI_EC_4_IBF_GIRQ_POS	13
#define MCHP_ACPI_EC_4_OBE_GIRQ_POS	14
#define MCHP_ACPI_PM1_CTL_GIRQ_POS	15
#define MCHP_ACPI_PM1_EN_GIRQ_POS	16
#define MCHP_ACPI_PM1_STS_GIRQ_POS	17
#define MCHP_KBC_OBE_GIRQ_POS		18
#define MCHP_KBC_IBF_GIRQ_POS		19
#define MCHP_MBOX_0_GIRQ_POS		20
#define MCHP_P80BD_0_GIRQ_POS		22

#define MCHP_UART_0_GIRQ_BIT		BIT(0)
#define MCHP_UART_1_GIRQ_BIT		BIT(1)
#define MCHP_EMI_0_GIRQ_BIT		BIT(2)
#define MCHP_EMI_1_GIRQ_BIT		BIT(3)
#define MCHP_EMI_2_GIRQ_BIT		BIT(4)
#define MCHP_ACPI_EC_0_IBF_GIRQ_BIT	BIT(5)
#define MCHP_ACPI_EC_0_OBE_GIRQ_BIT	BIT(6)
#define MCHP_ACPI_EC_1_IBF_GIRQ_BIT	BIT(7)
#define MCHP_ACPI_EC_1_OBE_GIRQ_BIT	BIT(8)
#define MCHP_ACPI_EC_2_IBF_GIRQ_BIT	BIT(9)
#define MCHP_ACPI_EC_2_OBE_GIRQ_BIT	BIT(10)
#define MCHP_ACPI_EC_3_IBF_GIRQ_BIT	BIT(11)
#define MCHP_ACPI_EC_3_OBE_GIRQ_BIT	BIT(12)
#define MCHP_ACPI_EC_4_IBF_GIRQ_BIT	BIT(13)
#define MCHP_ACPI_EC_4_OBE_GIRQ_BIT	BIT(14)
#define MCHP_ACPI_PM1_CTL_GIRQ_BIT	BIT(15)
#define MCHP_ACPI_PM1_EN_GIRQ_BIT	BIT(16)
#define MCHP_ACPI_PM1_STS_GIRQ_BIT	BIT(17)
#define MCHP_KBC_OBE_GIRQ_BIT		BIT(18)
#define MCHP_KBC_IBF_GIRQ_BIT		BIT(19)
#define MCHP_MBOX_0_GIRQ_BIT		BIT(20)
#define MCHP_P80BD_0_GIRQ_BIT		BIT(22)
/* Masks for blocks with multiple instances or sources */
#define MCHP_UART_GIRQ_MASK		0x03u
#define MCHP_UART_EMI_GIRQ_MASK		0x1cu
#define MCHP_ACPI_EC_GIRQ_MASK		0x07fe0u
#define MCHP_ACPI_PM1_GIRQ_MASK		0x38000u
#define MCHP_KBC_GIRQ_MASK		0xc0000u
#define MCHP_HOST_PERIPH_GIRQ_MASK	0x5fffffu

#define MCHP_GIRQ15_NVIC_AGGR		7u
/*
 * bitpos = MCHP_XXX_POS define
 * Example: MCHP_GIRQ15_NVIC_DIRECT(MCHP_KBC_IBF_GIRQ_POS)
 */
#define MCHP_GIRQ15_NVIC_DIRECT(bitpos)		(40u + (uint32_t)(bitpos))

/* GIRQ16 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_PK_GIRQ_POS		0
#define MCHP_RNG_GIRQ_POS		2
#define MCHP_AESH_GIRQ_POS		3

#define MCHP_PK_GIRQ_BIT		BIT(0)
#define MCHP_RNG_GIRQ_BIT		BIT(2)
#define MCHP_AESH_GIRQ_BIT		BIT(3)
#define MCHP_CRYPTO_GIRQ_MASK		0x0du

#define MCHP_GIRQ16_NVIC_AGGR		8u
#define MCHP_GIRQ16_NVIC_DIRECT(bitpos)	(65u + (uint32_t)(bitpos))

/* GIRQ17 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_PECI_GIRQ_POS		0
#define MCHP_TACH_0_GIRQ_POS		1
#define MCHP_TACH_1_GIRQ_POS		2
#define MCHP_TACH_2_GIRQ_POS		3
#define MCHP_TACH_3_GIRQ_POS		4
#define MCHP_ADC_0_SGL_GIRQ_POS		8
#define MCHP_ADC_0_RPT_GIRQ_POS		9
#define MCHP_RCID_0_GIRQ_POS		10
#define MCHP_RCID_1_GIRQ_POS		11
#define MCHP_RCID_2_GIRQ_POS		12
#define MCHP_LED_0_GIRQ_POS		13
#define MCHP_LED_1_GIRQ_POS		14
#define MCHP_LED_2_GIRQ_POS		15
#define MCHP_LED_3_GIRQ_POS		16
#define MCHP_PHOT_0_GIRQ_POS		17
#define MCHP_RPMFAN_0_FAIL_GIRQ_POS	20
#define MCHP_RPMFAN_0_STALL_GIRQ_POS	21
#define MCHP_RPMFAN_1_FAIL_GIRQ_POS	22
#define MCHP_RPMFAN_1_STALL_GIRQ_POS	23

#define MCHP_PECI_GIRQ_BIT		BIT(0)
#define MCHP_TACH_0_GIRQ_BIT		BIT(1)
#define MCHP_TACH_1_GIRQ_BIT		BIT(2)
#define MCHP_TACH_2_GIRQ_BIT		BIT(3)
#define MCHP_TACH_3_GIRQ_BIT		BIT(4)
#define MCHP_ADC_0_SGL_GIRQ_BIT		BIT(8)
#define MCHP_ADC_0_RPT_GIRQ_BIT		BIT(9)
#define MCHP_RCID_0_GIRQ_BIT		BIT(10)
#define MCHP_RCID_1_GIRQ_BIT		BIT(11)
#define MCHP_RCID_2_GIRQ_BIT		BIT(12)
#define MCHP_LED_0_GIRQ_BIT		BIT(13)
#define MCHP_LED_1_GIRQ_BIT		BIT(14)
#define MCHP_LED_2_GIRQ_BIT		BIT(15)
#define MCHP_LED_3_GIRQ_BIT		BIT(16)
#define MCHP_PHOT_0_GIRQ_BIT		BIT(17)
#define MCHP_RPMFAN_0_FAIL_GIRQ_BIT	BIT(20)
#define MCHP_RPMFAN_0_STALL_GIRQ_BIT	BIT(21)
#define MCHP_RPMFAN_1_FAIL_GIRQ_BIT	BIT(22)
#define MCHP_RPMFAN_1_STALL_GIRQ_BIT	BIT(23)
/* Masks for blocks with multiple instances or sources */
#define MCHP_TACH_GIRQ_MASK		0x1eu
#define MCHP_ADC_GIRQ_MASK		0x300u
#define MCHP_RCID_GIRQ_MASK		0x1c00u
#define MCHP_LED_GIRQ_MASK		0x1e000u
#define MCHP_RPMFAN_0_GIRQ_MASK		0x300000u
#define MCHP_RPMFAN_1_GIRQ_MASK		0xc00000u
#define MCHP_PERIPH_GROUP_1_MASK	0xf3ff1fu

#define MCHP_GIRQ17_NVIC_AGGR		9u
/* GIRQ17 direct mapping is not linear can't use parameterized macro ;<( */
#define MCHP_PECI_GIRQ_NVIC_DIRECT		70u
#define MCHP_TACH_0_GIRQ_NVIC_DIRECT		71u
#define MCHP_TACH_1_GIRQ_NVIC_DIRECT		72u
#define MCHP_TACH_2_GIRQ_NVIC_DIRECT		73u
#define MCHP_TACH_3_GIRQ_NVIC_DIRECT		159u
#define MCHP_ADC_0_SGL_GIRQ_NVIC_DIRECT		78u
#define MCHP_ADC_0_RPT_GIRQ_NVIC_DIRECT		79u
#define MCHP_RCID_0_GIRQ_NVIC_DIRECT		80u
#define MCHP_RCID_1_GIRQ_NVIC_DIRECT		81u
#define MCHP_RCID_2_GIRQ_NVIC_DIRECT		82u
#define MCHP_LED_0_GIRQ_NVIC_DIRECT		83u
#define MCHP_LED_1_GIRQ_NVIC_DIRECT		84u
#define MCHP_LED_2_GIRQ_NVIC_DIRECT		85u
#define MCHP_LED_3_GIRQ_NVIC_DIRECT		86u
#define MCHP_PHOT_0_GIRQ_NVIC_DIRECT		87u
#define MCHP_RPMFAN_0_FAIL_GIRQ_NVIC_DIRECT	74u
#define MCHP_RPMFAN_0_STALL_GIRQ_NVIC_DIRECT	75u
#define MCHP_RPMFAN_1_FAIL_GIRQ_NVIC_DIRECT	76u
#define MCHP_RPMFAN_1_STALL_GIRQ_NVIC_DIRECT	77u

/* GIRQ18 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_SPIEP_0_GIRQ_POS		0
#define MCHP_QMSPI_0_GIRQ_POS		1
#define MCHP_GPSPI_0_TXBE_GIRQ_POS	2
#define MCHP_GPSPI_0_RXBF_GIRQ_POS	3
#define MCHP_GPSPI_1_TXBE_GIRQ_POS	4
#define MCHP_GPSPI_1_RXBF_GIRQ_POS	5
#define MCHP_BCL_0_BCLR_GIRQ_POS	6
#define MCHP_BCL_0_BERR_GIRQ_POS	7
#define MCHP_PS2_0_ACT_GIRQ_POS		10
#define MCHP_EERPROMC_0_GIRQ_POS	13
#define MCHP_CCT_0_CNT_GIRQ_POS		20
#define MCHP_CCT_0_CAP0_GIRQ_POS	21
#define MCHP_CCT_0_CAP1_GIRQ_POS	22
#define MCHP_CCT_0_CAP2_GIRQ_POS	23
#define MCHP_CCT_0_CAP3_GIRQ_POS	24
#define MCHP_CCT_0_CAP4_GIRQ_POS	25
#define MCHP_CCT_0_CAP5_GIRQ_POS	26
#define MCHP_CCT_0_CMP0_GIRQ_POS	27
#define MCHP_CCT_0_CMP1_GIRQ_POS	28

#define MCHP_SPIEP_0_GIRQ_BIT		BIT(0)
#define MCHP_QMSPI_0_GIRQ_BIT		BIT(1)
#define MCHP_GPSPI_0_TXBE_GIRQ_BIT	BIT(2)
#define MCHP_GPSPI_0_RXBF_GIRQ_BIT	BIT(3)
#define MCHP_GPSPI_1_TXBE_GIRQ_BIT	BIT(4)
#define MCHP_GPSPI_1_RXBF_GIRQ_BIT	BIT(5)
#define MCHP_BCL_0_BCLR_GIRQ_BIT	BIT(6)
#define MCHP_BCL_0_BERR_GIRQ_BIT	BIT(7)
#define MCHP_PS2_0_ACT_GIRQ_BIT		BIT(10)
#define MCHP_EERPROMC_0_GIRQ_BIT	BIT(13)
#define MCHP_CCT_0_CNT_GIRQ_BIT		BIT(20)
#define MCHP_CCT_0_CAP0_GIRQ_BIT	BIT(21)
#define MCHP_CCT_0_CAP1_GIRQ_BIT	BIT(22)
#define MCHP_CCT_0_CAP2_GIRQ_BIT	BIT(23)
#define MCHP_CCT_0_CAP3_GIRQ_BIT	BIT(24)
#define MCHP_CCT_0_CAP4_GIRQ_BIT	BIT(25)
#define MCHP_CCT_0_CAP5_GIRQ_BIT	BIT(26)
#define MCHP_CCT_0_CMP0_GIRQ_BIT	BIT(27)
#define MCHP_CCT_0_CMP1_GIRQ_BIT	BIT(28)
/* Masks for blocks with multiple instances or sources */
#define MCHP_GPSPI_0_GIRQ_MASK		0x0cu
#define MCHP_GPSPI_1_GIRQ_MASK		0x30u
#define MCHP_BCL_0_GIRQ_MASK		0xc0u
#define MCHP_CCT_0_GIRQ_MASK		0x1ff00000u
#define MCHP_PERIPH_GROUP_2_MASK	0x1ff024ffu

#define MCHP_GIRQ18_NVIC_AGGR		10u
/* GIRQ18 direct mapping is not linear can't use parameterized macro ;<( */
#define MCHP_SPIEP_0_GIRQ_NVIC_DIRECT		90u
#define MCHP_QMSPI_0_GIRQ_NVIC_DIRECT		91u
#define MCHP_GPSPI_0_TXBE_GIRQ_NVIC_DIRECT	92u
#define MCHP_GPSPI_0_RXBF_GIRQ_NVIC_DIRECT	93u
#define MCHP_GPSPI_1_TXBE_GIRQ_NVIC_DIRECT	94u
#define MCHP_GPSPI_1_RXBF_GIRQ_NVIC_DIRECT	95u
#define MCHP_BCL_0_BCLR_GIRQ_NVIC_DIRECT	97u
#define MCHP_BCL_0_BERR_GIRQ_NVIC_DIRECT	96u
#define MCHP_PS2_0_ACT_GIRQ_NVIC_DIRECT		100u
#define MCHP_EERPROMC_0_GIRQ_NVIC_DIRECT	155u
#define MCHP_CCT_0_CNT_GIRQ_NVIC_DIRECT		146u
#define MCHP_CCT_0_CAP0_GIRQ_NVIC_DIRECT	147u
#define MCHP_CCT_0_CAP1_GIRQ_NVIC_DIRECT	148u
#define MCHP_CCT_0_CAP2_GIRQ_NVIC_DIRECT	149u
#define MCHP_CCT_0_CAP3_GIRQ_NVIC_DIRECT	150u
#define MCHP_CCT_0_CAP4_GIRQ_NVIC_DIRECT	151u
#define MCHP_CCT_0_CAP5_GIRQ_NVIC_DIRECT	152u
#define MCHP_CCT_0_CMP0_GIRQ_NVIC_DIRECT	153u
#define MCHP_CCT_0_CMP1_GIRQ_NVIC_DIRECT	154u

/* GIRQ19 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_ESPI_PC_GIRQ_POS		0
#define MCHP_ESPI_BM1_GIRQ_POS		1
#define MCHP_ESPI_BM2_GIRQ_POS		2
#define MCHP_ESPI_LTR_GIRQ_POS		3
#define MCHP_ESPI_OOB_UP_GIRQ_POS	4
#define MCHP_ESPI_OOB_DN_GIRQ_POS	5
#define MCHP_ESPI_FC_GIRQ_POS		6
#define MCHP_ESPI_RESET_GIRQ_POS	7
#define MCHP_ESPI_VW_EN_GIRQ_POS	8
#define MCHP_ESPI_SAF_DONE_GIRQ_POS	9
#define MCHP_ESPI_SAF_ERR_GIRQ_POS	10
#define MCHP_ESPI_SAF_CACHE_GIRQ_POS	11

#define MCHP_ESPI_PC_GIRQ_BIT		BIT(0)
#define MCHP_ESPI_BM1_GIRQ_BIT		BIT(1)
#define MCHP_ESPI_BM2_GIRQ_BIT		BIT(2)
#define MCHP_ESPI_LTR_GIRQ_BIT		BIT(3)
#define MCHP_ESPI_OOB_UP_GIRQ_BIT	BIT(4)
#define MCHP_ESPI_OOB_DN_GIRQ_BIT	BIT(5)
#define MCHP_ESPI_FC_GIRQ_BIT		BIT(6)
#define MCHP_ESPI_RESET_GIRQ_BIT	BIT(7)
#define MCHP_ESPI_VWEN_GIRQ_BIT		BIT(8)
#define MCHP_ESPI_SAF_DONE_GIRQ_BIT	BIT(9)
#define MCHP_ESPI_SAF_ERR_GIRQ_BIT	BIT(10)
#define MCHP_ESPI_SAF_CACHE_GIRQ_BIT	BIT(11)
/* Masks for blocks with multiple instances or sources */
#define MCHP_ESPI_BM_GIRQ_MASK		0x006u
#define MCHP_ESPI_OOB_GIRQ_MASK		0x030u
#define MCHP_ESPI_SAF_GIRQ_MASK		0xe00u
#define MCHP_ESPI_GIRQ_MASK		0xfffu

#define MCHP_GIRQ19_NVIC_AGGR			11u
#define MCHP_ESPI_PC_GIRQ_NVIC_DIRECT		103u
#define MCHP_ESPI_BM1_GIRQ_NVIC_DIRECT		104u
#define MCHP_ESPI_BM2_GIRQ_NVIC_DIRECT		105u
#define MCHP_ESPI_LTR_GIRQ_NVIC_DIRECT		106u
#define MCHP_ESPI_OOB_UP_GIRQ_NVIC_DIRECT	107u
#define MCHP_ESPI_OOB_DN_GIRQ_NVIC_DIRECT	108u
#define MCHP_ESPI_FC_GIRQ_NVIC_DIRECT		109u
#define MCHP_ESPI_RESET_GIRQ_NVIC_DIRECT	110u
#define MCHP_ESPI_VWEN_GIRQ_NVIC_DIRECT		156u
#define MCHP_ESPI_SAF_DONE_GIRQ_NVIC_DIRECT	166u
#define MCHP_ESPI_SAF_ERR_GIRQ_NVIC_DIRECT	167u
#define MCHP_ESPI_SAF_CACHE_GIRQ_NVIC_DIRECT	169u

/* GIRQ20 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_STAP_OBF_GIRQ_POS		0
#define MCHP_STAP_IBF_GIRQ_POS		1
#define MCHP_STAP_WAKE_GIRQ_POS		2
#define MCHP_OTP_READY_GIRQ_POS		3
#define MCHP_ISPI_ERR_GIRQ_POS		8
#define MCHP_CLK32_MON_GIRQ_POS		9

#define MCHP_STAP_OBF_GIRQ_BIT		BIT(0)
#define MCHP_STAP_IBF_GIRQ_BIT		BIT(1)
#define MCHP_STAP_WAKE_GIRQ_BIT		BIT(2)
#define MCHP_OTP_READY_GIRQ_BIT		BIT(3)
#define MCHP_ISPI_ERR_GIRQ_BIT		BIT(8)
#define MCHP_CLK32_MON_GIRQ_BIT		BIT(9)
/* Masks for blocks with multiple instances or sources */
#define MCHP_STAP_GIRQ_MASK		0x007u
#define MCHP_PERIPH_GROUP_3_MASK	0x30fu

#define MCHP_GIRQ20_NVIC_AGGR		12u
/*
 * Warning: STAP has no direct connections. If STAP and other source are
 * enabled GIRQ20 must be used in aggregated mode.
 */
#define MCHP_OTP_READY_GIRQ_NVIC_DIRECT		173u
#define MCHP_CLK32_MON_GIRQ_NVIC_DIRECT		174u

/* GIRQ21 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_WDT_GIRQ_POS		2
#define MCHP_WTMR_ALARM_GIRQ_POS	3
#define MCHP_WTMR_SUBWK_GIRQ_POS	4
#define MCHP_WTMR_ONESEC_GIRQ_POS	5
#define MCHP_WTMR_SUBSEC_GIRQ_POS	6
#define MCHP_WTMR_SPP_GIRQ_POS		7
#define MCHP_RTC_GIRQ_POS		8
#define MCHP_RTC_ALARM_GIRQ_POS		9
#define MCHP_VCI_OVRD_IN_GIRQ_POS	10
#define MCHP_VCI_IN0_GIRQ_POS		11
#define MCHP_VCI_IN1_GIRQ_POS		12
#define MCHP_VCI_IN2_GIRQ_POS		13
#define MCHP_VCI_IN3_GIRQ_POS		14
#define MCHP_VCI_IN4_GIRQ_POS		15
#define MCHP_PS2_0_PORT0A_WK_GIRQ_POS	18
#define MCHP_PS2_0_PORT0B_WK_GIRQ_POS	19
#define MCHP_KEYSCAN_GIRQ_POS		25
#define MCHP_GLUE_GIRQ_POS		26

#define MCHP_WDT_GIRQ_BIT		BIT(2)
#define MCHP_WTMR_ALARM_GIRQ_BIT	BIT(3)
#define MCHP_WTMR_SUBWK_GIRQ_BIT	BIT(4)
#define MCHP_WTMR_ONESEC_GIRQ_BIT	BIT(5)
#define MCHP_WTMR_SUBSEC_GIRQ_BIT	BIT(6)
#define MCHP_WTMR_SPP_GIRQ_BIT		BIT(7)
#define MCHP_RTC_GIRQ_BIT		BIT(8)
#define MCHP_RTC_ALARM_GIRQ_BIT		BIT(9)
#define MCHP_VCI_OVRD_IN_GIRQ_BIT	BIT(10)
#define MCHP_VCI_IN0_GIRQ_BIT		BIT(11)
#define MCHP_VCI_IN1_GIRQ_BIT		BIT(12)
#define MCHP_VCI_IN2_GIRQ_BIT		BIT(13)
#define MCHP_VCI_IN3_GIRQ_BIT		BIT(14)
#define MCHP_VCI_IN4_GIRQ_BIT		BIT(15)
#define MCHP_PS2_0_PORT0A_WK_GIRQ_BIT	BIT(18)
#define MCHP_PS2_0_PORT0B_WK_GIRQ_BIT	BIT(19)
#define MCHP_KEYSCAN_GIRQ_BIT		BIT(25)
#define MCHP_GLUE_GIRQ_BIT		BIT(26)
/* Masks for blocks with multiple instances or sources */
#define MCHP_WTMR_GIRQ_MASK		0xf8u
#define MCHP_RTC_GIRQ_MASK		0x300u
#define MCHP_VCI_GIRQ_MASK		0xfc00u
#define MCHP_PS2_PORT_WK_GIRQ_MASK	0xc0000u
#define MCHP_PERIPH_GROUP_4_MASK	0x60cfffcu

#define MCHP_GIRQ21_NVIC_AGGR			13u
#define MCHP_WDT_GIRQ_NVIC_DIRECT		171u
#define MCHP_WTMR_ALARM_GIRQ_NVIC_DIRECT	114u
#define MCHP_WTMR_SUBWK_GIRQ_NVIC_DIRECT	115u
#define MCHP_WTMR_ONESEC_GIRQ_NVIC_DIRECT	116u
#define MCHP_WTMR_SUBSEC_GIRQ_NVIC_DIRECT	117u
#define MCHP_WTMR_SPP_GIRQ_NVIC_DIRECT		118u
#define MCHP_RTC_GIRQ_NVIC_DIRECT		119u
#define MCHP_RTC_ALARM_GIRQ_NVIC_DIRECT		120u
#define MCHP_VCI_OVRD_IN_GIRQ_NVIC_DIRECT	121u
#define MCHP_VCI_IN0_GIRQ_NVIC_DIRECT		122u
#define MCHP_VCI_IN1_GIRQ_NVIC_DIRECT		123u
#define MCHP_VCI_IN2_GIRQ_NVIC_DIRECT		124u
#define MCHP_VCI_IN3_GIRQ_NVIC_DIRECT		125u
#define MCHP_VCI_IN4_GIRQ_NVIC_DIRECT		126u
#define MCHP_PS2_0_PORT0A_WK_GIRQ_NVIC_DIRECT	129u
#define MCHP_PS2_0_PORT0B_WK_GIRQ_NVIC_DIRECT	130u
#define MCHP_KEYSCAN_GIRQ_NVIC_DIRECT		135u
#define MCHP_GLUE_GIRQ_NVIC_DIRECT		172u

/*
 * GIRQ22 Source, Enable_Set/Clr, Result registers bit positions
 * NOTE: These wake sources allow the peripheral to turn back on clocks
 * long enough to facilitate the data transfer. No interrupt to the EC occurs
 * unless the peripheral was configured to generate an EC interrupt for
 * the specific data transfer.
 */
#define MCHP_SPIEP_WK_CLK_GIRQ_POS	0
#define MCHP_I2C_SMB_0_WK_CLK_GIRQ_POS	1
#define MCHP_I2C_SMB_1_WK_CLK_GIRQ_POS	2
#define MCHP_I2C_SMB_2_WK_CLK_GIRQ_POS	3
#define MCHP_I2C_SMB_3_WK_CLK_GIRQ_POS	4
#define MCHP_I2C_SMB_4_WK_CLK_GIRQ_POS	5
#define MCHP_ESPI_WK_CLK_GIRQ_POS	9
#define MCHP_STAP_WK_CLK_GIRQ_POS	15

#define MCHP_SPIEP_WK_CLK_GIRQ_BIT	BIT(0)
#define MCHP_I2C_SMB_0_WK_CLK_GIRQ_BIT	BIT(1)
#define MCHP_I2C_SMB_1_WK_CLK_GIRQ_BIT	BIT(2)
#define MCHP_I2C_SMB_2_WK_CLK_GIRQ_BIT	BIT(3)
#define MCHP_I2C_SMB_3_WK_CLK_GIRQ_BIT	BIT(4)
#define MCHP_I2C_SMB_4_WK_CLK_GIRQ_BIT	BIT(5)
#define MCHP_ESPI_WK_CLK_GIRQ_BIT	BIT(9)
#define MCHP_STAP_WK_CLK_GIRQ_BIT	BIT(15)
/* Masks for blocks with multiple instances or sources */
#define MCHP_I2C_SMB_WK_CLK_GIRQ_MASK	0x3eu
#define MCHP_CLK_WK_CLK_GIRQ_MASK	0x823fu

/* GIRQ23 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_BTMR16_0_GIRQ_POS		0
#define MCHP_BTMR16_1_GIRQ_POS		1
#define MCHP_BTMR16_2_GIRQ_POS		2
#define MCHP_BTMR16_3_GIRQ_POS		3
#define MCHP_BTMR32_0_GIRQ_POS		4
#define MCHP_BTMR32_1_GIRQ_POS		5
#define MCHP_CTMR_0_GIRQ_POS		6
#define MCHP_CTMR_1_GIRQ_POS		7
#define MCHP_CTMR_2_GIRQ_POS		8
#define MCHP_CTMR_3_GIRQ_POS		9
#define MCHP_RTMR_0_GIRQ_POS		10
#define MCHP_RTMR_0_SWI0_GIRQ_POS	11
#define MCHP_RTMR_0_SWI1_GIRQ_POS	12
#define MCHP_RTMR_0_SWI2_GIRQ_POS	13
#define MCHP_RTMR_0_SWI3_GIRQ_POS	14
#define MCHP_HTMR_0_GIRQ_POS		16
#define MCHP_HTMR_1_GIRQ_POS		17

#define MCHP_BTMR16_0_GIRQ_BIT		BIT(0)
#define MCHP_BTMR16_1_GIRQ_BIT		BIT(1)
#define MCHP_BTMR16_2_GIRQ_BIT		BIT(2)
#define MCHP_BTMR16_3_GIRQ_BIT		BIT(3)
#define MCHP_BTMR32_0_GIRQ_BIT		BIT(4)
#define MCHP_BTMR32_1_GIRQ_BIT		BIT(5)
#define MCHP_CTMR_0_GIRQ_BIT		BIT(6)
#define MCHP_CTMR_1_GIRQ_BIT		BIT(7)
#define MCHP_CTMR_2_GIRQ_BIT		BIT(8)
#define MCHP_CTMR_3_GIRQ_BIT		BIT(9)
#define MCHP_RTMR_0_GIRQ_BIT		BIT(10)
#define MCHP_RTMR_0_SWI0_GIRQ_BIT	BIT(11)
#define MCHP_RTMR_0_SWI1_GIRQ_BIT	BIT(12)
#define MCHP_RTMR_0_SWI2_GIRQ_BIT	BIT(13)
#define MCHP_RTMR_0_SWI3_GIRQ_BIT	BIT(14)
#define MCHP_HTMR_0_GIRQ_BIT		BIT(16)
#define MCHP_HTMR_1_GIRQ_BIT		BIT(17)
/* Masks for blocks with multiple instances or sources */
#define MCHP_BTMR16_GIRQ_MASK		0x0fu
#define MCHP_BTMR32_GIRQ_MASK		0x30u
#define MCHP_CTMR_GIRQ_MASK		0x3c0u
#define MCHP_RMTR_GIRQ_MASK		0x7c00u
#define MCHP_HTMR_GIRQ_MASK		0x30000u
#define MCHP_PERIPH_GROUP_5_GIRQ_MASK	0x37fffu

#define MCHP_GIRQ23_NVIC_AGGR			14u
/*
 * Warning: RTMR SWI interrupts have no direct connection.
 * If RTMR SWI will be used GIRQ23 must be aggregated.
 */
#define MCHP_BTMR16_0_GIRQ_NVIC_DIRECT		136u
#define MCHP_BTMR16_1_GIRQ_NVIC_DIRECT		137u
#define MCHP_BTMR16_2_GIRQ_NVIC_DIRECT		138u
#define MCHP_BTMR16_3_GIRQ_NVIC_DIRECT		139u
#define MCHP_BTMR32_0_GIRQ_NVIC_DIRECT		140u
#define MCHP_BTMR32_1_GIRQ_NVIC_DIRECT		141u
#define MCHP_CTMR_0_GIRQ_NVIC_DIRECT		142u
#define MCHP_CTMR_1_GIRQ_NVIC_DIRECT		143u
#define MCHP_CTMR_2_GIRQ_NVIC_DIRECT		144u
#define MCHP_CTMR_3_GIRQ_NVIC_DIRECT		145u
#define MCHP_RTMR_0_GIRQ_NVIC_DIRECT		111u
#define MCHP_HTMR_0_GIRQ_NVIC_DIRECT		112u
#define MCHP_HTMR_1_GIRQ_NVIC_DIRECT		113u

/* GIRQ24 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_MSVW00_SRC0_GIRQ_POS	0
#define MCHP_MSVW00_SRC1_GIRQ_POS	1
#define MCHP_MSVW00_SRC2_GIRQ_POS	2
#define MCHP_MSVW00_SRC3_GIRQ_POS	3
#define MCHP_MSVW01_SRC0_GIRQ_POS	4
#define MCHP_MSVW01_SRC1_GIRQ_POS	5
#define MCHP_MSVW01_SRC2_GIRQ_POS	6
#define MCHP_MSVW01_SRC3_GIRQ_POS	7
#define MCHP_MSVW02_SRC0_GIRQ_POS	8
#define MCHP_MSVW02_SRC1_GIRQ_POS	9
#define MCHP_MSVW02_SRC2_GIRQ_POS	10
#define MCHP_MSVW02_SRC3_GIRQ_POS	11
#define MCHP_MSVW03_SRC0_GIRQ_POS	12
#define MCHP_MSVW03_SRC1_GIRQ_POS	13
#define MCHP_MSVW03_SRC2_GIRQ_POS	14
#define MCHP_MSVW03_SRC3_GIRQ_POS	15
#define MCHP_MSVW04_SRC0_GIRQ_POS	16
#define MCHP_MSVW04_SRC1_GIRQ_POS	17
#define MCHP_MSVW04_SRC2_GIRQ_POS	18
#define MCHP_MSVW04_SRC3_GIRQ_POS	19
#define MCHP_MSVW05_SRC0_GIRQ_POS	20
#define MCHP_MSVW05_SRC1_GIRQ_POS	21
#define MCHP_MSVW05_SRC2_GIRQ_POS	22
#define MCHP_MSVW05_SRC3_GIRQ_POS	23
#define MCHP_MSVW06_SRC0_GIRQ_POS	24
#define MCHP_MSVW06_SRC1_GIRQ_POS	25
#define MCHP_MSVW06_SRC2_GIRQ_POS	26
#define MCHP_MSVW06_SRC3_GIRQ_POS	27

#define MCHP_MSVW00_SRC0_GIRQ_BIT	BIT(0)
#define MCHP_MSVW00_SRC1_GIRQ_BIT	BIT(1)
#define MCHP_MSVW00_SRC2_GIRQ_BIT	BIT(2)
#define MCHP_MSVW00_SRC3_GIRQ_BIT	BIT(3)
#define MCHP_MSVW01_SRC0_GIRQ_BIT	BIT(4)
#define MCHP_MSVW01_SRC1_GIRQ_BIT	BIT(5)
#define MCHP_MSVW01_SRC2_GIRQ_BIT	BIT(6)
#define MCHP_MSVW01_SRC3_GIRQ_BIT	BIT(7)
#define MCHP_MSVW02_SRC0_GIRQ_BIT	BIT(8)
#define MCHP_MSVW02_SRC1_GIRQ_BIT	BIT(9)
#define MCHP_MSVW02_SRC2_GIRQ_BIT	BIT(10)
#define MCHP_MSVW02_SRC3_GIRQ_BIT	BIT(11)
#define MCHP_MSVW03_SRC0_GIRQ_BIT	BIT(12)
#define MCHP_MSVW03_SRC1_GIRQ_BIT	BIT(13)
#define MCHP_MSVW03_SRC2_GIRQ_BIT	BIT(14)
#define MCHP_MSVW03_SRC3_GIRQ_BIT	BIT(15)
#define MCHP_MSVW04_SRC0_GIRQ_BIT	BIT(16)
#define MCHP_MSVW04_SRC1_GIRQ_BIT	BIT(17)
#define MCHP_MSVW04_SRC2_GIRQ_BIT	BIT(18)
#define MCHP_MSVW04_SRC3_GIRQ_BIT	BIT(19)
#define MCHP_MSVW05_SRC0_GIRQ_BIT	BIT(20)
#define MCHP_MSVW05_SRC1_GIRQ_BIT	BIT(21)
#define MCHP_MSVW05_SRC2_GIRQ_BIT	BIT(22)
#define MCHP_MSVW05_SRC3_GIRQ_BIT	BIT(23)
#define MCHP_MSVW06_SRC0_GIRQ_BIT	BIT(24)
#define MCHP_MSVW06_SRC1_GIRQ_BIT	BIT(25)
#define MCHP_MSVW06_SRC2_GIRQ_BIT	BIT(26)
#define MCHP_MSVW06_SRC3_GIRQ_BIT	BIT(27)
/* Masks for blocks with multiple instances or sources */
#define MCHP_MSVW00_GIRQ_MASK		0xfu
#define MCHP_MSVW01_GIRQ_MASK		0xf0u
#define MCHP_MSVW02_GIRQ_MASK		0xf00u
#define MCHP_MSVW03_GIRQ_MASK		0xf000u
#define MCHP_MSVW04_GIRQ_MASK		0xf0000u
#define MCHP_MSVW05_GIRQ_MASK		0xf00000u
#define MCHP_MSVW06_GIRQ_MASK		0xf000000u
#define MCHP_MSVW00_06_GIRQ_MASK	0x0fffffffu

#define MCHP_GIRQ24_NVIC_AGGR		15u

/* GIRQ25 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_MSVW07_SRC0_GIRQ_POS	0
#define MCHP_MSVW07_SRC1_GIRQ_POS	1
#define MCHP_MSVW07_SRC2_GIRQ_POS	2
#define MCHP_MSVW07_SRC3_GIRQ_POS	3
#define MCHP_MSVW08_SRC0_GIRQ_POS	4
#define MCHP_MSVW08_SRC1_GIRQ_POS	5
#define MCHP_MSVW08_SRC2_GIRQ_POS	6
#define MCHP_MSVW08_SRC3_GIRQ_POS	7
#define MCHP_MSVW09_SRC0_GIRQ_POS	8
#define MCHP_MSVW09_SRC1_GIRQ_POS	9
#define MCHP_MSVW09_SRC2_GIRQ_POS	10
#define MCHP_MSVW09_SRC3_GIRQ_POS	11
#define MCHP_MSVW10_SRC0_GIRQ_POS	12
#define MCHP_MSVW10_SRC1_GIRQ_POS	13
#define MCHP_MSVW10_SRC2_GIRQ_POS	14
#define MCHP_MSVW10_SRC3_GIRQ_POS	15

#define MCHP_MSVW07_SRC0_GIRQ_BIT	BIT(0)
#define MCHP_MSVW07_SRC1_GIRQ_BIT	BIT(1)
#define MCHP_MSVW07_SRC2_GIRQ_BIT	BIT(2)
#define MCHP_MSVW07_SRC3_GIRQ_BIT	BIT(3)
#define MCHP_MSVW08_SRC0_GIRQ_BIT	BIT(4)
#define MCHP_MSVW08_SRC1_GIRQ_BIT	BIT(5)
#define MCHP_MSVW08_SRC2_GIRQ_BIT	BIT(6)
#define MCHP_MSVW08_SRC3_GIRQ_BIT	BIT(7)
#define MCHP_MSVW09_SRC0_GIRQ_BIT	BIT(8)
#define MCHP_MSVW09_SRC1_GIRQ_BIT	BIT(9)
#define MCHP_MSVW09_SRC2_GIRQ_BIT	BIT(10)
#define MCHP_MSVW09_SRC3_GIRQ_BIT	BIT(11)
#define MCHP_MSVW10_SRC0_GIRQ_BIT	BIT(12)
#define MCHP_MSVW10_SRC1_GIRQ_BIT	BIT(13)
#define MCHP_MSVW10_SRC2_GIRQ_BIT	BIT(14)
#define MCHP_MSVW10_SRC3_GIRQ_BIT	BIT(15)
/* Masks for blocks with multiple instances or sources */
#define MCHP_MSVW07_GIRQ_MASK		0xfu
#define MCHP_MSVW08_GIRQ_MASK		0xf0u
#define MCHP_MSVW09_GIRQ_MASK		0xf00u
#define MCHP_MSVW10_GIRQ_MASK		0xf000u
#define MCHP_MSVW07_10_GIRQ_MASK	0xffffu

#define MCHP_GIRQ25_NVIC_AGGR		16u

#define MCHP_MSVW00_GIRQ		24
#define MCHP_MSVW01_GIRQ		24
#define MCHP_MSVW02_GIRQ		24
#define MCHP_MSVW03_GIRQ		24
#define MCHP_MSVW04_GIRQ		24
#define MCHP_MSVW05_GIRQ		24
#define MCHP_MSVW06_GIRQ		24
#define MCHP_MSVW07_GIRQ		25
#define MCHP_MSVW08_GIRQ		25
#define MCHP_MSVW09_GIRQ		25
#define MCHP_MSVW10_GIRQ		25

/* GIRQ26 Source, Enable_Set/Clr, Result registers bit positions */
#define MCHP_GPIO_0240_GIRQ_POS		0
#define MCHP_GPIO_0241_GIRQ_POS		1
#define MCHP_GPIO_0242_GIRQ_POS		2
#define MCHP_GPIO_0243_GIRQ_POS		3
#define MCHP_GPIO_0244_GIRQ_POS		4
#define MCHP_GPIO_0245_GIRQ_POS		5
#define MCHP_GPIO_0246_GIRQ_POS		6
#define MCHP_GPIO_0254_GIRQ_POS		12
#define MCHP_GPIO_0255_GIRQ_POS		13

#define MCHP_GPIO_0240_GIRQ_BIT		BIT(0)
#define MCHP_GPIO_0241_GIRQ_BIT		BIT(1)
#define MCHP_GPIO_0242_GIRQ_BIT		BIT(2)
#define MCHP_GPIO_0243_GIRQ_BIT		BIT(3)
#define MCHP_GPIO_0244_GIRQ_BIT		BIT(4)
#define MCHP_GPIO_0245_GIRQ_BIT		BIT(5)
#define MCHP_GPIO_0246_GIRQ_BIT		BIT(6)
#define MCHP_GPIO_0254_GIRQ_BIT		BIT(12)
#define MCHP_GPIO_0255_GIRQ_BIT		BIT(13)
/* Masks for blocks with multiple instances or sources */
#define MCHP_GPIO_0240_0276_GIRQ_MASK	0x307fu

#define MCHP_GIRQ26_NVIC_AGGR		17u

/* GIRQ numbering */
#define MCHP_GIRQ_START_NUM	8u
#define MCHP_GIRQ_LAST_NUM	26u
#define MCHP_GIRQ_IDX(girq)	((uint32_t)(girq) - 8u)
#define MCHP_GIRQ_IDX_FIRST	0u
#define MCHP_GIRQ_IDX_MAX	19u
/* Number of NVIC Enable_Set/Clr, Pending_Set/Clr, Active 32-bit registers */
#define MCHP_MAX_NVIC_IDX	6u
/* Number of external NVIC inputs */
#define MCHP_MAX_NVIC_EXT_INPUTS 181u

/** @brief GIRQ registers. Total size = 20(0x14) bytes */
struct girq_regs {
	volatile uint32_t SRC;
	volatile uint32_t EN_SET;
	volatile uint32_t RESULT;
	volatile uint32_t EN_CLR;
	volatile uint32_t RSVD1;
};

/** @brief ECIA registers with each GIRQ elucidated */
struct ecia_named_regs {
	struct girq_regs GIRQ08;
	struct girq_regs GIRQ09;
	struct girq_regs GIRQ10;
	struct girq_regs GIRQ11;
	struct girq_regs GIRQ12;
	struct girq_regs GIRQ13;
	struct girq_regs GIRQ14;
	struct girq_regs GIRQ15;
	struct girq_regs GIRQ16;
	struct girq_regs GIRQ17;
	struct girq_regs GIRQ18;
	struct girq_regs GIRQ19;
	struct girq_regs GIRQ20;
	struct girq_regs GIRQ21;
	struct girq_regs GIRQ22;
	struct girq_regs GIRQ23;
	struct girq_regs GIRQ24;
	struct girq_regs GIRQ25;
	struct girq_regs GIRQ26;
	uint8_t RSVD2[(0x0200 - 0x017c)];
	volatile uint32_t BLK_EN_SET;
	volatile uint32_t BLK_EN_CLR;
	volatile uint32_t BLK_ACTIVE;
};

/** @brief ECIA registers with array of GIRQ's */
struct ecia_regs {
	struct girq_regs GIRQ[19];
	uint8_t RSVD2[(0x200 - 0x17c)];
	volatile uint32_t BLK_EN_SET;
	volatile uint32_t BLK_EN_CLR;
	volatile uint32_t BLK_ACTIVE;
};

/* Until XEC ECIA driver is available we define these locally */
static inline void mchp_soc_ecia_girq_aggr_en(uint8_t girq, uint8_t en)
{
	if ((girq < MCHP_FIRST_GIRQ_NOS) || (girq > MCHP_LAST_GIRQ_NOS)) {
		return;
	}

	struct ecia_regs *ecia = (struct ecia_regs *)(ECIA_BASE_ADDR);

	if (en) {
		ecia->BLK_EN_SET = BIT(girq);
	} else {
		ecia->BLK_EN_CLR = BIT(girq);
	}
}

static inline void mchp_soc_ecia_girq_src_clr(uint8_t girq, uint8_t pin)
{
	if ((girq < MCHP_FIRST_GIRQ_NOS) || (girq > MCHP_LAST_GIRQ_NOS) ||
	    (pin > 31)) {
		return;
	}

	struct ecia_regs *ecia = (struct ecia_regs *)(ECIA_BASE_ADDR);

	ecia->GIRQ[girq - 8u].SRC = BIT(pin);
}

static inline void mchp_soc_ecia_girq_src_clr_bitmap(uint8_t girq,
						     uint32_t bitmap)
{
	if ((girq < MCHP_FIRST_GIRQ_NOS) || (girq > MCHP_LAST_GIRQ_NOS)) {
		return;
	}

	struct ecia_regs *ecia = (struct ecia_regs *)(ECIA_BASE_ADDR);

	ecia->GIRQ[girq - 8u].SRC = bitmap;
}

static inline void mchp_soc_ecia_girq_src_dis(uint8_t girq, uint8_t pin)
{
	if ((girq < MCHP_FIRST_GIRQ_NOS) || (girq > MCHP_LAST_GIRQ_NOS) ||
	    (pin > 31)) {
		return;
	}

	struct ecia_regs *ecia = (struct ecia_regs *)(ECIA_BASE_ADDR);

	ecia->GIRQ[girq - 8u].EN_CLR = BIT(pin);
}

static inline void mchp_soc_ecia_girq_src_en(uint8_t girq, uint8_t pin)
{
	if ((girq < MCHP_FIRST_GIRQ_NOS) || (girq > MCHP_LAST_GIRQ_NOS) ||
	    (pin > 31)) {
		return;
	}

	struct ecia_regs *ecia = (struct ecia_regs *)(ECIA_BASE_ADDR);

	ecia->GIRQ[girq - 8u].EN_SET = BIT(pin);
}

static inline uint32_t mchp_soc_ecia_girq_result(uint8_t girq)
{
	if ((girq < MCHP_FIRST_GIRQ_NOS) || (girq > MCHP_LAST_GIRQ_NOS)) {
		return 0u;
	}

	struct ecia_regs *ecia = (struct ecia_regs *)(ECIA_BASE_ADDR);

	return ecia->GIRQ[girq - 8u].RESULT;
}

#endif /* #ifndef _MEC172X_ECIA_H */
