/*
 * Copyright (c) 2025 Microchip Technology Inc.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @file mchp_sam_d5x_e5x_clock.h
 * @brief List clock subsystem IDs for sam_d5x_e5x family.
 *
 * Clock subsystem IDs. To be used in devicetree nodes, and as argument for clock API.
 */

#ifndef INCLUDE_ZEPHYR_DT_BINDINGS_CLOCK_MCHP_SAM_D5X_E5X_CLOCK_H_
#define INCLUDE_ZEPHYR_DT_BINDINGS_CLOCK_MCHP_SAM_D5X_E5X_CLOCK_H_

/**
 * Encode clock type, mclk bus, mclk mask bit, gclk pch and instance number,
 * to clock subsystem.
 *
 * - 00..07 (8 bits): inst
 *
 * - 08..13 (6 bits): gclkperiph
 * (values from 0 to 47)
 *
 * - 14..19 (6 bits): mclkmaskbit
 * (values from 0 to 31)
 *
 * - 20..25 (6 bits): mclkbus
 * following values
 * MBUS_AHB  (0)
 * MBUS_APBA (1)
 * MBUS_APBB (2)
 * MBUS_APBC (3)
 * MBUS_APBD (4)
 *
 * - 26..31 (6 bits): type
 * following values
 * SUBSYS_TYPE_XOSC       (0)
 * SUBSYS_TYPE_DFLL       (1)
 * SUBSYS_TYPE_FDPLL      (2)
 * SUBSYS_TYPE_RTC        (3)
 * SUBSYS_TYPE_XOSC32K    (4)
 * SUBSYS_TYPE_GCLKGEN    (5)
 * SUBSYS_TYPE_GCLKPERIPH (6)
 * SUBSYS_TYPE_MCLKCPU    (7)
 * SUBSYS_TYPE_MCLKPERIPH (8)
 *
 * @param type clock subsystem type
 * @param mclkbus select from the AHBx and the APBx buses
 * @param mclkmaskbit select the module connected to AHBx or APBx bus (0 to 31)
 * @param gclkperiph gclk peripheral channel number m in PCHTRLm (0 to 47)
 * @param inst instance number of the specified clock type
 */
#define MCHP_CLOCK_DERIVE_ID(type, mclkbus, mclkmaskbit, gclkperiph, inst)                         \
	(((type) << 26) | ((mclkbus) << 20) | ((mclkmaskbit) << 14) | ((gclkperiph) << 8) | inst)

/* XOSC_TYPE ids */
#define CLOCK_MCHP_XOSC_ID_XOSC0 MCHP_CLOCK_DERIVE_ID(0, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_XOSC_ID_XOSC1 MCHP_CLOCK_DERIVE_ID(0, 0x3f, 0x3f, 0x3f, 1)
#define CLOCK_MCHP_XOSC_ID_MAX   (1)

/* DFLL_TYPE ids */
#define CLOCK_MCHP_DFLL_ID  MCHP_CLOCK_DERIVE_ID(1, 0x3f, 0x3f, 0, 0)
#define CLOCK_MCHP_DFLL_MAX (0)

/* FDPLL_TYPE id */
#define CLOCK_MCHP_FDPLL_ID_FDPLL0 MCHP_CLOCK_DERIVE_ID(2, 0x3f, 0x3f, 1, 0)
#define CLOCK_MCHP_FDPLL_ID_FDPLL1 MCHP_CLOCK_DERIVE_ID(2, 0x3f, 0x3f, 2, 1)
#define CLOCK_MCHP_FDPLL_ID_MAX    (1)

/* RTC_TYPE ids */
#define CLOCK_MCHP_RTC_ID     MCHP_CLOCK_DERIVE_ID(3, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_RTC_ID_MAX (0)

/* XOSC32K_TYPE ids */
#define CLOCK_MCHP_XOSC32K_ID_XOSC1K  MCHP_CLOCK_DERIVE_ID(4, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_XOSC32K_ID_XOSC32K MCHP_CLOCK_DERIVE_ID(4, 0x3f, 0x3f, 0x3f, 1)
#define CLOCK_MCHP_XOSC32K_ID_MAX     (1)

/* GCLKGEN_TYPE ids */
#define CLOCK_MCHP_GCLKGEN_ID_GEN0  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_GCLKGEN_ID_GEN1  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 1)
#define CLOCK_MCHP_GCLKGEN_ID_GEN2  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 2)
#define CLOCK_MCHP_GCLKGEN_ID_GEN3  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 3)
#define CLOCK_MCHP_GCLKGEN_ID_GEN4  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 4)
#define CLOCK_MCHP_GCLKGEN_ID_GEN5  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 5)
#define CLOCK_MCHP_GCLKGEN_ID_GEN6  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 6)
#define CLOCK_MCHP_GCLKGEN_ID_GEN7  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 7)
#define CLOCK_MCHP_GCLKGEN_ID_GEN8  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 8)
#define CLOCK_MCHP_GCLKGEN_ID_GEN9  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 9)
#define CLOCK_MCHP_GCLKGEN_ID_GEN10 MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 10)
#define CLOCK_MCHP_GCLKGEN_ID_GEN11 MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 11)
#define CLOCK_MCHP_GCLKGEN_ID_MAX   (11)

/* GCLKPERIPH_TYPE ids */
#define CLOCK_MCHP_GCLKPERIPH_ID_SDHC0_SLOW   MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 0)
#define CLOCK_MCHP_GCLKPERIPH_ID_SDHC1_SLOW   MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 1)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM0_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 2)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM1_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 3)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM2_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 4)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM3_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 5)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM4_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 6)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM5_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 7)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM6_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 8)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM7_SLOW MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 3, 9)
#define CLOCK_MCHP_GCLKPERIPH_ID_EIC          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 4, 10)
#define CLOCK_MCHP_GCLKPERIPH_ID_FREQM_MSR    MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 5, 11)
#define CLOCK_MCHP_GCLKPERIPH_ID_FREQM_REF    MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 6, 12)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM0_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 7, 13)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM1_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 8, 14)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC0          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 9, 15)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC1          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 9, 16)
#define CLOCK_MCHP_GCLKPERIPH_ID_USB          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 10, 17)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS0       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 11, 18)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS1       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 12, 19)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS2       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 13, 20)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS3       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 14, 21)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS4       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 15, 22)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS5       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 16, 23)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS6       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 17, 24)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS7       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 18, 25)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS8       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 19, 26)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS9       MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 20, 27)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS10      MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 21, 28)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS11      MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 22, 29)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM2_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 23, 30)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM3_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 24, 31)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC0         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 25, 32)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC1         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 25, 33)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC2          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 26, 34)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC3          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 26, 35)
#define CLOCK_MCHP_GCLKPERIPH_ID_CAN0         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 27, 36)
#define CLOCK_MCHP_GCLKPERIPH_ID_CAN1         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 28, 37)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC2         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 29, 38)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC3         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 29, 39)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC4          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 30, 40)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC5          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 30, 41)
#define CLOCK_MCHP_GCLKPERIPH_ID_PDEC         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 31, 42)
#define CLOCK_MCHP_GCLKPERIPH_ID_AC           MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 32, 43)
#define CLOCK_MCHP_GCLKPERIPH_ID_CCL          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 33, 44)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM4_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 34, 45)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM5_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 35, 46)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM6_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 36, 47)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM7_CORE MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 37, 48)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC4         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 38, 49)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC6          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 39, 50)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC7          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 39, 51)
#define CLOCK_MCHP_GCLKPERIPH_ID_ADC0         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 40, 52)
#define CLOCK_MCHP_GCLKPERIPH_ID_ADC1         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 41, 53)
#define CLOCK_MCHP_GCLKPERIPH_ID_DAC          MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 42, 54)
#define CLOCK_MCHP_GCLKPERIPH_ID_I2S0         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 43, 55)
#define CLOCK_MCHP_GCLKPERIPH_ID_I2S1         MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 44, 56)
#define CLOCK_MCHP_GCLKPERIPH_ID_SDHC0        MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 45, 57)
#define CLOCK_MCHP_GCLKPERIPH_ID_SDHC1        MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 46, 58)
#define CLOCK_MCHP_GCLKPERIPH_ID_CM4_TRACE    MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 47, 59)
#define CLOCK_MCHP_GCLKPERIPH_ID_MAX          (59)

/* MCLKCPU_TYPE ids */
#define CLOCK_MCHP_MCLKCPU_ID  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_MCLKCPU_MAX (0)

/* MCLKPERIPH_TYPE ids */
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_HPB0             MCHP_CLOCK_DERIVE_ID(8, 0, 0, 0x3f, 0)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_HPB1             MCHP_CLOCK_DERIVE_ID(8, 0, 1, 0x3f, 1)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_HPB2             MCHP_CLOCK_DERIVE_ID(8, 0, 2, 0x3f, 2)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_HPB3             MCHP_CLOCK_DERIVE_ID(8, 0, 3, 0x3f, 3)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_DSU              MCHP_CLOCK_DERIVE_ID(8, 0, 4, 0x3f, 4)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_NVMCTRL          MCHP_CLOCK_DERIVE_ID(8, 0, 6, 0x3f, 5)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_CMCC             MCHP_CLOCK_DERIVE_ID(8, 0, 8, 0x3f, 6)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_DMAC             MCHP_CLOCK_DERIVE_ID(8, 0, 9, 0x3f, 7)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_USB              MCHP_CLOCK_DERIVE_ID(8, 0, 10, 0x3f, 8)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_PAC              MCHP_CLOCK_DERIVE_ID(8, 0, 12, 0x3f, 9)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_QSPI             MCHP_CLOCK_DERIVE_ID(8, 0, 13, 0x3f, 10)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_GMAC             MCHP_CLOCK_DERIVE_ID(8, 0, 14, 0x3f, 11)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_SDHC0            MCHP_CLOCK_DERIVE_ID(8, 0, 15, 0x3f, 12)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_SDHC1            MCHP_CLOCK_DERIVE_ID(8, 0, 16, 0x3f, 13)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_CAN0             MCHP_CLOCK_DERIVE_ID(8, 0, 17, 0x3f, 14)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_CAN1             MCHP_CLOCK_DERIVE_ID(8, 0, 18, 0x3f, 15)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_ICM              MCHP_CLOCK_DERIVE_ID(8, 0, 19, 0x3f, 16)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_PUKCC            MCHP_CLOCK_DERIVE_ID(8, 0, 20, 0x3f, 17)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_QSPI_2X          MCHP_CLOCK_DERIVE_ID(8, 0, 21, 0x3f, 18)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_NVMCTRL_SMEEPROM MCHP_CLOCK_DERIVE_ID(8, 0, 22, 0x3f, 19)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_NVMCTRL_CACHE    MCHP_CLOCK_DERIVE_ID(8, 0, 23, 0x3f, 20)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_PAC        MCHP_CLOCK_DERIVE_ID(8, 1, 0, 0x3f, 21)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_PM         MCHP_CLOCK_DERIVE_ID(8, 1, 1, 0x3f, 22)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_MCLK       MCHP_CLOCK_DERIVE_ID(8, 1, 2, 0x3f, 23)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_RSTC       MCHP_CLOCK_DERIVE_ID(8, 1, 3, 0x3f, 24)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_OSCCTRL    MCHP_CLOCK_DERIVE_ID(8, 1, 4, 0x3f, 25)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_OSC32KCTRL MCHP_CLOCK_DERIVE_ID(8, 1, 5, 0x3f, 26)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_SUPC       MCHP_CLOCK_DERIVE_ID(8, 1, 6, 0x3f, 27)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_GCLK       MCHP_CLOCK_DERIVE_ID(8, 1, 7, 0x3f, 28)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_WDT        MCHP_CLOCK_DERIVE_ID(8, 1, 8, 0x3f, 29)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_RTC        MCHP_CLOCK_DERIVE_ID(8, 1, 9, 0x3f, 30)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_EIC        MCHP_CLOCK_DERIVE_ID(8, 1, 10, 0x3f, 31)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_FREQM      MCHP_CLOCK_DERIVE_ID(8, 1, 11, 0x3f, 32)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_SERCOM0    MCHP_CLOCK_DERIVE_ID(8, 1, 12, 0x3f, 33)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_SERCOM1    MCHP_CLOCK_DERIVE_ID(8, 1, 13, 0x3f, 34)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_TC0        MCHP_CLOCK_DERIVE_ID(8, 1, 14, 0x3f, 35)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_TC1        MCHP_CLOCK_DERIVE_ID(8, 1, 15, 0x3f, 36)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_USB     MCHP_CLOCK_DERIVE_ID(8, 2, 0, 0x3f, 37)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_DSU     MCHP_CLOCK_DERIVE_ID(8, 2, 1, 0x3f, 38)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_NVMCTRL MCHP_CLOCK_DERIVE_ID(8, 2, 2, 0x3f, 39)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_PORT    MCHP_CLOCK_DERIVE_ID(8, 2, 4, 0x3f, 40)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_EVSYS   MCHP_CLOCK_DERIVE_ID(8, 2, 7, 0x3f, 41)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_SERCOM2 MCHP_CLOCK_DERIVE_ID(8, 2, 9, 0x3f, 42)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_SERCOM3 MCHP_CLOCK_DERIVE_ID(8, 2, 10, 0x3f, 43)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_TCC0    MCHP_CLOCK_DERIVE_ID(8, 2, 11, 0x3f, 44)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_TCC1    MCHP_CLOCK_DERIVE_ID(8, 2, 12, 0x3f, 45)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_TC2     MCHP_CLOCK_DERIVE_ID(8, 2, 13, 0x3f, 46)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_TC3     MCHP_CLOCK_DERIVE_ID(8, 2, 14, 0x3f, 47)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_RAMECC  MCHP_CLOCK_DERIVE_ID(8, 2, 16, 0x3f, 48)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_GMAC MCHP_CLOCK_DERIVE_ID(8, 3, 2, 0x3f, 49)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TCC2 MCHP_CLOCK_DERIVE_ID(8, 3, 3, 0x3f, 50)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TCC3 MCHP_CLOCK_DERIVE_ID(8, 3, 4, 0x3f, 51)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC4  MCHP_CLOCK_DERIVE_ID(8, 3, 5, 0x3f, 52)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC5  MCHP_CLOCK_DERIVE_ID(8, 3, 6, 0x3f, 53)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_PDEC MCHP_CLOCK_DERIVE_ID(8, 3, 7, 0x3f, 54)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_AC   MCHP_CLOCK_DERIVE_ID(8, 3, 8, 0x3f, 55)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_AES  MCHP_CLOCK_DERIVE_ID(8, 3, 9, 0x3f, 56)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TRNG MCHP_CLOCK_DERIVE_ID(8, 3, 10, 0x3f, 57)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_ICM  MCHP_CLOCK_DERIVE_ID(8, 3, 11, 0x3f, 58)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_QSPI MCHP_CLOCK_DERIVE_ID(8, 3, 13, 0x3f, 59)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_CCL  MCHP_CLOCK_DERIVE_ID(8, 3, 14, 0x3f, 60)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_SERCOM4 MCHP_CLOCK_DERIVE_ID(8, 4, 0, 0x3f, 61)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_SERCOM5 MCHP_CLOCK_DERIVE_ID(8, 4, 1, 0x3f, 62)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_SERCOM6 MCHP_CLOCK_DERIVE_ID(8, 4, 2, 0x3f, 63)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_SERCOM7 MCHP_CLOCK_DERIVE_ID(8, 4, 3, 0x3f, 64)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_TCC4    MCHP_CLOCK_DERIVE_ID(8, 4, 4, 0x3f, 65)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_TC6     MCHP_CLOCK_DERIVE_ID(8, 4, 5, 0x3f, 66)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_TC7     MCHP_CLOCK_DERIVE_ID(8, 4, 6, 0x3f, 67)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_ADC0    MCHP_CLOCK_DERIVE_ID(8, 4, 7, 0x3f, 68)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_ADC1    MCHP_CLOCK_DERIVE_ID(8, 4, 8, 0x3f, 69)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_DAC     MCHP_CLOCK_DERIVE_ID(8, 4, 9, 0x3f, 70)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_I2S     MCHP_CLOCK_DERIVE_ID(8, 4, 10, 0x3f, 71)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_PCC     MCHP_CLOCK_DERIVE_ID(8, 4, 11, 0x3f, 72)
#define CLOCK_MCHP_MCLKPERIPH_ID_MAX          (72)

#endif /* INCLUDE_ZEPHYR_DT_BINDINGS_CLOCK_MCHP_SAM_D5X_E5X_CLOCK_H_ */
