# mortal_write/views/writer.py

import streamlit as st
import streamlit.components.v1 as components  # 🔥 [新增导入] 必须引入这个
import time
import json
import pandas as pd
import os 
import re 
from utils import (
    render_header, 
    update_chapter_title_db, 
    update_chapter_summary_db,
    resequence_chapters,
    log_operation,      
    ensure_log_file     
)
# 🔥 [修改 1] 引入 get_beijing_time
from views.books import record_token_usage, get_beijing_time
from logic import FEATURE_MODELS, MODEL_MAPPING

# ==============================================================================
# 🛠️ Helpers (保持不变)
# ==============================================================================

# 🔥 从 config 导入 DATA_DIR
try:
    from config import DATA_DIR
except ImportError:
    # 如果 config 中没有定义，使用默认值
    DATA_DIR = "data"

# 🔥 [新增函数] 删除书籍缓存文件
def delete_book_cache(book_id, book_title):
    """删除书籍的缓存文件"""
    try:
        # 使用与books.py相同的路径逻辑
        safe_title = re.sub(r'[\\/*?:"<>|]', "", str(book_title)).strip()
        # 使用从 config 导入的 DATA_DIR
        cache_file = os.path.join(DATA_DIR, "exports", f"{book_id}_{safe_title}.txt")
        if os.path.exists(cache_file):
            os.remove(cache_file)
            print(f"✅ 已删除缓存文件: {cache_file}")
    except Exception as e:
        print(f"⚠️ 删除缓存文件失败: {e}")

# 🔥 [修改 2] 修改force_update_book_time函数，添加缓存清理
def force_update_book_time(db_mgr, book_id):
    """强制使用北京时间更新书籍的 updated_at 字段，并删除缓存文件"""
    try:
        current_time = get_beijing_time()
        db_mgr.execute("UPDATE books SET updated_at=? WHERE id=?", (current_time, book_id))
        
        # 获取书籍标题并删除缓存
        try:
            book_info = db_mgr.query("SELECT title FROM books WHERE id=?", (book_id,))
            if book_info:
                book_title = book_info[0]['title']
                delete_book_cache(book_id, book_title)
        except Exception as e:
            print(f"⚠️ 删除缓存失败: {e}")
    except Exception as e:
        print(f"Update time failed: {e}")

def get_safe_model_default(feature_key, hard_coded_default):
    if feature_key in FEATURE_MODELS:
        return FEATURE_MODELS[feature_key].get('default', hard_coded_default)
    return hard_coded_default

def safe_get_content(chunk):
    """Safe chunk extraction"""
    try:
        if isinstance(chunk, str): return chunk
        if hasattr(chunk, 'choices') and chunk.choices:
            if len(chunk.choices) > 0:
                delta = chunk.choices[0].delta
                if hasattr(delta, 'content'): return delta.content
        if isinstance(chunk, dict):
            return chunk.get('choices', [{}])[0].get('delta', {}).get('content', '')
        return None
    except Exception: return None

def _ensure_part_volume_schema(db_mgr):
    if st.session_state.get('schema_checked', False): return
    try:
        db_mgr.execute("""
            CREATE TABLE IF NOT EXISTS parts (
                id INTEGER PRIMARY KEY AUTOINCREMENT, 
                book_id INTEGER, name TEXT, summary TEXT, sort_order INTEGER
            )
        """)
        try: db_mgr.query("SELECT part_id FROM volumes LIMIT 1")
        except: 
            try: db_mgr.execute("ALTER TABLE volumes ADD COLUMN part_id INTEGER")
            except: pass
        st.session_state.schema_checked = True
    except Exception: pass

def _update_chapter_title_db_logged(chap_id, new_title_key):
    new_title = st.session_state[new_title_key]
    db_mgr = st.session_state.db
    if new_title:
        db_mgr.execute("UPDATE chapters SET title=? WHERE id=?", (new_title, chap_id))
        # 🔥 [修改 3] 替换 update_book_timestamp
        force_update_book_time(db_mgr, st.session_state.current_book_id)
        st.session_state.chapter_title_cache[chap_id] = new_title
        ensure_log_file()
        log_operation("章节管理", f"重命名章节 ID:{chap_id} 为 {new_title}")
        st.session_state.rerun_flag = True

def _update_chapter_summary_db_logged(chap_id, new_summary_key):
    new_summary = st.session_state[new_summary_key]
    db_mgr = st.session_state.db
    db_mgr.execute("UPDATE chapters SET summary=? WHERE id=?", (new_summary, chap_id))
    # 🔥 [修改 4] 替换 update_book_timestamp
    force_update_book_time(db_mgr, st.session_state.current_book_id)
    ensure_log_file()
    log_operation("章节管理", f"更新章节大纲 ID:{chap_id}")
    st.session_state.rerun_flag = True

# ==============================================================================
# 🧠 Context Manager (保持不变)
# ==============================================================================
def get_full_context(db_mgr, book_id, current_chap_id=None):
    context_parts = []
    # 1. 世界观
    try:
        settings = db_mgr.query("SELECT content FROM plots WHERE book_id=? AND status LIKE 'Setting_%'", (book_id,))
        if settings:
            s_text = "\n".join([f"- {s['content']}" for s in settings])
            context_parts.append(f"【世界观与设定】\n{s_text}")
    except: pass
    
    # 2. 角色
    try:
        chars = db_mgr.query("SELECT name, role, gender, desc, cheat_ability, power_level, debts_and_feuds FROM characters WHERE book_id=? AND is_major=1", (book_id,))
        if chars:
            c_list = []
            for c in chars:
                info = f"{c['name']}({c['role']}): {c['desc']}"
                if c['cheat_ability']: info += f" | 金手指: {c['cheat_ability']}"
                c_list.append(info)
            context_parts.append(f"【核心角色档案】\n" + "\n".join(c_list))
    except: pass
    
    # 3. 前情
    try:
        all_chaps = db_mgr.query("""
            SELECT c.id, c.title, c.content, c.summary 
            FROM chapters c
            JOIN volumes v ON c.volume_id = v.id
            JOIN parts p ON v.part_id = p.id
            WHERE p.book_id = ?
            ORDER BY p.sort_order, v.sort_order, c.sort_order
        """, (book_id,))
        curr_idx = -1
        if current_chap_id:
            for idx, item in enumerate(all_chaps):
                if item['id'] == current_chap_id:
                    curr_idx = idx; break
        target_idx = curr_idx if curr_idx != -1 else len(all_chaps)
        start_idx = max(0, target_idx - 5)
        prev_chaps = all_chaps[start_idx:target_idx]
        if prev_chaps:
            prev_text = []
            for pc in prev_chaps:
                content_preview = pc['summary'] or ""
                if not content_preview or len(content_preview) < 10:
                    raw_cont = pc['content'] or ""
                    content_preview = raw_cont[-500:] if len(raw_cont) > 500 else raw_cont
                prev_text.append(f"Chapter: {pc['title']}\nSummary/Excerpt: {content_preview}")
            context_parts.append(f"【前情回顾】\n" + "\n---\n".join(prev_text))
    except Exception as e: print(f"Context Error: {e}")
    return "\n\n".join(context_parts)

# ==============================================================================
# 🪟 Dialogs
# ==============================================================================
if hasattr(st, "dialog"): dialog_decorator = st.dialog
else: dialog_decorator = st.experimental_dialog

@dialog_decorator("💾 保存确认")
def dialog_save_chapter_content(db_mgr, chapter_id, new_content, book_id, chapter_title):
    # 🌟 美化：标题与信息展示
    st.markdown(f"### 正在保存：{chapter_title}")
    
    # 使用 metric 组件展示关键指标，比纯文本更好看
    st.metric(label="当前文档字数", value=f"{len(new_content)}", delta="准备写入数据库", delta_color="off")
    
    st.divider() # 分割线
    
    # 警示信息
    st.warning("⚠️ 此操作将覆盖数据库中的原有内容，确定要继续吗？")
    
    # 增加一点间距
    st.markdown("<div style='margin-bottom: 15px;'></div>", unsafe_allow_html=True)

    # 🌟 美化：按钮布局优化
    col_cancel, col_confirm = st.columns([1, 1], gap="medium")
    
    with col_cancel:
        # 使用 use_container_width=True 让按钮填满列宽，看起来更整齐
        if st.button("🚫 取消", type="secondary", use_container_width=True): 
            st.rerun()
            
    with col_confirm:
        if st.button("💾 确认覆盖", type="primary", use_container_width=True):
            try:
                db_mgr.execute("UPDATE chapters SET content=? WHERE id=?", (new_content, chapter_id)) 
                # 🔥 [修改 5] 替换 update_book_timestamp
                force_update_book_time(db_mgr, book_id)
                ensure_log_file()
                log_operation("更新章节", f"保存正文: {chapter_title}")
                st.session_state.rerun_flag = True
                st.toast("✅ 已保存", icon="💾"); time.sleep(0.5); st.rerun()
            except Exception as e: st.error(f"保存失败: {e}")

@dialog_decorator("➕ 新建")
def dialog_add_node(type_label, parent_id, book_id=None):
    # 此处无需更新书籍时间，或者根据需要更新
    st.caption(f"正在创建新的 **{type_label}**")
    name_input = st.text_input("名称", key="dialog_name_input")
    if st.button("确认创建", type="primary"):
        if not name_input.strip(): st.error("名称不能为空"); return
        db_mgr = st.session_state.db
        if type_label == "篇":
            mx = db_mgr.query("SELECT MAX(sort_order) as m FROM parts WHERE book_id=?", (book_id,))[0]['m'] or 0
            db_mgr.execute("INSERT INTO parts (book_id, name, sort_order) VALUES (?,?,?)", (book_id, name_input, mx+100))
        elif type_label == "卷":
            mx = db_mgr.query("SELECT MAX(sort_order) as m FROM volumes WHERE part_id=?", (parent_id,))[0]['m'] or 0
            db_mgr.execute("INSERT INTO volumes (book_id, part_id, name, sort_order) VALUES (?,?,?,?)", (st.session_state.current_book_id, parent_id, name_input, mx+100))
            if 'expanded_parts' not in st.session_state: st.session_state.expanded_parts = set()
            st.session_state.expanded_parts.add(parent_id)
        elif type_label == "章":
            mx = db_mgr.query("SELECT MAX(sort_order) as m FROM chapters WHERE volume_id=?", (parent_id,))[0]['m'] or 0
            cid = db_mgr.execute("INSERT INTO chapters (volume_id, title, summary, content, sort_order) VALUES (?,?,?,?,?)",
                           (parent_id, name_input, "", "", mx+1))
            st.session_state.current_chapter_id = cid
            if 'expanded_volumes' not in st.session_state: st.session_state.expanded_volumes = set()
            st.session_state.expanded_volumes.add(parent_id)
        ensure_log_file()
        log_operation("结构管理", f"新建{type_label}: {name_input}")
        st.session_state.rerun_flag = True; st.rerun()

@dialog_decorator("⚙️ 管理")
def dialog_manage_node(type_label, node_id, current_name):
    st.caption(f"正在管理 {type_label}: **{current_name}**")
    new_name = st.text_input("重命名", value=current_name)
    c1, c2 = st.columns(2)
    if c1.button("💾 保存修改", type="primary"):
        db_mgr = st.session_state.db
        table = "parts" if type_label == "篇" else "volumes"
        db_mgr.execute(f"UPDATE {table} SET name=? WHERE id=?", (new_name, node_id))
        log_operation("结构管理", f"重命名{type_label}: {current_name} -> {new_name}")
        st.rerun()
    if c2.button("🗑️ 删除", type="secondary"):
        db_mgr = st.session_state.db
        table = "parts" if type_label == "篇" else "volumes"
        db_mgr.execute(f"DELETE FROM {table} WHERE id=?", (node_id,))
        log_operation("结构管理", f"删除{type_label}: {current_name}")
        st.rerun()

# ==============================================================================
# 🎨 Explorer Logic (保持不变)
# ==============================================================================
def toggle_state(key, item_id):
    if key not in st.session_state: st.session_state[key] = set()
    if item_id in st.session_state[key]: st.session_state[key].remove(item_id)
    else: st.session_state[key].add(item_id)

def render_explorer_node_part(db_mgr, part, current_book_id):
    if 'expanded_parts' not in st.session_state: st.session_state.expanded_parts = set()
    is_expanded = part['id'] in st.session_state.expanded_parts
    icon = "📂" if not is_expanded else "📖"
    if st.button(f"{icon} {part['name']}", key=f"p_btn_{part['id']}", use_container_width=True):
        toggle_state('expanded_parts', part['id']); st.rerun()
    if is_expanded:
        st.markdown("""<div style="margin-top: -12px; margin-bottom: 5px;"></div>""", unsafe_allow_html=True)
        c_i, c_act1, c_act2 = st.columns([0.1, 1, 1])
        with c_act1:
            if st.button("➕ 加卷", key=f"add_v_{part['id']}", use_container_width=True): dialog_add_node("卷", part['id'])
        with c_act2:
            if st.button("⚙️ 管理", key=f"mng_p_{part['id']}", use_container_width=True): dialog_manage_node("篇", part['id'], part['name'])
        vols = db_mgr.query("SELECT id, name, part_id FROM volumes WHERE part_id=? ORDER BY sort_order", (part['id'],))
        if not vols: st.markdown("<div style='padding-left: 15px; color: gray; font-size: 12px; margin-bottom: 10px;'>└─ (暂无卷)</div>", unsafe_allow_html=True)
        else:
            for vol in vols: render_explorer_node_volume(db_mgr, vol)
        st.markdown("""<div style="margin-bottom: 10px;"></div>""", unsafe_allow_html=True)

def render_explorer_node_volume(db_mgr, vol):
    if 'expanded_volumes' not in st.session_state: st.session_state.expanded_volumes = set()
    is_expanded = vol['id'] in st.session_state.expanded_volumes
    icon = "📁" if not is_expanded else "📂"
    c_indent, c_main = st.columns([0.2, 5.8])
    with c_main:
        if st.button(f"{icon} {vol['name']}", key=f"v_btn_{vol['id']}", use_container_width=True):
            toggle_state('expanded_volumes', vol['id']); st.rerun()
    if is_expanded:
        st.markdown("""<div style="margin-top: -12px; margin-bottom: 5px;"></div>""", unsafe_allow_html=True)
        c_i, c_act1, c_act2 = st.columns([0.3, 1, 1])
        with c_act1:
            if st.button("➕ 加章", key=f"add_c_{vol['id']}", use_container_width=True): dialog_add_node("章", vol['id'])
        with c_act2:
            if st.button("⚙️ 管理", key=f"mng_v_{vol['id']}", use_container_width=True): dialog_manage_node("卷", vol['id'], vol['name'])
        chaps = db_mgr.query("SELECT id, title FROM chapters WHERE volume_id=? ORDER BY sort_order", (vol['id'],))
        if not chaps: st.markdown("<div style='padding-left: 35px; color: gray; font-size: 12px;'>└─ (暂无章节)</div>", unsafe_allow_html=True)
        else:
            st.markdown("""<style>div[data-testid="stVerticalBlock"] > div > button.chap-btn { text-align: left; padding-left: 35px !important; border: none; font-size: 14px; }</style>""", unsafe_allow_html=True)
            with st.container():
                for chap in chaps:
                    is_active = (st.session_state.get('current_chapter_id') == chap['id'])
                    display_title = st.session_state.chapter_title_cache.get(chap['id'], chap['title'])
                    b_type = "primary" if is_active else "secondary"
                    if st.button(f"　　{display_title}", key=f"c_{chap['id']}", use_container_width=True, type=b_type):
                        ensure_log_file()
                        log_operation("阅读章节", f"切换章节: {display_title} (ID:{chap['id']})")
                        st.session_state.current_chapter_id = chap['id']
                        st.session_state.current_part_id = vol['part_id']
                        st.rerun()
        st.markdown("""<div style="margin-bottom: 8px;"></div>""", unsafe_allow_html=True)

# ==============================================================================
# 🚀 Render Main Logic
# ==============================================================================
def render_writer(engine, current_book, current_chapter):
    # 🔥 [新增逻辑]：检查是否需要强制滚动
    if st.session_state.get("trigger_scroll_to_top"):
        # 强制滚动的 JS 代码：多点定位，多次执行，确保对抗浏览器的位置恢复
        js_code = """
        <script>
            function scrollNode(node) {
                if (node) {
                    node.scrollTop = 0;
                    node.scrollLeft = 0;
                }
            }
            function forceScroll() {
                try {
                    var parent = window.parent.document;
                    // 覆盖 Streamlit 常用的几种容器 class
                    scrollNode(parent.querySelector('[data-testid="stAppViewContainer"]'));
                    scrollNode(parent.querySelector('section.main'));
                    scrollNode(parent.documentElement);
                    scrollNode(parent.body);
                } catch(e) {
                    console.log("Scroll error:", e);
                }
            }
            // 连环执行，对抗浏览器的恢复行为 (持续1秒)
            for (let i = 0; i < 20; i++) {
                window.setTimeout(forceScroll, i * 50);
            }
        </script>
        """
        components.html(js_code, height=0, width=0)
        st.session_state.trigger_scroll_to_top = False # 消费掉这个 Flag
        
    db_mgr = st.session_state.db
    _ensure_part_volume_schema(db_mgr)
    if 'generation_running' not in st.session_state: st.session_state.generation_running = False
    st.session_state.model_assignments = engine.get_config_db("model_assignments", {})

    if not current_book:
        st.warning("请先在 [书籍管理] 中选择一本书"); return
        
    render_header("✍️", current_book['title'])
    full_context_str = get_full_context(db_mgr, current_book['id'], current_chapter['id'] if current_chapter else None)
    
    st.markdown("""
    <style>
    /* 目录树样式 */
    div[data-testid="column"]:nth-of-type(1) button { border: 0px solid transparent !important; background: transparent !important; box-shadow: none !important; text-align: left !important; } 
    div[data-testid="column"]:nth-of-type(1) button:hover { background-color: rgba(150, 150, 150, 0.1) !important; color: #3eaf7c !important; } 
    div[data-testid="column"]:nth-of-type(1) button[kind="primary"] { background-color: rgba(62, 175, 124, 0.15) !important; border-left: 3px solid #3eaf7c !important; color: #3eaf7c !important; padding-left: 8px !important; }
    .custom-select-label { font-size: 14px; font-weight: 600; color: #444; margin-bottom: 4px; display: block; }
    div[data-testid="stSelectbox"] div[data-baseweb="select"] > div { background-color: #fcfcfc; border: 1px solid #ddd; border-radius: 6px; }
    textarea { font-size: 16px !important; line-height: 1.6 !important; font-family: 'PingFang SC', sans-serif; }
    </style>
    """, unsafe_allow_html=True)
    
    col_explorer, col_editor = st.columns([1.4, 2.6], gap="medium")
    
    with col_explorer:
        all_books = db_mgr.query("SELECT id, title FROM books")
        opts = {b['title']: b['id'] for b in all_books}
        idx = list(opts.values()).index(current_book['id']) if current_book['id'] in opts.values() else 0
        sel_title = st.selectbox("当前书籍", list(opts.keys()), index=idx, label_visibility="collapsed", key="write_book_selector")
        if opts[sel_title] != current_book['id']:
            ensure_log_file(); log_operation("页面跳转", f"切换书籍: {sel_title}")
            st.session_state.current_book_id = opts[sel_title]
            st.session_state.current_chapter_id = None; st.session_state.current_part_id = None
            st.rerun()
        
        c_label, c_add = st.columns([4, 2])
        with c_label: st.caption("🗂️ 目录结构")
        with c_add:
            if st.button("➕ 新建篇", key="root_add_p_btn", use_container_width=True): dialog_add_node("篇", None, current_book['id'])
        parts = db_mgr.query("SELECT id, name FROM parts WHERE book_id=? ORDER BY sort_order", (current_book['id'],))
        if not parts: st.info('暂无内容，请点击上方"新建篇"')
        else:
            for part in parts: render_explorer_node_part(db_mgr, part, current_book['id'])

    with col_editor:
        tab_write, tab_outline, tab_assist = st.tabs(["📝 沉浸写作", "🧠 AI 批量生成", "✨ 写作辅助"])
        
        # ======================================================================
        # TAB 1: 沉浸写作
        # ======================================================================
        with tab_write:
            if not current_chapter: st.info("👈 请先从左侧选择一个章节。")
            else:
                title_key = f"chap_title_{current_chapter['id']}"
                st.text_input("章节标题", current_chapter['title'], key=title_key, on_change=_update_chapter_title_db_logged, args=(current_chapter['id'], title_key))
                
                st.markdown("##### 🤖 智能写作控制台")
                current_summary = current_chapter['summary'] or ""
                outline_key = f"ai_outline_input_{current_chapter['id']}"
                outline_input = st.text_area("本章大纲/提示词", current_summary, height=80, key=outline_key, 
                                           placeholder="在此输入剧情简述...",
                                           on_change=_update_chapter_summary_db_logged, args=(current_chapter['id'], outline_key))
                
                c_m, c_l = st.columns([1.2, 1.8]) 
                with c_m:
                    def_write = get_safe_model_default("write_quick_gen", "DSK_V3")
                    assigned_write = st.session_state.model_assignments.get("write_quick_gen", def_write)
                    model_display_name = MODEL_MAPPING.get(assigned_write, {}).get('name', assigned_write)
                    st.caption(f"当前模型: {model_display_name}")
                    model_pk = assigned_write if assigned_write in MODEL_MAPPING else None
                with c_l: 
                    target_k = st.slider("本章预计字数 (k)", 1.0, 10.0, 3.0, 0.1, key="word_slider")
                    target_words_num = int(target_k * 1000)

                st.markdown("<div style='margin-bottom: 10px;'></div>", unsafe_allow_html=True)
                
                btn_col1, btn_col2 = st.columns(2)
                with btn_col1:
                    btn_gen = st.button("✨ 开始生成", type="secondary", disabled=st.session_state.generation_running, use_container_width=True)
                with btn_col2:
                    btn_stop = st.button("⏹️ 停止生成", type="secondary", disabled=not st.session_state.generation_running, use_container_width=True)

                content_key = f"chapter_content_{current_chapter['id']}"
                if content_key not in st.session_state:
                    st.session_state[content_key] = current_chapter['content'] or ""

                if st.button("💾 保存当前正文", type="primary", use_container_width=True):
                     content_to_save = st.session_state.get(content_key, current_chapter['content'] or "")
                     dialog_save_chapter_content(db_mgr, current_chapter['id'], content_to_save, current_book['id'], current_chapter['title'])

                st.markdown("<div style='margin-bottom: 20px;'></div>", unsafe_allow_html=True)

                current_text = st.session_state.get(content_key, "")
                current_len = len(current_text)
                if current_len > target_words_num + 1000: len_color = "red"
                elif current_len < 100: len_color = "orange"
                else: len_color = "green"

                # 🔥 [修改核心]：将原来的两列改为三列，中间留给 AI 状态显示
                c_head, c_status, c_count = st.columns([1.2, 2.5, 1.3])
                
                with c_head: 
                    st.caption("📖 正文编辑")
                
                # 创建一个占位符，专门用于显示 "AI 正在码字..."
                ai_status_box = c_status.empty()
                
                with c_count: 
                    st.markdown(f"<div style='text-align:right; font-size:14px; padding-bottom:5px;'>字数: <span style='color:{len_color}; font-weight:bold'>{current_len}</span> / {target_words_num}</div>", unsafe_allow_html=True)

                editor_placeholder = st.empty()
                should_render_editor = True

                if btn_gen and model_pk:
                    should_render_editor = False 
                    client, m_name, m_key = engine.get_client(model_pk)
                    if not client: st.error("API Key 未配置")
                    else:
                        db_mgr.execute("UPDATE chapters SET summary=? WHERE id=?", (outline_input, current_chapter['id']))
                        st.session_state.generation_running = True
                        ensure_log_file(); log_operation("AI生成", f"开始生成: {current_chapter['title']}")
                        
                        final_prompt = (
                            f"{full_context_str}\n\n"
                            f"【本章写作要求】\n"
                            f"1. 这是一个小说正文生成任务。\n"
                            f"2. 请严格控制篇幅在 {target_words_num} 字左右。\n"
                            f"3. 严禁注水，严禁重复。\n"
                            f"4. 剧情大纲：{outline_input}"
                        )
                        hard_stop_limit = int(target_words_num * 1.5)
                        
                        # 🔥 [修改]：移除 st.spinner，改为在标题右侧更新状态文字
                        ai_status_box.markdown(f":blue[⚡ AI 正在码字 ({m_name})...] <span style='font-size:12px'>Thinking...</span>", unsafe_allow_html=True)
                        
                        ok, stream = engine.generate_content_from_outline_ai_stream(current_chapter['id'], final_prompt, current_book, target_words_num, client, m_name, m_key)
                        if ok:
                            buf = ""
                            full_existing = st.session_state[content_key] or ""
                            full_existing = full_existing.strip() + "\n" if full_existing else ""
                            
                            for chunk in stream:
                                if not st.session_state.generation_running: break
                                content_text = safe_get_content(chunk)
                                if content_text:
                                    buf += content_text
                                    simulated_content = full_existing + buf
                                    editor_placeholder.markdown(
                                        f"""<div style="height: 600px; overflow-y: auto; border: 1px solid rgba(49, 51, 63, 0.2); border-radius: 0.25rem; padding: 1rem; font-family: 'Source Sans Pro', sans-serif; white-space: pre-wrap; background-color: transparent;">{simulated_content}</div>""",
                                        unsafe_allow_html=True
                                    )
                                    if len(buf) > hard_stop_limit:
                                        st.session_state.generation_running = False
                                        st.toast(f"⚠️ 达到字数上限保护，已停止", icon="🛑"); break
                                        
                            full_new = full_existing + buf
                            db_mgr.execute("UPDATE chapters SET content=? WHERE id=?", (full_new, current_chapter['id']))
                            # 🔥 [修改 6] 保持不变
                            force_update_book_time(db_mgr, current_book['id'])
                            st.session_state[content_key] = full_new 
                            record_token_usage(provider="AI", model=m_name, tokens=len(buf), action_name="沉浸写作", book_title=current_book['title'])
                            log_operation("AI生成", f"生成完成: {current_chapter['title']}")
                            st.session_state.generation_running = False
                            
                            # 🔥 [修改]：生成完成后清空状态
                            ai_status_box.empty()
                            st.rerun() 
                        else:
                            ai_status_box.error("生成失败") 
                            st.error(f"生成失败: {stream}"); st.session_state.generation_running = False
                
                if btn_stop:
                    # 停止时也清空状态
                    ai_status_box.empty()
                    st.session_state.generation_running = False; log_operation("AI生成", "用户停止"); st.rerun()

                if should_render_editor:
                    with editor_placeholder:
                        st.text_area(label="hidden_content", value=current_text, height=600, label_visibility="collapsed", key=content_key)

        # ======================================================================
        # TAB 2: 批量生成
        # ======================================================================
        with tab_outline:
             if not current_book: st.warning("请选择书籍")
             elif not parts: st.warning("无结构")
             else:
                st.subheader("✍️ 批量生成")
                default_p_idx = 0; default_v_idx = 0
                part_opts = {p['name']: p['id'] for p in parts}
                if current_chapter:
                    curr_vol = db_mgr.query("SELECT * FROM volumes WHERE id=?", (current_chapter['volume_id'],))[0]
                    if curr_vol['part_id'] in part_opts.values(): default_p_idx = list(part_opts.values()).index(curr_vol['part_id'])
                
                c_p, c_v = st.columns(2)
                sel_p_name = c_p.selectbox("选择篇", list(part_opts.keys()), index=default_p_idx, key="bg_p")
                sel_p_id = part_opts[sel_p_name]
                bg_vols = db_mgr.query("SELECT * FROM volumes WHERE part_id=? ORDER BY sort_order", (sel_p_id,))
                
                if not bg_vols: st.warning("该篇无卷")
                else:
                    bg_v_opts = {v['name']: v['id'] for v in bg_vols}
                    if current_chapter and current_chapter['volume_id'] in bg_v_opts.values(): default_v_idx = list(bg_v_opts.values()).index(current_chapter['volume_id'])
                    sel_v_name = c_v.selectbox("选择卷", list(bg_v_opts.keys()), index=default_v_idx, key="bg_v")
                    sel_v_id = bg_v_opts[sel_v_name]
                    bg_chaps = db_mgr.query("SELECT id, title, summary FROM chapters WHERE volume_id=? ORDER BY sort_order", (sel_v_id,))
                    
                    if not bg_chaps: st.info("该卷无章节")
                    else:
                        c_names = [c['title'] for c in bg_chaps]
                        c1, c2 = st.columns(2)
                        s_start = c1.selectbox("起始章", c_names, 0, key="bg_s")
                        s_idx = c_names.index(s_start)
                        s_end = c2.selectbox("结束章", c_names[s_idx:], len(c_names[s_idx:])-1, key="bg_e")
                        e_idx = c_names.index(s_end)
                        target_chaps = bg_chaps[s_idx:e_idx+1]
                        st.info(f"选中: {len(target_chaps)} 章")
                        gen_prompt = st.text_area("通用大纲/指令", height=100, placeholder="例如：主角在这一段剧情中...")
                        
                        cm1, cm2, cm3 = st.columns([1, 1, 1])
                        with cm1:
                            def_b = get_safe_model_default("write_batch_gen", "GPT_4o")
                            assigned_b = st.session_state.model_assignments.get("write_batch_gen", def_b)
                            model_display_name_b = MODEL_MAPPING.get(assigned_b, {}).get('name', assigned_b)
                            st.markdown(f"**模型**"); st.caption(f"🚀 {model_display_name_b}")
                            pk_b = assigned_b if assigned_b in MODEL_MAPPING else None
                        with cm2: len_b = st.slider("单章字数(k)", 1, 10, 3, key="bg_l")
                        with cm3:
                            st.markdown('<div style="padding-top: 29px;"></div>', unsafe_allow_html=True)
                            btn_bg = st.button("🚀 开始批量", type="primary", use_container_width=True, disabled=st.session_state.generation_running)
                        
                        if btn_bg and pk_b:
                            if not gen_prompt.strip(): st.error("请输入大纲")
                            else:
                                client, m_name, m_key = engine.get_client(pk_b)
                                if not client: st.error("API Key 未配置")
                                else:
                                    st.session_state.generation_running = True
                                    ph = st.empty(); cnt = 0
                                    ensure_log_file(); log_operation("AI批量", f"开始批量 {len(target_chaps)} 章")
                                    batch_hard_limit = len_b * 1000 * 1.5

                                    try:
                                        for idx, ch in enumerate(target_chaps):
                                            if not st.session_state.generation_running: ph.warning("已停止"); break
                                            ph.info(f"⏳ ({idx+1}/{len(target_chaps)}) 生成：{ch['title']}...")
                                            batch_context = get_full_context(db_mgr, current_book['id'], ch['id'])
                                            combined_outline = f"{gen_prompt}\n{ch.get('summary', '')}"
                                            final_batch_prompt = f"{batch_context}\n\n【批量指令】{combined_outline}\n【章节】{ch['title']}\n【要求】字数 {len_b*1000} 字左右。"
                                            db_mgr.execute("UPDATE chapters SET summary=? WHERE id=?", (combined_outline, ch['id']))
                                            full_c = ""
                                            ok, stream = engine.generate_content_from_outline_ai_stream(ch['id'], final_batch_prompt, current_book, len_b*1000, client, m_name, m_key)
                                            if ok:
                                                for chunk in stream:
                                                    content_text = safe_get_content(chunk)
                                                    if content_text:
                                                        full_c += content_text
                                                        if len(full_c) > batch_hard_limit: break 
                                                db_mgr.execute("UPDATE chapters SET content=? WHERE id=?", (full_c, ch['id']))
                                                record_token_usage(provider="AI", model=m_name, tokens=len(full_c), action_name="批量生成", book_title=current_book['title'])
                                                cnt += 1
                                            else: st.error(f"失败: {ch['title']}")
                                        # 🔥 [修改 7] 替换 update_book_timestamp
                                        force_update_book_time(db_mgr, current_book['id'])
                                        ph.success(f"🎉 完成！共 {cnt} 章")
                                    except Exception as e: ph.error(f"错误: {e}")
                                    finally: st.session_state.generation_running = False; time.sleep(2); st.rerun()

        # ======================================================================
        # TAB 3: 写作辅助
        # ======================================================================
        with tab_assist:
            all_chaps_in_book = db_mgr.query("SELECT c.id, c.title, c.content, c.summary FROM chapters c JOIN volumes v ON c.volume_id = v.id JOIN parts p ON v.part_id = p.id WHERE p.book_id = ? ORDER BY p.sort_order, v.sort_order, c.sort_order", (current_book['id'],))
            
            if not all_chaps_in_book: st.info("请先创建章节。")
            else:
                chap_options = {c['title']: c['id'] for c in all_chaps_in_book}
                default_idx = 0
                if current_chapter and current_chapter['title'] in chap_options:
                    default_idx = list(chap_options.keys()).index(current_chapter['title'])
                target_chap_title = st.selectbox("🎯 选择目标章节", list(chap_options.keys()), index=default_idx)
                target_chap_id = chap_options[target_chap_title]
                target_chap_data = next((c for c in all_chaps_in_book if c['id'] == target_chap_id), None)
                target_content = target_chap_data['content'] if target_chap_data else ""
                
                def_cf = get_safe_model_default("write_logic_assist", "GPT_4o_Mini")
                def_rw = get_safe_model_default("write_rewrite", "DSK_V3")
                as_cf = st.session_state.model_assignments.get("write_logic_assist", def_cf)
                as_rw = st.session_state.model_assignments.get("write_rewrite", def_rw)
                
                st.subheader("🔎 矛盾检测")
                m_info = MODEL_MAPPING.get(as_cf, {'name': '未配置或无效'})
                st.caption(f"模型: **{m_info['name']}**")
                if st.button("🚨 检测设定冲突", use_container_width=True):
                    client, m_name, m_key = engine.get_client(as_cf)
                    if not client: st.error("API Key 未配置")
                    else:
                        with st.spinner("正在对比设定集与前文..."):
                            ensure_log_file(); log_operation("AI辅助", f"矛盾检测: {target_chap_title}")
                            assist_context = get_full_context(db_mgr, current_book['id'], target_chap_id)
                            final_input = f"{assist_context}\n\n【待检测正文】\n{target_content}"
                            rep = engine.analyze_chapter_conflict(final_input, current_book, client, m_name, m_key)
                            if isinstance(rep, tuple): rep = rep[1]
                            st.session_state[f"conflict_report_{target_chap_id}"] = rep
                        st.success("完成")
                rep_val = st.session_state.get(f"conflict_report_{target_chap_id}", "")
                if rep_val: st.info("检测报告："); st.text_area("report", rep_val, height=150, disabled=True, label_visibility="collapsed")
                
                st.subheader("🔄 一键重写")
                
                c_style_1, c_style_2 = st.columns([2, 1])
                with c_style_1:
                    style_rows = db_mgr.query("SELECT content FROM plots WHERE status='StyleDNA' AND book_id=?", (current_book['id'],))
                    styles = [f"🎨 风格: {s['content']}" for s in style_rows]
                    setting_rows = db_mgr.query("SELECT content FROM plots WHERE status LIKE 'Setting_%' AND book_id=?", (current_book['id'],))
                    settings = []
                    setting_content_map = {}
                    for s in setting_rows:
                        title = s['content'].split('\n')[0].strip()[:20]
                        label = f"🌍 设定: {title}"
                        settings.append(label)
                        setting_content_map[label] = s['content']
                    
                    all_opts = ["(不使用额外参考)"] + styles + settings
                    st.markdown('<label class="custom-select-label">🎨 参考风格/设定 (知识库)</label>', unsafe_allow_html=True)
                    sel_opt = st.selectbox("参考风格/设定", all_opts, label_visibility="collapsed")
                    
                    target_style_content = ""
                    if sel_opt != "(不使用额外参考)":
                        if sel_opt.startswith("🎨"): target_style_content = sel_opt.replace("🎨 风格: ", "")
                        elif sel_opt.startswith("🌍"): target_style_content = setting_content_map.get(sel_opt, "")

                m_info_rw = MODEL_MAPPING.get(as_rw, {'name': '未配置或无效'})
                with c_style_2:
                    st.caption(f"模型: **{m_info_rw['name']}**")
                    st.markdown("<div style='height: 28px'></div>", unsafe_allow_html=True) 
                
                if st.button("🚀 根据建议/设定重写本章", type="primary", use_container_width=True):
                    if not target_content: st.error("章节内容为空")
                    else:
                        client, m_name, m_key = engine.get_client(as_rw)
                        if not client: st.error("API Key 未配置")
                        else:
                            with st.spinner("AI 正在重塑章节..."):
                                ensure_log_file(); log_operation("AI辅助", f"一键重写: {target_chap_title}")
                                report_context = st.session_state.get(f"conflict_report_{target_chap_id}", "无特殊报告")
                                
                                res = engine.rewrite_chapter_ai(target_content, report_context, target_style_content, client, m_name, m_key)
                                
                                if isinstance(res, tuple): res = res[1]
                                st.session_state[f"rewritten_content_{target_chap_id}"] = res
                                record_token_usage(provider="AI", model=m_name, tokens=len(res) if res else 0, action_name="章节重写", book_title=current_book['title'])
                            st.success("完成")
                rw_val = st.session_state.get(f"rewritten_content_{target_chap_id}", "")
                if rw_val:
                    st.markdown("##### 预览重写结果")
                    st.text_area("preview", rw_val, height=300, label_visibility="collapsed")
                    if st.button(f"✅ 覆盖【{target_chap_title}】原内容", use_container_width=True):
                        db_mgr.execute("UPDATE chapters SET content=? WHERE id=?", (rw_val, target_chap_id))
                        # 🔥 [修改 8] 替换 update_book_timestamp
                        force_update_book_time(db_mgr, current_book['id'])
                        if current_chapter and target_chap_id == current_chapter['id']:
                            st.session_state[f"chapter_content_{target_chap_id}"] = rw_val
                        log_operation("更新章节", f"应用重写: {target_chap_title}")
                        st.session_state.rerun_flag = True; st.rerun()