import { Component, OnInit, ViewChild } from '@angular/core';
import { AbstractControl, FormGroup, ValidationErrors, Validators } from '@angular/forms';
import { ActivatedRoute, Router } from '@angular/router';

import _ from 'lodash';
import { concat as observableConcat, forkJoin as observableForkJoin, Observable } from 'rxjs';

import { RgwUserService } from '~/app/shared/api/rgw-user.service';
import { ActionLabelsI18n, URLVerbs, USER } from '~/app/shared/constants/app.constants';
import { Icons } from '~/app/shared/enum/icons.enum';
import { NotificationType } from '~/app/shared/enum/notification-type.enum';
import { CdForm } from '~/app/shared/forms/cd-form';
import { CdFormBuilder } from '~/app/shared/forms/cd-form-builder';
import { CdFormGroup } from '~/app/shared/forms/cd-form-group';
import { CdValidators } from '~/app/shared/forms/cd-validators';
import { FormatterService } from '~/app/shared/services/formatter.service';
import { NotificationService } from '~/app/shared/services/notification.service';
import { RgwUserCapabilities } from '../models/rgw-user-capabilities';
import { RgwUserCapability } from '../models/rgw-user-capability';
import { RgwUserS3Key } from '../models/rgw-user-s3-key';
import { RgwUserSubuser } from '../models/rgw-user-subuser';
import { RgwUserSwiftKey } from '../models/rgw-user-swift-key';
import { RgwUserCapabilityModalComponent } from '../rgw-user-capability-modal/rgw-user-capability-modal.component';
import { RgwUserS3KeyModalComponent } from '../rgw-user-s3-key-modal/rgw-user-s3-key-modal.component';
import { RgwUserSubuserModalComponent } from '../rgw-user-subuser-modal/rgw-user-subuser-modal.component';
import { RgwUserSwiftKeyModalComponent } from '../rgw-user-swift-key-modal/rgw-user-swift-key-modal.component';
import { RgwRateLimitComponent } from '../rgw-rate-limit/rgw-rate-limit.component';
import { RgwRateLimitConfig } from '../models/rgw-rate-limit';
import { ModalCdsService } from '~/app/shared/services/modal-cds.service';
import { RgwUserAccountsService } from '~/app/shared/api/rgw-user-accounts.service';
import { Account } from '../models/rgw-user-accounts';
import { ManagedPolicyArnMap, ManagedPolicyName, RGW } from '../utils/constants';
import { ComboBoxItem } from '~/app/shared/models/combo-box.model';

@Component({
  selector: 'cd-rgw-user-form',
  templateUrl: './rgw-user-form.component.html',
  styleUrls: ['./rgw-user-form.component.scss']
})
export class RgwUserFormComponent extends CdForm implements OnInit {
  userForm: CdFormGroup;
  editing = false;
  submitObservables: Observable<Object>[] = [];
  icons = Icons;
  subusers: RgwUserSubuser[] = [];
  s3Keys: RgwUserS3Key[] = [];
  swiftKeys: RgwUserSwiftKey[] = [];
  capabilities: RgwUserCapability[] = [];
  uid: string;
  action: string;
  resource: string;
  subuserLabel: string;
  s3keyLabel: string;
  capabilityLabel: string;
  usernameExists: boolean;
  showTenant = false;
  previousTenant: string = null;
  @ViewChild(RgwRateLimitComponent, { static: false }) rateLimitComponent!: RgwRateLimitComponent;
  accounts: Account[] = [];
  initialUserPolicies: string[] = [];
  managedPolicies: ComboBoxItem[] = [
    {
      content: ManagedPolicyName.AmazonS3FullAccess,
      name: ManagedPolicyArnMap[ManagedPolicyName.AmazonS3FullAccess],
      selected: false
    },
    {
      content: ManagedPolicyName.AmazonS3ReadOnlyAccess,
      name: ManagedPolicyArnMap[ManagedPolicyName.AmazonS3ReadOnlyAccess],
      selected: false
    }
  ];

  constructor(
    private formBuilder: CdFormBuilder,
    private route: ActivatedRoute,
    private router: Router,
    private rgwUserService: RgwUserService,
    private modalService: ModalCdsService,
    private notificationService: NotificationService,
    public actionLabels: ActionLabelsI18n,
    private rgwUserAccountService: RgwUserAccountsService
  ) {
    super();
    this.resource = $localize`user`;
    this.subuserLabel = $localize`subuser`;
    this.s3keyLabel = $localize`S3 Key`;
    this.capabilityLabel = $localize`capability`;
    this.editing = this.router.url.startsWith(`/rgw/user/${URLVerbs.EDIT}`);
    this.action = this.editing ? this.actionLabels.EDIT : this.actionLabels.CREATE;
    this.createForm();
  }

  createForm() {
    this.userForm = this.formBuilder.group({
      // General
      user_id: [
        null,
        [Validators.required, Validators.pattern(/^[a-zA-Z0-9!@#%^&*()._-]+$/)],
        this.editing
          ? []
          : [
              CdValidators.unique(this.rgwUserService.exists, this.rgwUserService, () =>
                this.userForm.getValue('tenant')
              )
            ]
      ],
      show_tenant: [this.editing],
      tenant: [
        null,
        [Validators.pattern(/^[a-zA-Z0-9_]+$/)],
        this.editing
          ? []
          : [
              CdValidators.unique(
                this.rgwUserService.exists,
                this.rgwUserService,
                () => this.userForm.getValue('user_id'),
                true
              )
            ]
      ],
      display_name: [
        null,
        [Validators.required, Validators.pattern(/^[a-zA-Z0-9!@#%^&*()._ -]+$/)]
      ],
      email: [
        null,
        [CdValidators.email],
        [CdValidators.unique(this.rgwUserService.emailExists, this.rgwUserService)]
      ],
      account_id: [null, [this.tenantedAccountValidator.bind(this)]],
      account_root_user: [false],
      account_policies: [[]],
      max_buckets_mode: [1],
      max_buckets: [
        1000,
        [CdValidators.requiredIf({ max_buckets_mode: '1' }), CdValidators.number(false)]
      ],
      system: [false],
      suspended: [false],
      // S3 key
      generate_key: [true],
      access_key: [null, [CdValidators.requiredIf({ generate_key: false })]],
      secret_key: [null, [CdValidators.requiredIf({ generate_key: false })]],
      // User quota
      user_quota_enabled: [false],
      user_quota_max_size_unlimited: [true],
      user_quota_max_size: [
        null,
        [
          CdValidators.composeIf(
            {
              user_quota_enabled: true,
              user_quota_max_size_unlimited: false
            },
            [Validators.required, this.quotaMaxSizeValidator]
          )
        ]
      ],
      user_quota_max_objects_unlimited: [true],
      user_quota_max_objects: [
        null,
        [
          CdValidators.requiredIf({
            user_quota_enabled: true,
            user_quota_max_objects_unlimited: false
          })
        ]
      ],
      // Bucket quota
      bucket_quota_enabled: [false],
      bucket_quota_max_size_unlimited: [true],
      bucket_quota_max_size: [
        null,
        [
          CdValidators.composeIf(
            {
              bucket_quota_enabled: true,
              bucket_quota_max_size_unlimited: false
            },
            [Validators.required, this.quotaMaxSizeValidator]
          )
        ]
      ],
      bucket_quota_max_objects_unlimited: [true],
      bucket_quota_max_objects: [
        null,
        [
          CdValidators.requiredIf({
            bucket_quota_enabled: true,
            bucket_quota_max_objects_unlimited: false
          })
        ]
      ]
    });
  }

  ngOnInit() {
    // Process route parameters.
    this.route.params.subscribe((params: { uid: string }) => {
      if (!params.hasOwnProperty('uid')) {
        return;
      }
      const uid = decodeURIComponent(params.uid);
      // Load the user and quota information.
      const observables = [];
      observables.push(this.rgwUserService.get(uid));
      observables.push(this.rgwUserService.getQuota(uid));
      observableForkJoin(observables).subscribe(
        (resp: any[]) => {
          // Get the default values.
          const defaults = _.clone(this.userForm.value);
          // Extract the values displayed in the form.
          let value = _.pick(resp[0], _.keys(this.userForm.value));
          // Map the max. buckets values.
          switch (value['max_buckets']) {
            case -1:
              value['max_buckets_mode'] = -1;
              value['max_buckets'] = '';
              break;
            case 0:
              value['max_buckets_mode'] = 0;
              value['max_buckets'] = '';
              break;
            default:
              value['max_buckets_mode'] = 1;
              break;
          }
          // Map the quota values.
          [USER, 'bucket'].forEach((type) => {
            const quota = resp[1][type + '_quota'];
            value[type + '_quota_enabled'] = quota.enabled;
            if (quota.max_size < 0) {
              value[type + '_quota_max_size_unlimited'] = true;
              value[type + '_quota_max_size'] = null;
            } else {
              value[type + '_quota_max_size_unlimited'] = false;
              value[type + '_quota_max_size'] = `${quota.max_size} B`;
            }
            if (quota.max_objects < 0) {
              value[type + '_quota_max_objects_unlimited'] = true;
              value[type + '_quota_max_objects'] = null;
            } else {
              value[type + '_quota_max_objects_unlimited'] = false;
              value[type + '_quota_max_objects'] = quota.max_objects;
            }
          });

          // Merge with default values.
          value = _.merge(defaults, value);
          // Update the form.
          this.userForm.setValue(value);

          if (this.userForm.getValue('account_id')) {
            this.userForm.get('account_id').disable();
          } else {
            this.userForm.get('account_id').setValue(null);
          }
          const isAccountRoot: boolean = resp[0]['type'] !== RGW;
          this.userForm.get('account_root_user').setValue(isAccountRoot);
          // Get the sub users.
          this.subusers = resp[0].subusers;

          // Get the keys.
          this.s3Keys = resp[0].keys;
          this.swiftKeys = resp[0].swift_keys;

          // Process the capabilities.
          const mapPerm = { 'read, write': '*' };
          resp[0].caps.forEach((cap: any) => {
            if (cap.perm in mapPerm) {
              cap.perm = mapPerm[cap.perm];
            }
          });
          this.capabilities = resp[0].caps;
          this.uid = this.getUID();
          this.initialUserPolicies = resp[0].managed_user_policies ?? [];

          this.managedPolicies.forEach((policy) => {
            policy.selected = this.initialUserPolicies.includes(policy.name);
          });

          // Optionally, update form control with selected items
          const selectedItems = this.managedPolicies.filter((p) => p.selected).map((p) => p.name);
          this.userForm.get('account_policies')?.setValue(selectedItems);
        },
        () => {
          this.loadingError();
        }
      );
    });
    this.rgwUserAccountService.list(true).subscribe(
      (accounts: Account[]) => {
        this.accounts = accounts;
        if (!this.editing) {
          // needed to disable checkbox on create form load
          this.userForm.get('account_id').reset();
        }
        this.loadingReady();
      },
      () => {
        this.loadingError();
      }
    );
    this.userForm.get('account_id').valueChanges.subscribe((value) => {
      if (!value) {
        this.userForm
          .get('display_name')
          .setValidators([Validators.pattern(/^[a-zA-Z0-9!@#%^&*()._ -]+$/), Validators.required]);
        this.userForm.get('display_name').updateValueAndValidity();
        this.userForm.get('account_root_user').disable();
      } else {
        this.userForm
          .get('display_name')
          .setValidators([Validators.pattern(/^[\w+=,.@-]+$/), Validators.required]);
        this.userForm.get('display_name').updateValueAndValidity();
        this.userForm.get('account_root_user').enable();
      }
    });
  }

  multiSelector(event: ComboBoxItem[]) {
    this.managedPolicies.forEach((policy) => {
      policy.selected = !!event.find((selected) => selected.name === policy.name);
    });
  }

  tenantedAccountValidator(control: AbstractControl): ValidationErrors | null {
    if (this?.userForm?.getValue('tenant') && this.accounts.length > 0) {
      const index: number = this.accounts.findIndex(
        (account: Account) => account.id === control.value
      );
      if (index !== -1) {
        return this.userForm.getValue('tenant') !== this.accounts[index].tenant
          ? { tenantedAccount: true }
          : null;
      }
    }
    return null;
  }

  rateLimitFormInit(rateLimitForm: FormGroup) {
    this.userForm.addControl('rateLimit', rateLimitForm);
  }

  goToListView() {
    this.router.navigate(['/rgw/user']);
  }

  onSubmit() {
    this.uid = this.getUID();
    let notificationTitle: string;
    // Exit immediately if the form isn't dirty.
    if (this.userForm.pristine && this.rateLimitComponent.form.pristine) {
      this.goToListView();
      return;
    }
    if (this.editing) {
      // Edit
      if (this._isGeneralDirty()) {
        const args = this._getUpdateArgs();
        this.submitObservables.push(this.rgwUserService.update(this.uid, args));
      }
      notificationTitle = $localize`Updated Object Gateway user '${this.uid}'`;
    } else {
      // Add
      const args = this._getCreateArgs();
      this.submitObservables.push(this.rgwUserService.create(args));
      notificationTitle = $localize`Created Object Gateway user '${this.uid}'`;
    }
    // Check if user quota has been modified.
    if (this._isUserQuotaDirty()) {
      const userQuotaArgs = this._getUserQuotaArgs();
      this.submitObservables.push(this.rgwUserService.updateQuota(this.uid, userQuotaArgs));
    }
    // Check if bucket quota has been modified.
    if (this._isBucketQuotaDirty()) {
      const bucketQuotaArgs = this._getBucketQuotaArgs();
      this.submitObservables.push(this.rgwUserService.updateQuota(this.uid, bucketQuotaArgs));
    }

    // Check if user ratelimit has been modified.
    const ratelimitvalue: RgwRateLimitConfig = this.rateLimitComponent.getRateLimitFormValue();
    if (!!ratelimitvalue) {
      this.submitObservables.push(
        this.rgwUserService.updateUserRateLimit(this.userForm.getValue('user_id'), ratelimitvalue)
      );
    }
    // Finally execute all observables one by one in serial.
    observableConcat(...this.submitObservables).subscribe({
      error: () => {
        // Reset the 'Submit' button.
        this.userForm.setErrors({ cdSubmitButton: true });
      },
      complete: () => {
        this.notificationService.show(NotificationType.success, notificationTitle);
        this.goToListView();
      }
    });
  }

  updateFieldsWhenTenanted() {
    this.showTenant = this.userForm.getValue('show_tenant');
    if (!this.showTenant) {
      this.userForm.get('user_id').markAsUntouched();
      this.userForm.get('tenant').patchValue(this.previousTenant);
    } else {
      this.userForm.get('user_id').markAsTouched();
      this.previousTenant = this.userForm.get('tenant').value;
      this.userForm.get('tenant').patchValue(null);
    }
  }

  getUID(): string {
    let uid = this.userForm.getValue('user_id');
    const tenant = this.userForm?.getValue('tenant');
    if (tenant && tenant.length > 0) {
      uid = `${this.userForm.getValue('tenant')}$${uid}`;
    }
    return uid;
  }

  /**
   * Validate the quota maximum size, e.g. 1096, 1K, 30M or 1.9MiB.
   */
  quotaMaxSizeValidator(control: AbstractControl): ValidationErrors | null {
    return new FormatterService().performValidation(
      control,
      '^(\\d+(\\.\\d+)?)\\s*(B|K(B|iB)?|M(B|iB)?|G(B|iB)?|T(B|iB)?)?$',
      { quotaMaxSize: true },
      'quota'
    );
  }

  /**
   * Add/Update a subuser.
   */
  setSubuser(subuser: RgwUserSubuser, index?: number) {
    const mapPermissions: Record<string, string> = {
      'full-control': 'full',
      'read-write': 'readwrite'
    };
    const uid = this.getUID();
    const args = {
      subuser: subuser.id,
      access:
        subuser.permissions in mapPermissions
          ? mapPermissions[subuser.permissions]
          : subuser.permissions,
      key_type: 'swift',
      secret_key: subuser.secret_key,
      generate_secret: subuser.generate_secret ? 'true' : 'false'
    };
    this.submitObservables.push(this.rgwUserService.createSubuser(uid, args));
    if (_.isNumber(index)) {
      // Modify
      // Create an observable to modify the subuser when the form is submitted.
      this.subusers[index] = subuser;
    } else {
      // Add
      // Create an observable to add the subuser when the form is submitted.
      this.subusers.push(subuser);
      // Add a Swift key. If the secret key is auto-generated, then visualize
      // this to the user by displaying a notification instead of the key.
      this.swiftKeys.push({
        user: subuser.id,
        secret_key: subuser.generate_secret ? 'Apply your changes first...' : subuser.secret_key
      });
    }
    // Mark the form as dirty to be able to submit it.
    this.userForm.markAsDirty();
  }

  /**
   * Delete a subuser.
   * @param {number} index The subuser to delete.
   */
  deleteSubuser(index: number) {
    const subuser = this.subusers[index];
    // Create an observable to delete the subuser when the form is submitted.
    this.submitObservables.push(this.rgwUserService.deleteSubuser(this.getUID(), subuser.id));
    // Remove the associated S3 keys.
    this.s3Keys = this.s3Keys.filter((key) => {
      return key.user !== subuser.id;
    });
    // Remove the associated Swift keys.
    this.swiftKeys = this.swiftKeys.filter((key) => {
      return key.user !== subuser.id;
    });
    // Remove the subuser to update the UI.
    this.subusers.splice(index, 1);
    // Mark the form as dirty to be able to submit it.
    this.userForm.markAsDirty();
  }

  /**
   * Add/Update a capability.
   */
  setCapability(cap: RgwUserCapability, index?: number) {
    const uid = this.getUID();
    if (_.isNumber(index)) {
      // Modify
      const oldCap = this.capabilities[index];
      // Note, the RadosGW Admin OPS API does not support the modification of
      // user capabilities. Because of that it is necessary to delete it and
      // then to re-add the capability with its new value/permission.
      this.submitObservables.push(
        this.rgwUserService.deleteCapability(uid, oldCap.type, oldCap.perm)
      );
      this.submitObservables.push(this.rgwUserService.addCapability(uid, cap.type, cap.perm));
      this.capabilities[index] = cap;
    } else {
      // Add
      // Create an observable to add the capability when the form is submitted.
      this.submitObservables.push(this.rgwUserService.addCapability(uid, cap.type, cap.perm));
      this.capabilities = [...this.capabilities, cap]; // Notify Angular CD
    }
    // Mark the form as dirty to be able to submit it.
    this.userForm.markAsDirty();
  }

  /**
   * Delete the given capability:
   * - Delete it from the local array to update the UI
   * - Create an observable that will be executed on form submit
   * @param {number} index The capability to delete.
   */
  deleteCapability(index: number) {
    const cap = this.capabilities[index];
    // Create an observable to delete the capability when the form is submitted.
    this.submitObservables.push(
      this.rgwUserService.deleteCapability(this.getUID(), cap.type, cap.perm)
    );
    // Remove the capability to update the UI.
    this.capabilities.splice(index, 1);
    this.capabilities = [...this.capabilities]; // Notify Angular CD
    // Mark the form as dirty to be able to submit it.
    this.userForm.markAsDirty();
  }

  hasAllCapabilities(capabilities: RgwUserCapability[]) {
    return !_.difference(RgwUserCapabilities.getAll(), _.map(capabilities, 'type')).length;
  }

  /**
   * Add/Update a S3 key.
   */
  setS3Key(key: RgwUserS3Key, index?: number) {
    if (_.isNumber(index)) {
      // Modify
      // Nothing to do here at the moment.
    } else {
      // Add
      // Split the key's user name into its user and subuser parts.
      const userMatches = key.user.match(/([^:]+)(:(.+))?/);
      // Create an observable to add the S3 key when the form is submitted.
      const uid = userMatches[1];
      const args = {
        subuser: userMatches[2] ? userMatches[3] : '',
        generate_key: key.generate_key ? 'true' : 'false'
      };
      if (args['generate_key'] === 'false') {
        if (!_.isNil(key.access_key)) {
          args['access_key'] = key.access_key;
        }
        if (!_.isNil(key.secret_key)) {
          args['secret_key'] = key.secret_key;
        }
      }
      this.submitObservables.push(this.rgwUserService.addS3Key(uid, args));
      // If the access and the secret key are auto-generated, then visualize
      // this to the user by displaying a notification instead of the key.
      this.s3Keys.push({
        user: key.user,
        access_key: key.generate_key ? 'Apply your changes first...' : key.access_key,
        secret_key: key.generate_key ? 'Apply your changes first...' : key.secret_key
      });
    }
    // Mark the form as dirty to be able to submit it.
    this.userForm.markAsDirty();
  }

  /**
   * Delete a S3 key.
   * @param {number} index The S3 key to delete.
   */
  deleteS3Key(index: number) {
    const key = this.s3Keys[index];
    // Create an observable to delete the S3 key when the form is submitted.
    this.submitObservables.push(this.rgwUserService.deleteS3Key(this.getUID(), key.access_key));
    // Remove the S3 key to update the UI.
    this.s3Keys.splice(index, 1);
    // Mark the form as dirty to be able to submit it.
    this.userForm.markAsDirty();
  }

  /**
   * Show the specified subuser in a modal dialog.
   * @param {number | undefined} index The subuser to show.
   */
  showSubuserModal(index?: number) {
    const uid = this.getUID();
    const modalRef = this.modalService.show(RgwUserSubuserModalComponent);
    if (_.isNumber(index)) {
      // Edit
      const subuser = this.subusers[index];
      modalRef.setEditing();
      modalRef.setValues(uid, subuser.id, subuser.permissions);
    } else {
      // Add
      modalRef.setEditing(false);
      modalRef.setValues(uid);
      modalRef.setSubusers(this.subusers);
    }
    modalRef.submitAction.subscribe((subuser: RgwUserSubuser) => {
      this.setSubuser(subuser, index);
    });
  }

  /**
   * Show the specified S3 key in a modal dialog.
   * @param {number | undefined} index The S3 key to show.
   */
  showS3KeyModal(index?: number) {
    const modalRef = this.modalService.show(RgwUserS3KeyModalComponent);
    if (_.isNumber(index)) {
      // View
      const key = this.s3Keys[index];
      modalRef.setViewing();
      modalRef.setValues(key.user, key.access_key, key.secret_key);
    } else {
      // Add
      const candidates = this._getS3KeyUserCandidates();
      modalRef.setViewing(false);
      modalRef.setUserCandidates(candidates);
      modalRef.submitAction.subscribe((key: RgwUserS3Key) => {
        this.setS3Key(key);
      });
    }
  }

  /**
   * Show the specified Swift key in a modal dialog.
   * @param {number} index The Swift key to show.
   */
  showSwiftKeyModal(index: number) {
    const modalRef = this.modalService.show(RgwUserSwiftKeyModalComponent);
    const key = this.swiftKeys[index];
    modalRef.setValues(key.user, key.secret_key);
  }

  /**
   * Show the specified capability in a modal dialog.
   * @param {number | undefined} index The S3 key to show.
   */
  showCapabilityModal(index?: number) {
    const modalRef = this.modalService.show(RgwUserCapabilityModalComponent);
    if (_.isNumber(index)) {
      // Edit
      const cap = this.capabilities[index];
      modalRef.setEditing();
      modalRef.setValues(cap.type, cap.perm);
    } else {
      // Add
      modalRef.setEditing(false);
      modalRef.setCapabilities(this.capabilities);
    }
    modalRef.submitAction.subscribe((cap: RgwUserCapability) => {
      this.setCapability(cap, index);
    });
  }

  /**
   * Check if the general user settings (display name, email, ...) have been modified.
   * @return {Boolean} Returns TRUE if the general user settings have been modified.
   */
  private _isGeneralDirty(): boolean {
    return [
      'display_name',
      'email',
      'max_buckets_mode',
      'max_buckets',
      'system',
      'suspended',
      'account_id',
      'account_root_user',
      'account_policies'
    ].some((path) => {
      return this.userForm.get(path).dirty;
    });
  }

  /**
   * Check if the user quota has been modified.
   * @return {Boolean} Returns TRUE if the user quota has been modified.
   */
  private _isUserQuotaDirty(): boolean {
    return [
      'user_quota_enabled',
      'user_quota_max_size_unlimited',
      'user_quota_max_size',
      'user_quota_max_objects_unlimited',
      'user_quota_max_objects'
    ].some((path) => {
      return this.userForm.get(path).dirty;
    });
  }

  /**
   * Check if the bucket quota has been modified.
   * @return {Boolean} Returns TRUE if the bucket quota has been modified.
   */
  private _isBucketQuotaDirty(): boolean {
    return [
      'bucket_quota_enabled',
      'bucket_quota_max_size_unlimited',
      'bucket_quota_max_size',
      'bucket_quota_max_objects_unlimited',
      'bucket_quota_max_objects'
    ].some((path) => {
      return this.userForm.get(path).dirty;
    });
  }

  /**
   * Helper function to get the arguments of the API request when a new
   * user is created.
   */
  private _getCreateArgs() {
    const result = {
      uid: this.getUID(),
      display_name: this.userForm.getValue('display_name'),
      system: this.userForm.getValue('system'),
      suspended: this.userForm.getValue('suspended'),
      email: '',
      max_buckets: this.userForm.getValue('max_buckets'),
      generate_key: this.userForm.getValue('generate_key'),
      access_key: '',
      secret_key: ''
    };
    const email = this.userForm.getValue('email');
    if (_.isString(email) && email.length > 0) {
      _.merge(result, { email: email });
    }
    const generateKey = this.userForm.getValue('generate_key');
    if (!generateKey) {
      _.merge(result, {
        generate_key: false,
        access_key: this.userForm.getValue('access_key'),
        secret_key: this.userForm.getValue('secret_key')
      });
    }
    const maxBucketsMode = parseInt(this.userForm.getValue('max_buckets_mode'), 10);
    if (_.includes([-1, 0], maxBucketsMode)) {
      // -1 => Disable bucket creation.
      //  0 => Unlimited bucket creation.
      _.merge(result, { max_buckets: maxBucketsMode });
    }
    if (this.userForm.getValue('account_id')) {
      _.merge(result, {
        account_id: this.userForm.getValue('account_id'),
        account_root_user: this.userForm.getValue('account_root_user')
      });
    }
    const accountPolicies = this._getAccountManagedPolicies();
    if (this.userForm.getValue('account_id') && !this.userForm.getValue('account_root_user')) {
      _.merge(result, { account_policies: accountPolicies });
    }
    return result;
  }

  /**
   * Helper function to get the arguments for the API request when the user
   * configuration has been modified.
   */
  private _getUpdateArgs() {
    const result: Record<string, any> = {};
    const keys = ['display_name', 'email', 'max_buckets', 'system', 'suspended'];
    for (const key of keys) {
      result[key] = this.userForm.getValue(key);
    }
    if (this.userForm.getValue('account_id')) {
      result['account_id'] = this.userForm.getValue('account_id');
      result['account_root_user'] = this.userForm.getValue('account_root_user');
    }
    const maxBucketsMode = parseInt(this.userForm.getValue('max_buckets_mode'), 10);
    if (_.includes([-1, 0], maxBucketsMode)) {
      // -1 => Disable bucket creation.
      //  0 => Unlimited bucket creation.
      result['max_buckets'] = maxBucketsMode;
    }
    const accountPolicies = this._getAccountManagedPolicies();
    if (this.userForm.getValue('account_id') && !this.userForm.getValue('account_root_user')) {
      result['account_policies'] = accountPolicies;
    }
    return result;
  }

  /**
   * Helper function to get the arguments for the API request when the user
   * quota configuration has been modified.
   */
  _getUserQuotaArgs(): Record<string, any> {
    const result = {
      quota_type: USER,
      enabled: this.userForm.getValue('user_quota_enabled'),
      max_size_kb: -1,
      max_objects: -1
    };
    if (!this.userForm.getValue('user_quota_max_size_unlimited')) {
      // Convert the given value to bytes.
      const bytes = new FormatterService().toBytes(this.userForm.getValue('user_quota_max_size'));
      // Finally convert the value to KiB.
      result['max_size_kb'] = (bytes / 1024).toFixed(0) as any;
    }
    if (!this.userForm.getValue('user_quota_max_objects_unlimited')) {
      result['max_objects'] = this.userForm.getValue('user_quota_max_objects');
    }
    return result;
  }

  /**
   * Helper function to get the arguments for the API request when the bucket
   * quota configuration has been modified.
   */
  private _getBucketQuotaArgs(): Record<string, any> {
    const result = {
      quota_type: 'bucket',
      enabled: this.userForm.getValue('bucket_quota_enabled'),
      max_size_kb: -1,
      max_objects: -1
    };
    if (!this.userForm.getValue('bucket_quota_max_size_unlimited')) {
      // Convert the given value to bytes.
      const bytes = new FormatterService().toBytes(this.userForm.getValue('bucket_quota_max_size'));
      // Finally convert the value to KiB.
      result['max_size_kb'] = (bytes / 1024).toFixed(0) as any;
    }
    if (!this.userForm.getValue('bucket_quota_max_objects_unlimited')) {
      result['max_objects'] = this.userForm.getValue('bucket_quota_max_objects');
    }
    return result;
  }

  /**
   * Helper method to get the user candidates for S3 keys.
   * @returns {Array} Returns a list of user identifiers.
   */
  private _getS3KeyUserCandidates() {
    let result = [];
    // Add the current user id.
    const uid = this.getUID();
    if (_.isString(uid) && !_.isEmpty(uid)) {
      result.push(uid);
    }
    // Append the subusers.
    this.subusers.forEach((subUser) => {
      result.push(subUser.id);
    });
    // Note that it's possible to create multiple S3 key pairs for a user,
    // thus we append already configured users, too.
    this.s3Keys.forEach((key) => {
      result.push(key.user);
    });
    result = _.uniq(result);
    return result;
  }

  /**
   * Get the account managed policies to attach/detach.
   * @returns {Object} Returns an object with attach and detach arrays.
   */
  private _getAccountManagedPolicies() {
    const selectedPolicies = this.managedPolicies.filter((p) => p.selected).map((p) => p.name);

    const initialPolicies = this.initialUserPolicies;
    const toAttach = selectedPolicies.filter((p) => !initialPolicies.includes(p));
    const toDetach = initialPolicies.filter((p) => !selectedPolicies.includes(p));

    const payload = {
      attach: toAttach,
      detach: toDetach
    };

    return payload;
  }

  onMaxBucketsModeChange(mode: string) {
    if (mode === '1') {
      // If 'Custom' mode is selected, then ensure that the form field
      // 'Max. buckets' contains a valid value. Set it to default if
      // necessary.
      if (!this.userForm.get('max_buckets').valid) {
        this.userForm.patchValue({
          max_buckets: 1000
        });
      }
    }
  }

  goToCreateAccountForm() {
    this.router.navigate(['rgw/accounts/create']);
  }
}
