// -*- mode:C++; tab-width:8; c-basic-offset:2; indent-tabs-mode:nil -*-
// vim: ts=8 sw=2 sts=2 expandtab

#ifndef CEPH_MDS_METRIC_AGGREGATOR_H
#define CEPH_MDS_METRIC_AGGREGATOR_H

#include <map>
#include <set>
#include <thread>
#include <unordered_set>

#include "msg/msg_types.h" // for entity_inst_t
#include "msg/Dispatcher.h"
#include "common/ceph_mutex.h"
#include "common/perf_counters.h"
#include "include/common_fwd.h"

#include "mgr/Types.h" // for PerformanceCounters
#include "mgr/MetricTypes.h" // for MetricPayload
#include "mgr/MDSPerfMetricTypes.h"

#include "mdstypes.h"
#include "MDSPinger.h"

class MDSMap;
class MDSRank;
class MMDSMetrics;
struct Metrics;
class MgrClient;

class MetricAggregator : public Dispatcher {
public:
  MetricAggregator(CephContext *cct, MDSRank *mds, MgrClient *mgrc);

  int init();
  void shutdown();

  void notify_mdsmap(const MDSMap &mdsmap);

  Dispatcher::dispatch_result_t ms_dispatch2(const ref_t<Message> &m) override;

  void ms_handle_connect(Connection *c) override {
  }
  bool ms_handle_reset(Connection *c) override {
    return false;
  }
  void ms_handle_remote_reset(Connection *c) override {
  }
  bool ms_handle_refused(Connection *c) override {
    return false;
  }

private:
  // drop this lock when calling ->send_message_mds() else mds might
  // deadlock
  ceph::mutex lock = ceph::make_mutex("MetricAggregator::lock");
  CephContext *m_cct;
  MDSRank *mds;
  MgrClient *mgrc;

  // maintain a map of rank to list of clients so that when a rank
  // goes away we cull metrics of clients connected to that rank.
  std::map<mds_rank_t, std::unordered_set<entity_inst_t>> clients_by_rank;

  // user query to metrics map
  std::map<MDSPerfMetricQuery, std::map<MDSPerfMetricKey, PerformanceCounters>> query_metrics_map;

  MDSPinger mds_pinger;
  std::thread pinger;

  std::map<mds_rank_t, entity_addrvec_t> active_rank_addrs;

  bool stopping = false;

  PerfCounters *m_perf_counters;
  std::map<std::pair<entity_inst_t, mds_rank_t>, PerfCounters*> client_perf_counters;
  uint64_t subv_window_sec;
  std::unordered_map<std::string, SlidingWindowTracker<SubvolumeMetric>> subvolume_aggregated_metrics;
  std::map<std::string, PerfCounters*> subvolume_perf_counters;

  void handle_mds_metrics(const cref_t<MMDSMetrics> &m);

  void refresh_metrics_for_rank(const entity_inst_t &client, mds_rank_t rank,
                                const Metrics &metrics);
  void refresh_subvolume_metrics_for_rank(mds_rank_t rank, const std::vector<SubvolumeMetric> &metrics);
  void remove_metrics_for_rank(const entity_inst_t &client, mds_rank_t rank, bool remove);

  void cull_metrics_for_rank(mds_rank_t rank);

  void ping_all_active_ranks();

  void set_perf_queries(const ConfigPayload &config_payload);
  MetricPayload get_perf_reports();
};

#endif // CEPH_MDS_METRIC_AGGREGATOR_H
