#pragma once

#include <array>
#include <memory>
#include <optional>
#include <string>

#include "OpType.h"

/* Overview
 *
 * class IoOp
 *   Stores details for an I/O operation. Generated by IoSequences
 *   and applied by IoExerciser's
 */

namespace ceph {
namespace io_exerciser {

class IoOp {
 public:
  IoOp();
  virtual ~IoOp() = default;
  virtual std::string to_string(uint64_t block_size) const = 0;
  virtual constexpr OpType getOpType() const = 0;
};

template <OpType opType>
class TestOp : public IoOp {
 public:
  TestOp();
  constexpr OpType getOpType() const override { return opType; }
};

class DoneOp : public TestOp<OpType::Done> {
 public:
  DoneOp();
  static std::unique_ptr<DoneOp> generate();
  std::string to_string(uint64_t block_size) const override;
};

class BarrierOp : public TestOp<OpType::Barrier> {
 public:
  BarrierOp();
  static std::unique_ptr<BarrierOp> generate();
  std::string to_string(uint64_t block_size) const override;
};

class CreateOp : public TestOp<OpType::Create> {
 public:
  CreateOp(uint64_t size);
  static std::unique_ptr<CreateOp> generate(uint64_t size);
  std::string to_string(uint64_t block_size) const override;
  uint64_t size;
};

class RemoveOp : public TestOp<OpType::Remove> {
 public:
  RemoveOp();
  static std::unique_ptr<RemoveOp> generate();
  std::string to_string(uint64_t block_size) const override;
};

class ConsistencyOp : public TestOp<OpType::Consistency> {
  public:
   ConsistencyOp();
   static std::unique_ptr<ConsistencyOp> generate();
   std::string to_string(uint64_t block_size) const override;
 };

class SwapOp : public TestOp<OpType::Swap> {
 public:
  SwapOp();
  static std::unique_ptr<SwapOp> generate();
  std::string to_string(uint64_t block_size) const override;
};

class CopyOp : public TestOp<OpType::Copy> {
 public:
  CopyOp();
  static std::unique_ptr<CopyOp> generate();
  std::string to_string(uint64_t block_size) const override;
};

template <OpType opType, int numIOs>
class ReadWriteOp : public TestOp<opType> {
 public:
  std::array<uint64_t, numIOs> offset;
  std::array<uint64_t, numIOs> length;

 protected:
  ReadWriteOp(std::array<uint64_t, numIOs>&& offset,
              std::array<uint64_t, numIOs>&& length);
  std::string to_string(uint64_t block_size) const override;
};

class SingleReadOp : public ReadWriteOp<OpType::Read, 1> {
 public:
  SingleReadOp(uint64_t offset, uint64_t length);
  static std::unique_ptr<SingleReadOp> generate(uint64_t offset,
                                                uint64_t length);
};

class DoubleReadOp : public ReadWriteOp<OpType::Read2, 2> {
 public:
  DoubleReadOp(uint64_t offset1, uint64_t length1, uint64_t offset2,
               uint64_t length2);
  static std::unique_ptr<DoubleReadOp> generate(uint64_t offset1,
                                                uint64_t length1,
                                                uint64_t offset2,
                                                uint64_t length2);
};

class TripleReadOp : public ReadWriteOp<OpType::Read3, 3> {
 public:
  TripleReadOp(uint64_t offset1, uint64_t length1, uint64_t offset2,
               uint64_t length2, uint64_t offset3, uint64_t length3);
  static std::unique_ptr<TripleReadOp> generate(
      uint64_t offset1, uint64_t length1, uint64_t offset2, uint64_t length2,
      uint64_t offset3, uint64_t length3);
};

class SingleWriteOp : public ReadWriteOp<OpType::Write, 1> {
 public:
  SingleWriteOp(uint64_t offset, uint64_t length);
  static std::unique_ptr<SingleWriteOp> generate(uint64_t offset,
                                                 uint64_t length);
};

class DoubleWriteOp : public ReadWriteOp<OpType::Write2, 2> {
 public:
  DoubleWriteOp(uint64_t offset1, uint64_t length1, uint64_t offset2,
                uint64_t length2);
  static std::unique_ptr<DoubleWriteOp> generate(uint64_t offset1,
                                                 uint64_t length1,
                                                 uint64_t offset2,
                                                 uint64_t length2);
};

class TripleWriteOp : public ReadWriteOp<OpType::Write3, 3> {
 public:
  TripleWriteOp(uint64_t offset1, uint64_t length1, uint64_t offset2,
                uint64_t length2, uint64_t offset3, uint64_t length3);
  static std::unique_ptr<TripleWriteOp> generate(
      uint64_t offset1, uint64_t length1, uint64_t offset2, uint64_t length2,
      uint64_t offset3, uint64_t length3);
};

class SingleAppendOp : public ReadWriteOp<OpType::Append, 1> {
 public:
  SingleAppendOp(uint64_t length);
  static std::unique_ptr<SingleAppendOp> generate(uint64_t length);
};

class TruncateOp : public TestOp<OpType::Truncate> {
 public:
  TruncateOp(uint64_t size);
  static std::unique_ptr<TruncateOp> generate(uint64_t size);
  std::string to_string(uint64_t block_size) const override;
  uint64_t size;
};

class SingleFailedWriteOp : public ReadWriteOp<OpType::FailedWrite, 1> {
 public:
  SingleFailedWriteOp(uint64_t offset, uint64_t length);
  static std::unique_ptr<SingleFailedWriteOp> generate(uint64_t offset,
                                                       uint64_t length);
};

class DoubleFailedWriteOp : public ReadWriteOp<OpType::FailedWrite2, 2> {
 public:
  DoubleFailedWriteOp(uint64_t offset1, uint64_t length1, uint64_t offset2,
                      uint64_t length2);
  static std::unique_ptr<DoubleFailedWriteOp> generate(uint64_t offset1,
                                                       uint64_t length1,
                                                       uint64_t offset2,
                                                       uint64_t length2);
};

class TripleFailedWriteOp : public ReadWriteOp<OpType::FailedWrite3, 3> {
 public:
  TripleFailedWriteOp(uint64_t offset1, uint64_t length1, uint64_t offset2,
                      uint64_t length2, uint64_t offset3, uint64_t length3);
  static std::unique_ptr<TripleFailedWriteOp> generate(
      uint64_t offset1, uint64_t length1, uint64_t offset2, uint64_t length2,
      uint64_t offset3, uint64_t length3);
};

template <ceph::io_exerciser::OpType opType>
class InjectErrorOp : public TestOp<opType> {
 public:
  InjectErrorOp(int shard, const std::optional<uint64_t>& type,
                const std::optional<uint64_t>& when,
                const std::optional<uint64_t>& duration);

  std::string to_string(uint64_t block_size) const override;

  int shard;
  std::optional<uint64_t> type;
  std::optional<uint64_t> when;
  std::optional<uint64_t> duration;

 protected:
  virtual inline constexpr std::string_view get_inject_type_string() const = 0;
};

class InjectReadErrorOp : public InjectErrorOp<OpType::InjectReadError> {
 public:
  InjectReadErrorOp(int shard, const std::optional<uint64_t>& type,
                    const std::optional<uint64_t>& when,
                    const std::optional<uint64_t>& duration);

  static std::unique_ptr<InjectReadErrorOp> generate(
      int shard, const std::optional<uint64_t>& type,
      const std::optional<uint64_t>& when,
      const std::optional<uint64_t>& duration);

 protected:
  inline constexpr std::string_view get_inject_type_string() const override {
    return "read";
  }
};

class InjectWriteErrorOp : public InjectErrorOp<OpType::InjectWriteError> {
 public:
  InjectWriteErrorOp(int shard, const std::optional<uint64_t>& type,
                     const std::optional<uint64_t>& when,
                     const std::optional<uint64_t>& duration);

  static std::unique_ptr<InjectWriteErrorOp> generate(
      int shard, const std::optional<uint64_t>& type,
      const std::optional<uint64_t>& when,
      const std::optional<uint64_t>& duration);

 protected:
  inline constexpr std::string_view get_inject_type_string() const override {
    return "write";
  }
};

template <ceph::io_exerciser::OpType opType>
class ClearErrorInjectOp : public TestOp<opType> {
 public:
  ClearErrorInjectOp(int shard, const std::optional<uint64_t>& type);

  std::string to_string(uint64_t block_size) const override;

  int shard;
  std::optional<uint64_t> type;

 protected:
  virtual inline constexpr std::string_view get_inject_type_string() const = 0;
};

class ClearReadErrorInjectOp
    : public ClearErrorInjectOp<OpType::ClearReadErrorInject> {
 public:
  ClearReadErrorInjectOp(int shard, const std::optional<uint64_t>& type);

  static std::unique_ptr<ClearReadErrorInjectOp> generate(
      int shard, const std::optional<uint64_t>& type);

 protected:
  inline constexpr std::string_view get_inject_type_string() const override {
    return "read";
  }
};

class ClearWriteErrorInjectOp
    : public ClearErrorInjectOp<OpType::ClearWriteErrorInject> {
 public:
  ClearWriteErrorInjectOp(int shard, const std::optional<uint64_t>& type);

  static std::unique_ptr<ClearWriteErrorInjectOp> generate(
      int shard, const std::optional<uint64_t>& type);

 protected:
  inline constexpr std::string_view get_inject_type_string() const override {
    return "write";
  }
};
}  // namespace io_exerciser
}  // namespace ceph
