# 角色行为系统

## 概述

角色行为系统是游戏的核心机制，负责管理玩家和NPC的所有行动、时间流逝、状态变化和数值结算。该系统采用分阶段处理的方式，确保所有角色的行为能够按照正确的时间顺序执行。

## 系统架构

### 1. 核心组件

#### 1.1 主行为循环 (`character_behavior.py`)
- **`init_character_behavior()`**: 行为树总控制函数，管理整个游戏的行为循环
- **`character_behavior()`**: 单个角色的行为处理函数
- **`judge_character_status()`**: 校验并结算角色状态
- **`judge_character_status_time_over()`**: 判断行为是否已经结束

#### 1.2 行为数据结构 (`game_type.py`)
```python
class Behavior:
    start_time: datetime.datetime  # 行为开始时间
    duration: int                  # 行为持续时间(分钟)
    behavior_id: str              # 行为ID
    move_target: List[str]        # 移动目标坐标
    move_src: List[str]           # 移动出发坐标
```

#### 1.3 角色数据结构
```python
class Character:
    behavior: Behavior                    # 当前行为状态
    second_behavior: Dict[str, int]      # 二段行为状态
    last_behavior_id_list: List          # 历史行为列表
    state: int                           # 当前状态
    target_character_id: int             # 交互对象ID
```

### 2. 行为处理流程

#### 2.1 主循环流程
```
┌─────────────────────────────────────────────┐
│           init_character_behavior()          │
└─────────────────────────────────────────────┘
                      │
                      ▼
┌─────────────────────────────────────────────┐
│              玩家阶段处理                    │
│  1. 玩家选择指令                            │
│  2. 设置行为ID、持续时间、开始时间           │
│  3. 循环处理直到玩家行为完成                 │
└─────────────────────────────────────────────┘
                      │
                      ▼
┌─────────────────────────────────────────────┐
│               NPC阶段处理                    │
│  1. 遍历所有NPC                             │
│  2. AI决策确定行为                          │
│  3. 执行行为直到所有NPC完成                 │
└─────────────────────────────────────────────┘
                      │
                      ▼
┌─────────────────────────────────────────────┐
│              时间管理与更新                  │
│  1. 更新游戏时间                            │
│  2. 检查是否进入新的一天                     │
│  3. 处理自动存档等特殊事件                  │
└─────────────────────────────────────────────┘
```

#### 2.2 单个角色行为处理流程
```
character_behavior(character_id)
          │
          ▼
    特殊模式处理
    - 疲劳/睡眠判断
    - 无法移动判断
    - 助理模式
    - 跟随模式
    - H状态处理
          │
          ▼
    行为前置事件
    (event_before_instruct)
          │
          ▼
    行为数值结算
    (settle_behavior)
          │
          ▼
    行为后置事件
    (event_after_instruct)
          │
          ▼
    实时数值变化
    (realtime_settle)
          │
          ▼
    状态持续判断
          │
          ▼
    行为完成判断
          │
          ▼
    自动获得素质
```

### 3. 关键系统集成

#### 3.1 结算系统 (`settle_behavior.py`)
- **作用**: 处理行为产生的所有数值变化和状态更新
- **主要函数**: 
  - `handle_settle_behavior()`: 主结算入口
  - `handle_instruct_data()`: 处理指令相关的结算
  - `handle_event_data()`: 处理事件相关的结算
- **结算内容**:
  - 数值变化（HP、MP、好感度等）
  - 状态变化（疲劳、饥饿、情欲等）
  - 特殊效果（刻印、堕落等）

#### 3.2 NPC AI系统 (`handle_npc_ai.py`)
- **作用**: 决定NPC的行为选择
- **主要函数**:
  - `find_character_target()`: AI主决策函数
  - `search_target()`: 搜索可用行为
  - `judge_character_follow()`: 跟随模式判断
- **决策优先级**:
  1. 紧急状态（疲劳、饥饿等）
  2. 工作/娱乐计划
  3. 普通行为选择

#### 3.3 前提系统 (`handle_premise.py`)
- **作用**: 验证行为和事件的执行条件
- **主要函数**:
  - `handle_premise()`: 主前提判断函数
  - `get_weight_from_premise_dict()`: 计算前提权重
- **前提类型**:
  - 基础前提（位置、时间、状态等）
  - 综合前提（CVP，复杂条件组合）
  - 特殊前提（无意识、口球等）

#### 3.4 事件系统 (`event.py`)
- **作用**: 处理剧情事件和特殊触发
- **主要函数**:
  - `handle_event()`: 事件处理主函数
  - `get_event_list_now()`: 获取当前可触发事件
- **事件类型**:
  - 指令前置事件
  - 指令后置事件
  - 状态触发事件

#### 3.5 实时结算系统 (`realtime_settle.py`)
- **作用**: 处理基于时间的自动数值变化
- **主要函数**:
  - `character_aotu_change_value()`: 自动数值变化
  - `change_character_persistent_state()`: 持续状态变化
- **结算内容**:
  - 疲劳度恢复/增加
  - 饥饿度增加
  - 尿意增加
  - 睡眠恢复

### 4. 数据流向

```
用户输入/AI决策
      │
      ▼
设置Behavior对象
(behavior_id, duration, start_time)
      │
      ▼
行为循环处理
      │
      ├──> 前提验证 (handle_premise)
      │
      ├──> 事件触发 (event system)
      │
      ├──> 数值结算 (settle_behavior)
      │
      ├──> 实时更新 (realtime_settle)
      │
      └──> 时间推进
            │
            ▼
      行为完成判断
            │
            ▼
      重置为空闲状态
```

### 5. 关键缓存变量

- `cache.over_behavior_character`: 已完成当前行为的角色集合
- `cache.game_time`: 当前游戏时间
- `cache.npc_id_got`: 已获得的NPC ID集合
- `cache.character_data`: 所有角色数据字典
- `cache.time_stop_mode`: 时停模式标志

### 6. 特殊机制

#### 6.1 时停模式
- 玩家行动后时间不推进
- 仅玩家可以行动
- 用于特殊玩法场景

#### 6.2 跟随模式
- NPC自动跟随玩家移动
- 智能跟随判断
- 可手动开启/关闭

#### 6.3 助理模式
- 特定NPC成为助理
- 享有特殊行为权限
- 自动处理部分事务

#### 6.4 H模式
- 特殊的行为处理逻辑
- 独立的AI决策系统
- 复杂的状态管理

### 7. 扩展说明

#### 7.1 添加新行为
1. 在 `Behavior.py` 中定义行为常量
2. 在配置文件中设置行为效果
3. 实现对应的结算函数
4. 添加必要的前提条件

#### 7.2 修改AI逻辑
1. 在 `handle_npc_ai.py` 中调整决策优先级
2. 修改搜索目标的权重计算
3. 添加新的特殊处理逻辑

#### 7.3 自定义事件
1. 在事件配置中定义新事件
2. 设置触发条件（前提）
3. 实现事件效果处理

### 8. 性能考虑

- NPC数量直接影响每轮循环时间
- 复杂前提判断会增加计算负担
- 事件检查频率需要平衡
- 实时结算精度与性能的权衡

### 9. 调试技巧

- 使用 `cache.over_behavior_character` 追踪行为完成情况
- 通过日志记录关键时间点
- 监控 `character_data.behavior` 查看当前行为状态
- 使用debug模式跳过等待时间

### 10. 相关文件

- **核心文件**:
  - `Script/Design/character_behavior.py`: 主行为循环
  - `Script/Design/settle_behavior.py`: 行为结算
  - `Script/Design/handle_npc_ai.py`: NPC AI
  - `Script/Design/handle_premise.py`: 前提系统
  - `Script/Design/event.py`: 事件系统

- **配置文件**:
  - `Script/Core/constant/Behavior.py`: 行为常量定义
  - `data/csv/Behavior.csv`: 行为配置数据
  - `data/csv/BehaviorEffect.csv`: 行为效果配置

- **结算实现**:
  - `Script/Settle/default.py`: 默认结算实现
  - `Script/Settle/default_second.py`: 二段行为结算
  - `Script/Settle/realtime_settle.py`: 实时结算