# 存档系统

## 系统概述

erArk的存档系统是游戏的核心基础架构，负责游戏数据的持久化存储和跨版本兼容性管理。该系统采用二进制序列化存储，支持完整的游戏状态保存与恢复，并具备强大的版本兼容性更新机制。

## 存档系统架构

### 核心文件结构

```
Script/Core/
├── save_handle.py          # 存档处理核心模块
├── cache_control.py        # 游戏缓存控制
├── game_type.py           # 数据类型定义
└── constant/__init__.py   # 系统常量定义

Script/Config/
└── normal_config.py       # 配置管理

save/                      # 存档目录
├── {save_id}/            # 具体存档文件夹
│   ├── 0                 # 存档头部信息
│   └── 1                 # 存档游戏数据
└── auto/                 # 自动存档
```

### 存档数据结构

#### 存档头部信息 (文件"0")
```python
save_version = {
    "game_verson": "版本号",
    "game_time": datetime,      # 游戏内时间
    "character_name": "博士名称",
    "save_time": datetime       # 存档保存时间
}
```

#### 存档游戏数据 (文件"1")
存储完整的`cache`对象，包含：
- 角色数据 (`character_data`)
- 地图数据 (`map_data`)
- 罗德岛基地数据 (`rhodes_island`)
- 游戏设置 (`game_setting`)
- 时间状态 (`game_time`)

### 核心功能模块

#### 1. 存档文件管理

**基础文件操作：**
```python
get_save_dir_path(save_id: str) → str
├── 构建存档目录路径: save/{save_id}/
├── 自动创建不存在的目录
└── 返回完整路径

judge_save_file_exist(save_id: str) → bool
├── 检查头部文件存在性: save/{save_id}/0
├── 检查数据文件存在性: save/{save_id}/1
└── 返回存档完整性状态

write_save_data(save_id: str, data_id: str, write_data: dict)
├── 创建存档目录
├── 使用pickle.dump()序列化数据
└── 写入指定文件: save/{save_id}/{data_id}

remove_save(save_id: str)
├── 获取存档目录路径
├── 使用shutil.rmtree()删除整个目录
└── 完全清除存档数据
```

#### 2. 存档保存系统

**存档创建流程：**
```python
establish_save(save_id: str)
├── 调用establish_save_linux()执行实际保存
└── 支持跨平台扩展

establish_save_linux(save_id: str)
├── 创建存档头部信息
│   ├── game_verson: 游戏版本号
│   ├── game_time: 当前游戏时间
│   ├── character_name: 博士角色名称
│   └── save_time: 实际存档时间
├── 组织存档数据结构
│   ├── "0": 存档头部信息
│   └── "1": 完整cache对象
└── 分别写入两个文件
```

#### 3. 存档读取系统

**数据载入流程：**
```python
load_save(save_id: str) → dict
├── 构建数据文件路径: save/{save_id}/1
├── 使用pickle.load()反序列化
└── 返回完整cache数据

load_save_info_head(save_id: str) → dict
├── 构建头部文件路径: save/{save_id}/0
├── 使用pickle.load()反序列化
└── 返回存档头部信息

input_load_save(save_id: str)
├── 创建全新Cache实例作为模板
├── 加载存档数据到内存
├── 执行全面跨版本兼容性更新
├── 覆盖当前游戏cache
└── 显示更新统计信息
```

#### 4. 跨版本兼容性系统

这是存档系统最复杂也最重要的组件，确保不同版本间的存档互通。

**核心更新机制：**

```python
input_load_save(save_id: str)
├── 数据结构基础更新
│   ├── update_dict_with_default() - 递归补充缺失字段
│   ├── update_tem_character() - 更新角色模板数据
│   └── recursive_update() - 深度合并对象属性
├── 角色系统全面更新
│   ├── update_character_config_data() - 更新角色配置
│   ├── update_chara_cloth() - 重置服装系统
│   ├── update_new_character() - 添加新角色
│   └── 角色特殊属性同步
├── 游戏内容系统更新
│   ├── 罗德岛资源同步
│   ├── 设施开启状态检查
│   ├── 食谱数据刷新
│   └── 图书借阅系统更新
├── 设置与配置更新
│   ├── update_settings() - 系统设置更新
│   ├── update_map() - 地图数据更新
│   └── 大地图势力数据更新
└── 数据验证与统计输出
```

**详细更新流程分析：**

**1. 基础数据结构更新 (`update_dict_with_default`)**
```python
功能: 递归对比新旧数据结构，补充缺失字段
处理逻辑:
├── 跳过Python内置方法(__开头结尾的属性)
├── 添加存档中不存在的新字段
├── 递归处理嵌套对象和Cache类型
├── 类型不匹配时重置为默认值
└── 记录更新次数并显示调试信息
```

**2. 角色模板数据更新 (`update_tem_character`)**
```python
功能: 同步角色预设数据，处理新增角色
处理流程:
├── 将cache.npc_tem_data转换为查找字典
├── 修正特殊角色序号错误(如深靛496→469)
├── 用新预设覆盖旧预设数据
├── 添加新增角色到预设列表末尾
├── 修正角色序号与实际位置不一致问题
├── 为缺失预设的角色创建空白模板
└── 清理多余的空白预设
```

**3. 角色配置数据更新 (`update_character_config_data`)**
```python
功能: 更新角色的各种属性字典
更新内容:
├── 素质(talent) - 对比game_config.config_talent
├── 经验(experience) - 对比game_config.config_experience  
├── 宝珠(juel) - 对比game_config.config_juel
├── 能力(ability) - 对比game_config.config_ability
├── 状态(status_data) - 对比game_config.config_character_state
├── 设置(chara_setting) - 对比game_config.config_chara_setting
├── 道具(item) - 对比game_config.config_item
├── 身体管理(body_manage) - 初始化或补充缺失项
├── 助理服务(assistant_services) - 补充新服务类型
├── 身体道具(body_item) - 从列表转换为字典格式
└── 行为系统更新 - 从int ID转换为字符串常量
```

**4. 服装系统更新 (`update_chara_cloth`)**
```python
功能: 检查并重置不兼容的服装数据
检查内容:
├── 各类型服装数据长度一致性检查
├── 脏污数据结构同步
├── 服装ID有效性验证
└── 无效服装数据时完整重置
重置流程:
├── 清空所有服装数据
├── 根据角色模板重新分配服装
├── 自动配备随机内衣
└── 返回重置标志
```

**5. 新角色添加 (`update_new_character`)**
```python
功能: 检测并添加新版本中的角色
处理逻辑:
├── 对比角色预设与实际角色数据
├── 找出预设中存在但角色数据中缺失的角色
├── 跳过特殊角色(如深靛)的重复添加
├── 为新角色分配连续的CID
├── 调用character_handle.init_character()创建完整角色
└── 添加到loaded_dict["character_data"]
```

**6. 系统设置更新 (`update_settings`)**
```python
功能: 更新各类游戏设置和配置
更新范围:
├── 基础系统设置(base_setting)
├── 绘制设置(draw_setting)  
├── 难度设置(difficulty_setting)
├── 角色口上选择设置(character_text_version)
├── AI聊天设置(ai_chat_setting)
├── 体检设置(physical_examination_setting)
└── 监禁调教设置(confinement_training_setting)
```

**7. 地图数据更新 (`update_map`)**
```python
功能: 同步地图场景和路径数据
处理内容:
├── 添加新场景到存档
├── 更新场景标签(scene_tag)
├── 更新房间区域信息(room_area)
├── 处理场景重命名(如人气快餐→约翰老妈汉堡店)
├── 移动角色位置到新场景名称
├── 删除不存在的旧场景数据
└── 同步完整地图路径数据(map_data)
```

**特殊处理机制：**

**角色行为系统转换:**
- 旧版本使用整数ID存储行为
- 新版本使用字符串常量存储
- 自动将int类型转换为对应的常量名

**深靛角色序号修正:**
- 修正深靛角色的ADV编号从496到469
- 特殊处理避免序号冲突

**服装系统兼容:**
- 检测无效服装ID并重置
- 自动配备基础内衣
- 保持服装数据结构一致性

**数据类型转换:**
- body_item从列表转换为字典
- 修正orgasm_count数据格式错误
- 类型不匹配时重置为默认值

#### 3. 缓存控制 (`cache_control.py`)

```python
cache: game_type.Cache = None
```
- 全局游戏状态管理器
- 所有游戏数据的统一入口
- 存档系统的数据源

## 存档访问接口

### 存档信息面板 (`see_save_info_panel.py`)

存档系统通过专门的面板为用户提供存档管理界面：

**核心功能：**
- 存档列表展示与分页浏览
- 存档信息格式化显示
- 存档操作确认机制
- 自动存档特殊处理

**存档信息显示格式：**
```python
# 季节显示转换
if game_time.month == 3: month_text = "春"
elif game_time.month == 6: month_text = "夏"  
elif game_time.month == 9: month_text = "秋"
elif game_time.month == 12: month_text = "冬"

# 完整存档信息
save_name = f"No.{save_id} {version} {game_time_text} {character_name}博士 {save_time_text}"
```

**存档操作流程：**
```
选择存档 → 操作菜单 → 二次确认 → 执行操作
├── 读取存档：载入游戏数据并切换到游戏场景
├── 覆盖存档：保存当前游戏状态到指定槽位
├── 删除存档：从文件系统中移除存档文件
└── 返回列表：取消操作返回存档列表
```

## 系统数据流向

### 存档系统数据流向图

```
游戏运行时:
Cache ←→ 各种游戏系统 (角色、地图、基建等)
  ↓
存档保存流程:
Cache → save_handle.establish_save() → pickle序列化 → 磁盘文件

存档读取流程:
磁盘文件 → pickle反序列化 → save_handle.input_load_save() → 
跨版本兼容性更新 → Cache覆盖 → 游戏系统重新初始化
```

### 存档操作调用链

```
存档保存:
用户触发保存 → establish_save(save_id) → 
创建存档头信息 → pickle.dump(cache) → 写入文件系统

存档载入:
用户选择存档 → input_load_save(save_id) → 
load_save() → 跨版本更新 → cache覆盖 → 切换游戏面板

存档管理:
judge_save_file_exist() → 检查存档存在性
load_save_info_head() → 获取存档头信息  
remove_save() → 删除存档文件
```

## 配置与设置

### 存档相关配置项 (`config.ini`)

```ini
[game]
max_save = 100        # 最大存档数量
save_page = 10        # 存档页面显示数量  
debug = 1             # 调试模式 (影响存档调试信息显示)
verson_time = @2021-2025    # 版本时间标识
```

### 版本管理配置

```python
# normal_config.py 中的版本管理
if os.path.exists("package.json"):
    with open("package.json", "r") as package_file:
        version_data = json.load(package_file)
        config_normal.verson = version_data["version"]
else:
    config_normal.verson = "Orgin"
```

## 技术特点与优势

### 存档系统核心优势

1. **跨版本兼容性**: 
   - 完善的版本检测和数据结构更新机制
   - 自动添加新增字段，保持原有数据不丢失
   - 智能的角色数据、配置数据、资源数据同步更新

2. **数据完整性**:
   - 使用Python pickle进行二进制序列化
   - 保证复杂对象结构的完整保存和恢复
   - 分离式存储设计（头信息+游戏数据）

3. **高可靠性**:
   - 二次确认机制防止误操作
   - 存档存在性检查和错误处理
   - 安全的文件读写操作

4. **高效性能**:
   - 按需加载存档信息
   - 分页显示支持大量存档管理
   - 增量更新机制减少兼容性处理开销

5. **扩展性设计**:
   - 模块化的更新函数设计
   - 易于添加新的数据类型和兼容性逻辑
   - 统一的缓存管理接口

## 开发指导

### 扩展存档系统

#### 1. 添加新数据类型
```python
# 在 game_type.py 中定义新的数据结构
class NewDataType:
    def __init__(self):
        self.new_field = "default_value"
        self.new_dict = {}
        
# 在 Cache 类中添加新字段
class Cache:
    def __init__(self):
        # 现有字段...
        self.new_data = NewDataType()
```

#### 2. 添加兼容性更新逻辑
```python
# 在 save_handle.py 的 input_load_save() 中添加更新函数
def update_new_data(loaded_dict):
    """更新新数据类型的兼容性逻辑"""
    update_count = 0
    
    # 检查并添加缺失的字段
    if "new_data" not in loaded_dict:
        loaded_dict["new_data"] = NewDataType()
        update_count += 1
        
    # 更新现有数据结构
    if hasattr(loaded_dict["new_data"], "old_field"):
        # 迁移逻辑
        loaded_dict["new_data"].new_field = process_old_field(loaded_dict["new_data"].old_field)
        update_count += 1
        
    return update_count

# 在主更新流程中调用
update_count += update_new_data(loaded_dict)
```

#### 3. 存档调试和测试

**调试方法：**
- 设置 `debug = 1` 查看详细更新信息
- 使用 `print()` 输出关键数据变化
- 对比更新前后的数据结构

**测试流程：**
1. 创建老版本存档
2. 更新代码添加新字段  
3. 载入老存档验证兼容性
4. 检查数据完整性和新功能
5. 测试新存档的保存和载入

### 存档系统最佳实践

#### 1. 数据结构设计原则
- 使用明确的默认值
- 避免循环引用
- 保持向后兼容的命名规范
- 合理组织数据层次结构

#### 2. 兼容性更新原则
- 只添加字段，不删除字段
- 提供数据迁移路径
- 记录更新操作数量
- 给用户明确的更新提示

#### 3. 性能优化建议
- 避免在更新函数中进行大量计算
- 使用批量更新减少I/O操作
- 合理设计缓存失效策略
- 考虑存档大小对加载时间的影响

## 高级技术细节

### 存档系统深度技术分析

基于对`save_handle.py`的完整分析，以下是存档系统的高级技术实现细节：

#### 1. 递归数据更新机制

**`recursive_update()` 深度对象合并:**
```python
功能: 递归更新目标对象的所有属性
实现原理:
├── 遍历源对象的所有属性
├── 检查属性是否为具有__dict__的对象
├── 递归处理嵌套对象
├── 直接赋值基础数据类型
└── 保持对象引用关系的完整性

应用场景:
- 角色数据的深度合并
- 复杂嵌套对象的属性同步
- 保持对象间引用关系不变
```

#### 2. 智能类型转换系统

**行为系统数据迁移:**
```python
# 从整数ID转换为字符串常量
旧格式: behavior.behavior_id = 1001 (int)
新格式: behavior.behavior_id = "SHARE_BLANKLY" (str)

转换逻辑:
├── 遍历Behavior_Int常量定义
├── 匹配整数值与常量名
├── 使用getattr()动态获取字符串常量
└── 更新为新的字符串格式

二级行为处理:
├── 复制原始字典避免迭代冲突
├── 处理整型键的批量转换
├── 删除无效的旧格式数据
└── 保持数据的完整性
```

#### 3. 角色收藏品系统自动扩展

**收藏品数据结构自动维护:**
```python
自动添加逻辑:
├── 检查角色是否在玩家收藏品列表中
├── 为新角色添加收藏品记录结构:
│   ├── token_list[cid] = False (代币收集状态)
│   ├── first_panties[cid] = "" (初次内裤记录)
│   ├── npc_panties[cid] = [] (内裤收藏列表)
│   └── npc_socks[cid] = [] (袜子收藏列表)
└── 确保所有角色都有完整收藏品数据
```

#### 4. 监禁系统状态同步

**囚犯列表自动维护:**
```python
检查逻辑:
├── 遍历所有角色的监禁状态
├── 检查sp_flag.imprisonment标志
├── 排除正在逃跑的角色(escaping=True)
├── 自动添加到current_prisoners列表
└── 记录监禁时间和状态

数据结构:
current_prisoners[cid] = [监禁时间, 状态码]
```

#### 5. 资源与设施智能检测

**罗德岛资源同步:**
```python
资源更新流程:
├── 遍历game_config.config_resouce中的所有资源
├── 检查存档中是否存在该资源记录
├── 不存在则初始化为0
└── 确保资源列表的完整性

设施开启状态检测:
├── 检查设施配置中的NPC_id要求
├── 遍历已获得角色列表
├── 匹配角色adv与设施要求
├── 自动开启满足条件的设施
└── 保持设施状态的合理性
```

#### 6. 特殊数据修正机制

**orgasm_count数据修正:**
```python
问题: 旧版本中可能存在orgasm_count = 0的错误数据
修正: 检测到该情况时，重置为[0, 0]的正确格式
确保: 高潮计数数据的数据类型一致性
```

**地图场景重命名处理:**
```python
具体案例: "人气快餐开封菜" → "约翰老妈汉堡店"
处理流程:
├── 检测存档中是否存在旧场景名
├── 遍历所有角色的位置信息
├── 批量更新角色位置到新场景名
├── 删除旧场景数据
└── 输出更新日志
```

#### 7. 调试与监控系统

**更新过程可视化:**
```python
统计机制:
├── 每个更新函数返回update_count
├── 累计所有更新操作数量
├── 分类显示不同类型的更新
└── 最终输出完整统计报告

调试信息:
├── 存档跨版本更新详细日志
├── 服装数据重置统计
├── 口上颜色更新计数
├── 地图更新状态提示
└── 系统设置更新通知
```

### 存档系统性能优化

#### 1. 内存优化策略

**数据预处理:**
- 使用字典转换优化查找性能
- cache_dict = {npc.AdvNpc: npc for npc in cache.npc_tem_data}
- 避免嵌套循环带来的性能问题

**增量更新:**
- 只更新真正需要变更的数据
- 跳过已正确的数据结构
- 减少不必要的对象创建

#### 2. 错误处理机制

**容错设计:**
- 类型检查避免运行时错误
- 默认值机制保证数据完整性
- 异常情况的优雅降级处理

**数据验证:**
- 存档文件完整性检查
- 数据结构合法性验证
- 关键字段的空值处理

这套存档系统为erArk提供了强大的数据持久化能力，是游戏数据管理的核心基础设施。通过精密的跨版本兼容性设计和智能的数据更新机制，确保游戏数据在版本迭代中的稳定性和完整性。