# 地图系统

## 概述

erArk的地图系统是一个多层级的场景导航系统，支持复杂的地图结构、路径寻找、访问控制和可视化展示。

## 系统架构

### 1. 核心数据结构

#### Map类 (game_type.py)
```python
class Map:
    path: List[str]           # 地图路径坐标
    map_name: str            # 地图名称  
    path_edge: Dict[str, Dict[str, TargetPath]]  # 路径边缘数据
    map_draw: MapDraw        # 地图绘制数据
    sorted_path: Dict[int, Dict[int, List[str]]]  # 排序后的路径
```

#### Scene类 (game_type.py)
```python
class Scene:
    scene_path: str          # 场景路径
    scene_name: str          # 场景名称
    in_door: bool           # 是否为室内
    exposed: int            # 暴露等级(0=私密,1=普通,2=公开)
    have_furniture: int     # 家具等级(0-3)
    close_type: int         # 门类型(0=无门,1=普通门,2=隔间门)
    room_area: int          # 房间大小(0=10人,1=50人,2=100人,3=无限)
    scene_tag: str          # 场景标签
    character_list: Set[int] # 当前场景内的角色列表
```

### 2. 地图层级结构

```
世界地图
└── 国家/地区 (如：罗德岛)
    └── 建筑物 (如：中枢、宿舍)
        └── 楼层 (如：1楼、2楼)
            └── 房间 (如：博士办公室、101房间)
```

### 3. 路径表示方式

- **列表形式**: `["罗德岛", "中枢", "1", "博士办公室"]`
- **字符串形式**: `"罗德岛-中枢-1-博士办公室"`
- **出口标识**: 路径中的"0"表示该层级的出口

## 主要组件

### 1. 地图配置系统 (map_config.py)

负责加载和解析地图数据：

- 从`/data/map/`目录加载地图结构
- 解析`Map.json`文件获取路径边缘数据
- 解析`Scene.json`文件获取场景属性
- 读取`Map`文件获取ASCII艺术地图
- 生成pickle缓存文件提升性能

### 2. 地图处理模块 (map_handle.py)

提供地图操作的核心功能：

#### 路径转换
- `get_map_system_path_str_for_list()`: 列表路径转字符串
- `get_map_system_path_for_str()`: 字符串路径转列表

#### 场景查询
- `get_scene_data_for_map()`: 获取指定场景数据
- `get_map_scene_id_for_scene_path()`: 获取场景在地图中的ID

#### 路径寻找
- `get_path()`: 使用Dijkstra算法计算最短路径
- `get_map_for_path()`: 获取场景所属的地图路径

#### 访问控制
- `judge_scene_accessible()`: 判断场景是否可进入
  - `open`: 开放访问
  - `wait_open`: 等待解锁(需要设施或NPC)
  - `door_lock`: 门已锁定
  - `private`: 私密区域(宿舍、浴室等)
  - `full`: 已满员

### 3. 角色移动系统 (character_move.py)

#### 玩家移动
```python
def own_charcter_move(target_scene: List[str]) -> int:
    """控制玩家角色移动到目标场景"""
    # 1. 计算路径
    # 2. 检查访问权限
    # 3. 处理私密区域
    # 4. 执行移动
    # 5. 更新时间
```

#### 通用移动
```python
def character_move(character_id: int, now_position: List[str], 
                  target_scene: List[str]) -> int:
    """移动任意角色到目标位置"""
    # 处理NPC移动逻辑
```

### 4. 地图显示面板 (see_map_panel.py)

#### SeeMapPanel类
主要功能：
- 显示当前地图的ASCII艺术
- 高亮显示玩家位置(金色)
- 标记有NPC的位置(绿色)
- 显示不可进入区域(灰色)
- 提供移动按钮

#### 快捷移动系统
- **MapSceneNameDraw**: 显示所有可达场景列表
- **CollectionSceneNamePanel**: 收藏地点快捷导航
- **MoveMenuPanel**: 切换不同的快捷移动视图

#### 地图导航功能
- 查看上级/下级地图
- 干员位置一览
- 收藏/取消收藏当前地点

### 5. 导航面板 (navigation_panel.py)

用于基地间移动：
- 显示燃料消耗
- 计算移动时间
- 处理外交关系

## 场景属性说明

### 暴露等级 (exposed)
- 0: 私密场所(如个人房间)
- 1: 普通场所
- 2: 公开场所

### 家具等级 (have_furniture)
- 0: 无家具
- 1: 简单家具
- 2: 普通家具
- 3: 豪华家具

### 门类型 (close_type)
- 0: 无门
- 1: 普通门(可锁)
- 2: 隔间门(如厕所隔间)

### 房间容量 (room_area)
- 0: 最多10人
- 1: 最多50人
- 2: 最多100人
- 3: 无限制

## 特殊机制

### 1. 设施解锁
某些区域需要满足条件才能进入：
- 基地升级到特定等级
- 特定NPC入驻
- 完成特定任务

### 2. 门锁系统
- 角色可以锁门防止他人进入
- 玩家可使用一次性钥匙开锁
- 某些角色拥有通用钥匙

### 3. 跟随系统
- 跟随玩家的NPC在私密区域有特殊处理
- 根据关系等级决定是否可以一同进入

### 4. 场景标签系统
用于分类和特殊行为：
- `Dormitory`: 宿舍
- `Dr_room`: 博士相关房间
- `Toilet`: 厕所
- `Bathroom`: 浴室

## 数据流程

1. **初始化阶段**
   - 加载地图配置文件
   - 构建场景数据结构
   - 预计算所有路径

2. **运行时**
   - 玩家选择目标位置
   - 系统计算最短路径
   - 检查访问权限
   - 执行移动并更新状态

3. **显示更新**
   - 刷新地图显示
   - 更新角色位置标记
   - 显示可用操作

## 相关配置文件

- `/data/map/`: 地图结构目录
- `Map.json`: 路径边缘定义
- `Scene.json`: 场景属性定义
- `Map`: ASCII地图文件
- `/data/csv/Map.csv`: 地图配置表
- `/data/csv/Scene.csv`: 场景配置表

## 使用示例

### 移动到指定位置
```python
# 移动玩家到博士办公室
target_path = ["罗德岛", "中枢", "1", "博士办公室"]
character_move.own_charcter_move(target_path)
```

### 检查场景是否可进入
```python
scene_path_str = "罗德岛-中枢-1-博士办公室"
access_status = map_handle.judge_scene_accessible(scene_path_str, 0)
if access_status == "open":
    # 可以进入
```

### 获取当前地图数据
```python
map_path = map_handle.get_map_for_path(character.position)
map_path_str = map_handle.get_map_system_path_str_for_list(map_path)
map_data = cache.map_data[map_path_str]
```

## 注意事项

1. 地图路径必须使用统一格式(列表或字符串)
2. 移动前需检查目标场景的可达性
3. 私密区域需要特殊权限处理
4. 地图数据缓存在pickle文件中，修改配置后需重新生成
5. ASCII地图中的按钮位置与场景ID对应