# 势力外交系统

## 系统概述

势力外交系统是erArk游戏中的核心政治管理模块，允许玩家通过派遣外交官执行不同的外交方针来影响各势力的声望和地区的源石病感染率。该系统涉及多个数据层面的交互，包括势力管理、外交官分配、政策执行和周期性结算。

## 核心组件架构

### 1. 数据结构层 (`Script/Core/game_type.py`)

#### Country类 (lines 1133-1141)
```python
class Country:
    def __init__(self):
        self.nation_reputation: Dict[int, float] = {}  # 势力声望字典
        self.country_infection_rate: Dict[int, int] = {}  # 国家矿石病感染率字典
```

#### RhodesIsland类外交数据 (line 1083)
```python
self.diplomat_of_country: Dict[int, Tuple[int, int]] = {}
# 格式：出身地国家id:[外交官角色id, 外交方针id]
```

### 2. 配置定义层 (`Script/Config/config_def.py`)

#### Nation类 (lines 883-896)
- **字段**：势力id、势力名、所属国别、领导人、介绍
- **功能**：定义各个势力的基本属性和归属关系

#### Birthplace类 (lines 159-169)
- **字段**：出生地id、出生地名、是否在大地图显示、初始感染率
- **功能**：定义各国家/地区的基础数据和感染率基线

#### Diplomatic_Policy类 (lines 461-473)
- **字段**：政策id、政策名、难易度、需求、政策介绍
- **功能**：定义可执行的外交方针和其效果描述

### 3. 配置加载层 (`Script/Config/game_config.py`)

#### 关键配置变量
```python
config_nation = {}  # 势力配置字典
config_birthplace = {}  # 国家/地区配置字典
config_diplomatic_policy = {}  # 外交政策配置字典
config_nation_data_of_country_subordinate = {}  # 主要势力到附属势力的映射
config_resouce_data_of_nation = {}  # 国家到特产资源的映射
```

#### 数据加载函数
- `load_nation()` (lines 606-625)：加载势力数据并构建层级关系
- `load_birthplace()` (lines 596-603)：加载国家地区数据
- `load_diplomatic_policy()` (lines 628-635)：加载外交政策数据

## 数据文件结构

### 1. CSV数据文件

#### Nation.csv (势力数据)
```csv
cid,name,country,leader,introduction
0,罗德岛,0,博士,致力于矿石病研究和感染者救治的国际性医疗机构
11,阿戈尔,1,待填写,地处深海的国家，对外相对封锁...
12,深海猎人,1,歌蕾蒂娅,隶属于深海国家阿戈尔的军事力量...
```
- **层级结构**：主要国家(country字段为正数) -> 附属势力(country字段指向主要国家)
- **特殊标识**：country=-1表示独立势力，country=0表示罗德岛相关

#### Birthplace.csv (国家地区数据)
```csv
cid,name,inmap,infect_rate
1,阿戈尔,1,5
4,哥伦比亚,1,45
6,卡兹戴尔,1,95
7,拉特兰,1,0
```
- **感染率**：各地区的初始源石病感染率(0-100%)
- **地图显示**：inmap字段控制是否在大地图中显示

#### Diplomatic_Policy.csv (外交政策)
```csv
cid,name,difficulty,need,info
1,在民间开展源石病治疗工作,1,无,小幅度降低当地的源石病感染率
2,设立和完善外勤办事处,2,无,小幅度提高地区声望
5,与势力领导人进行密切交流,5,无,大幅度提高势力声望和地区声望
```
- **难易度**：1-5级，限制可执行的政策
- **效果**：影响感染率或声望的具体数值在代码中定义

## 主要界面系统 (`Script/UI/Panel/nation_diplomacy_panel.py`)

### Nation_Diplomacy_Panel类

#### 核心方法

##### 1. `draw()` - 主界面绘制 (lines 155-297)
- **界面切换**：国家及附属势力 / 其他势力
- **势力列表**：显示势力名称、声望、感染率、负责外交官
- **交互逻辑**：点击势力进入详细信息页面

##### 2. `nation_info()` - 势力详细信息 (lines 308-432)
- **信息展示**：势力介绍、声望状态、特产资源、附属势力
- **操作选项**：调整负责外交官、调整外交方针
- **条件限制**：必须先任命外交官才能制定方针

##### 3. `adjust_NPC()` - 外交官分配 (lines 435-496)
- **候选条件**：出身地或势力归属匹配的干员
- **能力显示**：外交能力等级和效果预览
- **状态切换**：本地办公 vs 外派状态

##### 4. `sure_NPC()` - 确认外交官任命 (lines 499-549)
- **状态处理**：解除旧外交官、任命新外交官
- **工作分配**：设置工作类型为131(外交官)
- **位置逻辑**：根据罗德岛当前位置决定本地办公或外派

##### 5. `adjust_diplomatic_policy()` - 外交方针选择 (lines 552-614)
- **可用检查**：基于设施等级和外交官能力过滤
- **方针展示**：显示政策名称和效果描述
- **选择确认**：设置当前生效的外交方针

### 支持函数

#### `update_nation_data()` - 数据重置 (lines 20-43)
- **检查条件**：全势力0声望且全地区0感染率
- **重置逻辑**：恢复各地区的初始感染率数据
- **触发时机**：每次进入外交界面时调用

#### `judge_diplomatic_policy()` - 周期结算 (lines 45-135)
- **结算周期**：每周执行一次外交政策效果
- **能力计算**：基于外交官的外交能力(ability[40])进行随机调整
- **效果应用**：
  - 政策1,4：降低感染率 0.01/0.05 * 能力修正 * 随机因子
  - 政策2,3,5：提高声望 0.1/0.1/0.3 * 能力修正 * 随机因子
- **经验奖励**：外交官获得对话和学识经验
- **结果输出**：显示本周声望和感染率总变化

## 系统集成关系

### 1. 与结算系统的集成 (`Script/Settle/past_day_settle.py`)
- **调用点**：line 78调用`judge_diplomatic_policy()`
- **触发条件**：每周的特定时间点
- **数据持久化**：结算结果保存到cache.country中

### 2. 与能力计算系统的集成 (`Script/Design/attr_calculation.py`)
- **国家数据初始化**：`get_country_reset()`函数
- **声望等级转换**：`get_reputation_level()`函数
- **能力效果计算**：`get_ability_adjust()`函数

### 3. 与外勤委托系统的集成 (`Script/UI/Panel/field_commission_panel.py`)
- **声望奖励**：完成特定国家的委托可获得对应势力声望
- **委托过滤**：根据外交方针优先显示特定类型的委托
- **地区关系**：外交关系影响委托的可用性和奖励

### 4. 与角色系统的集成
- **出身地绑定**：character.relationship.birthplace字段
- **势力归属**：character.relationship.nation字段
- **工作分配**：character.work.work_type = 131表示外交官
- **特殊标记**：character.sp_flag.in_diplomatic_visit标记外派状态

## 数据流图

```
CSV文件 -> 配置加载 -> 游戏缓存 -> 界面显示 -> 用户操作 -> 数据更新 -> 周期结算
   ↓           ↓          ↓          ↓          ↓          ↓          ↓
Nation.csv -> config_nation -> cache.country -> 外交界面 -> 外交官任命 -> 声望变化 -> 每周结算
Birthplace.csv -> config_birthplace -> nation_reputation -> 方针选择 -> 政策执行 -> 感染率变化 -> 经验奖励
Diplomatic_Policy.csv -> config_diplomatic_policy -> country_infection_rate -> 状态更新 -> 效果计算 -> 结果展示 -> 数据保存
```

## 重要算法逻辑

### 1. 声望计算算法
```python
# 基础公式
ability_adjust = attr_calculation.get_ability_adjust(diplomat_ability_level)
random_factor = random.uniform(-6 + ability_level, 6 + ability_level) / 10
final_effect = base_effect * ability_adjust * random_factor

# 附属势力声望传播
main_nation_reputation_change = final_effect
subordinate_reputation_change = final_effect * multiplier  # 0.5或3.0根据政策
```

### 2. 感染率计算算法
```python
# 感染率降低计算
down_rate = policy_base_rate * ability_adjust * random_factor
new_infection_rate = current_rate - down_rate
# 政策1: base_rate = 0.01, 政策4: base_rate = 0.05
```

### 3. 外交官匹配算法
```python
# 外交官候选条件
def is_diplomat_candidate(character_data, target_country_id, target_nation_id):
    return (character_data.relationship.birthplace == target_country_id or 
            character_data.relationship.nation == target_nation_id)
```

## 系统特点与设计思路

### 1. 层级化势力系统
- **主要国家**：如维多利亚、哥伦比亚等，拥有多个附属势力
- **附属势力**：如深海猎人(属于阿戈尔)，声望变化会传播到主要国家
- **独立势力**：不属于任何国家的特殊组织

### 2. 能力驱动的效果系统
- **外交官能力**：ability[40]外交能力直接影响政策效果
- **随机性引入**：避免固定数值，增加游戏的不确定性
- **等级限制**：高级政策需要高能力外交官和高级设施

### 3. 长期发展机制
- **周期性结算**：外交效果不是立即生效，而是持续产生影响
- **经验成长**：外交官在执行任务中获得经验，形成正循环
- **状态持久化**：外交关系和感染率变化会永久保存

### 4. 地理位置相关性
- **本地 vs 外派**：罗德岛位置影响外交官的工作状态
- **地区特产**：不同国家有不同的资源特色
- **移动影响**：罗德岛移动会影响外交官的在线状态

## 扩展性设计

### 1. 可扩展的政策系统
- 新增外交政策只需在CSV中添加数据，代码中添加对应的effect处理
- 政策效果可以灵活调整，支持复杂的条件判断

### 2. 动态的势力关系
- 势力间的关系可以通过配置文件灵活定义
- 支持势力归属关系的动态变化

### 3. 模块化的界面系统
- 外交界面独立于其他系统，便于维护和扩展
- 支持不同类型的外交操作和显示模式

## 性能考虑

### 1. 数据缓存机制
- 配置数据在游戏启动时一次性加载，避免重复IO操作
- 游戏状态数据使用内存缓存，提高访问效率

### 2. 计算优化
- 外交效果计算只在周期结算时进行，避免实时计算开销
- 界面显示数据按需计算，减少不必要的处理

### 3. 存储优化
- 只存储变化的数据，初始数据通过配置文件恢复
- 使用字典结构提高数据查找效率

这个势力外交系统体现了erArk游戏在政治模拟方面的深度设计，通过多层次的数据结构、复杂的计算逻辑和丰富的交互界面，为玩家提供了一个真实而有趣的外交管理体验。