# 助理系统说明文档

## 系统概述

助理系统是erArk游戏中的一个重要功能模块，允许玩家指定一名角色作为助理，该助理可以提供多种服务，包括跟随、问候、同居、辅佐等功能。系统通过CSV配置、Python代码逻辑和UI界面相结合，实现了完整的助理管理功能。

## 核心文件结构

### 主要代码文件

1. **`Script/UI/Panel/assistant_panel.py`** - 助理面板UI主文件
2. **`data/csv/AssistantServices.csv`** - 助理服务配置数据
3. **`Script/Config/game_config.py`** - 游戏配置加载器
4. **`Script/Config/config_def.py`** - 配置数据结构定义
5. **`Script/Core/game_type.py`** - 角色数据结构定义
6. **`Script/Design/attr_calculation.py`** - 属性计算和初始化
7. **`Script/Design/handle_premise.py`** - 前提条件处理
8. **`Script/Design/handle_npc_ai.py`** - NPC AI行为控制

### CSV数据文件

- **`data/csv/AssistantServices.csv`** - 定义了所有助理服务的配置信息

## 数据结构与流程

### 助理服务配置结构

#### CSV数据格式 (`AssistantServices.csv`)
```csv
cid,name,require,option
服务id,服务名,服务需求(用#分隔每个选项的需求)(F好感，X信任),各个选项
```

#### 具体服务配置
- **服务ID 2**: 跟随服务 - `否|是，智能跟随（在吃饭/上厕所/休息/睡觉等生理需求时会暂离，其他情况下跟随）`
- **服务ID 3**: 辅佐服务 - `否|是，在非跟随状态下，会自己在博士办公室里处理公务`
- **服务ID 4**: 送饭服务 - `否|帮忙买午饭|亲手做午饭|亲手做三餐`
- **服务ID 5**: 早安服务 - `否|早上叫起床|早安吻|早安咬`
- **服务ID 6**: 晚安服务 - `否|晚上催睡觉|晚安吻|晚安咬`
- **服务ID 7**: 同居服务 - `否|是`
- **服务ID 8**: 助攻服务 - `否|是，助理在场时对其他干员的指令实行值+50`
- **服务ID 9**: 性处理服务 - (未实装)
- **服务ID 10**: 每周轮换 - `否|是，每周一随机指定一名干员担任本周的助理`

### 角色数据结构

#### Character类中的助理相关属性
```python
class Character:
    assistant_character_id: int = 0          # 助理角色ID（仅玩家角色使用）
    assistant_services: Dict[int, int] = {}  # 助理服务配置 {服务ID: 选项值}
    sp_flag.is_follow: int = 0              # 跟随状态标记
    second_behavior: Dict[str, int] = {}     # 二段行为标记
    dormitory: str                          # 宿舍位置
    pre_dormitory: str                      # 之前的宿舍位置
    action_info.plan_to_wake_time: List[int] # 预定起床时间
    action_info.plan_to_sleep_time: List[int] # 预定睡觉时间
```

#### 配置数据结构
```python
class AssistantServices:
    cid: int        # 服务ID
    name: str       # 服务名
    require: str    # 服务需求（F好感度，X信任度）
    option: str     # 各个选项
```

## 系统工作流程

### 1. 数据加载流程

1. **CSV读取**: `buildconfig.py` 读取 `AssistantServices.csv`
2. **数据解析**: `game_config.py` 中的 `load_assistant_services()` 函数解析CSV数据
3. **配置存储**: 数据存储在 `config_assistant_services` 和 `config_assistant_services_option` 中
4. **角色初始化**: `attr_calculation.py` 中的 `get_assistant_services_zero()` 初始化角色助理服务

### 2. 助理选择流程

#### 候选角色筛选 (`get_assistant_candidates()`)
```python
def get_assistant_candidates():
    candidates = []
    for npc_id in sorted(cache.npc_id_got):
        # 跳过玩家自己
        if npc_id == 0: continue
        # 跳过异常状态角色
        if handle_premise.handle_unnormal_27(npc_id): continue
        # 跳过监狱长
        if npc_id == cache.rhodes_island.current_warden_id: continue
        # 跳过访客
        if npc_id in cache.rhodes_island.visitor_info: continue
        candidates.append(npc_id)
    return candidates
```

#### 助理替换流程 (`assistant_replace()`)
```python
def assistant_replace(new_assistant_id: int):
    # 1. 处理旧助理
    if character_data.assistant_character_id != 0:
        old_assistant_data.sp_flag.is_follow = 0  # 取消跟随
        # 恢复宿舍位置
        if old_assistant_data.dormitory == "中枢/博士房间":
            old_assistant_data.dormitory = old_assistant_data.pre_dormitory
        # 重置助理服务数据
        old_assistant_data.assistant_services = get_assistant_services_zero()
        # 更新二段行为标记
        old_assistant_data.second_behavior["chosen_as_assistant"] = 0
        old_assistant_data.second_behavior["not_as_assistant"] = 1
    
    # 2. 设置新助理
    if new_assistant_id != old_id and 状态正常:
        character_data.assistant_character_id = new_assistant_id
        new_assistant_data.sp_flag.is_follow = 1  # 开启跟随
        new_assistant_data.assistant_services[2] = 1  # 默认开启智能跟随
        # 更新二段行为标记
        new_assistant_data.second_behavior["chosen_as_assistant"] = 1
        new_assistant_data.second_behavior["not_as_assistant"] = 0
```

### 3. 服务配置流程

#### 服务选项切换逻辑
```python
def chose_button(self, service_cid: int, service_option_len: int):
    # 1. 检查解锁条件
    service_require_text_next = service_require_text_all[current_option + 1]
    judge, reason = attr_calculation.judge_require(
        [service_require_text_next], 
        assistant_character_id, 
        hypnosis_replace_trust_flag=True
    )
    
    # 2. 更新服务选项
    if judge or cache.debug_mode:
        target_data.assistant_services[service_cid] += 1
    else:
        # 不符合条件时归零并显示失败信息
        target_data.assistant_services[service_cid] = 0
    
    # 3. 结算附带属性变化
    settlement_of_associated_attribute(service_cid, service_option_len)
```

#### 附带属性变化处理
```python
def settlement_of_associated_attribute(self, service_cid: int, service_option_len: int):
    # 多选项行为前缀映射
    behavior_prefixes = {
        2: "ai_follow_",      # 跟随服务
        4: "assistant_send_food_",  # 送饭服务
        5: "morning_service_",      # 早安服务
        6: "night_service_"         # 晚安服务
    }
    
    # 二值开关行为前缀映射
    binary_prefixes = {
        3: "support_service_",  # 辅佐服务
        7: "cohabitation_",     # 同居服务
        8: "love_support_"      # 助攻服务
    }
    
    # 处理二段行为标记设置
    if service_cid in behavior_prefixes:
        # 清除同类所有标记，设置当前选项标记
    elif service_cid in binary_prefixes:
        # 设置开启/关闭标记
    
    # 特殊处理：同居服务的宿舍变更
    if service_cid == 7:
        if target_data.assistant_services[service_cid] == 1:
            target_data.pre_dormitory = target_data.dormitory
            target_data.dormitory = "中枢/博士房间"
        else:
            target_data.dormitory = target_data.pre_dormitory
```

### 4. UI界面流程

#### 主面板显示 (`Assistant_Panel.draw()`)
```python
def draw(self):
    while 1:
        # 1. 显示助理基本信息
        if character_data.assistant_character_id == 0:
            button_text += "当前无助理"
        else:
            button_text += f"当前助理：{target_data.name}"
        
        # 2. 遍历显示所有服务选项
        if character_data.assistant_character_id != 0:
            for cid in game_config.config_assistant_services:
                if cid == 9: continue  # 跳过未实装的服务
                service_data = game_config.config_assistant_services[cid]
                service_option_data = game_config.config_assistant_services_option[cid]
                current_option = target_data.assistant_services.get(cid, 0)
                option_text = service_option_data[0][current_option]
                
                # 显示服务按钮
                button_text = f"[{cid:03d}]{service_data.name}    {option_text}"
                
                # 特殊处理：时间设置按钮
                if cid == 5 and target_data.assistant_services[cid]:  # 早安服务
                    # 显示早安时间设置按钮
                if cid == 6 and target_data.assistant_services[cid]:  # 晚安服务
                    # 显示晚安时间设置按钮
```

#### 时间设置功能
```python
def select_morning_salutation_time(self):
    # 显示当前预定时间和说明
    plan_to_wake_time = character_data.action_info.plan_to_wake_time
    # 提供时、分设置按钮
    
def change_hour(self, morning_flag: bool = True):
    user_input = flow_handle.askfor_str("请输入小时(24小时制)：")
    if morning_flag:
        # 早安时间限制：0-9时
        user_input = max(0, min(9, int(user_input)))
        character_data.action_info.plan_to_wake_time[0] = user_input
    else:
        # 晚安时间限制：18-23时
        user_input = max(18, min(23, int(user_input)))
        character_data.action_info.plan_to_sleep_time[0] = user_input
```

### 5. AI行为集成

#### 跟随逻辑 (`handle_npc_ai.py`)
```python
# 跟随状态检查
if character_data.sp_flag.is_follow == 1:
    # 智能跟随模式：基础跟随
if character_data.sp_flag.is_follow == 2:
    # 高级跟随模式：更紧密跟随

# 跟随行为中断检查
if character_data.sp_flag.is_follow and character_data.behavior.behavior_id != constant.Behavior.WAIT:
    # 如果助理正在执行其他行为且不是等待，则暂时取消跟随
    character_data.sp_flag.is_follow = 0
```

#### 助理服务触发
通过二段行为标记 (`second_behavior`) 系统触发特定的助理服务：
- `chosen_as_assistant`: 被选为助理标记
- `not_as_assistant`: 非助理状态标记
- `ai_follow_X`: 跟随服务等级标记
- `morning_service_X`: 早安服务等级标记
- `night_service_X`: 晚安服务等级标记
- `support_service_on/off`: 辅佐服务开关标记
- `cohabitation_on/off`: 同居服务开关标记
- `love_support_on/off`: 助攻服务开关标记

## 系统特点与设计思路

### 1. 模块化设计
- **UI层**: `assistant_panel.py` 负责界面展示和用户交互
- **数据层**: CSV配置文件提供基础数据
- **逻辑层**: 各种处理函数负责具体功能实现
- **AI集成**: 通过二段行为标记与NPC AI系统无缝集成

### 2. 灵活的配置系统
- **多选项支持**: 每个服务可以有多个等级选项
- **条件检查**: 支持基于好感度、信任度的解锁条件
- **动态配置**: 运行时可以随时调整服务配置

### 3. 状态管理
- **角色状态同步**: 助理的跟随、宿舍等状态与游戏世界同步
- **行为标记系统**: 通过二段行为标记触发相应的AI行为
- **状态持久化**: 助理配置通过存档系统持久化保存

### 4. 用户体验
- **直观界面**: 清晰的服务选项展示和状态反馈
- **智能提示**: 解锁条件不满足时提供明确的失败原因
- **时间定制**: 支持精确的时间设置（早安/晚安服务）

### 5. 扩展性
- **新服务添加**: 通过修改CSV文件和添加相应处理逻辑可以轻松添加新服务
- **AI行为扩展**: 通过二段行为标记系统可以轻松扩展AI行为
- **UI组件复用**: 使用通用的选择面板组件，便于维护和扩展

## 关键交互点

### 1. 与角色系统的交互
- 角色的基础属性（好感度、信任度）影响服务解锁
- 角色的特殊状态（异常、监禁等）影响助理候选资格
- 角色的位置信息用于跟随和同居功能

### 2. 与时间系统的交互
- 早安/晚安服务依赖游戏时间系统
- 每周轮换功能与日期系统集成

### 3. 与AI系统的交互
- 通过二段行为标记触发特定AI行为
- 跟随状态直接影响NPC的行为决策
- 助理服务与NPC日常行为相集成

### 4. 与地图系统的交互
- 同居服务涉及角色宿舍位置的变更
- 跟随功能依赖地图系统的位置追踪

### 5. 与存档系统的交互
- 助理配置和状态通过存档系统持久化
- 支持游戏重启后状态恢复

## 总结

助理系统是erArk游戏中一个设计精良的功能模块，通过CSV配置、Python逻辑处理、UI界面展示和AI行为集成，提供了完整的助理管理功能。系统具有良好的模块化设计、灵活的配置机制、清晰的状态管理和良好的扩展性，为玩家提供了丰富的助理互动体验。