/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.common.math.DoubleUtils;
import com.google.common.math.ElementTypesAreNonnullByDefault;
import com.google.common.math.IntMath;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.math.ToDoubleRounder;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

@ElementTypesAreNonnullByDefault
@GwtCompatible(emulated=true)
public final class BigIntegerMath {
    @VisibleForTesting
    static final int SQRT2_PRECOMPUTE_THRESHOLD = 256;
    @VisibleForTesting
    static final BigInteger SQRT2_PRECOMPUTED_BITS = new BigInteger("16a09e667f3bcc908b2fb1366ea957d3e3adec17512775099da2f590b0667322a", 16);
    private static final double LN_10 = Math.log(10.0);
    private static final double LN_2 = Math.log(2.0);

    @Beta
    public static BigInteger ceilingPowerOfTwo(BigInteger x) {
        BigInteger bigInteger;
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(bigInteger, RoundingMode.CEILING));
    }

    @Beta
    public static BigInteger floorPowerOfTwo(BigInteger x) {
        BigInteger bigInteger;
        return BigInteger.ZERO.setBit(BigIntegerMath.log2(bigInteger, RoundingMode.FLOOR));
    }

    public static boolean isPowerOfTwo(BigInteger x) {
        BigInteger bigInteger;
        Preconditions.checkNotNull(x);
        return x.signum() > 0 && x.getLowestSetBit() == bigInteger.bitLength() - 1;
    }

    /*
     * WARNING - void declaration
     */
    public static int log2(BigInteger x, RoundingMode mode) {
        MathPreconditions.checkPositive("x", Preconditions.checkNotNull(x));
        int logFloor = x.bitLength() - 1;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(BigIntegerMath.isPowerOfTwo(x));
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                if (BigIntegerMath.isPowerOfTwo(x)) {
                    return logFloor;
                }
                return logFloor + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                void var2_3;
                BigInteger bigInteger;
                BigInteger bigInteger2;
                if (logFloor < 256) {
                    BigInteger halfPower = SQRT2_PRECOMPUTED_BITS.shiftRight(256 - logFloor);
                    if (x.compareTo(bigInteger2) <= 0) {
                        return logFloor;
                    }
                    return logFloor + 1;
                }
                bigInteger2 = bigInteger.pow(2);
                int n = bigInteger2.bitLength() - 1;
                if (n < 2 * logFloor + 1) {
                    return logFloor;
                }
                return (int)(var2_3 + true);
            }
        }
        throw new AssertionError();
    }

    /*
     * WARNING - void declaration
     */
    @GwtIncompatible
    public static int log10(BigInteger x, RoundingMode mode) {
        void var3_3;
        MathPreconditions.checkPositive("x", x);
        if (BigIntegerMath.fitsInLong(x)) {
            return LongMath.log10(x.longValue(), mode);
        }
        int approxLog10 = (int)((double)BigIntegerMath.log2(x, RoundingMode.FLOOR) * LN_2 / LN_10);
        BigInteger approxPow = BigInteger.TEN.pow(approxLog10);
        int approxCmp = approxPow.compareTo(x);
        if (approxCmp > 0) {
            do {
                --approxLog10;
            } while ((approxCmp = (approxPow = approxPow.divide(BigInteger.TEN)).compareTo(x)) > 0);
        } else {
            BigInteger nextPow;
            int nextCmp;
            while ((nextCmp = (nextPow = BigInteger.TEN.multiply(approxPow)).compareTo(x)) <= 0) {
                ++approxLog10;
                approxPow = nextPow;
                approxCmp = nextCmp;
            }
        }
        int floorLog = approxLog10;
        void floorPow = var3_3;
        int floorCmp = approxCmp;
        switch (mode) {
            case UNNECESSARY: {
                void var2_2;
                MathPreconditions.checkRoundingUnnecessary(var2_2 == false);
            }
            case DOWN: 
            case FLOOR: {
                return floorLog;
            }
            case UP: 
            case CEILING: {
                if (floorPow.equals(x)) {
                    return floorLog;
                }
                return floorLog + 1;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                void var1_1;
                BigInteger bigInteger;
                BigInteger x2 = x.pow(2);
                BigInteger halfPowerSquared = floorPow.pow(2).multiply(BigInteger.TEN);
                if (bigInteger.compareTo((BigInteger)var1_1) <= 0) {
                    return floorLog;
                }
                return floorLog + 1;
            }
        }
        throw new AssertionError();
    }

    /*
     * WARNING - void declaration
     */
    @GwtIncompatible
    public static BigInteger sqrt(BigInteger x2, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x2);
        if (BigIntegerMath.fitsInLong(x2)) {
            return BigInteger.valueOf(LongMath.sqrt(x2.longValue(), mode));
        }
        BigInteger sqrtFloor = BigIntegerMath.sqrtFloor(x2);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(sqrtFloor.pow(2).equals(x2));
            }
            case DOWN: 
            case FLOOR: {
                return sqrtFloor;
            }
            case UP: 
            case CEILING: {
                void var1_3;
                int sqrtFloorInt = sqrtFloor.intValue();
                boolean x2 = sqrtFloorInt * var1_3 == x2.intValue() && sqrtFloor.pow(2).equals(x2);
                if (x2) {
                    return sqrtFloor;
                }
                return sqrtFloor.add(BigInteger.ONE);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                void var2_4;
                BigInteger bigInteger;
                BigInteger bigInteger2 = sqrtFloor.pow(2).add(sqrtFloor);
                if (bigInteger2.compareTo(bigInteger) >= 0) {
                    return sqrtFloor;
                }
                return var2_4.add(BigInteger.ONE);
            }
        }
        throw new AssertionError();
    }

    /*
     * WARNING - void declaration
     */
    @GwtIncompatible
    private static BigInteger sqrtFloor(BigInteger x) {
        void var1_2;
        BigInteger sqrt22;
        int log2 = BigIntegerMath.log2(x, RoundingMode.FLOOR);
        if (log2 < 1023) {
            sqrt22 = BigIntegerMath.sqrtApproxWithDoubles(x);
        } else {
            void sqrt22;
            int shift = sqrt22 - 52 & 0xFFFFFFFE;
            sqrt22 = BigIntegerMath.sqrtApproxWithDoubles(x.shiftRight(shift)).shiftLeft(shift >> 1);
        }
        BigInteger sqrt1 = sqrt22.add(x.divide(sqrt22)).shiftRight(1);
        if (sqrt22.equals(sqrt1)) {
            return sqrt22;
        }
        while ((sqrt1 = (sqrt22 = sqrt1).add(x.divide(sqrt22)).shiftRight(1)).compareTo(sqrt22) < 0) {
        }
        return var1_2;
    }

    @GwtIncompatible
    private static BigInteger sqrtApproxWithDoubles(BigInteger x) {
        return DoubleMath.roundToBigInteger(Math.sqrt(DoubleUtils.bigToDouble(x)), RoundingMode.HALF_EVEN);
    }

    /*
     * WARNING - void declaration
     */
    @GwtIncompatible
    public static double roundToDouble(BigInteger x, RoundingMode mode) {
        void var1_1;
        BigInteger bigInteger;
        return BigIntegerToDoubleRounder.INSTANCE.roundToDouble(bigInteger, (RoundingMode)var1_1);
    }

    /*
     * WARNING - void declaration
     */
    @GwtIncompatible
    public static BigInteger divide(BigInteger p, BigInteger q, RoundingMode mode) {
        void var2_2;
        void var1_1;
        Number number;
        Number pDec = new BigDecimal(p);
        BigDecimal qDec = new BigDecimal(q);
        return ((BigDecimal)number).divide((BigDecimal)var1_1, 0, (RoundingMode)var2_2).toBigIntegerExact();
    }

    /*
     * WARNING - void declaration
     */
    public static BigInteger factorial(int n) {
        void var3_5;
        void var1_2;
        MathPreconditions.checkNonNegative("n", n);
        if (n < LongMath.factorials.length) {
            return BigInteger.valueOf(LongMath.factorials[n]);
        }
        int n2 = n;
        int approxSize = IntMath.divide(n2 * IntMath.log2(n2, RoundingMode.CEILING), 64, RoundingMode.CEILING);
        ArrayList<BigInteger> bignums = new ArrayList<BigInteger>(approxSize);
        int startingNumber = LongMath.factorials.length;
        long product = LongMath.factorials[startingNumber - 1];
        int shift = Long.numberOfTrailingZeros(product);
        int productBits = LongMath.log2(product >>= shift, RoundingMode.FLOOR) + 1;
        int bits = LongMath.log2(startingNumber, RoundingMode.FLOOR) + 1;
        int nextPowerOfTwo = 1 << bits - 1;
        for (long num = (long)startingNumber; num <= (long)n; ++num) {
            int n3;
            if ((num & (long)nextPowerOfTwo) != 0L) {
                nextPowerOfTwo <<= 1;
                ++bits;
            }
            int tz = Long.numberOfTrailingZeros(num);
            long normalizedNum = num >> tz;
            shift += tz;
            if ((n3 = bits - n3) + productBits >= 64) {
                bignums.add(BigInteger.valueOf(product));
                product = 1L;
            }
            productBits = LongMath.log2(product *= normalizedNum, RoundingMode.FLOOR) + 1;
        }
        if (product > 1L) {
            bignums.add(BigInteger.valueOf(product));
        }
        return BigIntegerMath.listProduct((List<BigInteger>)var1_2).shiftLeft((int)var3_5);
    }

    static BigInteger listProduct(List<BigInteger> nums) {
        List<BigInteger> list;
        return BigIntegerMath.listProduct(nums, 0, list.size());
    }

    /*
     * WARNING - void declaration
     */
    static BigInteger listProduct(List<BigInteger> nums, int start, int end) {
        void var2_2;
        void var3_3;
        List<BigInteger> list;
        void var1_1;
        switch (end - start) {
            case 0: {
                return BigInteger.ONE;
            }
            case 1: {
                return nums.get(start);
            }
            case 2: {
                return nums.get(start).multiply(nums.get(start + 1));
            }
            case 3: {
                return nums.get(start).multiply(nums.get(start + 1)).multiply(nums.get(start + 2));
            }
        }
        int m = end + start >>> 1;
        return BigIntegerMath.listProduct(nums, (int)var1_1, m).multiply(BigIntegerMath.listProduct(list, (int)var3_3, (int)var2_2));
    }

    /*
     * WARNING - void declaration
     */
    public static BigInteger binomial(int n, int k) {
        void var3_3;
        void var2_2;
        int bits;
        MathPreconditions.checkNonNegative("n", n);
        MathPreconditions.checkNonNegative("k", k);
        Preconditions.checkArgument(k <= n, "k (%s) > n (%s)", k, n);
        if (k > n >> 1) {
            k = n - k;
        }
        if (k < LongMath.biggestBinomials.length && n <= LongMath.biggestBinomials[k]) {
            return BigInteger.valueOf(LongMath.binomial(n, k));
        }
        BigInteger accum = BigInteger.ONE;
        long numeratorAccum = n;
        long denominatorAccum = 1L;
        int numeratorBits = bits = LongMath.log2(n, RoundingMode.CEILING);
        for (int i = 1; i < k; ++i) {
            int p = n - i;
            int q = i + 1;
            if (numeratorBits + bits >= 63) {
                accum = accum.multiply(BigInteger.valueOf(numeratorAccum)).divide(BigInteger.valueOf(denominatorAccum));
                numeratorAccum = p;
                denominatorAccum = q;
                numeratorBits = bits;
                continue;
            }
            numeratorAccum *= (long)p;
            denominatorAccum *= (long)q;
            numeratorBits += bits;
        }
        return var2_2.multiply(BigInteger.valueOf((long)var3_3)).divide(BigInteger.valueOf(denominatorAccum));
    }

    @GwtIncompatible
    static boolean fitsInLong(BigInteger x) {
        return x.bitLength() <= 63;
    }

    private BigIntegerMath() {
    }

    @GwtIncompatible
    private static class BigIntegerToDoubleRounder
    extends ToDoubleRounder<BigInteger> {
        static final BigIntegerToDoubleRounder INSTANCE = new BigIntegerToDoubleRounder();

        private BigIntegerToDoubleRounder() {
        }

        @Override
        double roundToDoubleArbitrarily(BigInteger bigInteger) {
            return DoubleUtils.bigToDouble(bigInteger);
        }

        @Override
        int sign(BigInteger bigInteger) {
            return bigInteger.signum();
        }

        /*
         * WARNING - void declaration
         */
        @Override
        BigInteger toX(double d, RoundingMode mode) {
            void var3_2;
            return DoubleMath.roundToBigInteger(d, (RoundingMode)var3_2);
        }

        /*
         * WARNING - void declaration
         */
        @Override
        BigInteger minus(BigInteger a, BigInteger b) {
            void var2_2;
            return a.subtract((BigInteger)var2_2);
        }
    }
}

