/*
 * This file is part of the TrinityCore Project. See AUTHORS file for Copyright information
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include "ScriptMgr.h"
#include "molten_core.h"
#include "ScriptedCreature.h"
#include "SpellScript.h"

enum Spells
{
    SPELL_ARCANE_EXPLOSION      = 19712,
    SPELL_SHAZZRAH_CURSE        = 19713,
    SPELL_MAGIC_GROUNDING       = 19714,
    SPELL_COUNTERSPELL          = 19715,
    SPELL_SHAZZRAH_GATE_DUMMY   = 23138, // Teleports to and attacks a random target.
    SPELL_SHAZZRAH_GATE         = 23139,
};

enum Events
{
    EVENT_ARCANE_EXPLOSION              = 1,
    EVENT_ARCANE_EXPLOSION_TRIGGERED    = 2,
    EVENT_SHAZZRAH_CURSE                = 3,
    EVENT_MAGIC_GROUNDING               = 4,
    EVENT_COUNTERSPELL                  = 5,
    EVENT_SHAZZRAH_GATE                 = 6,
};

struct boss_shazzrah : public BossAI
{
    boss_shazzrah(Creature* creature) : BossAI(creature, BOSS_SHAZZRAH) { }

    void JustEngagedWith(Unit* target) override
    {
        BossAI::JustEngagedWith(target);
        events.ScheduleEvent(EVENT_ARCANE_EXPLOSION, 6s);
        events.ScheduleEvent(EVENT_SHAZZRAH_CURSE, 10s);
        events.ScheduleEvent(EVENT_MAGIC_GROUNDING, 24s);
        events.ScheduleEvent(EVENT_COUNTERSPELL, 15s);
        events.ScheduleEvent(EVENT_SHAZZRAH_GATE, 45s);
    }

    void UpdateAI(uint32 diff) override
    {
        if (!UpdateVictim())
            return;

        events.Update(diff);

        if (me->HasUnitState(UNIT_STATE_CASTING))
            return;

        while (uint32 eventId = events.ExecuteEvent())
        {
            switch (eventId)
            {
                case EVENT_ARCANE_EXPLOSION:
                    DoCastVictim(SPELL_ARCANE_EXPLOSION);
                    events.ScheduleEvent(EVENT_ARCANE_EXPLOSION, 4s, 7s);
                    break;
                // Triggered subsequent to using "Gate of Shazzrah".
                case EVENT_ARCANE_EXPLOSION_TRIGGERED:
                    DoCastVictim(SPELL_ARCANE_EXPLOSION);
                    break;
                case EVENT_SHAZZRAH_CURSE:
                    if (Unit* target = SelectTarget(SelectTargetMethod::Random, 0, 0.0f, true, true, -SPELL_SHAZZRAH_CURSE))
                        DoCast(target, SPELL_SHAZZRAH_CURSE);
                    events.ScheduleEvent(EVENT_SHAZZRAH_CURSE, 25s, 30s);
                    break;
                case EVENT_MAGIC_GROUNDING:
                    DoCast(me, SPELL_MAGIC_GROUNDING);
                    events.ScheduleEvent(EVENT_MAGIC_GROUNDING, 35s);
                    break;
                case EVENT_COUNTERSPELL:
                    DoCastVictim(SPELL_COUNTERSPELL);
                    events.ScheduleEvent(EVENT_COUNTERSPELL, 16s, 20s);
                    break;
                case EVENT_SHAZZRAH_GATE:
                    ResetThreatList();
                    DoCastAOE(SPELL_SHAZZRAH_GATE_DUMMY);
                    events.ScheduleEvent(EVENT_ARCANE_EXPLOSION_TRIGGERED, 2s);
                    events.RescheduleEvent(EVENT_ARCANE_EXPLOSION, 3s, 6s);
                    events.ScheduleEvent(EVENT_SHAZZRAH_GATE, 45s);
                    break;
                default:
                    break;
            }

            if (me->HasUnitState(UNIT_STATE_CASTING))
                return;
        }

        DoMeleeAttackIfReady();
    }
};

// 23138 - Gate of Shazzrah
class spell_shazzrah_gate_dummy : public SpellScript
{
    PrepareSpellScript(spell_shazzrah_gate_dummy);

    bool Validate(SpellInfo const* /*spellInfo*/) override
    {
        return ValidateSpellInfo({ SPELL_SHAZZRAH_GATE });
    }

    void FilterTargets(std::list<WorldObject*>& targets)
    {
        if (targets.empty())
            return;

        WorldObject* target = Trinity::Containers::SelectRandomContainerElement(targets);
        targets.clear();
        targets.push_back(target);
    }

    void HandleScript(SpellEffIndex /*effIndex*/)
    {
        if (Unit* target = GetHitUnit())
        {
            target->CastSpell(GetCaster(), SPELL_SHAZZRAH_GATE, true);
            if (Creature* creature = GetCaster()->ToCreature())
                creature->AI()->AttackStart(target); // Attack the target which caster will teleport to.
        }
    }

    void Register() override
    {
        OnObjectAreaTargetSelect += SpellObjectAreaTargetSelectFn(spell_shazzrah_gate_dummy::FilterTargets, EFFECT_0, TARGET_UNIT_SRC_AREA_ENEMY);
        OnEffectHitTarget += SpellEffectFn(spell_shazzrah_gate_dummy::HandleScript, EFFECT_0, SPELL_EFFECT_DUMMY);
    }
};

void AddSC_boss_shazzrah()
{
    RegisterMoltenCoreCreatureAI(boss_shazzrah);
    RegisterSpellScript(spell_shazzrah_gate_dummy);
}
