# 层次化配置系统说明

## 概述

层次化配置系统按照以下结构组织所有组件信息：

```
窗口类 (Window Classes)
├── 画布代码 (Canvas Code)
│   ├── 画布尺寸
│   ├── 画布背景
│   ├── 网格设置
│   └── 代码生成方法
├── 功能代码 (Functional Code)
│   └── 文件
│       ├── code_generator
│       ├── event_function_generator
│       ├── control_manager
│       └── event_manager
├── 窗口组件 (Window Components)
│   └── 各种组件
│       ├── 基本信息
│       ├── 默认属性
│       └── 绑定的事件 (Events)
│           ├── 事件ID
│           ├── 事件名称
│           ├── 信号
│           ├── 描述
│           └── 函数名称模板
└── 函数和类 (Functions and Classes)
    ├── 函数 (Functions)
    │   └── 函数信息
    │       ├── 名称
    │       ├── 所属类
    │       ├── 签名
    │       └── 文档字符串
    └── 类 (Classes)
        └── 类信息
            ├── 名称
            ├── 基类
            ├── 文档字符串
            ├── 成员 (Members)
            │   ├── 方法 (Methods)
            │   ├── 属性 (Properties)
            │   └── 属性 (Attributes)
            └── 数据结构 (Data Structures)
                └── 数据结构成员
                    ├── 类型
                    └── 结构
```

## 配置文件结构

### hierarchical_config.json

主配置文件，包含完整的层次化结构：

```json
{
  "window_classes": {
    "GeneratedWindow": {
      "name": "GeneratedWindow",
      "base_class": "QMainWindow",
      "canvas_code": { ... },
      "functional_code": { ... },
      "window_components": { ... },
      "functions_and_classes": { ... }
    }
  }
}
```

## 使用示例

### 1. 导出层次化配置

```bash
python lib/hierarchical_config_exporter.py
```

### 2. 加载层次化配置

```python
from lib.hierarchical_config_loader import HierarchicalConfigLoader

loader = HierarchicalConfigLoader()
if loader.load():
    # 获取窗口类信息
    window_class = loader.get_window_class("GeneratedWindow")
    
    # 获取画布代码信息
    canvas_code = loader.get_canvas_code("GeneratedWindow")
    
    # 获取功能代码信息
    functional_code = loader.get_functional_code("GeneratedWindow")
    
    # 获取窗口组件
    components = loader.get_window_components("GeneratedWindow")
    
    # 获取组件的绑定事件
    events = loader.get_component_events("GeneratedWindow", "Label")
    
    # 获取函数和类
    functions_and_classes = loader.get_functions_and_classes("GeneratedWindow")
    
    # 获取类信息（包括成员和数据结构）
    class_info = loader.get_class_info("GeneratedWindow", "CodeGenerator")
    
    # 获取类的成员
    members = loader.get_class_members("GeneratedWindow", "CodeGenerator")
    
    # 获取类的数据结构
    data_structures = loader.get_data_structures("GeneratedWindow", "CodeGenerator")
    
    # 获取数据结构的成员
    ds_members = loader.get_data_structure_members(
        "GeneratedWindow", 
        "CodeGenerator", 
        "control_manager"
    )
```

### 3. 遍历所有信息

```python
loader = HierarchicalConfigLoader()
loader.load()

# 列出所有窗口类
window_classes = loader.list_all_window_classes()

for window_name in window_classes:
    print(f"窗口类: {window_name}")
    
    # 列出所有组件
    components = loader.list_all_components(window_name)
    for component_name in components:
        events = loader.get_component_events(window_name, component_name)
        print(f"  组件: {component_name}, 事件数: {len(events)}")
    
    # 列出所有函数
    functions = loader.list_all_functions(window_name)
    print(f"  函数数: {len(functions)}")
    
    # 列出所有类
    classes = loader.list_all_classes(window_name)
    for class_name in classes:
        members = loader.get_class_members(window_name, class_name)
        data_structures = loader.get_data_structures(window_name, class_name)
        print(f"  类: {class_name}")
        print(f"    成员数: {len(members)}")
        print(f"    数据结构数: {len(data_structures)}")
```

## 配置层次说明

### 1. 窗口类 (Window Classes)

顶层结构，包含所有窗口类的定义。

### 2. 画布代码 (Canvas Code)

包含画布相关的配置和代码生成方法：
- 画布尺寸
- 画布背景色
- 网格设置
- 代码生成方法映射

### 3. 功能代码 (Functional Code)

包含所有功能模块的信息：
- 模块名称
- 文件路径
- 文档字符串
- 函数和类列表

### 4. 窗口组件 (Window Components)

包含所有可用的窗口组件：
- 组件基本信息
- 默认属性
- 默认大小
- 支持的事件列表

### 5. 绑定的事件 (Events)

每个组件下的事件信息：
- 事件ID
- 事件名称
- Qt信号名称
- 事件描述
- 函数名称模板
- 是否已绑定

### 6. 函数和类 (Functions and Classes)

包含所有函数和类的详细信息：

#### 函数 (Functions)
- 函数名称
- 所属类
- 函数签名
- 参数信息
- 返回类型
- 文档字符串

#### 类 (Classes)
- 类名称
- 基类列表
- 文档字符串
- 成员（方法、属性、变量）
- 数据结构

#### 成员 (Members)
- 方法 (Methods): 包含签名、参数、文档
- 属性 (Properties): 包含文档
- 属性 (Attributes): 包含值和类型

#### 数据结构 (Data Structures)
- 数据结构名称
- 数据类型
- 数据结构成员
  - 类型
  - 结构描述

## 优势

1. **层次清晰**: 按照逻辑层次组织，易于理解和导航
2. **信息完整**: 包含所有相关信息，从窗口到数据结构成员
3. **易于查询**: 提供丰富的查询接口，可以快速定位信息
4. **可扩展**: 结构设计灵活，易于添加新的层次和信息

## 文件清单

- `hierarchical_config_exporter.py` - 层次化配置导出工具
- `hierarchical_config_loader.py` - 层次化配置加载器
- `hierarchical_config.json` - 层次化配置文件（自动生成）
- `HIERARCHICAL_CONFIG_README.md` - 本说明文件

