#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
可视化编程工具主程序
基于PySide6实现的类似易语言的图形化编程工具
"""

# ========== 抑制调试器相关的警告（必须在其他导入之前） ========== 
import warnings  # 警告过滤器，用于控制警告消息的显示
warnings.filterwarnings('ignore', message='.*incompatible copy of pydevd.*')
warnings.filterwarnings('ignore', message='.*pydevd.*already imported.*')
warnings.filterwarnings('ignore', category=UserWarning, module='debugpy.*')

# 导入标准库模块
import sys  # 系统相关功能和参数
import os  # 操作系统接口，用于文件和目录操作
from typing import Optional  # 类型提示，用于标注可选类型

# 从PySide6.QtWidgets导入Qt界面组件
from PySide6.QtWidgets import (
    QApplication,  # Qt应用程序主类，管理整个应用程序的控制流
    QMainWindow,  # 主窗口基类，提供标准的应用程序窗口框架
    QWidget,  # 所有用户界面对象的基类
    QVBoxLayout,  # 垂直布局管理器
    QHBoxLayout,  # 水平布局管理器
    QSplitter,  # 分割器，用于创建可调整大小的面板
    QMenuBar,  # 菜单栏组件
    QMenu,  # 菜单组件
    QToolBar,  # 工具栏组件
    QStatusBar,  # 状态栏组件
    QFileDialog,  # 文件选择对话框
    QMessageBox,  # 消息提示框
    QDockWidget,  # 停靠窗口组件
    QListWidget,  # 列表控件
    QListWidgetItem,  # 列表项
    QLabel,  # 文本标签控件
    QPushButton,  # 按钮控件
    QFrame,  # 框架控件
    QScrollArea,  # 滚动区域控件
    QTabWidget,  # 选项卡控件
    QToolButton,  # 工具按钮
    QButtonGroup,  # 按钮组
    QGroupBox,  # 分组框
    QGridLayout,  # 网格布局
    QSizePolicy,  # 大小策略
    QComboBox,  # 下拉选择框
    QCheckBox,  # 复选框
    QPlainTextEdit,  # 纯文本编辑器
    QSlider,  # 滑动条
    QTextEdit  # 富文本编辑器
)
# 从PySide6.QtCore导入Qt核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间，包含各种常量
    QSize,  # 大小对象
    Signal,  # 信号定义
    QPoint,  # 点坐标对象
    QMimeData  # MIME数据对象，用于拖放操作
)
# 从PySide6.QtGui导入Qt图形界面相关类
from PySide6.QtGui import (
    QIcon,  # 图标类
    QPixmap,  # 像素图
    QFont,  # 字体类
    QDrag,  # 拖放操作
    QPainter,  # 绘图器
    QPen,  # 画笔
    QBrush,  # 画刷
    QAction,  # 动作类
    QColor,  # 颜色类
    QTextCursor  # 文本光标
)

# 导入自定义模块
from module.control_pyside import ControlManager  # 控件管理器，管理所有控件的创建、删除和操作
from module.control_library_pyside import ControlLibrary  # 控件库，提供控件定义和默认属性
from module.design_canvas_pyside import DesignCanvas  # 设计画布，用于可视化编辑界面
from module.control_palette_ui import ControlPaletteWidget  # 控件调色板UI，显示可用控件列表
from module.main_window_ui import MainWindowUI  # 主窗口UI管理器，负责创建菜单栏、工具栏等
from module.project_manager_pyside import ProjectManager  # 项目管理器，处理项目的保存和加载
from module.code_generator_pyside import CodeGenerator  # 代码生成器，将设计转换为Python代码
from module.code_dialog_pyside import CodeDialog  # 代码对话框，用于显示和编辑生成的代码
from module.undo_redo_manager_pyside import UndoRedoManager  # 撤销重做管理器，实现操作历史管理
from module.preview_window_pyside import BuiltInPreviewWindow  # 内置预览窗口，实时预览设计效果
from module.version import about_text  # 关于对话框文本内容
from module.property_editor_pyside import PropertyEditor  # 属性编辑器，编辑控件的属性
from module.monaco_editor_widget import MonacoEditorWidget  # Monaco Editor 代码编辑器
from module.main_window_pyside import MainWindow  # 主窗口类（新版本）
from module.code_parser_pyside import CodeParser  # 代码解析器，解析Python代码并转换为控件


def _resolve_app_secrets_path() -> str:
    """
    解析应用程序密钥文件的路径
    
    Returns:
        str: 应用程序密钥文件的完整路径
    """
    # 重新导入系统模块（函数内部导入，避免全局命名空间污染）
    import sys  # 系统相关功能
    import os  # 操作系统接口
    
    # ========== 判断应用是否被打包 ==========
    # 判断应用是否被打包（通过pyinstaller等工具打包成可执行文件）
    # getattr(sys, 'frozen', False)：检查sys模块是否有frozen属性（打包后的程序会有此属性）
    # 如果是打包后的可执行文件，使用可执行文件所在目录；否则使用脚本所在目录
    base = os.path.dirname(sys.executable) if getattr(sys, 'frozen', False) else os.path.dirname(os.path.abspath(__file__))
    
    # ========== 构建可能的密钥文件路径 ==========
    # 尝试第一个路径：Python/app_secrets.json（大写P，Windows系统常见的大小写）
    p1 = os.path.join(base, "Python", "app_secrets.json")
    # 尝试第二个路径：python/app_secrets.json（小写p，Linux/Mac系统常见的大小写）
    p2 = os.path.join(base, "python", "app_secrets.json")
    
    # ========== 返回存在的路径 ==========
    # 检查第一个路径是否存在文件
    # 如果第一个路径存在则返回第一个，否则如果第二个存在则返回第二个，都不存在则返回第一个路径
    # os.path.isfile(p1)：检查p1路径是否为文件且存在
    # 三元表达式嵌套：先检查p1，如果不存在则检查p2，如果p2也不存在则返回p1（默认路径）
    return p1 if os.path.isfile(p1) else (p2 if os.path.isfile(p2) else p1)

class _LegacyMainWindow(QMainWindow):
    """
    主窗口类（旧版）
    继承自QMainWindow，提供完整的可视化编程工具主窗口功能
    包括：控件管理、项目管理、代码生成、撤销重做等功能
    """
    
    def __init__(self):
        """
        初始化主窗口
        创建所有必要的管理器组件并初始化用户界面
        """
        # 调用父类构造函数，初始化Qt主窗口
        super().__init__()
        
        # 控件管理器：负责所有控件的创建、删除、选择和属性管理
        self.control_manager = ControlManager()
        
        # 创建项目管理器：负责项目的保存、加载、修改状态跟踪等
        # 传入控件管理器，使其能够访问和操作控件数据
        self.project_manager = ProjectManager(self.control_manager)
        
        # 创建代码生成器：将画布上的控件设计转换为Python代码
        # 传入控件管理器，以便读取控件信息并生成代码
        self.code_generator = CodeGenerator(self.control_manager)
        
        # 创建撤销重做管理器：实现操作的撤销和重做功能
        # 传入控件管理器，以便保存和恢复控件状态
        self.undo_redo_manager = UndoRedoManager(self.control_manager)
        
        # 创建UI组件管理器：负责创建和管理菜单栏、工具栏、状态栏等UI组件
        # 传入self（主窗口），使其能够在主窗口上创建UI元素
        self.ui_manager = MainWindowUI(self)
        
        # 创建代码解析器：将Python代码解析并转换为控件信息
        self.code_parser = CodeParser()
        
        # 预览窗口对象，初始为None，当需要预览时会创建实例
        self.preview_window = None
        
        # 初始化UI：创建所有界面元素，包括布局、控件、菜单等
        self._init_ui()
        
        # 设置窗口标题：显示应用程序名称和作者信息
        self.setWindowTitle("python Visual Studio Code")
        # 设置窗口最小尺寸：宽度1200像素，高度800像素
        self.setMinimumSize(1200, 800)
        
        # 连接信号：将各个组件的事件信号连接到对应的处理函数
        self._connect_signals()
    
    def _init_ui(self):
        """
        初始化用户界面
        创建主窗口的所有UI组件，包括代码编辑器、设计画布、控件调色板、属性编辑器等
        """
        # 创建中央部件：主窗口的中央区域，所有主要内容都在这里
        central_widget = QWidget()
        # 将中央部件设置为主窗口的中央部件
        self.setCentralWidget(central_widget)
        
        # 创建主布局：使用垂直布局以便更好地利用垂直空间
        main_layout = QVBoxLayout(central_widget)
        # 设置布局边距：左、上、右、下各5像素
        main_layout.setContentsMargins(5, 5, 5, 5)
        # 设置布局中组件之间的间距为5像素
        main_layout.setSpacing(5)
        
        # 创建水平分割器作为主要内容区域：用于分割左侧代码区、中间画布区、右侧面板区
        splitter = QSplitter(Qt.Horizontal)  # 水平方向的分割器
        # 允许子部件折叠
        splitter.setChildrenCollapsible(True)
        # 设置大小策略：水平和垂直方向都可以扩展
        splitter.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        # 将分割器添加到主布局中
        main_layout.addWidget(splitter)

        # ========== 创建左侧代码展示页 ==========
        # 创建代码容器：包含代码编辑器和工具栏的容器
        code_container = QWidget()
        # 为代码容器创建垂直布局
        code_layout = QVBoxLayout(code_container)
        # 设置代码容器布局的边距：左、上、右、下各4像素
        code_layout.setContentsMargins(4, 4, 4, 4)
        # 设置代码容器布局中组件间距为4像素
        code_layout.setSpacing(4)
        
        # 创建工具栏：水平布局，包含框架选择、同步选项、刷新、保存、复制、解析按钮
        header = QHBoxLayout()
        # 添加"框架："标签
        header.addWidget(QLabel("框架："))
        # 创建框架选择下拉框：用于选择代码生成的目标框架（PySide或TKinter）
        self.framework_combo_left = QComboBox()
        # 添加框架选项：PySide和TKinter
        self.framework_combo_left.addItems(["PySide", "TKinter"])
        # 将框架下拉框添加到工具栏
        header.addWidget(self.framework_combo_left)
        # 创建"同步生成"复选框：勾选后，画布变化时自动刷新代码
        self.sync_checkbox_left = QCheckBox("同步生成")
        # 默认选中同步生成
        self.sync_checkbox_left.setChecked(True)
        # 将同步生成复选框添加到工具栏
        header.addWidget(self.sync_checkbox_left)
        # 创建"刷新"按钮：手动刷新代码视图
        self.refresh_button_left = QPushButton("刷新")
        # 将刷新按钮添加到工具栏
        header.addWidget(self.refresh_button_left)
        # 创建"保存"按钮：将代码保存到文件
        self.save_button_left = QPushButton("保存")
        # 将保存按钮添加到工具栏
        header.addWidget(self.save_button_left)
        # 创建"复制"按钮：将代码复制到剪贴板
        self.copy_button_left = QPushButton("复制")
        # 将复制按钮添加到工具栏
        header.addWidget(self.copy_button_left)
        # 创建"解析"按钮：将代码解析并同步到画布
        self.sync_btn_left = QPushButton("解析")
        # 将解析按钮添加到工具栏
        header.addWidget(self.sync_btn_left)
        # 添加弹性空间，使按钮靠左对齐
        header.addStretch()
        # 将工具栏布局添加到代码容器的布局中
        code_layout.addLayout(header)
        
        # 创建代码编辑器：Monaco Editor 代码编辑组件
        self.code_view_editor_left = MonacoEditorWidget()
        # 设置为可编辑（非只读）
        self.code_view_editor_left.setReadOnly(False)
        # 设置不自动换行：代码编辑器应该保持原始格式
        self.code_view_editor_left.setLineWrapMode(QPlainTextEdit.NoWrap)
        # 设置字体：使用Consolas等宽字体，字号10，便于代码阅读
        self.code_view_editor_left.setFont(QFont("Consolas", 10))
        # 将代码编辑器添加到代码容器布局中
        code_layout.addWidget(self.code_view_editor_left)
        # 将代码容器添加到分割器的左侧
        splitter.addWidget(code_container)

        # ========== 创建设计画布（中间区域）==========
        # 创建设计画布对象：用于可视化编辑界面控件的画布
        self.design_canvas = DesignCanvas()
        # 将控件管理器传递给画布，使画布能够管理控件
        self.design_canvas.set_control_manager(self.control_manager)
        # 设置画布大小策略：固定大小（不随父容器变化）
        self.design_canvas.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
        
        # 创建滚动区域：包裹画布，当画布超出可视区域时显示滚动条
        canvas_scroll = QScrollArea()
        # 禁用自动调整大小：保持画布的固定尺寸
        canvas_scroll.setWidgetResizable(False)
        # 设置水平滚动条策略：需要时显示滚动条
        canvas_scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        # 设置垂直滚动条策略：需要时显示滚动条
        canvas_scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        # 将画布设置为滚动区域的子部件
        canvas_scroll.setWidget(self.design_canvas)
        # 将滚动区域（包含画布）添加到分割器的中间位置
        splitter.addWidget(canvas_scroll)
        
        # ========== 创建右侧垂直分割面板 ==========
        # 创建控件调色板：显示所有可用的控件类型，用户可以从这里拖拽控件到画布
        self.control_palette = ControlPaletteWidget()
        # 设置控件调色板的最小宽度为0（允许完全折叠）
        self.control_palette.setMinimumWidth(0)
        
        # 创建属性编辑器：用于编辑选中控件的属性
        self.property_editor = PropertyEditor()
        # 设置属性编辑器的最小宽度为240像素
        self.property_editor.setMinimumWidth(240)
        
        # 创建右侧垂直分割器：将控件调色板和属性编辑器上下排列
        right_splitter = QSplitter(Qt.Vertical)  # 垂直方向的分割器
        # 允许子部件折叠
        right_splitter.setChildrenCollapsible(True)
        # 将控件调色板添加到上方
        right_splitter.addWidget(self.control_palette)
        # 将属性编辑器添加到下方
        right_splitter.addWidget(self.property_editor)
        # 设置分割条宽度为6像素
        right_splitter.setHandleWidth(6)
        # 设置两个区域的高度比例：上300像素，下400像素
        right_splitter.setSizes([300, 400])
        # 将右侧分割器添加到主分割器的最右侧
        splitter.addWidget(right_splitter)
        
        # 将设计画布传递给撤销重做管理器：使撤销重做功能能够操作画布
        self.undo_redo_manager.set_design_canvas(self.design_canvas)
        
        # ========== 设置主分割器的比例和属性 ==========
        # 设置分割条宽度为6像素
        splitter.setHandleWidth(6)
        # 设置三个区域的宽度：左侧代码区520像素，中间画布区980像素，右侧面板区300像素
        splitter.setSizes([520, 980, 300])
        # 设置左侧区域的拉伸因子为0（不拉伸）
        splitter.setStretchFactor(0, 0)
        # 设置中间区域的拉伸因子为1（可以拉伸，占据剩余空间）
        splitter.setStretchFactor(1, 1)
        # 设置右侧区域的拉伸因子为0（不拉伸）
        splitter.setStretchFactor(2, 0)
        # 尝试设置中间区域不可折叠（防止误操作导致画布消失）
        try:
            splitter.setCollapsible(1, False)
        except Exception:
            pass

        # 计算画布的显示尺寸：取画布默认宽度和600中的较大值
        display_width = max(600, self.design_canvas.canvas_width)
        # 计算画布的显示高度：取画布默认高度和400中的较大值
        display_height = max(400, self.design_canvas.canvas_height)
        # 尝试设置画布的固定尺寸
        try:
            self.design_canvas.setFixedSize(display_width, display_height)
        except Exception:
            pass

        # 取消底部宽度滑条，保持画布宽度固定，通过分割条调节左右面板
        
        # 创建菜单栏
        self.ui_manager.create_menu_bar()
        self.menuBar().setMaximumHeight(28)  # 限制菜单栏高度
        
        # 创建工具栏
        self.ui_manager.create_tool_bar()
        # ========== 设置工具栏高度 ==========
        # 查找主窗口中所有的工具栏组件（可能有多个工具栏）
        for toolbar in self.findChildren(QToolBar):
            # 为每个工具栏设置最大高度为36像素（限制工具栏高度，保持界面整洁）
            toolbar.setMaximumHeight(36)
        
        # 创建状态栏
        self.ui_manager.create_status_bar()

        # 初始化控件选择器列表
        self.property_editor.refresh_control_list(self.control_manager)
        # 建立初始撤销快照作为基线
        try:
            self.undo_redo_manager.save_state("初始状态")
        except Exception:
            pass

        # 代码展示页交互
        self.framework_combo_left.currentTextChanged.connect(self._refresh_code_view_left)
        self.refresh_button_left.clicked.connect(self._refresh_code_view_left)
        self.sync_btn_left.clicked.connect(self._sync_code_to_canvas)
        self.code_view_editor_left.textChanged.connect(self._on_code_text_changed_left)
        self.save_button_left.clicked.connect(self._save_current_code_left)
        self.copy_button_left.clicked.connect(self._copy_current_code_left)
        self._user_edited_left = False
        self._updating_code_left = False
        self._ensure_canvas_signal_hooks_left()
        self._refresh_code_view_left()
        self._update_code_highlight_from_selection()
    
    def _connect_signals(self):
        """
        连接信号和槽函数
        将各个组件发出的事件信号连接到对应的处理函数
        """
        # ========== 设计画布信号连接 ==========
        # 控件被选中时发出信号，连接到选中处理函数
        self.design_canvas.control_selected.connect(self._on_control_selected)
        # 控件被添加时发出信号，连接到添加处理函数
        self.design_canvas.control_added.connect(self._on_control_added)
        # 控件被删除时发出信号，连接到删除处理函数
        self.design_canvas.control_deleted.connect(self._on_control_deleted)
        # 控件被移动时发出信号，连接到移动处理函数
        self.design_canvas.control_moved.connect(self._on_control_moved)
        # 控件被调整大小时发出信号，连接到调整大小处理函数
        self.design_canvas.control_resized.connect(self._on_control_resized)
        # 画布大小改变时发出信号，连接到画布大小改变处理函数
        self.design_canvas.canvas_resized.connect(self._on_canvas_resized)
        
        # ========== 属性编辑器信号连接 ==========
        # 控件属性被修改时发出信号，连接到属性改变处理函数
        self.property_editor.property_changed.connect(self._on_property_changed)

    def _refresh_code_view_left(self):
        """
        刷新左侧代码视图
        根据当前画布状态重新生成代码并更新代码编辑器
        如果用户正在编辑代码且不是手动点击刷新，则不自动更新
        """
        # 获取信号发送者（触发刷新的控件）
        sender = None
        try:
            sender = self.sender()  # 获取发出信号的控件对象
        except Exception:
            sender = None
        
        # ========== 检查同步生成选项 ==========
        # 检查是否启用了同步生成功能（复选框是否被选中）
        # hasattr检查对象是否有sync_checkbox_left属性（确保属性存在）
        # isChecked()检查复选框是否被选中
        if hasattr(self, 'sync_checkbox_left') and not self.sync_checkbox_left.isChecked():
            # 如果同步生成复选框未选中，不刷新代码（用户可能正在手动编辑代码）
            return
        
        # ========== 检查用户编辑状态 ==========
        # 检查用户是否正在编辑代码，并且触发刷新的不是刷新按钮本身
        # _user_edited_left：标记用户是否手动编辑过代码
        # sender：触发刷新信号的控件（如果为None或不是刷新按钮，说明是自动刷新）
        if self._user_edited_left and sender is not getattr(self, 'refresh_button_left', None):
            # 如果用户正在编辑代码且触发刷新的不是刷新按钮，不自动刷新
            # 这样可以避免在用户编辑代码时自动覆盖用户的内容
            return
        
        # 获取画布的宽度和高度
        canvas_width = self.design_canvas.canvas_width
        canvas_height = self.design_canvas.canvas_height
        
        try:
            # 设置代码生成器的窗口标题
            self.code_generator.set_window_title(self.design_canvas.window_title)
            # 设置代码生成器的窗口尺寸
            self.code_generator.set_window_size(canvas_width, canvas_height)
            # 设置代码生成器的窗口类名
            self.code_generator.set_window_class_name("GeneratedWindow")
            
            # ========== 设置窗口图标 ==========
            # 获取窗口图标路径（如果存在）
            # 使用getattr从设计画布获取window_icon属性，如果不存在则返回None
            icon_path = getattr(self.design_canvas, 'window_icon', None)
            # 检查图标路径是否为有效的字符串（不为None且不为空）
            if isinstance(icon_path, str) and icon_path:
                # 如果图标路径有效，设置代码生成器的窗口图标
                self.code_generator.set_window_icon(icon_path)
            
            # ========== 设置窗口菜单（页面）列表 ==========
            # 获取窗口菜单（页面）列表
            # 使用getattr从设计画布获取window_menus属性，如果不存在则返回空列表
            pages = getattr(self.design_canvas, 'window_menus', [])
            # 检查菜单页面列表是否存在且不为空
            if pages:
                # 如果存在菜单页面，设置到代码生成器（用于生成多页应用的代码）
                self.code_generator.set_pages(pages)
            
            # ========== 设置窗口下拉菜单列表 ==========
            # 获取窗口下拉菜单列表
            # 使用getattr从设计画布获取window_dropdown_menus属性，如果不存在则返回空列表
            menus = getattr(self.design_canvas, 'window_dropdown_menus', [])
            # 检查下拉菜单列表是否存在且不为空
            if menus:
                # 如果存在下拉菜单，设置到代码生成器（用于生成菜单栏代码）
                self.code_generator.set_dropdown_menus(menus)
            
            # 获取当前选择的框架（PySide或TKinter），默认为PySide
            framework = self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide"
            # 根据画布状态和选择的框架生成代码
            code = self.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            # 获取当前代码编辑器中的代码内容
            current = self.code_view_editor_left.toPlainText()
            
            # 如果生成的代码与当前代码不同，则更新代码编辑器
            if code != current:
                # 保存当前垂直滚动条的位置，以便更新后恢复
                vsb = self.code_view_editor_left.verticalScrollBar()
                pos = vsb.value() if vsb else 0
                # 设置更新标志，防止触发文本改变事件
                self._updating_code_left = True
                # 更新代码编辑器内容
                self.code_view_editor_left.setPlainText(code)
                # 清除更新标志
                self._updating_code_left = False
                # 尝试恢复滚动条位置
                try:
                    if vsb:
                        vsb.setValue(pos)
                except Exception:
                    pass
                # 清除用户编辑标志
                self._user_edited_left = False
            
            # 更新代码高亮：根据当前选中的控件高亮相关代码行
            self._update_code_highlight_from_selection()
        except Exception:
            pass

    def _sync_code_to_canvas(self):
        """
        将代码同步到画布
        解析代码编辑器中的Python代码，并将其转换为画布上的控件
        注意：此操作会清空当前画布上的所有内容
        """
        # 获取代码编辑器中的代码文本
        code = self.code_view_editor_left.toPlainText()
        # 如果代码为空，直接返回
        if not code:
            return
            
        # 使用代码解析器解析代码，提取窗口属性和控件信息
        result = self.code_parser.parse_code(code)
        # 如果解析失败，显示错误提示并返回
        if not result:
            QMessageBox.warning(self, "错误", "无法解析代码，请检查语法")
            return
            
        # 确认操作：显示确认对话框，因为同步会覆盖当前画布内容
        reply = QMessageBox.question(
            self, "确认同步", 
            "同步操作将覆盖当前画布上的所有内容，是否继续？",
            QMessageBox.Yes | QMessageBox.No
        )
        
        # 如果用户选择不继续，则返回
        if reply != QMessageBox.Yes:
            return
            
        try:
            # ========== 1. 更新窗口属性 ==========
            # 从解析结果中获取窗口属性字典
            window_props = result.get("window", {})
            # 如果解析结果中包含窗口宽度和高度，则更新画布大小
            if "width" in window_props and "height" in window_props:
                w, h = window_props["width"], window_props["height"]
                # 更新画布的宽度和高度
                self.design_canvas.set_canvas_size(w, h)
                # 同时更新项目管理器中的画布大小信息
                self.project_manager.set_canvas_size(w, h)
            
            # 如果解析结果中包含窗口标题，则更新窗口标题
            if "title" in window_props:
                self.design_canvas.set_window_property("window_title", window_props["title"])
                
            # ========== 2. 清空现有控件 ==========
            # 清空控件管理器中所有的控件，为加载新控件做准备
            self.control_manager.clear_all()
            
            # ========== 3. 创建新控件 ==========
            # 从解析结果中获取控件数据列表
            controls_data = result.get("controls", [])
            
            # 创建ID映射字典：将代码中的变量名映射到实际的控件ID
            id_map = {} 
            
            # 遍历所有控件数据，逐个创建控件
            for data in controls_data:
                # 获取控件ID（代码中的变量名）
                control_id = data["id"]
                # 获取控件类型（如Button、Label等）
                control_type = data["type"]
                # 获取父控件的变量名
                parent_var = data["parent"]
                
                # 确定父控件ID：根据父控件的变量名找到对应的实际控件ID
                parent_id = None
                if parent_var:
                    # 如果父控件变量名已经在ID映射中，直接使用映射的ID
                    if parent_var in id_map:
                        parent_id = id_map[parent_var]
                    # 如果父控件是页面（以page_开头），跳过父控件设置
                    elif parent_var.startswith("page_"):
                        pass
                    # 如果父控件是central_widget或tabs等特殊容器，也跳过
                    elif parent_var not in ["central_widget", "tabs"]:
                        parent_id = parent_var
                
                # 创建控件：调用控件管理器添加控件
                control = self.control_manager.add_control(
                    control_type=control_type,  # 控件类型
                    x=0, y=0,  # 初始位置（后续会从属性中更新）
                    parent_id=parent_id,  # 父控件ID
                    control_id=control_id  # 控件ID
                )
                
                # 如果控件创建成功
                if control:
                    # 将代码中的变量名映射到实际的控件ID
                    id_map[control_id] = control.id
                    
                    # 获取控件的属性字典
                    props = data.get("properties", {})
                    
                    # ========== 应用位置和大小属性 ==========
                    # 设置控件的X坐标
                    if "x" in props: control.x = props["x"]
                    # 设置控件的Y坐标
                    if "y" in props: control.y = props["y"]
                    # 设置控件的宽度
                    if "width" in props: control.width = props["width"]
                    # 设置控件的高度
                    if "height" in props: control.height = props["height"]
                    
                    # ========== 应用其他属性 ==========
                    # 遍历属性字典，设置控件的其他属性
                    for k, v in props.items():
                        # 跳过位置和大小属性（已单独处理）
                        if k not in ["x", "y", "width", "height"]:
                            control.properties[k] = v
                            
            # ========== 4. 更新画布和界面 ==========
            # 刷新画布显示，使新创建的控件可见
            self.design_canvas.update()
            # 刷新属性编辑器中的控件列表，使新控件出现在列表中
            self.property_editor.refresh_control_list(self.control_manager)
            # 更新状态栏显示同步成功信息
            self.status_label.setText("已从代码同步到画布")
            
            # 重置用户编辑标志：代码已从画布同步，不再是用户手动编辑的代码
            self._user_edited_left = False
            
        except Exception as e:
            # 如果同步过程中发生异常，显示错误提示
            QMessageBox.warning(self, "错误", f"同步失败: {str(e)}")
            # 导入traceback模块用于打印详细错误信息
            import traceback
            # 打印完整的错误堆栈跟踪，便于调试
            traceback.print_exc()

    def _ensure_canvas_signal_hooks_left(self):
        """
        确保画布信号连接到左侧代码视图更新
        将画布的各种变化信号连接到左侧代码视图的刷新函数，实现画布变化时自动更新代码
        """
        try:
            # ========== 连接画布变化信号 ==========
            # 连接控件添加信号：当有控件被添加到画布时，触发左侧代码视图刷新
            self.design_canvas.control_added.connect(self._on_canvas_changed_left)
            # 连接控件删除信号：当有控件被从画布删除时，触发左侧代码视图刷新
            self.design_canvas.control_deleted.connect(self._on_canvas_changed_left)
            # 连接控件移动信号：当控件被移动时，触发左侧代码视图刷新
            # 使用lambda忽略信号参数，直接调用刷新函数
            self.design_canvas.control_moved.connect(lambda *_: self._on_canvas_changed_left())
            # 连接控件大小调整信号：当控件大小被调整时，触发左侧代码视图刷新
            # 使用lambda忽略信号参数，直接调用刷新函数
            self.design_canvas.control_resized.connect(lambda *_: self._on_canvas_changed_left())
            # 连接画布大小调整信号：当画布大小被调整时，触发左侧代码视图刷新
            # 使用lambda忽略信号参数，直接调用刷新函数
            self.design_canvas.canvas_resized.connect(lambda *_: self._on_canvas_changed_left())
        except Exception:
            # 如果连接信号时发生异常，忽略异常（不中断程序运行）
            pass

    def _on_canvas_changed_left(self, *args, **kwargs):
        """
        画布变化时的处理函数（左侧代码视图）
        当画布发生任何变化时（控件添加、删除、移动、调整大小等），刷新左侧代码视图
        
        Args:
            *args: 可变位置参数（由信号传递，在此函数中不使用）
            **kwargs: 可变关键字参数（由信号传递，在此函数中不使用）
        """
        # ========== 刷新左侧代码视图 ==========
        # 调用刷新函数，重新生成代码并更新左侧代码编辑器
        self._refresh_code_view_left()

    def _on_code_text_changed_left(self):
        """
        左侧代码编辑器文本改变事件处理函数
        当用户在代码编辑器中修改代码时调用，用于标记代码已被用户编辑
        """
        # ========== 检查是否正在更新代码 ==========
        # 检查是否正在程序更新代码（避免程序更新时触发用户编辑标记）
        if self._updating_code_left:
            # 如果正在更新代码，直接返回（不标记为用户编辑）
            return
        
        # ========== 标记为用户编辑 ==========
        # 设置用户编辑标志为True，表示用户手动修改了代码
        # 这个标志用于判断是否在画布变化时自动刷新代码（如果用户正在编辑，不自动刷新）
        self._user_edited_left = True

    def _save_current_code_left(self):
        """
        保存左侧代码编辑器中的代码到文件
        显示文件保存对话框，让用户选择保存路径
        """
        try:
            # ========== 显示文件保存对话框 ==========
            # 显示文件保存对话框，获取用户选择的保存路径
            # 参数：父窗口、对话框标题、默认文件名、文件类型过滤器
            file_path, _ = QFileDialog.getSaveFileName(
                self,  # 父窗口（主窗口）
                "保存代码",  # 对话框标题
                "generated.py",  # 默认文件名
                "Python 文件 (*.py);;所有文件 (*)"  # 文件类型过滤器
            )
            
            # ========== 保存文件 ==========
            # 检查用户是否选择了文件路径
            if file_path:
                # 如果选择了路径，打开文件并写入代码
                # 以写入模式打开文件，使用UTF-8编码（支持中文字符）
                with open(file_path, 'w', encoding='utf-8') as f:
                    # 将代码编辑器中的纯文本内容写入文件
                    f.write(self.code_view_editor_left.toPlainText())
        except Exception:
            # ========== 处理保存异常 ==========
            # 如果保存过程中发生任何异常（如权限错误、磁盘空间不足等），忽略异常
            # 不显示错误提示，避免干扰用户体验
            pass

    def _copy_current_code_left(self):
        """
        复制左侧代码编辑器中的代码到剪贴板
        将代码编辑器的内容复制到系统剪贴板，用户可以在其他地方粘贴使用
        """
        try:
            # ========== 复制代码到剪贴板 ==========
            # 获取系统剪贴板对象
            # 将代码编辑器中的纯文本内容设置到剪贴板
            QApplication.clipboard().setText(self.code_view_editor_left.toPlainText())
        except Exception:
            # ========== 处理复制异常 ==========
            # 如果复制过程中发生任何异常（如剪贴板访问失败等），忽略异常
            # 不显示错误提示，避免干扰用户体验
            pass
    
    def _on_control_selected(self, control_id: str):
        """控件被选中时的处理"""
        if control_id == "canvas":
            self.property_editor.set_control(None, is_canvas=True)
            self.status_label.setText("已选中画布")
        elif control_id == "multiple_selected":
            # 处理多选情况
            selected_ids = self.control_manager.get_selected_ids()
            if selected_ids:
                # 显示选中的控件数量
                self.status_label.setText(f"已选中 {len(selected_ids)} 个控件")
                # 对于多选，我们可以设置第一个控件到property_editor，或者不设置
                # 保持property_editor的状态不变
            
            # 无论如何都要更新代码高亮
            self._update_code_highlight_from_selection()
            return
        else:
            control = self.control_manager.get_control(control_id)
            if control:
                self.property_editor.set_control(control)
                self.status_label.setText(f"已选中控件: {control.id}")
            else:
                self.property_editor.set_control(None)
                self.status_label.setText("未选中任何控件")
        self._update_code_highlight_from_selection()

    def _update_code_highlight_from_selection(self):
        """
        根据选中的控件更新代码编辑器中的高亮显示
        在代码编辑器中高亮显示与选中控件相关的代码行（控件创建、属性设置、信号连接等）
        """
        # ========== 获取选中的控件ID列表 ==========
        # 初始化选中控件ID列表
        ids = []
        try:
            # 尝试使用get_selected_ids方法获取选中控件ID列表
            if hasattr(self.control_manager, 'get_selected_ids'):
                # 如果控件管理器有get_selected_ids方法，调用该方法获取选中ID列表
                ids = list(self.control_manager.get_selected_ids()) or []
            else:
                # 如果控件管理器没有get_selected_ids方法，尝试从属性中获取
                ids = list(getattr(self.control_manager, 'selected_control_ids', []))
        except Exception:
            # 如果获取过程中发生异常，使用备用方法从属性中获取
            ids = list(getattr(self.control_manager, 'selected_control_ids', []))
        
        # ========== 获取代码文本和行列表 ==========
        # 获取代码编辑器中的纯文本内容
        text = self.code_view_editor_left.toPlainText()
        # 将代码文本按行分割，转换为行列表
        lines = text.splitlines()
        # 初始化高亮选区列表，用于存储需要高亮的代码区域
        selections = []
        
        # ========== 遍历所有选中的控件 ==========
        # 遍历每个选中控件的ID，为其查找相关的代码行并添加高亮
        for cid in ids:
            # ========== 转换控件ID为变量名 ==========
            try:
                # 尝试将控件ID转换为有效的Python变量名（如去除特殊字符等）
                var = self.code_generator._to_valid_variable_name(cid)
            except Exception:
                # 如果转换失败，直接使用控件ID作为变量名
                var = cid
            
            # ========== 准备匹配字符串 ==========
            # 将变量名转换为小写，用于不区分大小写的匹配
            var_lower = var.lower()
            # 将控件ID转换为小写字符串，用于匹配
            cid_lower = str(cid).lower()
            # 构建目标匹配字符串：self.变量名（如 self.button1）
            target_lower = f"self.{var_lower}"
            
            # ========== 查找相关的代码行索引 ==========
            # 初始化索引列表，存储需要高亮的行号
            indexes = []
            # 遍历代码的每一行，查找与当前控件相关的代码
            for i, line in enumerate(lines):
                # 将当前行转换为小写，用于不区分大小写的匹配
                ll = line.lower()
                
                # ========== 匹配控件创建行 ==========
                # 检查是否为控件创建行（包含 self.变量名 和 = 号）
                # 例如：self.button1 = QPushButton()
                if target_lower in ll and "=" in ll:
                    # 如果是控件创建行，添加行号到索引列表
                    indexes.append(i)
                    # 继续检查下一行
                    continue
                
                # ========== 匹配信号连接行 ==========
                # 检查是否为信号连接行（包含 self.变量名 和 .connect(）
                # 例如：self.button1.clicked.connect(self._on_button1_clicked)
                if target_lower in ll and ".connect(" in ll:
                    # 如果是信号连接行，添加行号到索引列表
                    indexes.append(i)
                    # 继续检查下一行
                    continue
                
                # ========== 匹配方法调用行 ==========
                # 检查是否为控件方法调用行（包含 self.变量名、点号和方法调用）
                # 例如：self.button1.setText("Hello")
                if target_lower in ll and "." in ll and "(" in ll:
                    # 如果是方法调用行，添加行号到索引列表
                    indexes.append(i)
                    # 继续检查下一行
                    continue
                
                # ========== 匹配事件处理函数定义行 ==========
                # 检查是否为控件的事件处理函数定义行（格式：def _on_变量名_...）
                # 例如：def _on_button1_clicked(self):
                if f"def _on_{var_lower}_" in ll:
                    # 如果是事件处理函数定义，添加行号和类型标记到索引列表
                    # 使用元组 (行号, 'method') 标记这是一个方法定义
                    indexes.append((i, 'method'))
                    # 继续检查下一行
                    continue
                
                # ========== 匹配控件ID字符串行 ==========
                # 检查代码行中是否直接包含控件ID字符串
                # 例如：注释中的控件ID或其他位置的ID引用
                if cid_lower and cid_lower in ll:
                    # 如果包含控件ID，添加行号和类型标记到索引列表
                    # 使用元组 (行号, 'id') 标记这是一个ID引用
                    indexes.append((i, 'id'))
                    # 继续检查下一行
                    continue
            # ========== 为找到的代码行创建高亮选区 ==========
            # 遍历所有找到的索引，为每个索引创建高亮选区
            for idx in indexes:
                # ========== 处理事件处理函数定义的高亮 ==========
                # 检查当前索引是否为方法定义类型（元组格式：(行号, 'method')）
                if isinstance(idx, tuple) and idx[1] == 'method':
                    # 获取方法定义的起始行号
                    start_i = idx[0]
                    # 初始化方法定义的结束行号（默认与起始行相同）
                    end_i = start_i
                    # 从方法定义的下一行开始查找方法体的结束位置
                    j = start_i + 1
                    # ========== 查找方法体的结束位置 ==========
                    # 循环查找方法体的结束行（直到遇到空行或下一个方法定义）
                    while j < len(lines):
                        # 获取当前行的文本
                        s = lines[j]
                        # 如果当前行为空行（去除空白后为空），方法体结束
                        if not s.strip():
                            # 跳出循环，方法体在此行之前结束
                            break
                        # 如果当前行是下一个方法定义（以def开头），方法体结束
                        if s.lstrip().startswith('def '):
                            # 跳出循环，方法体在此行之前结束
                            break
                        # 更新方法体的结束行号
                        end_i = j
                        # 移动到下一行继续查找
                        j += 1
                    
                    # ========== 创建方法定义的高亮选区 ==========
                    # 从文档中获取方法定义的起始文本块（行）
                    start_block = self.code_view_editor_left.document().findBlockByNumber(start_i)
                    # 从文档中获取方法定义的结束文本块（行）
                    end_block = self.code_view_editor_left.document().findBlockByNumber(end_i)
                    # 创建文本光标对象，用于选择文本
                    cursor = self.code_view_editor_left.textCursor()
                    # 将光标定位到起始文本块的位置（方法定义行的开始）
                    cursor.setPosition(start_block.position())
                    # 移动到当前行的末尾，选中整行
                    cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                    # 从方法定义的下一行开始，逐行选中到方法体结束
                    k = start_i + 1
                    while k <= end_i:
                        # 移动到下一个文本块（下一行）
                        cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
                        # 移动到当前行的末尾，选中整行
                        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                        # 移动到下一行
                        k += 1
                    
                    # ========== 创建高亮样式并添加到选区列表 ==========
                    # 创建额外的文本选区对象（用于高亮显示）
                    sel = QTextEdit.ExtraSelection()
                    # 设置高亮背景颜色（浅黄色 RGB(255, 245, 200)）
                    sel.format.setBackground(QColor(255, 245, 200))
                    # 将光标选区设置到额外选区对象
                    sel.cursor = cursor
                    # 将高亮选区添加到选区列表
                    selections.append(sel)
                # ========== 处理控件ID引用的高亮 ==========
                # 检查当前索引是否为ID引用类型（元组格式：(行号, 'id')）
                elif isinstance(idx, tuple) and idx[1] == 'id':
                    # 获取包含控件ID的行的行号（锚点行）
                    anchor = idx[0]
                    # 初始化方法定义行号（默认为锚点行）
                    def_line = anchor
                    # 从锚点行向上查找，寻找包含该ID的方法定义
                    k = anchor
                    while k >= 0:
                        # 检查当前行是否为方法定义行（以def开头）
                        if lines[k].lstrip().lower().startswith('def '):
                            # 找到方法定义行，保存行号
                            def_line = k
                            # 跳出循环，不再向上查找
                            break
                        # 继续向上查找（行号递减）
                        k -= 1
                    
                    # ========== 如果找到了包含该ID的方法，高亮整个方法 ==========
                    # 检查是否找到了包含控件ID的方法定义（方法定义行与锚点行不同）
                    if def_line != anchor:
                        # 将方法定义行作为高亮起始行
                        start_i = def_line
                        # 初始化方法定义的高亮结束行（默认与起始行相同）
                        end_i = start_i
                        # 从方法定义的下一行开始查找方法体的结束位置
                        j = start_i + 1
                        # ========== 查找方法体的结束位置 ==========
                        # 循环查找方法体的结束行（直到遇到空行或下一个方法定义）
                        while j < len(lines):
                            # 获取当前行的文本
                            s = lines[j]
                            # 如果当前行为空行（去除空白后为空），方法体结束
                            if not s.strip():
                                # 跳出循环，方法体在此行之前结束
                                break
                            # 如果当前行是下一个方法定义（以def开头），方法体结束
                            if s.lstrip().startswith('def '):
                                # 跳出循环，方法体在此行之前结束
                                break
                            # 更新方法体的结束行号
                            end_i = j
                            # 移动到下一行继续查找
                            j += 1
                        
                        # ========== 创建整个方法的高亮选区 ==========
                        # 从文档中获取方法定义的起始文本块（行）
                        start_block = self.code_view_editor_left.document().findBlockByNumber(start_i)
                        # 从文档中获取方法定义的结束文本块（行）
                        end_block = self.code_view_editor_left.document().findBlockByNumber(end_i)
                        # 创建文本光标对象，用于选择文本
                        cursor = self.code_view_editor_left.textCursor()
                        # 将光标定位到起始文本块的位置（方法定义行的开始）
                        cursor.setPosition(start_block.position())
                        # 移动到当前行的末尾，选中整行
                        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                        # 从方法定义的下一行开始，逐行选中到方法体结束
                        m = start_i + 1
                        while m <= end_i:
                            # 移动到下一个文本块（下一行）
                            cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
                            # 移动到当前行的末尾，选中整行
                            cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                            # 移动到下一行
                            m += 1
                        
                        # ========== 创建高亮样式并添加到选区列表 ==========
                        # 创建额外的文本选区对象（用于高亮显示）
                        sel = QTextEdit.ExtraSelection()
                        # 设置高亮背景颜色（浅黄色 RGB(255, 245, 200)）
                        sel.format.setBackground(QColor(255, 245, 200))
                        # 将光标选区设置到额外选区对象
                        sel.cursor = cursor
                        # 将高亮选区添加到选区列表
                        selections.append(sel)
                    else:
                        # ========== 如果没有找到包含ID的方法，只高亮单行 ==========
                        # 如果未找到方法定义（ID不在方法中，可能是独立语句），只高亮包含ID的单行
                        # 从文档中获取锚点行的文本块
                        block = self.code_view_editor_left.document().findBlockByNumber(anchor)
                        # 创建文本光标对象
                        cursor = self.code_view_editor_left.textCursor()
                        # 将光标定位到文本块的位置（行的开始）
                        cursor.setPosition(block.position())
                        # 移动到当前行的末尾，选中整行
                        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                        # ========== 创建高亮样式并添加到选区列表 ==========
                        # 创建额外的文本选区对象（用于高亮显示）
                        sel = QTextEdit.ExtraSelection()
                        # 设置高亮背景颜色（浅黄色 RGB(255, 245, 200)）
                        sel.format.setBackground(QColor(255, 245, 200))
                        # 将光标选区设置到额外选区对象
                        sel.cursor = cursor
                        # 将高亮选区添加到选区列表
                        selections.append(sel)
                else:
                    # ========== 处理普通代码行的高亮 ==========
                    # 如果索引不是元组（普通整数行号），只高亮单行
                    # 从索引中提取行号（如果是元组则取第一个元素，否则直接使用）
                    idx_i = idx if isinstance(idx, int) else idx[0]
                    # 从文档中获取对应行号的文本块
                    block = self.code_view_editor_left.document().findBlockByNumber(idx_i)
                    # 创建文本光标对象
                    cursor = self.code_view_editor_left.textCursor()
                    # 将光标定位到文本块的位置（行的开始）
                    cursor.setPosition(block.position())
                    # 移动到当前行的末尾，选中整行
                    cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                    # ========== 创建高亮样式并添加到选区列表 ==========
                    # 创建额外的文本选区对象（用于高亮显示）
                    sel = QTextEdit.ExtraSelection()
                    # 设置高亮背景颜色（浅黄色 RGB(255, 245, 200)）
                    sel.format.setBackground(QColor(255, 245, 200))
                    # 将光标选区设置到额外选区对象
                    sel.cursor = cursor
                    # 将高亮选区添加到选区列表
                    selections.append(sel)
        
        # ========== 应用高亮选区到代码编辑器 ==========
        # 尝试将生成的所有高亮选区应用到代码编辑器
        try:
            # 检查代码编辑器是否有set_external_highlights方法
            if hasattr(self.code_view_editor_left, 'set_external_highlights'):
                # 如果方法存在，调用它应用所有高亮选区
                self.code_view_editor_left.set_external_highlights(selections)
        except Exception:
            # 如果应用高亮时发生异常，忽略异常（不中断程序运行）
            pass
    
    def _on_control_added(self, control_id: str):
        """控件被添加时的处理"""
        control = self.control_manager.get_control(control_id)
        if control:
            self.project_manager.set_modified(True)
            self.status_label.setText(f"已添加控件: {control.id}")
            # 保存状态到撤销栈
            self.undo_redo_manager.save_state(f"添加 {control.control_type} 控件")
            self.property_editor.refresh_control_list(self.control_manager)
    
    def _on_control_deleted(self, control_id: str):
        """控件被删除时的处理"""
        self.project_manager.set_modified(True)
        self.status_label.setText(f"已删除控件: {control_id}")
        # 保存状态到撤销栈
        self.undo_redo_manager.save_state("删除控件")
        self.property_editor.refresh_control_list(self.control_manager)
    
    def _on_control_moved(self, control_id: str, dx: int, dy: int):
        """控件被移动时的处理"""
        control = self.control_manager.get_control(control_id)
        if control:
            self.coord_label.setText(f"X: {control.x}, Y: {control.y}")
            self.project_manager.set_modified(True)
            # 保存状态到撤销栈
            self.undo_redo_manager.save_state(f"移动控件 {control_id}")
    
    def _on_control_resized(self, control_id: str, handle: str, dx: int, dy: int):
        """控件被调整大小时的处理"""
        control = self.control_manager.get_control(control_id)
        if control:
            self.canvas_size_label.setText(f"控件: {control.width}x{control.height}")
            self.project_manager.set_modified(True)
            # 保存状态到撤销栈
            self.undo_redo_manager.save_state(f"调整控件 {control_id} 大小")
    
    def _on_property_changed(self, control_id: str, property_name: str, value):
        """属性变化时的处理"""
        if control_id == "canvas":
            # 画布属性变化处理映射
            canvas_property_handlers = {
                # 画布属性
                "canvas_width": lambda: (self.design_canvas.set_canvas_width(value), 
                                       self.project_manager.set_canvas_size(value, self.design_canvas.get_canvas_height())),
                "canvas_height": lambda: (self.design_canvas.set_canvas_height(value), 
                                        self.project_manager.set_canvas_size(self.design_canvas.get_canvas_width(), value)),
                "canvas_bg_color": lambda: self.design_canvas.set_canvas_bg_color(value),
                "show_grid": lambda: self.design_canvas.set_show_grid(value),
                "grid_size": lambda: self.design_canvas.set_grid_size(value),
                "snap_to_grid": lambda: self.design_canvas.set_snap_to_grid(value),
                
                # 使用通用窗口属性设置方法
                "window_name": lambda: self.design_canvas.set_window_property("window_name", value),
                "window_comment": lambda: self.design_canvas.set_window_property("window_comment", value),
                "window_left": lambda: self.design_canvas.set_window_property("window_left", value),
                "window_top": lambda: self.design_canvas.set_window_property("window_top", value),
                "window_width": lambda: self.design_canvas.set_window_property("window_width", value),
                "window_height": lambda: self.design_canvas.set_window_property("window_height", value),
                "window_visible": lambda: self.design_canvas.set_window_property("window_visible", value),
                "window_enabled": lambda: self.design_canvas.set_window_property("window_enabled", value),
                "window_cursor": lambda: self.design_canvas.set_window_property("window_cursor", value),
                "window_title": lambda: self.design_canvas.set_window_property("window_title", value),
                "window_border": lambda: self.design_canvas.set_window_property("window_border", value),
                "window_bg_color": lambda: self.design_canvas.set_window_property("window_bg_color", value),
                "window_bg_image": lambda: self.design_canvas.set_window_property("window_bg_image", value),
                "window_bg_image_mode": lambda: self.design_canvas.set_window_property("window_bg_image_mode", value),
                "window_bg_music": lambda: self.design_canvas.set_window_property("window_bg_music", value),
                "window_music_play_count": lambda: self.design_canvas.set_window_property("window_music_play_count", value),
                "window_show_control_button": lambda: self.design_canvas.set_window_property("window_show_control_button", value),
                "window_show_maximize_button": lambda: self.design_canvas.set_window_property("window_show_maximize_button", value),
                "window_show_minimize_button": lambda: self.design_canvas.set_window_property("window_show_minimize_button", value),
                "window_position": lambda: self.design_canvas.set_window_property("window_position", value),
                "window_movable": lambda: self.design_canvas.set_window_property("window_movable", value),
                "window_icon": lambda: self.design_canvas.set_window_property("window_icon", value),
                "window_enter_move_focus": lambda: self.design_canvas.set_window_property("window_enter_move_focus", value),
                "window_esc_close": lambda: self.design_canvas.set_window_property("window_esc_close", value),
                "window_f1_help": lambda: self.design_canvas.set_window_property("window_f1_help", value),
                "window_help_file": lambda: self.design_canvas.set_window_property("window_help_file", value),
                "window_help_flag": lambda: self.design_canvas.set_window_property("window_help_flag", value),
                "window_show_in_taskbar": lambda: self.design_canvas.set_window_property("window_show_in_taskbar", value),
                "window_free_move": lambda: self.design_canvas.set_window_property("window_free_move", value),
                "window_shape": lambda: self.design_canvas.set_window_property("window_shape", value),
                "window_always_on_top": lambda: self.design_canvas.set_window_property("window_always_on_top", value),
                "window_keep_title_active": lambda: self.design_canvas.set_window_property("window_keep_title_active", value),
                "window_class_name": lambda: self.design_canvas.set_window_property("window_class_name", value),
            }
            
            # 执行对应的属性处理函数
            if property_name in canvas_property_handlers:
                canvas_property_handlers[property_name]()
            
            # 更新画布
            self.design_canvas.update()
            
            # 更新状态栏
            self.status_label.setText(f"已更新画布属性: {property_name}")
            
            # 标记项目已修改
            self.project_manager.set_modified(True)
            # 保存状态到撤销栈
            self.undo_redo_manager.save_state(f"修改画布属性")
        else:
            # 控件属性变化
            control = self.control_manager.get_control(control_id)
            if control:
                # 更新控件属性
                if property_name in ["x", "y", "width", "height"]:
                    if property_name == "x":
                        control.x = value
                    elif property_name == "y":
                        control.y = value
                    elif property_name == "width":
                        control.width = value
                    elif property_name == "height":
                        control.height = value
                elif property_name == "z_order":
                    control.z_order = value
                else:
                    control.properties[property_name] = value
                    if property_name == "name":
                        control.name = value
                        self.property_editor.refresh_control_list(self.control_manager)
                
                # 更新画布
                self.design_canvas.update()
                
                # 更新状态栏
                self.status_label.setText(f"已更新属性: {property_name}")
                
                # 标记项目已修改
                self.project_manager.set_modified(True)
                # 保存状态到撤销栈
                self.undo_redo_manager.save_state(f"修改控件 {control_id} 属性")
    
    def _new_project(self):
        """新建项目"""
        # 检查是否有未保存的修改
        if self.project_manager.is_modified():
            reply = QMessageBox.question(
                self, "确认", "当前项目有未保存的修改，是否保存？",
                QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel
            )
            
            if reply == QMessageBox.Yes:
                self._save_project()
            elif reply == QMessageBox.Cancel:
                return
        
        # 清空画布
        self.design_canvas.clear_controls()
        
        # 重置项目管理器
        self.project_manager.new_project()
        
        # 更新窗口标题
        self.setWindowTitle("可视化编程工具 - 无标题")
        
        # 更新状态栏
        self.statusBar().showMessage("已创建新项目")
        
        # 清空撤销重做栈
        self.undo_redo_manager.clear()
        try:
            self.undo_redo_manager.save_state("初始状态")
        except Exception:
            pass
    
    def _open_project(self):
        """打开项目"""
        # 检查当前项目是否已修改
        if self.project_manager.is_modified():
            reply = QMessageBox.question(
                self, "保存项目", 
                "当前项目已修改，是否保存？",
                QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel
            )
            
            if reply == QMessageBox.Yes:
                if not self._save_project_impl():
                    return
            elif reply == QMessageBox.Cancel:
                return
        
        # 选择要打开的项目文件
        file_path, _ = QFileDialog.getOpenFileName(
            self, "打开项目", "", "项目文件 (*.vpp);;所有文件 (*)"
        )
        
        if file_path:
            # 加载项目
            if self.project_manager.load_project(file_path):
                self.design_canvas.update()
                
                # 更新画布大小
                canvas_width, canvas_height = self.project_manager.get_canvas_size()
                self.design_canvas.set_canvas_size(canvas_width, canvas_height)
                
                self.property_editor.set_control(None)
                self.status_label.setText(f"已打开项目: {self.project_manager.get_project_name()}")
                
                # 清空撤销重做栈
                self.undo_redo_manager.clear()
                try:
                    self.undo_redo_manager.save_state("初始状态")
                except Exception:
                    pass
            else:
                QMessageBox.warning(self, "错误", "无法打开项目文件")
    
    def _save_project(self):
        """保存项目"""
        if self._save_project_impl():
            self.status_label.setText(f"项目已保存: {self.project_manager.get_project_name()}")
    
    def _save_as_project(self):
        """另存为项目"""
        file_path, _ = QFileDialog.getSaveFileName(
            self, "另存为项目", f"{self.project_manager.get_project_name()}.vpp", 
            "项目文件 (*.vpp);;所有文件 (*)"
        )
        
        if file_path:
            if self.project_manager.save_project(file_path):
                self.status_label.setText(f"项目已保存为: {self.project_manager.get_project_name()}")
            else:
                QMessageBox.warning(self, "错误", "无法保存项目文件")
    
    def _save_project_impl(self) -> bool:
        """实现项目保存的内部方法
        
        Returns:
            bool: 是否成功保存
        """
        # 如果没有项目文件路径，显示另存为对话框
        if self.project_manager.get_project_file() is None:
            file_path, _ = QFileDialog.getSaveFileName(
                self, "保存项目", f"{self.project_manager.get_project_name()}.vpp", 
                "项目文件 (*.vpp);;所有文件 (*)"
            )
            
            if not file_path:
                return False
                
            result = self.project_manager.save_project(file_path)
            if result:
                # 初始化预览窗口
                self.preview_window = None
            return result
        else:
            result = self.project_manager.save_project()
            if result:
                # 初始化预览窗口
                self.preview_window = None
            return result
    
    def _undo(self):
        """撤销操作"""
        if self.undo_redo_manager.undo():
            # 重新加载画布
            self._reload_canvas()
            # 更新状态栏
            description = self.undo_redo_manager.get_undo_description()
            if description:
                self.status_label.setText(f"上一步: {description}")
            else:
                self.status_label.setText("上一步")
            # 标记项目已修改
            self.project_manager.set_modified(True)
        else:
            self.status_label.setText("无法执行上一步")
    
    def _redo(self):
        """重做操作"""
        if self.undo_redo_manager.redo():
            # 重新加载画布
            self._reload_canvas()
            # 更新状态栏
            description = self.undo_redo_manager.get_redo_description()
            if description:
                self.status_label.setText(f"重做: {description}")
            else:
                self.status_label.setText("重做")
            # 标记项目已修改
            self.project_manager.set_modified(True)
        else:
            self.status_label.setText("无法重做")
    
    def _reload_canvas(self):
        self.design_canvas.update()
        self.property_editor.refresh_control_list(self.control_manager)

    # 取消底部滑条联动方法

    def _on_canvas_resized(self, w: int, h: int):
        try:
            self.design_canvas.setFixedSize(max(1, w), max(1, h))
        except Exception:
            pass
        self.project_manager.set_canvas_size(w, h)
        try:
            self.canvas_size_label.setText(f"画布: {w}x{h}")
        except Exception:
            pass
        self.undo_redo_manager.save_state("调整画布大小")
    
    def _delete_selected(self):
        """删除选中的控件"""
        self.design_canvas.delete_selected_control()
    
    def _toggle_grid(self):
        """切换网格显示"""
        self.design_canvas.toggle_grid()
        self.status_label.setText("网格显示已切换")
    
    def _toggle_snap_to_grid(self):
        """切换网格对齐"""
        self.design_canvas.toggle_snap_to_grid()
        self.status_label.setText("网格对齐已切换")
    
    def _generate_code(self):
        """生成代码"""
        # 获取画布尺寸
        canvas_width = self.design_canvas.canvas_width
        canvas_height = self.design_canvas.canvas_height
        
        # 设置代码生成器参数
        self.code_generator.set_window_title(self.design_canvas.window_title)
        self.code_generator.set_window_size(canvas_width, canvas_height)
        self.code_generator.set_window_class_name("GeneratedWindow")
        # 设置窗口图标（若为路径）
        try:
            icon_path = self.design_canvas.window_icon
            if isinstance(icon_path, str) and icon_path:
                self.code_generator.set_window_icon(icon_path)
        except Exception:
            pass
        # 设置页面（来自菜单编辑器）
        try:
            pages = getattr(self.design_canvas, 'window_menus', [])
            if pages:
                self.code_generator.set_pages(pages)
        except Exception:
            pass
        # 设置下拉菜单（来自菜单编辑器）
        try:
            menus = getattr(self.design_canvas, 'window_dropdown_menus', [])
            if menus:
                self.code_generator.set_dropdown_menus(menus)
        except Exception:
            pass
        
        # 生成代码
        try:
            initial_code = self.code_generator.generate_code(canvas_width, canvas_height, framework="PySide")
            def provider(framework: str):
                return self.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            dialog = CodeDialog(self, initial_code, code_provider=provider, initial_framework="PySide")
            dialog.exec()
            self.statusBar().showMessage("代码生成成功")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"代码生成失败: {str(e)}")
            self.statusBar().showMessage("代码生成失败")
    
    def _find_python_interpreter(self):
        """
        智能查找可用的Python解释器
        """
        import sys
        import os
        import subprocess
        script_dir = os.path.dirname(os.path.abspath(__file__))
        if getattr(sys, 'frozen', False):
            base_dir = os.path.dirname(sys.executable)
        else:
            base_dir = script_dir
        cwd_dir = os.getcwd()
        candidates = []
        if os.name == 'nt':
            venv_env = os.environ.get('VIRTUAL_ENV', None)
            if venv_env:
                candidates.append(os.path.join(venv_env, 'Scripts', 'python.exe'))
            candidates.extend([
                os.path.join(cwd_dir, 'Python', 'python.exe'),
                os.path.join(base_dir, 'Python', 'python.exe'),
                os.path.join(script_dir, 'venv', 'Scripts', 'python.exe'),
                os.path.join(script_dir, '.venv', 'Scripts', 'python.exe'),
            ])
        else:
            venv_env = os.environ.get('VIRTUAL_ENV', None)
            if venv_env:
                candidates.append(os.path.join(venv_env, 'bin', 'python'))
            candidates.extend([
                os.path.join(cwd_dir, 'Python', 'python'),
                os.path.join(base_dir, 'Python', 'python'),
                os.path.join(script_dir, 'venv', 'bin', 'python'),
                os.path.join(script_dir, '.venv', 'bin', 'python'),
            ])
        for path in candidates:
            if path and os.path.isfile(path):
                return path
        if os.path.isfile(sys.executable):
            return sys.executable
        
        # 检查环境变量PATH中的python
        for python_name in ['python.exe', 'python3.exe', 'python']:
            try:
                # 使用where命令在Windows上查找
                if os.name == 'nt':  # Windows系统
                    result = subprocess.run(
                        ['where', python_name], 
                        capture_output=True, 
                        text=True, 
                        check=False
                    )
                    if result.returncode == 0 and result.stdout:
                        paths = result.stdout.strip().split('\n')
                        for path in paths:
                            if os.path.isfile(path):
                                return path
                # 使用which命令在Unix/Linux系统上查找
                else:
                    result = subprocess.run(
                        ['which', python_name], 
                        capture_output=True, 
                        text=True, 
                        check=False
                    )
                    found = result.stdout.strip()
                    if result.returncode == 0 and os.path.isfile(found):
                        return found
            except Exception:
                pass
        
        # 检查常见的Python安装路径
        common_paths = []
        if os.name == 'nt':
            system_drive = os.environ.get('SYSTEMDRIVE', 'C:')
            common_paths.extend([
                os.path.join(system_drive, 'Python39', 'python.exe'),
                os.path.join(system_drive, 'Python310', 'python.exe'),
                os.path.join(system_drive, 'Python311', 'python.exe'),
                os.path.join(system_drive, 'Python312', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python39', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python310', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python311', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python312', 'python.exe'),
            ])
        else:
            common_paths.extend([
                '/usr/bin/python3',
                '/usr/local/bin/python3',
                '/bin/python3',
            ])
        
        # 检查常见路径
        for path in common_paths:
            if os.path.isfile(path):
                return path
        
        return None
    
    def _run_project(self):
        import tempfile
        import subprocess
        import sys
        import os
        canvas_width = self.design_canvas.canvas_width
        canvas_height = self.design_canvas.canvas_height
        self.code_generator.set_window_title(self.design_canvas.window_title)
        self.code_generator.set_window_size(canvas_width, canvas_height)
        
        # ========== 设置窗口类名和窗口名称，确保绑定 ==========
        # 使用窗口类名作为基础，确保事件函数文件名和窗口类名绑定，避免冲突
        window_class_name = "GeneratedWindow"
        self.code_generator.set_window_class_name(window_class_name)
        self.code_generator.set_window_name(window_class_name)  # 确保窗口名称和类名一致
        
        try:
            icon_path = self.design_canvas.window_icon
            if isinstance(icon_path, str) and icon_path:
                self.code_generator.set_window_icon(icon_path)
        except Exception:
            pass
        try:
            pages = getattr(self.design_canvas, 'window_menus', [])
            if pages:
                self.code_generator.set_pages(pages)
        except Exception:
            pass
        try:
            menus = getattr(self.design_canvas, 'window_dropdown_menus', [])
            if menus:
                self.code_generator.set_dropdown_menus(menus)
        except Exception:
            pass
        try:
            code_text = None
            if hasattr(self, 'code_view_editor_left'):
                t = self.code_view_editor_left.toPlainText().strip()
                if t:
                    code_text = t
            if not code_text:
                framework = self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide"
                code_text = self.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            
            # ========== 创建临时目录用于存放代码文件 ==========
            # 使用临时目录而不是单个临时文件，这样可以同时保存主代码文件和事件函数文件
            temp_dir = tempfile.mkdtemp()
            temp_file_path = os.path.join(temp_dir, f"{window_class_name}.py")
            
            # ========== 保存主代码文件 ==========
            with open(temp_file_path, 'w', encoding='utf-8') as f:
                f.write(code_text)
            
            # ========== 检查是否有绑定的事件，如果有则生成事件函数文件 ==========
            has_events = False
            for control in self.control_manager.controls.values():
                bound_events = getattr(control, 'events', {})
                if isinstance(bound_events, dict) and bound_events:
                    # 检查是否有绑定的事件（值为True）
                    if any(is_bound for is_bound in bound_events.values()):
                        has_events = True
                        break
            
            # ========== 如果有绑定的事件，生成事件函数文件 ==========
            if has_events and framework.lower().startswith("pyside"):
                from module.event_function_generator_pyside import EventFunctionGenerator
                event_gen = EventFunctionGenerator(self.control_manager)
                event_gen.set_window_name(window_class_name)  # 使用窗口类名作为事件函数文件名
                function_content = event_gen.generate_function_file_content()
                function_file_name = event_gen.get_function_file_name()
                function_file_path = os.path.join(temp_dir, function_file_name)
                
                # 保存事件函数文件
                with open(function_file_path, 'w', encoding='utf-8') as f:
                    f.write(function_content)
            
            python_exe = self._find_python_interpreter()
            if not python_exe:
                raise RuntimeError("未找到可用的Python解释器")
            print(f"使用Python解释器: {python_exe}")
            subprocess.Popen([python_exe, temp_file_path])
            self.statusBar().showMessage(f"运行代码展示页 (使用: {os.path.basename(python_exe)})...")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"运行项目失败: {str(e)}")
            self.statusBar().showMessage("项目运行失败")
    
    def _preview_project(self):
        """预览项目"""
        try:
            if self.preview_window is None or not self.preview_window.isVisible():
                self.preview_window = BuiltInPreviewWindow(self, self.control_manager, self.design_canvas)
            else:
                self.preview_window.set_data(self.control_manager, self.design_canvas)
            self.preview_window.show()
            self.statusBar().showMessage("项目预览窗口已打开")
        except Exception as e:
            QMessageBox.warning(self, "预览错误", f"预览项目时出错: {str(e)}")
            self.statusBar().showMessage("预览项目失败")
    
    def _show_about(self):
        """显示关于对话框"""
        QMessageBox.about(
            self, 
            "关于可视化编程工具", 
            about_text()
        )


def _validate_license() -> bool:
    """
    验证许可证
    通过访问远程服务器验证软件许可证是否有效
    
    Returns:
        bool: 许可证验证是否通过
    """
    return True  # 许可证验证通过


def main():
    """
    主函数
    程序的入口点，负责初始化应用程序、创建主窗口并启动事件循环
    """
    # 打印启动信息
    print("正在创建应用程序...")
    
    # ========== 加载Monaco Editor配置 ==========
    # 在创建应用程序之前加载Monaco Editor本地路径配置
    try:
        from module.monaco_config import load_monaco_config
        load_monaco_config()
    except Exception as e:
        print(f"加载Monaco Editor配置时出错: {e}")
        import traceback
        traceback.print_exc()
    
    # ========== 加载控件配置文件 ==========
    # 在创建应用程序之前加载控件配置，实现动态加载
    # 支持一个组件一个配置文件的模式，方便动态加载和修改
    try:
        import os
        lib_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "lib")
        from lib.control_config_loader import ControlConfigLoader
        
        # 优先使用单个配置文件模式（一个组件一个文件）
        control_configs_dir = os.path.join(lib_dir, "control_configs")
        if os.path.exists(control_configs_dir):
            # 使用单个配置文件模式
            loader = ControlConfigLoader(lib_dir, use_individual_files=True)
            loader.load_all(lazy=False)  # 立即加载所有配置
            print("已从单个配置文件加载控件")
        elif os.path.exists(os.path.join(lib_dir, "controls.json")):
            # 使用旧的单个文件模式（向后兼容）
            loader = ControlConfigLoader(lib_dir, use_individual_files=False)
            loader.load_all()
            print("已从 controls.json 加载控件")
            # 提示可以迁移到单个文件模式
            print("提示: 可以运行 lib/migrate_to_individual_configs.py 迁移到单个配置文件模式")
        else:
            print("未找到控件配置文件，使用默认控件")
    except Exception as e:
        print(f"加载控件配置时出错: {e}")
        import traceback
        traceback.print_exc()
    
    # 创建Qt应用程序实例，sys.argv包含命令行参数
    app = QApplication(sys.argv)
    
    # 验证许可证：如果验证失败，显示错误信息并退出程序
    if not _validate_license():
        QMessageBox.critical(None, "错误", "软件错误联系作者QQ304888898")
        sys.exit(1)  # 退出程序，返回状态码1表示错误
    
    # 设置应用程序样式
    app.setStyle("Fusion")
    
    # 设置白色主题样式
    app.setStyleSheet("""
        QMainWindow {
            background-color: white;
            color: black;
        }
        QWidget {
            background-color: white;
            color: black;
        }
        QMenuBar {
            background-color: white;
            color: black;
            border-bottom: 1px solid #cccccc;
        }
        QMenuBar::item {
            background-color: white;
            color: black;
        }
        QMenuBar::item:selected {
            background-color: #e0e0e0;
        }
        QMenu {
            background-color: white;
            color: black;
            border: 1px solid #cccccc;
        }
        QMenu::item {
            background-color: white;
            color: black;
        }
        QMenu::item:selected {
            background-color: #e0e0e0;
        }
        QToolBar {
            background-color: white;
            color: black;
            border: 1px solid #cccccc;
        }
        QStatusBar {
            background-color: white;
            color: black;
            border-top: 1px solid #cccccc;
        }
        QSplitter {
            background-color: white;
        }
        QSplitter::handle {
            background-color: #cccccc;
        }
        QTabWidget::pane {
            border: 1px solid #cccccc;
            background-color: white;
        }
        QTabBar::tab {
            background-color: #f0f0f0;
            color: black;
            border: 1px solid #cccccc;
            padding: 5px;
        }
        QTabBar::tab:selected {
            background-color: white;
        }
        QGroupBox {
            background-color: white;
            color: black;
            border: 1px solid #cccccc;
            border-radius: 3px;
            margin-top: 10px;
            padding-top: 10px;
        }
        QGroupBox::title {
            subcontrol-origin: margin;
            left: 10px;
            padding: 0 5px 0 5px;
        }
        QPushButton {
            background-color: white;
            color: black;
            border: 1px solid #cccccc;
            padding: 5px;
            border-radius: 3px;
        }
        QPushButton:hover {
            background-color: #f0f0f0;
        }
        QPushButton:pressed {
            background-color: #e0e0e0;
        }
        QLineEdit {
            background-color: white;
            color: black;
            border: 1px solid #cccccc;
            padding: 3px;
            border-radius: 3px;
        }
        QTextEdit {
            background-color: white;
            color: black;
            border: 1px solid #cccccc;
            border-radius: 3px;
        }
        QLabel {
            color: black;
        }
        QScrollArea {
            background-color: white;
            border: 1px solid #cccccc;
        }
    """)
    
    import os
    try:
        p = _resolve_app_secrets_path()
        os.environ["APP_SECRETS_PATH"] = p
    except Exception:
        pass
    print("正在创建主窗口...")
    try:
        window = MainWindow()
        print("主窗口创建成功")
        window.showMaximized()
        print("主窗口已显示")
    except Exception as e:
        print(f"创建主窗口时出错: {e}")
        import traceback
        traceback.print_exc()
        return
    
    # 运行应用程序
    print("正在启动应用程序事件循环...")
    sys.exit(app.exec())


# 程序入口：当脚本直接运行时（非作为模块导入），执行main函数
if __name__ == "__main__":
    main()  # 调用主函数启动应用程序
