import os
from PIL import Image
import argparse

def is_invalid_color(color):
    """
    判断颜色是否为无效颜色 (255, 0, 0, 0)
    """
    return color == (255, 0, 0, 0)

def process_image(image_path, target_color=(255, 0, 0)):
    """
    处理图像：将黑色像素替换为指定位置的颜色，然后将该颜色替换为目标颜色
    """
    try:
        # 打开图像并转换为RGBA模式
        img = Image.open(image_path).convert("RGBA")
        pixels = img.load()
        
        # 获取图像尺寸
        width, height = img.size
        
        # 尝试获取参考颜色 - 先尝试(39,26)，如果无效则尝试(100,180)
        ref_color = None
        ref_positions = [(39, 26), (100, 180)]
        
        for pos in ref_positions:
            x, y = pos
            if x < width and y < height:
                color = pixels[x, y]
                if not is_invalid_color(color):
                    ref_color = color
                    print(f"使用位置 {pos} 的颜色作为参考: {ref_color}")
                    break
                else:
                    print(f"位置 {pos} 的颜色无效 (255,0,0,0)，尝试下一个位置")
        
        if ref_color is None:
            print(f"所有参考位置的颜色都无效或图像尺寸太小: {image_path}")
            return
        
        # 第一步：将黑色像素替换为参考颜色
        for x in range(width):
            for y in range(height):
                r, g, b, a = pixels[x, y]
                
                # 判断是否为黑色像素
                if r == 0 and g == 0 and b == 0:
                    # 将黑色替换为参考颜色，保持原有透明度
                    pixels[x, y] = (ref_color[0], ref_color[1], ref_color[2], a)
        
        # 第二步：将所有参考颜色替换为目标颜色
        for x in range(width):
            for y in range(height):
                r, g, b, a = pixels[x, y]
                
                # 判断是否为参考颜色
                if (r, g, b) == (ref_color[0], ref_color[1], ref_color[2]):
                    # 将参考颜色替换为目标颜色，保持原有透明度
                    # 如果目标颜色只有RGB值，添加透明度
                    if len(target_color) == 3:
                        pixels[x, y] = (target_color[0], target_color[1], target_color[2], a)
                    else:
                        pixels[x, y] = target_color
        
        # 覆盖保存原文件
        img.save(image_path, "PNG")
        print(f"已处理: {image_path}")
        
    except Exception as e:
        print(f"处理 {image_path} 时出错: {e}")

def process_directory(directory_path, target_color):
    """
    遍历目录下的所有PNG文件
    """
    if not os.path.exists(directory_path):
        print(f"目录不存在: {directory_path}")
        return
    
    png_count = 0
    for root, dirs, files in os.walk(directory_path):
        for file in files:
            if file.lower().endswith('.png'):
                file_path = os.path.join(root, file)
                process_image(file_path, target_color)
                png_count += 1
    
    if png_count == 0:
        print("在指定目录中未找到PNG文件")

def parse_color(color_str):
    """
    解析颜色字符串，支持格式: "255,0,0" 或 "255,0,0,255"
    """
    try:
        components = [int(c.strip()) for c in color_str.split(',')]
        if len(components) == 3:
            return tuple(components)  # RGB
        elif len(components) == 4:
            return tuple(components)  # RGBA
        else:
            raise ValueError("颜色格式不正确")
    except Exception as e:
        print(f"颜色解析错误: {e}，使用默认颜色红色(255,0,0)")
        return (255, 0, 0)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='处理PNG图像：将黑色像素替换为指定位置颜色，然后替换为指定颜色')
    parser.add_argument('directory', nargs='?', default='.', 
                       help='要处理的目录路径（默认为当前目录）')
    parser.add_argument('--color', '-c', default='255,0,0',
                       help='目标颜色，格式: "R,G,B" 或 "R,G,B,A" (默认为红色: 255,0,0)')
    
    args = parser.parse_args()
    
    # 解析目标颜色
    target_color = parse_color(args.color)
    
    print(f"开始处理目录: {args.directory}")
    print(f"目标颜色: {target_color}")
    print("参考位置: (39,26) 或 (100,180) - 跳过无效颜色(255,0,0,0)")
    print("正在处理图像...")
    
    process_directory(args.directory, target_color)
    
    print("处理完成！")