#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
窗口类管理器
管理多个窗口类，确保互不干扰、不重名
"""

from typing import Dict, List, Optional, Set
from .control_pyside import ControlManager
from .code_generator_pyside import CodeGenerator


class WindowClass:
    """窗口类数据类"""
    
    def __init__(self, name: str, control_manager: ControlManager = None):
        """
        初始化窗口类
        
        Args:
            name: 窗口类名称
            control_manager: 控件管理器，如果为None则创建新的
        """
        self.name = name
        self.control_manager = control_manager or ControlManager()
        self.code_generator = CodeGenerator(self.control_manager)
        self.code_generator.set_window_class_name(name)
        self.code_generator.set_window_name(name)
        self.canvas_width = 800
        self.canvas_height = 600
        self.window_title = "我的窗口"
        self.window_icon_path = ""
        self.pages = []
        self.dropdown_menus = []
        self.event_function_file = ""  # 功能函数文件内容，每个窗口类独立
        self.canvas_code = ""  # 画布代码内容，每个窗口类独立
    
    def to_dict(self) -> Dict:
        """转换为字典"""
        return {
            "name": self.name,
            "controls": [c.to_dict() for c in self.control_manager.controls.values()],
            "canvas_width": self.canvas_width,
            "canvas_height": self.canvas_height,
            "window_title": self.window_title,
            "window_icon_path": self.window_icon_path,
            "pages": self.pages,
            "dropdown_menus": self.dropdown_menus,
            "event_function_file": getattr(self, 'event_function_file', ""),  # 保存功能函数文件内容
            "canvas_code": getattr(self, 'canvas_code', "")  # 保存画布代码内容
        }
    
    @classmethod
    def from_dict(cls, data: Dict, name: str = None) -> 'WindowClass':
        """从字典创建"""
        window_name = name or data.get("name", "GeneratedWindow")
        window_class = cls(window_name)
        
        # 恢复控件
        from .control_pyside import Control
        for control_data in data.get("controls", []):
            control = Control.from_dict(control_data)
            window_class.control_manager.controls[control.id] = control
        
        # 恢复其他属性
        window_class.canvas_width = data.get("canvas_width", 800)
        window_class.canvas_height = data.get("canvas_height", 600)
        window_class.window_title = data.get("window_title", "我的窗口")
        window_class.window_icon_path = data.get("window_icon_path", "")
        window_class.pages = data.get("pages", [])
        window_class.dropdown_menus = data.get("dropdown_menus", [])
        window_class.event_function_file = data.get("event_function_file", "")  # 恢复功能函数文件内容
        window_class.canvas_code = data.get("canvas_code", "")  # 恢复画布代码内容
        
        # 更新代码生成器
        window_class.code_generator.set_window_class_name(window_name)
        window_class.code_generator.set_window_name(window_name)
        window_class.code_generator.set_window_title(window_class.window_title)
        window_class.code_generator.set_window_size(window_class.canvas_width, window_class.canvas_height)
        window_class.code_generator.set_window_icon(window_class.window_icon_path)
        window_class.code_generator.set_pages(window_class.pages)
        window_class.code_generator.set_dropdown_menus(window_class.dropdown_menus)
        
        return window_class


class WindowClassManager:
    """窗口类管理器"""
    
    def __init__(self):
        """初始化管理器"""
        self.window_classes: Dict[str, WindowClass] = {}
        self.current_window_class_name: Optional[str] = None
    
    def add_window_class(self, name: str) -> Optional[WindowClass]:
        """
        添加新窗口类
        
        Args:
            name: 窗口类名称
            
        Returns:
            创建的窗口类，如果名称已存在则返回None
        """
        if name in self.window_classes:
            return None
        
        window_class = WindowClass(name)
        self.window_classes[name] = window_class
        
        # 如果是第一个窗口类，设为当前
        if self.current_window_class_name is None:
            self.current_window_class_name = name
        
        return window_class
    
    def remove_window_class(self, name: str) -> bool:
        """
        删除窗口类
        
        Args:
            name: 窗口类名称
            
        Returns:
            是否成功删除
        """
        if name not in self.window_classes:
            return False
        
        # 如果删除的是当前窗口类，切换到其他窗口类
        if self.current_window_class_name == name:
            remaining = [n for n in self.window_classes.keys() if n != name]
            if remaining:
                self.current_window_class_name = remaining[0]
            else:
                self.current_window_class_name = None
        
        del self.window_classes[name]
        return True
    
    def get_window_class(self, name: str) -> Optional[WindowClass]:
        """获取窗口类"""
        return self.window_classes.get(name)
    
    def get_current_window_class(self) -> Optional[WindowClass]:
        """获取当前窗口类"""
        if self.current_window_class_name:
            return self.window_classes.get(self.current_window_class_name)
        return None
    
    def set_current_window_class(self, name: str) -> bool:
        """
        设置当前窗口类
        
        Args:
            name: 窗口类名称
            
        Returns:
            是否成功设置
        """
        if name in self.window_classes:
            self.current_window_class_name = name
            return True
        return False
    
    def rename_window_class(self, old_name: str, new_name: str) -> bool:
        """
        重命名窗口类
        
        Args:
            old_name: 旧名称
            new_name: 新名称
            
        Returns:
            是否成功重命名
        """
        if old_name not in self.window_classes:
            return False
        
        if new_name in self.window_classes:
            return False  # 新名称已存在
        
        window_class = self.window_classes.pop(old_name)
        window_class.name = new_name
        window_class.code_generator.set_window_class_name(new_name)
        window_class.code_generator.set_window_name(new_name)
        self.window_classes[new_name] = window_class
        
        # 更新当前窗口类名称
        if self.current_window_class_name == old_name:
            self.current_window_class_name = new_name
        
        return True
    
    def copy_window_class(self, source_name: str, target_name: str = None) -> Optional[WindowClass]:
        """
        复制窗口类
        
        Args:
            source_name: 源窗口类名称
            target_name: 目标窗口类名称，如果为None则自动生成
            
        Returns:
            复制的窗口类，如果失败则返回None
        """
        if source_name not in self.window_classes:
            return None
        
        # 生成目标名称
        if target_name is None:
            target_name = self._generate_unique_name(source_name)
        
        if target_name in self.window_classes:
            return None  # 目标名称已存在
        
        # 复制窗口类
        source = self.window_classes[source_name]
        source_dict = source.to_dict()
        target = WindowClass.from_dict(source_dict, target_name)
        
        self.window_classes[target_name] = target
        return target
    
    def _generate_unique_name(self, base_name: str) -> str:
        """
        生成唯一名称
        
        Args:
            base_name: 基础名称
            
        Returns:
            唯一的名称
        """
        if base_name not in self.window_classes:
            return base_name
        
        counter = 1
        while True:
            new_name = f"{base_name}_{counter}"
            if new_name not in self.window_classes:
                return new_name
            counter += 1
    
    def get_all_names(self) -> List[str]:
        """获取所有窗口类名称"""
        return list(self.window_classes.keys())
    
    def to_dict(self) -> Dict:
        """转换为字典"""
        return {
            "window_classes": {name: wc.to_dict() for name, wc in self.window_classes.items()},
            "current_window_class_name": self.current_window_class_name
        }
    
    @classmethod
    def from_dict(cls, data: Dict) -> 'WindowClassManager':
        """从字典创建"""
        manager = cls()
        
        for name, window_data in data.get("window_classes", {}).items():
            window_class = WindowClass.from_dict(window_data, name)
            manager.window_classes[name] = window_class
        
        manager.current_window_class_name = data.get("current_window_class_name")
        
        return manager

