#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
窗口类管理对话框
用于新建、删除、重命名窗口类
"""

from PySide6.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QListWidget, QListWidgetItem,
    QPushButton, QLineEdit, QLabel, QMessageBox, QDialogButtonBox
)
from PySide6.QtCore import Qt
from typing import Optional, List
from .frameless_window import FramelessWindow


class WindowClassDialog(FramelessWindow):
    """窗口类管理对话框"""
    
    def __init__(self, parent=None, window_class_manager=None):
        """
        初始化对话框
        
        Args:
            parent: 父窗口
            window_class_manager: 窗口类管理器
        """
        super().__init__(parent)
        self.setWindowTitle("窗口类管理")
        self.setMinimumWidth(400)
        self.setMinimumHeight(300)
        
        self.window_class_manager = window_class_manager
        self._init_ui()
        self._refresh_list()
    
    def _init_ui(self):
        """初始化UI"""
        # 使用FramelessWindow的内容区域
        content_widget = self.contentWidget()
        # 清除可能存在的布局
        if content_widget.layout():
            QWidget().setLayout(content_widget.layout())
        
        layout = QVBoxLayout(content_widget)
        
        # 标题
        title_label = QLabel("窗口类列表:")
        layout.addWidget(title_label)
        
        # 窗口类列表
        self.list_widget = QListWidget()
        self.list_widget.setSelectionMode(QListWidget.SingleSelection)
        layout.addWidget(self.list_widget)
        
        # 新建窗口类
        new_layout = QHBoxLayout()
        self.name_edit = QLineEdit()
        self.name_edit.setPlaceholderText("输入窗口类名称")
        new_btn = QPushButton("新建窗口")
        new_btn.setShortcut("Ctrl+N")
        new_btn.clicked.connect(self._new_window)
        new_layout.addWidget(QLabel("名称:"))
        new_layout.addWidget(self.name_edit)
        new_layout.addWidget(new_btn)
        layout.addLayout(new_layout)
        
        # 操作按钮
        btn_layout = QHBoxLayout()
        rename_btn = QPushButton("重命名")
        rename_btn.clicked.connect(self._rename_window)
        copy_btn = QPushButton("复制")
        copy_btn.setShortcut("Ctrl+C")
        copy_btn.clicked.connect(self._copy_window)
        delete_btn = QPushButton("删除")
        delete_btn.setShortcut("Delete")
        delete_btn.clicked.connect(self._delete_window)
        
        btn_layout.addWidget(rename_btn)
        btn_layout.addWidget(copy_btn)
        btn_layout.addWidget(delete_btn)
        btn_layout.addStretch()
        layout.addLayout(btn_layout)
        
        # 确定/取消按钮
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self._on_accept)
        button_box.rejected.connect(self.close)
        layout.addWidget(button_box)
    
    def _on_accept(self):
        """确定按钮（兼容FramelessWindow）"""
        self.close()
    
    def _refresh_list(self):
        """刷新列表"""
        self.list_widget.clear()
        if self.window_class_manager:
            for name in self.window_class_manager.get_all_names():
                item = QListWidgetItem(name)
                if name == self.window_class_manager.current_window_class_name:
                    item.setBackground(Qt.GlobalColor.lightGray)
                self.list_widget.addItem(item)
    
    def _new_window(self):
        """新建窗口"""
        name = self.name_edit.text().strip()
        if not name:
            QMessageBox.warning(self, "警告", "请输入窗口类名称")
            return
        
        # 验证名称是否有效
        if not self._is_valid_name(name):
            QMessageBox.warning(self, "警告", "窗口类名称只能包含字母、数字和下划线，且不能以数字开头")
            return
        
        # 检查是否重名
        if self.window_class_manager and name in self.window_class_manager.get_all_names():
            # 自动重命名
            name = self.window_class_manager._generate_unique_name(name)
            QMessageBox.information(self, "提示", f"名称已存在，已自动重命名为: {name}")
        
        if self.window_class_manager:
            window_class = self.window_class_manager.add_window_class(name)
            if window_class:
                self._refresh_list()
                self.name_edit.clear()
                # 选中新创建的窗口类
                items = self.list_widget.findItems(name, Qt.MatchExactly)
                if items:
                    self.list_widget.setCurrentItem(items[0])
            else:
                QMessageBox.warning(self, "错误", "创建窗口类失败")
    
    def _rename_window(self):
        """重命名窗口"""
        current_item = self.list_widget.currentItem()
        if not current_item:
            QMessageBox.warning(self, "警告", "请先选择要重命名的窗口类")
            return
        
        old_name = current_item.text()
        new_name, ok = self._get_text_input("重命名窗口类", "新名称:", old_name)
        if not ok or not new_name:
            return
        
        if not self._is_valid_name(new_name):
            QMessageBox.warning(self, "警告", "窗口类名称只能包含字母、数字和下划线，且不能以数字开头")
            return
        
        if self.window_class_manager:
            if self.window_class_manager.rename_window_class(old_name, new_name):
                self._refresh_list()
            else:
                QMessageBox.warning(self, "错误", "重命名失败，可能名称已存在")
    
    def _copy_window(self):
        """复制窗口"""
        current_item = self.list_widget.currentItem()
        if not current_item:
            QMessageBox.warning(self, "警告", "请先选择要复制的窗口类")
            return
        
        source_name = current_item.text()
        target_name, ok = self._get_text_input("复制窗口类", "新名称:", self.window_class_manager._generate_unique_name(source_name) if self.window_class_manager else f"{source_name}_copy")
        if not ok or not target_name:
            return
        
        if not self._is_valid_name(target_name):
            QMessageBox.warning(self, "警告", "窗口类名称只能包含字母、数字和下划线，且不能以数字开头")
            return
        
        if self.window_class_manager:
            # 检查是否重名，如果重名则自动重命名
            if target_name in self.window_class_manager.get_all_names():
                target_name = self.window_class_manager._generate_unique_name(target_name)
                QMessageBox.information(self, "提示", f"名称已存在，已自动重命名为: {target_name}")
            
            window_class = self.window_class_manager.copy_window_class(source_name, target_name)
            if window_class:
                self._refresh_list()
            else:
                QMessageBox.warning(self, "错误", "复制窗口类失败")
    
    def _delete_window(self):
        """删除窗口"""
        current_item = self.list_widget.currentItem()
        if not current_item:
            QMessageBox.warning(self, "警告", "请先选择要删除的窗口类")
            return
        
        name = current_item.text()
        
        # 至少保留一个窗口类
        if self.window_class_manager and len(self.window_class_manager.get_all_names()) <= 1:
            QMessageBox.warning(self, "警告", "至少需要保留一个窗口类")
            return
        
        reply = QMessageBox.question(
            self, "确认删除",
            f"确定要删除窗口类 '{name}' 吗？\n此操作不可撤销！",
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply == QMessageBox.Yes:
            if self.window_class_manager and self.window_class_manager.remove_window_class(name):
                self._refresh_list()
            else:
                QMessageBox.warning(self, "错误", "删除窗口类失败")
    
    def _is_valid_name(self, name: str) -> bool:
        """验证名称是否有效"""
        import re
        return bool(re.match(r'^[a-zA-Z_][a-zA-Z0-9_]*$', name))
    
    def _get_text_input(self, title: str, label: str, default: str = "") -> tuple:
        """获取文本输入"""
        from PySide6.QtWidgets import QInputDialog
        text, ok = QInputDialog.getText(self, title, label, text=default)
        return text.strip(), ok
    
    def get_selected_window_class_name(self) -> Optional[str]:
        """获取选中的窗口类名称"""
        current_item = self.list_widget.currentItem()
        if current_item:
            return current_item.text()
        return None

