#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
撤销重做管理器模块
实现撤销和重做功能
"""

"""
撤销重做管理器模块
实现撤销和重做功能，通过保存和恢复状态快照来实现
"""

# 导入类型提示
from typing import Dict, List, Optional, Any
# 导入PySide6图形界面相关类
from PySide6.QtGui import QColor  # 颜色类（可能用于界面显示）
from PySide6.QtWidgets import QMessageBox  # 消息框（可能用于错误提示）
# 导入控件类
from .control_pyside import Control  # 控件基类


class UndoRedoManager:
    """
    撤销重做管理器类
    通过保存和恢复状态快照实现操作的撤销和重做功能
    使用栈结构存储历史状态
    """
    
    def __init__(self, control_manager):
        """
        初始化撤销重做管理器
        
        Args:
            control_manager: 控件管理器实例，用于访问和控制控件数据
        """
        # 保存控件管理器引用
        self.control_manager = control_manager
        # 撤销栈：存储历史状态，栈顶为当前状态
        self.undo_stack = []
        # 重做栈：存储被撤销的状态，用于重做操作
        self.redo_stack = []
        # 最大撤销步数：限制撤销栈的大小，防止内存占用过大
        self.max_undo_steps = 50
        # 设计画布引用：用于保存和恢复画布相关状态（如画布大小、背景色等）
        self.design_canvas = None
        # 功能代码文件内容引用：用于保存和恢复功能代码文件内容
        self.event_function_file_content = None
        # 窗口类名称：用于标识当前窗口类
        self.window_class_name = None

    def set_design_canvas(self, canvas):
        """
        设置设计画布引用
        用于保存和恢复画布相关状态（如画布大小、背景色等）
        
        Args:
            canvas: 设计画布对象
        """
        self.design_canvas = canvas
    
    def set_event_function_file_content(self, content: str):
        """
        设置功能代码文件内容引用
        
        Args:
            content: 功能代码文件内容（字符串或可调用对象）
        """
        self.event_function_file_content = content
    
    def set_window_class_name(self, name: str):
        """
        设置窗口类名称
        
        Args:
            name: 窗口类名称
        """
        self.window_class_name = name
        
    def save_state(self, action_description: str = "", include_function_file: bool = False, include_canvas_code: bool = False, canvas_code_content: str = None):
        """
        保存当前状态到撤销栈
        在每次操作后调用此方法，保存操作前的状态，以便后续撤销
        
        Args:
            action_description: 操作描述（如"添加控件"、"移动控件"等），用于显示在撤销菜单中
            include_function_file: 是否包含功能代码文件内容
            include_canvas_code: 是否包含画布代码内容
            canvas_code_content: 画布代码内容（如果提供）
        """
        # 创建当前状态的快照
        state = {
            "controls": self._create_controls_snapshot(),  # 控件状态快照
            "canvas": self._create_canvas_snapshot(),  # 画布状态快照
            "description": action_description  # 操作描述
        }
        
        # 如果包含功能代码文件，保存其内容
        if include_function_file:
            if callable(self.event_function_file_content):
                # 如果是可调用对象（如lambda），调用它获取内容
                state["event_function_file"] = self.event_function_file_content()
            elif self.event_function_file_content is not None:
                # 如果是字符串，直接保存
                state["event_function_file"] = self.event_function_file_content
            else:
                state["event_function_file"] = ""
        
        # 如果包含画布代码，保存其内容
        if include_canvas_code:
            if canvas_code_content is not None:
                state["canvas_code"] = canvas_code_content
            else:
                state["canvas_code"] = ""
        
        # 将状态快照添加到撤销栈
        self.undo_stack.append(state)
        
        # ========== 限制撤销栈大小 ==========
        # 检查撤销栈的大小是否超过最大允许步数
        if len(self.undo_stack) > self.max_undo_steps:
            # 如果超过最大步数，移除栈底（最旧）的状态，保持栈大小在限制内
            # pop(0)从列表开头移除元素（栈底，最早的状态）
            self.undo_stack.pop(0)
        
        # ========== 清空重做栈 ==========
        # 执行新操作后，清空重做栈
        # 原因：新操作会覆盖历史，之前被撤销的操作无法再重做
        self.redo_stack.clear()
    
    def undo(self) -> tuple:
        """
        执行撤销操作
        
        Returns:
            (是否成功撤销, 功能代码文件内容, 画布代码内容) 元组
        """
        # ========== 检查是否可以进行撤销操作 ==========
        # 需要至少两个快照才能撤销：当前状态和前一状态
        # 如果栈中只有0个或1个状态，无法撤销
        if len(self.undo_stack) < 2:
            # 返回False表示无法撤销
            return False, None, None
        
        # ========== 将当前状态移动到重做栈 ==========
        # 从撤销栈中弹出最近的状态（栈顶，当前状态）
        last_state = self.undo_stack.pop()
        # 将弹出的状态添加到重做栈，以便后续可以重做
        self.redo_stack.append(last_state)
        
        # ========== 恢复到前一状态 ==========
        # 获取撤销栈当前顶部状态（前一状态，即要恢复到的状态）
        prev_state = self.undo_stack[-1]
        # 调用恢复状态方法，传入前一状态的控件快照和画布快照
        # prev_state.get("canvas")：如果画布快照不存在则返回None
        event_function_content = prev_state.get("event_function_file")
        canvas_code_content = prev_state.get("canvas_code")
        self._restore_state(prev_state["controls"], prev_state.get("canvas"), event_function_content, canvas_code_content)
        
        # 返回功能代码文件内容和画布代码内容，供主窗口使用
        return True, event_function_content if event_function_content is not None else None, canvas_code_content if canvas_code_content is not None else None
    
    def get_undo_event_function_content(self) -> Optional[str]:
        """
        获取撤销操作中的功能代码文件内容
        
        Returns:
            功能代码文件内容，如果不存在则返回None
        """
        if len(self.undo_stack) < 2:
            return None
        prev_state = self.undo_stack[-1]
        return prev_state.get("event_function_file")
    
    def redo(self) -> tuple:
        """
        执行重做操作
        从重做栈中恢复之前被撤销的状态
        
        Returns:
            (是否成功重做, 功能代码文件内容, 画布代码内容) 元组
        """
        # ========== 检查是否可以进行重做操作 ==========
        # 检查重做栈是否为空
        if len(self.redo_stack) == 0:
            # 如果重做栈为空，无法重做，返回False
            return False, None, None
        
        # ========== 从重做栈中恢复状态 ==========
        # 从重做栈中弹出最近的状态（栈顶，要恢复的状态）
        redo_state = self.redo_stack.pop()
        # 将当前状态保存到撤销栈（以便可以再次撤销）
        current_state = {
            "controls": self._create_controls_snapshot(),
            "canvas": self._create_canvas_snapshot(),
            "description": ""
        }
        if "event_function_file" in redo_state:
            if callable(self.event_function_file_content):
                current_state["event_function_file"] = self.event_function_file_content()
            elif self.event_function_file_content is not None:
                current_state["event_function_file"] = self.event_function_file_content
            else:
                current_state["event_function_file"] = ""
        if "canvas_code" in redo_state:
            # 保存当前画布代码（需要从外部传入）
            current_state["canvas_code"] = redo_state.get("canvas_code", "")
        self.undo_stack.append(current_state)
        
        # ========== 恢复重做状态 ==========
        # 调用恢复状态方法，传入重做状态的控件快照和画布快照
        event_function_content = redo_state.get("event_function_file")
        canvas_code_content = redo_state.get("canvas_code")
        self._restore_state(redo_state["controls"], redo_state.get("canvas"), event_function_content, canvas_code_content)
        
        # 返回功能代码文件内容和画布代码内容，供主窗口使用
        return True, event_function_content if event_function_content is not None else None, canvas_code_content if canvas_code_content is not None else None
    
    def get_redo_event_function_content(self) -> Optional[str]:
        """
        获取重做操作中的功能代码文件内容
        
        Returns:
            功能代码文件内容，如果不存在则返回None
        """
        if len(self.redo_stack) == 0:
            return None
        redo_state = self.redo_stack[-1]
        return redo_state.get("event_function_file")
    
    def can_undo(self) -> bool:
        """
        检查是否可以撤销
        
        Returns:
            是否可以撤销
        """
        return len(self.undo_stack) > 0
    
    def can_redo(self) -> bool:
        """
        检查是否可以重做
        
        Returns:
            是否可以重做
        """
        # 检查重做栈是否为空
        return len(self.redo_stack) > 0
    
    def clear(self):
        """清空撤销和重做栈"""
        self.undo_stack.clear()
        self.redo_stack.clear()
    
    def get_undo_description(self) -> str:
        """
        获取撤销操作的描述
        
        Returns:
            撤销操作的描述
        """
        if not self.can_undo():
            return ""
        
        return self.undo_stack[-1].get("description", "")
    
    def get_redo_description(self) -> str:
        """
        获取重做操作的描述
        
        Returns:
            重做操作的描述
        """
        if not self.can_redo():
            return ""
        # 返回重做栈顶部状态的描述
        return self.redo_stack[-1].get("description", "")
    
    def _create_controls_snapshot(self) -> Dict[str, Dict]:
        """
        创建控件快照
        
        Returns:
            控件快照字典
        """
        # ========== 初始化快照字典 ==========
        # 创建空字典，用于存储所有控件的快照数据
        snapshot = {}
        
        # ========== 遍历所有控件并创建快照 ==========
        # 遍历控件管理器中所有控件的字典（键为控件ID，值为控件对象）
        for control_id, control in self.control_manager.controls.items():
            # 为每个控件创建数据字典，保存控件的关键属性
            control_data = {
                "id": control.id,  # 控件ID（唯一标识符）
                "control_type": control.control_type,  # 控件类型（如"Button"、"Label"等）
                "x": control.x,  # 控件X坐标（位置）
                "y": control.y,  # 控件Y坐标（位置）
                "width": control.width,
                "height": control.height,
                "z_order": control.z_order,
                "properties": control.properties.copy()
            }
            snapshot[control_id] = control_data
        
        return snapshot
    
    def _restore_state(self, controls_snapshot: Dict[str, Dict], canvas_snapshot: Dict[str, Any] = None, event_function_file_content: str = None, canvas_code_content: str = None):
        """
        恢复控件状态
        
        Args:
            controls_snapshot: 控件快照字典
            canvas_snapshot: 画布快照字典
            event_function_file_content: 功能代码文件内容（可选）
            canvas_code_content: 画布代码内容（可选）
        """
        # 清空当前控件
        self.control_manager.controls.clear()
        
        # 从快照中恢复控件
        for _, control_data in controls_snapshot.items():
            control = Control.from_dict(control_data)
            self.control_manager.controls[control.id] = control

        if self.design_canvas and canvas_snapshot:
            try:
                self.design_canvas.begin_state_restore()
                w = canvas_snapshot.get("canvas_width")
                h = canvas_snapshot.get("canvas_height")
                if isinstance(w, int) and isinstance(h, int):
                    self.design_canvas.set_canvas_size(w, h)
                sg = canvas_snapshot.get("show_grid")
                if isinstance(sg, bool):
                    self.design_canvas.set_show_grid(sg)
                gs = canvas_snapshot.get("grid_size")
                if isinstance(gs, int):
                    self.design_canvas.set_grid_size(gs)
                stg = canvas_snapshot.get("snap_to_grid")
                if isinstance(stg, bool):
                    self.design_canvas.set_snap_to_grid(stg)
                bg_hex = canvas_snapshot.get("bg_color")
                if isinstance(bg_hex, str) and bg_hex:
                    self.design_canvas.set_canvas_bg_color(QColor(bg_hex))
                for prop in [
                    "window_title", "window_border", "window_bg_color",
                    "window_bg_image", "window_bg_image_mode"
                ]:
                    if prop in canvas_snapshot:
                        self.design_canvas.set_window_property(prop, canvas_snapshot.get(prop))
                self.design_canvas.window_menus = canvas_snapshot.get("window_menus", [])
                self.design_canvas.window_dropdown_menus = canvas_snapshot.get("window_dropdown_menus", [])
                self.design_canvas.active_page = canvas_snapshot.get("active_page", "页面1")
                self.design_canvas.active_index = canvas_snapshot.get("active_index", 0)
                self.design_canvas.update()
                self.design_canvas.end_state_restore()
            except Exception:
                pass
        
        # 恢复功能代码文件内容和画布代码内容（如果提供）
        # 这些内容需要主窗口处理，这里只保存到状态中
        # 主窗口会在恢复状态后调用相应的恢复方法
        self._pending_event_function_content = event_function_file_content
        self._pending_canvas_code_content = canvas_code_content
    
    def get_pending_event_function_content(self) -> Optional[str]:
        """获取待恢复的功能代码文件内容"""
        return getattr(self, '_pending_event_function_content', None)
    
    def get_pending_canvas_code_content(self) -> Optional[str]:
        """获取待恢复的画布代码内容"""
        return getattr(self, '_pending_canvas_code_content', None)
    
    def clear_pending_content(self):
        """清除待恢复的内容"""
        if hasattr(self, '_pending_event_function_content'):
            delattr(self, '_pending_event_function_content')
        if hasattr(self, '_pending_canvas_code_content'):
            delattr(self, '_pending_canvas_code_content')

    def _create_canvas_snapshot(self) -> Dict[str, Any]:
        snap = {}
        if self.design_canvas:
            try:
                snap = {
                    "canvas_width": getattr(self.design_canvas, "canvas_width", 800),
                    "canvas_height": getattr(self.design_canvas, "canvas_height", 520),
                    "show_grid": getattr(self.design_canvas, "show_grid", False),
                    "grid_size": getattr(self.design_canvas, "grid_size", 10),
                    "snap_to_grid": getattr(self.design_canvas, "snap_to_grid", False),
                    "bg_color": getattr(self.design_canvas, "bg_color", QColor(255, 255, 255)).name(),
                    "window_title": getattr(self.design_canvas, "window_title", ""),
                    "window_border": getattr(self.design_canvas, "window_border", ""),
                    "window_bg_color": getattr(self.design_canvas, "window_bg_color", ""),
                    "window_bg_image": getattr(self.design_canvas, "window_bg_image", ""),
                    "window_bg_image_mode": getattr(self.design_canvas, "window_bg_image_mode", ""),
                    "window_menus": getattr(self.design_canvas, "window_menus", []),
                    "window_dropdown_menus": getattr(self.design_canvas, "window_dropdown_menus", []),
                    "active_page": getattr(self.design_canvas, "active_page", "页面1"),
                    "active_index": getattr(self.design_canvas, "active_index", 0)
                }
            except Exception:
                snap = {}
        return snap
