#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
主题管理器模块
负责管理主题和图标颜色的自动选择
"""

import os
import shutil
import subprocess
from pathlib import Path
from typing import Optional, Dict, List, Tuple
from PySide6.QtCore import QSettings


class ThemeManager:
    """主题管理器"""
    
    # 主题到图标颜色的映射
    THEME_TO_ICON_COLOR = {
        "white": "white",
        "black": "black",
        "blackblue": "blue",
        "blackpink": "pink",
        "bluewhite": "blue",
        # 可以添加更多映射
    }
    
    # 默认图标颜色（当主题没有对应颜色时使用）
    DEFAULT_ICON_COLOR = "white"
    
    def __init__(self, base_dir: Optional[Path] = None):
        """
        初始化主题管理器
        
        Args:
            base_dir: 项目根目录路径，如果为None则自动检测
        """
        if base_dir is None:
            # 自动检测项目根目录
            current_file = Path(__file__).resolve()
            base_dir = current_file.parent.parent
        
        self.base_dir = Path(base_dir)
        self.icon_dir = self.base_dir / "ui" / "icon"
        self.theme_dir = self.base_dir / "ui" / "color"
        
        # 扫描并同步图标颜色和主题
        self._sync_icon_colors_and_themes()
    
    def get_icon_path(self, icon_name: str, theme_name: Optional[str] = None, icon_color: Optional[str] = None) -> str:
        """
        根据主题或图标颜色获取图标路径
        
        Args:
            icon_name: 图标文件名（如 "file.png"）
            theme_name: 主题名称，如果为None则从设置中读取
            icon_color: 图标颜色名称，如果指定则优先使用，否则根据主题自动选择
        
        Returns:
            图标文件的完整路径
        """
        # 如果指定了图标颜色，直接使用
        if icon_color:
            icon_path = self.icon_dir / icon_color / icon_name
            if icon_path.exists():
                return str(icon_path)
        
        # 否则根据主题获取图标颜色
        if theme_name is None:
            settings = QSettings("VisualProgramming", "Settings")
            theme_name = settings.value("theme/name", "white", type=str)
            # 尝试从设置中获取自定义图标颜色
            custom_icon_color = settings.value("icon/color", "", type=str)
            if custom_icon_color:
                icon_path = self.icon_dir / custom_icon_color / icon_name
                if icon_path.exists():
                    return str(icon_path)
        
        # 获取对应的图标颜色
        icon_color = self._get_icon_color_for_theme(theme_name)
        
        # 构建图标路径
        icon_path = self.icon_dir / icon_color / icon_name
        
        # 如果图标不存在，尝试使用默认颜色
        if not icon_path.exists():
            default_path = self.icon_dir / self.DEFAULT_ICON_COLOR / icon_name
            if default_path.exists():
                return str(default_path)
            # 如果默认颜色也不存在，返回第一个找到的
            for color_dir in self.icon_dir.iterdir():
                if color_dir.is_dir():
                    test_path = color_dir / icon_name
                    if test_path.exists():
                        return str(test_path)
        
        return str(icon_path)
    
    def _get_icon_color_for_theme(self, theme_name: str) -> str:
        """
        根据主题名称获取对应的图标颜色
        
        Args:
            theme_name: 主题名称
        
        Returns:
            图标颜色名称
        """
        # 从映射中获取
        if theme_name in self.THEME_TO_ICON_COLOR:
            return self.THEME_TO_ICON_COLOR[theme_name]
        
        # 如果主题名称直接对应图标颜色，直接返回
        icon_color_path = self.icon_dir / theme_name
        if icon_color_path.exists() and icon_color_path.is_dir():
            return theme_name
        
        # 尝试从主题名称中提取颜色（如 "blackblue" -> "blue"）
        for color in ["blue", "pink", "red", "green", "black", "white"]:
            if color in theme_name.lower():
                icon_color_path = self.icon_dir / color
                if icon_color_path.exists() and icon_color_path.is_dir():
                    return color
        
        # 默认返回白色
        return self.DEFAULT_ICON_COLOR
    
    def _sync_icon_colors_and_themes(self):
        """
        同步图标颜色和主题
        扫描图标文件夹，为没有对应主题的颜色创建主题
        """
        if not self.icon_dir.exists():
            return
        
        # 获取所有图标颜色文件夹
        icon_colors = []
        for item in self.icon_dir.iterdir():
            if item.is_dir() and not item.name.startswith('.'):
                icon_colors.append(item.name)
        
        # 获取所有现有主题
        existing_themes = set()
        if self.theme_dir.exists():
            for file in self.theme_dir.glob("*.qss"):
                existing_themes.add(file.stem)
        
        # 为没有对应主题的图标颜色创建主题
        for icon_color in icon_colors:
            # 检查是否已有对应主题
            has_theme = False
            for theme in existing_themes:
                if icon_color in theme.lower() or self._get_icon_color_for_theme(theme) == icon_color:
                    has_theme = True
                    break
            
            # 如果没有主题，创建一个
            if not has_theme:
                self._create_theme_for_color(icon_color)
    
    def _create_theme_for_color(self, icon_color: str):
        """
        为图标颜色创建对应的主题文件
        
        Args:
            icon_color: 图标颜色名称
        """
        if not self.theme_dir.exists():
            self.theme_dir.mkdir(parents=True, exist_ok=True)
        
        theme_file = self.theme_dir / f"{icon_color}.qss"
        
        # 如果主题文件已存在，不覆盖
        if theme_file.exists():
            return
        
        # 根据颜色生成主题样式
        theme_style = self._generate_theme_style(icon_color)
        
        # 写入主题文件
        with open(theme_file, 'w', encoding='utf-8') as f:
            f.write(theme_style)
        
        print(f"已为图标颜色 '{icon_color}' 创建主题文件: {theme_file}")
    
    def _generate_theme_style(self, icon_color: str) -> str:
        """
        根据图标颜色生成主题样式
        
        Args:
            icon_color: 图标颜色名称
        
        Returns:
            QSS样式字符串
        """
        # 定义颜色方案
        color_schemes = {
            "blue": {
                "bg_main": "#0066cc",
                "bg_secondary": "#0088ff",
                "bg_tertiary": "#004499",
                "text": "white",
                "border": "#004499",
            },
            "green": {
                "bg_main": "#006600",
                "bg_secondary": "#008800",
                "bg_tertiary": "#004400",
                "text": "white",
                "border": "#004400",
            },
            "red": {
                "bg_main": "#cc0000",
                "bg_secondary": "#ff0000",
                "bg_tertiary": "#990000",
                "text": "white",
                "border": "#990000",
            },
            "pink": {
                "bg_main": "#cc0066",
                "bg_secondary": "#ff0088",
                "bg_tertiary": "#990044",
                "text": "white",
                "border": "#990044",
            },
            "black": {
                "bg_main": "#1a1a1a",
                "bg_secondary": "#1a1a1a",
                "bg_tertiary": "#000000",
                "text": "#e0e0e0",
                "border": "#666666",
            },
            "white": {
                "bg_main": "white",
                "bg_secondary": "#f0f0f0",
                "bg_tertiary": "#e0e0e0",
                "text": "black",
                "border": "#cccccc",
            },
        }
        
        # 获取颜色方案，如果不存在则使用默认
        scheme = color_schemes.get(icon_color, color_schemes["white"])
        
        # 生成QSS样式
        qss = f"""/* {icon_color}主题 - 完整组件样式 */

QMainWindow {{
    background-color: {scheme['bg_main']};
    color: {scheme['text']};
}}

QWidget {{
    background-color: {scheme['bg_main']};
    color: {scheme['text']};
}}

QMenuBar {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    border-bottom: 1px solid {scheme['border']};
}}
QMenuBar::item {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    padding: 4px 8px;
}}
QMenuBar::item:selected {{
    background-color: {scheme['bg_tertiary']};
}}
QMenuBar::item:pressed {{
    background-color: {scheme['bg_tertiary']};
}}

QMenu {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    border: 1px solid {scheme['border']};
    padding: 4px;
}}
QMenu::item {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    padding: 4px 20px 4px 30px;
}}
QMenu::item:selected {{
    background-color: {scheme['bg_tertiary']};
}}
QMenu::item:disabled {{
    color: #666666;
}}
QMenu::separator {{
    height: 1px;
    background-color: {scheme['border']};
    margin: 4px 0px;
}}

QToolBar {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    border: 1px solid {scheme['border']};
    spacing: 3px;
}}
QToolBar::separator {{
    background-color: {scheme['border']};
    width: 1px;
    margin: 3px;
}}

QToolButton {{
    background-color: transparent;
    color: {scheme['text']};
    border: 1px solid transparent;
    border-radius: 3px;
    padding: 4px;
}}
QToolButton:hover {{
    background-color: {scheme['bg_tertiary']};
    border: 1px solid {scheme['border']};
}}
QToolButton:pressed {{
    background-color: {scheme['bg_tertiary']};
}}
QToolButton:checked {{
    background-color: {scheme['bg_tertiary']};
    border: 1px solid {scheme['border']};
}}

/* 左侧图标栏 */
QWidget#LeftIconBar {{
    background-color: {scheme['bg_secondary']};
    border-right: 1px solid {scheme['border']};
}}
QWidget#LeftIconBar QToolButton {{
    background-color: transparent;
    border: none;
    padding: 5px;
    text-align: center;
}}
QWidget#LeftIconBar QToolButton:hover {{
    background-color: {scheme['bg_tertiary']};
}}
QWidget#LeftIconBar QToolButton:checked {{
    background-color: {scheme['bg_tertiary']};
}}

QStatusBar {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    border-top: 1px solid {scheme['border']};
}}

QPushButton {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    border: 1px solid {scheme['border']};
    padding: 5px;
    border-radius: 3px;
}}
QPushButton:hover {{
    background-color: {scheme['bg_tertiary']};
}}
QPushButton:pressed {{
    background-color: {scheme['bg_tertiary']};
}}

QLineEdit {{
    background-color: {scheme['bg_main']};
    color: {scheme['text']};
    border: 1px solid {scheme['border']};
    padding: 3px;
    border-radius: 3px;
}}

QTextEdit {{
    background-color: {scheme['bg_main']};
    color: {scheme['text']};
    border: 1px solid {scheme['border']};
    border-radius: 3px;
}}

QScrollArea {{
    background-color: {scheme['bg_main']};
    border: 1px solid {scheme['border']};
}}

QTabWidget::pane {{
    border: 1px solid {scheme['border']};
    background-color: {scheme['bg_main']};
}}

QTabBar::tab {{
    background-color: {scheme['bg_secondary']};
    color: {scheme['text']};
    border: 1px solid {scheme['border']};
    padding: 5px;
}}
QTabBar::tab:selected {{
    background-color: {scheme['bg_main']};
}}
"""
        return qss
    
    def create_custom_color_theme(self, color_name: str, rgb_color: Tuple[int, int, int]) -> bool:
        """
        创建自定义颜色的主题和图标
        
        Args:
            color_name: 颜色名称（将用作文件夹名和主题名）
            rgb_color: RGB颜色值，格式为 (R, G, B)
        
        Returns:
            是否创建成功
        """
        try:
            # 1. 创建图标颜色文件夹
            icon_color_dir = self.icon_dir / color_name
            if not icon_color_dir.exists():
                icon_color_dir.mkdir(parents=True, exist_ok=True)
            
            # 2. 复制基础图标（使用white作为基础）
            base_color_dir = self.icon_dir / "white"
            if not base_color_dir.exists():
                # 如果white不存在，使用第一个可用的颜色文件夹
                for item in self.icon_dir.iterdir():
                    if item.is_dir() and not item.name.startswith('.'):
                        base_color_dir = item
                        break
            
            if base_color_dir.exists():
                # 复制所有PNG文件
                for png_file in base_color_dir.glob("*.png"):
                    dest_file = icon_color_dir / png_file.name
                    shutil.copy2(png_file, dest_file)
                
                # 复制color.py文件（如果存在）
                color_py_source = base_color_dir / "color.py"
                if color_py_source.exists():
                    color_py_dest = icon_color_dir / "color.py"
                    shutil.copy2(color_py_source, color_py_dest)
            
            # 3. 使用color.py处理图标
            color_py_path = self.icon_dir / "blue" / "color.py"
            if not color_py_path.exists():
                # 尝试从其他颜色文件夹查找
                for item in self.icon_dir.iterdir():
                    if item.is_dir():
                        test_path = item / "color.py"
                        if test_path.exists():
                            color_py_path = test_path
                            break
            
            if color_py_path.exists():
                # 运行color.py处理图标
                color_str = f"{rgb_color[0]},{rgb_color[1]},{rgb_color[2]}"
                try:
                    import sys
                    python_exe = sys.executable
                    result = subprocess.run(
                        [python_exe, str(color_py_path), str(icon_color_dir), "--color", color_str],
                        capture_output=True,
                        text=True,
                        timeout=60,
                        cwd=str(self.base_dir)
                    )
                    if result.returncode != 0:
                        print(f"处理图标时出错: {result.stderr}")
                        if result.stdout:
                            print(f"输出: {result.stdout}")
                        return False
                    else:
                        print(f"图标处理成功: {result.stdout}")
                except Exception as e:
                    print(f"运行color.py时出错: {e}")
                    import traceback
                    traceback.print_exc()
                    return False
            else:
                print("未找到color.py文件，无法处理图标颜色")
                return False
            
            # 4. 创建主题文件
            self._create_theme_for_color(color_name)
            
            # 5. 更新主题到图标颜色的映射
            self.THEME_TO_ICON_COLOR[color_name] = color_name
            
            print(f"成功创建自定义颜色主题: {color_name}")
            return True
            
        except Exception as e:
            print(f"创建自定义颜色主题时出错: {e}")
            import traceback
            traceback.print_exc()
            return False
    
    def get_available_icon_colors(self) -> List[str]:
        """
        获取所有可用的图标颜色
        
        Returns:
            图标颜色名称列表
        """
        if not self.icon_dir.exists():
            return []
        
        colors = []
        for item in self.icon_dir.iterdir():
            if item.is_dir() and not item.name.startswith('.'):
                colors.append(item.name)
        
        return sorted(colors)
    
    def get_available_themes(self) -> List[str]:
        """
        获取所有可用的主题
        
        Returns:
            主题名称列表
        """
        if not self.theme_dir.exists():
            return []
        
        themes = []
        for file in self.theme_dir.glob("*.qss"):
            themes.append(file.stem)
        
        return sorted(themes)

