#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
主题和字体统一管理器
负责统一管理所有组件的主题样式和字体设置
"""

import json
from pathlib import Path
from typing import Dict, Optional
from PySide6.QtCore import QSettings
from PySide6.QtGui import QFont


class ThemeFontManager:
    """主题和字体统一管理器"""
    
    def __init__(self, base_dir: Optional[Path] = None):
        """
        初始化管理器
        
        Args:
            base_dir: 项目根目录路径
        """
        if base_dir is None:
            current_file = Path(__file__).resolve()
            base_dir = current_file.parent.parent
        
        self.base_dir = Path(base_dir)
        self.theme_dir = self.base_dir / "ui" / "color"
        self.config_dir = self.base_dir / "lib" / "component_configs"
        
        # 确保配置目录存在
        self.config_dir.mkdir(parents=True, exist_ok=True)
        
        # 加载设置
        self.settings = QSettings("VisualProgramming", "Settings")
    
    def get_current_theme_name(self) -> str:
        """获取当前主题名称"""
        return self.settings.value("theme/name", "white", type=str)
    
    def get_unified_font(self) -> QFont:
        """获取统一字体"""
        font_family = self.settings.value("theme/font_family", "Microsoft YaHei UI", type=str)
        font_size = self.settings.value("theme/font_size", 10, type=int)
        font = QFont(font_family, font_size)
        return font
    
    def get_component_style(self, component_name: str) -> str:
        """
        获取组件的样式
        
        Args:
            component_name: 组件名称
        
        Returns:
            QSS样式字符串
        """
        theme_name = self.get_current_theme_name()
        theme_path = self.theme_dir / f"{theme_name}.qss"
        
        if not theme_path.exists():
            return ""
        
        # 读取主题文件
        with open(theme_path, 'r', encoding='utf-8') as f:
            theme_content = f.read()
        
        # 获取组件特定样式
        component_config_path = self.config_dir / f"{component_name}.json"
        if component_config_path.exists():
            with open(component_config_path, 'r', encoding='utf-8') as f:
                config = json.load(f)
                if 'style' in config:
                    return config['style']
        
        # 返回基础样式
        return theme_content
    
    def get_component_config(self, component_name: str) -> Dict:
        """
        获取组件配置
        
        Args:
            component_name: 组件名称
        
        Returns:
            组件配置字典
        """
        component_config_path = self.config_dir / f"{component_name}.json"
        if component_config_path.exists():
            with open(component_config_path, 'r', encoding='utf-8') as f:
                return json.load(f)
        return {}
    
    def save_component_config(self, component_name: str, config: Dict):
        """
        保存组件配置
        
        Args:
            component_name: 组件名称
            config: 配置字典
        """
        component_config_path = self.config_dir / f"{component_name}.json"
        with open(component_config_path, 'w', encoding='utf-8') as f:
            json.dump(config, f, ensure_ascii=False, indent=2)

