#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
设置对话框模块
提供软件的各种设置功能，包括基础设置、主题设置、控件设置、Python环境设置、AI设置等
"""

import os
import json
import shutil
from pathlib import Path
from typing import Dict, List, Optional

from PySide6.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QTreeWidget, QTreeWidgetItem,
    QStackedWidget, QPushButton, QLabel, QLineEdit, QSpinBox, QComboBox,
    QCheckBox, QListWidget, QListWidgetItem, QFileDialog, QGroupBox,
    QFormLayout, QMessageBox, QWidget, QScrollArea, QFrame, QColorDialog
)
from PySide6.QtCore import QEventLoop, QSize
from PySide6.QtCore import Qt, Signal, QSettings
from PySide6.QtGui import QFont, QIcon, QColor
from .frameless_window import FramelessWindow

from .control_library_pyside import ControlLibrary
from .theme_manager import ThemeManager


class SettingsDialog(FramelessWindow):
    """设置对话框（无标题栏）"""
    
    # 设置改变信号
    settings_changed = Signal()
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("设置")
        self.setMinimumSize(800, 600)
        self.resize(1000, 700)
        
        # 加载设置
        self.settings = QSettings("VisualProgramming", "Settings")
        
        # 初始化主题管理器
        self.theme_manager = ThemeManager()
        
        # 初始化UI
        self._init_ui()
        
        # 加载所有设置
        
        # 应用主题样式
        self._apply_theme_style()
        
        # 加载所有设置
        self._load_all_settings()
    
    def _init_ui(self):
        """初始化UI"""
        # 使用FramelessWindow的内容区域
        content_widget = self.contentWidget()
        # 清除可能存在的布局
        if content_widget.layout():
            QWidget().setLayout(content_widget.layout())
        
        main_layout = QHBoxLayout(content_widget)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)
        
        # 先初始化 pages 字典，避免在信号连接时访问出错
        self.pages = {}
        
        # 左侧树形框
        self.tree_widget = QTreeWidget()
        self.tree_widget.setHeaderLabel("设置分类")
        self.tree_widget.setFixedWidth(200)
        main_layout.addWidget(self.tree_widget)
        
        # 右侧内容区域（包含堆叠窗口和按钮）
        right_widget = QWidget()
        right_layout = QVBoxLayout(right_widget)
        right_layout.setContentsMargins(0, 0, 0, 0)
        right_layout.setSpacing(0)
        
        # 右侧堆叠窗口
        self.stacked_widget = QStackedWidget()
        right_layout.addWidget(self.stacked_widget, 1)
        
        # 创建各个设置页面（必须在连接信号之前创建）
        self._create_tree_items()
        self._create_setting_pages()
        
        # 在创建完页面后再连接信号
        self.tree_widget.currentItemChanged.connect(self._on_tree_item_changed)
        
        # 底部按钮区域
        button_container = QWidget()
        button_container.setFixedHeight(60)
        button_layout = QHBoxLayout(button_container)
        button_layout.setContentsMargins(20, 10, 20, 10)
        button_layout.setSpacing(10)
        button_layout.addStretch()
        
        # 按钮顺序：取消、应用、确定（更符合常见习惯）
        self.cancel_button = QPushButton("取消")
        self.cancel_button.setFixedSize(80, 32)
        self.cancel_button.clicked.connect(self.reject)
        button_layout.addWidget(self.cancel_button)
        
        self.apply_button = QPushButton("应用")
        self.apply_button.setFixedSize(80, 32)
        self.apply_button.clicked.connect(self._on_apply_clicked)
        button_layout.addWidget(self.apply_button)
        
        self.ok_button = QPushButton("确定")
        self.ok_button.setFixedSize(80, 32)
        self.ok_button.setDefault(True)  # 设置为默认按钮
        self.ok_button.clicked.connect(self._on_ok_clicked)
        button_layout.addWidget(self.ok_button)
        
        right_layout.addWidget(button_container)
        main_layout.addWidget(right_widget, 1)
    
    def _apply_theme_style(self):
        """应用主题样式"""
        from PySide6.QtCore import QSettings
        from pathlib import Path
        settings = QSettings("VisualProgramming", "Settings")
        theme_name = settings.value("theme/name", "white", type=str)
        theme_path = Path(__file__).parent.parent / "ui" / "color" / f"{theme_name}.qss"
        if theme_path.exists():
            with open(theme_path, "r", encoding="utf-8") as f:
                theme_style = f.read()
            # 判断是否为黑色主题
            black_themes = ["black", "blackblue", "blackpink"]
            is_black_theme = theme_name.lower() in black_themes
            
            # 根据主题类型设置标题栏样式
            if is_black_theme:
                # 黑色主题：使用纯色 #333333
                title_bar_style = """
QFrame#TitleBar {
    background-color: #333333;
    border-bottom: 1px solid #000000;
}
QLabel#TitleLabel {
    background-color: transparent;
    color: #e0e0e0;
    font-weight: bold;
    font-size: 12px;
}
QPushButton#MinimizeButton, QPushButton#MaximizeButton, QPushButton#CloseButton {
    background-color: transparent;
    border: none;
    border-radius: 3px;
    color: #e0e0e0;
    font-size: 16px;
    font-weight: bold;
}
QPushButton#MinimizeButton:hover {
    background-color: #3d3d3d;
}
QPushButton#MaximizeButton:hover {
    background-color: #3d3d3d;
}
QPushButton#CloseButton:hover {
    background-color: #ff4444;
    color: white;
}
"""
            else:
                # 非黑色主题：使用原有样式
                title_bar_style = """
QFrame#TitleBar {
    background-color: qlineargradient(x1:0, y1:0, x2:0, y2:1, 
        stop:0 #f0f0f0, stop:1 #333333);
    border-bottom: 1px solid #333333;
}
QLabel#TitleLabel {
    background-color: transparent;
    color: #333333;
    font-weight: bold;
    font-size: 12px;
}
QPushButton#MinimizeButton, QPushButton#MaximizeButton, QPushButton#CloseButton {
    background-color: transparent;
    border: none;
    border-radius: 3px;
}
QPushButton#MinimizeButton:hover {
    background-color: #333333;
}
QPushButton#MaximizeButton:hover {
    background-color: #333333;
}
QPushButton#CloseButton:hover {
    background-color: #ff4444;
    color: white;
}
"""
            self.setStyleSheet(theme_style + title_bar_style)
    
    def accept(self):
        """确定按钮（重写以兼容FramelessWindow）"""
        self._on_ok_clicked()
        super().accept()
    
    def reject(self):
        """取消按钮（重写以兼容FramelessWindow）"""
        super().reject()
    
    def _create_tree_items(self):
        """创建树形框项目"""
        items = [
            ("基础设置", "basic"),
            ("主题设置", "theme"),
            ("页面设置", "page"),
            ("打包设置", "package"),
            ("控件设置", "controls"),
            ("函数库设置", "fnpy_libraries"),
            ("Python环境设置", "python"),
            ("AI设置", "ai")
        ]
        
        for name, key in items:
            item = QTreeWidgetItem(self.tree_widget, [name])
            item.setData(0, Qt.UserRole, key)
            self.tree_widget.addTopLevelItem(item)
        
        # 默认选中第一项
        if self.tree_widget.topLevelItem(0):
            self.tree_widget.setCurrentItem(self.tree_widget.topLevelItem(0))
    
    def _create_setting_pages(self):
        """创建各个设置页面"""
        # 先初始化 pages 字典，避免在 _on_tree_item_changed 中访问时出错
        self.pages = {}
        
        # 基础设置
        self.pages["basic"] = self._create_basic_settings_page()
        self.stacked_widget.addWidget(self.pages["basic"])
        
        # 主题设置
        self.pages["theme"] = self._create_theme_settings_page()
        self.stacked_widget.addWidget(self.pages["theme"])
        
        # 页面设置
        self.pages["page"] = self._create_page_settings_page()
        self.stacked_widget.addWidget(self.pages["page"])
        
        # 打包设置
        self.pages["package"] = self._create_package_settings_page()
        self.stacked_widget.addWidget(self.pages["package"])
        
        # 控件设置
        self.pages["controls"] = self._create_controls_settings_page()
        self.stacked_widget.addWidget(self.pages["controls"])
        
        # 函数库设置
        self.pages["fnpy_libraries"] = self._create_fnpy_libraries_settings_page()
        self.stacked_widget.addWidget(self.pages["fnpy_libraries"])
        
        # Python环境设置
        self.pages["python"] = self._create_python_settings_page()
        self.stacked_widget.addWidget(self.pages["python"])
        
        # AI设置
        self.pages["ai"] = self._create_ai_settings_page()
        self.stacked_widget.addWidget(self.pages["ai"])
    
    def _create_basic_settings_page(self) -> QWidget:
        """创建基础设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        content = QWidget()
        content_layout = QVBoxLayout(content)
        
        # 通用设置组
        general_group = QGroupBox("通用设置")
        general_layout = QFormLayout()
        
        # 自动保存
        self.auto_save_checkbox = QCheckBox("启用自动保存")
        general_layout.addRow("", self.auto_save_checkbox)
        
        # 自动保存间隔（秒）
        self.auto_save_interval = QSpinBox()
        self.auto_save_interval.setRange(30, 3600)
        self.auto_save_interval.setValue(300)
        self.auto_save_interval.setSuffix(" 秒")
        general_layout.addRow("自动保存间隔:", self.auto_save_interval)
        
        # 启动时行为
        self.startup_behavior_combo = QComboBox()
        self.startup_behavior_combo.addItems(["打开新项目", "打开上次项目", "显示欢迎页面"])
        general_layout.addRow("启动时行为:", self.startup_behavior_combo)
        
        # 最近打开文件数量
        self.recent_files_count = QSpinBox()
        self.recent_files_count.setRange(0, 20)
        self.recent_files_count.setValue(10)
        general_layout.addRow("最近文件数量:", self.recent_files_count)
        
        general_group.setLayout(general_layout)
        content_layout.addWidget(general_group)
        
        # 窗口设置组
        window_group = QGroupBox("窗口设置")
        window_layout = QFormLayout()
        
        # 记住窗口大小和位置
        self.remember_window_state = QCheckBox("记住窗口大小和位置")
        window_layout.addRow("", self.remember_window_state)
        
        # 最小化到系统托盘
        self.minimize_to_tray = QCheckBox("最小化到系统托盘")
        window_layout.addRow("", self.minimize_to_tray)
        
        # 关闭时确认
        self.confirm_on_close = QCheckBox("关闭时确认")
        window_layout.addRow("", self.confirm_on_close)
        
        window_group.setLayout(window_layout)
        content_layout.addWidget(window_group)
        
        # 编辑器设置组
        editor_group = QGroupBox("编辑器行为")
        editor_layout = QFormLayout()
        
        # 自动换行
        self.auto_wrap_checkbox = QCheckBox("自动换行")
        editor_layout.addRow("", self.auto_wrap_checkbox)
        
        # 显示行号
        self.show_line_numbers = QCheckBox("显示行号")
        self.show_line_numbers.setChecked(True)
        editor_layout.addRow("", self.show_line_numbers)
        
        # 高亮当前行
        self.highlight_current_line = QCheckBox("高亮当前行")
        self.highlight_current_line.setChecked(True)
        editor_layout.addRow("", self.highlight_current_line)
        
        # Tab键空格数
        self.tab_size = QSpinBox()
        self.tab_size.setRange(1, 8)
        self.tab_size.setValue(4)
        editor_layout.addRow("Tab键空格数:", self.tab_size)
        
        # 使用空格代替Tab
        self.use_spaces_for_tab = QCheckBox("使用空格代替Tab")
        self.use_spaces_for_tab.setChecked(True)
        editor_layout.addRow("", self.use_spaces_for_tab)
        
        editor_group.setLayout(editor_layout)
        content_layout.addWidget(editor_group)
        
        # 快捷键设置组
        shortcuts_group = QGroupBox("快捷键设置")
        shortcuts_layout = QFormLayout()
        
        # 快捷键列表（可以扩展）
        self.shortcuts_label = QLabel("快捷键设置功能待实现\n（后续版本将支持自定义快捷键）")
        shortcuts_layout.addRow("", self.shortcuts_label)
        
        shortcuts_group.setLayout(shortcuts_layout)
        content_layout.addWidget(shortcuts_group)
        
        # 其他设置组
        other_group = QGroupBox("其他设置")
        other_layout = QFormLayout()
        
        # 启用调试模式
        self.debug_mode_checkbox = QCheckBox("启用调试模式")
        other_layout.addRow("", self.debug_mode_checkbox)
        
        # 日志级别
        self.log_level_combo = QComboBox()
        self.log_level_combo.addItems(["DEBUG", "INFO", "WARNING", "ERROR"])
        self.log_level_combo.setCurrentText("INFO")
        other_layout.addRow("日志级别:", self.log_level_combo)
        
        # 检查更新
        self.check_updates_checkbox = QCheckBox("启动时检查更新")
        self.check_updates_checkbox.setChecked(True)
        other_layout.addRow("", self.check_updates_checkbox)
        
        other_group.setLayout(other_layout)
        content_layout.addWidget(other_group)
        
        content_layout.addStretch()
        scroll.setWidget(content)
        layout.addWidget(scroll)
        
        return page
    
    def _create_theme_settings_page(self) -> QWidget:
        """创建主题设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        content = QWidget()
        content_layout = QVBoxLayout(content)
        
        # 软件主题设置
        app_theme_group = QGroupBox("软件主题")
        app_theme_layout = QFormLayout()
        
        # 主题列表
        self.theme_combo = QComboBox()
        self._load_themes()
        app_theme_layout.addRow("主题:", self.theme_combo)
        
        app_theme_group.setLayout(app_theme_layout)
        content_layout.addWidget(app_theme_group)
        
        # 代码编辑器设置
        editor_group = QGroupBox("代码编辑器设置")
        editor_layout = QFormLayout()
        
        # 编辑器主题
        self.editor_theme_combo = QComboBox()
        self.editor_theme_combo.addItems(["vs", "vs-dark", "hc-black"])
        editor_layout.addRow("编辑器主题:", self.editor_theme_combo)
        
        # 语言风格
        self.editor_language_combo = QComboBox()
        self.editor_language_combo.addItems(["python", "javascript", "typescript", "html", "css", "json"])
        editor_layout.addRow("语言风格:", self.editor_language_combo)
        
        # 字体大小
        self.editor_font_size = QSpinBox()
        self.editor_font_size.setRange(8, 72)
        self.editor_font_size.setValue(14)
        editor_layout.addRow("字体大小:", self.editor_font_size)
        
        # 字体
        self.editor_font_combo = QComboBox()
        self.editor_font_combo.addItems(["Consolas", "Courier New", "Monaco", "Menlo", "Source Code Pro", "Fira Code"])
        editor_layout.addRow("字体:", self.editor_font_combo)
        
        editor_group.setLayout(editor_layout)
        content_layout.addWidget(editor_group)
        
        # 图标设置
        icon_group = QGroupBox("图标设置")
        icon_layout = QFormLayout()
        
        self.icon_path_edit = QLineEdit()
        self.icon_path_edit.setText(r".\ui\icon")
        icon_browse_btn = QPushButton("浏览...")
        icon_browse_btn.clicked.connect(self._browse_icon_path)
        icon_path_layout = QHBoxLayout()
        icon_path_layout.addWidget(self.icon_path_edit)
        icon_path_layout.addWidget(icon_browse_btn)
        icon_layout.addRow("图标目录:", icon_path_layout)
        
        # 图标大小设置
        self.icon_size_spinbox = QSpinBox()
        self.icon_size_spinbox.setRange(32, 200)  # 允许32到200像素
        self.icon_size_spinbox.setValue(64)  # 默认值64
        self.icon_size_spinbox.setSuffix(" 像素")
        icon_layout.addRow("图标大小:", self.icon_size_spinbox)
        
        # 图标颜色设置
        icon_color_layout = QHBoxLayout()
        self.icon_color_combo = QComboBox()
        self.icon_color_combo.setEditable(True)  # 允许自定义输入
        self._load_icon_colors()
        icon_color_layout.addWidget(self.icon_color_combo)
        
        # 刷新按钮，重新加载图标颜色列表
        refresh_icon_colors_btn = QPushButton("刷新")
        refresh_icon_colors_btn.setToolTip("重新扫描图标目录，更新可用颜色列表")
        refresh_icon_colors_btn.clicked.connect(self._refresh_icon_colors)
        icon_color_layout.addWidget(refresh_icon_colors_btn)
        
        icon_layout.addRow("图标颜色:", icon_color_layout)
        
        icon_group.setLayout(icon_layout)
        content_layout.addWidget(icon_group)
        
        # 自定义颜色主题设置
        custom_color_group = QGroupBox("自定义颜色主题")
        custom_color_layout = QFormLayout()
        
        # 颜色名称输入
        self.custom_color_name_edit = QLineEdit()
        self.custom_color_name_edit.setPlaceholderText("例如: purple, orange, cyan")
        custom_color_layout.addRow("颜色名称:", self.custom_color_name_edit)
        
        # 颜色选择
        color_select_layout = QHBoxLayout()
        self.custom_color_preview = QLabel()
        self.custom_color_preview.setFixedSize(50, 30)
        self.custom_color_preview.setStyleSheet("background-color: #ffffff; border: 1px solid #cccccc;")
        self.custom_color_preview.setText("预览")
        self.custom_color_preview.setAlignment(Qt.AlignCenter)
        
        self.custom_color_btn = QPushButton("选择颜色...")
        self.custom_color_btn.clicked.connect(self._select_custom_color)
        color_select_layout.addWidget(self.custom_color_preview)
        color_select_layout.addWidget(self.custom_color_btn)
        custom_color_layout.addRow("颜色:", color_select_layout)
        
        # RGB值显示和编辑
        rgb_layout = QHBoxLayout()
        self.custom_color_r = QSpinBox()
        self.custom_color_r.setRange(0, 255)
        self.custom_color_r.setValue(255)
        self.custom_color_g = QSpinBox()
        self.custom_color_g.setRange(0, 255)
        self.custom_color_g.setValue(0)
        self.custom_color_b = QSpinBox()
        self.custom_color_b.setRange(0, 255)
        self.custom_color_b.setValue(0)
        rgb_layout.addWidget(QLabel("R:"))
        rgb_layout.addWidget(self.custom_color_r)
        rgb_layout.addWidget(QLabel("G:"))
        rgb_layout.addWidget(self.custom_color_g)
        rgb_layout.addWidget(QLabel("B:"))
        rgb_layout.addWidget(self.custom_color_b)
        custom_color_layout.addRow("RGB值:", rgb_layout)
        
        # 连接RGB值变化事件，更新预览
        self.custom_color_r.valueChanged.connect(self._update_color_preview)
        self.custom_color_g.valueChanged.connect(self._update_color_preview)
        self.custom_color_b.valueChanged.connect(self._update_color_preview)
        
        # 创建主题按钮
        self.create_custom_theme_btn = QPushButton("创建自定义颜色主题")
        self.create_custom_theme_btn.clicked.connect(self._create_custom_color_theme)
        custom_color_layout.addRow("", self.create_custom_theme_btn)
        
        custom_color_group.setLayout(custom_color_layout)
        content_layout.addWidget(custom_color_group)
        
        content_layout.addStretch()
        scroll.setWidget(content)
        layout.addWidget(scroll)
        
        return page
    
    def _create_page_settings_page(self) -> QWidget:
        """创建页面设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        content = QWidget()
        content_layout = QVBoxLayout(content)
        
        # 画布设置组
        canvas_group = QGroupBox("画布设置")
        canvas_layout = QFormLayout()
        
        # 默认画布大小
        canvas_size_layout = QHBoxLayout()
        self.canvas_width = QSpinBox()
        self.canvas_width.setRange(100, 5000)
        self.canvas_width.setValue(800)
        self.canvas_height = QSpinBox()
        self.canvas_height.setRange(100, 5000)
        self.canvas_height.setValue(600)
        canvas_size_layout.addWidget(self.canvas_width)
        canvas_size_layout.addWidget(QLabel(" x "))
        canvas_size_layout.addWidget(self.canvas_height)
        canvas_size_layout.addWidget(QLabel(" 像素"))
        canvas_layout.addRow("默认画布大小:", canvas_size_layout)
        
        # 显示网格
        self.show_grid_checkbox = QCheckBox("显示网格")
        self.show_grid_checkbox.setChecked(True)
        canvas_layout.addRow("", self.show_grid_checkbox)
        
        # 网格大小
        self.grid_size = QSpinBox()
        self.grid_size.setRange(5, 50)
        self.grid_size.setValue(10)
        canvas_layout.addRow("网格大小:", self.grid_size)
        
        # 网格对齐
        self.snap_to_grid_checkbox = QCheckBox("启用网格对齐")
        self.snap_to_grid_checkbox.setChecked(True)
        canvas_layout.addRow("", self.snap_to_grid_checkbox)
        
        # 对齐距离
        self.snap_distance = QSpinBox()
        self.snap_distance.setRange(1, 20)
        self.snap_distance.setValue(5)
        canvas_layout.addRow("对齐距离:", self.snap_distance)
        
        # 显示标尺
        self.show_ruler_checkbox = QCheckBox("显示标尺")
        canvas_layout.addRow("", self.show_ruler_checkbox)
        
        canvas_group.setLayout(canvas_layout)
        content_layout.addWidget(canvas_group)
        
        # 预览设置组
        preview_group = QGroupBox("预览设置")
        preview_layout = QFormLayout()
        
        # 预览窗口大小
        preview_size_layout = QHBoxLayout()
        self.preview_width = QSpinBox()
        self.preview_width.setRange(200, 2000)
        self.preview_width.setValue(1024)
        self.preview_height = QSpinBox()
        self.preview_height.setRange(200, 2000)
        self.preview_height.setValue(768)
        preview_size_layout.addWidget(self.preview_width)
        preview_size_layout.addWidget(QLabel(" x "))
        preview_size_layout.addWidget(self.preview_height)
        preview_size_layout.addWidget(QLabel(" 像素"))
        preview_layout.addRow("预览窗口大小:", preview_size_layout)
        
        # 自动刷新预览
        self.auto_refresh_preview = QCheckBox("自动刷新预览")
        self.auto_refresh_preview.setChecked(True)
        preview_layout.addRow("", self.auto_refresh_preview)
        
        # 预览刷新延迟（毫秒）
        self.preview_refresh_delay = QSpinBox()
        self.preview_refresh_delay.setRange(100, 5000)
        self.preview_refresh_delay.setValue(500)
        self.preview_refresh_delay.setSuffix(" 毫秒")
        preview_layout.addRow("刷新延迟:", self.preview_refresh_delay)
        
        preview_group.setLayout(preview_layout)
        content_layout.addWidget(preview_group)
        
        # 代码生成设置组
        code_gen_group = QGroupBox("代码生成设置")
        code_gen_layout = QFormLayout()
        
        # 代码风格
        self.code_style_combo = QComboBox()
        self.code_style_combo.addItems(["PEP 8", "Google", "自定义"])
        code_gen_layout.addRow("代码风格:", self.code_style_combo)
        
        # 自动格式化
        self.auto_format_code = QCheckBox("自动格式化代码")
        self.auto_format_code.setChecked(True)
        code_gen_layout.addRow("", self.auto_format_code)
        
        # 生成注释
        self.generate_comments = QCheckBox("生成代码注释")
        self.generate_comments.setChecked(True)
        code_gen_layout.addRow("", self.generate_comments)
        
        # 生成类型提示
        self.generate_type_hints = QCheckBox("生成类型提示")
        code_gen_layout.addRow("", self.generate_type_hints)
        
        code_gen_group.setLayout(code_gen_layout)
        content_layout.addWidget(code_gen_group)
        
        content_layout.addStretch()
        scroll.setWidget(content)
        layout.addWidget(scroll)
        
        return page
    
    def _create_package_settings_page(self) -> QWidget:
        """创建打包设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        label = QLabel("打包设置功能待实现")
        label.setAlignment(Qt.AlignCenter)
        layout.addWidget(label)
        
        return page
    
    def _create_fnpy_libraries_settings_page(self) -> QWidget:
        """创建函数库设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        content = QWidget()
        content_layout = QVBoxLayout(content)
        
        # 函数库设置组
        libraries_group = QGroupBox("函数库设置")
        libraries_layout = QVBoxLayout()
        
        # 说明标签
        info_label = QLabel("选择要加载的函数库。选中的函数库将在运行代码时自动包含。")
        info_label.setWordWrap(True)
        libraries_layout.addWidget(info_label)
        
        # 函数库列表
        try:
            from pathlib import Path
            import sys
            sys.path.insert(0, str(Path(__file__).parent.parent))
            from lib.fnpy_library_manager import fnpyLibraryManager
            
            self.fnpy_library_manager = fnpyLibraryManager()
            library_names = self.fnpy_library_manager.scan_libraries()
            
            # 创建复选框列表
            self.fnpy_library_checkboxes = {}
            for library_name in library_names:
                # 加载函数库信息以获取中文名称
                library_info = self.fnpy_library_manager.load_library(library_name)
                if library_info:
                    # 显示中文名（英文名）
                    display_name = f"{library_info.chinese_name} ({library_info.name})"
                    checkbox = QCheckBox(display_name)
                    checkbox.setChecked(library_info.enabled)
                else:
                    # 如果没有信息，只显示文件名（去掉.fnpy后缀）
                    display_name = library_name.replace('.fnpy', '')
                    checkbox = QCheckBox(display_name)
                    checkbox.setChecked(False)
                self.fnpy_library_checkboxes[library_name] = checkbox
                libraries_layout.addWidget(checkbox)
        except Exception as e:
            error_label = QLabel(f"加载函数库列表失败: {e}")
            error_label.setStyleSheet("color: red;")
            libraries_layout.addWidget(error_label)
            self.fnpy_library_checkboxes = {}
        
        libraries_group.setLayout(libraries_layout)
        content_layout.addWidget(libraries_group)
        
        content_layout.addStretch()
        
        scroll.setWidget(content)
        layout.addWidget(scroll)
        
        return page
    
    def _create_controls_settings_page(self) -> QWidget:
        """创建控件设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        # 添加说明标签
        info_label = QLabel("选择要禁用的控件（取消勾选将不加载）:")
        layout.addWidget(info_label)
        
        # 控件列表 - 使用滚动区域包装，避免重叠
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        # 控件列表
        self.controls_list = QListWidget()
        self.controls_list.setSelectionMode(QListWidget.MultiSelection)
        # 设置列表项间距，避免重叠
        self.controls_list.setSpacing(2)
        # 设置统一的项高度，确保不重叠
        self.controls_list.setUniformItemSizes(True)
        
        # 加载所有控件
        self._load_controls()
        
        # 将列表放入滚动区域
        scroll.setWidget(self.controls_list)
        layout.addWidget(scroll, 1)  # 使用拉伸因子1，让列表占据剩余空间
        
        return page
    
    def _create_python_settings_page(self) -> QWidget:
        """创建Python环境设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        content = QWidget()
        content_layout = QVBoxLayout(content)
        
        # Python环境选择
        env_group = QGroupBox("Python环境")
        env_layout = QFormLayout()
        
        self.python_env_type = QComboBox()
        self.python_env_type.addItems(["环境变量", "内置环境"])
        self.python_env_type.currentIndexChanged.connect(self._on_python_env_type_changed)
        env_layout.addRow("环境类型:", self.python_env_type)
        
        # 内置环境列表
        self.builtin_python_combo = QComboBox()
        self.builtin_python_combo.currentIndexChanged.connect(self._update_python_path)
        env_layout.addRow("内置Python版本:", self.builtin_python_combo)
        
        # Python路径显示
        self.python_path_label = QLabel("")
        env_layout.addRow("Python路径:", self.python_path_label)
        
        # 在创建完所有控件后再加载Python版本和更新路径
        self._load_builtin_python_versions()
        
        env_group.setLayout(env_layout)
        content_layout.addWidget(env_group)
        
        # Pip设置
        pip_group = QGroupBox("Pip设置")
        pip_layout = QFormLayout()
        
        self.auto_pip_checkbox = QCheckBox("自动生成pip命令")
        pip_layout.addRow("", self.auto_pip_checkbox)
        
        self.pip_path_edit = QLineEdit()
        self.pip_path_edit.setEnabled(False)
        pip_browse_btn = QPushButton("浏览...")
        pip_browse_btn.setEnabled(False)
        pip_browse_btn.clicked.connect(self._browse_pip_path)
        pip_path_layout = QHBoxLayout()
        pip_path_layout.addWidget(self.pip_path_edit)
        pip_path_layout.addWidget(pip_browse_btn)
        pip_layout.addRow("Pip路径:", pip_path_layout)
        
        self.auto_pip_checkbox.toggled.connect(lambda checked: (
            self.pip_path_edit.setEnabled(checked and self.python_env_type.currentIndex() == 1),
            pip_browse_btn.setEnabled(checked and self.python_env_type.currentIndex() == 1)
        ))
        
        # Pip下载源设置
        self.pip_index_url_combo = QComboBox()
        self.pip_index_url_combo.setEditable(True)
        self.pip_index_url_combo.addItems([
            "https://pypi.org/simple",
            "https://pypi.tuna.tsinghua.edu.cn/simple",
            "https://mirrors.aliyun.com/pypi/simple",
            "https://pypi.douban.com/simple",
            "https://mirrors.cloud.tencent.com/pypi/simple",
            "https://mirrors.bfsu.edu.cn/pypi/web/simple"
        ])
        self.pip_index_url_combo.setCurrentText("https://pypi.org/simple")
        pip_layout.addRow("Pip下载源:", self.pip_index_url_combo)
        
        # 使用国内镜像加速
        self.use_pip_mirror = QCheckBox("使用国内镜像加速（推荐）")
        self.use_pip_mirror.setChecked(False)
        self.use_pip_mirror.toggled.connect(self._on_pip_mirror_toggled)
        pip_layout.addRow("", self.use_pip_mirror)
        
        # 信任主机设置
        self.pip_trust_host_edit = QLineEdit()
        self.pip_trust_host_edit.setPlaceholderText("例如: pypi.org, pypi.tuna.tsinghua.edu.cn")
        pip_layout.addRow("信任主机:", self.pip_trust_host_edit)
        
        # 超时设置
        self.pip_timeout = QSpinBox()
        self.pip_timeout.setRange(10, 600)
        self.pip_timeout.setValue(60)
        self.pip_timeout.setSuffix(" 秒")
        pip_layout.addRow("超时时间:", self.pip_timeout)
        
        pip_group.setLayout(pip_layout)
        content_layout.addWidget(pip_group)
        
        content_layout.addStretch()
        scroll.setWidget(content)
        layout.addWidget(scroll)
        
        return page
    
    def _create_ai_settings_page(self) -> QWidget:
        """创建AI设置页面"""
        page = QWidget()
        layout = QVBoxLayout(page)
        layout.setContentsMargins(20, 20, 20, 20)
        
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.NoFrame)
        
        content = QWidget()
        content_layout = QVBoxLayout(content)
        
        # API设置
        api_group = QGroupBox("API设置")
        api_layout = QFormLayout()
        
        # API Key
        self.ai_api_key_edit = QLineEdit()
        self.ai_api_key_edit.setEchoMode(QLineEdit.Password)
        api_layout.addRow("API Key:", self.ai_api_key_edit)
        
        # API URL
        self.ai_api_url_edit = QLineEdit()
        self.ai_api_url_edit.setText("https://api.deepseek.com/v1/chat/completions")
        api_layout.addRow("API URL:", self.ai_api_url_edit)
        
        # 服务商
        self.ai_provider_combo = QComboBox()
        self.ai_provider_combo.addItems(["DeepSeek", "GLM"])
        self.ai_provider_combo.currentTextChanged.connect(self._on_ai_provider_changed)
        api_layout.addRow("服务商:", self.ai_provider_combo)
        
        # 模型
        self.ai_model_combo = QComboBox()
        self.ai_model_combo.addItems(["deepseek-chat", "deepseek-coder"])
        api_layout.addRow("模型:", self.ai_model_combo)
        
        api_group.setLayout(api_layout)
        content_layout.addWidget(api_group)
        
        content_layout.addStretch()
        scroll.setWidget(content)
        layout.addWidget(scroll)
        
        return page
    
    def _load_themes(self):
        """加载主题列表"""
        self.theme_combo.clear()
        if hasattr(self, 'theme_manager'):
            themes = self.theme_manager.get_available_themes()
            self.theme_combo.addItems(themes)
        else:
            theme_dir = Path(__file__).parent.parent / "ui" / "color"
            if theme_dir.exists():
                themes = []
                for file in theme_dir.glob("*.qss"):
                    themes.append(file.stem)
                self.theme_combo.addItems(sorted(themes))
    
    def _load_icon_colors(self):
        """加载图标颜色列表"""
        if hasattr(self, 'icon_color_combo'):
            current_text = self.icon_color_combo.currentText()  # 保存当前选择
            self.icon_color_combo.clear()
            
            if hasattr(self, 'theme_manager'):
                colors = self.theme_manager.get_available_icon_colors()
                self.icon_color_combo.addItems(colors)
            else:
                # 如果没有主题管理器，直接扫描目录
                icon_dir = Path(__file__).parent.parent / "ui" / "icon"
                if icon_dir.exists():
                    colors = []
                    for item in icon_dir.iterdir():
                        if item.is_dir() and not item.name.startswith('.'):
                            colors.append(item.name)
                    self.icon_color_combo.addItems(sorted(colors))
            
            # 恢复之前的选择
            if current_text:
                index = self.icon_color_combo.findText(current_text)
                if index >= 0:
                    self.icon_color_combo.setCurrentIndex(index)
                else:
                    self.icon_color_combo.setCurrentText(current_text)
    
    def _refresh_icon_colors(self):
        """刷新图标颜色列表"""
        self._load_icon_colors()
        QMessageBox.information(self, "提示", "图标颜色列表已刷新")
    
    def _load_controls(self):
        """加载控件列表"""
        try:
            # 清空列表，避免重复加载
            self.controls_list.clear()
            
            categories = ControlLibrary.get_all_categories()
            for cat_id, cat_name in categories:
                controls = ControlLibrary.get_controls_by_category(cat_id)
                for control_type, control_info in controls:
                    display_name = control_info.get('display_name', control_type)
                    item = QListWidgetItem(f"{cat_name} - {display_name}")
                    item.setData(Qt.UserRole, control_type)
                    item.setCheckState(Qt.Checked)
                    # 设置项高度，确保不重叠
                    item.setSizeHint(item.sizeHint())
                    self.controls_list.addItem(item)
            
            # 调整列表项高度，确保有足够的间距
            for i in range(self.controls_list.count()):
                item = self.controls_list.item(i)
                if item:
                    size = item.sizeHint()
                    if size.height() < 24:  # 如果高度太小，设置为24
                        item.setSizeHint(QSize(item.sizeHint().width(), 24))
        except Exception as e:
            print(f"加载控件列表失败: {e}")
    
    def _load_builtin_python_versions(self):
        """加载内置Python版本"""
        python_dir = Path(__file__).parent.parent / "Python"
        if python_dir.exists():
            versions = []
            for item in python_dir.iterdir():
                if item.is_dir() and not item.name.startswith('.'):
                    versions.append(item.name)
            self.builtin_python_combo.addItems(sorted(versions))
        # 只有在 python_path_label 已创建时才更新路径
        if hasattr(self, 'python_path_label'):
            self._update_python_path()
    
    def _on_tree_item_changed(self, current, previous):
        """树形框项目改变"""
        if current:
            key = current.data(0, Qt.UserRole)
            if key in self.pages:
                self.stacked_widget.setCurrentWidget(self.pages[key])
    
    def _on_python_env_type_changed(self, index):
        """Python环境类型改变"""
        self.builtin_python_combo.setEnabled(index == 1)
        self._update_python_path()
        
        # 更新pip路径编辑框状态
        is_builtin = index == 1
        auto_pip = self.auto_pip_checkbox.isChecked()
        self.pip_path_edit.setEnabled(is_builtin and auto_pip)
        # 找到pip浏览按钮并更新状态
        pip_browse_btn = None
        for widget in self.pages["python"].findChildren(QPushButton):
            if widget.text() == "浏览..." and widget != self.pages["python"].findChild(QPushButton, "icon_browse"):
                pip_browse_btn = widget
                break
        if pip_browse_btn:
            pip_browse_btn.setEnabled(is_builtin and auto_pip)
    
    def _update_python_path(self):
        """更新Python路径显示"""
        if self.python_env_type.currentIndex() == 0:
            # 环境变量
            import sys
            self.python_path_label.setText(sys.executable)
        else:
            # 内置环境
            version = self.builtin_python_combo.currentText()
            if version:
                python_dir = Path(__file__).parent.parent / "Python" / version
                python_exe = python_dir / "python.exe"
                if python_exe.exists():
                    self.python_path_label.setText(str(python_exe))
                else:
                    self.python_path_label.setText(f"未找到: {python_exe}")
            else:
                self.python_path_label.setText("请选择Python版本")
    
    def _on_ai_provider_changed(self, provider):
        """AI服务商改变"""
        if provider == "DeepSeek":
            self.ai_api_url_edit.setText("https://api.deepseek.com/v1/chat/completions")
            self.ai_model_combo.clear()
            self.ai_model_combo.addItems(["deepseek-chat", "deepseek-coder"])
        else:  # GLM
            self.ai_api_url_edit.setText("https://open.bigmodel.cn/api/paas/v4/chat/completions")
            self.ai_model_combo.clear()
            self.ai_model_combo.addItems(["GLM-4.5-Flash", "glm-4-flash-250414"])
    
    def _on_pip_mirror_toggled(self, checked):
        """Pip镜像切换"""
        if checked:
            # 切换到清华镜像
            index = self.pip_index_url_combo.findText("https://pypi.tuna.tsinghua.edu.cn/simple")
            if index >= 0:
                self.pip_index_url_combo.setCurrentIndex(index)
            else:
                self.pip_index_url_combo.setCurrentText("https://pypi.tuna.tsinghua.edu.cn/simple")
    
    def _browse_icon_path(self):
        """浏览图标路径"""
        path = QFileDialog.getExistingDirectory(self, "选择图标目录", self.icon_path_edit.text())
        if path:
            self.icon_path_edit.setText(path)
    
    def _select_custom_color(self):
        """选择自定义颜色"""
        current_r = self.custom_color_r.value()
        current_g = self.custom_color_g.value()
        current_b = self.custom_color_b.value()
        color = QColorDialog.getColor(QColor(current_r, current_g, current_b), self, "选择颜色")
        if color.isValid():
            self.custom_color_r.setValue(color.red())
            self.custom_color_g.setValue(color.green())
            self.custom_color_b.setValue(color.blue())
            self._update_color_preview()
    
    def _update_color_preview(self):
        """更新颜色预览"""
        r = self.custom_color_r.value()
        g = self.custom_color_g.value()
        b = self.custom_color_b.value()
        color_str = f"rgb({r}, {g}, {b})"
        self.custom_color_preview.setStyleSheet(
            f"background-color: {color_str}; "
            f"border: 1px solid #cccccc; "
            f"color: {'white' if (r + g + b) < 384 else 'black'};"
        )
        self.custom_color_preview.setText(f"#{r:02x}{g:02x}{b:02x}")
    
    def _create_custom_color_theme(self):
        """创建自定义颜色主题"""
        color_name = self.custom_color_name_edit.text().strip()
        if not color_name:
            QMessageBox.warning(self, "错误", "请输入颜色名称")
            return
        
        # 检查颜色名称是否有效（只允许字母、数字、下划线）
        if not color_name.replace('_', '').isalnum():
            QMessageBox.warning(self, "错误", "颜色名称只能包含字母、数字和下划线")
            return
        
        r = self.custom_color_r.value()
        g = self.custom_color_g.value()
        b = self.custom_color_b.value()
        
        # 检查主题是否已存在
        existing_themes = self.theme_manager.get_available_themes()
        if color_name in existing_themes:
            reply = QMessageBox.question(
                self, "确认", 
                f"主题 '{color_name}' 已存在，是否覆盖？",
                QMessageBox.Yes | QMessageBox.No
            )
            if reply != QMessageBox.Yes:
                return
        
        # 显示处理提示
        QMessageBox.information(
            self, "提示", 
            f"正在创建自定义颜色主题 '{color_name}'...\n"
            f"这可能需要几秒钟时间，请稍候。"
        )
        
        # 创建主题
        success = self.theme_manager.create_custom_color_theme(color_name, (r, g, b))
        
        if success:
            # 刷新主题列表
            self._load_themes()
            # 选择新创建的主题
            index = self.theme_combo.findText(color_name)
            if index >= 0:
                self.theme_combo.setCurrentIndex(index)
            
            QMessageBox.information(
                self, "成功", 
                f"自定义颜色主题 '{color_name}' 创建成功！\n"
                f"主题文件和图标已生成。"
            )
        else:
            QMessageBox.warning(
                self, "错误", 
                f"创建自定义颜色主题 '{color_name}' 失败。\n"
                f"请检查控制台输出以获取详细信息。"
            )
    
    def _browse_pip_path(self):
        """浏览pip路径"""
        path, _ = QFileDialog.getOpenFileName(self, "选择pip可执行文件", self.pip_path_edit.text(), "可执行文件 (*.exe);;所有文件 (*)")
        if path:
            self.pip_path_edit.setText(path)
    
    def _load_all_settings(self):
        """加载所有设置"""
        # 基础设置
        self.auto_save_checkbox.setChecked(self.settings.value("basic/auto_save", False, type=bool))
        self.auto_save_interval.setValue(self.settings.value("basic/auto_save_interval", 300, type=int))
        startup_behavior = self.settings.value("basic/startup_behavior", 0, type=int)
        self.startup_behavior_combo.setCurrentIndex(startup_behavior)
        self.recent_files_count.setValue(self.settings.value("basic/recent_files_count", 10, type=int))
        
        self.remember_window_state.setChecked(self.settings.value("basic/remember_window_state", True, type=bool))
        self.minimize_to_tray.setChecked(self.settings.value("basic/minimize_to_tray", False, type=bool))
        self.confirm_on_close.setChecked(self.settings.value("basic/confirm_on_close", False, type=bool))
        
        self.auto_wrap_checkbox.setChecked(self.settings.value("basic/auto_wrap", False, type=bool))
        self.show_line_numbers.setChecked(self.settings.value("basic/show_line_numbers", True, type=bool))
        self.highlight_current_line.setChecked(self.settings.value("basic/highlight_current_line", True, type=bool))
        self.tab_size.setValue(self.settings.value("basic/tab_size", 4, type=int))
        self.use_spaces_for_tab.setChecked(self.settings.value("basic/use_spaces_for_tab", True, type=bool))
        
        self.debug_mode_checkbox.setChecked(self.settings.value("basic/debug_mode", False, type=bool))
        log_level = self.settings.value("basic/log_level", "INFO", type=str)
        index = self.log_level_combo.findText(log_level)
        if index >= 0:
            self.log_level_combo.setCurrentIndex(index)
        self.check_updates_checkbox.setChecked(self.settings.value("basic/check_updates", True, type=bool))
        
        # 主题设置
        theme = self.settings.value("theme/name", "white", type=str)
        index = self.theme_combo.findText(theme)
        if index >= 0:
            self.theme_combo.setCurrentIndex(index)
        
        # 编辑器设置
        self.editor_theme_combo.setCurrentText(self.settings.value("editor/theme", "vs", type=str))
        self.editor_language_combo.setCurrentText(self.settings.value("editor/language", "python", type=str))
        self.editor_font_size.setValue(self.settings.value("editor/font_size", 14, type=int))
        self.editor_font_combo.setCurrentText(self.settings.value("editor/font", "Consolas", type=str))
        
        # 图标路径
        self.icon_path_edit.setText(self.settings.value("icon/path", r".\ui\icon", type=str))
        # 图标大小
        self.icon_size_spinbox.setValue(self.settings.value("icon/size", 64, type=int))
        # 图标颜色
        icon_color = self.settings.value("icon/color", "", type=str)
        if icon_color:
            index = self.icon_color_combo.findText(icon_color)
            if index >= 0:
                self.icon_color_combo.setCurrentIndex(index)
            else:
                # 如果颜色不在列表中，添加到可编辑框
                self.icon_color_combo.setCurrentText(icon_color)
        
        # 页面设置
        self.canvas_width.setValue(self.settings.value("page/canvas_width", 800, type=int))
        self.canvas_height.setValue(self.settings.value("page/canvas_height", 600, type=int))
        self.show_grid_checkbox.setChecked(self.settings.value("page/show_grid", True, type=bool))
        self.grid_size.setValue(self.settings.value("page/grid_size", 10, type=int))
        self.snap_to_grid_checkbox.setChecked(self.settings.value("page/snap_to_grid", True, type=bool))
        self.snap_distance.setValue(self.settings.value("page/snap_distance", 5, type=int))
        self.show_ruler_checkbox.setChecked(self.settings.value("page/show_ruler", False, type=bool))
        
        self.preview_width.setValue(self.settings.value("page/preview_width", 1024, type=int))
        self.preview_height.setValue(self.settings.value("page/preview_height", 768, type=int))
        self.auto_refresh_preview.setChecked(self.settings.value("page/auto_refresh_preview", True, type=bool))
        self.preview_refresh_delay.setValue(self.settings.value("page/preview_refresh_delay", 500, type=int))
        
        code_style = self.settings.value("page/code_style", "PEP 8", type=str)
        index = self.code_style_combo.findText(code_style)
        if index >= 0:
            self.code_style_combo.setCurrentIndex(index)
        self.auto_format_code.setChecked(self.settings.value("page/auto_format_code", True, type=bool))
        self.generate_comments.setChecked(self.settings.value("page/generate_comments", True, type=bool))
        self.generate_type_hints.setChecked(self.settings.value("page/generate_type_hints", False, type=bool))
        
        # Python环境设置
        env_type = self.settings.value("python/env_type", 0, type=int)
        self.python_env_type.setCurrentIndex(env_type)
        builtin_version = self.settings.value("python/builtin_version", "", type=str)
        if builtin_version:
            index = self.builtin_python_combo.findText(builtin_version)
            if index >= 0:
                self.builtin_python_combo.setCurrentIndex(index)
        self.auto_pip_checkbox.setChecked(self.settings.value("python/auto_pip", True, type=bool))
        self.pip_path_edit.setText(self.settings.value("python/pip_path", "", type=str))
        
        # Pip下载源设置
        pip_index_url = self.settings.value("python/pip_index_url", "https://pypi.org/simple", type=str)
        self.pip_index_url_combo.setCurrentText(pip_index_url)
        self.use_pip_mirror.setChecked(self.settings.value("python/use_pip_mirror", False, type=bool))
        self.pip_trust_host_edit.setText(self.settings.value("python/pip_trust_host", "", type=str))
        self.pip_timeout.setValue(self.settings.value("python/pip_timeout", 60, type=int))
        
        # AI设置
        self.ai_api_key_edit.setText(self.settings.value("ai/api_key", "", type=str))
        self.ai_api_url_edit.setText(self.settings.value("ai/api_url", "https://api.deepseek.com/v1/chat/completions", type=str))
        provider = self.settings.value("ai/provider", "DeepSeek", type=str)
        index = self.ai_provider_combo.findText(provider)
        if index >= 0:
            self.ai_provider_combo.setCurrentIndex(index)
        model = self.settings.value("ai/model", "deepseek-chat", type=str)
        index = self.ai_model_combo.findText(model)
        if index >= 0:
            self.ai_model_combo.setCurrentIndex(index)
        
        # 控件设置
        disabled_controls = self.settings.value("controls/disabled", [], type=list)
        if not disabled_controls:
            disabled_controls = []
        for i in range(self.controls_list.count()):
            item = self.controls_list.item(i)
            control_type = item.data(Qt.UserRole)
            item.setCheckState(Qt.Unchecked if control_type in disabled_controls else Qt.Checked)
        
        # 函数库设置
        try:
            if hasattr(self, 'fnpy_library_checkboxes') and hasattr(self, 'fnpy_library_manager'):
                enabled_libraries = self.settings.value("fnpy_libraries/enabled", [], type=list)
                if not enabled_libraries:
                    enabled_libraries = []
                for library_name, checkbox in self.fnpy_library_checkboxes.items():
                    checkbox.setChecked(library_name in enabled_libraries)
                    self.fnpy_library_manager.set_library_enabled(library_name, library_name in enabled_libraries)
        except Exception as e:
            print(f"加载函数库设置失败: {e}")
    
    def _save_all_settings(self):
        """保存所有设置"""
        # 基础设置
        self.settings.setValue("basic/auto_save", self.auto_save_checkbox.isChecked())
        self.settings.setValue("basic/auto_save_interval", self.auto_save_interval.value())
        self.settings.setValue("basic/startup_behavior", self.startup_behavior_combo.currentIndex())
        self.settings.setValue("basic/recent_files_count", self.recent_files_count.value())
        
        self.settings.setValue("basic/remember_window_state", self.remember_window_state.isChecked())
        self.settings.setValue("basic/minimize_to_tray", self.minimize_to_tray.isChecked())
        self.settings.setValue("basic/confirm_on_close", self.confirm_on_close.isChecked())
        
        self.settings.setValue("basic/auto_wrap", self.auto_wrap_checkbox.isChecked())
        self.settings.setValue("basic/show_line_numbers", self.show_line_numbers.isChecked())
        self.settings.setValue("basic/highlight_current_line", self.highlight_current_line.isChecked())
        self.settings.setValue("basic/tab_size", self.tab_size.value())
        self.settings.setValue("basic/use_spaces_for_tab", self.use_spaces_for_tab.isChecked())
        
        self.settings.setValue("basic/debug_mode", self.debug_mode_checkbox.isChecked())
        self.settings.setValue("basic/log_level", self.log_level_combo.currentText())
        self.settings.setValue("basic/check_updates", self.check_updates_checkbox.isChecked())
        
        # 主题设置
        self.settings.setValue("theme/name", self.theme_combo.currentText())
        
        # 编辑器设置
        self.settings.setValue("editor/theme", self.editor_theme_combo.currentText())
        self.settings.setValue("editor/language", self.editor_language_combo.currentText())
        self.settings.setValue("editor/font_size", self.editor_font_size.value())
        self.settings.setValue("editor/font", self.editor_font_combo.currentText())
        
        # 图标路径
        self.settings.setValue("icon/path", self.icon_path_edit.text())
        # 图标大小
        self.settings.setValue("icon/size", self.icon_size_spinbox.value())
        # 图标颜色
        icon_color = self.icon_color_combo.currentText().strip()
        if icon_color:
            self.settings.setValue("icon/color", icon_color)
        
        # 页面设置
        self.settings.setValue("page/canvas_width", self.canvas_width.value())
        self.settings.setValue("page/canvas_height", self.canvas_height.value())
        self.settings.setValue("page/show_grid", self.show_grid_checkbox.isChecked())
        self.settings.setValue("page/grid_size", self.grid_size.value())
        self.settings.setValue("page/snap_to_grid", self.snap_to_grid_checkbox.isChecked())
        self.settings.setValue("page/snap_distance", self.snap_distance.value())
        self.settings.setValue("page/show_ruler", self.show_ruler_checkbox.isChecked())
        
        self.settings.setValue("page/preview_width", self.preview_width.value())
        self.settings.setValue("page/preview_height", self.preview_height.value())
        self.settings.setValue("page/auto_refresh_preview", self.auto_refresh_preview.isChecked())
        self.settings.setValue("page/preview_refresh_delay", self.preview_refresh_delay.value())
        
        self.settings.setValue("page/code_style", self.code_style_combo.currentText())
        self.settings.setValue("page/auto_format_code", self.auto_format_code.isChecked())
        self.settings.setValue("page/generate_comments", self.generate_comments.isChecked())
        self.settings.setValue("page/generate_type_hints", self.generate_type_hints.isChecked())
        
        # Python环境设置
        self.settings.setValue("python/env_type", self.python_env_type.currentIndex())
        self.settings.setValue("python/builtin_version", self.builtin_python_combo.currentText())
        self.settings.setValue("python/auto_pip", self.auto_pip_checkbox.isChecked())
        self.settings.setValue("python/pip_path", self.pip_path_edit.text())
        
        # Pip下载源设置
        self.settings.setValue("python/pip_index_url", self.pip_index_url_combo.currentText())
        self.settings.setValue("python/use_pip_mirror", self.use_pip_mirror.isChecked())
        self.settings.setValue("python/pip_trust_host", self.pip_trust_host_edit.text())
        self.settings.setValue("python/pip_timeout", self.pip_timeout.value())
        
        # AI设置
        self.settings.setValue("ai/api_key", self.ai_api_key_edit.text())
        self.settings.setValue("ai/api_url", self.ai_api_url_edit.text())
        self.settings.setValue("ai/provider", self.ai_provider_combo.currentText())
        self.settings.setValue("ai/model", self.ai_model_combo.currentText())
        
        # 控件设置
        disabled_controls = []
        for i in range(self.controls_list.count()):
            item = self.controls_list.item(i)
            if item.checkState() == Qt.Unchecked:
                disabled_controls.append(item.data(Qt.UserRole))
        self.settings.setValue("controls/disabled", disabled_controls)
        
        # 函数库设置
        try:
            if hasattr(self, 'fnpy_library_checkboxes') and hasattr(self, 'fnpy_library_manager'):
                enabled_libraries = []
                for library_name, checkbox in self.fnpy_library_checkboxes.items():
                    enabled = checkbox.isChecked()
                    self.fnpy_library_manager.set_library_enabled(library_name, enabled)
                    if enabled:
                        enabled_libraries.append(library_name)
                self.settings.setValue("fnpy_libraries/enabled", enabled_libraries)
        except Exception as e:
            print(f"保存函数库设置失败: {e}")
        
        self.settings.sync()
        self.settings_changed.emit()
    
    def _on_ok_clicked(self):
        """确定按钮点击"""
        self._save_all_settings()
        self.accept()
    
    def _on_apply_clicked(self):
        """应用按钮点击"""
        self._save_all_settings()
        QMessageBox.information(self, "提示", "设置已保存")

