#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
属性编辑器模块
实现控件属性编辑功能，支持不同类型的属性编辑器
"""

"""
属性编辑器模块
实现控件属性编辑功能，支持不同类型的属性编辑器
提供直观的属性编辑界面，支持文本、数字、颜色、字体等多种属性类型
"""

# 导入类型提示
from typing import Any, Dict, List, Optional, Callable
# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QWidget,  # 基础窗口部件
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QLabel,  # 标签
    QLineEdit,  # 单行文本输入框
    QSpinBox,  # 整数旋转框
    QDoubleSpinBox,  # 浮点数旋转框
    QCheckBox,  # 复选框
    QComboBox,  # 下拉选择框
    QPushButton,  # 按钮
    QGroupBox,  # 分组框
    QScrollArea,  # 滚动区域
    QFrame,  # 框架
    QSlider,  # 滑动条
    QTextEdit,  # 富文本编辑器
    QColorDialog,  # 颜色选择对话框
    QFontDialog,  # 字体选择对话框
    QFileDialog,  # 文件选择对话框
    QTabWidget,  # 选项卡控件
    QSplitter,  # 分割器
    QTreeWidget,  # 树形控件
    QTreeWidgetItem,  # 树形控件项
    QFormLayout,  # 表单布局
    QSizePolicy,  # 大小策略
    QGridLayout,  # 网格布局
    QButtonGroup,  # 按钮组
    QRadioButton,  # 单选按钮
    QStackedWidget,  # 堆叠控件
    QPlainTextEdit,  # 纯文本编辑器
    QCompleter  # 自动补全
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    Signal,  # 信号定义
    Property,  # 属性
    QTimer,  # 定时器
    QSize,  # 大小对象
    QStringListModel  # 字符串列表模型
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QFont,  # 字体
    QColor,  # 颜色
    QIntValidator,  # 整数验证器
    QDoubleValidator,  # 浮点数验证器
    QPainter,  # 绘图器
    QTextCursor  # 文本光标
)

# 导入自定义模块
from .control_pyside import Control  # 控件类
from .control_library_pyside import ControlLibrary  # 控件库
from .monaco_editor_widget import MonacoEditorWidget  # Monaco Editor 代码编辑器
from .event_manager_pyside import EventManager  # 事件管理器


class PropertyEditor(QWidget):
    """
    属性编辑器主类
    提供可视化的属性编辑界面，支持编辑控件的各种属性
    当选中控件或画布时，显示对应的属性编辑器
    """
    
    # ========== 信号定义 ==========
    # 属性值改变时发出信号，参数为控件ID、属性名称、新属性值
    property_changed = Signal(str, str, object)
    # 事件绑定改变时发出信号，参数为控件ID、事件ID、是否绑定
    event_binding_changed = Signal(str, str, bool)
    
    def __init__(self, parent=None):
        """
        初始化属性编辑器
        
        Args:
            parent: 父窗口部件
        """
        # 调用父类构造函数
        super().__init__(parent)
        
        # 当前选中的控件对象（None表示未选中控件或选中画布）
        self.current_control = None
        
        # 标记是否为画布选中状态（True表示正在编辑画布属性）
        self.is_canvas = False
        
        # 属性编辑器缓存：存储已创建的属性编辑器控件，避免重复创建
        self.property_editors = {}
        
        # 初始化用户界面
        self._init_ui()
    
    def _init_ui(self):
        """
        初始化用户界面
        创建属性编辑器的所有UI组件，包括标题、控件选择器、滚动区域等
        """
        # ========== 创建主布局 ==========
        # 创建垂直布局作为属性编辑器的主布局
        layout = QVBoxLayout(self)
        # 设置布局边距：左、上、右、下各5像素
        layout.setContentsMargins(5, 5, 5, 5)

        # ========== 创建标题标签 ==========
        # 创建标题标签，显示"属性编辑器"
        title_label = QLabel("属性编辑器")
        # 设置标题样式：加粗、字号14px、内边距5px
        title_label.setStyleSheet("font-weight: bold; font-size: 14px; padding: 5px;")
        # 将标题标签添加到布局
        layout.addWidget(title_label)

        # ========== 创建控件选择器 ==========
        # 创建下拉选择框，用于选择要编辑的控件或画布
        self.control_selector = QComboBox()
        # 设置选择器为不可编辑（用户不能直接输入，只能从列表中选择）
        self.control_selector.setEditable(False)
        # ========== 连接选择变化信号 ==========
        # 连接选择变化信号，当用户选择不同控件时触发处理函数
        self.control_selector.currentIndexChanged.connect(self._on_select_control)
        # 将控件选择器添加到布局
        layout.addWidget(self.control_selector)
        
        # ========== 创建滚动区域 ==========
        # 创建滚动区域，用于包含属性编辑器容器（当属性较多时可以滚动查看）
        scroll_area = QScrollArea()
        # 设置滚动区域内的控件可以自动调整大小（适应滚动区域）
        scroll_area.setWidgetResizable(True)
        # 设置水平滚动条策略：需要时显示滚动条
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        # 设置垂直滚动条策略：需要时显示滚动条
        scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        
        # ========== 创建编辑器容器 ==========
        # 创建编辑器容器窗口部件，用于包含所有属性编辑器
        self.editor_container = QWidget()
        # 为编辑器容器创建垂直布局
        self.editor_layout = QVBoxLayout(self.editor_container)
        # 设置布局边距：左、上、右、下各5像素
        self.editor_layout.setContentsMargins(5, 5, 5, 5)
        # 设置布局中组件间距为5像素
        self.editor_layout.setSpacing(5)
        
        # ========== 设置容器样式 ==========
        # 设置编辑器容器的样式表（为复选框添加边框等样式）
        self.editor_container.setStyleSheet("QCheckBox { border: 1px solid #000; padding: 2px; border-radius: 2px; }")
        
        # ========== 创建提示标签 ==========
        # 创建提示标签，当没有选中控件时显示提示信息
        self.no_control_label = QLabel("请选择一个控件以编辑其属性")
        # 设置标签文本居中对齐
        self.no_control_label.setAlignment(Qt.AlignCenter)
        # 设置标签样式：灰色文字、内边距20px
        self.no_control_label.setStyleSheet("color: #888; padding: 20px;")
        # 将提示标签添加到编辑器布局
        self.editor_layout.addWidget(self.no_control_label)
        
        # ========== 将容器添加到滚动区域 ==========
        # 将编辑器容器设置为滚动区域的子部件
        scroll_area.setWidget(self.editor_container)
        # 将滚动区域添加到主布局
        layout.addWidget(scroll_area)

    def refresh_control_list(self, control_manager):
        """
        刷新控件选择器列表
        更新下拉框中的控件列表，显示所有可用的控件
        
        Args:
            control_manager: 控件管理器，包含所有控件的信息
        """
        # ========== 阻止信号触发 ==========
        # 在更新列表期间阻止信号触发，避免选择变化时触发不必要的事件
        self.control_selector.blockSignals(True)
        
        # ========== 清空现有列表 ==========
        # 清空控件选择器中的所有选项，准备重新填充
        self.control_selector.clear()
        
        # ========== 添加画布选项 ==========
        # 添加"画布"选项，用户可以选择编辑画布属性
        # 参数：显示文本"画布"，关联数据"canvas"
        self.control_selector.addItem("画布", "canvas")
        
        # ========== 添加控件选项 ==========
        try:
            # 尝试导入控件库，获取控件的显示名称
            from .control_library_pyside import ControlLibrary
            
            # ========== 遍历所有控件 ==========
            # 遍历控件管理器中的所有控件（控件ID和控件对象）
            for cid, control in control_manager.controls.items():
                # ========== 获取控件信息 ==========
                # 从控件库获取控件的详细信息（包括显示名称等）
                info = ControlLibrary.get_control_info(control.control_type)
                
                # ========== 获取显示名称 ==========
                # 从控件信息中获取显示名称，如果不存在则使用控件类型名称
                display_name = info.get("display_name", control.control_type)
                
                # ========== 构建显示文本 ==========
                # 构建控件选择器中的显示文本：显示名称 - 控件名称
                display = f"{display_name} - {control.name}"
                
                # ========== 添加到选择器 ==========
                # 将控件添加到下拉框，显示文本为display，关联数据为控件ID
                self.control_selector.addItem(display, cid)
        except Exception:
            # ========== 异常处理 ==========
            # 如果获取控件信息失败（如控件库不可用），使用简化方式显示控件
            # 遍历所有控件，使用控件类型和名称构建显示文本
            for cid, control in control_manager.controls.items():
                # 构建简化的显示文本：控件类型 - 控件名称
                display = f"{control.control_type} - {control.name}"
                # 将控件添加到下拉框
                self.control_selector.addItem(display, cid)
        
        # ========== 恢复信号触发 ==========
        # 更新完成后，恢复信号触发功能，允许选择变化时触发事件
        self.control_selector.blockSignals(False)

    def _on_select_control(self, index: int):
        """
        控件选择器选择变化事件处理函数
        当用户在控件选择器中选择不同的控件时调用，通知画布选中对应的控件
        
        Args:
            index: 选择器中的索引号（-1表示未选择）
        """
        # ========== 检查索引有效性 ==========
        # 检查选择的索引是否有效（索引小于0表示未选择或选择无效）
        if index < 0:
            # 如果索引无效，直接返回，不执行任何操作
            return
        
        # ========== 获取控件ID ==========
        # 从控件选择器中获取选中项关联的数据（控件ID或"canvas"）
        control_id = self.control_selector.itemData(index)
        
        # ========== 查找主窗口 ==========
        # 从属性编辑器的父窗口开始，向上查找主窗口（包含设计画布的窗口）
        main = self.parent()
        
        # ========== 向上遍历父窗口 ==========
        # 循环向上查找父窗口，直到找到包含设计画布的窗口
        while main:
            # ========== 检查是否有设计画布 ==========
            # 检查当前窗口是否有design_canvas属性（表示这是主窗口）
            if hasattr(main, 'design_canvas'):
                # ========== 选中控件 ==========
                # 如果找到了设计画布，通知画布选中对应的控件
                # 如果control_id为'canvas'，则选中画布；否则选中指定ID的控件
                main.design_canvas.select_control(control_id if control_id != 'canvas' else 'canvas')
                # 找到主窗口后，退出循环
                break
            # ========== 继续向上查找 ==========
            # 如果当前窗口不是主窗口，继续向上查找父窗口
            main = main.parent()
    
    def set_control(self, control: Optional[Control], is_canvas: bool = False):
        """设置要编辑的控件或画布
        
        Args:
            control: 要编辑的控件，如果是画布则为None
            is_canvas: 是否为画布选中状态
        """
        # 清空之前的编辑器
        self._clear_editors()
        
        self.current_control = control
        self.is_canvas = is_canvas
        
        # ========== 根据选择类型创建属性编辑器 ==========
        # 检查是否为画布选中状态
        if is_canvas:
            # ========== 创建画布属性编辑器 ==========
            # 如果选中了画布，创建画布属性编辑器（包含画布和窗口属性）
            self._create_canvas_property_editors()
            # ========== 隐藏提示标签 ==========
            # 隐藏"请选择一个控件"的提示标签，因为已经选中了画布
            self.no_control_label.hide()
        # ========== 检查是否选中了控件 ==========
        elif control:
            # ========== 创建控件属性编辑器 ==========
            # 如果选中了控件，创建控件的属性编辑器（包含控件所有可编辑属性）
            self._create_property_editors(control)
            # ========== 隐藏提示标签 ==========
            # 隐藏"请选择一个控件"的提示标签，因为已经选中了控件
            self.no_control_label.hide()
        else:
            # ========== 显示提示标签 ==========
            # 如果没有选中任何控件或画布，显示提示标签，提示用户选择一个控件
            self.no_control_label.show()
    
    def _clear_editors(self):
        """
        清空所有属性编辑器
        移除布局中的所有编辑器控件，为新的属性编辑器腾出空间
        """
        # ========== 清空布局中的所有控件 ==========
        # 循环移除布局中的所有子控件（除了提示标签）
        # 检查布局中是否还有子控件
        while self.editor_layout.count():
            # ========== 取出第一个子项 ==========
            # 从布局中取出第一个子项（布局项对象，包含控件信息）
            child = self.editor_layout.takeAt(0)
            
            # ========== 检查子项是否有控件 ==========
            # 检查子项是否包含控件，且该控件不是提示标签
            if child.widget() and child.widget() != self.no_control_label:
                # ========== 延迟删除控件 ==========
                # 如果子项包含控件且不是提示标签，延迟删除该控件
                # deleteLater()会在事件循环的下一次迭代中删除控件，避免立即删除导致的错误
                child.widget().deleteLater()
        
        # ========== 清空编辑器缓存 ==========
        # 清空属性编辑器字典，移除所有已缓存的编辑器引用
        self.property_editors.clear()
    
    def _create_property_editor(self, property_type: str, property_name: str, label_text: str, 
                               layout, default_value=None, items=None, min_val=None, max_val=None, options=None):
        """通用的属性编辑器创建方法
        
        Args:
            property_type: 属性类型 ('spinbox', 'checkbox', 'combobox', 'lineedit', 'textedit')
            property_name: 属性名称
            label_text: 标签文本
            layout: 布局对象
            default_value: 默认值
            items: 列表项（仅用于combobox）
            min_val: 最小值（仅用于spinbox）
            max_val: 最大值（仅用于spinbox）
            options: 额外选项（向后兼容）
            
        Returns:
            创建的编辑器控件
        """
        # ========== 初始化编辑器变量 ==========
        # 初始化编辑器变量为None，后续根据属性类型创建对应的编辑器
        editor = None
        
        # ========== 创建数字输入框编辑器 ==========
        # 检查属性类型是否为'spinbox'（整数输入框）
        if property_type == 'spinbox':
            # 创建整数旋转框控件
            editor = QSpinBox()
            
            # ========== 设置最小值 ==========
            # 检查是否提供了最小值
            if min_val is not None:
                # 如果提供了最小值，设置输入框的最小值限制
                editor.setMinimum(min_val)
            
            # ========== 设置最大值 ==========
            # 检查是否提供了最大值
            if max_val is not None:
                # 如果提供了最大值，设置输入框的最大值限制
                editor.setMaximum(max_val)
            
            # ========== 设置默认值 ==========
            # 检查是否提供了默认值
            if default_value is not None:
                # 如果提供了默认值，设置输入框的初始值
                editor.setValue(default_value)
            
            # ========== 连接值改变信号 ==========
            # 连接值改变信号，当用户修改值时触发属性改变事件
            editor.valueChanged.connect(lambda value: self._on_property_changed(property_name, value))
        
        # ========== 创建复选框编辑器 ==========
        # 检查属性类型是否为'checkbox'（复选框）
        elif property_type == 'checkbox':
            # 创建复选框控件
            editor = QCheckBox()
            
            # ========== 设置默认选中状态 ==========
            # 检查是否提供了默认值
            if default_value is not None:
                # 如果提供了默认值（布尔值），设置复选框的选中状态
                editor.setChecked(default_value)
            
            # ========== 连接状态改变信号 ==========
            # 连接状态改变信号，当用户点击复选框时触发属性改变事件
            # 将Qt.Checked状态转换为布尔值（True/False）
            editor.stateChanged.connect(lambda state: self._on_property_changed(property_name, state == Qt.Checked))
        
        # ========== 创建下拉选择框编辑器 ==========
        # 检查属性类型是否为'combobox'（下拉选择框）
        elif property_type == 'combobox':
            # 创建下拉选择框控件
            editor = QComboBox()
            
            # ========== 添加选项 ==========
            # 检查是否提供了选项列表
            if items:
                # 如果提供了选项列表，将所有选项添加到下拉框
                editor.addItems(items)
                
                # ========== 设置默认选中项 ==========
                # 检查是否提供了默认值且默认值在选项列表中
                if default_value is not None and default_value in items:
                    # 查找默认值在选项列表中的索引
                    index = editor.findText(default_value)
                    # ========== 检查索引有效性 ==========
                    # 检查是否找到了索引（索引>=0表示找到）
                    if index >= 0:
                        # 如果找到了索引，设置当前选中项为该索引
                        editor.setCurrentIndex(index)
            
            # ========== 连接文本改变信号 ==========
            # 连接当前文本改变信号，当用户选择不同选项时触发属性改变事件
            editor.currentTextChanged.connect(lambda text: self._on_property_changed(property_name, text))
        
        # ========== 创建单行文本输入框编辑器 ==========
        # 检查属性类型是否为'lineedit'（单行文本输入框）
        elif property_type == 'lineedit':
            # 创建单行文本输入框控件
            editor = QLineEdit()
            
            # ========== 设置默认文本 ==========
            # 检查是否提供了默认值
            if default_value is not None:
                # 如果提供了默认值，设置输入框的初始文本（转换为字符串）
                editor.setText(default_value)
            
            # ========== 连接文本改变信号 ==========
            # 连接文本改变信号，当用户输入文本时触发属性改变事件
            editor.textChanged.connect(lambda text: self._on_property_changed(property_name, text))
        
        # ========== 创建多行文本编辑器 ==========
        # 检查属性类型是否为'textedit'（多行文本编辑器）
        elif property_type == 'textedit':
            # 创建多行文本编辑器控件
            editor = QTextEdit()
            
            # ========== 设置最大高度 ==========
            # 设置编辑器的最大高度为60像素，避免占用过多空间
            editor.setMaximumHeight(60)
            
            # ========== 设置默认文本 ==========
            # 检查是否提供了默认值
            if default_value is not None:
                # 如果提供了默认值，设置编辑器的初始纯文本内容（转换为字符串）
                editor.setPlainText(default_value)
            
            # ========== 连接文本改变信号 ==========
            # 连接文本改变信号，当用户编辑文本时触发属性改变事件
            # 使用lambda函数获取编辑器的当前纯文本内容
            editor.textChanged.connect(lambda: self._on_property_changed(property_name, editor.toPlainText()))
        
        # ========== 将编辑器添加到布局 ==========
        # 检查编辑器是否创建成功
        if editor:
            # ========== 添加到表单布局 ==========
            # 将标签和编辑器添加到表单布局中（标签在左，编辑器在右）
            layout.addRow(label_text, editor)
            
            # ========== 缓存编辑器 ==========
            # 将编辑器保存到属性编辑器字典中，方便后续访问和更新
            self.property_editors[property_name] = editor
        
        # ========== 返回创建的编辑器 ==========
        # 返回创建的编辑器控件（如果没有创建则返回None）
        return editor
    
    def _create_canvas_property_editors(self):
        """创建画布属性编辑器"""
        # 获取画布当前属性
        from .design_canvas_pyside import DesignCanvas
        
        # 获取画布实例 - 通过主窗口获取
        canvas = None
        main_window = self.parent()
        while main_window:
            if hasattr(main_window, 'design_canvas'):
                canvas = main_window.design_canvas
                break
            main_window = main_window.parent()
        
        # 创建单一的画布属性选项卡
        canvas_tab = QWidget()
        canvas_layout = QFormLayout(canvas_tab)
        
        # === 画布属性部分（精简可用项） ===
        
        # 画布宽度
        canvas_width = canvas.get_canvas_width() if canvas else 380
        self._create_property_editor('spinbox', 'canvas_width', '画布宽度:', canvas_layout, 
                                    default_value=canvas_width, min_val=380, max_val=3840)
        
        # 画布高度
        canvas_height = canvas.get_canvas_height() if canvas else 300
        self._create_property_editor('spinbox', 'canvas_height', '画布高度:', canvas_layout, 
                                    default_value=canvas_height, min_val=300, max_val=2160)
        
        # 背景颜色
        bg_color_button = QPushButton("选择背景色")
        bg_color_button.clicked.connect(self._choose_canvas_bg_color)
        canvas_layout.addRow("背景颜色:", bg_color_button)
        
        # 当前背景颜色显示
        self.canvas_bg_color_label = QLabel("当前背景色: #FFFFFF")
        if canvas:
            self.canvas_bg_color_label.setText(f"当前背景色: {canvas.bg_color.name()}")
            self.canvas_bg_color_label.setStyleSheet(f"background-color: {canvas.bg_color.name()};")
        else:
            self.canvas_bg_color_label.setStyleSheet("background-color: #FFFFFF;")
        canvas_layout.addRow("", self.canvas_bg_color_label)
        
        # 网格显示
        show_grid = canvas.show_grid if canvas else True
        self._create_property_editor('checkbox', 'show_grid', '显示网格:', canvas_layout, 
                                    default_value=show_grid)
        
        # 网格大小
        grid_size = canvas.grid_size if canvas else 10
        self._create_property_editor('spinbox', 'grid_size', '网格大小:', canvas_layout, 
                                    default_value=grid_size, min_val=5, max_val=50)
        
        # 吸附网格
        snap_to_grid = canvas.snap_to_grid if canvas else True
        self._create_property_editor('checkbox', 'snap_to_grid', '吸附网格:', canvas_layout, 
                                    default_value=snap_to_grid)
        
        # === 窗口属性部分 ===
        
        # 窗口名称
        window_name = canvas.window_name if canvas else "启动窗口"
        self._create_property_editor('lineedit', 'window_name', '窗口名称:', canvas_layout, 
                                    default_value=window_name)
        
        # 窗口备注
        window_comment = canvas.window_comment if canvas else "在程序启动后自动调入本窗口"
        self._create_property_editor('textedit', 'window_comment', '窗口备注:', canvas_layout, 
                                    default_value=window_comment)
        
        # 窗口位置和尺寸
        window_left = canvas.window_left if canvas else 50
        self._create_property_editor('spinbox', 'window_left', '左边:', canvas_layout, 
                                    default_value=window_left, min_val=-32768, max_val=32767)
        
        window_top = canvas.window_top if canvas else 50
        self._create_property_editor('spinbox', 'window_top', '顶边:', canvas_layout, 
                                    default_value=window_top, min_val=-32768, max_val=32767)
        
        window_width = canvas.window_width if canvas else 380
        self._create_property_editor('spinbox', 'window_width', '窗口宽度:', canvas_layout, 
                                    default_value=window_width, min_val=0, max_val=32767)
        
        window_height = canvas.window_height if canvas else 250
        self._create_property_editor('spinbox', 'window_height', '窗口高度:', canvas_layout, 
                                    default_value=window_height, min_val=0, max_val=32767)
        
        # 窗口状态
        window_visible = canvas.window_visible if canvas else True
        self._create_property_editor('checkbox', 'window_visible', '可视:', canvas_layout, 
                                    default_value=window_visible)
        
        window_enabled = canvas.window_enabled if canvas else True
        self._create_property_editor('checkbox', 'window_enabled', '禁止:', canvas_layout, 
                                    default_value=window_enabled)
        
        # 窗口标题
        window_title = canvas.window_title if canvas else "启动窗口"
        self._create_property_editor('lineedit', 'window_title', '标题:', canvas_layout, 
                                    default_value=window_title)
        
        # 底色
        bg_color_items = ["默认底色", "黑色", "深灰", "灰色", "浅灰", "白色", "红色", "绿色", "蓝色"]
        window_bg_color = canvas.window_bg_color if canvas else "默认底色"
        self._create_property_editor('combobox', 'window_bg_color', '底色:', canvas_layout, 
                                    default_value=window_bg_color, items=bg_color_items)
        
        # 底图选择
        bg_img_button = QPushButton("选择底图")
        bg_img_button.clicked.connect(lambda: self._choose_file_for_canvas('window_bg_image', '选择图片', '图像文件 (*.png *.jpg *.jpeg *.bmp);;所有文件 (*)'))
        canvas_layout.addRow("底图:", bg_img_button)
        
        self.canvas_bg_image_label = QLabel(canvas.window_bg_image if canvas and canvas.window_bg_image else "未选择")
        canvas_layout.addRow("", self.canvas_bg_image_label)
        
        # 底图方式
        bg_image_mode_items = ["图片平铺", "图片居中", "图片拉伸", "图片缩放"]
        window_bg_image_mode = canvas.window_bg_image_mode if canvas else "图片平铺"
        self._create_property_editor('combobox', 'window_bg_image_mode', '底图方式:', canvas_layout, 
                                    default_value=window_bg_image_mode, items=bg_image_mode_items)
        
        # 控制按钮
        window_show_control_button = canvas.window_show_control_button if canvas else True
        self._create_property_editor('checkbox', 'window_show_control_button', '控制按钮:', canvas_layout, 
                                    default_value=window_show_control_button)
        
        # 最大化按钮
        window_show_maximize_button = canvas.window_show_maximize_button if canvas else False
        self._create_property_editor('checkbox', 'window_show_maximize_button', '最大化按钮:', canvas_layout, 
                                    default_value=window_show_maximize_button)
        
        # 最小化按钮
        window_show_minimize_button = canvas.window_show_minimize_button if canvas else True
        self._create_property_editor('checkbox', 'window_show_minimize_button', '最小化按钮:', canvas_layout, 
                                    default_value=window_show_minimize_button)
        
        # 图标选择
        icon_button = QPushButton("选择图标")
        icon_button.clicked.connect(lambda: self._choose_file_for_canvas('window_icon', '选择图标', '图像文件 (*.png *.ico *.jpg *.jpeg *.bmp);;所有文件 (*)'))
        canvas_layout.addRow("图标:", icon_button)
        self.canvas_icon_label = QLabel(canvas.window_icon if canvas and canvas.window_icon else "未选择")
        canvas_layout.addRow("", self.canvas_icon_label)
        
        # 回车下移焦点
        window_enter_move_focus = canvas.window_enter_move_focus if canvas else False
        self._create_property_editor('checkbox', 'window_enter_move_focus', '回车下移焦点:', canvas_layout,
                                    default_value=window_enter_move_focus)
        
        # Esc键关闭
        window_esc_close = canvas.window_esc_close if canvas else True
        self._create_property_editor('checkbox', 'window_esc_close', 'Esc键关闭:', canvas_layout,
                                    default_value=window_esc_close)
        
        # F1键打开帮助
        window_f1_help = canvas.window_f1_help if canvas else False
        self._create_property_editor('checkbox', 'window_f1_help', 'F1键打开帮助:', canvas_layout,
                                    default_value=window_f1_help)
        
        # 帮助文件名
        window_help_file = canvas.window_help_file if canvas else ""
        self._create_property_editor('lineedit', 'window_help_file', '帮助文件名:', canvas_layout,
                                    default_value=window_help_file)
        
        # 帮助标志值
        window_help_flag = canvas.window_help_flag if canvas else 0
        self._create_property_editor('spinbox', 'window_help_flag', '帮助标志值:', canvas_layout,
                                    default_value=window_help_flag, min_val=0, max_val=32767)
        
        # 在任务条中显示
        window_show_in_taskbar = canvas.window_show_in_taskbar if canvas else True
        self._create_property_editor('checkbox', 'window_show_in_taskbar', '在任务条中显示:', canvas_layout,
                                    default_value=window_show_in_taskbar)
        
        # 随意移动
        window_free_move = canvas.window_free_move if canvas else False
        self._create_property_editor('checkbox', 'window_free_move', '随意移动:', canvas_layout,
                                    default_value=window_free_move)
        
        # 移除无效外形设置
        
        # 总在最前
        window_always_on_top = canvas.window_always_on_top if canvas else False
        self._create_property_editor('checkbox', 'window_always_on_top', '总在最前:', canvas_layout,
                                    default_value=window_always_on_top)
        
        # 保持标题条激活
        window_keep_title_active = canvas.window_keep_title_active if canvas else False
        self._create_property_editor('checkbox', 'window_keep_title_active', '保持标题条激活:', canvas_layout,
                                    default_value=window_keep_title_active)
        
        # 窗口类名
        window_class_name = canvas.window_class_name if canvas else ""
        self._create_property_editor('lineedit', 'window_class_name', '窗口类名:', canvas_layout,
                                    default_value=window_class_name)
        
        # 将选项卡添加到编辑器布局
        self.editor_layout.addWidget(canvas_tab)

    def _build_code_view_widget(self):
        container = QWidget()
        v = QVBoxLayout(container)
        h = QHBoxLayout()
        h.addWidget(QLabel("框架："))
        self.framework_combo = QComboBox()
        self.framework_combo.addItems(["PySide", "TKinter"])
        self.framework_combo.currentTextChanged.connect(self._refresh_code_view)
        h.addWidget(self.framework_combo)
        self.sync_checkbox = QCheckBox("同步生成")
        self.sync_checkbox.setChecked(True)
        self.sync_checkbox.stateChanged.connect(lambda *_: self._refresh_code_view())
        h.addWidget(self.sync_checkbox)
        self.refresh_button = QPushButton("刷新")
        self.refresh_button.clicked.connect(self._refresh_code_view)
        h.addWidget(self.refresh_button)
        self.save_button2 = QPushButton("保存")
        self.save_button2.clicked.connect(self._save_current_code)
        h.addWidget(self.save_button2)
        self.copy_button2 = QPushButton("复制")
        self.copy_button2.clicked.connect(self._copy_current_code)
        h.addWidget(self.copy_button2)
        h.addStretch()
        v.addLayout(h)
        self.code_view_editor = MonacoEditorWidget()
        self.code_view_editor.setReadOnly(False)
        self.code_view_editor.setLineWrapMode(QPlainTextEdit.NoWrap)
        self.code_view_editor.setFont(QFont("Consolas", 10))
        self.code_view_editor.textChanged.connect(self._on_code_text_changed)
        v.addWidget(self.code_view_editor)
        return container

    def set_fixed_framework(self, framework: str):
        try:
            if hasattr(self, 'framework_combo'):
                self.framework_combo.setCurrentText(framework)
                self.framework_combo.setEnabled(False)
        except Exception:
            pass

    def _switch_mode(self, index: int):
        self.stacked.setCurrentIndex(index)
        self._update_mode_buttons(index)
        if index == 1:
            self._refresh_code_view()

    def _update_mode_buttons(self, index: int):
        self.btn_properties.setEnabled(index != 0)
        self.btn_code.setEnabled(index != 1)

    def _refresh_code_view(self):
        main_window = self.parent()
        while main_window and not hasattr(main_window, 'code_generator'):
            main_window = main_window.parent()
        if not main_window:
            return
        if hasattr(self, 'sync_checkbox') and not self.sync_checkbox.isChecked() and self._user_edited:
            return
        canvas_width = main_window.design_canvas.canvas_width
        canvas_height = main_window.design_canvas.canvas_height
        try:
            main_window.code_generator.set_window_title(main_window.design_canvas.window_title)
            main_window.code_generator.set_window_size(canvas_width, canvas_height)
            main_window.code_generator.set_window_class_name("GeneratedWindow")
            icon_path = getattr(main_window.design_canvas, 'window_icon', None)
            if isinstance(icon_path, str) and icon_path:
                main_window.code_generator.set_window_icon(icon_path)
            pages = getattr(main_window.design_canvas, 'window_menus', [])
            if pages:
                main_window.code_generator.set_pages(pages)
            menus = getattr(main_window.design_canvas, 'window_dropdown_menus', [])
            if menus:
                main_window.code_generator.set_dropdown_menus(menus)
            fw = getattr(main_window, 'fixed_framework', None)
            framework = fw if fw else (self.framework_combo.currentText() if hasattr(self, 'framework_combo') else "PySide")
            code = main_window.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            self._updating_code = True
            self.code_view_editor.setPlainText(code)
            self._updating_code = False
            self._user_edited = False
        except Exception:
            pass
        self._ensure_canvas_signal_hooks()

    def _ensure_canvas_signal_hooks(self):
        if self._canvas_hooked:
            return
        main_window = self.parent()
        while main_window and not hasattr(main_window, 'design_canvas'):
            main_window = main_window.parent()
        if not main_window:
            return
        canvas = main_window.design_canvas
        try:
            canvas.control_selected.connect(self._on_canvas_changed)
            canvas.control_added.connect(self._on_canvas_changed)
            canvas.control_deleted.connect(self._on_canvas_changed)
            canvas.control_moved.connect(lambda *_: self._on_canvas_changed())
            canvas.control_resized.connect(lambda *_: self._on_canvas_changed())
            canvas.canvas_resized.connect(lambda *_: self._on_canvas_changed())
            self._canvas_hooked = True
        except Exception:
            pass

    def _on_canvas_changed(self, *args, **kwargs):
        if hasattr(self, 'stacked') and self.stacked.currentIndex() == 1:
            QTimer.singleShot(0, self._refresh_code_view)

    def _choose_file_for_canvas(self, property_name: str, title: str, filter_str: str):
        from PySide6.QtWidgets import QFileDialog
        file_path, _ = QFileDialog.getOpenFileName(self, title, "", filter_str)
        if file_path:
            self._on_property_changed(property_name, file_path)
            if property_name == 'window_bg_image':
                self.canvas_bg_image_label.setText(file_path)
            if property_name == 'window_icon':
                self.canvas_icon_label.setText(file_path)
    
    def _choose_canvas_bg_color(self):
        """选择画布背景颜色"""
        color = QColor(255, 255, 255)
        selected_color = QColorDialog.getColor(color, self)
        
        if selected_color.isValid():
            self._on_property_changed("canvas_bg_color", selected_color)
            self.canvas_bg_color_label.setText(f"当前背景色: {selected_color.name()}")
            self.canvas_bg_color_label.setStyleSheet(f"background-color: {selected_color.name()};")
    
    def _create_property_editors(self, control: Control):
        """为控件创建属性编辑器"""
        # 获取控件信息
        control_info = ControlLibrary.get_control_info(control.control_type)
        
        # 创建基本属性组
        self._create_basic_properties_group(control)
        
        # 创建特定属性组
        self._create_specific_properties_group(control, control_info)
        
        # 创建布局属性组
        self._create_layout_properties_group(control)
        
        # 创建样式属性组
        self._create_style_properties_group(control)
        
        # 创建事件绑定组
        self._create_event_binding_group(control)
    
    def _create_basic_properties_group(self, control: Control):
        """创建基本属性组"""
        group = QGroupBox("基本属性")
        group_layout = QFormLayout(group)
        
        # ID属性（只读）
        id_edit = QLineEdit(control.id)
        id_edit.setReadOnly(True)
        group_layout.addRow("ID:", id_edit)
        
        # 类型属性（只读）
        type_edit = QLineEdit(control.control_type)
        type_edit.setReadOnly(True)
        group_layout.addRow("类型:", type_edit)
        
        # 名称属性
        name_edit = QLineEdit(control.properties.get("name", ""))
        name_edit.textChanged.connect(lambda text: self._on_property_changed("name", text))
        group_layout.addRow("名称:", name_edit)
        self.property_editors["name"] = name_edit
        
        # 可见性属性
        visible_check = QCheckBox()
        visible_check.setChecked(control.properties.get("visible", True))
        visible_check.stateChanged.connect(lambda state: self._on_property_changed("visible", state == Qt.Checked))
        group_layout.addRow("可见:", visible_check)
        self.property_editors["visible"] = visible_check
        
        # 启用状态属性
        enabled_check = QCheckBox()
        enabled_check.setChecked(control.properties.get("enabled", True))
        enabled_check.stateChanged.connect(lambda state: self._on_property_changed("enabled", state == Qt.Checked))
        group_layout.addRow("启用:", enabled_check)
        self.property_editors["enabled"] = enabled_check
        
        self.editor_layout.addWidget(group)
    
    def _create_specific_properties_group(self, control: Control, control_info: Dict):
        """创建特定属性组"""
        group = QGroupBox("特定属性")
        group_layout = QFormLayout(group)
        
        # 根据控件类型添加特定属性
        if control.control_type == "Label":
            # 文本属性
            text_edit = QLineEdit(control.properties.get("text", "Label"))
            text_edit.textChanged.connect(lambda text: self._on_property_changed("text", text))
            group_layout.addRow("文本:", text_edit)
            self.property_editors["text"] = text_edit
            
            # 对齐方式
            alignment_combo = QComboBox()
            alignment_combo.addItems(["左对齐", "居中", "右对齐"])
            alignment = control.properties.get("alignment", "center")
            if alignment == "left":
                alignment_combo.setCurrentIndex(0)
            elif alignment == "center":
                alignment_combo.setCurrentIndex(1)
            elif alignment == "right":
                alignment_combo.setCurrentIndex(2)
            alignment_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("alignment", ["left", "center", "right"][index]))
            group_layout.addRow("对齐:", alignment_combo)
            self.property_editors["alignment"] = alignment_combo
        
        elif control.control_type == "Button":
            # 文本属性
            text_edit = QLineEdit(control.properties.get("text", "Button"))
            text_edit.textChanged.connect(lambda text: self._on_property_changed("text", text))
            group_layout.addRow("文本:", text_edit)
            self.property_editors["text"] = text_edit
            
            # 默认按钮
            default_check = QCheckBox()
            default_check.setChecked(control.properties.get("default", False))
            default_check.stateChanged.connect(lambda state: self._on_property_changed("default", state == Qt.Checked))
            group_layout.addRow("默认:", default_check)
            self.property_editors["default"] = default_check
        
        elif control.control_type in ["LineEdit", "TextEdit"]:
            # 文本属性
            if control.control_type == "LineEdit":
                text_edit = QLineEdit(control.properties.get("text", ""))
                text_edit.textChanged.connect(lambda text: self._on_property_changed("text", text))
                group_layout.addRow("文本:", text_edit)
                self.property_editors["text"] = text_edit
                
                # 占位符文本
                placeholder_edit = QLineEdit(control.properties.get("placeholder_text", ""))
                placeholder_edit.textChanged.connect(lambda text: self._on_property_changed("placeholder_text", text))
                group_layout.addRow("占位符:", placeholder_edit)
                self.property_editors["placeholder_text"] = placeholder_edit
                
                # 只读属性
                readonly_check = QCheckBox()
                readonly_check.setChecked(control.properties.get("read_only", False))
                readonly_check.stateChanged.connect(lambda state: self._on_property_changed("read_only", state == Qt.Checked))
                group_layout.addRow("只读:", readonly_check)
                self.property_editors["read_only"] = readonly_check
                
                # 回显模式
                echo_combo = QComboBox()
                echo_combo.addItems(["正常", "密码", "密码时回显"])
                echo_mode = control.properties.get("echo_mode", "normal")
                if echo_mode == "normal":
                    echo_combo.setCurrentIndex(0)
                elif echo_mode == "password":
                    echo_combo.setCurrentIndex(1)
                elif echo_mode == "password_echo_on_edit":
                    echo_combo.setCurrentIndex(2)
                echo_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("echo_mode", ["normal", "password", "password_echo_on_edit"][index]))
                group_layout.addRow("回显模式:", echo_combo)
                self.property_editors["echo_mode"] = echo_combo
            else:  # TextEdit
                text_edit = QTextEdit()
                text_edit.setPlainText(control.properties.get("plain_text", ""))
                text_edit.textChanged.connect(lambda: self._on_property_changed("plain_text", text_edit.toPlainText()))
                text_edit.setMaximumHeight(100)
                group_layout.addRow("文本:", text_edit)
                self.property_editors["plain_text"] = text_edit
                
                # 占位符文本
                placeholder_edit = QLineEdit(control.properties.get("placeholder_text", ""))
                placeholder_edit.textChanged.connect(lambda text: self._on_property_changed("placeholder_text", text))
                group_layout.addRow("占位符:", placeholder_edit)
                self.property_editors["placeholder_text"] = placeholder_edit
                
                # 只读属性
                readonly_check = QCheckBox()
                readonly_check.setChecked(control.properties.get("read_only", False))
                readonly_check.stateChanged.connect(lambda state: self._on_property_changed("read_only", state == Qt.Checked))
                group_layout.addRow("只读:", readonly_check)
                self.property_editors["read_only"] = readonly_check
        
        elif control.control_type == "Frame":
            # 框架形状
            frame_shape_combo = QComboBox()
            frame_shape_combo.addItems(["无", "方框", "面板", "样式化面板"])
            frame_shape = control.properties.get("frame_shape", "box")
            if frame_shape == "no_frame":
                frame_shape_combo.setCurrentIndex(0)
            elif frame_shape == "box":
                frame_shape_combo.setCurrentIndex(1)
            elif frame_shape == "panel":
                frame_shape_combo.setCurrentIndex(2)
            elif frame_shape == "styled_panel":
                frame_shape_combo.setCurrentIndex(3)
            frame_shape_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("frame_shape", ["no_frame", "box", "panel", "styled_panel"][index]))
            group_layout.addRow("框架形状:", frame_shape_combo)
            self.property_editors["frame_shape"] = frame_shape_combo
        
        elif control.control_type == "GroupBox":
            # 标题
            title_edit = QLineEdit(control.properties.get("title", "GroupBox"))
            title_edit.textChanged.connect(lambda text: self._on_property_changed("title", text))
            group_layout.addRow("标题:", title_edit)
            self.property_editors["title"] = title_edit
            
            # 平坦属性
            flat_check = QCheckBox()
            flat_check.setChecked(control.properties.get("flat", False))
            flat_check.stateChanged.connect(lambda state: self._on_property_changed("flat", state == Qt.Checked))
            group_layout.addRow("平坦:", flat_check)
            self.property_editors["flat"] = flat_check
            
            # 可选中属性
            checkable_check = QCheckBox()
            checkable_check.setChecked(control.properties.get("checkable", False))
            checkable_check.stateChanged.connect(lambda state: self._on_property_changed("checkable", state == Qt.Checked))
            group_layout.addRow("可选中:", checkable_check)
            self.property_editors["checkable"] = checkable_check
        
        elif control.control_type == "TabWidget":
            # 标签位置
            tab_position_combo = QComboBox()
            tab_position_combo.addItems(["北", "南", "西", "东"])
            tab_position = control.properties.get("tab_position", "north")
            if tab_position == "north":
                tab_position_combo.setCurrentIndex(0)
            elif tab_position == "south":
                tab_position_combo.setCurrentIndex(1)
            elif tab_position == "west":
                tab_position_combo.setCurrentIndex(2)
            elif tab_position == "east":
                tab_position_combo.setCurrentIndex(3)
            tab_position_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("tab_position", ["north", "south", "west", "east"][index]))
            group_layout.addRow("标签位置:", tab_position_combo)
            self.property_editors["tab_position"] = tab_position_combo
            
            # 可移动属性
            movable_check = QCheckBox()
            movable_check.setChecked(control.properties.get("movable", False))
            movable_check.stateChanged.connect(lambda state: self._on_property_changed("movable", state == Qt.Checked))
            group_layout.addRow("可移动:", movable_check)
            self.property_editors["movable"] = movable_check
            # 标签列表
            tabs_edit = QLineEdit(
                ",".join(control.properties.get("tabs", ["页面1","页面2"]))
            )
            tabs_edit.textChanged.connect(lambda text: self._on_property_changed("tabs", [s.strip() for s in text.split(",") if s.strip()]))
            group_layout.addRow("标签列表:", tabs_edit)
            self.property_editors["tabs"] = tabs_edit
        
        elif control.control_type == "StackedWidget":
            # 页面列表
            pages_edit = QLineEdit(
                ",".join(control.properties.get("pages", ["页面1","页面2"]))
            )
            pages_edit.textChanged.connect(lambda text: self._on_property_changed("pages", [s.strip() for s in text.split(",") if s.strip()]))
            group_layout.addRow("页面列表:", pages_edit)
            self.property_editors["pages"] = pages_edit
            # 当前页索引
            index_spin = QSpinBox()
            index_spin.setMinimum(0)
            index_spin.setMaximum(999)
            index_spin.setValue(control.properties.get("current_index", 0))
            index_spin.valueChanged.connect(lambda value: self._on_property_changed("current_index", value))
            group_layout.addRow("当前页索引:", index_spin)
            self.property_editors["current_index"] = index_spin
        
        elif control.control_type == "CheckBox":
            # 文本
            text_edit = QLineEdit(control.properties.get("text", "CheckBox"))
            text_edit.textChanged.connect(lambda text: self._on_property_changed("text", text))
            group_layout.addRow("文本:", text_edit)
            self.property_editors["text"] = text_edit
            
            # 选中状态
            checked_check = QCheckBox()
            checked_check.setChecked(control.properties.get("checked", False))
            checked_check.stateChanged.connect(lambda state: self._on_property_changed("checked", state == Qt.Checked))
            group_layout.addRow("选中:", checked_check)
            self.property_editors["checked"] = checked_check
            
            # 三态属性
            tristate_check = QCheckBox()
            tristate_check.setChecked(control.properties.get("tristate", False))
            tristate_check.stateChanged.connect(lambda state: self._on_property_changed("tristate", state == Qt.Checked))
            group_layout.addRow("三态:", tristate_check)
            self.property_editors["tristate"] = tristate_check
        
        elif control.control_type == "RadioButton":
            # 文本
            text_edit = QLineEdit(control.properties.get("text", "RadioButton"))
            text_edit.textChanged.connect(lambda text: self._on_property_changed("text", text))
            group_layout.addRow("文本:", text_edit)
            self.property_editors["text"] = text_edit
            
            # 选中状态
            checked_check = QCheckBox()
            checked_check.setChecked(control.properties.get("checked", False))
            checked_check.stateChanged.connect(lambda state: self._on_property_changed("checked", state == Qt.Checked))
            group_layout.addRow("选中:", checked_check)
            self.property_editors["checked"] = checked_check
        
        elif control.control_type == "ComboBox":
            # 可编辑属性
            editable_check = QCheckBox()
            editable_check.setChecked(control.properties.get("editable", False))
            editable_check.stateChanged.connect(lambda state: self._on_property_changed("editable", state == Qt.Checked))
            group_layout.addRow("可编辑:", editable_check)
            self.property_editors["editable"] = editable_check
            
            # 当前项
            current_index_spin = QSpinBox()
            current_index_spin.setMinimum(0)
            current_index_spin.setMaximum(99)
            current_index_spin.setValue(control.properties.get("current_index", 0))
            current_index_spin.valueChanged.connect(lambda value: self._on_property_changed("current_index", value))
            group_layout.addRow("当前项:", current_index_spin)
            self.property_editors["current_index"] = current_index_spin
        
        elif control.control_type == "Slider":
            # 方向
            orientation_combo = QComboBox()
            orientation_combo.addItems(["水平", "垂直"])
            orientation = control.properties.get("orientation", "horizontal")
            if orientation == "horizontal":
                orientation_combo.setCurrentIndex(0)
            elif orientation == "vertical":
                orientation_combo.setCurrentIndex(1)
            orientation_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("orientation", ["horizontal", "vertical"][index]))
            group_layout.addRow("方向:", orientation_combo)
            self.property_editors["orientation"] = orientation_combo
            
            # 最小值
            minimum_spin = QSpinBox()
            minimum_spin.setMinimum(-9999)
            minimum_spin.setMaximum(9999)
            minimum_spin.setValue(control.properties.get("minimum", 0))
            minimum_spin.valueChanged.connect(lambda value: self._on_property_changed("minimum", value))
            group_layout.addRow("最小值:", minimum_spin)
            self.property_editors["minimum"] = minimum_spin
            
            # 最大值
            maximum_spin = QSpinBox()
            maximum_spin.setMinimum(-9999)
            maximum_spin.setMaximum(9999)
            maximum_spin.setValue(control.properties.get("maximum", 100))
            maximum_spin.valueChanged.connect(lambda value: self._on_property_changed("maximum", value))
            group_layout.addRow("最大值:", maximum_spin)
            self.property_editors["maximum"] = maximum_spin
            
            # 当前值
            value_spin = QSpinBox()
            value_spin.setMinimum(-9999)
            value_spin.setMaximum(9999)
            value_spin.setValue(control.properties.get("value", 50))
            value_spin.valueChanged.connect(lambda value: self._on_property_changed("value", value))
            group_layout.addRow("当前值:", value_spin)
            self.property_editors["value"] = value_spin
            
            # 刻度显示
            tick_position_combo = QComboBox()
            tick_position_combo.addItems(["无", "两侧", "上方", "下方", "左侧", "右侧"])
            tick_position = control.properties.get("tick_position", "no_ticks")
            if tick_position == "no_ticks":
                tick_position_combo.setCurrentIndex(0)
            elif tick_position == "ticks_both_sides":
                tick_position_combo.setCurrentIndex(1)
            elif tick_position == "ticks_above":
                tick_position_combo.setCurrentIndex(2)
            elif tick_position == "ticks_below":
                tick_position_combo.setCurrentIndex(3)
            elif tick_position == "ticks_left":
                tick_position_combo.setCurrentIndex(4)
            elif tick_position == "ticks_right":
                tick_position_combo.setCurrentIndex(5)
            tick_position_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("tick_position", ["no_ticks", "ticks_both_sides", "ticks_above", "ticks_below", "ticks_left", "ticks_right"][index]))
            group_layout.addRow("刻度位置:", tick_position_combo)
            self.property_editors["tick_position"] = tick_position_combo
        
        elif control.control_type == "ProgressBar":
            # 方向
            orientation_combo = QComboBox()
            orientation_combo.addItems(["水平", "垂直"])
            orientation = control.properties.get("orientation", "horizontal")
            if orientation == "horizontal":
                orientation_combo.setCurrentIndex(0)
            elif orientation == "vertical":
                orientation_combo.setCurrentIndex(1)
            orientation_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("orientation", ["horizontal", "vertical"][index]))
            group_layout.addRow("方向:", orientation_combo)
            self.property_editors["orientation"] = orientation_combo
            
            # 最小值
            minimum_spin = QSpinBox()
            minimum_spin.setMinimum(0)
            minimum_spin.setMaximum(9999)
            minimum_spin.setValue(control.properties.get("minimum", 0))
            minimum_spin.valueChanged.connect(lambda value: self._on_property_changed("minimum", value))
            group_layout.addRow("最小值:", minimum_spin)
            self.property_editors["minimum"] = minimum_spin
            
            # 最大值
            maximum_spin = QSpinBox()
            maximum_spin.setMinimum(1)
            maximum_spin.setMaximum(9999)
            maximum_spin.setValue(control.properties.get("maximum", 100))
            maximum_spin.valueChanged.connect(lambda value: self._on_property_changed("maximum", value))
            group_layout.addRow("最大值:", maximum_spin)
            self.property_editors["maximum"] = maximum_spin
            
            # 当前值
            value_spin = QSpinBox()
            value_spin.setMinimum(0)
            value_spin.setMaximum(9999)
            value_spin.setValue(control.properties.get("value", 30))
            value_spin.valueChanged.connect(lambda value: self._on_property_changed("value", value))
            group_layout.addRow("当前值:", value_spin)
            self.property_editors["value"] = value_spin
            
            # 文本可见
            text_visible_check = QCheckBox()
            text_visible_check.setChecked(control.properties.get("text_visible", True))
            text_visible_check.stateChanged.connect(lambda state: self._on_property_changed("text_visible", state == Qt.Checked))
            group_layout.addRow("文本可见:", text_visible_check)
            self.property_editors["text_visible"] = text_visible_check
        
        elif control.control_type == "ListWidget":
            # 选择模式
            selection_mode_combo = QComboBox()
            selection_mode_combo.addItems(["单选", "多选", "扩展", "无选择"])
            selection_mode = control.properties.get("selection_mode", "single_selection")
            if selection_mode == "single_selection":
                selection_mode_combo.setCurrentIndex(0)
            elif selection_mode == "multi_selection":
                selection_mode_combo.setCurrentIndex(1)
            elif selection_mode == "extended_selection":
                selection_mode_combo.setCurrentIndex(2)
            elif selection_mode == "no_selection":
                selection_mode_combo.setCurrentIndex(3)
            selection_mode_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("selection_mode", ["single_selection", "multi_selection", "extended_selection", "no_selection"][index]))
            group_layout.addRow("选择模式:", selection_mode_combo)
            self.property_editors["selection_mode"] = selection_mode_combo
            
            # 编辑触发器
            edit_triggers_combo = QComboBox()
            edit_triggers_combo.addItems(["无", "双击", "选中时", "编辑键", "任意键"])
            edit_triggers = control.properties.get("edit_triggers", "double_clicked")
            if edit_triggers == "no_edit_triggers":
                edit_triggers_combo.setCurrentIndex(0)
            elif edit_triggers == "double_clicked":
                edit_triggers_combo.setCurrentIndex(1)
            elif edit_triggers == "edit_key_pressed":
                edit_triggers_combo.setCurrentIndex(2)
            elif edit_triggers == "any_key_pressed":
                edit_triggers_combo.setCurrentIndex(3)
            elif edit_triggers == "selected_clicked":
                edit_triggers_combo.setCurrentIndex(4)
            edit_triggers_combo.currentIndexChanged.connect(lambda index: self._on_property_changed("edit_triggers", ["no_edit_triggers", "double_clicked", "edit_key_pressed", "any_key_pressed", "selected_clicked"][index]))
            group_layout.addRow("编辑触发:", edit_triggers_combo)
            self.property_editors["edit_triggers"] = edit_triggers_combo
        
        elif control.control_type in ["SpinBox", "DoubleSpinBox"]:
            # 前缀
            prefix_edit = QLineEdit(control.properties.get("prefix", ""))
            prefix_edit.textChanged.connect(lambda text: self._on_property_changed("prefix", text))
            group_layout.addRow("前缀:", prefix_edit)
            self.property_editors["prefix"] = prefix_edit
            
            # 后缀
            suffix_edit = QLineEdit(control.properties.get("suffix", ""))
            suffix_edit.textChanged.connect(lambda text: self._on_property_changed("suffix", text))
            group_layout.addRow("后缀:", suffix_edit)
            self.property_editors["suffix"] = suffix_edit
            
            # 最小值
            if control.control_type == "SpinBox":
                minimum_spin = QSpinBox()
                minimum_spin.setMinimum(-9999)
                minimum_spin.setMaximum(9999)
                minimum_spin.setValue(control.properties.get("minimum", 0))
                minimum_spin.valueChanged.connect(lambda value: self._on_property_changed("minimum", value))
                self.property_editors["minimum"] = minimum_spin
            else:  # DoubleSpinBox
                minimum_spin = QDoubleSpinBox()
                minimum_spin.setMinimum(-9999.99)
                minimum_spin.setMaximum(9999.99)
                minimum_spin.setValue(control.properties.get("minimum", 0.0))
                minimum_spin.valueChanged.connect(lambda value: self._on_property_changed("minimum", value))
                self.property_editors["minimum"] = minimum_spin
            group_layout.addRow("最小值:", minimum_spin)
            
            # 最大值
            if control.control_type == "SpinBox":
                maximum_spin = QSpinBox()
                maximum_spin.setMinimum(-9999)
                maximum_spin.setMaximum(9999)
                maximum_spin.setValue(control.properties.get("maximum", 99))
                maximum_spin.valueChanged.connect(lambda value: self._on_property_changed("maximum", value))
                self.property_editors["maximum"] = maximum_spin
            else:  # DoubleSpinBox
                maximum_spin = QDoubleSpinBox()
                maximum_spin.setMinimum(-9999.99)
                maximum_spin.setMaximum(9999.99)
                maximum_spin.setValue(control.properties.get("maximum", 99.99))
                maximum_spin.valueChanged.connect(lambda value: self._on_property_changed("maximum", value))
                self.property_editors["maximum"] = maximum_spin
            group_layout.addRow("最大值:", maximum_spin)
            
            # 当前值
            if control.control_type == "SpinBox":
                value_spin = QSpinBox()
                value_spin.setMinimum(-9999)
                value_spin.setMaximum(9999)
                value_spin.setValue(control.properties.get("value", 0))
                value_spin.valueChanged.connect(lambda value: self._on_property_changed("value", value))
                self.property_editors["value"] = value_spin
            else:  # DoubleSpinBox
                value_spin = QDoubleSpinBox()
                value_spin.setMinimum(-9999.99)
                value_spin.setMaximum(9999.99)
                value_spin.setValue(control.properties.get("value", 0.0))
                value_spin.valueChanged.connect(lambda value: self._on_property_changed("value", value))
                self.property_editors["value"] = value_spin
            group_layout.addRow("当前值:", value_spin)
            
            # 步长
            if control.control_type == "SpinBox":
                step_spin = QSpinBox()
                step_spin.setMinimum(1)
                step_spin.setMaximum(9999)
                step_spin.setValue(control.properties.get("single_step", 1))
                step_spin.valueChanged.connect(lambda value: self._on_property_changed("single_step", value))
                self.property_editors["single_step"] = step_spin
            else:  # DoubleSpinBox
                step_spin = QDoubleSpinBox()
                step_spin.setMinimum(0.01)
                step_spin.setMaximum(9999.99)
                step_spin.setValue(control.properties.get("single_step", 0.1))
                step_spin.valueChanged.connect(lambda value: self._on_property_changed("single_step", value))
                self.property_editors["single_step"] = step_spin
            group_layout.addRow("步长:", step_spin)
            
            # 只读属性
            readonly_check = QCheckBox()
            readonly_check.setChecked(control.properties.get("read_only", False))
            readonly_check.stateChanged.connect(lambda state: self._on_property_changed("read_only", state == Qt.Checked))
            group_layout.addRow("只读:", readonly_check)
            self.property_editors["read_only"] = readonly_check
        elif control.control_type == "DateEdit":
            format_edit = QLineEdit(control.properties.get("display_format", "yyyy-MM-dd"))
            format_edit.textChanged.connect(lambda text: self._on_property_changed("display_format", text))
            group_layout.addRow("显示格式:", format_edit)
            self.property_editors["display_format"] = format_edit
            popup_check = QCheckBox()
            popup_check.setChecked(control.properties.get("calendar_popup", True))
            popup_check.stateChanged.connect(lambda state: self._on_property_changed("calendar_popup", state == Qt.Checked))
            group_layout.addRow("弹出日历:", popup_check)
            self.property_editors["calendar_popup"] = popup_check
        elif control.control_type == "TimeEdit":
            format_edit = QLineEdit(control.properties.get("display_format", "HH:mm:ss"))
            format_edit.textChanged.connect(lambda text: self._on_property_changed("display_format", text))
            group_layout.addRow("显示格式:", format_edit)
            self.property_editors["display_format"] = format_edit
        elif control.control_type == "DateTimeEdit":
            format_edit = QLineEdit(control.properties.get("display_format", "yyyy-MM-dd HH:mm:ss"))
            format_edit.textChanged.connect(lambda text: self._on_property_changed("display_format", text))
            group_layout.addRow("显示格式:", format_edit)
            self.property_editors["display_format"] = format_edit
        elif control.control_type == "TreeWidget":
            headers_edit = QLineEdit(
                ",".join(control.properties.get("header_labels", ["名称","类型"]))
            )
            headers_edit.textChanged.connect(lambda text: self._on_property_changed("header_labels", [s.strip() for s in text.split(",") if s.strip()]))
            group_layout.addRow("表头:", headers_edit)
            self.property_editors["header_labels"] = headers_edit
        
        self.editor_layout.addWidget(group)
    
    def _create_layout_properties_group(self, control: Control):
        """创建布局属性组"""
        group = QGroupBox("布局属性")
        group_layout = QFormLayout(group)
        
        # 位置X
        x_spin = QSpinBox()
        x_spin.setMinimum(0)
        x_spin.setMaximum(9999)
        x_spin.setValue(control.x)
        x_spin.valueChanged.connect(lambda value: self._on_property_changed("x", value))
        group_layout.addRow("X:", x_spin)
        self.property_editors["x"] = x_spin
        
        # 位置Y
        y_spin = QSpinBox()
        y_spin.setMinimum(0)
        y_spin.setMaximum(9999)
        y_spin.setValue(control.y)
        y_spin.valueChanged.connect(lambda value: self._on_property_changed("y", value))
        group_layout.addRow("Y:", y_spin)
        self.property_editors["y"] = y_spin
        
        # 宽度
        width_spin = QSpinBox()
        width_spin.setMinimum(10)
        width_spin.setMaximum(9999)
        width_spin.setValue(control.width)
        width_spin.valueChanged.connect(lambda value: self._on_property_changed("width", value))
        group_layout.addRow("宽度:", width_spin)
        self.property_editors["width"] = width_spin
        
        # 高度
        height_spin = QSpinBox()
        height_spin.setMinimum(10)
        height_spin.setMaximum(9999)
        height_spin.setValue(control.height)
        height_spin.valueChanged.connect(lambda value: self._on_property_changed("height", value))
        group_layout.addRow("高度:", height_spin)
        self.property_editors["height"] = height_spin
        
        # 最小宽度
        min_width_spin = QSpinBox()
        min_width_spin.setMinimum(0)
        min_width_spin.setMaximum(9999)
        min_width_spin.setValue(control.properties.get("minimum_width", 0))
        min_width_spin.valueChanged.connect(lambda value: self._on_property_changed("minimum_width", value))
        group_layout.addRow("最小宽度:", min_width_spin)
        self.property_editors["minimum_width"] = min_width_spin
        
        # 最小高度
        min_height_spin = QSpinBox()
        min_height_spin.setMinimum(0)
        min_height_spin.setMaximum(9999)
        min_height_spin.setValue(control.properties.get("minimum_height", 0))
        min_height_spin.valueChanged.connect(lambda value: self._on_property_changed("minimum_height", value))
        group_layout.addRow("最小高度:", min_height_spin)
        self.property_editors["minimum_height"] = min_height_spin
        
        # 最大宽度
        max_width_spin = QSpinBox()
        max_width_spin.setMinimum(0)
        max_width_spin.setMaximum(16777215)  # QWIDGETSIZE_MAX
        max_width_spin.setValue(control.properties.get("maximum_width", 16777215))
        max_width_spin.valueChanged.connect(lambda value: self._on_property_changed("maximum_width", value))
        group_layout.addRow("最大宽度:", max_width_spin)
        self.property_editors["maximum_width"] = max_width_spin
        
        # 最大高度
        max_height_spin = QSpinBox()
        max_height_spin.setMinimum(0)
        max_height_spin.setMaximum(16777215)  # QWIDGETSIZE_MAX
        max_height_spin.setValue(control.properties.get("maximum_height", 16777215))
        max_height_spin.valueChanged.connect(lambda value: self._on_property_changed("maximum_height", value))
        group_layout.addRow("最大高度:", max_height_spin)
        self.property_editors["maximum_height"] = max_height_spin
        
        # Z轴顺序
        z_order_spin = QSpinBox()
        z_order_spin.setMinimum(0)
        z_order_spin.setMaximum(9999)
        z_order_spin.setValue(control.z_order)
        z_order_spin.valueChanged.connect(lambda value: self._on_property_changed("z_order", value))
        group_layout.addRow("Z轴顺序:", z_order_spin)
        self.property_editors["z_order"] = z_order_spin
        
        self.editor_layout.addWidget(group)
    
    def _create_style_properties_group(self, control: Control):
        """创建样式属性组"""
        group = QGroupBox("样式属性")
        group_layout = QFormLayout(group)
        
        # 样式表
        style_edit = QTextEdit()
        style_edit.setPlainText(control.properties.get("style_sheet", ""))
        style_edit.setMaximumHeight(100)
        style_edit.textChanged.connect(lambda: self._on_property_changed("style_sheet", style_edit.toPlainText()))
        group_layout.addRow("样式表:", style_edit)
        self.property_editors["style_sheet"] = style_edit
        
        # 字体
        font_button = QPushButton("选择字体")
        font_button.clicked.connect(self._choose_font)
        group_layout.addRow("字体:", font_button)
        
        # 当前字体显示
        self.font_label = QLabel("当前字体: 默认")
        font = control.properties.get("font", None)
        if font:
            self.font_label.setText(f"当前字体: {font.family()}, {font.pointSize()}pt")
        group_layout.addRow("", self.font_label)
        
        # 字体颜色
        color_button = QPushButton("选择颜色")
        color_button.clicked.connect(self._choose_color)
        group_layout.addRow("文字颜色:", color_button)
        
        # 当前颜色显示
        self.color_label = QLabel("当前颜色: 默认")
        color = control.properties.get("color", None)
        if color:
            self.color_label.setText(f"当前颜色: {color.name()}")
            self.color_label.setStyleSheet(f"background-color: {color.name()}; color: white;")
        group_layout.addRow("", self.color_label)
        
        # 背景颜色
        bg_color_button = QPushButton("选择背景色")
        bg_color_button.clicked.connect(self._choose_bg_color)
        group_layout.addRow("背景颜色:", bg_color_button)
        
        # 当前背景颜色显示
        self.bg_color_label = QLabel("当前背景色: 默认")
        bg_color = control.properties.get("background_color", None)
        if bg_color:
            self.bg_color_label.setText(f"当前背景色: {bg_color.name()}")
            self.bg_color_label.setStyleSheet(f"background-color: {bg_color.name()};")
        group_layout.addRow("", self.bg_color_label)
        
        # 透明度
        opacity_spin = QDoubleSpinBox()
        opacity_spin.setMinimum(0.0)
        opacity_spin.setMaximum(1.0)
        opacity_spin.setSingleStep(0.1)
        opacity_spin.setValue(control.properties.get("opacity", 1.0))
        opacity_spin.valueChanged.connect(lambda value: self._on_property_changed("opacity", value))
        group_layout.addRow("透明度:", opacity_spin)
        self.property_editors["opacity"] = opacity_spin
        
        self.editor_layout.addWidget(group)
    
    def _on_property_changed(self, property_name: str, value: Any):
        """属性值变化时的处理"""
        if self.is_canvas:
            # 画布属性变化
            self.property_changed.emit("canvas", property_name, value)
        elif self.current_control:
            # 控件属性变化
            self.property_changed.emit(self.current_control.id, property_name, value)
        if hasattr(self, 'stacked') and self.stacked.currentIndex() == 1 and (not hasattr(self, 'sync_checkbox') or self.sync_checkbox.isChecked()):
            QTimer.singleShot(0, self._refresh_code_view)

    def _choose_font(self):
        if not self.current_control:
            return
        font = self.current_control.properties.get("font", QFont())
        selected_font, ok = QFontDialog.getFont(font, self)
        if ok:
            self._on_property_changed("font", selected_font)
            self.font_label.setText(f"当前字体: {selected_font.family()}, {selected_font.pointSize()}pt")

    def _choose_color(self):
        if not self.current_control:
            return
        color = self.current_control.properties.get("color", QColor(0, 0, 0))
        selected_color = QColorDialog.getColor(color, self)
        if selected_color.isValid():
            self._on_property_changed("color", selected_color)
            self.color_label.setText(f"当前颜色: {selected_color.name()}")
            self.color_label.setStyleSheet(f"background-color: {selected_color.name()}; color: white;")

    def _choose_bg_color(self):
        if not self.current_control:
            return
        bg_color = self.current_control.properties.get("background_color", QColor(240, 240, 240))
        selected_color = QColorDialog.getColor(bg_color, self)
        if selected_color.isValid():
            self._on_property_changed("background_color", selected_color)
            self.bg_color_label.setText(f"当前背景色: {selected_color.name()}")
            self.bg_color_label.setStyleSheet(f"background-color: {selected_color.name()};")

    def _on_code_text_changed(self):
        if self._updating_code:
            return
        self._user_edited = True

    def _run_current_code(self):
        main_window = self.parent()
        while main_window and not hasattr(main_window, '_find_python_interpreter'):
            main_window = main_window.parent()
        if not main_window:
            return
        code = self.code_view_editor.toPlainText()
        if not code.strip():
            return
        import tempfile
        import subprocess
        import os
        with tempfile.NamedTemporaryFile(mode='w', suffix='.py', delete=False, encoding='utf-8') as f:
            f.write(code)
            temp_file_path = f.name
        python_exe = main_window._find_python_interpreter()
        if not python_exe:
            return
        try:
            subprocess.Popen([python_exe, temp_file_path])
        except Exception:
            pass

    def _save_current_code(self):
        try:
            main_window = self.parent()
            while main_window and not hasattr(main_window, 'code_generator'):
                main_window = main_window.parent()
            framework = getattr(main_window, 'fixed_framework', None)
            project_dir = getattr(main_window, 'project_dir', None)
            if framework and project_dir:
                import os
                fname = 'main_PySide.py' if framework.lower().startswith('py') else 'main_TKinter.py'
                file_path = os.path.join(project_dir, fname)
                os.makedirs(os.path.dirname(file_path), exist_ok=True)
                with open(file_path, 'w', encoding='utf-8') as f:
                    f.write(self.code_view_editor.toPlainText())
            else:
                file_path, _ = QFileDialog.getSaveFileName(
                    self,
                    "保存代码",
                    "generated_window.py",
                    "Python文件 (*.py);;所有文件 (*)"
                )
                if file_path:
                    try:
                        import os
                        os.makedirs(os.path.dirname(file_path), exist_ok=True)
                        with open(file_path, 'w', encoding='utf-8') as f:
                            f.write(self.code_view_editor.toPlainText())
                    except Exception:
                        pass
        except Exception:
            pass

    def _copy_current_code(self):
        self.code_view_editor.selectAll()
        self.code_view_editor.copy()

    
    def update_property(self, property_name: str, value: Any):
        """更新属性值"""
        if property_name in self.property_editors:
            editor = self.property_editors[property_name]
            
            # 根据编辑器类型设置值
            if isinstance(editor, QLineEdit):
                editor.setText(str(value))
            elif isinstance(editor, QTextEdit):
                editor.setPlainText(str(value))
            elif isinstance(editor, QSpinBox):
                editor.setValue(int(value))
            elif isinstance(editor, QDoubleSpinBox):
                editor.setValue(float(value))
            elif isinstance(editor, QCheckBox):
                editor.setChecked(bool(value))
            elif isinstance(editor, QComboBox):
                # 特殊处理一些组合框
                if property_name == "alignment":
                    if value == "left":
                        editor.setCurrentIndex(0)
                    elif value == "center":
                        editor.setCurrentIndex(1)
                    elif value == "right":
                        editor.setCurrentIndex(2)
                elif property_name == "orientation":
                    if value == "horizontal":
                        editor.setCurrentIndex(0)
                    elif value == "vertical":
                        editor.setCurrentIndex(1)
                # 其他组合框处理...
            
            # 特殊处理字体和颜色
            elif property_name == "font":
                if isinstance(value, QFont):
                    self.font_label.setText(f"当前字体: {value.family()}, {value.pointSize()}pt")
            elif property_name == "color":
                if isinstance(value, QColor):
                    self.color_label.setText(f"当前颜色: {value.name()}")
                    self.color_label.setStyleSheet(f"background-color: {value.name()}; color: white;")
            elif property_name == "background_color":
                if isinstance(value, QColor):
                    self.bg_color_label.setText(f"当前背景色: {value.name()}")
                    self.bg_color_label.setStyleSheet(f"background-color: {value.name()};")
    
    def _create_event_binding_group(self, control: Control):
        """创建事件绑定组"""
        group = QGroupBox("事件绑定")
        group_layout = QFormLayout(group)
        
        # 获取控件支持的所有事件
        events = EventManager.get_events_for_control(control.control_type)
        
        # 创建事件下拉框
        event_combo = QComboBox()
        event_combo.addItem("-- 选择事件 --", None)
        
        # 获取控件已绑定的事件
        bound_events = control.events if hasattr(control, 'events') else {}
        if not isinstance(bound_events, dict):
            bound_events = {}
        
        # 添加所有可用事件到下拉框
        for event_id, event_info in events:
            display_name = event_info.get("name", event_id)
            # 检查是否已绑定
            is_bound = event_id in bound_events
            if is_bound:
                display_name = f"✓ {display_name}"
            event_combo.addItem(display_name, event_id)
        
        # 连接下拉框选择变化信号
        event_combo.currentIndexChanged.connect(
            lambda index: self._on_event_selected(control, event_combo, index)
        )
        
        group_layout.addRow("事件:", event_combo)
        self.property_editors["event_selector"] = event_combo
        
        # 存储事件列表供后续使用
        self._event_list = events
        self._current_control_events = bound_events
        
        self.editor_layout.addWidget(group)
    
    def _on_event_selected(self, control: Control, combo: QComboBox, index: int):
        """
        事件选择处理函数（画布代码绑定事件的核心处理函数）
        当用户在属性编辑器的事件下拉框中选择或取消选择事件时调用此函数
        
        此函数负责：
        1. 检查事件的绑定状态（已绑定或未绑定）
        2. 切换事件的绑定状态（绑定或取消绑定）
        3. 更新事件下拉框的显示（已绑定的事件前面显示✓标记）
        4. 触发信号通知主窗口更新事件函数文件
        5. 触发信号通知代码生成器更新生成的代码
        6. 高亮显示代码编辑器中对应的事件绑定代码行
        
        Args:
            control: 当前选中的控件对象
            combo: 事件选择下拉框控件
            index: 下拉框中选中项的索引（0表示"-- 选择事件 --"占位项）
        """
        # ========== 检查是否选择了有效事件 ==========
        # 如果索引小于等于0，表示选择了占位项"-- 选择事件 --"，不执行任何操作
        if index <= 0:  # "-- 选择事件 --"
            return
        
        # ========== 获取事件ID ==========
        # 从下拉框的项数据中获取事件ID（事件ID存储在itemData中）
        event_id = combo.itemData(index)
        # 如果事件ID不存在或为空，直接返回
        if not event_id:
            return
        
        # ========== 获取控件绑定的事件字典 ==========
        # 从控件对象中获取已绑定的事件字典
        # 如果控件没有events属性，初始化为空字典
        bound_events = control.events if hasattr(control, 'events') else {}
        # 确保bound_events是字典类型，如果不是则初始化为空字典
        if not isinstance(bound_events, dict):
            bound_events = {}
            control.events = bound_events
        
        # ========== 检查事件是否已绑定 ==========
        # 检查事件ID是否在已绑定事件字典中
        is_bound = event_id in bound_events
        
        # ========== 切换绑定状态 ==========
        # 如果事件已绑定，执行取消绑定操作
        if is_bound:
            # ========== 取消绑定 ==========
            # 从已绑定事件字典中删除该事件
            # 注意：只移除事件绑定代码，保留功能实现代码（用户可能已经编写了事件处理函数）
            del bound_events[event_id]
            # 更新下拉框显示，移除✓标记，恢复为普通的事件名称显示
            combo.setItemText(index, EventManager.get_event_display_name(event_id, control.control_type))
        else:
            # ========== 绑定事件 ==========
            # 将事件添加到已绑定事件字典中，值为True表示已绑定
            bound_events[event_id] = True
            # 更新下拉框显示，在事件名称前面添加✓标记，表示已绑定
            combo.setItemText(index, f"✓ {EventManager.get_event_display_name(event_id, control.control_type)}")
            
            # ========== 触发事件绑定信号 ==========
            # 触发事件绑定改变信号，通知主窗口创建或更新事件函数文件
            # 主窗口会收到此信号并调用_on_event_binding_changed处理函数
            if hasattr(self, 'event_binding_changed'):
                self.event_binding_changed.emit(control.id, event_id, True)
        
        # ========== 更新控件事件字典 ==========
        # 将更新后的事件字典保存回控件对象
        control.events = bound_events
        
        # ========== 触发属性变化信号 ==========
        # 触发属性变化信号，通知代码生成器更新生成的代码
        # 代码生成器会重新生成信号连接代码（_generate_signal_connections）
        self.property_changed.emit(control.id, "events", bound_events)
        
        # ========== 高亮显示事件绑定代码 ==========
        # 在代码编辑器中高亮显示与此事件相关的代码行（如信号连接代码）
        self._highlight_event_binding_code(control.id, event_id)
    
    def _highlight_event_binding_code(self, control_id: str, event_id: str):
        """
        高亮显示事件绑定代码（画布代码绑定函数的可视化反馈）
        在代码编辑器中高亮显示与指定控件和事件相关的代码行，帮助用户快速定位事件绑定代码
        
        此函数会：
        1. 查找主窗口和代码编辑器
        2. 获取控件的变量名和事件处理函数名
        3. 在代码文本中查找匹配的信号连接代码行
        4. 为匹配的代码行创建高亮选区并应用到代码编辑器
        
        Args:
            control_id: 控件ID（如"button1"、"label1"等）
            event_id: 事件ID（如"clicked"、"textChanged"等）
        """
        # ========== 查找主窗口 ==========
        # 从当前控件的父级开始，向上查找主窗口对象
        # 主窗口应该有code_view_editor_left属性（左侧代码编辑器）
        main_window = self.parent()
        while main_window and not hasattr(main_window, 'code_view_editor_left'):
            main_window = main_window.parent()
        
        # ========== 检查是否找到主窗口和代码编辑器 ==========
        # 如果未找到主窗口或主窗口没有代码编辑器，直接返回
        if not main_window or not hasattr(main_window, 'code_view_editor_left'):
            return
        
        # ========== 获取代码编辑器 ==========
        # 获取左侧代码编辑器对象
        code_editor = main_window.code_view_editor_left
        # 获取代码编辑器的纯文本内容
        code_text = code_editor.toPlainText()
        # 将代码文本按行分割，转换为行列表
        lines = code_text.splitlines()
        
        # ========== 获取控件变量名 ==========
        # 将控件ID转换为合法的Python变量名（用于在代码中查找）
        try:
            # 尝试从代码生成器中获取转换方法
            from .code_generator_pyside import CodeGenerator
            code_gen = getattr(main_window, 'code_generator', None)
            if code_gen:
                # 使用代码生成器的转换方法（与代码生成时使用的转换方法一致）
                var_name = code_gen._to_valid_variable_name(control_id)
            else:
                # 如果代码生成器不存在，使用正则表达式手动转换
                import re
                var_name = re.sub(r'[^a-zA-Z0-9_]', '_', control_id)
        except Exception:
            # 如果转换失败，使用正则表达式手动转换
            import re
            var_name = re.sub(r'[^a-zA-Z0-9_]', '_', control_id)
        
        # ========== 获取事件函数名 ==========
        # 根据控件ID和事件ID生成事件处理函数名称（格式：_on_{控件ID}_{事件名}）
        # 例如：控件ID="button1"，事件ID="clicked"，函数名="_on_button1_clicked"
        function_name = EventManager.get_event_function_name(control_id, event_id)
        
        # ========== 查找事件绑定代码行 ==========
        # 初始化高亮选区列表，用于存储需要高亮的代码行
        selections = []
        # 遍历代码的每一行，查找匹配的信号连接代码
        for i, line in enumerate(lines):
            # 将当前行转换为小写，用于不区分大小写的匹配
            line_lower = line.lower()
            # ========== 查找信号连接代码行 ==========
            # 检查代码行是否包含控件变量名和.connect(关键字
            # 例如：self.button1.clicked.connect(...) 或 self.label1.textChanged.connect(...)
            if f"self.{var_name.lower()}" in line_lower and ".connect(" in line_lower:
                # ========== 检查是否包含事件函数名 ==========
                # 进一步检查代码行是否包含对应的事件处理函数名
                # 这样可以确保高亮的是正确的事件绑定代码
                if function_name.lower() in line_lower:
                    # ========== 创建高亮选区 ==========
                    # 从文档中获取对应行号的文本块（行对象）
                    block = code_editor.document().findBlockByNumber(i)
                    # 创建文本光标对象，用于选择文本
                    cursor = code_editor.textCursor()
                    # 将光标定位到文本块的位置（行的开始位置）
                    cursor.setPosition(block.position())
                    # 移动到当前行的末尾，选中整行
                    cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                    
                    # ========== 创建额外选区对象 ==========
                    # 创建额外的文本选区对象（用于高亮显示）
                    sel = QTextEdit.ExtraSelection()
                    # 设置高亮背景颜色（黄色 RGB(255, 255, 0)），便于用户识别
                    sel.format.setBackground(QColor(255, 255, 0))
                    # 将光标选区设置到额外选区对象
                    sel.cursor = cursor
                    # 将高亮选区添加到选区列表
                    selections.append(sel)
        
        # ========== 应用高亮到代码编辑器 ==========
        # 如果找到了匹配的代码行，将高亮选区应用到代码编辑器
        if selections:
            code_editor.set_external_highlights(selections)
