"""
提示词模板模块
为AI代码生成提供标准化的提示词模板，支持PySide6和tkinter两种框架
"""

from typing import List  # 类型提示

class PromptTemplates:
    """
    提示词模板类
    用于构建AI代码生成时的系统提示词和用户提示词
    """
    def __init__(self):
        """
        初始化提示词模板类
        """
        # 存储模板列表（当前未使用，预留扩展）
        self._templates = []

    def list_templates(self) -> List[str]:
        """
        列出所有可用的模板名称
        
        Returns:
            List[str]: 模板名称列表
        """
        return list(self._templates)

    def normalize_framework(self, framework: str) -> str:
        """
        规范化框架名称
        将用户输入的框架名称转换为标准格式，统一处理各种变体
        
        Args:
            framework: 框架名称（如"tkinter"、"PySide"、"TKinter"、"tk"等）
            
        Returns:
            str: 规范化后的框架名称（"tkinter"或"PySide6"）
        """
        # ========== 处理输入并转换为小写 ==========
        # 如果framework为None或空，使用空字符串；否则使用原值
        # 转换为小写，统一处理大小写变体（如"PySide"、"pyside"、"PYSIDE"）
        f = (framework or "").lower()
        
        # ========== 判断框架类型 ==========
        # 检查规范化后的框架名称是否以"tk"开头（包括"tkinter"、"tk"等）
        if f.startswith("tk"):
            # 如果以"tk"开头，返回标准化的"tkinter"
            return "tkinter"
        
        # ========== 默认返回PySide6 ==========
        # 如果不符合tkinter条件，默认返回"PySide6"（包括"PySide"、"pyside"等变体）
        return "PySide6"

    def build_system_prompt(self, framework: str) -> str:
        """
        构建系统提示词
        为AI提供框架相关的代码生成指导
        
        Args:
            framework: 目标框架名称
            
        Returns:
            str: 系统提示词字符串
        """
        # ========== 规范化框架名称 ==========
        # 将输入的框架名称转换为标准格式（"tkinter"或"PySide6"）
        fw = self.normalize_framework(framework)
        
        # ========== 根据框架类型返回对应的系统提示词 ==========
        # 检查规范化后的框架名称是否为"pyside6"（不区分大小写）
        if fw.lower() == "pyside6":
            # 如果是PySide6框架，返回PySide6专用的系统提示词
            return (
                "请输出完整可运行的Python代码，使用三引号代码块包裹。"
                "使用PySide6框架，包含必要的导入语句。"
                "创建一个主窗口类，设置合适的窗口标题和大小。"
                "创建必要的中央部件并设置。"
                "根据需求创建适当数量和类型的控件。"
                "为可交互控件添加基本的信号槽连接。"
                "包含main()函数和程序入口点。"
                "确保中文文案使用中文显示。"
            )
        else:
            # ========== tkinter框架的系统提示词 ==========
            # 如果是tkinter框架（或其他非PySide6框架），返回tkinter专用的系统提示词
            return (
                "请输出完整可运行的Python代码，使用三引号代码块包裹。"
                "使用tkinter/ttk框架。"
                "创建一个主窗口，设置合适的窗口标题和大小。"
                "创建必要的控件并设置布局。"
                "包含基本的事件处理逻辑。"
                "包含main()函数和程序入口点。"
                "确保中文文案使用中文显示。"
            )

    def build_user_prompt(self, framework: str, task: str, mode: str = "单独GUI页面", logic: str = "无") -> str:
        """
        构建用户提示词
        根据任务描述和模式构建用户提示词
        
        Args:
            framework: 目标框架名称
            task: 任务描述（要生成的界面功能描述）
            mode: 生成模式，默认为"单独GUI页面"
            logic: 逻辑描述，默认为"无"
            
        Returns:
            str: 用户提示词字符串
        """
        # ========== 规范化框架名称 ==========
        # 将输入的框架名称转换为标准格式（"tkinter"或"PySide6"）
        fw = self.normalize_framework(framework)
        
        # ========== 构建基础提示词 ==========
        # 构建用户提示词的基础部分，包含任务描述和基本要求
        base = (
            f"请用{fw}生成完整可运行的Python脚本：{task}。"
            "根据需求灵活设计界面，可使用布局管理器或绝对定位。"
            "包含必要的控件和基本功能逻辑。"
            "确保代码可直接运行，界面美观实用。"
            "所有输出必须为代码，并使用三引号代码块包裹。"
            "界面文案请使用中文。"
        )
        
        # ========== 检查生成模式 ==========
        # 检查生成模式是否为"功能GUI页面"（需要添加功能逻辑）
        # (mode or '')：如果mode为None则使用空字符串
        # .strip()：去除首尾空白字符
        if (mode or '').strip() == "功能GUI页面":
            # 如果是功能GUI页面模式，在基础提示词后追加功能逻辑要求
            base += " 请添加适当的功能逻辑和用户交互反馈。"
        
        # 返回构建完成的用户提示词
        return base

# 创建全局提示词模板实例，供其他模块使用
prompt_templates = PromptTemplates()