#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
项目管理模块
负责项目的保存、加载和管理
"""

# 导入标准库模块
import json  # JSON数据处理，用于项目文件的序列化和反序列化
import os  # 操作系统接口，用于文件和目录操作
from datetime import datetime  # 日期时间处理，用于记录项目创建和修改时间
from typing import Dict, List, Optional, Any  # 类型提示

# 导入自定义模块
from module.control_pyside import ControlManager, Control  # 控件管理器和控件类
from module.version import PROJECT_SCHEMA_VERSION  # 项目架构版本号，用于兼容性检查
from module.window_class_manager import WindowClassManager  # 窗口类管理器


class ProjectManager:
    """
    项目管理类
    负责项目的创建、保存、加载和管理
    管理项目文件、控件数据、画布属性等信息
    """
    
    def __init__(self, control_manager: ControlManager = None):
        """
        初始化项目管理器
        
        Args:
            control_manager: 控件管理器实例，用于访问和管理控件数据（已废弃，使用window_class_manager）
        """
        # 保存控件管理器引用（向后兼容）
        self.control_manager = control_manager
        # 窗口类管理器（管理多个窗口类）
        self.window_class_manager = WindowClassManager()
        # 如果提供了control_manager，创建默认窗口类
        if control_manager:
            default_window = self.window_class_manager.add_window_class("GeneratedWindow")
            if default_window:
                default_window.control_manager = control_manager
        else:
            # 创建默认窗口类
            self.window_class_manager.add_window_class("GeneratedWindow")
        # 当前项目文件路径（None表示未保存的新项目）
        self.project_file = None
        # 项目名称，默认为"未命名项目"
        self.project_name = "未命名项目"
        # 项目是否已修改标志（True表示有未保存的更改）
        self.project_modified = False
        # 画布宽度，默认800像素（向后兼容）
        self.canvas_width = 800
        # 画布高度，默认520像素（向后兼容）
        self.canvas_height = 520
        # 窗口标题，默认"我的页面"（向后兼容）
        self.window_title = "我的页面"
        # 窗口头像/图标，默认使用用户表情符号
        self.window_avatar = "👤"
        # 窗口名称，用于生成事件函数文件名（向后兼容）
        self.window_name = "GeneratedWindow"
        # 事件函数文件内容
        self.event_function_file = ""
        
    def new_project(self) -> bool:
        """
        创建新项目
        清空所有控件并重置项目属性为默认值
        
        Returns:
            bool: 是否成功创建新项目（当前始终返回True）
        """
        # 清空控件管理器中的所有控件
        self.control_manager.clear_all()
        
        # ========== 重置项目属性为默认值 ==========
        # 清除项目文件路径（表示这是一个新项目）
        self.project_file = None
        # 重置项目名称为默认值
        self.project_name = "未命名项目"
        # 清除修改标志（新项目尚未被修改）
        self.project_modified = False
        # 重置画布尺寸为默认值
        self.canvas_width = 800
        self.canvas_height = 520
        # 重置窗口标题为默认值
        self.window_title = "我的页面"
        # 重置窗口头像为默认值
        self.window_avatar = "👤"
        # 重置窗口名称
        self.window_name = "GeneratedWindow"
        # 重置事件函数文件
        self.event_function_file = ""
        
        return True
    
    def save_project(self, file_path: Optional[str] = None) -> bool:
        """
        保存项目到文件
        将项目数据（控件、画布属性等）序列化为JSON格式并保存到文件
        
        Args:
            file_path: 保存路径，如果为None则使用当前项目文件路径
            
        Returns:
            bool: 是否成功保存项目
        """
        try:
            # ========== 确定保存路径 ==========
            # 如果没有指定文件路径，使用当前项目文件路径
            if file_path is None:
                # 如果当前项目文件路径也为None，说明项目尚未保存过，返回False
                if self.project_file is None:
                    return False
                # 使用当前项目文件路径
                file_path = self.project_file
            
            # ========== 创建保存目录 ==========
            # 获取文件所在目录路径
            dirpath = os.path.dirname(file_path)
            # 如果目录路径不为空，创建目录（如果不存在）
            if dirpath:
                os.makedirs(dirpath, exist_ok=True)
            
            # ========== 准备项目数据 ==========
            project_data = {
                "version": PROJECT_SCHEMA_VERSION,  # 项目架构版本号，用于兼容性检查
                "name": self.project_name,  # 项目名称
                "created_time": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),  # 创建时间（字符串格式）
                "canvas_width": self.canvas_width,  # 画布宽度（向后兼容）
                "canvas_height": self.canvas_height,  # 画布高度（向后兼容）
                "window_title": self.window_title,  # 窗口标题（向后兼容）
                "window_avatar": self.window_avatar,  # 窗口头像
                "window_name": getattr(self, 'window_name', "GeneratedWindow"),  # 窗口名称（向后兼容）
                "event_function_file": getattr(self, 'event_function_file', ""),  # 事件函数文件内容
                "window_class_manager": self.window_class_manager.to_dict(),  # 窗口类管理器数据
                "controls": []  # 控件列表（向后兼容，稍后填充）
            }
            
            # ========== 收集所有控件数据（向后兼容） ==========
            # 如果存在control_manager，保存其控件（向后兼容旧版本）
            if self.control_manager:
                # 遍历控件管理器中的所有控件ID
                for control_id in self.control_manager.get_all_control_ids():
                    # 获取控件对象
                    control = self.control_manager.get_control(control_id)
                    # 如果控件存在，提取其数据
                    if control:
                        # ========== 获取事件绑定字典 ==========
                        # 从控件对象中获取事件绑定字典
                        # 如果控件没有events属性，初始化为空字典
                        events_dict = {}
                        if hasattr(control, 'events') and isinstance(control.events, dict):
                            # 复制events字典，确保只保存可序列化的数据
                            for event_id, event_value in control.events.items():
                                # 只保存已绑定的事件（值为True的事件）
                                # 确保事件ID是字符串类型，便于JSON序列化
                                if event_value:  # 如果值为True或非空，表示事件已绑定
                                    events_dict[str(event_id)] = True
                        
                        # 构建控件数据字典
                        control_data = {
                            "id": control.id,  # 控件ID
                            "control_type": control.control_type,  # 控件类型（如Button、Label等）
                        "x": control.x,  # 控件X坐标
                        "y": control.y,  # 控件Y坐标
                        "width": control.width,  # 控件宽度
                        "height": control.height,  # 控件高度
                        "z_order": control.z_order,  # 控件层级顺序
                        "properties": control.properties,  # 控件属性字典
                        "events": events_dict  # 控件事件绑定字典（已处理确保可序列化）
                    }
                    # 将控件数据添加到项目数据的控件列表中
                    project_data["controls"].append(control_data)
            
            # 保存到文件
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(project_data, f, ensure_ascii=False, indent=2)
            
            # 更新项目状态
            self.project_file = file_path
            self.project_name = os.path.splitext(os.path.basename(file_path))[0]
            self.project_modified = False
            
            return True
            
        except Exception as e:
            print(f"保存项目失败: {e}")
            return False
    
    def load_project(self, file_path: str) -> bool:
        """加载项目
        
        Args:
            file_path: 项目文件路径
            
        Returns:
            bool: 是否成功加载
        """
        try:
            # 检查文件是否存在
            if not os.path.exists(file_path):
                return False
            
            # 读取项目数据
            with open(file_path, 'r', encoding='utf-8') as f:
                project_data = json.load(f)
            
            # 加载项目属性
            self.project_name = project_data.get("name", "未命名项目")
            self.canvas_width = project_data.get("canvas_width", 800)
            self.canvas_height = project_data.get("canvas_height", 520)
            self.window_title = project_data.get("window_title", "我的页面")
            self.window_avatar = project_data.get("window_avatar", "👤")
            self.window_name = project_data.get("window_name", "GeneratedWindow")
            self.event_function_file = project_data.get("event_function_file", "")
            
            # ========== 加载窗口类管理器数据 ==========
            if "window_class_manager" in project_data:
                # 新版本：使用窗口类管理器
                self.window_class_manager = WindowClassManager.from_dict(project_data["window_class_manager"])
                # 更新当前窗口类的control_manager（向后兼容）
                current_window = self.window_class_manager.get_current_window_class()
                if current_window and self.control_manager:
                    current_window.control_manager = self.control_manager
            else:
                # 旧版本：从controls加载，创建默认窗口类
                if self.control_manager:
                    # 先保存旧控件数据
                    old_controls = list(self.control_manager.controls.values())
                    self.control_manager.clear_all()
                else:
                    old_controls = []
                
                # 创建默认窗口类
                default_window = self.window_class_manager.add_window_class(self.window_name)
                if default_window:
                    # 恢复控件到默认窗口类
                    if self.control_manager:
                        default_window.control_manager = self.control_manager
                    # 从旧控件数据恢复
                    for control in old_controls:
                        default_window.control_manager.controls[control.id] = control
            
            # ========== 加载控件数据（向后兼容旧版本） ==========
            # 从项目数据中获取控件列表，如果不存在则使用空列表
            controls_data = project_data.get("controls", [])
            # 遍历控件数据列表，逐个创建控件对象
            for control_data in controls_data:
                # 获取控件类型：优先从"control_type"字段获取，如果不存在则从"type"字段获取（兼容不同版本）
                control_type = control_data.get("control_type") or control_data.get("type")
                # 检查控件类型是否存在
                if control_type:
                    # 如果控件类型存在，创建控件对象
                    control = Control(
                        control_type=control_type,  # 控件类型
                        x=control_data.get("x", 0),  # X坐标，如果不存在则默认为0
                        y=control_data.get("y", 0),  # Y坐标，如果不存在则默认为0
                        width=control_data.get("width", 100),  # 宽度，如果不存在则默认为100
                        height=control_data.get("height", 30)  # 高度，如果不存在则默认为30
                    )
                    # 获取控件ID（用于恢复控件的唯一标识符）
                    cid = control_data.get("id")
                    # 检查控件ID是否存在
                    if cid:
                        # 如果存在，使用保存的ID替换自动生成的ID（保持控件的唯一性）
                        control.id = cid
                    # 设置控件的Z轴顺序（层级顺序），如果不存在则默认为0
                    control.z_order = control_data.get("z_order", 0)
                    # 设置控件的属性字典，如果不存在则使用空字典
                    control.properties = control_data.get("properties", {})
                    # ========== 设置控件的事件绑定字典 ==========
                    # 从控件数据中获取事件绑定字典，如果不存在则使用空字典
                    events_data = control_data.get("events", {})
                    # 确保events属性存在且是字典类型
                    if not isinstance(events_data, dict):
                        events_data = {}
                    # 确保control.events属性存在且是字典类型
                    if not hasattr(control, 'events') or not isinstance(control.events, dict):
                        control.events = {}
                    # 将加载的事件绑定数据复制到控件对象
                    # 只恢复已绑定的事件（值为True的事件）
                    control.events = {}
                    for event_id, event_value in events_data.items():
                        # 只恢复值为True的事件绑定
                        if event_value:
                            control.events[str(event_id)] = True
                    # 将控件添加到控件管理器的控件字典中，使用控件ID作为键
                    self.control_manager.controls[control.id] = control
            
            # 更新项目状态
            self.project_file = file_path
            self.project_modified = False
            
            return True
            
        except Exception as e:
            print(f"加载项目失败: {e}")
            return False
    
    def set_modified(self, modified: bool = True):
        """设置项目修改状态
        
        Args:
            modified: 是否已修改
        """
        self.project_modified = modified
    
    def is_modified(self) -> bool:
        """检查项目是否已修改
        
        Returns:
            bool: 是否已修改
        """
        return self.project_modified
    
    def get_project_file(self) -> Optional[str]:
        """获取当前项目文件路径
        
        Returns:
            str: 项目文件路径，如果没有则返回None
        """
        return self.project_file
    
    def get_project_name(self) -> str:
        """获取项目名称
        
        Returns:
            str: 项目名称
        """
        return self.project_name
    
    def set_canvas_size(self, width: int, height: int):
        """设置画布大小
        
        Args:
            width: 画布宽度
            height: 画布高度
        """
        self.canvas_width = width
        self.canvas_height = height
        self.set_modified()
    
    def get_canvas_size(self) -> tuple:
        """获取画布大小
        
        Returns:
            tuple: (宽度, 高度)
        """
        return (self.canvas_width, self.canvas_height)
    
    def set_window_title(self, title: str):
        """设置窗口标题
        
        Args:
            title: 窗口标题
        """
        self.window_title = title
        self.set_modified()
    
    def get_window_title(self) -> str:
        """获取窗口标题
        
        Returns:
            str: 窗口标题
        """
        return self.window_title
    
    def set_window_avatar(self, avatar: str):
        """设置窗口头像
        
        Args:
            avatar: 窗口头像
        """
        self.window_avatar = avatar
        self.set_modified()
    
    def get_window_avatar(self) -> str:
        """获取窗口头像
        
        Returns:
            str: 窗口头像
        """
        return self.window_avatar
    
    def set_event_function_file(self, content: str):
        """设置事件函数文件内容
        
        Args:
            content: 事件函数文件内容
        """
        self.event_function_file = content
        self.set_modified()
    
    def get_event_function_file(self) -> str:
        """获取事件函数文件内容
        
        Returns:
            str: 事件函数文件内容
        """
        return self.event_function_file
    
    def set_window_name(self, name: str):
        """设置窗口名称
        
        Args:
            name: 窗口名称
        """
        self.window_name = name
        self.set_modified()
    
    def get_window_name(self) -> str:
        """获取窗口名称
        
        Returns:
            str: 窗口名称
        """
        return self.window_name
