#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
预览窗口模块
实现内置预览功能，实时预览设计画布上的界面效果
可以在不生成完整代码的情况下快速查看界面效果
"""

# 导入类型提示
from typing import Dict, Any, List
# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QDialog,  # 对话框基类
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QLabel,  # 标签
    QWidget,  # 基础窗口部件
    QMenuBar,  # 菜单栏
    QMenu,  # 菜单
    QTabWidget  # 选项卡控件（注意：重复导入了QHBoxLayout，已删除）
)
# 导入PySide6核心功能
from PySide6.QtCore import Qt  # Qt命名空间和常量
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QAction,  # 动作类
    QIcon,  # 图标
    QPixmap  # 像素图
)

# 导入自定义模块
from .control_pyside import ControlManager, Control  # 控件管理器和控件类
from .control_library_pyside import ControlLibrary  # 控件库


class BuiltInPreviewWindow(QDialog):
    """
    内置预览窗口类
    提供实时预览功能，展示设计画布上的界面效果
    支持菜单栏、选项卡页面等复杂布局的预览
    """
    def __init__(self, parent=None, control_manager: ControlManager = None, design_canvas=None):
        """
        初始化预览窗口
        
        Args:
            parent: 父窗口对象
            control_manager: 控件管理器实例，包含所有控件的设计信息
            design_canvas: 设计画布实例，包含窗口属性和画布配置
        """
        # 调用父类构造函数
        super().__init__(parent)
        # 保存控件管理器引用
        self.control_manager = control_manager
        # 保存设计画布引用
        self.design_canvas = design_canvas
        # 菜单栏组件（如果存在下拉菜单）
        self.menu_bar = None
        # 选项卡控件（如果存在多个页面）
        self.tab_widget = None
        # 中央部件（用于放置控件）
        self.central_widget = None
        # 标题栏部件（预留）
        self.header_widget = None
        # 标题栏标签（预留）
        self.header_label = None
        # 使用FramelessWindow的内容区域
        content_widget = self.contentWidget()
        # 清除可能存在的布局
        if content_widget.layout():
            from PySide6.QtWidgets import QWidget
            QWidget().setLayout(content_widget.layout())
        
        # 创建垂直布局作为主布局
        self._layout = QVBoxLayout(content_widget)
        # 设置布局边距为0（无边框）
        self._layout.setContentsMargins(0, 0, 0, 0)
        # 设置布局间距为0
        self._layout.setSpacing(0)
        # 构建用户界面
        self._build_ui()

    def _build_ui(self):
        self._clear_ui()
        w = getattr(self.design_canvas, "canvas_width", 800)
        h = getattr(self.design_canvas, "canvas_height", 520)
        self.resize(w, h)
        title = getattr(self.design_canvas, "window_title", "预览窗口")
        self.setWindowTitle(title)
        icon_path = getattr(self.design_canvas, "window_icon", "")
        try:
            if isinstance(icon_path, str) and icon_path:
                self.setWindowIcon(QIcon(icon_path))
        except Exception:
            pass
        frame_h = getattr(self.design_canvas, "title_bar_height", 0)
        menu_h = 24 if (getattr(self.design_canvas, "window_dropdown_menus", []) or []) else 0
        tab_h_const = 24
        self.header_widget = None
        self.header_label = None
        dropdowns = getattr(self.design_canvas, "window_dropdown_menus", []) or []
        if dropdowns:
            mb = QMenuBar(self)
            # 固定菜单栏高度与画布一致
            try:
                mb.setFixedHeight(menu_h)
            except Exception:
                pass
            menu_count = 0
            for m in dropdowns:
                if isinstance(m, dict):
                    title = m.get("title", "菜单")
                    menu = self._build_dropdown_menu(m)
                    if not menu:
                        menu = QMenu(title, self)
                    mb.addMenu(menu)
                    menu_count += 1
            if menu_count > 0:
                self.menu_bar = mb
                self._layout.addWidget(self.menu_bar)
        pages = getattr(self.design_canvas, "window_menus", []) or []
        if pages:
            self.tab_widget = QTabWidget(self)
            self._layout.addWidget(self.tab_widget)
            content_h = max(0, h)
            for pg in pages:
                page_widget = QWidget(self.tab_widget)
                page_widget.setFixedSize(w, content_h)
                self.tab_widget.addTab(page_widget, pg)
                self._populate_controls(page_widget, pg)
            try:
                self.tab_widget.setCurrentIndex(getattr(self.design_canvas, 'active_index', 0))
            except Exception:
                pass
            try:
                # 总高度 = 内容 + 标签栏高度
                self.tab_widget.setFixedSize(w, content_h + tab_h_const)
            except Exception:
                pass
        else:
            self.central_widget = QWidget(self)
            self._layout.addWidget(self.central_widget)
            content_h = max(0, h)
            self.central_widget.setFixedSize(w, content_h)
            self._populate_controls(self.central_widget, None)

        total_h = h + (menu_h if (getattr(self.design_canvas, "window_dropdown_menus", []) or []) else 0) + (tab_h_const if pages else 0)
        self.setFixedSize(w, total_h)

    def _clear_ui(self):
        while self._layout.count():
            item = self._layout.takeAt(0)
            w = item.widget()
            if w:
                w.deleteLater()
        self.menu_bar = None
        self.tab_widget = None
        self.central_widget = None

    def _populate_controls(self, parent: QWidget, page_name: Any):
        controls = sorted(self.control_manager.controls.values(), key=lambda c: c.z_order)
        for c in controls:
            if page_name is not None:
                p = c.properties.get("page", None)
                if p != page_name:
                    continue
            self._create_widget_for_control(c, parent, page_name)

    def _create_widget_for_control(self, control: Control, parent: QWidget, page_name: Any):
        cls = ControlLibrary.get_control_class(control.control_type)
        if not cls:
            return
        try:
            w = cls(parent)
        except Exception:
            return
        try:
            frame_h = getattr(self.design_canvas, "title_bar_height", 0)
            menu_h = 24 if (getattr(self.design_canvas, "window_dropdown_menus", []) or []) else 0
            tab_h_const = 24 if page_name is not None else 0
            y_off = frame_h + menu_h + tab_h_const
            y_pos = control.y - y_off
            if y_pos < 0:
                y_pos = 0
            w.setGeometry(control.x, y_pos, control.width, control.height)
            w.show()
        except Exception:
            pass
        # ========== 遍历控件的所有属性并设置 ==========
        # 遍历控件属性字典中的所有属性（键值对）
        # control.properties or {}：如果properties为None则使用空字典，避免None错误
        for pn, pv in (control.properties or {}).items():
            # 尝试设置属性，如果失败则跳过
            try:
                # ========== 处理文本属性 ==========
                # 检查属性名是否为"text"（通用文本属性）
                if pn == "text":
                    # 如果是文本属性，使用getattr动态调用setText方法，将属性值转换为字符串
                    getattr(w, "setText")(str(pv))
                
                # ========== 处理占位符文本（LineEdit控件） ==========
                # 检查控件类型是否为LineEdit且属性名为placeholder_text
                elif control.control_type == "LineEdit" and pn == "placeholder_text":
                    # 如果是LineEdit的占位符文本，调用setPlaceholderText方法
                    w.setPlaceholderText(str(pv))
                
                # ========== 处理纯文本内容（TextEdit控件） ==========
                # 检查控件类型是否为TextEdit且属性名为plain_text
                elif control.control_type == "TextEdit" and pn == "plain_text":
                    # 如果是TextEdit的纯文本内容，调用setPlainText方法
                    w.setPlainText(str(pv))
                
                # ========== 处理标题属性（GroupBox控件） ==========
                # 检查属性名是否为"title"且控件类型为GroupBox
                elif pn == "title" and control.control_type == "GroupBox":
                    # 如果是GroupBox的标题，使用getattr动态调用setTitle方法
                    getattr(w, "setTitle")(str(pv))
                
                # ========== 处理选中状态（CheckBox和RadioButton） ==========
                # 检查属性名是否为"checked"且控件类型为CheckBox或RadioButton
                elif pn == "checked" and control.control_type in ["CheckBox", "RadioButton"]:
                    # 如果是选中状态属性，调用setChecked方法，将属性值转换为布尔值
                    getattr(w, "setChecked")(bool(pv))
                
                # ========== 处理值属性（Slider、ProgressBar、SpinBox、DoubleSpinBox） ==========
                # 检查属性名是否为"value"且控件类型为数值控件之一
                elif pn == "value" and control.control_type in ["Slider", "ProgressBar", "SpinBox", "DoubleSpinBox"]:
                    # 如果是值属性，调用setValue方法（直接使用属性值，不转换类型）
                    getattr(w, "setValue")(pv)
                
                # ========== 处理最小值属性（数值控件） ==========
                # 检查属性名是否为"minimum"且控件类型为数值控件之一
                elif pn == "minimum" and control.control_type in ["Slider", "ProgressBar", "SpinBox", "DoubleSpinBox"]:
                    # 如果是最小值属性，调用setMinimum方法
                    getattr(w, "setMinimum")(pv)
                
                # ========== 处理最大值属性（数值控件） ==========
                # 检查属性名是否为"maximum"且控件类型为数值控件之一
                elif pn == "maximum" and control.control_type in ["Slider", "ProgressBar", "SpinBox", "DoubleSpinBox"]:
                    # 如果是最大值属性，调用setMaximum方法
                    getattr(w, "setMaximum")(pv)
                
                # ========== 处理方向属性（Slider控件） ==========
                # 检查控件类型是否为Slider且属性名为orientation
                elif control.control_type == "Slider" and pn == "orientation":
                    # 如果是Slider的方向属性，需要特别处理
                    try:
                        # 将属性值转换为字符串，判断是水平还是垂直方向
                        # 如果值为"Horizontal"则使用Qt.Horizontal，否则使用Qt.Vertical
                        w.setOrientation(Qt.Horizontal if str(pv) == "Horizontal" else Qt.Vertical)
                    except Exception:
                        # 如果设置方向失败，忽略异常
                        pass
                
                # ========== 处理启用状态 ==========
                # 检查属性名是否为"enabled"（通用属性）
                elif pn == "enabled":
                    # 如果是启用状态属性，调用setEnabled方法，将属性值转换为布尔值
                    getattr(w, "setEnabled")(bool(pv))
                
                # ========== 处理可见状态 ==========
                # 检查属性名是否为"visible"（通用属性）
                elif pn == "visible":
                    # 如果是可见状态属性，调用setVisible方法，将属性值转换为布尔值
                    getattr(w, "setVisible")(bool(pv))
                
                # ========== 处理字体大小 ==========
                # 检查属性名是否为"font_size"（通用属性）
                elif pn == "font_size":
                    # 如果是字体大小属性，需要获取字体对象并修改
                    # 获取控件当前字体对象
                    f = w.font()
                    # 设置字体的点大小（将属性值转换为整数）
                    f.setPointSize(int(pv))
                    # 将修改后的字体应用到控件
                    w.setFont(f)
                
                # ========== 处理字体加粗 ==========
                # 检查属性名是否为"font_bold"（通用属性）
                elif pn == "font_bold":
                    # 如果是字体加粗属性，需要获取字体对象并修改
                    # 获取控件当前字体对象
                    f = w.font()
                    # 设置字体是否加粗（将属性值转换为布尔值）
                    f.setBold(bool(pv))
                    # 将修改后的字体应用到控件
                    w.setFont(f)
                
                # ========== 处理对齐方式（Label控件） ==========
                # 检查属性名是否为"alignment"且控件类型为Label
                elif pn == "alignment" and control.control_type == "Label":
                    # 如果是Label的对齐方式属性，根据属性值设置不同的对齐方式
                    # 检查对齐值是否为"center"（居中）
                    if pv == "center":
                        # 设置为居中对齐
                        w.setAlignment(Qt.AlignCenter)
                    # 检查对齐值是否为"left"（左对齐）
                    elif pv == "left":
                        # 设置为左对齐
                        w.setAlignment(Qt.AlignLeft)
                    # 检查对齐值是否为"right"（右对齐）
                    elif pv == "right":
                        # 设置为右对齐
                        w.setAlignment(Qt.AlignRight)
                elif control.control_type == "TabWidget" and pn == "tabs":
                    try:
                        w.clear()
                    except Exception:
                        pass
                    for tab in control.properties.get("tabs", []) or []:
                        w.addTab(QWidget(), str(tab))
                elif control.control_type == "StackedWidget" and pn == "pages":
                    try:
                        w.clear()
                    except Exception:
                        pass
                    for _ in control.properties.get("pages", []) or []:
                        w.addWidget(QWidget())
                elif control.control_type == "StackedWidget" and pn == "current_index":
                    w.setCurrentIndex(int(pv))
                elif pn == "items" and control.control_type == "ComboBox":
                    w.clear()
                    for it in control.properties.get("items", []) or []:
                        w.addItem(str(it))
                    try:
                        idx = int(control.properties.get("current_index", 0))
                        w.setCurrentIndex(idx)
                    except Exception:
                        pass
                elif pn == "items" and control.control_type == "ListWidget":
                    w.clear()
                    for it in control.properties.get("items", []) or []:
                        w.addItem(str(it))
                elif control.control_type == "DateEdit" and pn == "display_format":
                    w.setDisplayFormat(str(pv))
                elif control.control_type == "DateEdit" and pn == "calendar_popup":
                    w.setCalendarPopup(bool(pv))
                elif control.control_type == "TimeEdit" and pn == "display_format":
                    w.setDisplayFormat(str(pv))
                elif control.control_type == "DateTimeEdit" and pn == "display_format":
                    w.setDisplayFormat(str(pv))
                elif control.control_type == "TreeWidget" and pn == "header_labels":
                    w.setHeaderLabels(control.properties.get("header_labels", []) or [])
                elif control.control_type == "TableWidget" and pn == "row_count":
                    w.setRowCount(int(pv))
                elif control.control_type == "TableWidget" and pn == "column_count":
                    w.setColumnCount(int(pv))
                elif control.control_type == "TableWidget" and pn == "headers":
                    w.setHorizontalHeaderLabels(control.properties.get("headers", []) or [])
                elif control.control_type == "ImageBox" and pn == "image_path":
                    try:
                        w.setPixmap(QPixmap(str(pv)))
                    except Exception:
                        pass
                elif control.control_type == "ImageBox" and pn == "scaled":
                    w.setScaledContents(bool(pv))
            except Exception:
                pass

    def set_data(self, control_manager: ControlManager, design_canvas):
        self.control_manager = control_manager
        self.design_canvas = design_canvas
        self._build_ui()

    def _build_dropdown_menu(self, m: dict):
        menu = QMenu(m.get("title", "菜单"), self)
        stack = [menu]
        for it in m.get("items", []) or []:
            label = it.get("label", "")
            if label.strip() in ["-", "—"]:
                stack[-1].addSeparator()
                continue
            level = max(0, int(it.get("level", 0)))
            while len(stack) - 1 > level:
                stack.pop()
            while len(stack) - 1 < level:
                sm = QMenu("", self)
                stack[-1].addMenu(sm)
                stack.append(sm)
            act = QAction(label, self)
            sc = it.get("shortcut", "(无)")
            if sc and sc != "(无)":
                act.setShortcut(sc)
            if it.get("checked", False):
                act.setCheckable(True)
                act.setChecked(True)
            if not it.get("allow", True):
                act.setEnabled(False)
            if not it.get("visible", True):
                act.setVisible(False)
            stack[-1].addAction(act)
        return menu if menu.actions() else None