"""
Monaco Editor 代码编辑器模块
使用 VSCode 的 Monaco Editor 作为代码编辑器
基于 QWebEngineView 实现
"""

import os
import json
from pathlib import Path
from PySide6.QtWidgets import QWidget, QScrollBar, QMenu, QApplication, QInputDialog
from PySide6.QtWebEngineWidgets import QWebEngineView
from PySide6.QtCore import Qt, Signal, QUrl, QTimer
from PySide6.QtGui import QFont, QAction, QClipboard

# 导入控件库，用于代码补全
from .control_library_pyside import ControlLibrary
# 导入Monaco配置模块
from .monaco_config import get_monaco_loader_url, get_monaco_vs_path


class MonacoEditorWidget(QWidget):
    """
    Monaco Editor 代码编辑器组件
    使用 QWebEngineView 加载 Monaco Editor，提供与 LineNumberedCodeEdit 兼容的接口
    """
    
    # 定义信号，与 QPlainTextEdit 兼容
    textChanged = Signal()  # 文本变化信号
    run_code_requested = Signal(str)  # 运行代码请求信号
    
    def __init__(self, parent=None):
        """
        初始化 Monaco Editor 组件
        
        Args:
            parent: 父窗口部件
        """
        super().__init__(parent)
        
        # 创建 WebEngineView
        self._web_view = QWebEngineView(self)
        # 启用自定义右键菜单
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self._show_context_menu)
        
        # 设置布局
        from PySide6.QtWidgets import QVBoxLayout
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)
        layout.addWidget(self._web_view)
        
        # 确保组件有最小尺寸
        self.setMinimumSize(200, 150)
        
        # 初始化状态
        self._is_ready = False
        self._pending_value = None
        self._read_only = False
        
        # 从设置加载编辑器配置
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        self._font_family = settings.value("editor/font", "Consolas", type=str)
        self._font_size = settings.value("editor/font_size", 14, type=int)
        self._theme = settings.value("editor/theme", "vs", type=str)
        self._language = settings.value("editor/language", "python", type=str)
        
        self._pending_theme = None  # 待应用的主题
        self._pending_language = None  # 待应用的语言
        self._auto_load_settings = True  # 是否自动从设置加载主题和语言
        self._external_highlights = []
        self._cached_text = ""  # 缓存当前文本
        self._text_change_timer = QTimer(self)  # 用于延迟触发 textChanged 信号
        self._text_change_timer.setSingleShot(True)
        self._text_change_timer.timeout.connect(self._emit_text_changed)
        
        # 滚动位置缓存
        self._scroll_position = 0
        self._horizontal_scroll_position = 0
        
        # 直接使用内联 HTML，避免本地文件加载时的跨域问题
        # 这样可以确保 Monaco Editor 的 CDN 资源能够正常加载
        self._load_inline_html()
        
        # 连接页面加载完成信号
        self._web_view.page().loadFinished.connect(self._on_page_loaded)
        
        # 设置 JavaScript 回调
        self._setup_javascript_bridge()
        
        # 初始化补全关键词
        self._init_completion_keywords()
    
    def _load_inline_html(self):
        """加载内联 HTML（如果文件不存在时使用）"""
        # 获取Monaco Editor路径配置
        loader_url = get_monaco_loader_url()
        vs_path = get_monaco_vs_path()
        
        # 使用当前配置的字体、主题和语言
        font_family_js = self._font_family.replace("'", "\\'").replace('"', '\\"')  # 转义引号
        font_size_js = self._font_size
        theme_js = self._theme.replace("'", "\\'").replace('"', '\\"')  # 转义引号
        language_js = self._language.replace("'", "\\'").replace('"', '\\"')  # 转义引号
        
        # 检查是否使用本地路径，如果是，需要设置base URL
        base_url = None
        relative_loader_path = loader_url  # 默认使用原始路径
        
        if loader_url.startswith('file://'):
            # 从loader_url提取基础路径（loader.js所在的目录）
            # 例如: file:///D:/.../monaco-editor/min/vs/loader.js
            # base_url应该是: file:///D:/.../monaco-editor/min/vs/
            try:
                # 移除 loader.js 部分，保留目录路径
                if loader_url.endswith('/loader.js'):
                    base_path = loader_url[:-10]  # 移除 '/loader.js'
                elif loader_url.endswith('loader.js'):
                    base_path = loader_url[:-9]   # 移除 'loader.js'
                else:
                    # 如果格式不对，使用 vs_path
                    base_path = vs_path
                
                # 确保base_path以/结尾
                if not base_path.endswith('/'):
                    base_path += '/'
                
                # 创建QUrl对象作为base URL
                base_url = QUrl(base_path)
                # 将绝对路径改为相对路径
                relative_loader_path = 'loader.js'
                #print(f"设置Monaco Editor base URL: {base_url.toString()}, 使用相对路径: {relative_loader_path}")
                print("正在加载代码编辑框")
            except Exception as e:
                print(f"设置base URL失败: {e}，使用绝对路径")
                import traceback
                traceback.print_exc()
        
        # 优先从文件加载模板，避免转义问题
        template_path = Path(__file__).parent.parent / "ui" / "monaco_editor_template.html"
        if template_path.exists():
            try:
                html_content = template_path.read_text(encoding='utf-8')
                # 替换模板中的CDN路径为本地路径或相对路径
                html_content = html_content.replace(
                    'https://cdn.jsdelivr.net/npm/monaco-editor@0.55.1/min/vs/loader.js',
                    relative_loader_path  # 使用相对路径或原始路径
                )
                html_content = html_content.replace(
                    "'https://cdn.jsdelivr.net/npm/monaco-editor@0.55.1/min/vs'",
                    f"'{vs_path}'"
                )
                html_content = html_content.replace(
                    '"https://cdn.jsdelivr.net/npm/monaco-editor@0.55.1/min/vs"',
                    f'"{vs_path}"'
                )
                
                # 替换模板中的配置值为当前设置
                # 使用当前配置的字体、主题和语言
                font_family_js = self._font_family.replace("'", "\\'").replace('"', '\\"')
                font_size_js = self._font_size
                theme_js = self._theme.replace("'", "\\'").replace('"', '\\"')
                language_js = self._language.replace("'", "\\'").replace('"', '\\"')
                
                # 替换语言
                html_content = html_content.replace(
                    "language: 'python',",
                    f"language: '{language_js}',"
                )
                # 替换主题
                html_content = html_content.replace(
                    "theme: 'vs',",
                    f"theme: '{theme_js}',"
                )
                # 替换字体大小
                html_content = html_content.replace(
                    "fontSize: 14,",
                    f"fontSize: {font_size_js},"
                )
                # 替换字体族
                html_content = html_content.replace(
                    "fontFamily: 'Consolas, Courier New, monospace',",
                    f"fontFamily: '{font_family_js}',"
                )
                
                # 如果使用本地路径，提供base URL；否则使用默认base URL
                if base_url:
                    self._web_view.setHtml(html_content, base_url)
                else:
                    self._web_view.setHtml(html_content)
                return
            except Exception as e:
                print(f"加载Monaco Editor模板文件失败: {e}，使用内联HTML")
        
        # 如果文件加载失败，使用内联HTML（简化版本，避免转义问题）
        # 使用字符串格式化而不是f-string，避免JavaScript花括号转义问题
        html_content = """<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        body { margin: 0; padding: 0; overflow: hidden; }
        #container { width: 100%; height: 100vh; }
    </style>
</head>
<body>
    <div id="container"></div>
    <script src="{}"></script>
    <script>
        // 等待 loader.js 加载完成
        if (typeof require === 'undefined') {
            // 如果 require 未定义，等待加载
            var checkRequire = setInterval(function() {
                if (typeof require !== 'undefined') {
                    clearInterval(checkRequire);
                    initializeMonaco();
                }
            }, 50);
        } else {
            initializeMonaco();
        }
        
        function initializeMonaco() {
            require.config({ paths: { vs: '{}' } });
            require(['vs/editor/editor.main'], function () {
            window.editor = monaco.editor.create(document.getElementById('container'), {{
                value: '',
                language: '{}',
                theme: '{}',
                automaticLayout: true,
                fontSize: {},
                fontFamily: '{}',
                minimap: {{ enabled: true }},
                scrollBeyondLastLine: false,
                wordWrap: 'off',
                lineNumbers: 'on',
                renderLineHighlight: 'all',
                selectOnLineNumbers: true,
                roundedSelection: false,
                readOnly: false,
                cursorStyle: 'line',
                suggestOnTriggerCharacters: true,
                acceptSuggestionOnEnter: 'on',
                tabCompletion: 'on',
                wordBasedSuggestions: 'off',
                quickSuggestions: {{ other: true, comments: false, strings: false }}
            }});
            window.editor.onDidChangeModelContent(function() {
                if (window.onContentChanged) {
                    window.onContentChanged(window.editor.getValue());
                }
            });
            window.editor.onDidChangeCursorPosition(function(e) {
                if (window.onCursorPositionChanged) {
                    window.onCursorPositionChanged({
                        line: e.position.lineNumber,
                        column: e.position.column
                    });
                }
            });
            
            // 注册代码补全提供者
            monaco.languages.registerCompletionItemProvider('python', {
                provideCompletionItems: function(model, position) {
                    var word = model.getWordUntilPosition(position);
                    var range = {
                        startLineNumber: position.lineNumber,
                        endLineNumber: position.lineNumber,
                        startColumn: word.startColumn,
                        endColumn: word.endColumn
                    };
                    
                    // Python关键字
                    var pythonKeywords = [
                        { label: 'def', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'def ', documentation: '定义函数' },
                        { label: 'class', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'class ', documentation: '定义类' },
                        { label: 'if', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'if ', documentation: '条件语句' },
                        { label: 'elif', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'elif ', documentation: '否则如果' },
                        { label: 'else', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'else:', documentation: '否则' },
                        { label: 'for', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'for ', documentation: '循环语句' },
                        { label: 'while', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'while ', documentation: '当循环' },
                        { label: 'import', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'import ', documentation: '导入模块' },
                        { label: 'from', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'from ', documentation: '从模块导入' },
                        { label: 'return', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'return ', documentation: '返回值' },
                        { label: 'self', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'self', documentation: '类实例自身' },
                        { label: 'super', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'super()', documentation: '调用父类' },
                        { label: 'True', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'True', documentation: '布尔真值' },
                        { label: 'False', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'False', documentation: '布尔假值' },
                        { label: 'None', kind: monaco.languages.CompletionItemKind.Keyword, insertText: 'None', documentation: '空值' },
                    ];
                    
                    // PySide6常用类
                    var pyside6Classes = [
                        { label: 'QApplication', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QApplication', documentation: 'Qt应用程序类' },
                        { label: 'QMainWindow', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QMainWindow', documentation: '主窗口类' },
                        { label: 'QWidget', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QWidget', documentation: '窗口部件基类' },
                        { label: 'QPushButton', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QPushButton', documentation: '按钮控件' },
                        { label: 'QLabel', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QLabel', documentation: '标签控件' },
                        { label: 'QLineEdit', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QLineEdit', documentation: '单行输入框' },
                        { label: 'QTextEdit', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QTextEdit', documentation: '多行文本编辑器' },
                        { label: 'QVBoxLayout', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QVBoxLayout', documentation: '垂直布局' },
                        { label: 'QHBoxLayout', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QHBoxLayout', documentation: '水平布局' },
                        { label: 'QMessageBox', kind: monaco.languages.CompletionItemKind.Class, insertText: 'QMessageBox', documentation: '消息框' },
                    ];
                    
                    // 控件类型（从Python获取）
                    var controlTypes = [];
                    if (window.controlTypes && Array.isArray(window.controlTypes)) {
                        controlTypes = window.controlTypes.map(function(ct) {
                            return {
                                label: ct,
                                kind: monaco.languages.CompletionItemKind.Class,
                                insertText: ct,
                                documentation: '控件类型: ' + ct
                            };
                        });
                    }
                    
                    var suggestions = pythonKeywords.concat(pyside6Classes).concat(controlTypes);
                    
                    return { suggestions: suggestions };
                },
                triggerCharacters: ['.', '(', ' ']
            });
            
            // 注册悬停提示提供者
            monaco.languages.registerHoverProvider('python', {
                provideHover: function(model, position) {
                    var word = model.getWordAtPosition(position);
                    if (!word) {
                        return null;
                    }
                    
                    // 提供常见Python和PySide6的提示信息
                    var hoverInfo = {
                        'QApplication': 'Qt应用程序类，管理应用程序的控制流和主要设置',
                        'QMainWindow': '主窗口类，提供带菜单栏、工具栏和状态栏的窗口框架',
                        'QWidget': '所有用户界面对象的基类',
                        'QPushButton': '按钮控件，用户可以点击执行操作',
                        'QLabel': '标签控件，用于显示文本或图片',
                        'QLineEdit': '单行文本输入框',
                        'QTextEdit': '多行富文本编辑器',
                        'QVBoxLayout': '垂直布局管理器',
                        'QHBoxLayout': '水平布局管理器',
                        'QMessageBox': '消息对话框，用于显示信息、警告、错误等',
                        'self': '类实例的引用，指向当前对象',
                        'super': '用于调用父类的方法',
                        'def': '定义函数的关键字',
                        'class': '定义类的关键字',
                        'import': '导入模块的关键字',
                        'from': '从模块导入特定内容的关键字'
                    };
                    
                    var wordText = word.word;
                    if (hoverInfo[wordText]) {
                        return {
                            range: new monaco.Range(
                                position.lineNumber,
                                word.startColumn,
                                position.lineNumber,
                                word.endColumn
                            ),
                            contents: [
                                { value: '**' + wordText + '**' },
                                { value: hoverInfo[wordText] }
                            ]
                        };
                    }
                    
                    return null;
                }
            });
            
            if (window.onEditorReady) {
                window.onEditorReady();
            }
            });
        }
        window.setEditorValue = function(value) {
            if (window.editor) {
                window.editor.setValue(value || '');
            }
        };
        window.getEditorValue = function() {
            if (window.editor) {
                return window.editor.getValue();
            }
            return '';
        };
        window.setEditorReadOnly = function(readOnly) {
            if (window.editor) {
                window.editor.updateOptions({ readOnly: readOnly });
            }
        };
        window.setEditorFont = function(fontFamily, fontSize) {
            if (window.editor) {
                window.editor.updateOptions({
                    fontFamily: fontFamily,
                    fontSize: fontSize
                });
            }
        };
        window.setEditorTheme = function(theme) {
            if (window.editor) {
                monaco.editor.setTheme(theme);
            }
        };
        window.highlightLines = function(lines) {
            if (!window.editor || !lines || lines.length === 0) {
                return;
            }
            var decorations = lines.map(function(line) {
                return {
                    range: new monaco.Range(line, 1, line, 1),
                    options: {
                        isWholeLine: true,
                        className: 'highlighted-line',
                        glyphMarginClassName: 'highlighted-line-glyph'
                    }
                };
            });
            window.editor.deltaDecorations([], decorations);
        };
        window.clearHighlights = function() {
            if (window.editor) {
                window.editor.deltaDecorations([], []);
            }
        };
        window.setExternalHighlights = function(highlights) {
            if (!window.editor || !highlights) {
                return;
            }
            var decorations = highlights.map(function(h) {
                return {
                    range: new monaco.Range(h.startLine, h.startColumn, h.endLine, h.endColumn),
                    options: {
                        inlineClassName: 'external-highlight',
                        className: 'external-highlight-line'
                    }
                };
            });
            window.editor.deltaDecorations([], decorations);
        };
        window.insertTextAtCursor = function(text) {
            try {
                if (!window.editor) {
                    return;
                }
                if (typeof text !== 'string') {
                    text = String(text);
                }
                var model = window.editor.getModel();
                if (!model) {
                    return;
                }
                var position = window.editor.getPosition();
                if (!position) {
                    return;
                }
                var selection = window.editor.getSelection();
                // 如果没有选择，使用当前光标位置创建一个空范围
                var range = selection && !selection.isEmpty() ? selection : 
                            new monaco.Range(position.lineNumber, position.column, position.lineNumber, position.column);
                // 使用标准的executeEdits API
                var edits = [{
                    range: range,
                    text: text
                }];
                // 执行编辑操作
                window.editor.executeEdits('insert', edits);
                // 计算插入后的新位置
                // 使用转义字符串避免Python三引号字符串的转义问题
                var lines = text.split(String.fromCharCode(10));
                var lastLine = lines[lines.length - 1];
                var newLine = range.startLineNumber + lines.length - 1;
                var newColumn = (lines.length === 1) ? 
                               (range.startColumn + lastLine.length) : 
                               (lastLine.length + 1);
                var newPosition = new monaco.Position(newLine, newColumn);
                // 设置光标到新位置
                window.editor.setPosition(newPosition);
                window.editor.revealPositionInCenter(newPosition);
            } catch (e) {
                console.error('Error in insertTextAtCursor:', e);
            }
        };
        window.getCursorPosition = function() {
            if (!window.editor) {
                return { line: 1, column: 1 };
            }
            var position = window.editor.getPosition();
            return {
                line: position ? position.lineNumber : 1,
                column: position ? position.column : 1
            };
        };
        window.setCursorPosition = function(line, column) {
            if (!window.editor) {
                return;
            }
            var position = new monaco.Position(line, column);
            window.editor.setPosition(position);
            window.editor.revealPositionInCenter(position);
        };
        window.getSelectionRange = function() {
            if (!window.editor) {
                return null;
            }
            var selection = window.editor.getSelection();
            if (!selection) {
                return null;
            }
            return {
                startLine: selection.startLineNumber,
                startColumn: selection.startColumn,
                endLine: selection.endLineNumber,
                endColumn: selection.endColumn
            };
        };
    </script>
    <style>
        .highlighted-line { background-color: rgba(235, 235, 255, 0.5) !important; }
        .external-highlight { background-color: rgba(255, 255, 0, 0.3) !important; }
        .external-highlight-line { background-color: rgba(255, 255, 0, 0.2) !important; }
    </style>
</body>
</html>""".format(relative_loader_path, vs_path, language_js, theme_js, font_size_js, font_family_js)
        
        # 如果使用本地路径，提供base URL；否则使用默认base URL
        if base_url:
            self._web_view.setHtml(html_content, base_url)
        else:
            self._web_view.setHtml(html_content)
    
    def _setup_javascript_bridge(self):
        """设置 JavaScript 桥接，用于 Python 和 JavaScript 之间的通信"""
        # 注意：在 PySide6 中，我们使用 runJavaScript 方法来与 JavaScript 通信
        # 不需要使用 javaScriptChannelObject（该方法在 PySide6 中不存在）
        # 文本变化监听通过定时器轮询实现
        pass
    
    def _on_page_loaded(self, success):
        """页面加载完成回调"""
        if success:
            # 等待更长时间确保 JavaScript 和资源加载完成
            # Monaco Editor 需要从本地路径或CDN加载，可能需要更长时间
            QTimer.singleShot(1500, self._initialize_editor)
        else:
            # 如果加载失败，尝试重新加载
            print("Monaco Editor 页面加载失败，尝试使用内联 HTML")
            self._load_inline_html()
    
    def _initialize_editor(self):
        """初始化编辑器"""
        # 设置控件类型到JavaScript环境，用于代码补全
        try:
            control_types_js = json.dumps(self._control_types)
            script = f"window.controlTypes = {control_types_js};"
            self._web_view.page().runJavaScript(script)
        except Exception:
            pass
        
        # 检查编辑器是否准备好
        script = """
        (function() {
            if (window.editor) {
                return true;
            }
            return false;
        })();
        """
        self._web_view.page().runJavaScript(script, self._check_editor_ready)
    
    def _check_editor_ready(self, result):
        """检查编辑器是否准备好"""
        if result:
            self._is_ready = True
            # 应用待处理的设置
            if self._pending_value is not None:
                self.setPlainText(self._pending_value)
                self._pending_value = None
            self.setReadOnly(self._read_only)
            # 字体、主题和语言已经在创建时设置了，这里不需要再次设置
            # 但如果这些设置在创建后发生了变化，仍然需要更新
            # self.setFont(QFont(self._font_family, self._font_size))
            
            # 应用待处理的主题和语言设置（仅在创建后发生变化时）
            if self._pending_theme is not None:
                theme = self._pending_theme
                self._pending_theme = None
                self.setTheme(theme)
            
            if self._pending_language is not None:
                language = self._pending_language
                self._pending_language = None
                self.setLanguage(language)
            
            # 设置内容变化监听
            self._setup_content_listener()
        else:
            # 如果编辑器未准备好，再次尝试初始化（可能是资源加载较慢）
            # 最多尝试 5 次
            if not hasattr(self, '_init_retry_count'):
                self._init_retry_count = 0
            self._init_retry_count += 1
            if self._init_retry_count < 5:
                QTimer.singleShot(1000, self._initialize_editor)
            else:
                print("Monaco Editor 初始化失败，已重试 5 次")
    
    def _setup_content_listener(self):
        """设置内容变化监听"""
        script = """
        (function() {
            if (window.editor && !window._contentListenerSetup) {
                var lastValue = window.editor.getValue();
                window.editor.onDidChangeModelContent(function() {
                    var currentValue = window.editor.getValue();
                    if (currentValue !== lastValue) {
                        lastValue = currentValue;
                        // 通过 URL 变化来通知 Python 端（临时方案）
                        // 更好的方案是使用 QWebChannel
                        if (window.onContentChanged) {
                            window.onContentChanged(currentValue);
                        }
                    }
                });
                window._contentListenerSetup = true;
            }
        })();
        """
        self._web_view.page().runJavaScript(script)
        
        # 设置定期检查文本变化的定时器
        self._text_check_timer = QTimer(self)
        # 使用 lambda 包装，避免槽查找问题
        self._text_check_timer.timeout.connect(lambda: self._check_text_changed())
        self._text_check_timer.start(300)  # 每 300ms 检查一次
    
    def _init_completion_keywords(self):
        """初始化补全关键词"""
        # 收集控件类型
        self._control_types = []
        try:
            cats = ControlLibrary.get_all_categories()
            for cat_id, _ in cats:
                for ct, _info in ControlLibrary.get_controls_by_category(cat_id):
                    if ct:
                        self._control_types.append(ct)
        except Exception:
            pass
        self._control_types = sorted(set(self._control_types))
    
    # ========== 公共接口方法，与 QPlainTextEdit 兼容 ==========
    
    def setPlainText(self, text):
        """
        设置编辑器文本内容
        
        Args:
            text: 文本内容
        """
        if text is None:
            text = ""
        
        # 更新缓存
        self._cached_text = text
        
        if not self._is_ready:
            self._pending_value = text
            return
        
        # 转义 JavaScript 字符串
        # 使用 JSON 编码更安全
        import json
        escaped_text = json.dumps(text)
        script = f"window.setEditorValue({escaped_text});"
        self._web_view.page().runJavaScript(script)
    
    def toPlainText(self):
        """
        获取编辑器文本内容
        
        Returns:
            str: 文本内容
        """
        if not self._is_ready:
            return self._pending_value or ""
        
        # 由于 runJavaScript 是异步的，我们返回缓存值
        # 同时触发一次异步获取以更新缓存
        script = "window.getEditorValue();"
        self._web_view.page().runJavaScript(script, self._on_text_retrieved)
        return self._cached_text
    
    def _on_text_retrieved(self, result):
        """文本获取回调"""
        if result is not None:
            self._cached_text = result
    
    def _check_text_changed(self):
        """检查文本是否变化"""
        if not self._is_ready:
            return
        
        script = "window.getEditorValue();"
        self._web_view.page().runJavaScript(script, self._on_text_changed)
    
    def _on_text_changed(self, result):
        """文本变化回调"""
        if result is not None and result != self._cached_text:
            self._cached_text = result
            # 延迟触发 textChanged 信号，避免频繁触发
            self._text_change_timer.stop()
            self._text_change_timer.start(100)
    
    def _emit_text_changed(self):
        """触发 textChanged 信号"""
        self.textChanged.emit()
    
    def setReadOnly(self, read_only):
        """
        设置只读模式
        
        Args:
            read_only: 是否只读
        """
        self._read_only = read_only
        if self._is_ready:
            script = f"window.setEditorReadOnly({str(read_only).lower()});"
            self._web_view.page().runJavaScript(script)
    
    def isReadOnly(self):
        """
        获取只读状态
        
        Returns:
            bool: 是否只读
        """
        return self._read_only
    
    def setFont(self, font):
        """
        设置字体
        
        Args:
            font: QFont 对象
        """
        self._font_family = font.family()
        self._font_size = font.pointSize()
        if self._is_ready:
            script = f"window.setEditorFont('{self._font_family}', {self._font_size});"
            self._web_view.page().runJavaScript(script)
    
    def font(self):
        """
        获取字体
        
        Returns:
            QFont: 字体对象
        """
        return QFont(self._font_family, self._font_size)
    
    def setLineWrapMode(self, mode):
        """
        设置换行模式（Monaco Editor 通过 wordWrap 选项控制）
        
        Args:
            mode: 换行模式（QPlainTextEdit.NoWrap 等）
        """
        if self._is_ready:
            # QPlainTextEdit.NoWrap = 0
            word_wrap = 'off' if mode == 0 else 'on'
            script = f"""
            if (window.editor) {{
                window.editor.updateOptions({{ wordWrap: '{word_wrap}' }});
            }}
            """
            self._web_view.page().runJavaScript(script)
    
    def highlight_lines(self, line_numbers):
        """
        高亮指定的行号（用于代码高亮）
        
        Args:
            line_numbers: 行号列表（从1开始，Monaco Editor的行号从1开始）
        """
        if not self._is_ready:
            return
        
        # 先清除之前的高亮（无论是否传入新的行号）
        script_clear = "window.clearHighlights();"
        self._web_view.page().runJavaScript(script_clear)
        
        # 如果传入空列表或None，只清除高亮，不应用新的
        if not line_numbers:
            return
        
        # 转换为JavaScript数组，行号从1开始
        import json
        lines_json = json.dumps([int(ln) for ln in line_numbers])
        script = f"window.highlightLines({lines_json});"
        self._web_view.page().runJavaScript(script)
    
    def set_external_highlights(self, selections):
        """
        设置外部高亮（用于代码搜索等）
        
        Args:
            selections: 高亮选择列表（QTextEdit.ExtraSelection 对象列表或行号列表）
        """
        self._externalHighlights = selections or []
        if not self._is_ready:
            return
        
        # 如果selections是行号列表（简单整数列表），使用highlight_lines
        if selections and isinstance(selections[0], int):
            self.highlight_lines(selections)
            return
        
        # 转换 QTextEdit.ExtraSelection 为 Monaco Editor 的装饰
        highlights = []
        for selection in self._externalHighlights:
            cursor = selection.cursor
            if cursor.hasSelection():
                start_pos = cursor.selectionStart()
                end_pos = cursor.selectionEnd()
                # 这里需要将字符位置转换为行列位置
                # 为了简化，我们暂时跳过这个功能
                pass
        
        # 如果有高亮，应用它们
        if highlights:
            import json
            highlights_json = json.dumps(highlights)
            script = f"window.setExternalHighlights({highlights_json});"
            self._web_view.page().runJavaScript(script)
        else:
            script = "window.clearHighlights();"
            self._web_view.page().runJavaScript(script)
    
    # ========== 其他辅助方法 ==========
    
    def setTheme(self, theme):
        """
        设置编辑器主题
        
        Args:
            theme: 主题名称（'vs', 'vs-dark', 'hc-black' 等）
        """
        if self._is_ready:
            script = f"window.setEditorTheme('{theme}');"
            self._web_view.page().runJavaScript(script)
        else:
            # 如果编辑器还没准备好，保存主题设置，等待编辑器准备好后再应用
            self._pending_theme = theme
    
    def setLanguage(self, language):
        """
        设置编辑器语言
        
        Args:
            language: 语言标识（'python', 'javascript', 'typescript' 等）
        """
        if self._is_ready:
            script = f"""
            if (window.editor) {{
                monaco.editor.setModelLanguage(window.editor.getModel(), '{language}');
            }}
            """
            self._web_view.page().runJavaScript(script)
        else:
            # 如果编辑器还没准备好，保存语言设置，等待编辑器准备好后再应用
            self._pending_language = language
    
    def verticalScrollBar(self):
        """
        获取垂直滚动条（兼容 QPlainTextEdit 接口）
        
        Returns:
            MonacoScrollBar: 模拟的滚动条对象
        """
        if not hasattr(self, '_vertical_scrollbar'):
            self._vertical_scrollbar = MonacoScrollBar(self, is_vertical=True)
        return self._vertical_scrollbar
    
    def horizontalScrollBar(self):
        """
        获取水平滚动条（兼容 QPlainTextEdit 接口）
        
        Returns:
            MonacoScrollBar: 模拟的滚动条对象
        """
        if not hasattr(self, '_horizontal_scrollbar'):
            self._horizontal_scrollbar = MonacoScrollBar(self, is_vertical=False)
        return self._horizontal_scrollbar
    
    def _show_context_menu(self, pos):
        """显示右键菜单（参考VSCode设计，全中文）"""
        menu = QMenu(self)
        
        # ========== 编辑操作组 ==========
        # 撤销
        undo_action = QAction("撤销", self)
        undo_action.setShortcut("Ctrl+Z")
        undo_action.triggered.connect(self._undo)
        menu.addAction(undo_action)
        
        # 重做
        redo_action = QAction("重做", self)
        redo_action.setShortcut("Ctrl+Y")
        redo_action.triggered.connect(self._redo)
        menu.addAction(redo_action)
        
        menu.addSeparator()
        
        # 剪切
        cut_action = QAction("剪切", self)
        cut_action.setShortcut("Ctrl+X")
        cut_action.triggered.connect(self._cut)
        menu.addAction(cut_action)
        
        # 复制
        copy_action = QAction("复制", self)
        copy_action.setShortcut("Ctrl+C")
        copy_action.triggered.connect(self._copy)
        menu.addAction(copy_action)
        
        # 粘贴
        paste_action = QAction("粘贴", self)
        paste_action.setShortcut("Ctrl+V")
        paste_action.triggered.connect(self._paste)
        menu.addAction(paste_action)
        
        menu.addSeparator()
        
        # 全选
        select_all_action = QAction("全选", self)
        select_all_action.setShortcut("Ctrl+A")
        select_all_action.triggered.connect(self._select_all)
        menu.addAction(select_all_action)
        
        menu.addSeparator()
        
        # ========== 查找和导航组 ==========
        # 查找
        find_action = QAction("查找", self)
        find_action.setShortcut("Ctrl+F")
        find_action.triggered.connect(self._find)
        menu.addAction(find_action)
        
        # 替换
        replace_action = QAction("替换", self)
        replace_action.setShortcut("Ctrl+H")
        replace_action.triggered.connect(self._replace)
        menu.addAction(replace_action)
        
        # 跳转到行
        goto_line_action = QAction("转到行...", self)
        goto_line_action.setShortcut("Ctrl+G")
        goto_line_action.triggered.connect(self._goto_line)
        menu.addAction(goto_line_action)
        
        menu.addSeparator()
        
        # ========== 代码操作组 ==========
        # 跳转到定义
        go_to_definition_action = QAction("转到定义", self)
        go_to_definition_action.setShortcut("F12")
        go_to_definition_action.triggered.connect(self._go_to_definition)
        menu.addAction(go_to_definition_action)
        
        # 查找所有引用
        find_references_action = QAction("查找所有引用", self)
        find_references_action.setShortcut("Shift+F12")
        find_references_action.triggered.connect(self._find_references)
        menu.addAction(find_references_action)
        
        menu.addSeparator()
        
        # 格式化文档
        format_document_action = QAction("格式化文档", self)
        format_document_action.setShortcut("Shift+Alt+F")
        format_document_action.triggered.connect(self._format_document)
        menu.addAction(format_document_action)
        
        menu.addSeparator()
        
        # ========== 运行和调试组 ==========
        # 运行代码
        run_code_action = QAction("运行代码", self)
        run_code_action.setShortcut("F5")
        run_code_action.triggered.connect(self._run_code)
        menu.addAction(run_code_action)
        
        # 运行选中代码
        run_selection_action = QAction("运行选中代码", self)
        run_selection_action.triggered.connect(self._run_selection)
        menu.addAction(run_selection_action)
        
        menu.addSeparator()
        
        # ========== 其他操作组 ==========
        # 切换注释
        toggle_comment_action = QAction("切换行注释", self)
        toggle_comment_action.setShortcut("Ctrl+/")
        toggle_comment_action.triggered.connect(self._toggle_comment)
        menu.addAction(toggle_comment_action)
        
        # 转换为大写
        to_upper_action = QAction("转换为大写", self)
        to_upper_action.triggered.connect(self._to_upper)
        menu.addAction(to_upper_action)
        
        # 转换为小写
        to_lower_action = QAction("转换为小写", self)
        to_lower_action.triggered.connect(self._to_lower)
        menu.addAction(to_lower_action)
        
        menu.addSeparator()
        
        # 显示命令面板
        command_palette_action = QAction("命令面板...", self)
        command_palette_action.setShortcut("Ctrl+Shift+P")
        command_palette_action.triggered.connect(self._show_command_palette)
        menu.addAction(command_palette_action)
        
        menu.exec(self.mapToGlobal(pos))
    
    def _undo(self):
        """撤销"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.trigger('undo', 'keyboard');
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _redo(self):
        """重做"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.trigger('redo', 'keyboard');
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _cut(self):
        """剪切"""
        if self._is_ready:
            script = """
            if (window.editor) {
                var selection = window.editor.getSelection();
                if (!selection.isEmpty()) {
                    var text = window.editor.getModel().getValueInRange(selection);
                    navigator.clipboard.writeText(text).then(function() {
                        window.editor.executeEdits('cut', [{
                            range: selection,
                            text: ''
                        }]);
                    });
                }
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _copy(self):
        """复制"""
        if self._is_ready:
            script = """
            if (window.editor) {
                var selection = window.editor.getSelection();
                if (!selection.isEmpty()) {
                    var text = window.editor.getModel().getValueInRange(selection);
                    navigator.clipboard.writeText(text);
                }
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _paste(self):
        """粘贴"""
        if self._is_ready:
            script = """
            if (window.editor && navigator.clipboard) {
                navigator.clipboard.readText().then(function(text) {
                    var selection = window.editor.getSelection();
                    window.editor.executeEdits('paste', [{
                        range: selection,
                        text: text
                    }]);
                });
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _select_all(self):
        """全选"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.setSelection(window.editor.getModel().getFullModelRange());
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def insertPlainText(self, text):
        """
        在光标位置插入文本（与QPlainTextEdit兼容）
        
        Args:
            text: 要插入的文本
        """
        if not self._is_ready:
            # 如果编辑器未准备好，将文本追加到待处理值
            if self._pending_value is None:
                self._pending_value = ""
            self._pending_value += text
            return
        
        if not text:
            return
        
        # 使用与 setPlainText 相同的方式传递文本
        import json
        escaped_text = json.dumps(text)
        # 直接调用函数，与 setPlainText 保持一致
        script = f"window.insertTextAtCursor({escaped_text});"
        self._web_view.page().runJavaScript(script)
        # 更新缓存
        QTimer.singleShot(100, self._update_text_cache)
        # 触发文本变化信号
        QTimer.singleShot(200, self._emit_text_changed)
    
    def textCursor(self):
        """
        获取文本光标对象（与QPlainTextEdit兼容）
        
        Returns:
            MonacoTextCursor: 模拟的文本光标对象
        """
        if not hasattr(self, '_text_cursor'):
            self._text_cursor = MonacoTextCursor(self)
        return self._text_cursor
    
    def getCursorPosition(self):
        """
        获取当前光标位置
        
        Returns:
            tuple: (行号, 列号)，从1开始
        """
        if not self._is_ready:
            return (1, 1)
        
        script = "window.getCursorPosition();"
        # 由于是异步的，我们使用回调获取结果
        result = [None]
        def callback(pos):
            if pos:
                result[0] = (pos.get('line', 1), pos.get('column', 1))
        
        self._web_view.page().runJavaScript(script, callback)
        # 如果还没获取到结果，返回默认值
        return result[0] if result[0] else (1, 1)
    
    def setCursorPosition(self, line, column):
        """
        设置光标位置
        
        Args:
            line: 行号（从1开始）
            column: 列号（从1开始）
        """
        if not self._is_ready:
            return
        
        script = f"window.setCursorPosition({line}, {column});"
        self._web_view.page().runJavaScript(script)
    
    def _update_text_cache(self):
        """更新文本缓存"""
        script = "window.getEditorValue();"
        self._web_view.page().runJavaScript(script, self._on_text_retrieved)
    
    def _find(self):
        """查找"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('actions.find').run();
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _replace(self):
        """替换"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('editor.action.startFindReplaceAction').run();
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _goto_line(self):
        """跳转到指定行"""
        if not self._is_ready:
            return
        line, ok = QInputDialog.getInt(self, "转到行", "行号:", 1, 1, 999999, 1)
        if ok:
            script = f"""
            if (window.editor) {{
                var lineNumber = {line};
                var column = 1;
                window.editor.setPosition({{ lineNumber: lineNumber, column: column }});
                window.editor.revealLineInCenter(lineNumber);
            }}
            """
            self._web_view.page().runJavaScript(script)
    
    def _go_to_definition(self):
        """跳转到定义"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('editor.action.revealDefinition').run();
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _find_references(self):
        """查找所有引用"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('editor.action.goToReferences').run();
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _format_document(self):
        """格式化文档"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('editor.action.formatDocument').run();
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _run_code(self):
        """运行代码"""
        code = self.toPlainText()
        if not code.strip():
            return
        # 发送信号通知主窗口运行代码
        if hasattr(self, 'run_code_requested'):
            self.run_code_requested.emit(code)
        else:
            # 如果没有信号，尝试查找父窗口
            parent = self.parent()
            while parent:
                if hasattr(parent, '_run_code_from_editor'):
                    parent._run_code_from_editor(code)
                    break
                parent = parent.parent()
    
    def _run_selection(self):
        """运行选中的代码"""
        if not self._is_ready:
            return
        script = """
        (function() {
            if (window.editor) {
                var selection = window.editor.getSelection();
                if (selection && !selection.isEmpty()) {
                    var model = window.editor.getModel();
                    return model.getValueInRange(selection);
                }
            }
            return null;
        })();
        """
        def on_selection_retrieved(result):
            if result:
                if hasattr(self, 'run_code_requested'):
                    self.run_code_requested.emit(result)
                else:
                    parent = self.parent()
                    while parent:
                        if hasattr(parent, '_run_code_from_editor'):
                            parent._run_code_from_editor(result)
                            break
                        parent = parent.parent()
        
        self._web_view.page().runJavaScript(script, on_selection_retrieved)
    
    def _toggle_comment(self):
        """切换行注释"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('editor.action.commentLine').run();
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _to_upper(self):
        """转换为大写"""
        if self._is_ready:
            script = """
            if (window.editor) {
                var selection = window.editor.getSelection();
                if (selection && !selection.isEmpty()) {
                    var model = window.editor.getModel();
                    var text = model.getValueInRange(selection);
                    window.editor.executeEdits('toUpper', [{
                        range: selection,
                        text: text.toUpperCase()
                    }]);
                }
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _to_lower(self):
        """转换为小写"""
        if self._is_ready:
            script = """
            if (window.editor) {
                var selection = window.editor.getSelection();
                if (selection && !selection.isEmpty()) {
                    var model = window.editor.getModel();
                    var text = model.getValueInRange(selection);
                    window.editor.executeEdits('toLower', [{
                        range: selection,
                        text: text.toLowerCase()
                    }]);
                }
            }
            """
            self._web_view.page().runJavaScript(script)
    
    def _show_command_palette(self):
        """显示命令面板"""
        if self._is_ready:
            script = """
            if (window.editor) {
                window.editor.getAction('workbench.action.showCommands').run();
            }
            """
            self._web_view.page().runJavaScript(script)


class MonacoTextCursor:
    """
    Monaco Editor 文本光标模拟类
    用于兼容 QPlainTextEdit 的 textCursor 接口
    """
    def __init__(self, editor_widget):
        """
        初始化文本光标模拟对象
        
        Args:
            editor_widget: MonacoEditorWidget 实例
        """
        self._editor = editor_widget
        self._position = (1, 1)  # (行号, 列号)，从1开始
    
    def insertText(self, text):
        """
        在光标位置插入文本
        
        Args:
            text: 要插入的文本
        """
        if self._editor._is_ready:
            self._editor.insertPlainText(text)
    
    def position(self):
        """
        获取光标位置（字符位置，从文档开始计算）
        
        Returns:
            int: 光标位置
        """
        # 简化实现：返回一个基于行列的位置值
        line, column = self._editor.getCursorPosition()
        # 获取当前文本内容来计算准确位置
        current_text = self._editor.toPlainText()
        lines = current_text.split('\n')
        pos = 0
        for i in range(min(line - 1, len(lines))):
            pos += len(lines[i]) + 1  # +1 for newline
        pos += column - 1
        return pos
    
    def setPosition(self, position):
        """
        设置光标位置
        
        Args:
            position: 字符位置
        """
        # 简化实现：基于行号设置
        current_text = self._editor.toPlainText()
        lines = current_text.split('\n')
        line = 1
        col = 1
        pos = 0
        for i, line_text in enumerate(lines):
            line_len = len(line_text) + 1  # +1 for newline
            if pos + line_len > position:
                col = position - pos + 1
                line = i + 1
                break
            pos += line_len
        self._editor.setCursorPosition(line, col)
    
    def movePosition(self, operation):
        """移动光标位置（简化实现）"""
        # 这个方法在实际使用中可能需要更多实现
        pass
    
    def hasSelection(self):
        """检查是否有选中的文本"""
        if not self._editor._is_ready:
            return False
        # 简化实现：检查选择范围
        script = """
        (function() {
            if (window.editor) {
                var selection = window.editor.getSelection();
                return selection && !selection.isEmpty();
            }
            return false;
        })();
        """
        result = [False]
        def callback(has_sel):
            result[0] = has_sel if has_sel is not None else False
        self._editor._web_view.page().runJavaScript(script, callback)
        return result[0]
    
    def deletePreviousChar(self):
        """删除光标前一个字符"""
        if self._editor._is_ready:
            script = """
            if (window.editor) {
                var selection = window.editor.getSelection();
                var model = window.editor.getModel();
                var position = window.editor.getPosition();
                if (position && position.column > 1) {
                    var newPosition = new monaco.Position(position.lineNumber, position.column - 1);
                    var range = new monaco.Range(position.lineNumber, position.column - 1, position.lineNumber, position.column);
                    window.editor.executeEdits('delete', [{
                        range: range,
                        text: ''
                    }]);
                    window.editor.setPosition(newPosition);
                } else if (position && position.lineNumber > 1) {
                    // 删除上一行的换行符
                    var prevLine = model.getLineContent(position.lineNumber - 1);
                    var prevLineLength = prevLine.length;
                    var newPosition = new monaco.Position(position.lineNumber - 1, prevLineLength + 1);
                    var range = new monaco.Range(position.lineNumber - 1, prevLineLength + 1, position.lineNumber, 1);
                    window.editor.executeEdits('delete', [{
                        range: range,
                        text: ''
                    }]);
                    window.editor.setPosition(newPosition);
                }
            }
            """
            self._editor._web_view.page().runJavaScript(script)
            QTimer.singleShot(100, self._editor._update_text_cache)


class MonacoScrollBar:
    """
    Monaco Editor 滚动条模拟类
    用于兼容 QPlainTextEdit 的滚动条接口
    """
    def __init__(self, editor_widget, is_vertical=True):
        """
        初始化滚动条模拟对象
        
        Args:
            editor_widget: MonacoEditorWidget 实例
            is_vertical: 是否为垂直滚动条
        """
        self._editor = editor_widget
        self._is_vertical = is_vertical
        self._value = 0
    
    def value(self):
        """
        获取当前滚动位置
        
        Returns:
            int: 滚动位置
        """
        if self._editor._is_ready:
            # 从 Monaco Editor 获取滚动位置
            script = """
            (function() {
                if (window.editor) {
                    var scrollTop = window.editor.getScrollTop();
                    return Math.round(scrollTop);
                }
                return 0;
            })();
            """
            # 注意：runJavaScript 是异步的，这里返回缓存值
            # 实际值会通过回调更新
            if self._is_vertical:
                self._editor._web_view.page().runJavaScript(
                    script, 
                    lambda val: setattr(self, '_value', val or 0)
                )
        return self._value
    
    def setValue(self, value):
        """
        设置滚动位置
        
        Args:
            value: 滚动位置
        """
        self._value = value
        if self._editor._is_ready:
            if self._is_vertical:
                script = f"""
                if (window.editor) {{
                    window.editor.setScrollTop({value});
                }}
                """
            else:
                script = f"""
                if (window.editor) {{
                    window.editor.setScrollLeft({value});
                }}
                """
            self._editor._web_view.page().runJavaScript(script)

