"""
Monaco Editor 配置模块
用于管理Monaco Editor的CDN路径配置
"""

import os
from pathlib import Path
from typing import Optional

# 默认CDN配置
_MONACO_VERSION = "0.55.1"
_CDN_BASE = f"https://cdn.jsdelivr.net/npm/monaco-editor@{_MONACO_VERSION}/min/vs"
_CDN_LOADER_URL = f"{_CDN_BASE}/loader.js"

# 全局缓存变量
_monaco_loader_url: Optional[str] = None
_monaco_vs_path: Optional[str] = None
_initialized: bool = False


def _path_to_file_url(path: Path) -> str:
    """将本地路径转换为file://协议URL"""
    try:
        from PySide6.QtCore import QUrl
        return QUrl.fromLocalFile(str(path)).toString()
    except ImportError:
        # 手动转换（Windows需要特殊处理）
        abs_path = str(path.resolve()).replace('\\', '/')
        if os.name == 'nt' and not abs_path.startswith('/'):
            abs_path = '/' + abs_path
        return f'file://{abs_path}'


def _get_local_monaco_path() -> Optional[Path]:
    """获取本地Monaco Editor路径"""
    # 从当前文件位置计算项目根目录
    project_root = Path(__file__).parent.parent
    local_path = project_root / "ui" / "monaco_editor_local" / "node_modules" / "monaco-editor"
    return local_path if local_path.exists() else None


def load_monaco_config(config_file_path: Optional[str] = None):
    """
    从配置文件加载Monaco Editor的本地路径
    
    Args:
        config_file_path: 配置文件路径（保留参数以兼容现有代码，当前未使用）
    """
    global _monaco_loader_url, _monaco_vs_path, _initialized
    
    if _initialized:
        return
    
    # 尝试使用本地路径
    local_path = _get_local_monaco_path()
    
    if local_path:
        file_url = _path_to_file_url(local_path)
        _monaco_loader_url = f"{file_url}/min/vs/loader.js"
        _monaco_vs_path = f"{file_url}/min/vs"
        #print(f"已加载Monaco Editor本地路径: {_monaco_vs_path}")
    else:
        # 使用CDN路径
        _monaco_loader_url = _CDN_LOADER_URL
        _monaco_vs_path = _CDN_BASE
        print("使用Monaco Editor CDN路径")
    
    _initialized = True


def get_monaco_loader_url() -> str:
    """
    获取Monaco Editor loader.js的URL
    
    Returns:
        str: loader.js的URL（本地file://路径或CDN路径）
    """
    if not _initialized:
        load_monaco_config()
    return _monaco_loader_url or _CDN_LOADER_URL


def get_monaco_vs_path() -> str:
    """
    获取Monaco Editor vs路径（用于require.config）
    
    Returns:
        str: vs路径（本地file://路径或CDN路径）
    """
    if not _initialized:
        load_monaco_config()
    return _monaco_vs_path or _CDN_BASE


def is_using_local() -> bool:
    """
    检查是否使用本地路径
    
    Returns:
        bool: 如果使用本地路径返回True，否则返回False
    """
    if not _initialized:
        load_monaco_config()
    return _monaco_loader_url is not None and _monaco_loader_url.startswith('file://')

