"""
菜单编辑器模块
提供菜单和菜单项的编辑功能
支持创建和编辑下拉菜单、菜单项、快捷键等
"""

# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QDialog,  # 对话框基类
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QFormLayout,  # 表单布局
    QLabel,  # 标签
    QLineEdit,  # 单行文本输入框
    QComboBox,  # 下拉选择框
    QCheckBox,  # 复选框
    QListWidget,  # 列表控件
    QListWidgetItem,  # 列表项
    QPushButton,  # 按钮
    QDialogButtonBox  # 对话框按钮盒
)

# 定义可用的快捷键列表
SHORTCUTS = ["(无)", "Ctrl+N", "Ctrl+O", "Ctrl+S", "Ctrl+Q", "Ctrl+P", "Ctrl+Z", "Ctrl+Y"]

from .frameless_window import FramelessWindow

class MenuEditorDialog(FramelessWindow):
    """
    菜单编辑器对话框类
    用于创建和编辑菜单及其菜单项
    支持设置菜单标题、名称、快捷键、可见性等属性
    """
    def __init__(self, parent=None, menu_data=None):
        """
        初始化菜单编辑器对话框
        
        Args:
            parent: 父窗口对象
            menu_data: 菜单数据字典，如果为None则创建新菜单
                格式：{
                    "title": "菜单标题",
                    "name": "菜单名称",
                    "shortcut": "快捷键",
                    "selected": False,
                    "allow": True,
                    "visible": True,
                    "items": []  # 菜单项列表
                }
        """
        # 调用父类构造函数
        super().__init__(parent)
        # 设置对话框标题
        self.setWindowTitle("菜单编辑器")
        # 保存菜单数据，如果未提供则使用默认值创建新菜单
        self.menu_data = menu_data or {
            "title": "新菜单",  # 菜单标题
            "name": "新菜单",  # 菜单名称（用于代码中）
            "shortcut": "(无)",  # 快捷键
            "selected": False,  # 是否选中
            "allow": True,  # 是否允许操作
            "visible": True,  # 是否可见
            "items": []  # 菜单项列表
        }
        # 初始化用户界面
        self._init_ui()

    def _init_ui(self):
        # 使用FramelessWindow的内容区域
        content_widget = self.contentWidget()
        # 清除可能存在的布局
        if content_widget.layout():
            from PySide6.QtWidgets import QWidget
            QWidget().setLayout(content_widget.layout())
        
        layout = QVBoxLayout(content_widget)
        form = QFormLayout()
        self.title_edit = QLineEdit(self.menu_data.get("title", ""))
        self.name_edit = QLineEdit(self.menu_data.get("name", ""))
        self.shortcut_combo = QComboBox()
        self.shortcut_combo.addItems(SHORTCUTS)
        self.shortcut_combo.setCurrentText(self.menu_data.get("shortcut", "(无)"))
        self.selected_check = QCheckBox("选中")
        self.selected_check.setChecked(self.menu_data.get("selected", False))
        self.allow_check = QCheckBox("允许操作")
        self.allow_check.setChecked(self.menu_data.get("allow", True))
        self.visible_check = QCheckBox("可视")
        self.visible_check.setChecked(self.menu_data.get("visible", True))
        form.addRow("标题:", self.title_edit)
        form.addRow("名称:", self.name_edit)
        form.addRow("快捷键:", self.shortcut_combo)
        box = QHBoxLayout()
        box.addWidget(self.selected_check)
        box.addWidget(self.allow_check)
        box.addWidget(self.visible_check)
        form.addRow("", QLabel(""))
        layout.addLayout(form)
        layout.addLayout(box)

        # ========== 创建菜单项列表控件 ==========
        # 创建列表控件，用于显示菜单项列表
        self.list_widget = QListWidget()
        
        # ========== 加载现有菜单项到列表 ==========
        # 遍历菜单数据中的菜单项列表（如果不存在则使用空列表）
        for item in self.menu_data.get("items", []):
            # 为每个菜单项创建列表项，显示项标签（如果不存在则使用空字符串）
            lw = QListWidgetItem(item.get("label", ""))
            # 将完整的菜单项数据保存到列表项的自定义数据中
            # 参数32是Qt.UserRole，用于存储自定义数据（整个item字典）
            lw.setData(32, item)
            # 将列表项添加到列表控件中
            self.list_widget.addItem(lw)
        
        # 将列表控件添加到主布局中
        layout.addWidget(self.list_widget)

        # 选中项编辑区
        self.item_form = QFormLayout()
        self.item_label_edit = QLineEdit()
        self.item_shortcut_combo = QComboBox(); self.item_shortcut_combo.addItems(SHORTCUTS)
        self.item_checked = QCheckBox("选中")
        self.item_allow = QCheckBox("允许操作")
        self.item_visible = QCheckBox("可视")
        wrap = QVBoxLayout()
        wrap.addLayout(self.item_form)
        self.item_form.addRow("项标题:", self.item_label_edit)
        self.item_form.addRow("项快捷键:", self.item_shortcut_combo)
        box2 = QHBoxLayout(); box2.addWidget(self.item_checked); box2.addWidget(self.item_allow); box2.addWidget(self.item_visible)
        wrap.addLayout(box2)
        btn_apply = QPushButton("应用到当前项")
        wrap.addWidget(btn_apply)
        layout.addLayout(wrap)

        btns = QHBoxLayout()
        self.btn_left = QPushButton("左移")
        self.btn_right = QPushButton("右移")
        self.btn_up = QPushButton("上移")
        self.btn_down = QPushButton("下移")
        self.btn_ins_before = QPushButton("向前插入")
        self.btn_ins_after = QPushButton("向后插入")
        self.btn_del = QPushButton("删除")
        self.btn_bulk = QPushButton("批量导入")
        # ========== 将所有操作按钮添加到布局 ==========
        # 遍历所有操作按钮的列表
        for b in [self.btn_left, self.btn_right, self.btn_up, self.btn_down, self.btn_ins_before, self.btn_ins_after, self.btn_del, self.btn_bulk]:
            # 将每个按钮添加到按钮布局中
            btns.addWidget(b)
        layout.addLayout(btns)

        self.btn_left.clicked.connect(self._move_left)
        self.btn_right.clicked.connect(self._move_right)
        self.btn_up.clicked.connect(self._move_up)
        self.btn_down.clicked.connect(self._move_down)
        self.btn_ins_before.clicked.connect(lambda: self._insert_item(before=True))
        self.btn_ins_after.clicked.connect(lambda: self._insert_item(before=False))
        self.btn_del.clicked.connect(self._delete_item)
        self.btn_bulk.clicked.connect(self._bulk_import)
        self.list_widget.currentItemChanged.connect(self._on_item_selected)
        btn_apply.clicked.connect(self._apply_to_item)

        tips = QLabel("提示: 如想加入菜单横向分隔线, 将标题置为一个破号( - )。若某项为子菜单的标题, 建议该项不可操作。访问键可在字前加上 & 字符。")
        layout.addWidget(tips)

        db = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        db.accepted.connect(self._on_accept)
        db.rejected.connect(self.close)
        layout.addWidget(db)
    
    def _on_accept(self):
        """确定按钮（兼容FramelessWindow）"""
        self.close()

    def _current_item(self):
        """
        获取当前选中的菜单项
        
        Returns:
            tuple: (当前列表项对象, 索引) 如果没有选中项则返回 (None, -1)
        """
        # ========== 获取当前选中项的索引 ==========
        # 获取列表控件中当前选中行的索引（如果没有选中则返回-1）
        idx = self.list_widget.currentRow()
        
        # ========== 检查是否有选中项 ==========
        # 检查索引是否小于0（表示没有选中项）
        if idx < 0:
            # 如果没有选中项，返回None和-1
            return None, -1
        
        # ========== 返回选中项对象和索引 ==========
        # 如果有效，返回列表项对象和索引
        return self.list_widget.item(idx), idx

    def _move_left(self):
        """
        将当前菜单项向左移动（降低层级，减少缩进）
        """
        # ========== 获取当前选中项 ==========
        # 获取当前选中的列表项对象和索引
        it, idx = self._current_item()
        
        # ========== 检查是否有选中项 ==========
        # 检查列表项对象是否存在
        if not it:
            # 如果没有选中项，直接返回，不执行任何操作
            return
        
        # ========== 获取菜单项数据 ==========
        # 从列表项的自定义数据中获取菜单项数据
        # 如果数据不存在（data(32)返回None），则创建新字典，使用文本作为标签，层级为0
        data = it.data(32) or {"label": it.text(), "level": 0}
        
        # ========== 降低层级 ==========
        # 将菜单项的层级减1（向左移动，减少缩进）
        # max(0, ...)确保层级不会小于0（最左层级）
        data["level"] = max(0, int(data.get("level", 0)) - 1)
        
        # ========== 更新列表项数据 ==========
        # 将修改后的数据保存回列表项的自定义数据中
        it.setData(32, data)
        # 更新列表项显示的文本（根据层级显示不同的缩进）
        it.setText(self._display_text(data))

    def _move_right(self):
        """
        将当前菜单项向右移动（提高层级，增加缩进）
        """
        # ========== 获取当前选中项 ==========
        # 获取当前选中的列表项对象和索引
        it, idx = self._current_item()
        
        # ========== 检查是否有选中项 ==========
        # 检查列表项对象是否存在
        if not it:
            # 如果没有选中项，直接返回，不执行任何操作
            return
        
        # ========== 获取菜单项数据 ==========
        # 从列表项的自定义数据中获取菜单项数据
        # 如果数据不存在（data(32)返回None），则创建新字典，使用文本作为标签，层级为0
        data = it.data(32) or {"label": it.text(), "level": 0}
        
        # ========== 提高层级 ==========
        # 将菜单项的层级加1（向右移动，增加缩进）
        # min(4, ...)确保层级不会超过4（最大层级限制）
        data["level"] = min(4, int(data.get("level", 0)) + 1)
        
        # ========== 更新列表项数据 ==========
        # 将修改后的数据保存回列表项的自定义数据中
        it.setData(32, data)
        # 更新列表项显示的文本（根据层级显示不同的缩进）
        it.setText(self._display_text(data))

    def _move_up(self):
        """
        将当前菜单项向上移动（在列表中的位置提前）
        """
        # ========== 获取当前选中项 ==========
        # 获取当前选中的列表项对象和索引
        it, idx = self._current_item()
        
        # ========== 检查是否可以上移 ==========
        # 检查列表项是否存在，并且索引是否大于0（不能是第一个）
        if not it or idx <= 0:
            # 如果列表项不存在或已经是第一个，无法上移，直接返回
            return
        
        # ========== 执行上移操作 ==========
        # 从列表中移除当前项（但保留项对象）
        self.list_widget.takeItem(idx)
        # 将项插入到前一个位置（索引减1）
        self.list_widget.insertItem(idx - 1, it)
        # 将新位置的项设置为当前选中项（保持选中状态）
        self.list_widget.setCurrentRow(idx - 1)

    def _move_down(self):
        """
        将当前菜单项向下移动（在列表中的位置延后）
        """
        # ========== 获取当前选中项 ==========
        # 获取当前选中的列表项对象和索引
        it, idx = self._current_item()
        
        # ========== 检查是否可以下移 ==========
        # 检查列表项是否存在，索引是否有效（不小于0），以及是否不是最后一个（索引小于总数-1）
        if not it or idx < 0 or idx >= self.list_widget.count() - 1:
            # 如果列表项不存在、索引无效或已经是最后一个，无法下移，直接返回
            return
        
        # ========== 执行下移操作 ==========
        # 从列表中移除当前项（但保留项对象）
        self.list_widget.takeItem(idx)
        # 将项插入到后一个位置（索引加1）
        self.list_widget.insertItem(idx + 1, it)
        # 将新位置的项设置为当前选中项（保持选中状态）
        self.list_widget.setCurrentRow(idx + 1)

    def _insert_item(self, before=True):
        """
        插入新菜单项
        在当前选中项的前面或后面插入一个新的菜单项
        
        Args:
            before: 是否在当前项之前插入（True表示之前，False表示之后）
        """
        # ========== 获取当前选中项 ==========
        # 获取当前选中的列表项对象和索引
        it, idx = self._current_item()
        
        # ========== 创建新菜单项 ==========
        # 创建新的列表项，显示文本为"新项"
        new = QListWidgetItem("新项")
        # 设置新项的自定义数据（完整的菜单项数据字典）
        # 包含：标签、层级、快捷键、选中状态、允许操作、可见性
        new.setData(32, {"label": "新项", "level": 0, "shortcut": "(无)", "checked": False, "allow": True, "visible": True})
        
        # ========== 计算插入位置 ==========
        # 如果before为True，插入到当前项之前（使用当前索引）
        # 如果before为False，插入到当前项之后（使用当前索引+1，如果索引有效）
        # 如果索引无效（<0），则插入到列表末尾
        pos = idx if before else idx + 1 if idx >= 0 else self.list_widget.count()
        
        # ========== 插入新项 ==========
        # 在计算出的位置插入新项
        self.list_widget.insertItem(pos, new)
        # 将新插入的项设置为当前选中项（焦点移到新项上）
        self.list_widget.setCurrentItem(new)

    def _delete_item(self):
        """
        删除当前选中的菜单项
        从列表控件中移除当前选中的项
        """
        # ========== 获取当前选中项 ==========
        # 获取当前选中的列表项对象和索引
        it, idx = self._current_item()
        
        # ========== 检查是否有选中项 ==========
        # 检查列表项对象是否存在
        if not it:
            # 如果没有选中项，无法删除，直接返回
            return
        
        # ========== 删除项 ==========
        # 从列表控件中移除指定索引的项（takeItem会移除但不删除项对象）
        self.list_widget.takeItem(idx)

    def _bulk_import(self):
        dlg = QDialog(self)
        dlg.setWindowTitle("批量导入菜单项")
        v = QVBoxLayout(dlg)
        tip = QLabel("每行一个菜单项；使用前导 . 或空格表示层级，例如 '..子项' 表示二级。输入 '-' 作为分隔线")
        v.addWidget(tip)
        edit = QTextEdit()
        v.addWidget(edit)
        db = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        v.addWidget(db)
        db.accepted.connect(dlg.accept)
        db.rejected.connect(dlg.reject)
        # ========== 处理对话框结果 ==========
        # 显示对话框并等待用户操作，如果用户点击了"确定"（返回Accepted）
        if dlg.exec():
            # ========== 获取输入的文本 ==========
            # 从文本编辑器中获取纯文本内容，并按行分割成列表
            text = edit.toPlainText().splitlines()
            
            # ========== 遍历每一行文本 ==========
            # 遍历文本列表中的每一行
            for line in text:
                # 去除行尾的空白字符（保留行首，因为行首可能包含层级标识）
                raw = line.rstrip()
                # 检查行是否为空（去除行尾空白后）
                if not raw:
                    # 如果行为空，跳过处理，继续下一行
                    continue
                
                # ========== 解析层级 ==========
                # 初始化层级为0（最顶层）
                lvl = 0
                # 初始化索引为0，用于遍历行首字符
                i = 0
                # 从行首开始，查找连续的 '.' 或 ' ' 字符（这些字符表示层级缩进）
                while i < len(raw) and raw[i] in ['.', ' ']:
                    # 如果当前字符是 '.' 或 ' '，继续检查下一个字符
                    i += 1
                # 如果找到了层级标识字符（i > 0）
                if i > 0:
                    # 计算层级：每2个标识字符表示1个层级，层级范围限制在0-4之间
                    lvl = max(0, min(4, i // 2))
                
                # ========== 提取标签文本 ==========
                # 从层级标识字符之后的位置开始，提取标签文本
                label = raw[i:]
                # 检查标签文本是否为空
                if not label:
                    # 如果标签为空，跳过此项，继续下一行
                    continue
                
                # ========== 创建并添加菜单项 ==========
                # 创建新的列表项，显示文本为标签
                it = QListWidgetItem(label)
                # 设置新项的自定义数据（包含标签、层级、快捷键等属性）
                it.setData(32, {"label": label, "level": lvl, "shortcut": "(无)", "checked": False, "allow": True, "visible": True})
                # 将新项添加到列表控件中
                self.list_widget.addItem(it)

    def _display_text(self, item):
        level = int(item.get("level", 0))
        label = item.get("label", "")
        return ("." * (level * 2)) + label

    def _on_item_selected(self, current, previous):
        if not current:
            self.item_label_edit.setText("")
            self.item_shortcut_combo.setCurrentIndex(0)
            self.item_checked.setChecked(False)
            self.item_allow.setChecked(True)
            self.item_visible.setChecked(True)
            return
        data = current.data(32) or {"label": current.text(), "level": 0}
        self.item_label_edit.setText(data.get("label", ""))
        self.item_shortcut_combo.setCurrentText(data.get("shortcut", "(无)"))
        self.item_checked.setChecked(bool(data.get("checked", False)))
        self.item_allow.setChecked(bool(data.get("allow", True)))
        self.item_visible.setChecked(bool(data.get("visible", True)))

    def _apply_to_item(self):
        it, idx = self._current_item()
        if not it:
            return
        data = it.data(32) or {"label": it.text(), "level": 0}
        data["label"] = self.item_label_edit.text().strip()
        data["shortcut"] = self.item_shortcut_combo.currentText()
        data["checked"] = self.item_checked.isChecked()
        data["allow"] = self.item_allow.isChecked()
        data["visible"] = self.item_visible.isChecked()
        it.setData(32, data)
        it.setText(self._display_text(data))

    def accept(self):
        """
        接受对话框（点击确定按钮时调用）
        收集所有编辑的菜单数据，保存到menu_data中，然后关闭对话框
        """
        # ========== 收集所有菜单项数据 ==========
        # 创建空列表，用于存储所有菜单项的数据
        items = []
        # 遍历列表控件中的所有项
        for i in range(self.list_widget.count()):
            # 获取指定索引的列表项对象
            it = self.list_widget.item(i)
            # 从列表项的自定义数据中获取菜单项数据，如果不存在则创建新字典
            data = it.data(32) or {"label": it.text(), "level": 0}
            # ========== 提取真实标签 ==========
            # 提取真实的标签文本（去掉显示时添加的点前缀）
            # 优先使用数据中的label，如果没有则从文本中提取（去掉左侧的点）
            txt = data.get("label") or it.text().lstrip('.')
            # 将清理后的标签文本保存回数据字典
            data["label"] = txt
            # 将菜单项数据添加到列表中
            items.append(data)
        
        # ========== 构建完整的菜单数据 ==========
        # 构建菜单数据字典，包含所有菜单属性
        self.menu_data = {
            "title": self.title_edit.text().strip() or "菜单",  # 菜单标题（去除空白后，如果为空则使用默认值）
            "name": self.name_edit.text().strip() or self.title_edit.text().strip() or "菜单",  # 菜单名称（优先使用名称，其次使用标题，最后使用默认值）
            "shortcut": self.shortcut_combo.currentText(),  # 快捷键
            "selected": self.selected_check.isChecked(),  # 是否选中
            "allow": self.allow_check.isChecked(),  # 是否允许操作
            "visible": self.visible_check.isChecked(),  # 是否可见
            "items": items  # 菜单项列表
        }
        
        # ========== 调用父类accept方法 ==========
        # 调用父类的accept方法，关闭对话框并返回Accepted状态
        self._on_accept()

    def get_data(self):
        return self.menu_data


class MenusManagerDialog(FramelessWindow):
    def __init__(self, parent=None, menus=None):
        super().__init__(parent)
        self.setWindowTitle("菜单管理器")
        self.menus = menus or []
        self._init_ui()

    def _init_ui(self):
        # 使用FramelessWindow的内容区域
        content_widget = self.contentWidget()
        # 清除可能存在的布局
        if content_widget.layout():
            from PySide6.QtWidgets import QWidget
            QWidget().setLayout(content_widget.layout())
        
        layout = QVBoxLayout(content_widget)
        self.list_widget = QListWidget()
        for m in self.menus:
            it = QListWidgetItem(m.get("title", "菜单"))
            it.setData(32, m)
            self.list_widget.addItem(it)
        layout.addWidget(self.list_widget)

        btns = QHBoxLayout()
        btn_add = QPushButton("添加菜单")
        btn_edit = QPushButton("编辑菜单")
        btn_del = QPushButton("删除菜单")
        btn_up = QPushButton("上移")
        btn_down = QPushButton("下移")
        for b in [btn_add, btn_edit, btn_del, btn_up, btn_down]:
            btns.addWidget(b)
        layout.addLayout(btns)

        btn_add.clicked.connect(self._add)
        btn_edit.clicked.connect(self._edit)
        btn_del.clicked.connect(self._del)
        btn_up.clicked.connect(self._up)
        btn_down.clicked.connect(self._down)

        db = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        db.accepted.connect(self._on_accept)
        db.rejected.connect(self.close)
        layout.addWidget(db)
    
    def _on_accept(self):
        """确定按钮（兼容FramelessWindow）"""
        self.close()

    def _current(self):
        idx = self.list_widget.currentRow()
        if idx < 0:
            return None, -1
        return self.list_widget.item(idx), idx

    def _add(self):
        dlg = MenuEditorDialog(self)
        if dlg.exec():
            data = dlg.get_data()
            it = QListWidgetItem(data.get("title", "菜单"))
            it.setData(32, data)
            self.list_widget.addItem(it)

    def _edit(self):
        it, idx = self._current()
        if not it:
            return
        dlg = MenuEditorDialog(self, it.data(32))
        if dlg.exec():
            data = dlg.get_data()
            it.setText(data.get("title", "菜单"))
            it.setData(32, data)

    def _del(self):
        it, idx = self._current()
        if not it:
            return
        self.list_widget.takeItem(idx)

    def _up(self):
        it, idx = self._current()
        if not it or idx <= 0:
            return
        self.list_widget.takeItem(idx)
        self.list_widget.insertItem(idx - 1, it)
        self.list_widget.setCurrentRow(idx - 1)

    def _down(self):
        it, idx = self._current()
        if not it or idx < 0 or idx >= self.list_widget.count() - 1:
            return
        self.list_widget.takeItem(idx)
        self.list_widget.insertItem(idx + 1, it)
        self.list_widget.setCurrentRow(idx + 1)

    def get_menus(self):
        menus = []
        for i in range(self.list_widget.count()):
            it = self.list_widget.item(i)
            menus.append(it.data(32))
        return menus
