#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
主窗口UI组件
包含菜单栏、工具栏和状态栏的创建方法
"""

"""
主窗口UI组件模块
包含菜单栏、工具栏和状态栏的创建方法
"""

# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QMenuBar,  # 菜单栏
    QMenu,  # 菜单
    QToolBar,  # 工具栏
    QStatusBar,  # 状态栏
    QLabel,  # 标签
    QFileDialog,  # 文件对话框
    QMessageBox  # 消息框
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import QAction  # 动作类，用于菜单项和工具栏按钮
# 导入PySide6核心功能
from PySide6.QtCore import QSize  # 大小对象
# 导入版本信息
from module.version import APP_VERSION, about_text


class MainWindowUI:
    """
    主窗口UI组件类
    提供菜单栏、工具栏和状态栏的创建方法
    将UI组件的创建逻辑从主窗口类中分离出来，使代码结构更清晰
    """
    
    def __init__(self, main_window):
        """
        初始化UI组件管理器
        
        Args:
            main_window: 主窗口实例，用于设置菜单栏、工具栏和状态栏
        """
        # 保存主窗口引用
        self.main_window = main_window
        # 状态栏标签：显示一般状态信息
        self.status_label = None
        # 坐标标签：显示鼠标或控件坐标
        self.coord_label = None
        # 画布大小标签：显示画布或控件尺寸
        self.canvas_size_label = None
    
    def create_menu_bar(self):
        """
        创建菜单栏
        包含文件、编辑、工具、视图、帮助等菜单
        """
        # 获取主窗口的菜单栏（如果不存在则创建）
        menu_bar = self.main_window.menuBar()
        
        # ========== 文件菜单 ==========
        file_menu = menu_bar.addMenu("文件")
        
        # 打开项目动作
        open_action = QAction("打开", self.main_window)
        open_action.setShortcut("Ctrl+O")  # 设置快捷键
        open_action.triggered.connect(self.main_window._open_project)  # 连接信号到打开项目方法
        file_menu.addAction(open_action)
        
        # 保存项目动作
        save_action = QAction("保存", self.main_window)
        save_action.setShortcut("Ctrl+S")  # 设置快捷键
        save_action.triggered.connect(self.main_window._save_project)  # 连接信号到保存项目方法
        file_menu.addAction(save_action)
        
        # 另存为动作
        save_as_action = QAction("另存为", self.main_window)
        save_as_action.setShortcut("Ctrl+Shift+S")  # 设置快捷键
        save_as_action.triggered.connect(self.main_window._save_as_project)  # 连接信号到另存为方法
        file_menu.addAction(save_as_action)
        
        # 添加分隔符
        file_menu.addSeparator()
        
        # 退出动作
        exit_action = QAction("退出", self.main_window)
        exit_action.setShortcut("Ctrl+Q")  # 设置快捷键
        exit_action.triggered.connect(self.main_window.close)  # 连接信号到关闭窗口方法
        file_menu.addAction(exit_action)
        
        # ========== 编辑菜单 ==========
        edit_menu = menu_bar.addMenu("编辑")
        
        # 撤销动作
        undo_action = QAction("上一步", self.main_window)
        undo_action.setShortcut("Ctrl+Z")  # 设置快捷键
        undo_action.triggered.connect(self.main_window._undo)  # 连接信号到撤销方法
        edit_menu.addAction(undo_action)
        
        # 重做动作
        redo_action = QAction("重做", self.main_window)
        redo_action.setShortcut("Ctrl+Y")  # 设置快捷键
        redo_action.triggered.connect(self.main_window._redo)  # 连接信号到重做方法
        edit_menu.addAction(redo_action)
        
        # 添加分隔符
        edit_menu.addSeparator()
        
        # 删除动作
        delete_action = QAction("删除", self.main_window)
        delete_action.setShortcut("Delete")  # 设置快捷键
        delete_action.triggered.connect(self.main_window._delete_selected)  # 连接信号到删除选中控件方法
        edit_menu.addAction(delete_action)
        
        # 添加分隔符
        edit_menu.addSeparator()
        
        # 新建窗口动作
        new_window_action = QAction("新建窗口", self.main_window)
        new_window_action.setShortcut("Ctrl+N+W")  # 设置快捷键 Ctrl+N+W
        new_window_action.triggered.connect(self.main_window._new_window_class)  # 连接信号到新建窗口方法
        edit_menu.addAction(new_window_action)
        
        # 删除窗口动作
        delete_window_action = QAction("删除窗口", self.main_window)
        delete_window_action.setShortcut("Ctrl+Shift+W")  # 设置快捷键 Ctrl+Shift+W
        delete_window_action.triggered.connect(self.main_window._delete_window_class)  # 连接信号到删除窗口方法
        edit_menu.addAction(delete_window_action)
        
        
        # 工具菜单
        tools_menu = menu_bar.addMenu("工具")
        
        parse_code_action = QAction("解析代码", self.main_window)
        parse_code_action.triggered.connect(self.main_window._parse_and_sync_code)
        tools_menu.addAction(parse_code_action)
        
        generate_code_action = QAction("生成代码", self.main_window)
        generate_code_action.triggered.connect(self.main_window._generate_code)
        tools_menu.addAction(generate_code_action)
        
        preview_action = QAction("预览", self.main_window)
        preview_action.setShortcut("F5")
        preview_action.triggered.connect(self.main_window._preview_project)
        tools_menu.addAction(preview_action)
        
        run_action = QAction("运行", self.main_window)
        run_action.setShortcut("F6")
        run_action.triggered.connect(self.main_window._run_project)
        tools_menu.addAction(run_action)
        
        
        
        # 帮助菜单
        help_menu = menu_bar.addMenu("帮助")
        
        about_action = QAction("关于", self.main_window)
        about_action.triggered.connect(self.main_window._show_about)
        help_menu.addAction(about_action)
    
    def create_tool_bar(self):
        """
        创建工具栏
        在主窗口中创建工具栏，包含常用的操作按钮（打开、保存、撤销、重做等）
        """
        # ========== 创建工具栏 ==========
        # 在主窗口中添加工具栏，名称为"主工具栏"
        tool_bar = self.main_window.addToolBar("主工具栏")
        # 设置工具栏的objectName，用于保存和恢复窗口状态
        tool_bar.setObjectName("MainToolBar")
        # 设置工具栏不可移动（固定位置）
        tool_bar.setMovable(False)
        # 设置工具栏图标大小：24x24像素（适合工具栏显示）
        tool_bar.setIconSize(QSize(24, 24))
        
        
        
        # 打开按钮
        open_action = QAction("打开", self.main_window)
        open_action.triggered.connect(self.main_window._open_project)
        tool_bar.addAction(open_action)
        
        # 保存按钮
        save_action = QAction("保存", self.main_window)
        save_action.triggered.connect(self.main_window._save_project)
        tool_bar.addAction(save_action)
        
        tool_bar.addSeparator()
        
        # 撤销按钮
        undo_action = QAction("上一步", self.main_window)
        undo_action.triggered.connect(self.main_window._undo)
        tool_bar.addAction(undo_action)
        
        # 重做按钮
        redo_action = QAction("重做", self.main_window)
        redo_action.triggered.connect(self.main_window._redo)
        tool_bar.addAction(redo_action)
        
        tool_bar.addSeparator()
        
        # 删除按钮
        delete_action = QAction("删除", self.main_window)
        delete_action.triggered.connect(self.main_window._delete_selected)
        tool_bar.addAction(delete_action)
        
        tool_bar.addSeparator()
        
        
        tool_bar.addSeparator()
        
        # 解析代码按钮
        parse_code_action = QAction("解析代码", self.main_window)
        parse_code_action.triggered.connect(self.main_window._parse_and_sync_code)
        tool_bar.addAction(parse_code_action)
        
        # 生成代码按钮
        generate_code_action = QAction("生成代码", self.main_window)
        generate_code_action.triggered.connect(self.main_window._generate_code)
        tool_bar.addAction(generate_code_action)
        
        
        
        # 运行按钮
        run_action = QAction("运行", self.main_window)
        run_action.triggered.connect(self.main_window._run_project)
        tool_bar.addAction(run_action)
        
        
    
    def create_status_bar(self):
        """
        创建状态栏
        在主窗口底部创建状态栏，显示程序状态、坐标、画布大小、版本等信息
        """
        # ========== 获取或创建状态栏 ==========
        # 获取主窗口的状态栏（如果不存在则创建）
        status_bar = self.main_window.statusBar()
        # 设置状态栏最大高度为24像素（限制高度，保持界面紧凑）
        status_bar.setMaximumHeight(24)
        
        # ========== 创建状态标签 ==========
        # 创建状态标签，显示一般状态信息（如"就绪"、"已选中控件"等）
        self.main_window.status_label = QLabel("就绪")
        # 将状态标签添加到状态栏（左侧，可被其他信息覆盖）
        status_bar.addWidget(self.main_window.status_label)
        
        # ========== 创建坐标标签 ==========
        # 创建坐标标签，显示鼠标或控件坐标（格式：X: 0, Y: 0）
        self.main_window.coord_label = QLabel("X: 0, Y: 0")
        # 将坐标标签添加到状态栏（右侧永久区域，不会被覆盖）
        status_bar.addPermanentWidget(self.main_window.coord_label)
        
        # ========== 创建画布大小标签 ==========
        # 创建画布大小标签，显示画布或控件尺寸（格式：画布: 380x300）
        self.main_window.canvas_size_label = QLabel("画布: 380x300")
        # 将画布大小标签添加到状态栏（右侧永久区域）
        status_bar.addPermanentWidget(self.main_window.canvas_size_label)
        
        # ========== 创建版本标签 ==========
        # 创建版本标签，显示应用程序版本号（从version模块导入）
        self.main_window.version_label = QLabel(f"版本: {APP_VERSION}")
        # 将版本标签添加到状态栏（右侧永久区域）
        status_bar.addPermanentWidget(self.main_window.version_label)
    
    def show_about_dialog(self):
        """显示关于对话框"""
        QMessageBox.about(
            self.main_window,
            "关于可视化编程工具",
            about_text()
        )
    
    def get_open_file_path(self, title="打开项目", filter_str="项目文件 (*.vpp);;所有文件 (*)"):
        """
        获取打开文件路径
        显示文件打开对话框，让用户选择要打开的项目文件
        
        Args:
            title: 对话框标题，默认为"打开项目"
            filter_str: 文件类型过滤器，默认为"项目文件 (*.vpp);;所有文件 (*)"
                       格式：显示名称 (扩展名);;显示名称 (扩展名)
            
        Returns:
            str: 用户选择的文件路径，如果用户取消选择则返回None
        """
        # ========== 显示文件打开对话框 ==========
        # 显示文件打开对话框，获取用户选择的文件路径
        # 参数：父窗口（主窗口）、对话框标题、起始目录（空字符串表示默认）、文件过滤器
        file_path, _ = QFileDialog.getOpenFileName(
            self.main_window,  # 父窗口
            title,  # 对话框标题
            "",  # 起始目录（空字符串表示使用默认目录）
            filter_str  # 文件类型过滤器
        )
        
        # ========== 返回文件路径 ==========
        # 如果用户选择了文件（file_path不为空字符串），返回路径；否则返回None
        return file_path if file_path else None
    
    def get_save_file_path(self, title="另存为项目", filter_str="项目文件 (*.vpp);;所有文件 (*)"):
        """
        获取保存文件路径
        显示文件保存对话框，让用户选择要保存的项目文件路径
        
        Args:
            title: 对话框标题，默认为"另存为项目"
            filter_str: 文件类型过滤器，默认为"项目文件 (*.vpp);;所有文件 (*)"
                       格式：显示名称 (扩展名);;显示名称 (扩展名)
            
        Returns:
            str: 用户选择的文件路径，如果用户取消选择则返回None
        """
        # ========== 显示文件保存对话框 ==========
        # 显示文件保存对话框，获取用户选择的保存路径
        # 参数：父窗口（主窗口）、对话框标题、起始目录（空字符串表示默认）、文件过滤器
        file_path, _ = QFileDialog.getSaveFileName(
            self.main_window,  # 父窗口
            title,  # 对话框标题
            "",  # 起始目录（空字符串表示使用默认目录）
            filter_str  # 文件类型过滤器
        )
        
        # ========== 返回文件路径 ==========
        # 如果用户选择了文件路径（file_path不为空字符串），返回路径；否则返回None
        return file_path if file_path else None
