"""
主窗口模块
提供完整的可视化编程工具主窗口功能
包含设计画布、代码编辑器、属性编辑器、控件调色板、AI对话等组件
"""

# 导入标准库模块
import sys  # 系统相关功能和参数
import os  # 操作系统接口
import re  # 正则表达式模块，用于文本处理
import io  # 输入输出流
import subprocess  # 子进程管理
from pathlib import Path  # 路径处理
from typing import Optional  # 类型提示
# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QApplication,  # 应用程序主类
    QMainWindow,  # 主窗口基类
    QWidget,  # 基础窗口部件
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QSplitter,  # 分割器
    QMenuBar,  # 菜单栏
    QMenu,  # 菜单
    QToolBar,  # 工具栏
    QStatusBar,  # 状态栏
    QFileDialog,  # 文件对话框
    QMessageBox,  # 消息框
    QDockWidget,  # 停靠窗口
    QListWidget,  # 列表控件
    QListWidgetItem,  # 列表项
    QTreeWidget,  # 树形控件
    QTreeWidgetItem,  # 树形控件项
    QLabel,  # 标签
    QPushButton,  # 按钮
    QFrame,  # 框架
    QScrollArea,  # 滚动区域
    QTabWidget,  # 选项卡控件
    QToolButton,  # 工具按钮
    QButtonGroup,  # 按钮组
    QGroupBox,  # 分组框
    QGridLayout,  # 网格布局
    QSizePolicy,  # 大小策略
    QComboBox,  # 下拉选择框
    QCheckBox,  # 复选框
    QPlainTextEdit,  # 纯文本编辑器
    QSlider,  # 滑动条
    QTextEdit  # 富文本编辑器
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QSize,  # 大小对象
    Signal,  # 信号定义
    QPoint,  # 点坐标
    QMimeData,  # MIME数据，用于拖放操作
    QTimer  # 定时器
)
from PySide6.QtWidgets import QApplication
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QIcon,  # 图标
    QPixmap,  # 像素图
    QFont,  # 字体
    QDrag,  # 拖放操作
    QPainter,  # 绘图器
    QPen,  # 画笔
    QBrush,  # 画刷
    QAction,  # 动作
    QColor,  # 颜色
    QTextCursor  # 文本光标
)

# 导入自定义模块
from .control_pyside import ControlManager  # 控件管理器
from .control_library_pyside import ControlLibrary  # 控件库
from .design_canvas_pyside import DesignCanvas  # 设计画布
from .control_palette_ui import ControlPaletteWidget  # 控件调色板
from .main_window_ui import MainWindowUI  # 主窗口UI管理器
from .project_manager_pyside import ProjectManager  # 项目管理器
from .code_generator_pyside import CodeGenerator  # 代码生成器
from .ai_chat_module import AIChatWidget  # AI对话组件
from .code_dialog_pyside import CodeDialog  # 代码对话框
from .undo_redo_manager_pyside import UndoRedoManager  # 撤销重做管理器
from .preview_window_pyside import BuiltInPreviewWindow  # 预览窗口
from .version import about_text  # 关于对话框文本
from module.property_editor_pyside import PropertyEditor  # 属性编辑器
from module.monaco_editor_widget import MonacoEditorWidget  # Monaco Editor 代码编辑器
from module.code_parser_pyside import CodeParser  # 代码解析器
from module.event_manager_pyside import EventManager  # 事件管理器
from module.settings_dialog import SettingsDialog  # 设置对话框
from module.theme_manager import ThemeManager  # 主题管理器
from module.file_manager import (
    is_text_file, get_file_display_name, create_file, create_directory,
    delete_file_or_directory, rename_file_or_directory, copy_file_or_directory,
    move_file_or_directory, get_relative_path, open_with_system_app,
    TEXT_FILE_EXTENSIONS
)  # 文件管理器


class MainWindow(QMainWindow):
    """
    主窗口类（新版本）
    继承自QMainWindow，提供完整的可视化编程工具主窗口功能
    包含设计画布、代码编辑器、属性编辑器、控件调色板、AI对话等组件
    """
    def __init__(self):
        """
        初始化主窗口
        创建所有必要的管理器组件并初始化用户界面
        """
        # 调用父类构造函数，初始化Qt主窗口
        super().__init__()
        
        # ========== 创建核心管理器组件 ==========
        # 控件管理器：负责所有控件的创建、删除、选择和属性管理
        self.control_manager = ControlManager()
        # 项目管理器：负责项目的保存、加载、修改状态跟踪等
        self.project_manager = ProjectManager(self.control_manager)
        # 代码生成器：将画布上的控件设计转换为Python代码
        self.code_generator = CodeGenerator(self.control_manager)
        # 撤销重做管理器：实现操作的撤销和重做功能
        self.undo_redo_manager = UndoRedoManager(self.control_manager)
        # 窗口操作撤销重做管理器：管理窗口创建/删除/重命名等操作的撤销重做
        self.window_operation_undo_stack = []
        self.window_operation_redo_stack = []
        # UI组件管理器：负责创建和管理菜单栏、工具栏、状态栏等UI组件
        self.ui_manager = MainWindowUI(self)
        # 代码解析器：将Python代码解析并转换为控件信息
        self.code_parser = CodeParser()
        # 主题管理器：管理主题和图标颜色
        self.theme_manager = ThemeManager()
        # 预览窗口对象，初始为None，当需要预览时会创建实例
        self.preview_window = None
        
        # 文件管理相关变量
        self._clipboard_file_path = None  # 剪贴板中的文件路径（用于复制/剪切）
        self._clipboard_is_cut = False  # 是否为剪切操作
        self._file_chinese_names = {}  # 文件中文名称字典 {文件路径: 中文名称}
        
        # 初始化UI：创建所有界面元素，包括布局、控件、菜单等
        self._init_ui()
        
        # 设置窗口标题：显示应用程序名称和作者信息
        self.setWindowTitle("python Visual Studio Code")
        # 设置窗口最小尺寸：宽度1200像素，高度800像素
        self.setMinimumSize(1200, 800)
        
        # 连接信号：将各个组件的事件信号连接到对应的处理函数
        self._connect_signals()
        
        # 加载设置：在窗口启动时加载所有保存的设置
        self._load_settings_on_startup()

    def _init_ui(self):
        # 从设置中获取图标大小，默认为64
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        com_weight = settings.value("icon/size", 64, type=int)
        
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        main_layout = QVBoxLayout(central_widget)
        main_layout.setContentsMargins(5, 5, 5, 5)
        main_layout.setSpacing(5)
        
        # 创建主水平分割器
        self.main_splitter = QSplitter(Qt.Horizontal)
        self.main_splitter.setChildrenCollapsible(True)
        self.main_splitter.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        main_layout.addWidget(self.main_splitter)
        
        # ========== 创建左侧面板（按钮栏 + 选择夹）==========
        left_panel = QWidget()
        left_panel_layout = QHBoxLayout(left_panel)
        left_panel_layout.setContentsMargins(0, 0, 0, 0)
        left_panel_layout.setSpacing(0)
        
        # 创建左侧按钮栏（垂直布局）
        button_bar = QWidget()
        button_bar.setObjectName("LeftIconBar")  # 设置唯一标识
        # 按钮栏宽度 = 图标大小 + 8像素边距
        button_bar.setFixedWidth(com_weight + 8)
        # 不设置独立样式，让按钮栏跟随主题变化
        button_bar_layout = QVBoxLayout(button_bar)
        button_bar_layout.setContentsMargins(2, 2, 2, 2)
        button_bar_layout.setSpacing(2)
        
        # 按钮顺序：项目目录（第一个）、其他按钮（中间）、AI（倒数第二个）、设置（倒数第一个）
        # 1. 项目目录按钮（第一个）
        self.btn_project = QToolButton()
        # 获取自定义图标颜色（如果设置了）
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        custom_icon_color = settings.value("icon/color", "", type=str)
        icon_path = self.theme_manager.get_icon_path("file.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_project.setIcon(QIcon(icon_path))
        self.btn_project.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_project.setCheckable(True)
        self.btn_project.setFixedSize(com_weight, com_weight)
        self.btn_project.setIconSize(QSize(com_weight, com_weight))
        self.btn_project.setToolTip("项目目录")
        self.btn_project.clicked.connect(lambda: self._toggle_left_panel_tab("项目目录"))
        button_bar_layout.addWidget(self.btn_project)
        
        # 2. 属性编辑按钮
        self.btn_property = QToolButton()
        icon_path = self.theme_manager.get_icon_path("property.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_property.setIcon(QIcon(icon_path))
        self.btn_property.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_property.setCheckable(True)
        self.btn_property.setChecked(True)
        self.btn_property.setFixedSize(com_weight, com_weight)
        self.btn_property.setIconSize(QSize(com_weight, com_weight))
        self.btn_property.setToolTip("属性编辑")
        self.btn_property.clicked.connect(lambda: self._toggle_left_panel_tab("属性编辑"))
        button_bar_layout.addWidget(self.btn_property)
        
        # 3. 组件库按钮
        self.btn_palette = QToolButton()
        icon_path = self.theme_manager.get_icon_path("component.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_palette.setIcon(QIcon(icon_path))
        self.btn_palette.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_palette.setCheckable(True)
        self.btn_palette.setFixedSize(com_weight, com_weight)
        self.btn_palette.setIconSize(QSize(com_weight, com_weight))
        self.btn_palette.setToolTip("组件库")
        self.btn_palette.clicked.connect(lambda: self._toggle_left_panel_tab("组件库"))
        button_bar_layout.addWidget(self.btn_palette)
        
        # 4. 函数库按钮
        self.btn_fnpy_library = QToolButton()
        icon_path = self.theme_manager.get_icon_path("fnpydata.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_fnpy_library.setIcon(QIcon(icon_path))
        self.btn_fnpy_library.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_fnpy_library.setCheckable(True)
        self.btn_fnpy_library.setFixedSize(com_weight, com_weight)
        self.btn_fnpy_library.setIconSize(QSize(com_weight, com_weight))
        self.btn_fnpy_library.setToolTip("函数库")
        self.btn_fnpy_library.clicked.connect(lambda: self._toggle_left_panel_tab("函数库"))
        button_bar_layout.addWidget(self.btn_fnpy_library)
        
        # 添加数据按钮（使用data.png图标）
        self.btn_data = QToolButton()
        icon_path = self.theme_manager.get_icon_path("data.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_data.setIcon(QIcon(icon_path))
        self.btn_data.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_data.setCheckable(False)
        self.btn_data.setFixedSize(com_weight, com_weight)
        self.btn_data.setIconSize(QSize(com_weight, com_weight))
        self.btn_data.setToolTip("数据")
        # 预留接口，暂时不做处理
        # self.btn_data.clicked.connect(self._show_data)
        button_bar_layout.addWidget(self.btn_data)
        
        button_bar_layout.addStretch()
        
        # 倒数第二个：AI聊天按钮
        self.btn_ai = QToolButton()
        icon_path = self.theme_manager.get_icon_path("AI.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_ai.setIcon(QIcon(icon_path))
        self.btn_ai.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_ai.setCheckable(True)
        self.btn_ai.setFixedSize(com_weight, com_weight)
        self.btn_ai.setIconSize(QSize(com_weight, com_weight))
        self.btn_ai.setToolTip("AI聊天")
        self.btn_ai.clicked.connect(lambda: self._toggle_left_panel_tab("AI聊天"))
        button_bar_layout.addWidget(self.btn_ai)
        
        # 倒数第一个：设置按钮
        self.btn_settings = QToolButton()
        icon_path = self.theme_manager.get_icon_path("setting.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_settings.setIcon(QIcon(icon_path))
        self.btn_settings.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_settings.setCheckable(False)
        self.btn_settings.setFixedSize(com_weight, com_weight)
        self.btn_settings.setIconSize(QSize(com_weight, com_weight))
        self.btn_settings.setToolTip("设置")
        self.btn_settings.clicked.connect(self._show_settings)
        button_bar_layout.addWidget(self.btn_settings)
        
        # 终端输出按钮（使用error.png图标）
        self.btn_toggle_output = QToolButton()
        icon_path = self.theme_manager.get_icon_path("error.png", icon_color=custom_icon_color if custom_icon_color else None)
        if os.path.exists(icon_path):
            self.btn_toggle_output.setIcon(QIcon(icon_path))
        self.btn_toggle_output.setToolButtonStyle(Qt.ToolButtonIconOnly)
        self.btn_toggle_output.setCheckable(True)
        self.btn_toggle_output.setFixedSize(com_weight, com_weight)
        self.btn_toggle_output.setIconSize(QSize(com_weight, com_weight))
        self.btn_toggle_output.setToolTip("显示/隐藏输出窗口")
        self.btn_toggle_output.clicked.connect(self._toggle_output_panel)
        button_bar_layout.addWidget(self.btn_toggle_output)
        
        # 创建选择夹（TabWidget）
        self.left_tab_widget = QTabWidget()
        self.left_tab_widget.setTabsClosable(True)
        self.left_tab_widget.tabCloseRequested.connect(self._close_left_tab)
        self.left_tab_widget.currentChanged.connect(self._on_left_tab_changed)
        self.left_tab_widget.setMinimumWidth(200)
        self.left_tab_widget.setMaximumWidth(400)
        
        # 创建各个组件
        self.property_editor = PropertyEditor()
        self.property_editor.setMinimumWidth(240)
        
        self.control_palette = ControlPaletteWidget()
        self.control_palette.setObjectName("ControlPaletteWidget")  # 设置唯一标识，用于排除主题设置
        self.control_palette.setMinimumWidth(0)
        
        self.project_preview_widget = self._create_project_preview_widget()
        
        self.ai_chat_widget = AIChatWidget()
        
        # 创建函数库组件
        from .fnpy_library_widget import fnpyLibraryWidget
        self.fnpy_library_widget = fnpyLibraryWidget()
        self.fnpy_library_widget.setMinimumWidth(0)
        # 连接函数点击信号，显示函数注释
        self.fnpy_library_widget.function_clicked.connect(self._on_function_clicked)
        
        # 将组件添加到选择夹
        self.left_tab_widget.addTab(self.property_editor, "属性编辑")
        self.left_tab_widget.addTab(self.control_palette, "组件库")
        self.left_tab_widget.addTab(self.fnpy_library_widget, "函数库")
        self.left_tab_widget.addTab(self.project_preview_widget, "项目目录")
        self.left_tab_widget.addTab(self.ai_chat_widget, "AI聊天")
        
        # 默认显示第一个标签页
        self.left_tab_widget.setCurrentIndex(0)
        
        # 将按钮栏和选择夹添加到左侧面板
        left_panel_layout.addWidget(button_bar)
        left_panel_layout.addWidget(self.left_tab_widget)
        
        # 保存左侧面板的引用，用于调整宽度
        self.left_panel = left_panel
        # 保存按钮栏的引用，用于调整图标大小
        self.button_bar = button_bar
        
        # 初始化时保存选择夹的默认宽度
        self._left_tab_widget_saved_width = 250
        
        # 将左侧面板添加到主分割器
        self.main_splitter.addWidget(left_panel)
        
        # ========== 创建中间区域（代码编辑器 + 设计画布）==========
        self.center_splitter = QSplitter(Qt.Horizontal)
        self.center_splitter.setChildrenCollapsible(True)
        
        # 代码编辑器容器
        code_container = QWidget()
        code_layout = QVBoxLayout(code_container)
        code_layout.setContentsMargins(4, 4, 4, 4)
        code_layout.setSpacing(4)
        header = QHBoxLayout()
        header.addWidget(QLabel("框架："))
        self.framework_combo_left = QComboBox()
        self.framework_combo_left.addItems(["PySide", "TKinter"])
        header.addWidget(self.framework_combo_left)
        self.sync_checkbox_left = QCheckBox("同步生成")
        self.sync_checkbox_left.setChecked(True)
        header.addWidget(self.sync_checkbox_left)
        self.refresh_button_left = QPushButton("刷新")
        header.addWidget(self.refresh_button_left)
        self.save_button_left = QPushButton("保存")
        header.addWidget(self.save_button_left)
        self.copy_button_left = QPushButton("复制")
        header.addWidget(self.copy_button_left)
        header.addStretch()
        code_layout.addLayout(header)
        
        # 创建多文件代码编辑器（使用TabWidget）
        self.code_tab_widget = QTabWidget()
        self.code_tab_widget.setTabsClosable(True)
        self.code_tab_widget.tabCloseRequested.connect(self._close_code_tab)
        self.code_tab_widget.currentChanged.connect(self._on_code_tab_changed)
        
        # 不再创建默认的代码编辑器，代码编辑器将根据窗口类动态创建
        # self.code_view_editor_left 保留作为向后兼容的引用，但不再添加到标签页
        self.code_view_editor_left = None  # 不再使用固定的默认编辑器
        self._code_editors = {}  # 字典，存储所有代码编辑器（包括画布代码和功能代码）
        
        code_layout.addWidget(self.code_tab_widget)
        self.center_splitter.addWidget(code_container)
        
        # 设计画布
        self.design_canvas = DesignCanvas()
        self.design_canvas.set_control_manager(self.control_manager)
        self.design_canvas.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
        
        # 创建设计画布的滚动区域
        canvas_scroll = QScrollArea()
        canvas_scroll.setWidgetResizable(False)
        canvas_scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        canvas_scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        canvas_scroll.setWidget(self.design_canvas)
        
        self.center_splitter.addWidget(canvas_scroll)
        
        # 设置中间分割器的比例
        self.center_splitter.setHandleWidth(6)
        self.center_splitter.setSizes([400, 600])
        self.center_splitter.setStretchFactor(0, 0)
        self.center_splitter.setStretchFactor(1, 1)
        
        # 将中间区域添加到主分割器
        self.main_splitter.addWidget(self.center_splitter)
        
        # 设置主分割器的比例
        self.undo_redo_manager.set_design_canvas(self.design_canvas)
        self.main_splitter.setHandleWidth(6)
        self.main_splitter.setSizes([250, 1200])
        self.main_splitter.setStretchFactor(0, 0)
        self.main_splitter.setStretchFactor(1, 1)
        
        # 设置画布尺寸
        display_width = max(600, self.design_canvas.canvas_width)
        display_height = max(400, self.design_canvas.canvas_height)
        try:
            self.design_canvas.setFixedSize(display_width, display_height)
        except Exception:
            pass
        
        # 创建菜单栏、工具栏、状态栏
        self.ui_manager.create_menu_bar()
        self.menuBar().setMaximumHeight(28)
        self.ui_manager.create_tool_bar()
        for toolbar in self.findChildren(QToolBar):
            toolbar.setMaximumHeight(36)
            # 确保所有工具栏都有objectName，避免saveState警告
            if not toolbar.objectName():
                toolbar.setObjectName(f"ToolBar_{id(toolbar)}")
        self.ui_manager.create_status_bar()
        self.property_editor.refresh_control_list(self.control_manager)
        try:
            self.undo_redo_manager.save_state("初始状态")
        except Exception:
            pass
        
        # 连接信号
        self.framework_combo_left.currentTextChanged.connect(self._refresh_code_view_left)
        self.refresh_button_left.clicked.connect(self._refresh_code_view_left)
        self.save_button_left.clicked.connect(self._save_current_code_left)
        self.copy_button_left.clicked.connect(self._copy_current_code_left)

        self._user_edited_left = False
        self._updating_code_left = False
        self._ensure_canvas_signal_hooks_left()
        self._refresh_code_view_left()
        self._update_code_highlight_from_selection()
        
        # 初始化项目预览
        self._refresh_project_preview()
        
        # 创建底部输出窗口
        self._create_output_panel()
        
        # 重定向print输出
        self._redirect_print_output()

    def _connect_signals(self):
        self.design_canvas.control_selected.connect(self._on_control_selected)
        self.design_canvas.control_added.connect(self._on_control_added)
        self.design_canvas.control_deleted.connect(self._on_control_deleted)
        self.design_canvas.control_moved.connect(self._on_control_moved)
        self.design_canvas.control_resized.connect(self._on_control_resized)
        self.design_canvas.canvas_resized.connect(self._on_canvas_resized)
        self.property_editor.property_changed.connect(self._on_property_changed)
        try:
            self.property_editor.event_binding_changed.connect(self._on_event_binding_changed)
        except Exception:
            pass
        try:
            self.ai_chat_widget.code_generated.connect(self._on_ai_code_generated_left)
        except Exception:
            pass

    def _refresh_code_view_left(self):
        sender = None
        try:
            sender = self.sender()
        except Exception:
            sender = None
        if hasattr(self, 'sync_checkbox_left') and not self.sync_checkbox_left.isChecked():
            return
        if self._user_edited_left and sender is not getattr(self, 'refresh_button_left', None):
            return
        canvas_width = self.design_canvas.canvas_width
        canvas_height = self.design_canvas.canvas_height
        try:
            # 获取当前窗口类
            current_window = None
            window_name = None
            if hasattr(self.project_manager, 'window_class_manager'):
                current_window = self.project_manager.window_class_manager.get_current_window_class()
                if current_window:
                    window_name = current_window.name
                    # 使用当前窗口类的代码生成器
                    self.code_generator = current_window.code_generator
                else:
                    window_name = "GeneratedWindow"
                    self.code_generator.set_window_class_name("GeneratedWindow")
                    window_name = self.project_manager.get_window_name() if hasattr(self.project_manager, 'get_window_name') else "GeneratedWindow"
                    self.code_generator.set_window_name(window_name)
            else:
                window_name = "GeneratedWindow"
                self.code_generator.set_window_class_name("GeneratedWindow")
                window_name = self.project_manager.get_window_name() if hasattr(self.project_manager, 'get_window_name') else "GeneratedWindow"
                self.code_generator.set_window_name(window_name)
            
            # 代码生成器属性会在下面根据当前窗口类设置，这里先跳过
            
            # 确保代码生成器使用正确的控件管理器
            # 代码生成器应该使用当前窗口类的控件管理器
            if current_window:
                # 使用当前窗口类的代码生成器（每个窗口类有自己的代码生成器）
                code_generator = current_window.code_generator
                # 确保代码生成器使用当前窗口类的控件管理器（重要！）
                code_generator.control_manager = current_window.control_manager
                # 确保控件管理器中有控件（调试用）
                if not current_window.control_manager.controls:
                    print(f"警告: 窗口 '{window_name}' 的控件管理器中没有控件")
            else:
                # 如果没有当前窗口类，使用主窗口的代码生成器
                code_generator = self.code_generator
                # 代码生成器应该使用画布当前使用的控件管理器
                if hasattr(self.design_canvas, 'control_manager'):
                    code_generator.control_manager = self.design_canvas.control_manager
            
            # 更新代码生成器的属性（使用当前窗口类的属性）
            if current_window:
                code_generator.set_window_title(current_window.window_title)
                code_generator.set_window_size(canvas_width, canvas_height)
                code_generator.set_window_class_name(window_name)
                code_generator.set_window_name(window_name)
                if current_window.window_icon_path:
                    code_generator.set_window_icon(current_window.window_icon_path)
                if current_window.pages:
                    code_generator.set_pages(current_window.pages)
                if current_window.dropdown_menus:
                    code_generator.set_dropdown_menus(current_window.dropdown_menus)
            
            # 生成代码
            fw = getattr(self, 'fixed_framework', None)
            framework = fw if fw else (self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide")
            code = code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            
            # 更新当前窗口类的画布代码标签页
            if window_name and hasattr(self, '_code_editors'):
                canvas_tab_name = f"{window_name}_canvas.py"
                if canvas_tab_name in self._code_editors:
                    # 更新对应的画布代码标签页
                    editor = self._code_editors[canvas_tab_name]
                    # 直接更新代码，不进行比较（因为MonacoEditorWidget的toPlainText可能返回缓存值）
                    vsb = editor.verticalScrollBar()
                    pos = vsb.value() if vsb else 0
                    self._updating_code_left = True
                    editor.setPlainText(code)
                    self._updating_code_left = False
                    try:
                        if vsb:
                            vsb.setValue(pos)
                    except Exception:
                        pass
                    # 保存到窗口类
                    if current_window:
                        current_window.canvas_code = code
                    self._user_edited_left = False
                else:
                    # 如果标签页不存在，创建它
                    if current_window:
                        self._update_canvas_code_tab(window_name, current_window)
                    else:
                        # 如果没有窗口类，尝试创建或使用当前活动的编辑器
                        current_editor = self._get_current_code_editor()
                        if current_editor:
                            vsb = current_editor.verticalScrollBar()
                            pos = vsb.value() if vsb else 0
                            self._updating_code_left = True
                            current_editor.setPlainText(code)
                            self._updating_code_left = False
                            try:
                                if vsb:
                                    vsb.setValue(pos)
                            except Exception:
                                pass
                            self._user_edited_left = False
            else:
                # 如果没有窗口类，尝试创建或使用当前活动的编辑器
                current_editor = self._get_current_code_editor()
                if current_editor:
                    vsb = current_editor.verticalScrollBar()
                    pos = vsb.value() if vsb else 0
                    self._updating_code_left = True
                    current_editor.setPlainText(code)
                    self._updating_code_left = False
                    try:
                        if vsb:
                            vsb.setValue(pos)
                    except Exception:
                        pass
                    self._user_edited_left = False
            
            # 只在有选中控件时才更新代码高亮
            try:
                selected_ids = []
                if hasattr(self.control_manager, 'get_selected_ids'):
                    selected_ids = list(self.control_manager.get_selected_ids()) or []
                else:
                    selected_ids = list(getattr(self.control_manager, 'selected_control_ids', []))
                if selected_ids:
                    self._update_code_highlight_from_selection()
            except Exception:
                pass
        except Exception as e:
            print(f"刷新代码视图失败: {e}")
            import traceback
            traceback.print_exc()

    def _sync_code_to_canvas(self):
        """将代码同步到画布"""
        # 获取当前活动的代码编辑器
        current_editor = self._get_current_code_editor()
        if not current_editor:
            QMessageBox.warning(self, "警告", "没有打开的代码编辑器，请先打开代码标签页")
            return
        
        code = current_editor.toPlainText()
        if not code:
            return
            
        # 解析代码
        result = self.code_parser.parse_code(code)
        if not result:
            QMessageBox.warning(self, "错误", "无法解析代码，请检查语法")
            return
            
        # 确认操作
        reply = QMessageBox.question(
            self, "确认同步", 
            "同步操作将覆盖当前画布上的所有内容，是否继续？",
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply != QMessageBox.Yes:
            return
            
        try:
            # 1. 更新窗口属性
            window_props = result.get("window", {})
            if "width" in window_props and "height" in window_props:
                w, h = window_props["width"], window_props["height"]
                self.design_canvas.set_canvas_size(w, h)
                self.project_manager.set_canvas_size(w, h)
            
            if "title" in window_props:
                self.design_canvas.set_window_property("window_title", window_props["title"])
                
            # 2. 清空现有控件
            self.control_manager.clear_all()
            
            # 3. 创建新控件
            controls_data = result.get("controls", [])
            
            id_map = {} 
            
            for data in controls_data:
                control_id = data["id"]
                control_type = data["type"]
                parent_var = data["parent"]
                
                # 确定父控件ID
                parent_id = None
                if parent_var:
                    if parent_var in id_map:
                        parent_id = id_map[parent_var]
                    elif parent_var.startswith("page_"):
                        pass
                    elif parent_var not in ["central_widget", "tabs"]:
                        parent_id = parent_var
                
                # 创建控件
                control = self.control_manager.add_control(
                    control_type=control_type,
                    x=0, y=0, 
                    parent_id=parent_id,
                    control_id=control_id
                )
                
                if control:
                    id_map[control_id] = control.id
                    
                    # 应用属性
                    props = data.get("properties", {})
                    
                    # 位置和大小
                    if "x" in data: control.x = data["x"]
                    if "y" in data: control.y = data["y"]
                    if "width" in data: control.width = data["width"]
                    if "height" in data: control.height = data["height"]
                    
                    # 其他属性
                    for k, v in props.items():
                        if k not in ["x", "y", "width", "height"]:
                            control.properties[k] = v
                    
                    # 确保有页面属性，否则可能不显示
                    if "page" not in control.properties:
                        control.properties["page"] = getattr(self.design_canvas, 'active_page', '页面1')
                            
            # 4. 更新画布
            self.design_canvas.update()
            self.property_editor.refresh_control_list(self.control_manager)
            self.status_label.setText("已从代码同步到画布")
            
            # 重置用户编辑标志
            self._user_edited_left = False
            
            # 5. 重新生成标准代码
            self._refresh_code_view_left()
            
            # 6. 刷新项目预览
            self._refresh_project_preview()
            
        except Exception as e:
            QMessageBox.warning(self, "错误", f"同步失败: {str(e)}")
            import traceback
            traceback.print_exc()

    def _ensure_canvas_signal_hooks_left(self):
        try:
            self.design_canvas.control_added.connect(self._on_canvas_changed_left)
            self.design_canvas.control_deleted.connect(self._on_canvas_changed_left)
            self.design_canvas.control_moved.connect(lambda *_: self._on_canvas_changed_left())
            self.design_canvas.control_resized.connect(lambda *_: self._on_canvas_changed_left())
            self.design_canvas.canvas_resized.connect(lambda *_: self._on_canvas_changed_left())
        except Exception:
            pass

    def _on_canvas_changed_left(self, *args, **kwargs):
        self._refresh_code_view_left()

    def _on_code_text_changed_left(self):
        if self._updating_code_left:
            return
        self._user_edited_left = True

    def _save_current_code_left(self):
        try:
            editor = self._get_current_code_editor()
            current_index = self.code_tab_widget.currentIndex()
            tab_name = self.code_tab_widget.tabText(current_index) if current_index >= 0 else ""
            
            # 如果是事件函数文件，保存到窗口类
            if "_function.py" in tab_name:
                content = editor.toPlainText()
                # 从标签名提取窗口名
                window_name = tab_name[:-13]  # 去掉 '_function.py'
                if hasattr(self.project_manager, 'window_class_manager'):
                    current_window = self.project_manager.window_class_manager.get_window_class(window_name)
                    if current_window:
                        current_window.event_function_file = content
                        self.project_manager.set_modified(True)
                        self.statusBar().showMessage(f"已保存事件函数文件: {tab_name}")
                        return
                # 向后兼容
                if hasattr(self.project_manager, 'set_event_function_file'):
                    self.project_manager.set_event_function_file(content)
                self.statusBar().showMessage(f"已保存事件函数文件: {tab_name}")
                return
            
            # 如果是画布代码文件，保存到窗口类
            if "_canvas.py" in tab_name:
                content = editor.toPlainText()
                # 从标签名提取窗口名
                window_name = tab_name[:-11]  # 去掉 '_canvas.py'
                if hasattr(self.project_manager, 'window_class_manager'):
                    current_window = self.project_manager.window_class_manager.get_window_class(window_name)
                    if current_window:
                        current_window.canvas_code = content
                        self.project_manager.set_modified(True)
                        self.statusBar().showMessage(f"已保存画布代码: {tab_name}")
                        return
            
            # 否则保存画布代码到文件（向后兼容）
            framework = getattr(self, 'fixed_framework', None)
            project_dir = getattr(self, 'project_dir', None)
            if framework and project_dir:
                import os
                fname = 'main_PySide.py' if framework.lower().startswith('py') else 'main_TKinter.py'
                file_path = os.path.join(project_dir, fname)
                os.makedirs(os.path.dirname(file_path), exist_ok=True)
                with open(file_path, 'w', encoding='utf-8') as f:
                    f.write(editor.toPlainText())
                self.statusBar().showMessage(f"已保存到项目: {file_path}")
            else:
                file_path, _ = QFileDialog.getSaveFileName(self, "保存代码", "generated.py", "Python 文件 (*.py);;所有文件 (*)")
                if file_path:
                    with open(file_path, 'w', encoding='utf-8') as f:
                        f.write(editor.toPlainText())
        except Exception as e:
            print(f"保存代码失败: {e}")
            import traceback
            traceback.print_exc()

    def _copy_current_code_left(self):
        try:
            editor = self._get_current_code_editor()
            QApplication.clipboard().setText(editor.toPlainText())
        except Exception:
            pass

    def _on_ai_code_generated_left(self, code: str, framework: str):
        """
        处理AI代码生成信号
        将生成的代码添加到代码编辑器
        
        Args:
            code: 生成的代码内容
            framework: 框架名称
        """
        try:
            print(f"[DEBUG] 收到代码生成信号，代码长度: {len(code) if code else 0}")
            
            # 获取当前活动的代码编辑器
            current_editor = self._get_current_code_editor()
            
            if not current_editor:
                # 如果没有活动的编辑器，尝试创建或切换到对应的代码标签页
                print("[DEBUG] 没有活动的代码编辑器，尝试创建标签页...")
                try:
                    # 获取当前窗口类
                    if hasattr(self.project_manager, 'window_class_manager'):
                        current_window = self.project_manager.window_class_manager.get_current_window_class()
                        if current_window:
                            window_name = current_window.name
                            # 更新画布代码标签页（如果不存在会创建）
                            self._update_canvas_code_tab(window_name, current_window)
                            # 再次尝试获取编辑器
                            current_editor = self._get_current_code_editor()
                except Exception as e:
                    print(f"[DEBUG] 创建标签页失败: {e}")
                    import traceback
                    traceback.print_exc()
            
            if current_editor:
                # 更新代码编辑器
                self._updating_code_left = True
                current_editor.setPlainText(code)
                self._updating_code_left = False
                self._user_edited_left = False
                print(f"[DEBUG] 代码已成功添加到编辑器，长度: {len(code)}")
            else:
                # 如果仍然没有编辑器，显示警告
                print("[DEBUG] 警告：无法找到或创建代码编辑器")
                QMessageBox.warning(
                    self, 
                    "无法添加代码", 
                    "没有活动的代码编辑器。\n\n请先打开或创建一个代码标签页，然后重试。"
                )
                return
            
            # 更新框架选择
            try:
                if framework:
                    self.framework_combo_left.setCurrentText(framework)
            except Exception:
                pass
            
            # 更新状态栏
            try:
                self.statusBar().showMessage(f"AI生成的代码已添加到代码编辑器（{len(code)} 字符）")
            except Exception:
                pass
            
            # 同步到画布（可选，根据需求决定是否启用）
            try:
                # 注释掉自动同步，避免覆盖用户的设计
                # self._sync_canvas_from_code(code, framework)
                pass
            except Exception:
                pass
            
            # 刷新代码视图（可选）
            try:
                # 注释掉自动刷新，避免覆盖用户编辑
                # self._refresh_code_view_left()
                pass
            except Exception:
                pass
                
        except Exception as e:
            print(f"[DEBUG] 处理代码生成信号时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.warning(
                self,
                "添加代码失败",
                f"无法将AI生成的代码添加到编辑器：\n{str(e)}"
            )
            
    def _parse_and_sync_code(self):
        """
        解析代码页中的代码并同步到设计画布，同时更新代码页为标准格式
        """
        try:
            # 获取当前活动的代码编辑器中的代码
            current_editor = self._get_current_code_editor()
            if not current_editor:
                QMessageBox.warning(self, "警告", "没有打开的代码编辑器，请先打开代码标签页")
                return
            
            code_text = current_editor.toPlainText().strip()
            if not code_text:
                QMessageBox.warning(self, "警告", "代码页为空，请先输入代码")
                return
            
            # 使用CodeParser解析代码
            parser = CodeParser()
            parse_result = parser.parse_code(code_text)
            
            # 使用现有的_sync_canvas_from_code方法同步到画布
            framework = self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide"
            self._sync_canvas_from_code(code_text, framework)
            
            # 使用代码生成器重新生成标准格式的代码
            canvas_width = self.design_canvas.canvas_width
            canvas_height = self.design_canvas.canvas_height
            
            # 生成代码
            generated_code = self.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            
            # 更新代码编辑器
            if current_editor:
                self._updating_code_left = True
                current_editor.setPlainText(generated_code)
                self._updating_code_left = False
            
            # 刷新画布和代码高亮
            self._reload_canvas()
            self._refresh_code_view_left()
            
            self.statusBar().showMessage("代码解析成功并同步到画布和代码页")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"代码解析失败: {str(e)}")
            self.statusBar().showMessage("代码解析失败")

    def _sync_canvas_from_code(self, code_text: str, framework: str):
        """
        从代码同步到画布
        解析Python代码，提取窗口大小、标题、控件信息，并同步到设计画布
        
        Args:
            code_text: 要解析的Python代码文本
            framework: 框架名称（"PySide"或"TKinter"）
        """
        # ========== 初始化代码文本 ==========
        # 获取代码文本，如果为空则使用空字符串（避免None错误）
        t = code_text or ""
        
        # ========== 提取窗口大小 ==========
        # 使用正则表达式搜索窗口大小设置：支持setFixedSize和resize两种方法
        # 正则表达式匹配：setFixedSize(数字, 数字) 或 resize(数字, 数字)
        size_m = re.search(r"setFixedSize\s*\(\s*(\d+)\s*,\s*(\d+)\s*\)|resize\s*\(\s*(\d+)\s*,\s*(\d+)\s*\)", t)
        # 初始化宽度和高度变量为None
        w = None
        h = None
        # 检查是否找到大小设置匹配
        if size_m:
            # 如果找到匹配，尝试提取尺寸值
            # 检查是否为setFixedSize格式（group(1)和group(2)）
            if size_m.group(1) and size_m.group(2):
                # 如果是setFixedSize格式，提取第一个和第二个捕获组（宽度和高度）
                w = int(size_m.group(1))
                h = int(size_m.group(2))
            # 检查是否为resize格式（group(3)和group(4)）
            elif size_m.group(3) and size_m.group(4):
                # 如果是resize格式，提取第三个和第四个捕获组（宽度和高度）
                w = int(size_m.group(3))
                h = int(size_m.group(4))
        
        # ========== 如果未找到大小设置，尝试从setGeometry提取 ==========
        # 检查是否成功提取到宽度和高度（两者都不为None）
        if not (w and h):
            # 如果未找到大小，尝试从setGeometry方法中提取（该方法包含位置和大小）
            # 正则表达式匹配：setGeometry(数字, 数字, 宽度, 高度)，提取后两个参数（宽度和高度）
            gm = re.search(r"setGeometry\s*\(\s*\d+\s*,\s*\d+\s*,\s*(\d+)\s*,\s*(\d+)\s*\)", t)
            # 检查是否找到setGeometry匹配
            if gm:
                # 如果找到，提取宽度（第一个捕获组）和高度（第二个捕获组）
                w = int(gm.group(1))
                h = int(gm.group(2))
        
        # ========== 提取窗口标题 ==========
        # 使用正则表达式搜索窗口标题设置：setWindowTitle("标题") 或 setWindowTitle('标题')
        # 正则表达式匹配引号内的标题文本（支持单引号和双引号）
        title_m = re.search(r"setWindowTitle\s*\(\s*([\"\'])(.*?)\1\s*\)", t)
        # 如果找到匹配，提取标题文本（group(2)是标题内容）；否则使用None
        title = title_m.group(2) if title_m else None
        
        # ========== 应用窗口大小到画布 ==========
        # 检查是否成功提取到宽度和高度
        if w and h:
            # 如果提取成功，尝试设置画布大小
            try:
                # 设置画布宽度
                self.design_canvas.set_canvas_width(w)
                # 设置画布高度
                self.design_canvas.set_canvas_height(h)
            except Exception:
                # 如果设置失败，忽略异常（继续执行其他操作）
                pass
        
        # ========== 应用窗口标题到画布 ==========
        # 检查是否成功提取到标题
        if title:
            # 如果提取成功，尝试设置画布窗口标题
            try:
                # 设置窗口标题
                self.design_canvas.set_window_title(title)
            except Exception:
                # 如果设置失败，忽略异常
                pass
        
        # ========== 清空现有控件 ==========
        # 在添加新控件之前，先清空画布上的所有现有控件
        try:
            # 清空所有控件
            self.design_canvas.clear_controls()
        except Exception:
            # 如果清空失败，忽略异常（可能画布尚未初始化）
            pass
        
        # ========== 定义PySide类名到控件类型的映射 ==========
        # 创建字典，将PySide6的QWidget类名映射到内部控件类型名称
        class_map = {
            'QPushButton': 'Button',      # 按钮控件
            'QLabel': 'Label',            # 标签控件
            'QLineEdit': 'LineEdit',      # 单行文本输入框
            'QTextEdit': 'TextEdit',      # 多行文本编辑器
            'QCheckBox': 'CheckBox',      # 复选框
            'QRadioButton': 'RadioButton', # 单选按钮
            'QComboBox': 'ComboBox',      # 下拉选择框
            'QProgressBar': 'ProgressBar', # 进度条
            'QGroupBox': 'GroupBox',      # 分组框
            'QTabWidget': 'TabWidget'     # 选项卡控件
        }
        
        # ========== 定义正则表达式模式 ==========
        # 编译控件实例化模式：匹配 self.变量名 = Q类名( 的格式
        inst_pattern = re.compile(r"self\.(\w+)\s*=\s*(Q\w+)\s*\(")
        # 定义几何形状模式模板：匹配 setGeometry(x, y, width, height) 的格式
        geom_pattern_tpl = r"self\.%s\.setGeometry\s*\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)"
        # 定义文本设置模式模板：匹配 setText("文本") 的格式
        text_pattern_tpl = r"self\.%s\.setText\s*\(\s*([\"\'])(.*?)\1\s*\)"
        # 定义标题设置模式模板：匹配 setTitle("标题") 的格式
        title_set_tpl = r"self\.%s\.setTitle\s*\(\s*([\"\'])(.*?)\1\s*\)"
        
        # ========== 初始化默认位置和间距 ==========
        # 设置默认X坐标位置
        pos_x = 40
        # 设置默认Y坐标位置
        pos_y = 80
        # 设置垂直间距（控件之间的垂直距离）
        gap_y = 12
        # 初始化默认值字典（用于存储控件的默认属性）
        defaults = {}
        
        # ========== 导入控件库 ==========
        # 尝试导入控件库，用于获取控件的默认尺寸等信息
        try:
            from .control_library_pyside import ControlLibrary
        except Exception:
            # 如果导入失败，将ControlLibrary设置为None
            ControlLibrary = None
        
        # ========== 遍历代码中的所有控件实例化语句 ==========
        # 使用正则表达式查找所有控件实例化语句：self.变量名 = Q类名
        # finditer返回所有匹配的迭代器，每次迭代返回一个Match对象
        for m in re.finditer(r"self\.(\w+)\s*=\s*(Q\w+)", t):
            # ========== 提取控件变量名和类名 ==========
            # 从正则匹配结果中提取控件变量名（group(1)是变量名，如"button1"）
            var = m.group(1)
            # 从正则匹配结果中提取PySide类名（group(2)是QWidget类名，如"QPushButton"）
            qclass = m.group(2)
            # 通过映射表将QWidget类名转换为内部控件类型名称
            ctype = class_map.get(qclass)
            # 检查控件类型是否在映射表中（是否支持该控件类型）
            if not ctype:
                # 如果不在映射表中（未知的控件类型），跳过此控件，继续处理下一个
                continue
            # ========== 初始化几何属性变量 ==========
            # 初始化X坐标（位置）
            gx = None
            # 初始化Y坐标（位置）
            gy = None
            # 初始化宽度
            gw = None
            # 初始化高度
            gh = None
            
            # ========== 提取控件文本内容 ==========
            # 使用正则表达式搜索控件文本设置：self.变量名.setText("文本")
            tm = re.search(text_pattern_tpl % re.escape(var), t)
            # 如果找到匹配，提取文本内容（group(2)）；否则使用None
            text_val = tm.group(2) if tm else None
            # 检查是否成功提取到文本
            if not text_val:
                # 如果未找到setText调用，尝试从构造函数参数中提取文本
                # 搜索：变量名 = Q类名("文本") 的格式
                tm_init = re.search(rf"{re.escape(var)}\s*=\s*{re.escape(qclass)}\s*\(\s*([\"\'])(.*?)\1", t)
                # 如果找到构造函数中的文本参数
                if tm_init:
                    # 提取构造函数中的文本内容
                    text_val = tm_init.group(2)
            
            # ========== 提取控件几何形状（位置和大小） ==========
            # 使用正则表达式搜索控件的几何形状设置：self.变量名.setGeometry(x, y, width, height)
            gm = re.search(geom_pattern_tpl % re.escape(var), t)
            # 检查是否找到几何形状设置
            if gm:
                # 如果找到，提取X坐标（第一个捕获组）
                gx = int(gm.group(1))
                # 提取Y坐标（第二个捕获组）
                gy = int(gm.group(2))
                # 提取宽度（第三个捕获组）
                gw = int(gm.group(3))
                # 提取高度（第四个捕获组）
                gh = int(gm.group(4))
            
            # ========== 获取控件默认尺寸 ==========
            # 初始化默认宽度和高度
            tw = None
            th = None
            # 检查控件库是否可用
            if ControlLibrary:
                # 如果控件库可用，尝试获取该控件类型的默认尺寸
                try:
                    # 获取控件的默认尺寸（返回(width, height)元组）
                    ds = ControlLibrary.get_default_size(ctype)
                    # 解包元组，获取默认宽度和高度
                    tw, th = ds[0], ds[1]
                except Exception:
                    # 如果获取默认尺寸失败，保持tw和th为None
                    pass
            
            # ========== 确定控件的最终位置和大小 ==========
            # 确定X坐标：如果从代码中提取到则使用提取值，否则使用默认位置
            x = gx if gx is not None else pos_x
            # 确定Y坐标：如果从代码中提取到则使用提取值，否则使用默认位置
            y = gy if gy is not None else pos_y
            # 确定宽度：如果从代码中提取到则使用提取值，否则使用默认宽度（控件库默认值或120）
            wv = gw if gw is not None else (tw or 120)
            # 确定高度：如果从代码中提取到则使用提取值，否则使用默认高度（控件库默认值或30）
            hv = gh if gh is not None else (th or 30)
            ctrl = None
            try:
                ctrl = self.design_canvas.add_control(ctype, x, y)
            except Exception:
                ctrl = None
            if not ctrl:
                continue
            try:
                ctrl.width = wv
                ctrl.height = hv
            except Exception:
                pass
            if text_val:
                try:
                    ctrl.properties['text'] = text_val
                except Exception:
                    pass
            if ctype == 'GroupBox':
                tm2 = re.search(title_set_tpl % re.escape(var), t)
                title2 = tm2.group(2) if tm2 else None
                if title2:
                    try:
                        ctrl.properties['title'] = title2
                    except Exception:
                        pass
            pos_y += (hv + gap_y)
        try:
            self.property_editor.refresh_control_list(self.control_manager)
        except Exception:
            pass
        for m in re.finditer(r"\n(\w+)\s*=\s*(Q\w+)\s*\(", t):
            # ========== 提取控件变量名和类名 ==========
            # 从正则匹配结果中提取控件变量名（group(1)是变量名，如"button1"）
            var = m.group(1)
            # 从正则匹配结果中提取PySide类名（group(2)是QWidget类名，如"QPushButton"）
            qclass = m.group(2)
            # 通过映射表将QWidget类名转换为内部控件类型名称
            ctype = class_map.get(qclass)
            # 检查控件类型是否在映射表中（是否支持该控件类型）
            if not ctype:
                # 如果不在映射表中（未知的控件类型），跳过此控件，继续处理下一个
                continue
            
            # ========== 初始化几何属性变量 ==========
            # 初始化X坐标（位置）
            gx = None
            # 初始化Y坐标（位置）
            gy = None
            # 初始化宽度
            gw = None
            # 初始化高度
            gh = None
            
            # ========== 提取控件文本内容 ==========
            # 使用正则表达式搜索控件文本设置：self.变量名.setText("文本")
            tm = re.search(text_pattern_tpl % re.escape(var), t)
            # 如果找到匹配，提取文本内容（group(2)）；否则使用None
            text_val = tm.group(2) if tm else None
            # 检查是否成功提取到文本
            if not text_val:
                # 如果未找到setText调用，尝试从构造函数参数中提取文本
                # 搜索：变量名 = Q类名("文本") 的格式
                tm_init = re.search(rf"{re.escape(var)}\s*=\s*{re.escape(qclass)}\s*\(\s*([\"\'])(.*?)\1", t)
                # 如果找到构造函数中的文本参数
                if tm_init:
                    # 提取构造函数中的文本内容
                    text_val = tm_init.group(2)
            
            # ========== 提取控件几何形状（位置和大小） ==========
            # 使用正则表达式搜索控件的几何形状设置：self.变量名.setGeometry(x, y, width, height)
            gm = re.search(geom_pattern_tpl % re.escape(var), t)
            # 检查是否找到几何形状设置
            if gm:
                # 如果找到，提取X坐标（第一个捕获组）
                gx = int(gm.group(1))
                # 提取Y坐标（第二个捕获组）
                gy = int(gm.group(2))
                # 提取宽度（第三个捕获组）
                gw = int(gm.group(3))
                # 提取高度（第四个捕获组）
                gh = int(gm.group(4))
            
            # ========== 获取控件默认尺寸 ==========
            # 初始化默认宽度和高度
            tw = None
            th = None
            # 检查控件库是否可用
            if ControlLibrary:
                # 如果控件库可用，尝试获取该控件类型的默认尺寸
                try:
                    # 获取控件的默认尺寸（返回(width, height)元组）
                    ds = ControlLibrary.get_default_size(ctype)
                    # 解包元组，获取默认宽度和高度
                    tw, th = ds[0], ds[1]
                except Exception:
                    # 如果获取默认尺寸失败，保持tw和th为None
                    pass
            
            # ========== 确定控件的最终位置和大小 ==========
            # 确定X坐标：如果从代码中提取到则使用提取值，否则使用默认位置
            x = gx if gx is not None else pos_x
            # 确定Y坐标：如果从代码中提取到则使用提取值，否则使用默认位置
            y = gy if gy is not None else pos_y
            # 确定宽度：如果从代码中提取到则使用提取值，否则使用默认宽度（控件库默认值或120）
            wv = gw if gw is not None else (tw or 120)
            # 确定高度：如果从代码中提取到则使用提取值，否则使用默认高度（控件库默认值或30）
            hv = gh if gh is not None else (th or 30)
            ctrl = None
            try:
                ctrl = self.design_canvas.add_control(ctype, x, y)
            except Exception:
                ctrl = None
            if not ctrl:
                continue
            try:
                ctrl.width = wv
                ctrl.height = hv
            except Exception:
                pass
            if text_val:
                try:
                    ctrl.properties['text'] = text_val
                except Exception:
                    pass
            if ctype == 'GroupBox':
                tm2 = re.search(title_set_tpl % re.escape(var), t)
                title2 = tm2.group(2) if tm2 else None
                if title2:
                    try:
                        ctrl.properties['title'] = title2
                    except Exception:
                        pass
            pos_y += (hv + gap_y)
        try:
            self.property_editor.refresh_control_list(self.control_manager)
        except Exception:
            pass
        if 'tkinter' in t or 'tk.' in t or 'ttk.' in t:
            for m in re.finditer(r"\n(\w+)\s*=\s*(tk\.|ttk\.)?(\w+)\s*\(", t):
                var = m.group(1)
                base = m.group(3)
                qt = None
                if base.lower() == 'button':
                    qt = 'Button'
                elif base.lower() == 'label':
                    qt = 'Label'
                elif base.lower() == 'entry':
                    qt = 'LineEdit'
                elif base.lower() == 'text':
                    qt = 'TextEdit'
                elif base.lower() == 'checkbutton':
                    qt = 'CheckBox'
                elif base.lower() == 'radiobutton':
                    qt = 'RadioButton'
                elif base.lower() == 'combobox':
                    qt = 'ComboBox'
                else:
                    qt = None
                if not qt:
                    continue
                tw = None
                th = None
                gx = None
                gy = None
                if ControlLibrary:
                    try:
                        ds = ControlLibrary.get_default_size(qt)
                        tw, th = ds[0], ds[1]
                    except Exception:
                        pass
                x = gx if gx is not None else pos_x
                y = gy if gy is not None else pos_y
                wv = tw or 120
                hv = th or 30
                ctrl = None
                try:
                    ctrl = self.design_canvas.add_control(qt, x, y)
                except Exception:
                    ctrl = None
                if not ctrl:
                    continue
                try:
                    ctrl.width = wv
                    ctrl.height = hv
                except Exception:
                    pass
                pos_y += (hv + gap_y)
            try:
                self.property_editor.refresh_control_list(self.control_manager)
            except Exception:
                pass

    def _on_control_selected(self, control_id: str):
        """
        控件选中事件处理函数
        当用户在画布上选中控件或画布时调用此方法
        
        Args:
            control_id: 控件ID，如果是"canvas"表示选中画布，如果是"multiple_selected"表示多选
        """
        # ========== 处理画布选中 ==========
        # 检查是否选中了画布本身（而不是控件）
        if control_id == "canvas":
            # 如果选中画布，设置属性编辑器为画布模式（不显示控件属性，显示画布属性）
            self.property_editor.set_control(None, is_canvas=True)
            # 更新状态栏标签，显示"已选中画布"
            self.status_label.setText("已选中画布")
        
        # ========== 处理多选状态 ==========
        # 检查是否是多个控件被选中
        elif control_id == "multiple_selected":
            # 如果是多选状态，获取所有选中的控件ID列表
            selected_ids = self.control_manager.get_selected_ids()
            # 检查是否有选中的控件
            if selected_ids:
                # 如果有选中控件，更新状态栏显示选中数量
                self.status_label.setText(f"已选中 {len(selected_ids)} 个控件")
            # 更新代码高亮显示（高亮所有选中的控件对应的代码）
            self._update_code_highlight_from_selection()
            # 多选状态下不设置属性编辑器（因为多选时不编辑属性），直接返回
            return
        
        # ========== 处理单个控件选中 ==========
        else:
            # 如果选中了单个控件，获取该控件对象
            control = self.control_manager.get_control(control_id)
            # 检查控件对象是否存在
            if control:
                # 如果控件存在，设置属性编辑器为编辑该控件
                self.property_editor.set_control(control)
                # 更新状态栏标签，显示控件的ID
                self.status_label.setText(f"已选中控件: {control.id}")
            else:
                # 如果控件不存在，清空属性编辑器（不显示任何属性）
                self.property_editor.set_control(None)
                # 更新状态栏标签，显示未选中任何控件
                self.status_label.setText("未选中任何控件")
        
        # ========== 更新代码高亮 ==========
        # 无论选中什么，都要更新代码编辑器中的高亮显示（高亮对应的代码行）
        self._update_code_highlight_from_selection()

    def _update_code_highlight_from_selection(self):
        """
        根据选中的控件更新代码编辑器中的高亮显示
        在代码编辑器中高亮显示与选中控件相关的代码行
        """
        # ========== 获取当前窗口类和对应的代码编辑器 ==========
        current_window = None
        window_name = None
        code_editor = None
        code_generator = None
        
        # 获取当前窗口类
        if hasattr(self.project_manager, 'window_class_manager'):
            current_window = self.project_manager.window_class_manager.get_current_window_class()
            if current_window:
                window_name = current_window.name
                code_generator = current_window.code_generator
                # 获取当前窗口对应的画布代码编辑器
                canvas_tab_name = f"{window_name}_canvas.py"
                if hasattr(self, '_code_editors') and canvas_tab_name in self._code_editors:
                    code_editor = self._code_editors[canvas_tab_name]
        
        # 如果没有找到对应的编辑器，尝试获取当前活动的编辑器
        if not code_editor:
            code_editor = self._get_current_code_editor()
            if code_editor:
                # 尝试从当前窗口类获取代码生成器
                if current_window:
                    code_generator = current_window.code_generator
                else:
                    code_generator = self.code_generator
            else:
                # 如果没有活动的编辑器，无法高亮，直接返回
                return
        
        # ========== 先清除之前的高亮 ==========
        # 在应用新高亮之前，先清除所有之前的高亮
        try:
            is_monaco = isinstance(code_editor, MonacoEditorWidget)
            if is_monaco:
                # MonacoEditorWidget: 清除所有高亮
                if hasattr(code_editor, 'highlight_lines'):
                    code_editor.highlight_lines([])
            else:
                # QPlainTextEdit: 清除所有外部高亮
                if hasattr(code_editor, 'set_external_highlights'):
                    code_editor.set_external_highlights([])
                elif hasattr(code_editor, 'setExtraSelections'):
                    code_editor.setExtraSelections([])
        except Exception:
            # 如果清除高亮失败，忽略异常（不阻止程序继续运行）
            pass
        
        # ========== 获取选中的控件ID列表 ==========
        # 初始化选中ID列表为空列表
        ids = []
        try:
            # 尝试获取选中控件ID列表
            # 检查控件管理器是否有get_selected_ids方法
            if hasattr(self.control_manager, 'get_selected_ids'):
                # 如果有该方法，调用它获取选中ID列表，转换为列表（如果为None则使用空列表）
                ids = list(self.control_manager.get_selected_ids()) or []
            else:
                # 如果没有该方法，尝试从控件管理器的属性中获取选中ID列表
                ids = list(getattr(self.control_manager, 'selected_control_ids', []))
        except Exception:
            # 如果发生任何异常，尝试从属性中获取选中ID列表（降级方案）
            ids = list(getattr(self.control_manager, 'selected_control_ids', []))
        
        # ========== 如果没有选中任何控件，直接返回 ==========
        # 如果没有选中任何控件，高亮已经被清除，直接返回
        if not ids:
            return
        
        # ========== 检查编辑器类型 ==========
        # 检查是否为MonacoEditorWidget（基于QWebEngineView，没有document方法）
        is_monaco = isinstance(code_editor, MonacoEditorWidget)
        
        # ========== 获取代码文本 ==========
        # 从代码编辑器中获取纯文本内容
        text = code_editor.toPlainText()
        # 将代码文本按行分割成列表
        lines = text.splitlines()
        
        # ========== 初始化高亮选择列表 ==========
        # 创建空列表，用于存储需要高亮的文本选择区域
        selections = []
        # 对于MonacoEditorWidget，收集需要高亮的行号
        highlight_lines = []
        
        # ========== 遍历所有选中的控件ID ==========
        # 遍历选中控件ID列表，为每个控件查找并高亮相关代码
        for cid in ids:
            # ========== 将控件ID转换为变量名 ==========
            try:
                # 尝试将控件ID转换为合法的Python变量名（去除特殊字符等）
                var = code_generator._to_valid_variable_name(cid)
            except Exception:
                # 如果转换失败（如方法不存在），直接使用控件ID作为变量名
                var = cid
            
            # ========== 准备搜索字符串 ==========
            # 将变量名转换为小写（用于不区分大小写的搜索）
            var_lower = var.lower()
            # 将控件ID转换为小写字符串（用于搜索）
            cid_lower = str(cid).lower()
            # 构建目标字符串（如"self.button1"），用于搜索控件实例化语句
            target_lower = f"self.{var_lower}"
            
            # ========== 初始化匹配索引列表 ==========
            # 创建空列表，用于存储匹配的代码行索引
            indexes = []
            
            # ========== 遍历所有代码行，查找匹配 ==========
            # 遍历代码行列表，enumerate返回行号和行内容
            for i, line in enumerate(lines):
                # 将当前行转换为小写（用于不区分大小写的匹配）
                ll = line.lower()
                
                # ========== 匹配控件实例化语句 ==========
                # 检查行中是否包含目标变量名（如"self.button1"）和赋值符号"="
                # 这通常表示控件创建语句：self.button1 = QPushButton()
                if target_lower in ll and "=" in ll:
                    # 如果匹配，将行索引添加到列表中
                    indexes.append(i)
                    # 继续下一行（避免重复匹配）
                    continue
                
                # ========== 匹配信号连接语句 ==========
                # 检查行中是否包含目标变量名和".connect("
                # 这通常表示信号连接语句：self.button1.clicked.connect(...)
                if target_lower in ll and ".connect(" in ll:
                    # 如果匹配，将行索引添加到列表中
                    indexes.append(i)
                    # 继续下一行
                    continue
                
                # ========== 匹配方法调用语句 ==========
                # 检查行中是否包含目标变量名、点号和括号（方法调用）
                # 这通常表示属性设置或方法调用：self.button1.setText(...)
                if target_lower in ll and "." in ll and "(" in ll:
                    # 如果匹配，将行索引添加到列表中
                    indexes.append(i)
                    # 继续下一行
                    continue
                
                # ========== 匹配事件处理方法定义 ==========
                # 检查行中是否包含事件处理方法定义（如def _on_button1_clicked）
                if f"def _on_{var_lower}_" in ll:
                    # 如果匹配，将行索引和方法类型标记添加到列表中（元组形式）
                    indexes.append((i, 'method'))
                    # 继续下一行
                    continue
                
                # ========== 匹配控件ID注释或字符串 ==========
                # 检查行中是否包含控件ID（可能作为注释或字符串中的一部分）
                if cid_lower and cid_lower in ll:
                    # 如果匹配，将行索引和ID类型标记添加到列表中（元组形式）
                    indexes.append((i, 'id'))
                    # 继续下一行
                    continue
            # ========== 处理所有匹配的索引，创建高亮选择 ==========
            # 遍历所有匹配的索引（行号或元组）
            for idx in indexes:
                # ========== 处理方法定义类型的高亮 ==========
                # 检查索引是否为元组且类型为'method'（事件处理方法定义）
                if isinstance(idx, tuple) and idx[1] == 'method':
                    # 获取方法定义的起始行号
                    start_i = idx[0]
                    # 初始化结束行号为起始行号（如果方法只有一行，则结束行等于起始行）
                    end_i = start_i
                    # 初始化循环变量j为起始行号的下一条
                    j = start_i + 1
                    # ========== 查找方法结束位置 ==========
                    # 从方法定义的下一条开始，向下查找方法的结束位置
                    while j < len(lines):
                        # 获取当前行的内容
                        s = lines[j]
                        # ========== 检查是否为空行 ==========
                        # 检查行去除空白后是否为空（空行表示方法结束）
                        if not s.strip():
                            # 如果遇到空行，停止查找（方法结束）
                            break
                        # ========== 检查是否为下一个方法定义 ==========
                        # 检查行去除左侧空白后是否以'def '开头（下一个方法定义）
                        if s.lstrip().startswith('def '):
                            # 如果遇到下一个方法定义，停止查找（当前方法结束）
                            break
                        # ========== 更新结束行号 ==========
                        # 如果当前行属于方法体，更新结束行号为当前行
                        end_i = j
                        # 移动到下一行
                        j += 1
                    
                    # ========== 根据编辑器类型处理高亮 ==========
                    if is_monaco:
                        # MonacoEditorWidget: 添加所有行号（从start_i到end_i，包含两端）
                        for line_num in range(start_i, end_i + 1):
                            highlight_lines.append(line_num + 1)  # Monaco行号从1开始
                    else:
                        # QPlainTextEdit: 使用document方法
                        start_block = code_editor.document().findBlockByNumber(start_i)
                        end_block = code_editor.document().findBlockByNumber(end_i)
                        cursor = code_editor.textCursor()
                        cursor.setPosition(start_block.position())
                        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                        k = start_i + 1
                        while k <= end_i:
                            cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
                            cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                            k += 1
                        sel = QTextEdit.ExtraSelection()
                        sel.format.setBackground(QColor(255, 245, 200))
                        sel.cursor = cursor
                        selections.append(sel)
                # ========== 处理ID类型的高亮 ==========
                # 检查索引是否为元组且类型为'id'（控件ID出现在注释或字符串中）
                elif isinstance(idx, tuple) and idx[1] == 'id':
                    # 获取包含控件ID的行号（锚点行）
                    anchor = idx[0]
                    # 初始化函数定义行为锚点行（假设当前行就是函数定义）
                    def_line = anchor
                    # 初始化循环变量k为锚点行
                    k = anchor
                    # ========== 向上查找函数定义 ==========
                    # 从锚点行向上查找，找到包含此行的函数定义
                    while k >= 0:
                        # 检查当前行去除左侧空白并转小写后是否以'def '开头
                        if lines[k].lstrip().lower().startswith('def '):
                            # 如果找到函数定义行，保存行号并退出循环
                            def_line = k
                            break
                        # 向上移动到上一行
                        k -= 1
                    
                    # ========== 检查是否找到了函数定义 ==========
                    # 检查函数定义行是否与锚点行不同（说明找到了包含ID的函数）
                    if def_line != anchor:
                        # ========== 查找函数结束位置 ==========
                        # 如果找到了函数定义，获取函数体范围进行高亮
                        # 设置起始行为函数定义行
                        start_i = def_line
                        # 初始化结束行为起始行
                        end_i = start_i
                        # 初始化循环变量j为函数定义的下一条
                        j = start_i + 1
                        # ========== 向下查找函数结束位置 ==========
                        # 从函数定义的下一条开始，向下查找函数的结束位置
                        while j < len(lines):
                            # 获取当前行的内容
                            s = lines[j]
                            # ========== 检查是否为空行 ==========
                            # 检查行去除空白后是否为空（空行表示函数结束）
                            if not s.strip():
                                # 如果遇到空行，停止查找（函数结束）
                                break
                            # ========== 检查是否为下一个函数定义 ==========
                            # 检查行去除左侧空白后是否以'def '开头（下一个函数定义）
                            if s.lstrip().startswith('def '):
                                # 如果遇到下一个函数定义，停止查找（当前函数结束）
                                break
                            # ========== 更新结束行号 ==========
                            # 如果当前行属于函数体，更新结束行号为当前行
                            end_i = j
                            # 移动到下一行
                            j += 1
                        
                        # ========== 根据编辑器类型处理高亮 ==========
                        if is_monaco:
                            # MonacoEditorWidget: 添加所有行号
                            for line_num in range(start_i, end_i + 1):
                                highlight_lines.append(line_num + 1)  # Monaco行号从1开始
                        else:
                            # QPlainTextEdit: 使用document方法
                            start_block = code_editor.document().findBlockByNumber(start_i)
                            end_block = code_editor.document().findBlockByNumber(end_i)
                            cursor = code_editor.textCursor()
                            cursor.setPosition(start_block.position())
                            cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                            m = start_i + 1
                            while m <= end_i:
                                cursor.movePosition(QTextCursor.NextBlock, QTextCursor.KeepAnchor)
                                cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                                m += 1
                            sel = QTextEdit.ExtraSelection()
                            sel.format.setBackground(QColor(255, 245, 200))
                            sel.cursor = cursor
                            selections.append(sel)
                    else:
                        # ========== 如果未找到函数定义，只高亮锚点行 ==========
                        # 如果未找到函数定义（def_line == anchor），只高亮包含ID的那一行
                        if is_monaco:
                            # MonacoEditorWidget: 添加行号（Monaco行号从1开始）
                            highlight_lines.append(anchor + 1)
                        else:
                            # QPlainTextEdit: 使用document方法
                            block = code_editor.document().findBlockByNumber(anchor)
                            cursor = code_editor.textCursor()
                            cursor.setPosition(block.position())
                            cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                            sel = QTextEdit.ExtraSelection()
                            sel.format.setBackground(QColor(255, 245, 200))
                            sel.cursor = cursor
                            selections.append(sel)
                else:
                    # ========== 处理普通行索引（非方法定义，非ID） ==========
                    # 如果索引不是元组（普通整数索引），直接提取行号
                    # 如果是元组，提取第一个元素（行号）
                    idx_i = idx if isinstance(idx, int) else idx[0]
                    
                    # ========== 根据编辑器类型处理高亮 ==========
                    if is_monaco:
                        # MonacoEditorWidget: 添加行号（Monaco行号从1开始）
                        highlight_lines.append(idx_i + 1)
                    else:
                        # QPlainTextEdit: 使用document方法
                        block = code_editor.document().findBlockByNumber(idx_i)
                        cursor = code_editor.textCursor()
                        cursor.setPosition(block.position())
                        cursor.movePosition(QTextCursor.EndOfBlock, QTextCursor.KeepAnchor)
                        sel = QTextEdit.ExtraSelection()
                        sel.format.setBackground(QColor(255, 245, 200))
                        sel.cursor = cursor
                        selections.append(sel)
        
        # ========== 应用高亮到代码编辑器 ==========
        try:
            if is_monaco:
                # MonacoEditorWidget: 使用highlight_lines方法
                if hasattr(code_editor, 'highlight_lines'):
                    # 去重并排序行号
                    unique_lines = sorted(set(highlight_lines))
                    # highlight_lines方法内部会先清除之前的高亮，然后应用新的
                    code_editor.highlight_lines(unique_lines)
            else:
                # QPlainTextEdit: 使用set_external_highlights方法
                if hasattr(code_editor, 'set_external_highlights'):
                    # set_external_highlights方法会更新_externalHighlights并重新应用高亮
                    code_editor.set_external_highlights(selections)
                else:
                    # 如果没有set_external_highlights方法，使用setExtraSelections
                    if hasattr(code_editor, 'setExtraSelections'):
                        # 直接设置，这会替换所有之前的高亮
                        code_editor.setExtraSelections(selections)
            
            # 如果代码编辑器在标签页中，确保切换到该标签页
            if window_name and hasattr(self, '_code_editors'):
                canvas_tab_name = f"{window_name}_canvas.py"
                if canvas_tab_name in self._code_editors:
                    # 查找并切换到对应的标签页
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == canvas_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            break
        except Exception as e:
            # 如果应用高亮失败，打印错误信息以便调试
            print(f"应用代码高亮失败: {e}")
            import traceback
            traceback.print_exc()

    def _on_control_added(self, control_id: str):
        """
        控件添加事件处理函数
        当在设计画布上添加控件时调用此方法
        
        Args:
            control_id: 被添加的控件ID
        """
        # ========== 确保使用当前窗口类的控件管理器 ==========
        # 获取当前窗口类，确保控件添加到正确的窗口
        current_window = None
        if hasattr(self.project_manager, 'window_class_manager'):
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if current_name:
                current_window = self.project_manager.window_class_manager.get_window_class(current_name)
                if current_window:
                    # 确保使用当前窗口类的控件管理器
                    self.control_manager = current_window.control_manager
        
        # ========== 获取添加的控件对象 ==========
        # 从控件管理器中获取新添加的控件对象
        control = self.control_manager.get_control(control_id)
        
        # ========== 检查控件是否存在 ==========
        # 检查控件对象是否存在（不为None）
        if control:
            # 如果控件存在，执行以下操作
            # ========== 标记项目为已修改 ==========
            # 设置项目状态为已修改（需要在保存时提示用户）
            self.project_manager.set_modified(True)
            # ========== 更新状态栏 ==========
            # 更新状态栏标签，显示已添加的控件ID
            self.status_label.setText(f"已添加控件: {control.id}")
            # ========== 保存撤销状态 ==========
            # 在撤销管理器中保存当前状态，以便后续可以撤销此操作
            # 操作描述：添加控件类型名称
            self.undo_redo_manager.save_state(f"添加 {control.control_type} 控件")
            # ========== 刷新属性编辑器控件列表 ==========
            # 刷新属性编辑器中的控件列表，使新添加的控件出现在列表中
            self.property_editor.refresh_control_list(self.control_manager)
            # ========== 刷新代码视图 ==========
            # 刷新画布代码和功能代码
            self._refresh_code_view_left()
            # ========== 更新代码高亮 ==========
            # 选中新添加的控件并更新代码高亮
            self.control_manager.select_control(control_id)
            QTimer.singleShot(100, self._update_code_highlight_from_selection)
            # ========== 更新功能代码文件 ==========
            # 更新当前窗口类的功能代码文件
            self._update_event_function_tab()
            # ========== 刷新项目预览 ==========
            # 刷新项目预览树
            self._refresh_project_preview()

    def _on_control_deleted(self, control_id: str):
        """
        控件删除事件处理函数
        当在设计画布上删除控件时调用此方法
        
        Args:
            control_id: 被删除的控件ID
        """
        # ========== 标记项目为已修改 ==========
        # 设置项目状态为已修改（需要在保存时提示用户）
        self.project_manager.set_modified(True)
        # ========== 更新状态栏 ==========
        # 更新状态栏标签，显示已删除的控件ID
        self.status_label.setText(f"已删除控件: {control_id}")
        # ========== 保存撤销状态 ==========
        # 在撤销管理器中保存当前状态，以便后续可以撤销此操作
        # 操作描述：删除控件
        self.undo_redo_manager.save_state("删除控件")
        # ========== 刷新属性编辑器控件列表 ==========
        # 刷新属性编辑器中的控件列表，移除已删除的控件
        self.property_editor.refresh_control_list(self.control_manager)
        # 刷新项目预览
        self._refresh_project_preview()

    def _on_control_moved(self, control_id: str, dx: int, dy: int):
        """
        控件移动事件处理函数
        当在设计画布上移动控件时调用此方法
        
        Args:
            control_id: 被移动的控件ID
            dx: X方向的移动距离（像素）
            dy: Y方向的移动距离（像素）
        """
        # ========== 获取移动的控件对象 ==========
        # 从控件管理器中获取被移动的控件对象
        control = self.control_manager.get_control(control_id)
        
        # ========== 检查控件是否存在 ==========
        # 检查控件对象是否存在（不为None）
        if control:
            # 如果控件存在，执行以下操作
            # ========== 更新坐标标签 ==========
            # 更新状态栏中的坐标标签，显示控件的新位置
            self.coord_label.setText(f"X: {control.x}, Y: {control.y}")
            # ========== 标记项目为已修改 ==========
            # 设置项目状态为已修改（需要在保存时提示用户）
            self.project_manager.set_modified(True)
            # ========== 保存撤销状态 ==========
            # 在撤销管理器中保存当前状态，以便后续可以撤销此移动操作
            # 操作描述：移动控件和控件ID
            self.undo_redo_manager.save_state(f"移动控件 {control_id}")

    def _on_control_resized(self, control_id: str, handle: str, dx: int, dy: int):
        """
        控件大小调整事件处理函数
        当在设计画布上调整控件大小时调用此方法
        
        Args:
            control_id: 被调整大小的控件ID
            handle: 调整大小的句柄位置（如"top_left"、"bottom_right"等）
            dx: X方向的调整距离（像素，正数表示增加宽度，负数表示减少宽度）
            dy: Y方向的调整距离（像素，正数表示增加高度，负数表示减少高度）
        """
        # ========== 获取调整大小的控件对象 ==========
        # 从控件管理器中获取被调整大小的控件对象
        control = self.control_manager.get_control(control_id)
        
        # ========== 检查控件是否存在 ==========
        # 检查控件对象是否存在（不为None）
        if control:
            # 如果控件存在，执行以下操作
            # ========== 更新画布大小标签 ==========
            # 更新状态栏中的画布大小标签，显示控件的新尺寸
            self.canvas_size_label.setText(f"控件: {control.width}x{control.height}")
            # ========== 标记项目为已修改 ==========
            # 设置项目状态为已修改（需要在保存时提示用户）
            self.project_manager.set_modified(True)
            # ========== 保存撤销状态 ==========
            # 在撤销管理器中保存当前状态，以便后续可以撤销此大小调整操作
            # 操作描述：调整控件大小和控件ID
            self.undo_redo_manager.save_state(f"调整控件 {control_id} 大小")

    def _on_property_changed(self, control_id: str, property_name: str, value):
        """
        属性变化事件处理函数
        当控件的属性或画布的属性被修改时调用此方法
        
        Args:
            control_id: 控件ID，如果是"canvas"表示修改的是画布属性
            property_name: 属性名称（如"text"、"x"、"width"等）
            value: 新的属性值
        """
        # ========== 处理画布属性变化 ==========
        # 检查是否修改的是画布属性（而不是控件属性）
        if control_id == "canvas":
            # ========== 定义画布属性处理函数字典 ==========
            # 创建字典，将属性名映射到对应的处理函数（lambda函数）
            # 每个处理函数负责设置对应的画布属性
            canvas_property_handlers = {
                "canvas_width": lambda: (self.design_canvas.set_canvas_width(value), self.project_manager.set_canvas_size(value, self.design_canvas.get_canvas_height())),
                "canvas_height": lambda: (self.design_canvas.set_canvas_height(value), self.project_manager.set_canvas_size(self.design_canvas.get_canvas_width(), value)),
                "canvas_bg_color": lambda: self.design_canvas.set_canvas_bg_color(value),
                "show_grid": lambda: self.design_canvas.set_show_grid(value),
                "grid_size": lambda: self.design_canvas.set_grid_size(value),
                "snap_to_grid": lambda: self.design_canvas.set_snap_to_grid(value),
                "window_name": lambda: self.design_canvas.set_window_property("window_name", value),
                "window_comment": lambda: self.design_canvas.set_window_property("window_comment", value),
                "window_left": lambda: self.design_canvas.set_window_property("window_left", value),
                "window_top": lambda: self.design_canvas.set_window_property("window_top", value),
                "window_width": lambda: self.design_canvas.set_window_property("window_width", value),
                "window_height": lambda: self.design_canvas.set_window_property("window_height", value),
                "window_visible": lambda: self.design_canvas.set_window_property("window_visible", value),
                "window_enabled": lambda: self.design_canvas.set_window_property("window_enabled", value),
                "window_cursor": lambda: self.design_canvas.set_window_property("window_cursor", value),
                "window_title": lambda: self.design_canvas.set_window_property("window_title", value),
                "window_border": lambda: self.design_canvas.set_window_property("window_border", value),
                "window_bg_color": lambda: self.design_canvas.set_window_property("window_bg_color", value),
                "window_bg_image": lambda: self.design_canvas.set_window_property("window_bg_image", value),
                "window_bg_image_mode": lambda: self.design_canvas.set_window_property("window_bg_image_mode", value),
                "window_bg_music": lambda: self.design_canvas.set_window_property("window_bg_music", value),
                "window_music_play_count": lambda: self.design_canvas.set_window_property("window_music_play_count", value),
                "window_show_control_button": lambda: self.design_canvas.set_window_property("window_show_control_button", value),
                "window_show_maximize_button": lambda: self.design_canvas.set_window_property("window_show_maximize_button", value),
                "window_show_minimize_button": lambda: self.design_canvas.set_window_property("window_show_minimize_button", value),
                "window_position": lambda: self.design_canvas.set_window_property("window_position", value),
                "window_movable": lambda: self.design_canvas.set_window_property("window_movable", value),
                "window_icon": lambda: self.design_canvas.set_window_property("window_icon", value),
                "window_enter_move_focus": lambda: self.design_canvas.set_window_property("window_enter_move_focus", value),
                "window_esc_close": lambda: self.design_canvas.set_window_property("window_esc_close", value),
                "window_f1_help": lambda: self.design_canvas.set_window_property("window_f1_help", value),
                "window_help_file": lambda: self.design_canvas.set_window_property("window_help_file", value),
                "window_help_flag": lambda: self.design_canvas.set_window_property("window_help_flag", value),
                "window_show_in_taskbar": lambda: self.design_canvas.set_window_property("window_show_in_taskbar", value),
                "window_free_move": lambda: self.design_canvas.set_window_property("window_free_move", value),
                "window_shape": lambda: self.design_canvas.set_window_property("window_shape", value),
                "window_always_on_top": lambda: self.design_canvas.set_window_property("window_always_on_top", value),
                "window_keep_title_active": lambda: self.design_canvas.set_window_property("window_keep_title_active", value),
                "window_class_name": lambda: self.design_canvas.set_window_property("window_class_name", value),
            }
            
            # ========== 检查属性是否有对应的处理函数 ==========
            # 检查修改的属性名是否在画布属性处理函数字典中
            if property_name in canvas_property_handlers:
                # 如果属性有对应的处理函数，调用它来设置属性值
                canvas_property_handlers[property_name]()
            
            # ========== 更新画布显示 ==========
            # 更新画布显示，使属性更改生效
            self.design_canvas.update()
            # ========== 更新状态栏 ==========
            # 更新状态栏标签，显示已更新的画布属性名称
            self.status_label.setText(f"已更新画布属性: {property_name}")
            # ========== 标记项目为已修改 ==========
            # 设置项目状态为已修改（需要在保存时提示用户）
            self.project_manager.set_modified(True)
            # ========== 保存撤销状态 ==========
            # 在撤销管理器中保存当前状态，以便后续可以撤销此属性修改
            self.undo_redo_manager.save_state("修改画布属性")
        else:
            # ========== 处理控件属性变化 ==========
            # 如果修改的是控件属性（而不是画布属性），执行以下操作
            # 从控件管理器中获取被修改属性的控件对象
            control = self.control_manager.get_control(control_id)
            
            # ========== 检查控件是否存在 ==========
            # 检查控件对象是否存在（不为None）
            if control:
                # ========== 处理位置和大小属性 ==========
                # 检查属性名是否为位置或大小属性（这些属性需要特殊处理）
                if property_name in ["x", "y", "width", "height"]:
                    # 根据属性名设置对应的属性值
                    if property_name == "x":
                        # 如果属性名为"x"，设置控件的X坐标
                        control.x = value
                    elif property_name == "y":
                        # 如果属性名为"y"，设置控件的Y坐标
                        control.y = value
                    elif property_name == "width":
                        # 如果属性名为"width"，设置控件的宽度
                        control.width = value
                    elif property_name == "height":
                        # 如果属性名为"height"，设置控件的高度
                        control.height = value
                
                # ========== 处理层级顺序属性 ==========
                # 检查属性名是否为"z_order"（控件的层级顺序）
                elif property_name == "z_order":
                    # 如果属性名为"z_order"，设置控件的层级顺序（用于控制控件的显示顺序）
                    control.z_order = value
                
                else:
                    # ========== 处理其他属性 ==========
                    # 如果属性不是位置、大小或层级顺序，将其保存到控件的属性字典中
                    control.properties[property_name] = value
                    
                    # ========== 特殊处理name属性 ==========
                    # 检查属性名是否为"name"（控件名称）
                    if property_name == "name":
                        # 如果属性名为"name"，同时设置控件的name属性（可能是控件显示名称）
                        control.name = value
                        # 刷新属性编辑器中的控件列表（控件名称变化需要更新列表显示）
                        self.property_editor.refresh_control_list(self.control_manager)
                
                # ========== 更新画布显示 ==========
                # 更新画布显示，使控件属性更改生效
                self.design_canvas.update()
                # ========== 更新状态栏 ==========
                # 更新状态栏标签，显示已更新的属性名称
                self.status_label.setText(f"已更新属性: {property_name}")
                self.project_manager.set_modified(True)
                self.undo_redo_manager.save_state(f"修改控件 {control_id} 属性")

    def _new_project(self):
        """
        创建新项目
        如果当前项目有未保存的修改，先提示用户是否保存
        """
        # ========== 检查项目是否已修改 ==========
        # 检查当前项目是否有未保存的修改
        if self.project_manager.is_modified():
            # 如果项目已修改，显示确认对话框，询问用户是否保存
            reply = QMessageBox.question(
                self,  # 父窗口
                "确认",  # 对话框标题
                "当前项目有未保存的修改，是否保存？",  # 提示消息
                QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel  # 按钮选项（是、否、取消）
            )
            
            # ========== 处理用户选择 ==========
            # 检查用户在对话框中的选择
            if reply == QMessageBox.Yes:
                # 如果用户选择"是"，保存当前项目
                self._save_project()
            elif reply == QMessageBox.Cancel:
                # 如果用户选择"取消"，直接返回，不创建新项目
                return
        self.design_canvas.clear_controls()
        self.project_manager.new_project()
        import os, sys
        run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
        self.project_dir = os.path.join(run_dir, 'project')
        try:
            os.makedirs(self.project_dir, exist_ok=True)
        except Exception:
            pass
        try:
            from PySide6.QtWidgets import QInputDialog
            framework, ok = QInputDialog.getItem(self, "新建项目", "选择框架", ["PySide", "TKinter"], 0, False)
        except Exception:
            framework, ok = ("PySide", True)
        if not ok:
            return
        self.fixed_framework = framework
        try:
            if hasattr(self, 'framework_combo_left'):
                self.framework_combo_left.setCurrentText(framework)
                self.framework_combo_left.setEnabled(False)
        except Exception:
            pass
        try:
            if hasattr(self, 'property_editor') and hasattr(self.property_editor, 'set_fixed_framework'):
                self.property_editor.set_fixed_framework(framework)
        except Exception:
            pass
        self.setWindowTitle("可视化编程工具 - 无标题")
        self.statusBar().showMessage(f"已创建新项目: {self.project_dir}，框架: {framework}")
        
        # 设置窗口名称
        self.project_manager.set_window_name("GeneratedWindow")
        
        # 确保窗口类管理器存在并切换到当前窗口类
        if hasattr(self.project_manager, 'window_class_manager'):
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if current_name:
                # 切换到当前窗口类（确保所有区域正确更新）
                self._switch_to_window_class(current_name)
            else:
                # 如果没有当前窗口类，使用第一个
                all_names = self.project_manager.window_class_manager.get_all_names()
                if all_names:
                    self._switch_to_window_class(all_names[0])
        
        # 刷新项目预览
        self._refresh_project_preview()
        
        # 清空撤销重做栈并保存初始状态
        if hasattr(self, 'undo_redo_manager') and self.undo_redo_manager:
            self.undo_redo_manager.clear()
            try:
                self.undo_redo_manager.save_state("初始状态")
            except Exception:
                pass

    def _open_project(self):
        if self.project_manager.is_modified():
            reply = QMessageBox.question(
                self, "保存项目",
                "当前项目已修改，是否保存？",
                QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel
            )
            if reply == QMessageBox.Yes:
                if not self._save_project_impl():
                    return
            elif reply == QMessageBox.Cancel:
                return
        file_path, _ = QFileDialog.getOpenFileName(
            self, "打开项目", "", "项目文件 (*.vpp);;所有文件 (*)"
        )
        if file_path:
            if self.project_manager.load_project(file_path):
                # 关闭预览窗口（如果存在）
                if hasattr(self, 'preview_window') and self.preview_window:
                    try:
                        if self.preview_window.isVisible():
                            self.preview_window.close()
                        self.preview_window = None
                    except Exception:
                        self.preview_window = None
                
                # 如果有多窗口类管理器，切换到当前窗口类
                if hasattr(self.project_manager, 'window_class_manager'):
                    current_name = self.project_manager.window_class_manager.current_window_class_name
                    if current_name:
                        # 切换到当前窗口类（会自动更新所有区域）
                        if not self._switch_to_window_class(current_name):
                            # 如果切换失败，尝试使用第一个窗口类
                            all_names = self.project_manager.window_class_manager.get_all_names()
                            if all_names and all_names[0] != current_name:
                                self._switch_to_window_class(all_names[0])
                    else:
                        # 如果没有当前窗口类，使用第一个
                        all_names = self.project_manager.window_class_manager.get_all_names()
                        if all_names:
                            self._switch_to_window_class(all_names[0])
                else:
                    # 向后兼容：使用旧的加载方式
                    self.design_canvas.update()
                    canvas_width, canvas_height = self.project_manager.get_canvas_size()
                    self.design_canvas.set_canvas_size(canvas_width, canvas_height)
                    self.property_editor.set_control(None)
                    
                    # 加载事件函数文件（向后兼容）
                    event_function_content = self.project_manager.get_event_function_file()
                    if event_function_content:
                        self._update_event_function_tab()
                
                self.status_label.setText(f"已打开项目: {self.project_manager.get_project_name()}")
                
                # 刷新项目预览
                self._refresh_project_preview()
                
                self.undo_redo_manager.clear()
                try:
                    self.undo_redo_manager.save_state("初始状态")
                except Exception:
                    pass
            else:
                QMessageBox.warning(self, "错误", "无法打开项目文件")

    def _save_project(self):
        if self._save_project_impl():
            self.status_label.setText(f"项目已保存: {self.project_manager.get_project_name()}")

    def _save_as_project(self):
        file_path, _ = QFileDialog.getSaveFileName(
            self, "另存为项目", f"{self.project_manager.get_project_name()}.vpp",
            "项目文件 (*.vpp);;所有文件 (*)"
        )
        if file_path:
            if self.project_manager.save_project(file_path):
                self.status_label.setText(f"项目已保存为: {self.project_manager.get_project_name()}")
            else:
                QMessageBox.warning(self, "错误", "无法保存项目文件")

    def _save_project_impl(self) -> bool:
        # 在保存前，确保所有窗口类的状态都已保存
        try:
            # 保存当前窗口类的状态（包括功能函数文件）
            self._save_current_window_class_state()
            
            # 保存所有窗口类的功能函数文件（从代码编辑器同步到窗口类）
            if hasattr(self.project_manager, 'window_class_manager'):
                for window_name in self.project_manager.window_class_manager.get_all_names():
                    window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                    if window_class:
                        function_file_name = f"{window_name}_function.py"
                        # 优先从代码编辑器获取（实时内容）
                        if hasattr(self, '_code_editors') and function_file_name in self._code_editors:
                            editor = self._code_editors[function_file_name]
                            content = editor.toPlainText()
                            window_class.event_function_file = content
                        # 如果代码编辑器中没有，确保窗口类中有内容（至少是空模板）
                        elif not getattr(window_class, 'event_function_file', None):
                            # 如果窗口类中也没有，生成空模板
                            empty_function_file = "\n".join([
                                f"# -*- coding: utf-8 -*-",
                                f"\"\"\"",
                                f"事件处理函数模块",
                                f"窗口: {window_name}",
                                f"此文件包含所有控件的事件处理函数实现",
                                f"\"\"\"",
                                "",
                                "# 导入必要的模块",
                                "from PySide6.QtCore import Qt",
                                "from PySide6.QtGui import QMouseEvent, QKeyEvent, QWheelEvent",
                                "",
                            ])
                            window_class.event_function_file = empty_function_file
        except Exception as e:
            print(f"保存窗口类状态失败: {e}")
        
        if self.project_manager.get_project_file() is None:
            file_path, _ = QFileDialog.getSaveFileName(
                self, "保存项目", f"{self.project_manager.get_project_name()}.vpp",
                "项目文件 (*.vpp);;所有文件 (*)"
            )
            if not file_path:
                return False
            result = self.project_manager.save_project(file_path)
            if result:
                self.preview_window = None
            return result
        else:
            result = self.project_manager.save_project()
            if result:
                self.preview_window = None
            return result

    def _undo(self):
        """执行撤销操作"""
        # 先尝试撤销窗口操作
        if self._undo_window_operation():
            self.status_label.setText("上一步: 窗口操作")
            self.project_manager.set_modified(True)
            return
        
        # 再尝试撤销常规操作
        result = self.undo_redo_manager.undo()
        if isinstance(result, tuple):
            if len(result) >= 3:
                success, event_function_content, canvas_code_content = result
            elif len(result) >= 2:
                success, event_function_content = result
                canvas_code_content = None
            else:
                success = result[0] if result else False
                event_function_content = None
                canvas_code_content = None
        else:
            success = result
            event_function_content = None
            canvas_code_content = None
        
        if success:
            self._reload_canvas()
            description = self.undo_redo_manager.get_undo_description()
            
            # 恢复功能代码文件内容
            if event_function_content is not None:
                self._restore_event_function_content(event_function_content)
            
            # 恢复画布代码内容
            if canvas_code_content is not None:
                self._restore_canvas_code_content(canvas_code_content)
            
            if description:
                self.status_label.setText(f"上一步: {description}")
            else:
                self.status_label.setText("上一步")
            self.project_manager.set_modified(True)
        else:
            self.status_label.setText("无法执行上一步")

    def _redo(self):
        """执行重做操作"""
        # 先尝试重做窗口操作
        if self._redo_window_operation():
            self.status_label.setText("重做: 窗口操作")
            self.project_manager.set_modified(True)
            return
        
        # 再尝试重做常规操作
        result = self.undo_redo_manager.redo()
        if isinstance(result, tuple):
            if len(result) >= 3:
                success, event_function_content, canvas_code_content = result
            elif len(result) >= 2:
                success, event_function_content = result
                canvas_code_content = None
            else:
                success = result[0] if result else False
                event_function_content = None
                canvas_code_content = None
        else:
            success = result
            event_function_content = None
            canvas_code_content = None
        
        if success:
            self._reload_canvas()
            description = self.undo_redo_manager.get_redo_description()
            
            # 恢复功能代码文件内容
            if event_function_content is not None:
                self._restore_event_function_content(event_function_content)
            
            # 恢复画布代码内容
            if canvas_code_content is not None:
                self._restore_canvas_code_content(canvas_code_content)
            
            if description:
                self.status_label.setText(f"重做: {description}")
            else:
                self.status_label.setText("重做")
            self.project_manager.set_modified(True)
        else:
            self.status_label.setText("无法重做")
    
    def _restore_event_function_content(self, content: str):
        """恢复功能代码文件内容"""
        try:
            if not hasattr(self.project_manager, 'window_class_manager'):
                return
            
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if not current_name:
                return
            
            # 更新窗口类的功能代码文件内容
            current_window = self.project_manager.window_class_manager.get_window_class(current_name)
            if current_window:
                current_window.event_function_file = content
            
            # 更新代码编辑器
            tab_name = f"{current_name}_function.py"
            if hasattr(self, '_code_editors') and tab_name in self._code_editors:
                editor = self._code_editors[tab_name]
                # 先断开连接，避免触发保存
                try:
                    editor.textChanged.disconnect()
                except Exception:
                    pass
                editor.setPlainText(content)
                # 清除撤销状态标记（因为内容被恢复了）
                if hasattr(editor, '_undo_state_saved'):
                    delattr(editor, '_undo_state_saved')
                # 重新连接
                editor.textChanged.connect(lambda: self._on_event_function_changed(tab_name, editor, current_name))
        except Exception as e:
            print(f"恢复功能代码文件内容失败: {e}")
    
    def _restore_canvas_code_content(self, content: str):
        """恢复画布代码内容"""
        try:
            # 获取当前窗口类名称
            current_name = None
            if hasattr(self.project_manager, 'window_class_manager'):
                current_name = self.project_manager.window_class_manager.current_window_class_name
            if not current_name:
                current_name = self.project_manager.get_window_name() if hasattr(self.project_manager, 'get_window_name') else "GeneratedWindow"
            
            # 更新窗口类中的画布代码内容
            if hasattr(self.project_manager, 'window_class_manager'):
                current_window = self.project_manager.window_class_manager.get_window_class(current_name)
                if current_window:
                    current_window.canvas_code = content
            
            # 更新代码编辑器中的内容
            tab_name = f"{current_name}_canvas.py"
            if hasattr(self, '_code_editors') and tab_name in self._code_editors:
                editor = self._code_editors[tab_name]
                # 先断开连接，避免触发保存
                try:
                    editor.textChanged.disconnect()
                except Exception:
                    pass
                editor.setPlainText(content)
                # 清除撤销状态标记（因为内容被恢复了）
                if hasattr(editor, '_undo_state_saved'):
                    delattr(editor, '_undo_state_saved')
                # 重新连接
                editor.textChanged.connect(lambda: self._on_canvas_code_changed(tab_name, editor, current_name))
        except Exception as e:
            print(f"恢复画布代码内容失败: {e}")
    
    def _save_window_operation_state(self, description: str, operation_data: dict):
        """保存窗口操作状态到撤销栈"""
        try:
            state = {
                "description": description,
                "data": operation_data.copy(),
                "window_class_manager_state": self.project_manager.window_class_manager.to_dict()
            }
            self.window_operation_undo_stack.append(state)
            # 限制栈大小
            if len(self.window_operation_undo_stack) > 50:
                self.window_operation_undo_stack.pop(0)
            # 清空重做栈
            self.window_operation_redo_stack.clear()
        except Exception as e:
            print(f"保存窗口操作状态失败: {e}")
    
    def _undo_window_operation(self) -> bool:
        """撤销窗口操作"""
        try:
            if len(self.window_operation_undo_stack) < 2:
                return False
            
            # 将当前状态移到重做栈
            current_state = self.window_operation_undo_stack.pop()
            self.window_operation_redo_stack.append(current_state)
            
            # 恢复到前一状态
            prev_state = self.window_operation_undo_stack[-1]
            self._restore_window_class_manager_state(prev_state["window_class_manager_state"])
            
            return True
        except Exception as e:
            print(f"撤销窗口操作失败: {e}")
            return False
    
    def _redo_window_operation(self) -> bool:
        """重做窗口操作"""
        try:
            if len(self.window_operation_redo_stack) == 0:
                return False
            
            # 从重做栈恢复
            redo_state = self.window_operation_redo_stack.pop()
            # 保存当前状态到撤销栈
            current_state = {
                "description": "",
                "data": {},
                "window_class_manager_state": self.project_manager.window_class_manager.to_dict()
            }
            self.window_operation_undo_stack.append(current_state)
            
            # 恢复重做状态
            self._restore_window_class_manager_state(redo_state["window_class_manager_state"])
            
            return True
        except Exception as e:
            print(f"重做窗口操作失败: {e}")
            return False
    
    def _restore_window_class_manager_state(self, state_dict: dict):
        """恢复窗口类管理器状态"""
        try:
            from module.window_class_manager import WindowClassManager
            self.project_manager.window_class_manager = WindowClassManager.from_dict(state_dict)
            
            # 切换到当前窗口类
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if current_name:
                self._switch_to_window_class(current_name)
        except Exception as e:
            print(f"恢复窗口类管理器状态失败: {e}")

    def _reload_canvas(self):
        self.design_canvas.update()
        self.property_editor.refresh_control_list(self.control_manager)

    def _on_canvas_resized(self, w: int, h: int):
        try:
            self.design_canvas.setFixedSize(max(1, w), max(1, h))
        except Exception:
            pass
        self.project_manager.set_canvas_size(w, h)
        
        # 更新当前窗口类的画布大小
        if hasattr(self.project_manager, 'window_class_manager'):
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if current_name:
                current_window = self.project_manager.window_class_manager.get_window_class(current_name)
                if current_window:
                    current_window.canvas_width = w
                    current_window.canvas_height = h
                    current_window.code_generator.set_window_size(w, h)
        
        try:
            self.canvas_size_label.setText(f"画布: {w}x{h}")
        except Exception:
            pass
        self.undo_redo_manager.save_state("调整画布大小")

    def _delete_selected(self):
        self.design_canvas.delete_selected_control()
    
    def _save_current_window_class_state(self):
        """保存当前窗口类的状态到窗口类对象中"""
        try:
            if not hasattr(self.project_manager, 'window_class_manager'):
                return
            
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if not current_name:
                return
            
            current_window = self.project_manager.window_class_manager.get_window_class(current_name)
            if not current_window:
                return
            
            # 保存画布大小
            current_window.canvas_width = self.design_canvas.canvas_width
            current_window.canvas_height = self.design_canvas.canvas_height
            
            # 保存窗口标题
            current_window.window_title = getattr(self.design_canvas, 'window_title', '我的窗口')
            
            # 保存窗口图标
            current_window.window_icon_path = getattr(self.design_canvas, 'window_icon', '')
            
            # 保存菜单和页面
            current_window.pages = getattr(self.design_canvas, 'window_menus', [])
            current_window.dropdown_menus = getattr(self.design_canvas, 'window_dropdown_menus', [])
            
            # 保存画布代码内容（从代码编辑器，确保使用最新的内容）
            canvas_tab_name = f"{current_name}_canvas.py"
            if hasattr(self, '_code_editors') and canvas_tab_name in self._code_editors:
                editor = self._code_editors[canvas_tab_name]
                # 确保获取最新的代码内容
                canvas_code = editor.toPlainText()
                current_window.canvas_code = canvas_code
            else:
                # 尝试从当前活动的代码编辑器获取代码
                current_editor = self._get_current_code_editor()
                if current_editor:
                    current_window.canvas_code = current_editor.toPlainText()
            
            # 保存功能函数文件内容（从代码编辑器，确保使用最新的内容）
            function_tab_name = f"{current_name}_function.py"
            if hasattr(self, '_code_editors') and function_tab_name in self._code_editors:
                editor = self._code_editors[function_tab_name]
                # 确保获取最新的代码内容
                function_code = editor.toPlainText()
                current_window.event_function_file = function_code
            
            # 更新代码生成器的属性
            current_window.code_generator.set_window_title(current_window.window_title)
            current_window.code_generator.set_window_size(current_window.canvas_width, current_window.canvas_height)
            current_window.code_generator.set_window_icon(current_window.window_icon_path)
            current_window.code_generator.set_pages(current_window.pages)
            current_window.code_generator.set_dropdown_menus(current_window.dropdown_menus)
        except Exception as e:
            print(f"保存窗口类状态失败: {e}")
    
    def _switch_to_window_class(self, window_name: str):
        """
        切换到指定的窗口类
        更新所有相关区域：设计画布、代码编辑器、属性编辑器、撤销重做管理器等
        
        Args:
            window_name: 窗口类名称
        """
        try:
            # 检查窗口类管理器是否存在
            if not hasattr(self.project_manager, 'window_class_manager'):
                print("警告: 窗口类管理器不存在")
                return False
            
            # 检查窗口类是否存在
            if not self.project_manager.window_class_manager.get_window_class(window_name):
                print(f"警告: 窗口类 '{window_name}' 不存在")
                return False
            
            # 如果切换的是当前窗口类，直接返回成功（避免重复切换）
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if current_name == window_name:
                return True
            
            # 保存当前窗口类的状态（如果存在）
            if current_name:
                self._save_current_window_class_state()
            
            # 关闭预览窗口（如果存在），避免使用旧的控件管理器
            if hasattr(self, 'preview_window') and self.preview_window:
                try:
                    if self.preview_window.isVisible():
                        self.preview_window.close()
                    self.preview_window = None
                except Exception:
                    self.preview_window = None
            
            window_class = self.project_manager.window_class_manager.get_window_class(window_name)
            if not window_class:
                return False
            
            # 设置当前窗口类
            self.project_manager.window_class_manager.set_current_window_class(window_name)
            
            # 切换到新窗口类的控件管理器
            self.control_manager = window_class.control_manager
            
            # 更新设计画布
            self.design_canvas.set_control_manager(window_class.control_manager)
            
            # 更新画布大小
            self.design_canvas.set_canvas_size(window_class.canvas_width, window_class.canvas_height)
            
            # 更新窗口属性
            self.design_canvas.set_window_property("window_title", window_class.window_title)
            if window_class.window_icon_path:
                self.design_canvas.set_window_property("window_icon", window_class.window_icon_path)
            
            # 更新菜单和页面
            if hasattr(self.design_canvas, 'window_menus'):
                self.design_canvas.window_menus = window_class.pages or []
            if hasattr(self.design_canvas, 'window_dropdown_menus'):
                self.design_canvas.window_dropdown_menus = window_class.dropdown_menus or []
            
            # 更新代码生成器
            self.code_generator = window_class.code_generator
            
            # 更新撤销重做管理器
            from module.undo_redo_manager_pyside import UndoRedoManager
            self.undo_redo_manager = UndoRedoManager(window_class.control_manager)
            self.undo_redo_manager.set_design_canvas(self.design_canvas)
            # 设置功能代码文件内容获取函数
            self.undo_redo_manager.set_event_function_file_content(
                lambda: self._get_current_event_function_content()
            )
            self.undo_redo_manager.set_window_class_name(window_name)
            
            # 重新连接信号
            self._connect_signals()
            
            # 刷新属性编辑器
            self.property_editor.refresh_control_list(window_class.control_manager)
            self.property_editor.set_control(None)  # 清空选中状态
            
            # 刷新设计画布显示
            self.design_canvas.update()
            
            # 更新画布代码标签页（必须在刷新代码视图之前）
            self._update_canvas_code_tab(window_name, window_class)
            
            # 刷新代码编辑区
            if hasattr(self, '_refresh_code_view_left'):
                self._refresh_code_view_left()
            
            # 刷新功能函数文件标签页
            self._update_event_function_tab()
            
            # 刷新项目预览树形框（延迟刷新，避免在切换过程中刷新）
            # 使用QTimer延迟刷新，确保所有更新完成后再刷新
            from PySide6.QtCore import QTimer
            QTimer.singleShot(100, self._refresh_project_preview)
            
            # 保存初始撤销状态
            try:
                if hasattr(self, 'undo_redo_manager') and self.undo_redo_manager:
                    self.undo_redo_manager.save_state("切换窗口类")
            except Exception:
                pass
            
            return True
        except Exception as e:
            print(f"切换窗口类失败: {e}")
            import traceback
            traceback.print_exc()
            # 即使失败也刷新项目预览，确保UI状态正确
            try:
                self._refresh_project_preview()
            except Exception:
                pass
            return False
    
    def _new_window_class(self):
        """新建窗口类"""
        try:
            from PySide6.QtWidgets import QInputDialog, QMessageBox
            
            # 检查是否有窗口类管理器
            if not hasattr(self.project_manager, 'window_class_manager'):
                QMessageBox.warning(self, "错误", "窗口类管理器不可用")
                return
            
            # 获取新窗口类名称
            window_name, ok = QInputDialog.getText(
                self, 
                "新建窗口类", 
                "请输入窗口类名称:",
                text="new_window_class"
            )
            
            if not ok or not window_name.strip():
                return
            
            window_name = window_name.strip()
            
            # 检查名称是否已存在
            if window_name in self.project_manager.window_class_manager.get_all_names():
                # 如果重名，自动重命名
                counter = 1
                base_name = window_name
                while window_name in self.project_manager.window_class_manager.get_all_names():
                    window_name = f"{base_name}_{counter}"
                    counter += 1
                QMessageBox.information(
                    self, 
                    "提示", 
                    f"窗口类名称已存在，已自动重命名为: {window_name}"
                )
            
            # 保存窗口操作撤销状态
            self._save_window_operation_state("创建窗口类", {
                "operation": "create",
                "window_name": window_name,
                "before_state": self.project_manager.window_class_manager.to_dict()
            })
            
            # 创建新窗口类
            new_window = self.project_manager.window_class_manager.add_window_class(window_name)
            
            if new_window:
                # 清空新窗口类的控件（新窗口应该是空的）
                new_window.control_manager.clear_all()
                
                # 初始化新窗口类的代码生成器属性
                new_window.code_generator.set_window_class_name(window_name)
                new_window.code_generator.set_window_name(window_name)
                new_window.code_generator.set_window_title(new_window.window_title)
                new_window.code_generator.set_window_size(new_window.canvas_width, new_window.canvas_height)
                # 确保代码生成器使用新窗口类的控件管理器
                new_window.code_generator.control_manager = new_window.control_manager
                
                # 生成初始画布代码（空窗口的代码）
                fw = getattr(self, 'fixed_framework', None)
                framework = fw if fw else (self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide")
                initial_code = new_window.code_generator.generate_code(
                    new_window.canvas_width, 
                    new_window.canvas_height, 
                    framework=framework
                )
                new_window.canvas_code = initial_code
                
                # 创建空的功能函数文件内容
                empty_function_file = "\n".join([
                    f"# -*- coding: utf-8 -*-",
                    f"\"\"\"",
                    f"事件处理函数模块",
                    f"窗口: {window_name}",
                    f"此文件包含所有控件的事件处理函数实现",
                    f"\"\"\"",
                    "",
                    "# 导入必要的模块",
                    "from PySide6.QtCore import Qt",
                    "from PySide6.QtGui import QMouseEvent, QKeyEvent, QWheelEvent",
                    "",
                ])
                new_window.event_function_file = empty_function_file
                
                # 切换到新窗口类（会自动更新所有区域）
                if self._switch_to_window_class(window_name):
                    # 标记项目已修改
                    self.project_manager.set_modified(True)
                    self.status_label.setText(f"已创建新窗口类: {window_name}")
                else:
                    QMessageBox.warning(self, "错误", "切换窗口类失败")
            else:
                QMessageBox.warning(self, "错误", "无法创建窗口类")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"创建窗口类失败: {str(e)}")
    
    def _delete_window_class(self):
        """删除窗口类"""
        try:
            from PySide6.QtWidgets import QInputDialog, QMessageBox
            
            # 检查是否有窗口类管理器
            if not hasattr(self.project_manager, 'window_class_manager'):
                QMessageBox.warning(self, "错误", "窗口类管理器不可用")
                return
            
            window_class_manager = self.project_manager.window_class_manager
            all_names = window_class_manager.get_all_names()
            
            if len(all_names) <= 1:
                QMessageBox.warning(self, "警告", "至少需要保留一个窗口类")
                return
            
            # 让用户选择要删除的窗口类
            current_name = window_class_manager.current_window_class_name or all_names[0]
            window_name, ok = QInputDialog.getItem(
                self,
                "删除窗口类",
                "请选择要删除的窗口类:",
                all_names,
                all_names.index(current_name) if current_name in all_names else 0,
                False
            )
            
            if not ok or not window_name:
                return
            
            # 确认删除
            reply = QMessageBox.question(
                self,
                "确认删除",
                f"确定要删除窗口类 '{window_name}' 吗？\n此操作不可撤销！",
                QMessageBox.Yes | QMessageBox.No
            )
            
            if reply == QMessageBox.Yes:
                # 保存窗口操作撤销状态
                deleted_window_data = None
                deleted_window = window_class_manager.get_window_class(window_name)
                if deleted_window:
                    deleted_window_data = deleted_window.to_dict()
                
                self._save_window_operation_state("删除窗口类", {
                    "operation": "delete",
                    "window_name": window_name,
                    "window_data": deleted_window_data,
                    "before_state": window_class_manager.to_dict()
                })
                
                # 删除窗口类
                if window_class_manager.remove_window_class(window_name):
                    # 切换到剩余的第一个窗口类
                    remaining_names = window_class_manager.get_all_names()
                    if remaining_names:
                        new_current = remaining_names[0]
                        # 使用统一的切换方法
                        self._switch_to_window_class(new_current)
                    
                    # 标记项目已修改
                    self.project_manager.set_modified(True)
                    
                    self.status_label.setText(f"已删除窗口类: {window_name}")
                else:
                    QMessageBox.warning(self, "错误", "无法删除窗口类")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"删除窗口类失败: {str(e)}")

    def _toggle_grid(self):
        self.design_canvas.toggle_grid()
        self.status_label.setText("网格显示已切换")

    def _toggle_snap_to_grid(self):
        self.design_canvas.toggle_snap_to_grid()
        self.status_label.setText("网格对齐已切换")

    def _generate_code(self):
        canvas_width = self.design_canvas.canvas_width
        canvas_height = self.design_canvas.canvas_height
        self.code_generator.set_window_title(self.design_canvas.window_title)
        self.code_generator.set_window_size(canvas_width, canvas_height)
        self.code_generator.set_window_class_name("GeneratedWindow")
        try:
            icon_path = self.design_canvas.window_icon
            if isinstance(icon_path, str) and icon_path:
                self.code_generator.set_window_icon(icon_path)
        except Exception:
            pass
        try:
            pages = getattr(self.design_canvas, 'window_menus', [])
            if pages:
                self.code_generator.set_pages(pages)
        except Exception:
            pass
        try:
            menus = getattr(self.design_canvas, 'window_dropdown_menus', [])
            if menus:
                self.code_generator.set_dropdown_menus(menus)
        except Exception:
            pass
        try:
            initial_code = self.code_generator.generate_code(canvas_width, canvas_height, framework="PySide")
            def provider(framework: str):
                return self.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            dialog = CodeDialog(self, initial_code, code_provider=provider, initial_framework="PySide")
            dialog.exec()
            self.statusBar().showMessage("代码生成成功")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"代码生成失败: {str(e)}")
            self.statusBar().showMessage("代码生成失败")

    def _find_python_interpreter(self):
        import os
        import subprocess
        script_dir = os.path.dirname(os.path.abspath(__file__))
        if getattr(sys, 'frozen', False):
            base_dir = os.path.dirname(sys.executable)
        else:
            base_dir = script_dir
        cwd_dir = os.getcwd()
        candidates = []
        if os.name == 'nt':
            venv_env = os.environ.get('VIRTUAL_ENV', None)
            if venv_env:
                candidates.append(os.path.join(venv_env, 'Scripts', 'python.exe'))
            candidates.extend([
                os.path.join(cwd_dir, 'Python', 'python.exe'),
                os.path.join(base_dir, 'Python', 'python.exe'),
                os.path.join(script_dir, 'venv', 'Scripts', 'python.exe'),
                os.path.join(script_dir, '.venv', 'Scripts', 'python.exe'),
            ])
        else:
            venv_env = os.environ.get('VIRTUAL_ENV', None)
            if venv_env:
                candidates.append(os.path.join(venv_env, 'bin', 'python'))
            candidates.extend([
                os.path.join(cwd_dir, 'Python', 'python'),
                os.path.join(base_dir, 'Python', 'python'),
                os.path.join(script_dir, 'venv', 'bin', 'python'),
                os.path.join(script_dir, '.venv', 'bin', 'python'),
            ])
        for path in candidates:
            if path and os.path.isfile(path):
                return path
        if os.path.isfile(sys.executable):
            return sys.executable
        for python_name in ['python.exe', 'python3.exe', 'python']:
            try:
                if os.name == 'nt':
                    result = subprocess.run(['where', python_name], capture_output=True, text=True, check=False)
                    if result.returncode == 0 and result.stdout:
                        paths = result.stdout.strip().split('\n')
                        for path in paths:
                            if os.path.isfile(path):
                                return path
                else:
                    result = subprocess.run(['which', python_name], capture_output=True, text=True, check=False)
                    found = result.stdout.strip()
                    if result.returncode == 0 and os.path.isfile(found):
                        return found
            except Exception:
                pass
        common_paths = []
        if os.name == 'nt':
            system_drive = os.environ.get('SYSTEMDRIVE', 'C:')
            common_paths.extend([
                os.path.join(system_drive, 'Python39', 'python.exe'),
                os.path.join(system_drive, 'Python310', 'python.exe'),
                os.path.join(system_drive, 'Python311', 'python.exe'),
                os.path.join(system_drive, 'Python312', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python39', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python310', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python311', 'python.exe'),
                os.path.join(system_drive, 'Program Files', 'Python312', 'python.exe'),
            ])
        else:
            common_paths.extend([
                '/usr/bin/python3',
                '/usr/local/bin/python3',
                '/bin/python3',
            ])
        for path in common_paths:
            if os.path.isfile(path):
                return path
        return None

    def _run_project(self):
        import tempfile
        import subprocess
        import os
        import sys
        canvas_width = self.design_canvas.canvas_width
        canvas_height = self.design_canvas.canvas_height
        self.code_generator.set_window_title(self.design_canvas.window_title)
        self.code_generator.set_window_size(canvas_width, canvas_height)
        
        # ========== 设置窗口类名和窗口名称，确保绑定 ==========
        # 使用窗口类名作为基础，确保事件函数文件名和窗口类名绑定，避免冲突
        window_class_name = "GeneratedWindow"
        self.code_generator.set_window_class_name(window_class_name)
        self.code_generator.set_window_name(window_class_name)  # 确保窗口名称和类名一致
        
        try:
            icon_path = self.design_canvas.window_icon
            if isinstance(icon_path, str) and icon_path:
                self.code_generator.set_window_icon(icon_path)
        except Exception:
            pass
        try:
            pages = getattr(self.design_canvas, 'window_menus', [])
            if pages:
                self.code_generator.set_pages(pages)
        except Exception:
            pass
        try:
            menus = getattr(self.design_canvas, 'window_dropdown_menus', [])
            if menus:
                self.code_generator.set_dropdown_menus(menus)
        except Exception:
            pass
        try:
            # 先获取框架类型
            framework = self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide"
            
            code_text = None
            # 尝试从当前活动的代码编辑器获取代码
            current_editor = self._get_current_code_editor()
            if current_editor:
                t = current_editor.toPlainText().strip()
                if t:
                    code_text = t
            if not code_text:
                code_text = self.code_generator.generate_code(canvas_width, canvas_height, framework=framework)
            try:
                code_text = self._ensure_entrypoint_for_code(code_text)
            except Exception:
                pass
            
            # ========== 创建临时目录用于存放代码文件 ==========
            # 使用临时目录而不是单个临时文件，这样可以同时保存主代码文件和事件函数文件
            temp_dir = tempfile.mkdtemp()
            
            # ========== 复制函数库文件到临时目录 ==========
            try:
                from pathlib import Path
                import sys
                sys.path.insert(0, str(Path(__file__).parent.parent))
                from lib.fnpy_library_manager import fnpyLibraryManager
                import shutil
                
                library_manager = fnpyLibraryManager()
                enabled_libraries = library_manager.get_enabled_libraries()
                
                if enabled_libraries:
                    # 创建lib/fnpy目录
                    temp_lib_dir = os.path.join(temp_dir, "lib", "fnpy")
                    os.makedirs(temp_lib_dir, exist_ok=True)
                    
                    # 创建__init__.py文件
                    init_file = os.path.join(temp_lib_dir, "__init__.py")
                    with open(init_file, 'w', encoding='utf-8') as f:
                        f.write("# 函数库初始化文件\n")
                    
                    # 复制函数库文件（从.fnpy文件中提取代码并保存为.py文件）
                    for library in enabled_libraries:
                        source_path = library.file_path
                        # 目标文件名：去掉.fnpy后缀，添加.py后缀
                        dest_name = library.name.replace('.fnpy', '.py')
                        dest_path = os.path.join(temp_lib_dir, dest_name)
                        if os.path.exists(source_path):
                            # 从.fnpy文件中提取代码部分
                            with open(source_path, 'r', encoding='utf-8') as f:
                                content = f.read()
                            
                            # 提取代码部分（在---CODE---之后）
                            if "---CODE---" in content:
                                code = content.split("---CODE---", 1)[1].strip()
                            else:
                                # 如果没有---CODE---标记，尝试解析为纯代码
                                code = content
                            
                            # 保存为.py文件
                            with open(dest_path, 'w', encoding='utf-8') as f:
                                f.write(code)
            except Exception as e:
                print(f"复制函数库文件失败: {e}")
            
            # 支持多窗口类：为每个窗口类生成主文件
            if hasattr(self.project_manager, 'window_class_manager'):
                # 为每个窗口类生成代码文件
                for wc_name in self.project_manager.window_class_manager.get_all_names():
                    window_class = self.project_manager.window_class_manager.get_window_class(wc_name)
                    if not window_class:
                        continue
                    
                    # 切换到该窗口类生成代码
                    old_generator = self.code_generator
                    self.code_generator = window_class.code_generator
                    wc_code = self.code_generator.generate_code(
                        window_class.canvas_width, 
                        window_class.canvas_height, 
                        framework=framework
                    )
                    self.code_generator = old_generator
                    
                    # 确保文件名唯一
                    base_file_name = f"{wc_name}.py"
                    file_name = base_file_name
                    counter = 1
                    temp_file_path = os.path.join(temp_dir, file_name)
                    while os.path.exists(temp_file_path):
                        file_name = f"{wc_name}_{counter}.py"
                        temp_file_path = os.path.join(temp_dir, file_name)
                        counter += 1
                    
                    # 保存窗口类代码文件
                    with open(temp_file_path, 'w', encoding='utf-8') as f:
                        f.write(wc_code)
                
                # 使用第一个窗口类作为主文件运行
                first_window = self.project_manager.window_class_manager.get_all_names()[0]
                temp_file_path = os.path.join(temp_dir, f"{first_window}.py")
            else:
                # 单窗口类模式（向后兼容）
                temp_file_path = os.path.join(temp_dir, f"{window_class_name}.py")
                # ========== 保存主代码文件 ==========
                with open(temp_file_path, 'w', encoding='utf-8') as f:
                    f.write(code_text)
            
            # ========== 检查是否有绑定的事件，如果有则生成事件函数文件 ==========
            has_events = False
            for control in self.control_manager.controls.values():
                bound_events = getattr(control, 'events', {})
                if isinstance(bound_events, dict) and bound_events:
                    # 检查是否有绑定的事件（值为True）
                    if any(is_bound for is_bound in bound_events.values()):
                        has_events = True
                        break
            
            # ========== 如果有绑定的事件，生成事件函数文件 ==========
            # 支持多窗口类，为每个窗口类生成事件函数文件
            existing_function_files = set()
            if hasattr(self.project_manager, 'window_class_manager'):
                # 多窗口类模式：为每个窗口类生成事件函数文件
                for wc_name in self.project_manager.window_class_manager.get_all_names():
                    window_class = self.project_manager.window_class_manager.get_window_class(wc_name)
                    if not window_class:
                        continue
                    
                    # 检查该窗口类是否有事件
                    wc_has_events = False
                    for control in window_class.control_manager.controls.values():
                        bound_events = getattr(control, 'events', {})
                        if isinstance(bound_events, dict) and bound_events:
                            if any(is_bound for is_bound in bound_events.values()):
                                wc_has_events = True
                                break
                    
                    if wc_has_events and framework.lower().startswith("pyside"):
                        from .event_function_generator_pyside import EventFunctionGenerator
                        event_gen = EventFunctionGenerator(window_class.control_manager)
                        event_gen.set_window_name(wc_name)
                        function_content = event_gen.generate_function_file_content()
                        function_file_name = event_gen.get_function_file_name(existing_function_files)
                        existing_function_files.add(function_file_name)
                        function_file_path = os.path.join(temp_dir, function_file_name)
                        
                        # 保存事件函数文件
                        with open(function_file_path, 'w', encoding='utf-8') as f:
                            f.write(function_content)
            elif has_events and framework.lower().startswith("pyside"):
                # 单窗口类模式（向后兼容）
                from .event_function_generator_pyside import EventFunctionGenerator
                event_gen = EventFunctionGenerator(self.control_manager)
                event_gen.set_window_name(window_class_name)
                function_content = event_gen.generate_function_file_content()
                function_file_name = event_gen.get_function_file_name(existing_function_files)
                function_file_path = os.path.join(temp_dir, function_file_name)
                
                # 保存事件函数文件
                with open(function_file_path, 'w', encoding='utf-8') as f:
                    f.write(function_content)
            
            python_exe = self._find_python_interpreter()
            if not python_exe:
                try:
                    python_exe = sys.executable
                except Exception:
                    python_exe = None
            if not python_exe:
                raise RuntimeError("未找到可用的Python解释器")
            
            # 显示终端输出窗口
            if hasattr(self, 'output_panel'):
                self.output_panel.setVisible(True)
                self.btn_toggle_output.setChecked(True)
                self.output_tab_widget.setCurrentIndex(0)  # 切换到终端输出标签页
            
            # 构建命令
            cmd = [python_exe, temp_file_path]
            cmd_str = ' '.join(f'"{arg}"' if ' ' in arg else arg for arg in cmd)
            
            # 显示命令信息
            self._append_terminal_output(f"[命令] {cmd_str}")
            self._append_terminal_output(f"[工作目录] {temp_dir}")
            self._append_terminal_output("=" * 60)
            
            # 运行命令并捕获输出
            try:
                # 创建进程，捕获stdout和stderr
                process = subprocess.Popen(
                    cmd,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.STDOUT,  # 将stderr重定向到stdout
                    text=True,
                    encoding='utf-8',
                    errors='replace',  # 处理编码错误
                    bufsize=1,  # 行缓冲
                    universal_newlines=True,
                    cwd=temp_dir
                )
                
                # 记录进程信息
                self._append_terminal_output(f"[进程ID] {process.pid}")
                self._append_terminal_output(f"[状态] 进程已启动，正在读取输出...")
                self._append_terminal_output("=" * 60)
                
                # 创建后台线程来读取输出
                from PySide6.QtCore import QThread
                
                class OutputReaderThread(QThread):
                    # 定义信号，用于在主线程中更新UI
                    output_received = Signal(str)
                    process_finished = Signal(int)
                    
                    def __init__(self, process):
                        super().__init__()
                        self.process = process
                    
                    def run(self):
                        try:
                            # 实时读取输出
                            for line in iter(self.process.stdout.readline, ''):
                                if line:
                                    # 通过信号发送输出（线程安全）
                                    self.output_received.emit(line.rstrip())
                            
                            # 等待进程结束
                            self.process.wait()
                            # 发送进程结束信号
                            self.process_finished.emit(self.process.returncode)
                        except Exception as e:
                            self.output_received.emit(f"[读取错误] {str(e)}")
                            try:
                                self.process_finished.emit(self.process.returncode if self.process.poll() is not None else -1)
                            except:
                                self.process_finished.emit(-1)
                
                # 创建并启动输出读取线程
                self.output_thread = OutputReaderThread(process)
                # 连接信号到槽函数
                self.output_thread.output_received.connect(self._append_terminal_output)
                self.output_thread.process_finished.connect(self._on_process_finished)
                self.output_thread.start()
                
            except Exception as e:
                self._append_terminal_output(f"[错误] 运行失败: {str(e)}")
                import traceback
                self._append_terminal_output(f"[错误详情] {traceback.format_exc()}")
            
            self.statusBar().showMessage(f"运行代码展示页 (使用: {os.path.basename(python_exe)})...")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"运行项目失败: {str(e)}")
            self.statusBar().showMessage("项目运行失败")
            if hasattr(self, 'terminal_output'):
                self._append_terminal_output(f"[错误] {str(e)}")

    def _ensure_entrypoint_for_code(self, code_text: str) -> str:
        """
        确保代码有入口点（优化版）
        智能检测代码结构，避免重复添加入口点
        """
        t = code_text or ""
        if not t.strip():
            return t
        
        tl = t.lower()
        
        # 检查是否已经有入口点
        has_entrypoint = (
            "__main__" in tl or 
            "qapplication(" in tl or 
            "app.exec" in tl or
            "app.exec_()" in tl or
            "qapplication.instance()" in tl
        )
        
        if has_entrypoint:
            return code_text
        
        # 尝试提取主类名
        cls = "GeneratedWindow"
        try:
            import re
            # 查找最后一个类定义（通常是主窗口类）
            class_matches = list(re.finditer(r"class\s+([A-Za-z_]\w*)\s*(?:\(|:)", t))
            if class_matches:
                # 使用最后一个类（通常是主窗口）
                cls = class_matches[-1].group(1)
        except Exception:
            pass
        
        # 检查是否导入了必要的模块
        has_imports = "import sys" in t or "from sys import" in t
        has_qapp_import = "from PySide6.QtWidgets import" in t or "from PyQt6.QtWidgets import" in t
        
        # 构建入口点代码
        ep_lines = []
        
        # 添加必要的导入（如果缺失）
        if not has_imports:
            ep_lines.append("import sys")
        
        if not has_qapp_import:
            # 检测使用的框架
            if "PySide6" in t:
                ep_lines.append("from PySide6.QtWidgets import QApplication")
            elif "PyQt6" in t:
                ep_lines.append("from PyQt6.QtWidgets import QApplication")
            else:
                ep_lines.append("from PySide6.QtWidgets import QApplication")
        
        # 添加入口点
        if ep_lines:
            ep_lines.append("")
        
        ep_lines.extend([
            "if __name__ == \"__main__\":",
            "    app = QApplication(sys.argv)",
            f"    w = {cls}()",
            "    try:",
            "        w.show()",
            "    except Exception as e:",
            "        print(f\"显示窗口时出错: {{e}}\")",
            "    sys.exit(app.exec())"
        ])
        
        ep = "\n" + "\n".join(ep_lines)
        return t + ep
    
    def _run_code_from_editor(self, code: str):
        """
        从编辑器运行代码（优化版）
        
        Args:
            code: 要运行的代码
        """
        if not code or not code.strip():
            return
        
        import tempfile
        import subprocess
        import os
        import sys
        
        try:
            # 确保代码有入口点
            code = self._ensure_entrypoint_for_code(code)
            
            # 创建临时文件
            temp_dir = tempfile.mkdtemp()
            temp_file_path = os.path.join(temp_dir, "run_code.py")
            
            with open(temp_file_path, 'w', encoding='utf-8') as f:
                f.write(code)
            
            # 查找Python解释器
            python_exe = self._find_python_interpreter()
            if not python_exe:
                python_exe = sys.executable
            if not python_exe:
                from PySide6.QtWidgets import QMessageBox
                QMessageBox.warning(self, "错误", "未找到可用的Python解释器")
                return
            
            # 显示终端输出窗口
            if hasattr(self, 'output_panel'):
                self.output_panel.setVisible(True)
                self.btn_toggle_output.setChecked(True)
                self.output_tab_widget.setCurrentIndex(0)
            
            # 构建命令
            cmd = [python_exe, temp_file_path]
            cmd_str = ' '.join(f'"{arg}"' if ' ' in arg else arg for arg in cmd)
            
            # 显示命令信息
            self._append_terminal_output(f"[命令] {cmd_str}")
            self._append_terminal_output(f"[工作目录] {temp_dir}")
            self._append_terminal_output("=" * 60)
            
            # 运行命令并捕获输出
            process = subprocess.Popen(
                cmd,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                text=True,
                encoding='utf-8',
                errors='replace',
                bufsize=1,
                universal_newlines=True,
                cwd=temp_dir
            )
            
            # 记录进程信息
            self._append_terminal_output(f"[进程ID] {process.pid}")
            self._append_terminal_output(f"[状态] 进程已启动，正在读取输出...")
            self._append_terminal_output("=" * 60)
            
            # 创建后台线程来读取输出
            from PySide6.QtCore import QThread, Signal
            
            class OutputReaderThread(QThread):
                output_received = Signal(str)
                process_finished = Signal(int)
                
                def __init__(self, process):
                    super().__init__()
                    self.process = process
                
                def run(self):
                    try:
                        for line in iter(self.process.stdout.readline, ''):
                            if line:
                                self.output_received.emit(line.rstrip())
                        self.process.wait()
                        self.process_finished.emit(self.process.returncode)
                    except Exception as e:
                        self.output_received.emit(f"[读取错误] {str(e)}")
                        try:
                            self.process_finished.emit(self.process.returncode if self.process.poll() is not None else -1)
                        except:
                            self.process_finished.emit(-1)
            
            # 创建并启动输出读取线程
            self.output_thread = OutputReaderThread(process)
            self.output_thread.output_received.connect(self._append_terminal_output)
            self.output_thread.process_finished.connect(self._on_process_finished)
            self.output_thread.start()
            
            self.statusBar().showMessage(f"运行代码 (使用: {os.path.basename(python_exe)})...")
            
        except Exception as e:
            from PySide6.QtWidgets import QMessageBox
            QMessageBox.critical(self, "错误", f"运行代码失败: {str(e)}")
            self.statusBar().showMessage("代码运行失败")
            if hasattr(self, 'terminal_output'):
                self._append_terminal_output(f"[错误] {str(e)}")

    def _preview_project(self):
        try:
            if self.preview_window is None or not self.preview_window.isVisible():
                self.preview_window = BuiltInPreviewWindow(self, self.control_manager, self.design_canvas)
            else:
                self.preview_window.set_data(self.control_manager, self.design_canvas)
            self.preview_window.show()
            self.statusBar().showMessage("项目预览窗口已打开")
        except Exception as e:
            QMessageBox.warning(self, "预览错误", f"预览项目时出错: {str(e)}")
            self.statusBar().showMessage("预览项目失败")

    def _show_about(self):
        QMessageBox.about(self, "关于可视化编程工具", about_text())
    
    def _show_settings(self):
        """显示设置对话框"""
        dialog = SettingsDialog(self)
        dialog.settings_changed.connect(self._on_settings_changed)
        dialog.exec()
    
    def _on_settings_changed(self):
        """设置改变时的回调"""
        # 应用主题
        self._apply_theme()
        # 应用编辑器设置
        self._apply_editor_settings()
        # 应用画布设置
        self._apply_canvas_settings()
        # 更新AI设置
        self._update_ai_settings()
        # 更新图标大小
        self._update_icon_sizes()
        # 更新图标颜色
        self._update_icons_for_theme(None)
    
    def _apply_theme(self):
        """应用主题（组件库保持独立样式，左侧图标栏应用主题）"""
        from PySide6.QtCore import QSettings
        from .theme_font_manager import ThemeFontManager
        
        settings = QSettings("VisualProgramming", "Settings")
        theme_name = settings.value("theme/name", "white", type=str)
        theme_path = Path(__file__).parent.parent / "ui" / "color" / f"{theme_name}.qss"
        if theme_path.exists():
            with open(theme_path, "r", encoding="utf-8") as f:
                theme_style = f.read()
            
            # 应用主题样式
            self.setStyleSheet(theme_style)
            
            # 应用统一字体
            try:
                if not hasattr(self, '_theme_font_manager'):
                    self._theme_font_manager = ThemeFontManager()
                unified_font = self._theme_font_manager.get_unified_font()
                QApplication.instance().setFont(unified_font)
            except Exception:
                pass
            
            # 更新图标以匹配新主题
            self._update_icons_for_theme(theme_name)
            
            # 组件库保持独立样式，不受主题影响（组件库内的组件使用固定样式）
            if hasattr(self, 'control_palette'):
                self._apply_control_palette_style()
    
    def _update_icons_for_theme(self, theme_name: Optional[str] = None):
        """根据主题更新所有图标"""
        if not hasattr(self, 'theme_manager'):
            return
        
        # 获取自定义图标颜色（如果设置了）
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        custom_icon_color = settings.value("icon/color", "", type=str)
        
        # 更新所有按钮图标
        icon_mappings = [
            ("btn_project", "file.png"),
            ("btn_property", "property.png"),
            ("btn_palette", "component.png"),
            ("btn_data", "data.png"),
            ("btn_ai", "AI.png"),
            ("btn_settings", "setting.png"),
            ("btn_toggle_output", "error.png"),
        ]
        
        for attr_name, icon_name in icon_mappings:
            if hasattr(self, attr_name):
                button = getattr(self, attr_name)
                if button:
                    # 如果设置了自定义图标颜色，优先使用
                    icon_path = self.theme_manager.get_icon_path(
                        icon_name, 
                        theme_name, 
                        icon_color=custom_icon_color if custom_icon_color else None
                    )
                    if os.path.exists(icon_path):
                        button.setIcon(QIcon(icon_path))
    
    def _update_icon_sizes(self):
        """根据设置更新图标大小"""
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        icon_size = settings.value("icon/size", 64, type=int)
        
        # 更新所有按钮的大小和图标大小
        buttons = [
            self.btn_project,
            self.btn_property,
            self.btn_palette,
            self.btn_data,
            self.btn_ai,
            self.btn_settings,
            self.btn_toggle_output,
        ]
        
        for button in buttons:
            if button:
                button.setFixedSize(icon_size, icon_size)
                button.setIconSize(QSize(icon_size, icon_size))
        
        # 更新按钮栏宽度
        if hasattr(self, 'button_bar'):
            self.button_bar.setFixedWidth(icon_size + 8)
    
    def _apply_control_palette_style(self):
        """应用组件库的主题样式（标题、标签页表头应用主题，组件项保持独立样式）"""
        if not hasattr(self, 'control_palette') or not self.control_palette:
            return
        
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        theme_name = settings.value("theme/name", "white", type=str)
        
        # 根据主题设置组件库的样式
        # 组件库的标题和标签页表头应用主题样式，但组件项本身保持独立样式
        if theme_name == "white":
            # 白色主题
            control_palette_style = """
            QWidget#ControlPaletteWidget {
                background-color: white;
            }
            QWidget#ControlPaletteWidget QLabel {
                color: black;
                background-color: transparent;
                font-weight: bold;
                font-size: 14px;
                padding: 5px;
            }
            QWidget#ControlPaletteWidget QTabWidget::pane {
                border: 1px solid #cccccc;
                background-color: white;
                top: -1px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab {
                background-color: #f0f0f0;
                color: black;
                border: 1px solid #cccccc;
                border-bottom-color: #cccccc;
                padding: 5px 8px;
                margin-right: 2px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:selected {
                background-color: white;
                border-bottom-color: white;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:hover {
                background-color: #f5f5f5;
            }
            /* 组件项样式 */
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem {
                background-color: #f0f0f0;
                border: 1px solid #ccc;
                border-radius: 4px;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem:hover {
                background-color: #e0e0e0;
                border: 1px solid #999;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem QLabel {
                color: black;
                background-color: transparent;
            }
            """
        elif theme_name == "black":
            # 黑色主题
            control_palette_style = """
            QWidget#ControlPaletteWidget {
                background-color: #1a1a1a;
            }
            QWidget#ControlPaletteWidget QLabel {
                color: #e0e0e0;
                background-color: transparent;
                font-weight: bold;
                font-size: 14px;
                padding: 5px;
            }
            QWidget#ControlPaletteWidget QTabWidget::pane {
                border: 1px solid #000000;
                background-color: #1a1a1a;
                top: -1px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab {
                background-color: #1a1a1a;
                color: #e0e0e0;
                border: 1px solid #000000;
                border-bottom-color: #000000;
                padding: 5px 8px;
                margin-right: 2px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:selected {
                background-color: #3d3d3d;
                border-bottom-color: #3d3d3d;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:hover {
                background-color: #3d3d3d;
            }
            /* 组件项样式 */
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem {
                background-color: #1a1a1a;
                border: 1px solid #555555;
                border-radius: 4px;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem:hover {
                background-color: #3d3d3d;
                border: 1px solid #777777;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem QLabel {
                color: #e0e0e0;
                background-color: transparent;
            }
            """
        elif theme_name == "bluewhite":
            # 蓝白主题
            control_palette_style = """
            QWidget#ControlPaletteWidget {
                background-color: white;
            }
            QWidget#ControlPaletteWidget QLabel {
                color: #0066cc;
                background-color: transparent;
                font-weight: bold;
                font-size: 14px;
                padding: 5px;
            }
            QWidget#ControlPaletteWidget QTabWidget::pane {
                border: 1px solid #004499;
                background-color: white;
                top: -1px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab {
                background-color: #e6f2ff;
                color: #0066cc;
                border: 1px solid #004499;
                border-bottom-color: #004499;
                padding: 5px 8px;
                margin-right: 2px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:selected {
                background-color: white;
                border-bottom-color: white;
                color: #0066cc;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:hover {
                background-color: #cce6ff;
            }
            /* 组件项样式 */
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem {
                background-color: #e6f2ff;
                border: 1px solid #66aaff;
                border-radius: 4px;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem:hover {
                background-color: #cce6ff;
                border: 1px solid #004499;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem QLabel {
                color: #0066cc;
                background-color: transparent;
            }
            """
        elif theme_name == "blackpink":
            # 黑粉主题
            control_palette_style = """
            QWidget#ControlPaletteWidget {
                background-color: #1a1a1a;
            }
            QWidget#ControlPaletteWidget QLabel {
                color: #ff66aa;
                background-color: transparent;
                font-weight: bold;
                font-size: 14px;
                padding: 5px;
            }
            QWidget#ControlPaletteWidget QTabWidget::pane {
                border: 1px solid #ff66aa;
                background-color: #1a1a1a;
                top: -1px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab {
                background-color: #1a1a1a;
                color: #ff99cc;
                border: 1px solid #ff66aa;
                border-bottom-color: #ff66aa;
                padding: 5px 8px;
                margin-right: 2px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:selected {
                background-color: #3d3d3d;
                border-bottom-color: #3d3d3d;
                color: #ff66aa;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:hover {
                background-color: #3d3d3d;
            }
            /* 组件项样式 */
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem {
                background-color: #1a1a1a;
                border: 1px solid #ff66aa;
                border-radius: 4px;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem:hover {
                background-color: #3d3d3d;
                border: 1px solid #ff99cc;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem QLabel {
                color: #ff99cc;
                background-color: transparent;
            }
            """
        elif theme_name == "blackblue":
            # 黑蓝主题
            control_palette_style = """
            QWidget#ControlPaletteWidget {
                background-color: #1a1a1a;
            }
            QWidget#ControlPaletteWidget QLabel {
                color: #66aaff;
                background-color: transparent;
                font-weight: bold;
                font-size: 14px;
                padding: 5px;
            }
            QWidget#ControlPaletteWidget QTabWidget::pane {
                border: 1px solid #0066cc;
                background-color: #1a1a1a;
                top: -1px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab {
                background-color: #1a1a1a;
                color: #66aaff;
                border: 1px solid #0066cc;
                border-bottom-color: #0066cc;
                padding: 5px 8px;
                margin-right: 2px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:selected {
                background-color: #3d3d3d;
                border-bottom-color: #3d3d3d;
                color: #0088ff;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:hover {
                background-color: #3d3d3d;
            }
            /* 组件项样式 */
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem {
                background-color: #1a1a1a;
                border: 1px solid #0066cc;
                border-radius: 4px;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem:hover {
                background-color: #3d3d3d;
                border: 1px solid #0088ff;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem QLabel {
                color: #66aaff;
                background-color: transparent;
            }
            """
        else:
            # 默认白色主题
            control_palette_style = """
            QWidget#ControlPaletteWidget {
                background-color: white;
            }
            QWidget#ControlPaletteWidget QLabel {
                color: black;
                background-color: transparent;
                font-weight: bold;
                font-size: 14px;
                padding: 5px;
            }
            QWidget#ControlPaletteWidget QTabWidget::pane {
                border: 1px solid #cccccc;
                background-color: white;
                top: -1px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab {
                background-color: #f0f0f0;
                color: black;
                border: 1px solid #cccccc;
                border-bottom-color: #cccccc;
                padding: 5px 8px;
                margin-right: 2px;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:selected {
                background-color: white;
                border-bottom-color: white;
            }
            QWidget#ControlPaletteWidget QTabBar::tab:hover {
                background-color: #f5f5f5;
            }
            /* 组件项样式 */
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem {
                background-color: #f0f0f0;
                border: 1px solid #ccc;
                border-radius: 4px;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem:hover {
                background-color: #e0e0e0;
                border: 1px solid #999;
            }
            QWidget#ControlPaletteWidget QWidget#ControlPaletteItem QLabel {
                color: black;
                background-color: transparent;
            }
            """
        
        self.control_palette.setStyleSheet(control_palette_style)
    
    def _apply_editor_settings(self):
        """应用编辑器设置"""
        from PySide6.QtCore import QSettings
        from PySide6.QtGui import QFont
        settings = QSettings("VisualProgramming", "Settings")
        
        theme = settings.value("editor/theme", "vs", type=str)
        language = settings.value("editor/language", "python", type=str)
        font_size = settings.value("editor/font_size", 14, type=int)
        font_family = settings.value("editor/font", "Consolas", type=str)
        
        # 更新所有代码编辑器（通过标签页中的编辑器）
        # 不再使用固定的 code_view_editor_left
        
        # 更新所有标签页中的编辑器
        if hasattr(self, '_code_editors'):
            for editor in self._code_editors.values():
                if hasattr(editor, 'setTheme'):
                    editor.setTheme(theme)
                if hasattr(editor, 'setLanguage'):
                    editor.setLanguage(language)
                if hasattr(editor, 'setFont'):
                    editor.setFont(QFont(font_family, font_size))
    
    def _apply_editor_settings_to_editor(self, editor: MonacoEditorWidget):
        """为单个编辑器应用设置"""
        from PySide6.QtCore import QSettings
        from PySide6.QtGui import QFont
        settings = QSettings("VisualProgramming", "Settings")
        
        theme = settings.value("editor/theme", "vs", type=str)
        language = settings.value("editor/language", "python", type=str)
        font_size = settings.value("editor/font_size", 14, type=int)
        font_family = settings.value("editor/font", "Consolas", type=str)
        
        if hasattr(editor, 'setTheme'):
            editor.setTheme(theme)
        if hasattr(editor, 'setLanguage'):
            editor.setLanguage(language)
        if hasattr(editor, 'setFont'):
            editor.setFont(QFont(font_family, font_size))
    
    def _update_ai_settings(self):
        """更新AI设置"""
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        
        if hasattr(self, 'ai_chat_widget') and self.ai_chat_widget:
            api_key = settings.value("ai/api_key", "", type=str)
            api_url = settings.value("ai/api_url", "https://api.deepseek.com/v1/chat/completions", type=str)
            provider = settings.value("ai/provider", "DeepSeek", type=str)
            model = settings.value("ai/model", "deepseek-chat", type=str)
            
            self.ai_chat_widget.api_key = api_key
            self.ai_chat_widget.api_url = api_url
            self.ai_chat_widget.provider = provider
            self.ai_chat_widget.model_id = model
    
    def _load_settings_on_startup(self):
        """在窗口启动时加载所有设置"""
        from PySide6.QtCore import QSettings, QByteArray
        from PySide6.QtGui import QFont
        settings = QSettings("VisualProgramming", "Settings")
        
        # ========== 恢复窗口状态（位置、大小、分割器状态） ==========
        remember_window_state = settings.value("basic/remember_window_state", True, type=bool)
        if remember_window_state:
            # 恢复窗口几何信息（位置和大小）
            geometry = settings.value("window/geometry", None)
            if geometry is not None:
                try:
                    self.restoreGeometry(QByteArray.fromBase64(geometry.encode()))
                except Exception:
                    pass
            
            # 恢复窗口状态（最大化、最小化等）
            window_state = settings.value("window/state", None)
            if window_state is not None:
                try:
                    self.restoreState(QByteArray.fromBase64(window_state.encode()))
                except Exception:
                    pass
            
            # 恢复主分割器状态
            main_splitter_state = settings.value("window/main_splitter_state", None)
            if main_splitter_state is not None and hasattr(self, 'main_splitter'):
                try:
                    self.main_splitter.restoreState(QByteArray.fromBase64(main_splitter_state.encode()))
                except Exception:
                    pass
            
            # 恢复中间分割器状态（代码编辑器和画布）
            center_splitter_state = settings.value("window/center_splitter_state", None)
            if center_splitter_state is not None and hasattr(self, 'center_splitter'):
                try:
                    self.center_splitter.restoreState(QByteArray.fromBase64(center_splitter_state.encode()))
                except Exception:
                    pass
            
            # 恢复左侧面板宽度
            left_panel_width = settings.value("window/left_panel_width", 250, type=int)
            if hasattr(self, '_left_tab_widget_saved_width'):
                self._left_tab_widget_saved_width = left_panel_width
        
        # ========== 应用主题 ==========
        self._apply_theme()
        
        # ========== 应用编辑器设置（延迟执行，等待编辑器初始化完成）==========
        # Monaco Editor 需要时间加载和初始化，使用延迟调用确保编辑器已准备好
        QTimer.singleShot(3000, self._apply_editor_settings)
        
        # ========== 应用画布设置 ==========
        self._apply_canvas_settings()
        
        # ========== 应用组件库独立样式 ==========
        # 确保组件库在启动时也有独立样式，不受主题影响
        if hasattr(self, 'control_palette'):
            self._apply_control_palette_style()
        
        # ========== 更新AI设置 ==========
        self._update_ai_settings()
    
    def _apply_canvas_settings(self):
        """应用画布相关设置"""
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        
        if hasattr(self, 'design_canvas') and self.design_canvas:
            # 应用网格设置
            show_grid = settings.value("page/show_grid", True, type=bool)
            grid_size = settings.value("page/grid_size", 10, type=int)
            snap_to_grid = settings.value("page/snap_to_grid", True, type=bool)
            snap_distance = settings.value("page/snap_distance", 5, type=int)
            show_ruler = settings.value("page/show_ruler", False, type=bool)
            
            if hasattr(self.design_canvas, 'set_show_grid'):
                self.design_canvas.set_show_grid(show_grid)
            if hasattr(self.design_canvas, 'set_grid_size'):
                self.design_canvas.set_grid_size(grid_size)
            if hasattr(self.design_canvas, 'set_snap_to_grid'):
                self.design_canvas.set_snap_to_grid(snap_to_grid)
            if hasattr(self.design_canvas, 'set_snap_distance'):
                self.design_canvas.set_snap_distance(snap_distance)
            if hasattr(self.design_canvas, 'set_show_ruler'):
                self.design_canvas.set_show_ruler(show_ruler)
    
    def _close_code_tab(self, index: int):
        """关闭代码标签页"""
        # 允许关闭所有标签页，包括第一个
        # 获取标签页名称
        tab_text = self.code_tab_widget.tabText(index) if index < self.code_tab_widget.count() else ""
        if not tab_text:
            return
        
        # 在关闭前保存代码内容到窗口类
        if tab_text in self._code_editors:
            editor = self._code_editors[tab_text]
            content = editor.toPlainText()
            
            # 判断是画布代码还是功能代码
            if tab_text.endswith("_canvas.py"):
                # 画布代码：从标签名提取窗口名
                window_name = tab_text.replace("_canvas.py", "")
                if hasattr(self.project_manager, 'window_class_manager'):
                    window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                    if window_class:
                        window_class.canvas_code = content
            elif tab_text.endswith("_function.py"):
                # 功能代码：从标签名提取窗口名
                window_name = tab_text.replace("_function.py", "")
                if hasattr(self.project_manager, 'window_class_manager'):
                    window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                    if window_class:
                        window_class.event_function_file = content
        
        # 移除标签页
        self.code_tab_widget.removeTab(index)
        
        # 从编辑器中移除
        if tab_text in self._code_editors:
            del self._code_editors[tab_text]
    
    def _on_code_tab_changed(self, index: int):
        """代码标签页切换"""
        # 可以在这里添加切换时的处理逻辑
        pass
    
    def _toggle_left_panel_tab(self, tab_name: str):
        """切换左侧面板标签页"""
        # 查找标签页索引
        tab_index = -1
        for i in range(self.left_tab_widget.count()):
            if self.left_tab_widget.tabText(i) == tab_name:
                tab_index = i
                break
        
        # 如果标签页不存在，重新添加
        if tab_index == -1:
            if tab_name == "属性编辑":
                tab_index = self.left_tab_widget.addTab(self.property_editor, "属性编辑")
            elif tab_name == "组件库":
                tab_index = self.left_tab_widget.addTab(self.control_palette, "组件库")
            elif tab_name == "函数库":
                tab_index = self.left_tab_widget.addTab(self.fnpy_library_widget, "函数库")
            elif tab_name == "项目目录":
                tab_index = self.left_tab_widget.addTab(self.project_preview_widget, "项目目录")
            elif tab_name == "AI聊天":
                tab_index = self.left_tab_widget.addTab(self.ai_chat_widget, "AI聊天")
            else:
                return
        
        # 切换标签页
        current_index = self.left_tab_widget.currentIndex()
        if current_index == tab_index and self.left_tab_widget.isVisible():
            # 如果当前已选中且面板可见，则隐藏面板
            # 保存当前宽度
            self._left_tab_widget_saved_width = self.left_tab_widget.width()
            self.left_tab_widget.setVisible(False)
            # 调整左侧面板宽度为按钮栏宽度（50像素）
            self.left_panel.setFixedWidth(50)
            self.btn_property.setChecked(False)
            self.btn_palette.setChecked(False)
            self.btn_project.setChecked(False)
            self.btn_ai.setChecked(False)
            self.btn_settings.setChecked(False)
        else:
            # 切换到指定标签页并显示面板
            self.left_tab_widget.setCurrentIndex(tab_index)
            self.left_tab_widget.setVisible(True)
            # 恢复左侧面板宽度限制（允许调整大小）
            self.left_panel.setMinimumWidth(50)
            self.left_panel.setMaximumWidth(16777215)  # QWIDGETSIZE_MAX
            # 恢复选择夹宽度设置
            self.left_tab_widget.setMinimumWidth(200)
            self.left_tab_widget.setMaximumWidth(400)
            # 如果之前保存了宽度，尝试恢复
            if hasattr(self, '_left_tab_widget_saved_width') and self._left_tab_widget_saved_width > 0:
                try:
                    self.left_tab_widget.setMinimumWidth(max(200, self._left_tab_widget_saved_width))
                except Exception:
                    pass
            # 更新按钮状态
            self.btn_property.setChecked(tab_name == "属性编辑")
            self.btn_palette.setChecked(tab_name == "组件库")
            self.btn_project.setChecked(tab_name == "项目目录")
            self.btn_ai.setChecked(tab_name == "AI聊天")
            self.btn_settings.setChecked(False)  # 设置按钮不参与标签页切换
    
    def _close_left_tab(self, index: int):
        """关闭左侧标签页"""
        # 获取标签页名称
        tab_name = self.left_tab_widget.tabText(index)
        
        # 保存当前选择夹的宽度（如果可见）
        if self.left_tab_widget.isVisible():
            self._left_tab_widget_saved_width = self.left_tab_widget.width()
        
        # 移除标签页
        self.left_tab_widget.removeTab(index)
        
        # 更新按钮状态
        if tab_name == "属性编辑":
            self.btn_property.setChecked(False)
        elif tab_name == "组件库":
            self.btn_palette.setChecked(False)
        elif tab_name == "项目目录":
            self.btn_project.setChecked(False)
        elif tab_name == "AI聊天":
            self.btn_ai.setChecked(False)
        self.btn_settings.setChecked(False)
        
        # 如果所有标签页都关闭了，隐藏选择夹并调整宽度
        if self.left_tab_widget.count() == 0:
            self.left_tab_widget.setVisible(False)
            # 调整左侧面板宽度为按钮栏宽度（50像素）
            # 使用setFixedWidth确保宽度固定
            self.left_panel.setFixedWidth(50)
        else:
            # 切换到下一个标签页
            new_index = min(index, self.left_tab_widget.count() - 1)
            self.left_tab_widget.setCurrentIndex(new_index)
            # 更新按钮状态
            current_tab_name = self.left_tab_widget.tabText(new_index)
            self.btn_property.setChecked(current_tab_name == "属性编辑")
            self.btn_palette.setChecked(current_tab_name == "组件库")
            self.btn_project.setChecked(current_tab_name == "项目目录")
            self.btn_ai.setChecked(current_tab_name == "AI聊天")
            self.btn_settings.setChecked(False)
    
    def _on_left_tab_changed(self, index: int):
        """左侧标签页切换"""
        if index >= 0 and index < self.left_tab_widget.count():
            tab_name = self.left_tab_widget.tabText(index)
            # 更新按钮状态
            self.btn_property.setChecked(tab_name == "属性编辑")
            self.btn_palette.setChecked(tab_name == "组件库")
            self.btn_fnpy_library.setChecked(tab_name == "函数库")
            self.btn_project.setChecked(tab_name == "项目目录")
            self.btn_ai.setChecked(tab_name == "AI聊天")
            # 设置按钮不参与标签页切换
            self.btn_settings.setChecked(False)
    
    def _update_event_function_tab(self):
        """更新事件函数文件标签页"""
        try:
            # 获取当前窗口类名称
            window_name = None
            if hasattr(self.project_manager, 'window_class_manager'):
                window_name = self.project_manager.window_class_manager.current_window_class_name
            if not window_name:
                window_name = self.project_manager.get_window_name() if hasattr(self.project_manager, 'get_window_name') else "GeneratedWindow"
            
            # 获取当前窗口类
            current_window = None
            if hasattr(self.project_manager, 'window_class_manager'):
                current_window = self.project_manager.window_class_manager.get_window_class(window_name)
            
            # 获取当前窗口类的功能函数文件内容（优先从窗口类获取）
            function_content = ""
            if current_window:
                function_content = getattr(current_window, 'event_function_file', "") or ""
            
            # 如果窗口类中没有内容，使用事件函数生成器生成
            if not function_content:
                from module.event_function_generator_pyside import EventFunctionGenerator
                # 使用当前窗口类的控件管理器，如果没有则使用默认的
                control_mgr = current_window.control_manager if current_window else self.control_manager
                event_gen = EventFunctionGenerator(control_mgr)
                event_gen.set_window_name(window_name)
                function_content = event_gen.generate_function_file_content()
                
                # 保存到窗口类
                if current_window:
                    current_window.event_function_file = function_content
            
            # 检查标签页是否已存在
            tab_name = f"{window_name}_function.py"
            if tab_name in self._code_editors:
                # 更新现有标签页内容
                editor = self._code_editors[tab_name]
                current_content = editor.toPlainText()
                
                # 如果内容不同，更新标签页（确保使用窗口类中的最新代码）
                if current_content != function_content:
                    # 先断开旧的连接，避免触发保存
                    try:
                        editor.textChanged.disconnect()
                    except Exception:
                        pass
                    # 更新内容
                    editor.setPlainText(function_content)
                    # 清除撤销状态标记（因为内容被重置了）
                    if hasattr(editor, '_undo_state_saved'):
                        delattr(editor, '_undo_state_saved')
                    # 重新连接，确保使用正确的窗口名
                    editor.textChanged.connect(lambda: self._on_event_function_changed(tab_name, editor, window_name))
                
                # 切换到该标签页
                for i in range(self.code_tab_widget.count()):
                    if self.code_tab_widget.tabText(i) == tab_name:
                        self.code_tab_widget.setCurrentIndex(i)
                        break
            else:
                # 创建新标签页
                editor = MonacoEditorWidget()
                editor.setReadOnly(False)
                editor.setLineWrapMode(QPlainTextEdit.NoWrap)
                
                # 连接运行代码信号
                if hasattr(editor, 'run_code_requested'):
                    editor.run_code_requested.connect(self._run_code_from_editor)
                
                # 应用编辑器设置
                self._apply_editor_settings_to_editor(editor)
                editor.setPlainText(function_content)
                # 初始化撤销状态标记（Monaco Editor 内置撤销功能，此标记可能不需要）
                if hasattr(editor, '_undo_state_saved'):
                    editor._undo_state_saved = False
                editor.textChanged.connect(lambda: self._on_event_function_changed(tab_name, editor, window_name))
                
                self.code_tab_widget.addTab(editor, tab_name)
                self._code_editors[tab_name] = editor
                # 切换到新标签页
                self.code_tab_widget.setCurrentIndex(self.code_tab_widget.count() - 1)
        except Exception as e:
            import traceback
            traceback.print_exc()
    
    def _on_event_function_changed(self, tab_name: str, editor: MonacoEditorWidget, window_name: str = None):
        """事件函数文件内容改变"""
        try:
            content = editor.toPlainText()
            
            # 保存到窗口类（优先）
            if hasattr(self.project_manager, 'window_class_manager'):
                if not window_name:
                    # 从标签名提取窗口名
                    if tab_name.endswith('_function.py'):
                        window_name = tab_name[:-13]  # 去掉 '_function.py'
                    else:
                        window_name = self.project_manager.window_class_manager.current_window_class_name
                
                if window_name:
                    current_window = self.project_manager.window_class_manager.get_window_class(window_name)
                    if current_window:
                        # 检查内容是否真的改变了（避免重复保存）
                        old_content = getattr(current_window, 'event_function_file', "") or ""
                        if old_content != content:
                            # 只在第一次改变时保存撤销状态（使用标记避免频繁保存）
                            if not hasattr(editor, '_undo_state_saved'):
                                if hasattr(self, 'undo_redo_manager') and self.undo_redo_manager:
                                    # 保存当前状态（在改变之前）
                                    self.undo_redo_manager.save_state("编辑功能代码", include_function_file=True)
                                editor._undo_state_saved = True
                            
                            current_window.event_function_file = content
                            self.project_manager.set_modified(True)
                        return
            
            # 向后兼容：保存到项目管理器
            if hasattr(self.project_manager, 'set_event_function_file'):
                self.project_manager.set_event_function_file(content)
        except Exception as e:
            print(f"保存功能函数文件失败: {e}")
    
    def _get_current_event_function_content(self) -> str:
        """获取当前窗口类的功能代码文件内容"""
        try:
            if not hasattr(self.project_manager, 'window_class_manager'):
                return ""
            
            current_name = self.project_manager.window_class_manager.current_window_class_name
            if not current_name:
                return ""
            
            # 优先从代码编辑器获取（实时内容）
            tab_name = f"{current_name}_function.py"
            if hasattr(self, '_code_editors') and tab_name in self._code_editors:
                editor = self._code_editors[tab_name]
                return editor.toPlainText()
            
            # 从窗口类获取
            current_window = self.project_manager.window_class_manager.get_window_class(current_name)
            if current_window:
                return getattr(current_window, 'event_function_file', "") or ""
            
            return ""
        except Exception as e:
            print(f"获取功能代码文件内容失败: {e}")
            return ""
    
    def _update_canvas_code_tab(self, window_name: str, window_class):
        """更新画布代码标签页"""
        try:
            canvas_tab_name = f"{window_name}_canvas.py"
            
            # 确保代码生成器使用正确的控件管理器（重要！）
            window_class.code_generator.control_manager = window_class.control_manager
            
            # 获取画布代码内容（优先从窗口类获取，确保是最新的）
            # 重要：始终从窗口类获取保存的代码，不要重新生成，以保留用户的编辑
            canvas_code = getattr(window_class, 'canvas_code', "") or ""
            
            # 如果窗口类中没有保存的代码，才生成新的代码
            if not canvas_code:
                canvas_width = window_class.canvas_width
                canvas_height = window_class.canvas_height
                # 使用窗口类自己的代码生成器
                code_generator = window_class.code_generator
                # 确保代码生成器使用正确的控件管理器
                code_generator.control_manager = window_class.control_manager
                code_generator.set_window_title(window_class.window_title)
                code_generator.set_window_size(canvas_width, canvas_height)
                code_generator.set_window_class_name(window_name)
                code_generator.set_window_name(window_name)
                code_generator.set_window_icon(window_class.window_icon_path)
                code_generator.set_pages(window_class.pages)
                code_generator.set_dropdown_menus(window_class.dropdown_menus)
                
                framework = getattr(self, 'fixed_framework', None) or (
                    self.framework_combo_left.currentText() if hasattr(self, 'framework_combo_left') else "PySide"
                )
                canvas_code = code_generator.generate_code(canvas_width, canvas_height, framework=framework)
                # 保存生成的代码到窗口类
                window_class.canvas_code = canvas_code
            # 注意：如果窗口类中已经有保存的代码，我们使用它
            # 代码的更新由 _refresh_code_view_left 负责（当添加控件时调用）
            
            # 检查标签页是否已存在
            if canvas_tab_name in self._code_editors:
                # 标签页已存在，优先使用编辑器中的内容（保留用户编辑）
                editor = self._code_editors[canvas_tab_name]
                current_content = editor.toPlainText()
                
                # 只有当编辑器内容为空时，才使用窗口类中保存的代码
                # 这样可以保留用户编辑的内容
                if not current_content or current_content.strip() == "":
                    # 编辑器为空，使用窗口类中保存的代码
                    try:
                        editor.textChanged.disconnect()
                    except Exception:
                        pass
                    editor.setPlainText(canvas_code)
                    editor.textChanged.connect(lambda: self._on_canvas_code_changed(canvas_tab_name, editor, window_name))
                # 如果编辑器有内容，保留它（用户可能已经编辑过，或者这是重新打开的标签页）
                
                # 切换到该标签页
                for i in range(self.code_tab_widget.count()):
                    if self.code_tab_widget.tabText(i) == canvas_tab_name:
                        self.code_tab_widget.setCurrentIndex(i)
                        break
            else:
                # 创建新标签页
                editor = MonacoEditorWidget()
                editor.setReadOnly(False)
                editor.setLineWrapMode(QPlainTextEdit.NoWrap)
                
                # 连接运行代码信号
                if hasattr(editor, 'run_code_requested'):
                    editor.run_code_requested.connect(self._run_code_from_editor)
                
                # 应用编辑器设置
                self._apply_editor_settings_to_editor(editor)
                # 设置代码内容（从窗口类获取，确保正确）
                editor.setPlainText(canvas_code)
                # 连接文本改变信号，确保使用正确的窗口名
                editor.textChanged.connect(lambda: self._on_canvas_code_changed(canvas_tab_name, editor, window_name))
                
                self.code_tab_widget.addTab(editor, canvas_tab_name)
                self._code_editors[canvas_tab_name] = editor
                # 切换到新标签页
                self.code_tab_widget.setCurrentIndex(self.code_tab_widget.count() - 1)
        except Exception as e:
            print(f"更新画布代码标签页失败: {e}")
            import traceback
            traceback.print_exc()
    
    def _on_canvas_code_changed(self, tab_name: str, editor: MonacoEditorWidget, window_name: str = None):
        """画布代码内容改变"""
        try:
            content = editor.toPlainText()
            
            # 保存到窗口类
            if hasattr(self.project_manager, 'window_class_manager'):
                if not window_name:
                    # 从标签名提取窗口名
                    if tab_name.endswith('_canvas.py'):
                        window_name = tab_name[:-11]  # 去掉 '_canvas.py'
                    else:
                        window_name = self.project_manager.window_class_manager.current_window_class_name
                
                if window_name:
                    current_window = self.project_manager.window_class_manager.get_window_class(window_name)
                    if current_window:
                        # 检查内容是否真的改变了（避免重复保存）
                        old_content = getattr(current_window, 'canvas_code', "") or ""
                        if old_content != content:
                            # 只在第一次改变时保存撤销状态（使用标记避免频繁保存）
                            if not hasattr(editor, '_undo_state_saved') or not editor._undo_state_saved:
                                self.undo_redo_manager.save_state("编辑画布代码", include_canvas_code=True, canvas_code_content=old_content)
                                editor._undo_state_saved = True
                                # 重置标记，等待下一次编辑
                                from PySide6.QtCore import QTimer
                                QTimer.singleShot(1000, lambda: setattr(editor, '_undo_state_saved', False))
                        
                        current_window.canvas_code = content
                        self.project_manager.set_modified(True)
        except Exception as e:
            print(f"保存画布代码失败: {e}")
    
    def _get_current_code_editor(self) -> MonacoEditorWidget:
        """获取当前活动的代码编辑器"""
        current_index = self.code_tab_widget.currentIndex()
        if current_index >= 0 and current_index < self.code_tab_widget.count():
            editor = self.code_tab_widget.widget(current_index)
            if editor:
                return editor
        # 如果没有活动的标签页，返回None
        return None
    
    def _create_project_preview_widget(self) -> QWidget:
        """创建项目预览组件"""
        widget = QWidget()
        layout = QVBoxLayout(widget)
        layout.setContentsMargins(5, 5, 5, 5)
        
        # 标题
        title_label = QLabel("项目预览")
        title_label.setStyleSheet("font-weight: bold; font-size: 14px; padding: 5px;")
        layout.addWidget(title_label)
        
        # 创建树形控件显示层次化项目结构
        self.preview_tree = QTreeWidget()
        self.preview_tree.setHeaderLabel("项目结构")
        self.preview_tree.setColumnCount(1)
        # 连接单击和双击事件
        self.preview_tree.itemClicked.connect(self._on_preview_item_clicked)
        self.preview_tree.itemDoubleClicked.connect(self._on_preview_item_double_clicked)
        # 启用右键菜单
        self.preview_tree.setContextMenuPolicy(Qt.CustomContextMenu)
        self.preview_tree.customContextMenuRequested.connect(self._show_preview_tree_context_menu)
        # 支持键盘事件（用于Delete删除和Ctrl+C复制）
        self.preview_tree.setFocusPolicy(Qt.StrongFocus)
        # 创建键盘事件过滤器
        from PySide6.QtCore import QObject
        class TreeKeyFilter(QObject):
            def __init__(self, main_window):
                super().__init__()
                self.main_window = main_window
            
            def eventFilter(self, obj, event):
                if event.type() == event.Type.KeyPress:
                    key = event.key()
                    if key == Qt.Key_Delete:
                        self.main_window._delete_selected_tree_item()
                        return True
                    elif key == Qt.Key_C and (event.modifiers() & Qt.ControlModifier):
                        self.main_window._copy_selected_tree_item()
                        return True
                return super().eventFilter(obj, event)
        
        self.tree_key_filter = TreeKeyFilter(self)
        self.preview_tree.installEventFilter(self.tree_key_filter)
        layout.addWidget(self.preview_tree)
        
        # 刷新按钮
        refresh_btn = QPushButton("刷新预览")
        refresh_btn.clicked.connect(self._refresh_project_preview)
        layout.addWidget(refresh_btn)
        
        return widget
    
    def _get_target_folder_path(self, item_type, data):
        """
        根据节点类型获取目标文件夹路径
        
        Args:
            item_type: 节点类型
            data: 节点数据
            
        Returns:
            目标文件夹路径，如果无法确定则返回项目目录
        """
        # 如果是文件夹或项目根目录，直接返回路径
        if item_type == "folder":
            return data[1] if len(data) > 1 else None
        elif item_type == "project_root":
            return data[1] if len(data) > 1 else None
        
        # 如果是文件，返回文件所在目录
        elif item_type == "file":
            file_path = data[1] if len(data) > 1 else None
            if file_path:
                return os.path.dirname(file_path)
        
        # 其他情况，返回项目目录
        if hasattr(self, 'project_dir') and self.project_dir and os.path.exists(self.project_dir):
            return self.project_dir
        
        return None
    
    def _add_file_operations_with_selection(self, menu, default_path):
        """添加通过文件选择对话框的文件操作菜单项"""
        from PySide6.QtWidgets import QFileDialog
        
        def create_action(text, callback):
            action = QAction(text, self)
            action.triggered.connect(callback)
            return action
        
        # 新建文件（需要选择目录）
        def new_file_with_selection():
            folder_path = QFileDialog.getExistingDirectory(self, "选择要创建文件的文件夹", default_path)
            if folder_path:
                self._new_file_in_folder(folder_path)
        
        new_file_action = create_action("新建文件", new_file_with_selection)
        menu.addAction(new_file_action)
        
        # 新建文件夹（需要选择目录）
        def new_folder_with_selection():
            folder_path = QFileDialog.getExistingDirectory(self, "选择要创建文件夹的目录", default_path)
            if folder_path:
                self._new_folder_in_folder(folder_path)
        
        new_folder_action = create_action("新建文件夹", new_folder_with_selection)
        menu.addAction(new_folder_action)
        
        menu.addSeparator()
        
        # 粘贴（如果有剪贴板内容，需要选择目标目录）
        if self._clipboard_file_path:
            def paste_with_selection():
                folder_path = QFileDialog.getExistingDirectory(self, "选择要粘贴到的文件夹", default_path)
                if folder_path:
                    self._paste_file(folder_path)
            
            paste_action = create_action("粘贴", paste_with_selection)
            menu.addAction(paste_action)
            menu.addSeparator()
        
        # 复制文件
        def copy_with_selection():
            file_path, _ = QFileDialog.getOpenFileName(
                self, "选择要复制的文件", default_path, "所有文件 (*.*)"
            )
            if file_path:
                self._copy_file(file_path)
        
        copy_action = create_action("复制文件", copy_with_selection)
        menu.addAction(copy_action)
        
        # 剪切文件
        def cut_with_selection():
            file_path, _ = QFileDialog.getOpenFileName(
                self, "选择要剪切的文件", default_path, "所有文件 (*.*)"
            )
            if file_path:
                self._cut_file(file_path)
        
        cut_action = create_action("剪切文件", cut_with_selection)
        menu.addAction(cut_action)
        
        menu.addSeparator()
        
        # 重命名文件
        def rename_with_selection():
            file_path, _ = QFileDialog.getOpenFileName(
                self, "选择要重命名的文件", default_path, "所有文件 (*.*)"
            )
            if file_path:
                self._rename_file(file_path)
        
        rename_action = create_action("重命名文件", rename_with_selection)
        menu.addAction(rename_action)
        
        # 删除文件
        def delete_with_selection():
            file_path, _ = QFileDialog.getOpenFileName(
                self, "选择要删除的文件", default_path, "所有文件 (*.*)"
            )
            if file_path:
                self._delete_file(file_path)
        
        delete_action = create_action("删除文件", delete_with_selection)
        menu.addAction(delete_action)
        
        menu.addSeparator()
    
    def _add_common_file_operations(self, menu, target_folder_path, item_type=None, data=None, show_all_operations=False):
        """
        添加通用文件操作菜单项
        
        Args:
            menu: 菜单对象
            target_folder_path: 目标文件夹路径
            item_type: 节点类型（可选）
            data: 节点数据（可选）
            show_all_operations: 是否显示所有操作（即使没有具体路径）
        """
        # 如果目标文件夹路径不存在，尝试使用后备路径
        if not target_folder_path or not os.path.exists(target_folder_path):
            # 尝试使用当前工作目录或用户主目录作为后备
            if not target_folder_path:
                target_folder_path = os.getcwd() if os.path.exists(os.getcwd()) else os.path.expanduser("~")
            # 如果路径仍然不存在，使用文件选择对话框方式
            if not target_folder_path or not os.path.exists(target_folder_path):
                # 即使没有有效路径，也至少添加一些基本操作（通过文件选择对话框）
                default_path = os.path.expanduser("~") if os.path.exists(os.path.expanduser("~")) else os.getcwd()
                self._add_file_operations_with_selection(menu, default_path)
                return
        
        # 创建统一的菜单项创建函数
        def create_action(text, callback):
            action = QAction(text, self)
            action.triggered.connect(callback)
            return action
        
        # 新建文件
        new_file_action = create_action("新建文件", 
            lambda: self._new_file_in_folder(target_folder_path))
        menu.addAction(new_file_action)
        
        # 新建文件夹
        new_folder_action = create_action("新建文件夹", 
            lambda: self._new_folder_in_folder(target_folder_path))
        menu.addAction(new_folder_action)
        
        menu.addSeparator()
        
        # 粘贴（如果有剪贴板内容）
        if self._clipboard_file_path:
            paste_action = create_action("粘贴", 
                lambda: self._paste_file(target_folder_path))
            menu.addAction(paste_action)
            menu.addSeparator()
        
        # 获取文件/文件夹路径
        path = None
        is_file = False
        if item_type in ["file", "folder"] and data and len(data) > 1:
            path = data[1]
            is_file = item_type == "file"
        elif show_all_operations:
            # 如果没有具体路径但需要显示所有操作，使用文件选择对话框
            pass
        
        # 添加复制、剪切、重命名、删除操作
        if path or show_all_operations:
            if path:
                # 有具体路径，直接使用
                # 复制相对路径
                copy_path_action = create_action("复制相对路径", 
                    lambda: self._copy_file_path(path))
                menu.addAction(copy_path_action)
                
                # 复制
                copy_action = create_action("复制" + ("文件" if is_file else "文件夹"), 
                    lambda: self._copy_file(path))
                menu.addAction(copy_action)
                
                # 剪切
                cut_action = create_action("剪切" + ("文件" if is_file else "文件夹"), 
                    lambda: self._cut_file(path))
                menu.addAction(cut_action)
                
                menu.addSeparator()
                
                # 重命名
                rename_action = create_action("重命名" + ("文件" if is_file else "文件夹"), 
                    lambda: self._rename_file(path))
                menu.addAction(rename_action)
                
                # 删除
                delete_action = create_action("删除" + ("文件" if is_file else "文件夹"), 
                    lambda: self._delete_file(path))
                menu.addAction(delete_action)
            else:
                # 没有具体路径，通过文件选择对话框让用户选择
                from PySide6.QtWidgets import QFileDialog
                
                # 复制文件/文件夹
                def copy_with_selection():
                    file_path, _ = QFileDialog.getOpenFileName(
                        self, "选择要复制的文件", target_folder_path, "所有文件 (*.*)"
                    )
                    if file_path:
                        self._copy_file(file_path)
                
                copy_action = create_action("复制文件", copy_with_selection)
                menu.addAction(copy_action)
                
                # 剪切文件/文件夹
                def cut_with_selection():
                    file_path, _ = QFileDialog.getOpenFileName(
                        self, "选择要剪切的文件", target_folder_path, "所有文件 (*.*)"
                    )
                    if file_path:
                        self._cut_file(file_path)
                
                cut_action = create_action("剪切文件", cut_with_selection)
                menu.addAction(cut_action)
                
                menu.addSeparator()
                
                # 重命名文件/文件夹
                def rename_with_selection():
                    file_path, _ = QFileDialog.getOpenFileName(
                        self, "选择要重命名的文件", target_folder_path, "所有文件 (*.*)"
                    )
                    if file_path:
                        self._rename_file(file_path)
                
                rename_action = create_action("重命名文件", rename_with_selection)
                menu.addAction(rename_action)
                
                # 删除文件/文件夹
                def delete_with_selection():
                    file_path, _ = QFileDialog.getOpenFileName(
                        self, "选择要删除的文件", target_folder_path, "所有文件 (*.*)"
                    )
                    if file_path:
                        self._delete_file(file_path)
                
                delete_action = create_action("删除文件", delete_with_selection)
                menu.addAction(delete_action)
            
            menu.addSeparator()
    
    def _show_preview_tree_context_menu(self, pos):
        """显示项目预览树的右键菜单"""
        item = self.preview_tree.itemAt(pos)
        if not item:
            # 如果没有选中项，显示项目根目录菜单
            self._show_project_root_context_menu(pos)
            return
        
        menu = QMenu(self)
        
        # 获取选中项的数据
        data = item.data(0, Qt.UserRole)
        if not data:
            # 如果没有数据，显示通用菜单（使用项目目录作为目标）
            target_folder_path = None
            inferred_path = None
            
            # 获取项目目录
            if hasattr(self, 'project_dir') and self.project_dir and os.path.exists(self.project_dir):
                target_folder_path = self.project_dir
            else:
                # 如果没有项目目录，尝试从父节点获取
                parent = item.parent()
                if parent:
                    parent_data = parent.data(0, Qt.UserRole)
                    if parent_data and len(parent_data) > 1:
                        parent_path = parent_data[1]
                        if os.path.isdir(parent_path):
                            target_folder_path = parent_path
                        elif os.path.isfile(parent_path):
                            target_folder_path = os.path.dirname(parent_path)
            
            # 尝试从节点文本推断路径
            item_text = item.text(0) if item else ""
            if target_folder_path and item_text and item_text.strip():
                # 移除可能的括号内容，如 "文件名 (folder_name)"
                clean_text = item_text.split('(')[0].strip()
                # 尝试作为文件名或文件夹名
                possible_path = os.path.join(target_folder_path, clean_text)
                if os.path.exists(possible_path):
                    inferred_path = possible_path
                else:
                    # 尝试查找匹配的文件或文件夹（只搜索一级目录，避免性能问题）
                    try:
                        for entry in os.listdir(target_folder_path):
                            entry_path = os.path.join(target_folder_path, entry)
                            if entry == clean_text or clean_text in entry:
                                if os.path.exists(entry_path):
                                    inferred_path = entry_path
                                    break
                    except Exception:
                        pass
            
            # 添加通用文件操作菜单（包括新建、粘贴等）
            # 确保至少有一个有效的路径用于文件操作
            if not target_folder_path or not os.path.exists(target_folder_path):
                # 尝试使用当前工作目录或用户主目录作为后备
                target_folder_path = os.getcwd() if os.path.exists(os.getcwd()) else os.path.expanduser("~")
            
            # 如果有推断的路径，传递它以便显示复制、剪切、重命名、删除操作
            if inferred_path and os.path.exists(inferred_path):
                # 判断是文件还是文件夹
                inferred_type = "file" if os.path.isfile(inferred_path) else "folder"
                inferred_data = (inferred_type, inferred_path)
                self._add_common_file_operations(menu, target_folder_path, inferred_type, inferred_data)
            else:
                # 显示所有操作，包括复制、剪切、重命名、删除（通过文件选择对话框）
                self._add_common_file_operations(menu, target_folder_path, item_type=None, data=None, show_all_operations=True)
            
            # 添加刷新操作
            menu.addSeparator()
            refresh_action = QAction("刷新", self)
            refresh_action.triggered.connect(self._refresh_project_preview)
            menu.addAction(refresh_action)
            
            menu.exec(self.preview_tree.mapToGlobal(pos))
            return
        
        item_type = data[0]
        
        # 如果点击的是占位符节点，尝试获取其父节点（文件夹节点）来显示正确的菜单
        if item_type == "placeholder":
            parent = item.parent()
            if parent:
                parent_data = parent.data(0, Qt.UserRole)
                if parent_data and parent_data[0] == "folder":
                    # 使用父节点（文件夹节点）来显示菜单
                    item = parent
                    data = parent_data
                    item_type = data[0]
                else:
                    # 如果父节点不是文件夹，直接返回，不显示菜单
                    return
            else:
                # 如果没有父节点，直接返回
                return
        
        # 调试信息（用于排查问题，可以注释掉）
        # print(f"[DEBUG] 右键菜单 - 节点类型: {item_type}, 数据: {data}, 文本: {item.text(0)}")
        
        # 获取目标文件夹路径（用于通用文件操作）
        target_folder_path = self._get_target_folder_path(item_type, data)
        
        # 首先添加通用文件操作菜单（新建文件、新建文件夹、粘贴等）
        if target_folder_path:
            self._add_common_file_operations(menu, target_folder_path, item_type, data)
        
        # 根据节点类型显示不同的菜单
        menu_added = False  # 标记是否已添加特定菜单项
        
        # 创建统一的菜单项创建函数，支持类型标识
        def create_action(text, callback, menu_type=None, data_dict=None):
            """创建菜单项，支持类型标识"""
            action = QAction(text, self)
            if menu_type:
                # 将菜单类型和额外数据存储到action的data中
                action.setData({"menu_type": menu_type, "item_type": item_type, "data": data_dict or {}})
            action.triggered.connect(callback)
            return action
        
        if item_type == "window_class":
            menu_added = True
            # 窗口类右键菜单
            window_name = data[1]
            
            # 新建窗口
            new_window_action = create_action("新建窗口", 
                self._new_window_class, 
                menu_type="window_class", 
                data_dict={"action": "new_window", "window_name": window_name})
            menu.addAction(new_window_action)
            
            menu.addSeparator()
            
            # 重命名窗口
            rename_action = create_action("重命名窗口", 
                lambda: self._rename_window_class(window_name), 
                menu_type="window_class", 
                data_dict={"action": "rename", "window_name": window_name})
            menu.addAction(rename_action)
            
            # 删除窗口
            delete_action = create_action("删除窗口", 
                self._delete_selected_tree_item, 
                menu_type="window_class", 
                data_dict={"action": "delete", "window_name": window_name})
            menu.addAction(delete_action)
            
            menu.addSeparator()
            
            # 复制窗口
            copy_action = create_action("复制窗口", 
                self._copy_selected_tree_item, 
                menu_type="window_class", 
                data_dict={"action": "copy", "window_name": window_name})
            menu.addAction(copy_action)
            
        elif item_type == "control":
            menu_added = True
            # 控件右键菜单
            window_name = data[1] if len(data) > 1 else None
            control_id = data[2] if len(data) > 2 else data[1]
            
            # 选择控件
            select_action = create_action("选择控件", 
                lambda: self._select_control_from_tree(control_id), 
                menu_type="control", 
                data_dict={"action": "select", "control_id": control_id, "window_name": window_name})
            menu.addAction(select_action)
            
            menu.addSeparator()
            
            # 复制控件
            copy_action = create_action("复制控件", 
                lambda: self._copy_control_from_tree(control_id), 
                menu_type="control", 
                data_dict={"action": "copy", "control_id": control_id, "window_name": window_name})
            menu.addAction(copy_action)
            
            # 删除控件
            delete_action = create_action("删除控件", 
                self._delete_selected_tree_item, 
                menu_type="control", 
                data_dict={"action": "delete", "control_id": control_id, "window_name": window_name})
            menu.addAction(delete_action)
            
            menu.addSeparator()
            
            # 属性编辑
            property_action = create_action("编辑属性", 
                lambda: self._edit_control_property_from_tree(control_id), 
                menu_type="control", 
                data_dict={"action": "edit_property", "control_id": control_id, "window_name": window_name})
            menu.addAction(property_action)
        
        elif item_type == "event":
            menu_added = True
            # 事件右键菜单
            if len(data) >= 4:
                window_name = data[1]
                control_id = data[2]
                event_id = data[3]
                
                # 编辑事件
                edit_action = create_action("编辑事件", 
                    lambda: self._edit_event_from_tree(window_name, control_id, event_id), 
                    menu_type="event", 
                    data_dict={"action": "edit", "window_name": window_name, "control_id": control_id, "event_id": event_id})
                menu.addAction(edit_action)
                
                # 删除事件
                delete_action = create_action("删除事件", 
                    self._delete_selected_tree_item, 
                    menu_type="event", 
                    data_dict={"action": "delete", "window_name": window_name, "control_id": control_id, "event_id": event_id})
                menu.addAction(delete_action)
        
        elif item_type == "file":
            menu_added = True
            # 文件右键菜单（添加特定操作）
            file_path = data[1] if len(data) > 1 else None
            if file_path:
                # 打开文件
                open_action = create_action("打开文件", 
                    lambda: self._open_file(file_path), 
                    menu_type="file", 
                    data_dict={"action": "open", "path": file_path})
                menu.addAction(open_action)
                
                # 使用系统程序打开
                open_with_system_action = create_action("使用系统程序打开", 
                    lambda: self._open_file_with_system(file_path), 
                    menu_type="file", 
                    data_dict={"action": "open_with_system", "path": file_path})
                menu.addAction(open_with_system_action)
                
                menu.addSeparator()
                
                # 导出文件
                export_action = create_action("导出文件", 
                    lambda: self._export_file(file_path), 
                    menu_type="file", 
                    data_dict={"action": "export", "path": file_path})
                menu.addAction(export_action)
                
                menu.addSeparator()
        
        elif item_type == "folder":
            menu_added = True
            # 文件夹右键菜单（通用操作已在上面添加，这里只添加特定操作）
            # 如果需要添加文件夹特定的操作，可以在这里添加
            pass
        
        elif item_type == "project_root":
            menu_added = True
            # 项目根目录右键菜单（添加特定操作）
            project_dir = data[1] if len(data) > 1 else None
            if project_dir:
                # 导入文件
                import_action = create_action("导入文件", 
                    lambda: self._import_file(project_dir), 
                    menu_type="folder", 
                    data_dict={"action": "import", "path": project_dir, "context": "project_root"})
                menu.addAction(import_action)
                menu.addSeparator()
        
        elif item_type == "window_class_root":
            menu_added = True
            # 窗口类根节点右键菜单
            # 新建窗口
            new_window_action = create_action("新建窗口", 
                self._new_window_class, 
                menu_type="window_class", 
                data_dict={"action": "new_window", "context": "window_class_root"})
            menu.addAction(new_window_action)
        
        elif item_type == "canvas_code":
            menu_added = True
            # 画布代码右键菜单
            window_name = data[1] if len(data) > 1 else None
            if window_name:
                # 打开画布代码
                open_action = create_action("打开画布代码", 
                    lambda: self._open_canvas_code(window_name), 
                    menu_type="canvas_code", 
                    data_dict={"action": "open", "window_name": window_name})
                menu.addAction(open_action)
                
                menu.addSeparator()
                
                # 复制代码
                copy_action = create_action("复制代码", 
                    lambda: self._copy_canvas_code(window_name), 
                    menu_type="canvas_code", 
                    data_dict={"action": "copy", "window_name": window_name})
                menu.addAction(copy_action)
        
        elif item_type == "functional_code":
            menu_added = True
            # 功能代码右键菜单
            window_name = data[1] if len(data) > 1 else None
            if window_name:
                # 打开功能代码
                open_action = create_action("打开功能代码", 
                    lambda: self._open_functional_code(window_name), 
                    menu_type="functional_code", 
                    data_dict={"action": "open", "window_name": window_name})
                menu.addAction(open_action)
                
                menu.addSeparator()
                
                # 复制代码
                copy_action = create_action("复制代码", 
                    lambda: self._copy_functional_code(window_name), 
                    menu_type="functional_code", 
                    data_dict={"action": "copy", "window_name": window_name})
                menu.addAction(copy_action)
        
        elif item_type == "window_components":
            menu_added = True
            # 窗口组件右键菜单
            window_name = data[1] if len(data) > 1 else None
            if window_name:
                # 展开所有组件
                expand_action = QAction("展开所有组件", self)
                expand_action.triggered.connect(lambda: self._expand_all_components(item))
                menu.addAction(expand_action)
                
                # 折叠所有组件
                collapse_action = QAction("折叠所有组件", self)
                collapse_action.triggered.connect(lambda: self._collapse_all_components(item))
                menu.addAction(collapse_action)
        
        elif item_type in ["function", "class", "member", "data_structure", "data_type", "ds_member", "data_type_member"]:
            menu_added = True
            # 代码元素右键菜单（函数、类、成员等）
            # 跳转到代码位置
            if len(data) >= 3:
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    jump_action = QAction("跳转到代码", self)
                    # 获取行号（从数据中提取）
                    line = None
                    if len(data) >= 5:
                        # 尝试从第5个元素获取行号（通常是lineno）
                        try:
                            line = int(data[4]) if isinstance(data[4], int) else None
                        except:
                            pass
                    elif len(data) >= 4:
                        # 尝试从第4个元素获取行号
                        try:
                            line = int(data[3]) if isinstance(data[3], int) else None
                        except:
                            pass
                    
                    # 获取文件名（功能代码文件名）
                    function_file_name = data[2] if len(data) > 2 else None
                    if function_file_name:
                        if line:
                            jump_action.triggered.connect(lambda w=window_name, f=function_file_name, l=line: self._jump_to_function_code_location(w, f, l))
                        else:
                            jump_action.triggered.connect(lambda w=window_name, f=function_file_name: self._jump_to_function_code_location(w, f, 1))
                    else:
                        # 如果没有文件名，尝试跳转到功能代码标签页
                        if line:
                            jump_action.triggered.connect(lambda w=window_name, l=line: self._jump_to_function_code_location(w, None, l))
                        else:
                            jump_action.triggered.connect(lambda w=window_name: self._jump_to_function_code_location(w, None, 1))
                    menu.addAction(jump_action)
        
        elif item_type in ["functions_group", "classes_group", "members_group", "data_structures_group", 
                           "ds_members_group", "data_types_group", "data_type_members_group"]:
            menu_added = True
            # 代码元素分组右键菜单
            # 展开/折叠所有
            if item.childCount() > 0:
                if item.isExpanded():
                    collapse_action = QAction("折叠所有", self)
                    collapse_action.triggered.connect(lambda: self._collapse_item_recursive(item))
                    menu.addAction(collapse_action)
                else:
                    expand_action = QAction("展开所有", self)
                    expand_action.triggered.connect(lambda: self._expand_item_recursive(item))
                    menu.addAction(expand_action)
        
        elif item_type == "function_file":
            menu_added = True
            # 功能文件右键菜单
            file_name = data[1] if len(data) > 1 else None
            if file_name:
                # 打开文件
                open_action = QAction("打开文件", self)
                open_action.triggered.connect(lambda: self._open_function_file(file_name))
                menu.addAction(open_action)
        
        elif item_type == "component":
            menu_added = True
            # 组件右键菜单
            component_name = data[1] if len(data) > 1 else None
            if component_name:
                # 查看组件信息
                info_action = QAction("查看组件信息", self)
                info_action.triggered.connect(lambda: self._show_component_info(component_name))
                menu.addAction(info_action)
        
        elif item_type == "placeholder":
            # 占位符节点，不显示菜单
            return
        
        else:
            # 未知类型，添加调试信息
            print(f"[DEBUG] 未知的节点类型: {item_type}, 数据: {data}, 文本: {item.text(0)}")
            # 仍然显示通用菜单
        
        # 只有在添加了特定菜单项时才添加分隔符
        if menu_added:
            menu.addSeparator()
        
        # 通用操作 - 刷新
        refresh_action = QAction("刷新", self)
        refresh_action.triggered.connect(self._refresh_project_preview)
        menu.addAction(refresh_action)
        
        # 显示菜单
        menu.exec(self.preview_tree.mapToGlobal(pos))
    
    def _show_project_root_context_menu(self, pos):
        """显示项目根目录右键菜单（当没有选中项时）"""
        menu = QMenu(self)
        
        # 检查是否有项目目录
        if hasattr(self, 'project_dir') and self.project_dir and os.path.exists(self.project_dir):
            def create_action(text, callback, action_name):
                action = QAction(text, self)
                action.setData({
                    "menu_type": "folder",
                    "item_type": "project_root",
                    "data": {"action": action_name, "path": self.project_dir, "context": "project_root_menu"}
                })
                action.triggered.connect(callback)
                return action
            
            # 新建文件
            new_file_action = create_action("新建文件", lambda: self._new_file_in_folder(self.project_dir), "new_file")
            menu.addAction(new_file_action)
            
            # 新建文件夹
            new_folder_action = create_action("新建文件夹", lambda: self._new_folder_in_folder(self.project_dir), "new_folder")
            menu.addAction(new_folder_action)
            
            menu.addSeparator()
            
            # 粘贴（如果有剪贴板内容）
            if self._clipboard_file_path:
                paste_action = create_action("粘贴", lambda: self._paste_file(self.project_dir), "paste")
                menu.addAction(paste_action)
                menu.addSeparator()
            
            # 导入文件
            import_action = create_action("导入文件", lambda: self._import_file(self.project_dir), "import")
            menu.addAction(import_action)
            
            menu.addSeparator()
        
        # 刷新
        refresh_action = QAction("刷新", self)
        refresh_action.triggered.connect(self._refresh_project_preview)
        menu.addAction(refresh_action)
        
        # 显示菜单
        menu.exec(self.preview_tree.mapToGlobal(pos))

    def _select_control_from_tree(self, control_id):
        """从树中选择控件"""
        self.control_manager.select_control(control_id)
        self.design_canvas.update()
        self._update_code_highlight_from_selection()
    
    def _copy_control_from_tree(self, control_id):
        """从树中复制控件"""
        control = self.control_manager.get_control(control_id)
        if control:
            # 使用现有的复制逻辑
            self._copy_selected_control()
    
    def _edit_control_property_from_tree(self, control_id):
        """从树中编辑控件属性"""
        self.control_manager.select_control(control_id)
        self.design_canvas.update()
        # 切换到属性编辑标签页
        for i in range(self.left_tab_widget.count()):
            if self.left_tab_widget.tabText(i) == "属性编辑":
                self.left_tab_widget.setCurrentIndex(i)
                break
    
    def _edit_event_from_tree(self, window_name, control_id, event_id):
        """从树中编辑事件"""
        # 切换到功能代码标签页
        if hasattr(self.project_manager, 'window_class_manager'):
            window_class = self.project_manager.window_class_manager.get_window_class(window_name)
            if window_class:
                tab_name = f"{window_name}_functions.py"
                self._update_event_function_tab()
                # 切换到对应的标签页
                for i in range(self.code_tab_widget.count()):
                    if self.code_tab_widget.tabText(i) == tab_name:
                        self.code_tab_widget.setCurrentIndex(i)
                        break
    
    def _refresh_project_preview(self):
        """刷新项目预览 - 使用层次化配置结构"""
        try:
            self.preview_tree.clear()
            
            # 添加项目文件系统根节点
            if hasattr(self, 'project_dir') and self.project_dir and os.path.exists(self.project_dir):
                project_root_item = QTreeWidgetItem(self.preview_tree)
                project_root_item.setText(0, f"项目文件: {os.path.basename(self.project_dir)}")
                project_root_item.setData(0, Qt.UserRole, ("project_root", self.project_dir))
                project_root_item.setExpanded(True)
                
                # 构建项目文件系统树
                self._build_project_files_tree(project_root_item, self.project_dir)
            
            # 添加窗口类树结构
            window_class_root_item = QTreeWidgetItem(self.preview_tree)
            window_class_root_item.setText(0, "窗口类")
            window_class_root_item.setData(0, Qt.UserRole, ("window_class_root",))
            window_class_root_item.setExpanded(True)
            
            # 优先使用多窗口类树结构
            if hasattr(self.project_manager, 'window_class_manager'):
                # 将窗口类树添加到窗口类根节点下
                self._build_multi_window_tree(window_class_root_item)
                # 默认不展开，防止卡顿
                # self.preview_tree.expandAll()
                return
            
            # 尝试加载层次化配置
            try:
                from lib.hierarchical_config_loader import HierarchicalConfigLoader
                loader = HierarchicalConfigLoader()
                if loader.load():
                    self._build_hierarchical_tree(loader)
                    # 默认不展开，防止卡顿
                    # self.preview_tree.expandAll()
                    return
            except Exception as e:
                print(f"加载层次化配置失败，使用简化结构: {e}")
            
            # 如果层次化配置加载失败，使用简化结构
            self._build_simple_tree()
            # 默认不展开，防止卡顿
            # self.preview_tree.expandAll()
        except Exception as e:
            import traceback
            traceback.print_exc()
    
    def _build_multi_window_tree(self, parent_item=None):
        """构建多窗口类树形结构 - 按照完整的层次结构"""
        try:
            if not hasattr(self.project_manager, 'window_class_manager'):
                print("警告: 项目管理器没有window_class_manager属性，使用简化结构")
                self._build_simple_tree()
                return
            
            # 如果没有指定父节点，使用树形控件的根节点
            if parent_item is None:
                parent_item = self.preview_tree
            
            window_class_manager = self.project_manager.window_class_manager
            
            # 检查是否有窗口类
            all_names = window_class_manager.get_all_names()
            if not all_names:
                print("警告: 没有找到任何窗口类")
                # 创建一个默认的窗口类节点
                window_item = QTreeWidgetItem(parent_item)
                window_item.setText(0, "窗口类: GeneratedWindow [当前]")
                window_item.setData(0, Qt.UserRole, ("window_class", "GeneratedWindow"))
                window_item.setExpanded(True)  # 默认展开
                
                # 添加空的子节点
                canvas_item = QTreeWidgetItem(window_item)
                canvas_item.setText(0, "画布代码 (GeneratedWindow)")
                canvas_item.setData(0, Qt.UserRole, ("canvas_code", "GeneratedWindow"))
                
                functional_item = QTreeWidgetItem(window_item)
                functional_item.setText(0, "功能代码")
                functional_item.setData(0, Qt.UserRole, ("functional_code", "GeneratedWindow"))
                
                components_item = QTreeWidgetItem(window_item)
                components_item.setText(0, "窗口组件 (0)")
                components_item.setData(0, Qt.UserRole, ("window_components", "GeneratedWindow"))
                return
            
            # 为每个窗口类创建树节点
            for window_name in all_names:
                try:
                    window_class = window_class_manager.get_window_class(window_name)
                    if not window_class:
                        print(f"警告: 无法获取窗口类 '{window_name}'")
                        continue
                    
                    # 创建窗口类根节点
                    window_item = QTreeWidgetItem(parent_item)
                    display_name = f"窗口类: {window_name}"
                    if window_name == window_class_manager.current_window_class_name:
                        display_name += " [当前]"
                    window_item.setText(0, display_name)
                    window_item.setData(0, Qt.UserRole, ("window_class", window_name))
                    # 默认展开第一级，让用户能看到子节点
                    window_item.setExpanded(True)
                    
                    # 1. 画布代码（使用窗口类名称作为标识）
                    canvas_item = QTreeWidgetItem(window_item)
                    canvas_tab_name = f"{window_name}_canvas.py"
                    canvas_item.setText(0, f"画布代码")
                    canvas_item.setData(0, Qt.UserRole, ("canvas_code", window_name))
                    canvas_item.setExpanded(False)
                    
                    # 2. 功能代码
                    functional_item = QTreeWidgetItem(window_item)
                    functional_item.setText(0, "功能实现代码")
                    functional_item.setData(0, Qt.UserRole, ("functional_code", window_name))
                    functional_item.setExpanded(False)
                    
                    # 解析功能代码文件，显示函数和类
                    try:
                        self._build_functional_code_tree(functional_item, window_name, window_class)
                    except Exception as e:
                        print(f"构建功能代码树失败: {e}")
                        import traceback
                        traceback.print_exc()
                    
                    # 3. 窗口组件
                    control_count = len(window_class.control_manager.controls) if hasattr(window_class, 'control_manager') and window_class.control_manager else 0
                    components_item = QTreeWidgetItem(window_item)
                    components_item.setText(0, f"窗口组件 ({control_count})")
                    components_item.setData(0, Qt.UserRole, ("window_components", window_name))
                    components_item.setExpanded(False)
                    
                    # 添加控件
                    if hasattr(window_class, 'control_manager') and window_class.control_manager:
                        controls = sorted(
                            window_class.control_manager.controls.values(),
                            key=lambda c: c.z_order
                        )
                        
                        for control in controls:
                            control_item = QTreeWidgetItem(components_item)
                            control_item.setText(0, f"{control.name} ({control.control_type})")
                            control_item.setData(0, Qt.UserRole, ("control", window_name, control.id))
                            control_item.setExpanded(False)
                            
                            # 添加绑定的事件
                            bound_events = getattr(control, 'events', {})
                            if isinstance(bound_events, dict) and bound_events:
                                events_item = QTreeWidgetItem(control_item)
                                events_item.setText(0, "绑定的事件")
                                events_item.setExpanded(False)
                                for event_id, is_bound in bound_events.items():
                                    if is_bound:
                                        event_item = QTreeWidgetItem(events_item)
                                        event_name = EventManager.get_event_display_name(event_id, control.control_type)
                                        event_item.setText(0, f"✓ {event_name}")
                                        event_item.setData(0, Qt.UserRole, ("event", window_name, control.id, event_id))
                except Exception as e:
                    print(f"构建窗口类 '{window_name}' 的树节点失败: {e}")
                    import traceback
                    traceback.print_exc()
                    continue
        except Exception as e:
            print(f"构建多窗口树失败: {e}")
            import traceback
            traceback.print_exc()
            # 如果出错，尝试使用简化结构
            try:
                self._build_simple_tree()
            except:
                pass
    
    def _build_project_files_tree(self, parent_item, dir_path):
        """
        构建项目文件系统树形结构
        
        Args:
            parent_item: 父节点
            dir_path: 目录路径
        """
        try:
            if not os.path.isdir(dir_path):
                return
            
            # 获取目录中的所有项（文件和文件夹）
            try:
                items = sorted(os.listdir(dir_path), key=lambda x: (not os.path.isdir(os.path.join(dir_path, x)), x.lower()))
            except PermissionError:
                return
            
            for item_name in items:
                item_path = os.path.join(dir_path, item_name)
                
                # 跳过隐藏文件和系统文件
                if item_name.startswith('.'):
                    continue
                
                # 创建树节点
                item = QTreeWidgetItem(parent_item)
                
                if os.path.isdir(item_path):
                    # 文件夹
                    chinese_name = self._file_chinese_names.get(item_path, "")
                    display_name = f"{chinese_name} ({item_name})" if chinese_name else item_name
                    item.setText(0, display_name)
                    item.setData(0, Qt.UserRole, ("folder", item_path))
                    item.setExpanded(False)
                    
                    # 递归构建子目录（延迟加载，只在展开时加载）
                    # 先添加一个占位符，展开时再加载
                    placeholder = QTreeWidgetItem(item)
                    placeholder.setText(0, "...")
                    placeholder.setData(0, Qt.UserRole, ("placeholder",))
                else:
                    # 文件
                    chinese_name = self._file_chinese_names.get(item_path, "")
                    display_name = get_file_display_name(item_path, chinese_name)
                    item.setText(0, display_name)
                    item.setData(0, Qt.UserRole, ("file", item_path))
                    
                    # 根据文件类型设置图标（可选）
                    if is_text_file(item_path):
                        item.setData(0, Qt.UserRole + 1, "text_file")
        except Exception as e:
            print(f"构建项目文件树失败: {e}")
            import traceback
            traceback.print_exc()
    
    def _build_functional_code_tree(self, parent_item, window_name, window_class):
        """构建功能代码树形结构（显示函数和类）"""
        try:
            # 获取事件函数文件内容
            from module.event_function_generator_pyside import EventFunctionGenerator
            
            # 确保control_manager存在
            if not hasattr(window_class, 'control_manager') or not window_class.control_manager:
                # 如果没有control_manager，创建一个空的节点
                empty_item = QTreeWidgetItem(parent_item)
                empty_item.setText(0, "暂无功能代码")
                return
            
            event_gen = EventFunctionGenerator(window_class.control_manager)
            event_gen.set_window_name(window_name)
            
            # 尝试从项目文件中获取事件函数文件内容
            function_file_content = ""
            if hasattr(self.project_manager, 'event_function_file'):
                function_file_content = self.project_manager.event_function_file or ""
            
            # 如果没有内容，生成默认内容（仅用于显示结构）
            if not function_file_content:
                try:
                    function_file_content = event_gen.generate_function_file_content()
                except Exception as e:
                    print(f"生成功能代码内容失败: {e}")
                    function_file_content = ""
            
            # 如果还是没有内容，显示提示
            if not function_file_content:
                empty_item = QTreeWidgetItem(parent_item)
                empty_item.setText(0, "暂无功能代码")
                return
            
            # 解析功能代码文件
            import ast
            try:
                tree = ast.parse(function_file_content)
                
                # 存储文件信息用于跳转
                function_file_name = event_gen.get_function_file_name()
                
                functions = []
                classes = []
                data_structures = []  # 模块级别的数据结构（通常是类型定义等）
                
                # 只提取模块级别的函数、类和数据结构（不是嵌套在类中的）
                for node in tree.body:
                    if isinstance(node, ast.FunctionDef):
                        functions.append(node)
                    elif isinstance(node, ast.ClassDef):
                        # 判断是普通类还是数据结构（这里简化处理，可以根据命名约定判断）
                        # 如果类名以大写字母开头且较短，可能是数据结构
                        if len(node.name) <= 10 and node.name[0].isupper():
                            # 检查类体是否主要是属性定义（简单判断）
                            has_methods = any(isinstance(item, ast.FunctionDef) for item in node.body)
                            if not has_methods or len([item for item in node.body if isinstance(item, ast.FunctionDef)]) <= 2:
                                data_structures.append(node)
                            else:
                                classes.append(node)
                        else:
                            classes.append(node)
                
                # 添加函数节点（直接放在功能代码下）
                if functions:
                    funcs_item = QTreeWidgetItem(parent_item)
                    funcs_item.setText(0, f"函数 ({len(functions)})")
                    funcs_item.setData(0, Qt.UserRole, ("functions_group", window_name, function_file_name))
                    funcs_item.setExpanded(False)
                    for func_node in functions:
                        func_item = QTreeWidgetItem(funcs_item)
                        func_item.setText(0, func_node.name)
                        func_item.setData(0, Qt.UserRole, ("function", window_name, function_file_name, func_node.name, func_node.lineno))
                
                # 添加类节点（直接放在功能代码下）
                if classes:
                    classes_item = QTreeWidgetItem(parent_item)
                    classes_item.setText(0, f"类 ({len(classes)})")
                    classes_item.setData(0, Qt.UserRole, ("classes_group", window_name, function_file_name))
                    classes_item.setExpanded(False)
                    
                    for class_node in classes:
                        class_item = QTreeWidgetItem(classes_item)
                        class_item.setText(0, class_node.name)
                        class_item.setData(0, Qt.UserRole, ("class", window_name, function_file_name, class_node.name, class_node.lineno))
                        class_item.setExpanded(False)
                        
                        # 添加类的成员
                        members = []
                        for item in class_node.body:
                            if isinstance(item, ast.FunctionDef):
                                members.append(("方法", item.name, item.lineno))
                            elif isinstance(item, ast.Assign):
                                for target in item.targets:
                                    if isinstance(target, ast.Name):
                                        members.append(("属性", target.id, item.lineno))
                        
                        if members:
                            members_item = QTreeWidgetItem(class_item)
                            members_item.setText(0, f"成员 ({len(members)})")
                            members_item.setData(0, Qt.UserRole, ("members_group", window_name, function_file_name, class_node.name))
                            members_item.setExpanded(False)
                            for member_type, member_name, member_line in members:
                                member_item = QTreeWidgetItem(members_item)
                                member_item.setText(0, f"{member_name} ({member_type})")
                                member_item.setData(0, Qt.UserRole, ("member", window_name, function_file_name, class_node.name, member_name, member_line))
                        
                        # 添加数据结构（类中定义的嵌套类）
                        class_data_structures = []
                        for item in class_node.body:
                            if isinstance(item, ast.ClassDef):
                                class_data_structures.append(item)
                        
                        if class_data_structures:
                            ds_item = QTreeWidgetItem(class_item)
                            ds_item.setText(0, f"数据结构 ({len(class_data_structures)})")
                            ds_item.setData(0, Qt.UserRole, ("data_structures_group", window_name, function_file_name, class_node.name))
                            ds_item.setExpanded(False)
                            
                            for ds_node in class_data_structures:
                                ds_name_item = QTreeWidgetItem(ds_item)
                                ds_name_item.setText(0, ds_node.name)
                                ds_name_item.setData(0, Qt.UserRole, ("data_structure", window_name, function_file_name, class_node.name, ds_node.name, ds_node.lineno))
                                ds_name_item.setExpanded(False)
                                
                                # 添加数据结构的成员
                                ds_members = []
                                for item in ds_node.body:
                                    if isinstance(item, ast.Assign):
                                        for target in item.targets:
                                            if isinstance(target, ast.Name):
                                                ds_members.append((target.id, item.lineno))
                                    elif isinstance(item, ast.FunctionDef):
                                        ds_members.append((item.name, item.lineno))
                                
                                if ds_members:
                                    ds_members_item = QTreeWidgetItem(ds_name_item)
                                    ds_members_item.setText(0, "成员")
                                    ds_members_item.setData(0, Qt.UserRole, ("ds_members_group", window_name, function_file_name, class_node.name, ds_node.name))
                                    ds_members_item.setExpanded(False)
                                    for member_name, member_line in ds_members:
                                        member_item = QTreeWidgetItem(ds_members_item)
                                        member_item.setText(0, member_name)
                                        member_item.setData(0, Qt.UserRole, ("ds_member", window_name, function_file_name, class_node.name, ds_node.name, member_name, member_line))
                
                # 添加模块级别的数据结构（如果有）
                if data_structures:
                    ds_item = QTreeWidgetItem(parent_item)
                    ds_item.setText(0, f"数据类型 ({len(data_structures)})")
                    ds_item.setData(0, Qt.UserRole, ("data_types_group", window_name, function_file_name))
                    ds_item.setExpanded(False)
                    for ds_node in data_structures:
                        ds_name_item = QTreeWidgetItem(ds_item)
                        ds_name_item.setText(0, ds_node.name)
                        ds_name_item.setData(0, Qt.UserRole, ("data_type", window_name, function_file_name, ds_node.name, ds_node.lineno))
                        ds_name_item.setExpanded(False)
                        
                        # 添加数据类型的成员
                        ds_members = []
                        for item in ds_node.body:
                            if isinstance(item, ast.Assign):
                                for target in item.targets:
                                    if isinstance(target, ast.Name):
                                        ds_members.append((target.id, item.lineno))
                            elif isinstance(item, ast.FunctionDef):
                                ds_members.append((item.name, item.lineno))
                        
                        if ds_members:
                            ds_members_item = QTreeWidgetItem(ds_name_item)
                            ds_members_item.setText(0, "成员")
                            ds_members_item.setData(0, Qt.UserRole, ("data_type_members_group", window_name, function_file_name, ds_node.name))
                            ds_members_item.setExpanded(False)
                            for member_name, member_line in ds_members:
                                member_item = QTreeWidgetItem(ds_members_item)
                                member_item.setText(0, member_name)
                                member_item.setData(0, Qt.UserRole, ("data_type_member", window_name, function_file_name, ds_node.name, member_name, member_line))
                        
            except SyntaxError as e:
                # 如果解析失败，显示错误信息
                print(f"解析功能代码文件失败: {e}")
                error_item = QTreeWidgetItem(parent_item)
                error_item.setText(0, f"解析错误: {str(e)}")
        except Exception as e:
            # 如果出错，显示错误信息
            print(f"构建功能代码树失败: {e}")
            import traceback
            traceback.print_exc()
            error_item = QTreeWidgetItem(parent_item)
            error_item.setText(0, f"构建失败: {str(e)}")
    
    def _build_hierarchical_tree(self, loader):
        """构建层次化树形结构（支持多窗口类）"""
        # 如果有多窗口类管理器，使用它
        if hasattr(self.project_manager, 'window_class_manager'):
            self._build_multi_window_tree()
            return
        
        # 否则使用单个窗口类
        window_name = self.code_generator.window_class_name or "GeneratedWindow"
        
        # 获取窗口类信息
        window_class = loader.get_window_class(window_name)
        if not window_class:
            # 如果找不到，尝试使用默认名称
            window_name = "GeneratedWindow"
            window_class = loader.get_window_class(window_name)
        
        if not window_class:
            self._build_simple_tree()
            return
        
        # 创建窗口类根节点
        window_item = QTreeWidgetItem(self.preview_tree)
        window_item.setText(0, f"窗口类: {window_name}")
        window_item.setData(0, Qt.UserRole, ("window_class", window_name))
        
        # 1. 画布代码
        canvas_code = loader.get_canvas_code(window_name)
        if canvas_code:
            canvas_item = QTreeWidgetItem(window_item)
            canvas_item.setText(0, "画布代码")
            canvas_item.setData(0, Qt.UserRole, ("canvas_code", None))
            
            # 添加画布代码的子项
            if "canvas_width" in canvas_code:
                size_item = QTreeWidgetItem(canvas_item)
                size_item.setText(0, f"尺寸: {canvas_code.get('canvas_width')}x{canvas_code.get('canvas_height')}")
            
            if "code_generation_methods" in canvas_code:
                methods_item = QTreeWidgetItem(canvas_item)
                methods_item.setText(0, "代码生成方法")
                for method_name, method_desc in canvas_code["code_generation_methods"].items():
                    method_item = QTreeWidgetItem(methods_item)
                    method_item.setText(0, f"{method_name}: {method_desc}")
        
        # 2. 功能代码
        functional_code = loader.get_functional_code(window_name)
        if functional_code:
            functional_item = QTreeWidgetItem(window_item)
            functional_item.setText(0, "功能代码")
            functional_item.setData(0, Qt.UserRole, ("functional_code", None))
            
            # 添加功能代码文件
            files = functional_code.get("files", {})
            for file_name, file_info in files.items():
                file_item = QTreeWidgetItem(functional_item)
                file_item.setText(0, file_name)
                file_item.setData(0, Qt.UserRole, ("function_file", file_name))
                
                # 添加函数和类
                functions_and_classes = loader.get_functions_and_classes(window_name)
                if functions_and_classes:
                    fc_item = QTreeWidgetItem(file_item)
                    fc_item.setText(0, "函数和类")
                    
                    # 添加函数
                    functions = functions_and_classes.get("functions", {})
                    if functions:
                        funcs_item = QTreeWidgetItem(fc_item)
                        funcs_item.setText(0, f"函数 ({len(functions)})")
                        # 只显示前10个函数，避免过多
                        for func_name in list(functions.keys())[:10]:
                            func_item = QTreeWidgetItem(funcs_item)
                            func_info = functions[func_name]
                            func_item.setText(0, f"{func_info.get('name', func_name)}")
                            func_item.setData(0, Qt.UserRole, ("function", func_name))
                    
                    # 添加类
                    classes = functions_and_classes.get("classes", {})
                    if classes:
                        classes_item = QTreeWidgetItem(fc_item)
                        classes_item.setText(0, f"类 ({len(classes)})")
                        
                        for class_name, class_info in classes.items():
                            class_item = QTreeWidgetItem(classes_item)
                            class_item.setText(0, class_name)
                            class_item.setData(0, Qt.UserRole, ("class", class_name))
                            
                            # 添加类的成员
                            members = loader.get_class_members(window_name, class_name)
                            if members:
                                members_item = QTreeWidgetItem(class_item)
                                members_item.setText(0, f"成员 ({len(members)})")
                                # 只显示前5个成员
                                for member_name, member_info in list(members.items())[:5]:
                                    member_item = QTreeWidgetItem(members_item)
                                    member_type = member_info.get("type", "unknown")
                                    member_item.setText(0, f"{member_name} ({member_type})")
                            
                            # 添加类的数据结构
                            data_structures = loader.get_data_structures(window_name, class_name)
                            if data_structures:
                                ds_item = QTreeWidgetItem(class_item)
                                ds_item.setText(0, f"数据结构 ({len(data_structures)})")
                                
                                for ds_name, ds_info in data_structures.items():
                                    ds_name_item = QTreeWidgetItem(ds_item)
                                    ds_name_item.setText(0, ds_name)
                                    
                                    # 添加数据结构的成员
                                    ds_members = loader.get_data_structure_members(window_name, class_name, ds_name)
                                    if ds_members:
                                        ds_members_item = QTreeWidgetItem(ds_name_item)
                                        ds_members_item.setText(0, "成员")
                                        for member_key, member_value in ds_members.items():
                                            member_item = QTreeWidgetItem(ds_members_item)
                                            member_item.setText(0, f"{member_key}: {member_value}")
        
        # 3. 窗口组件
        components = loader.get_window_components(window_name)
        if components:
            components_item = QTreeWidgetItem(window_item)
            components_item.setText(0, f"窗口组件 ({len(components)})")
            components_item.setData(0, Qt.UserRole, ("window_components", None))
            
            # 添加所有组件
            for component_name, component_info in components.items():
                component_item = QTreeWidgetItem(components_item)
                display_name = component_info.get("display_name", component_name)
                category = component_info.get("category", "")
                component_item.setText(0, f"{display_name} ({category})")
                component_item.setData(0, Qt.UserRole, ("component", component_name))
                
                # 添加组件绑定的事件
                events = loader.get_component_events(window_name, component_name)
                if events:
                    events_item = QTreeWidgetItem(component_item)
                    events_item.setText(0, f"绑定的事件 ({len(events)})")
                    
                    # 只显示已绑定的事件（如果有实际控件）
                    control = None
                    for c in self.control_manager.controls.values():
                        if c.control_type == component_name:
                            control = c
                            break
                    
                    if control:
                        bound_events = getattr(control, 'events', {})
                        for event_id, event_info in events.items():
                            # 检查是否绑定
                            is_bound = bound_events.get(event_id, False) if isinstance(bound_events, dict) else False
                            if is_bound:
                                event_item = QTreeWidgetItem(events_item)
                                event_name = event_info.get("name", event_id)
                                event_item.setText(0, f"✓ {event_name}")
                                event_item.setData(0, Qt.UserRole, ("event", control.id, event_id))
                    else:
                        # 如果没有实际控件，显示所有可用事件
                        for event_id, event_info in list(events.items())[:5]:
                            event_item = QTreeWidgetItem(events_item)
                            event_name = event_info.get("name", event_id)
                            event_item.setText(0, event_name)
    
    def _build_simple_tree(self):
        """构建简化的树形结构（兼容模式）"""
        # 添加画布代码项
        canvas_item = QTreeWidgetItem(self.preview_tree)
        canvas_item.setText(0, "画布代码")
        canvas_item.setData(0, Qt.UserRole, ("canvas_code", None))
        
        # 添加事件函数文件项
        window_name = self.project_manager.get_window_name()
        function_file_name = f"{window_name}_function.py"
        function_item = QTreeWidgetItem(self.preview_tree)
        function_item.setText(0, function_file_name)
        function_item.setData(0, Qt.UserRole, ("function_file", function_file_name))
        
        # 添加所有控件
        controls = sorted(
            self.control_manager.controls.values(),
            key=lambda c: c.z_order
        )
        
        for control in controls:
            control_item = QTreeWidgetItem(self.preview_tree)
            control_item.setText(0, f"{control.name} ({control.control_type})")
            control_item.setData(0, Qt.UserRole, ("control", control.id))
            
            # 添加绑定的事件
            bound_events = getattr(control, 'events', {})
            if isinstance(bound_events, dict) and bound_events:
                for event_id, is_bound in bound_events.items():
                    if is_bound:
                        event_item = QTreeWidgetItem(control_item)
                        event_name = EventManager.get_event_display_name(event_id, control.control_type)
                        event_item.setText(0, f"  {event_name}")
                        event_item.setData(0, Qt.UserRole, ("event", control.id, event_id))
    
    def _on_preview_item_clicked(self, item: QTreeWidgetItem, column: int):
        """预览项单击事件"""
        try:
            data = item.data(0, Qt.UserRole)
            if not data:
                # 如果没有数据，检查是否有子节点，如果有则展开/折叠
                if item.childCount() > 0:
                    item.setExpanded(not item.isExpanded())
                return
            
            item_type = data[0]
            
            # 处理占位符（延迟加载文件夹内容）
            if item_type == "placeholder":
                parent = item.parent()
                if parent:
                    parent_data = parent.data(0, Qt.UserRole)
                    if parent_data and parent_data[0] == "folder":
                        folder_path = parent_data[1]
                        # 清除占位符并加载实际内容
                        parent.removeChild(item)
                        self._build_project_files_tree(parent, folder_path)
                        parent.setExpanded(True)
                return
            
            # 检查是否有子节点（非叶子节点）
            has_children = item.childCount() > 0
            
            if item_type == "window_class":
                # 窗口类，单击时切换到该窗口类
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    # 切换窗口类
                    if self._switch_to_window_class(window_name):
                        self.status_label.setText(f"已切换到窗口类: {window_name}")
                # 如果有子节点，展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
            elif item_type == "control":
                # 单击控件时选中控件（不跳转，跳转由双击处理）
                window_name = data[1] if len(data) > 1 else None
                control_id = data[2] if len(data) > 2 else data[1]
                if window_name and hasattr(self.project_manager, 'window_class_manager'):
                    window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                    if window_class:
                        if window_name != self.project_manager.window_class_manager.current_window_class_name:
                            self._switch_to_window_class(window_name)
                        self.design_canvas.select_control(control_id)
                else:
                    self.design_canvas.select_control(control_id)
                # 如果有子节点（比如有事件），展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
            elif item_type == "canvas_code":
                # 画布代码，单击时跳转到对应窗口的画布代码
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    # 先切换到该窗口类（确保代码正确）
                    if hasattr(self.project_manager, 'window_class_manager'):
                        current_name = self.project_manager.window_class_manager.current_window_class_name
                        if current_name != window_name:
                            self._switch_to_window_class(window_name)
                    
                    canvas_tab_name = f"{window_name}_canvas.py"
                    # 查找对应的标签页
                    tab_found = False
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == canvas_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            tab_found = True
                            break
                    
                    # 如果标签页不存在，创建它
                    if not tab_found:
                        if hasattr(self.project_manager, 'window_class_manager'):
                            window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                            if window_class:
                                self._update_canvas_code_tab(window_name, window_class)
                # 如果有子节点，展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
            elif item_type == "functional_code":
                # 功能代码，单击时跳转到事件实现文件
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    # 先切换到该窗口类（确保代码正确）
                    if hasattr(self.project_manager, 'window_class_manager'):
                        current_name = self.project_manager.window_class_manager.current_window_class_name
                        if current_name != window_name:
                            self._switch_to_window_class(window_name)
                    
                    function_tab_name = f"{window_name}_function.py"
                    # 查找对应的标签页
                    tab_found = False
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == function_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            tab_found = True
                            break
                    
                    # 如果标签页不存在，创建它
                    if not tab_found:
                        self._update_event_function_tab()
                # 如果有子节点，展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
            elif item_type in ["window_components", 
                              "functions_group", "classes_group", "members_group", 
                              "data_structures_group", "ds_members_group", "data_types_group",
                              "data_type_members_group"]:
                # 这些是中间节点，如果有子节点，展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
            elif item_type in ["class", "component", "data_structure", "data_type"]:
                # 这些节点可能有子节点，如果有则展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
            else:
                # 其他节点（如函数、成员、事件等），如果有子节点，展开/折叠
                if has_children:
                    item.setExpanded(not item.isExpanded())
        except Exception as e:
            print(f"预览项单击事件处理失败: {e}")
            import traceback
            traceback.print_exc()
    
    def _on_preview_item_double_clicked(self, item: QTreeWidgetItem, column: int):
        """预览项双击事件"""
        try:
            data = item.data(0, Qt.UserRole)
            if not data:
                return
            
            item_type = data[0]
            
            # 处理文件双击
            if item_type == "file":
                file_path = data[1] if len(data) > 1 else None
                if file_path:
                    self._open_file(file_path)
                return
            
            # 处理文件夹双击（展开/折叠）
            if item_type == "folder":
                item.setExpanded(not item.isExpanded())
                # 如果展开且只有占位符，加载实际内容
                if item.isExpanded() and item.childCount() == 1:
                    child = item.child(0)
                    child_data = child.data(0, Qt.UserRole)
                    if child_data and child_data[0] == "placeholder":
                        folder_path = data[1]
                        item.removeChild(child)
                        self._build_project_files_tree(item, folder_path)
                return
            
            if item_type == "window_class":
                # 窗口类，双击时切换并展开/折叠
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    # 切换窗口类
                    if self._switch_to_window_class(window_name):
                        self.status_label.setText(f"已切换到窗口类: {window_name}")
                    # 展开/折叠
                    item.setExpanded(not item.isExpanded())
                else:
                    # 如果没有窗口名，只是展开/折叠
                    item.setExpanded(not item.isExpanded())
            elif item_type == "canvas_code":
                # 切换到画布代码标签页
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    # 先切换到该窗口类（确保代码正确）
                    if hasattr(self.project_manager, 'window_class_manager'):
                        current_name = self.project_manager.window_class_manager.current_window_class_name
                        if current_name != window_name:
                            self._switch_to_window_class(window_name)
                    
                    canvas_tab_name = f"{window_name}_canvas.py"
                    # 查找对应的标签页
                    tab_found = False
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == canvas_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            tab_found = True
                            break
                    
                    # 如果标签页不存在，创建它
                    if not tab_found:
                        if hasattr(self.project_manager, 'window_class_manager'):
                            window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                            if window_class:
                                self._update_canvas_code_tab(window_name, window_class)
                else:
                    self.code_tab_widget.setCurrentIndex(0)
            elif item_type == "functional_code":
                # 功能代码，双击时跳转到事件实现文件
                window_name = data[1] if len(data) > 1 else None
                if window_name:
                    # 先切换到该窗口类（确保代码正确）
                    if hasattr(self.project_manager, 'window_class_manager'):
                        current_name = self.project_manager.window_class_manager.current_window_class_name
                        if current_name != window_name:
                            self._switch_to_window_class(window_name)
                    
                    function_tab_name = f"{window_name}_function.py"
                    # 查找对应的标签页
                    tab_found = False
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == function_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            tab_found = True
                            break
                    
                    # 如果标签页不存在，创建它
                    if not tab_found:
                        self._update_event_function_tab()
                else:
                    # 如果没有窗口名，只是展开/折叠
                    item.setExpanded(not item.isExpanded())
            elif item_type == "function_file":
                # 切换到事件函数文件标签页
                self._update_event_function_tab()
            elif item_type == "function":
                # 跳转到函数定义
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                function_name = data[3] if len(data) > 3 else None
                function_line = data[4] if len(data) > 4 else None
                if function_name:
                    self._jump_to_code_location(function_file_name, function_name, function_line, "function")
            elif item_type == "class":
                # 类，展开/折叠或跳转
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                class_name = data[3] if len(data) > 3 else None
                class_line = data[4] if len(data) > 4 else None
                if class_name and class_line:
                    # 双击时跳转到类定义
                    self._jump_to_code_location(function_file_name, class_name, class_line, "class")
                else:
                    # 如果没有行号，只是展开/折叠
                    item.setExpanded(not item.isExpanded())
            elif item_type == "member":
                # 跳转到类成员定义
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                class_name = data[3] if len(data) > 3 else None
                member_name = data[4] if len(data) > 4 else None
                member_line = data[5] if len(data) > 5 else None
                if member_name and member_line:
                    self._jump_to_code_location(function_file_name, member_name, member_line, "member", class_name)
            elif item_type == "data_structure":
                # 数据结构，展开/折叠或跳转
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                class_name = data[3] if len(data) > 3 else None
                ds_name = data[4] if len(data) > 4 else None
                ds_line = data[5] if len(data) > 5 else None
                if ds_name and ds_line:
                    self._jump_to_code_location(function_file_name, ds_name, ds_line, "data_structure", class_name)
                else:
                    item.setExpanded(not item.isExpanded())
            elif item_type == "ds_member":
                # 跳转到数据结构成员定义
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                class_name = data[3] if len(data) > 3 else None
                ds_name = data[4] if len(data) > 4 else None
                member_name = data[5] if len(data) > 5 else None
                member_line = data[6] if len(data) > 6 else None
                if member_name and member_line:
                    self._jump_to_code_location(function_file_name, member_name, member_line, "ds_member", class_name, ds_name)
            elif item_type == "data_type":
                # 数据类型，展开/折叠或跳转
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                type_name = data[3] if len(data) > 3 else None
                type_line = data[4] if len(data) > 4 else None
                if type_name and type_line:
                    self._jump_to_code_location(function_file_name, type_name, type_line, "data_type")
                else:
                    item.setExpanded(not item.isExpanded())
            elif item_type == "data_type_member":
                # 跳转到数据类型成员定义
                window_name = data[1] if len(data) > 1 else None
                function_file_name = data[2] if len(data) > 2 else None
                type_name = data[3] if len(data) > 3 else None
                member_name = data[4] if len(data) > 4 else None
                member_line = data[5] if len(data) > 5 else None
                if member_name and member_line:
                    self._jump_to_code_location(function_file_name, member_name, member_line, "data_type_member", None, type_name)
            elif item_type == "window_components":
                # 窗口组件，展开/折叠
                item.setExpanded(not item.isExpanded())
            elif item_type == "component":
                # 组件，展开/折叠
                item.setExpanded(not item.isExpanded())
            elif item_type == "control":
                # 选中控件并跳转到代码
                window_name = data[1] if len(data) > 1 else None
                control_id = data[2] if len(data) > 2 else data[1]
                # 选中控件
                if window_name and hasattr(self.project_manager, 'window_class_manager'):
                    window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                    if window_class:
                        # 切换到对应的窗口类
                        if window_name != self.project_manager.window_class_manager.current_window_class_name:
                            self._switch_to_window_class(window_name)
                        # 选中控件
                        self.design_canvas.select_control(control_id)
                else:
                    self.design_canvas.select_control(control_id)
                # 跳转到画布代码中的控件定义
                self._jump_to_control_in_canvas_code(window_name, control_id)
            elif item_type == "event":
                # 跳转到事件函数
                window_name = data[1] if len(data) > 1 else None
                control_id = data[2] if len(data) > 2 else data[1]
                event_id = data[3] if len(data) > 3 else data[2]
                self._jump_to_event_function(control_id, event_id, window_name)
        except Exception:
            pass
    
    def _delete_selected_tree_item(self):
        """删除树形框中选中的项"""
        current_item = self.preview_tree.currentItem()
        if not current_item:
            return
        
        data = current_item.data(0, Qt.UserRole)
        if not data:
            return
        
        item_type = data[0]
        
        if item_type == "window_class":
            # 删除窗口类
            window_name = data[1]
            if hasattr(self.project_manager, 'window_class_manager'):
                if len(self.project_manager.window_class_manager.get_all_names()) <= 1:
                    QMessageBox.warning(self, "警告", "至少需要保留一个窗口类")
                    return
                
                reply = QMessageBox.question(
                    self, "确认删除",
                    f"确定要删除窗口类 '{window_name}' 吗？\n此操作不可撤销！",
                    QMessageBox.Yes | QMessageBox.No
                )
                
                if reply == QMessageBox.Yes:
                    if self.project_manager.window_class_manager.remove_window_class(window_name):
                        self._refresh_project_preview()
                        self.project_manager.set_modified(True)
        
        elif item_type == "control":
            # 删除控件
            window_name = data[1] if len(data) > 1 else None
            control_id = data[2] if len(data) > 2 else data[1]
            
            if window_name and hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class:
                    if window_class.control_manager.delete_control(control_id):
                        self._refresh_project_preview()
                        self.project_manager.set_modified(True)
                        self.design_canvas.update()
            else:
                # 向后兼容：使用当前control_manager
                if self.control_manager.delete_control(control_id):
                    self._refresh_project_preview()
                    self.project_manager.set_modified(True)
                    self.design_canvas.update()
        
        elif item_type == "event":
            # 删除事件绑定
            if len(data) >= 4:
                window_name = data[1]
                control_id = data[2]
                event_id = data[3]
                if hasattr(self.project_manager, 'window_class_manager'):
                    window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                    if window_class:
                        control = window_class.control_manager.get_control(control_id)
                        if control and hasattr(control, 'events'):
                            control.events[event_id] = False
                            self._refresh_project_preview()
                            self.project_manager.set_modified(True)
    
    def _copy_selected_tree_item(self):
        """复制树形框中选中的项"""
        current_item = self.preview_tree.currentItem()
        if not current_item:
            return
        
        data = current_item.data(0, Qt.UserRole)
        if not data:
            return
        
        item_type = data[0]
        
        if item_type == "window_class":
            # 复制窗口类
            source_name = data[1]
            if hasattr(self.project_manager, 'window_class_manager'):
                target_name = self.project_manager.window_class_manager._generate_unique_name(source_name)
                window_class = self.project_manager.window_class_manager.copy_window_class(source_name, target_name)
                if window_class:
                    self._refresh_project_preview()
                    self.project_manager.set_modified(True)
                    QMessageBox.information(self, "成功", f"已复制窗口类为: {target_name}")
        
        elif item_type == "control":
            # 复制控件
            window_name = data[1] if len(data) > 1 else None
            control_id = data[2] if len(data) > 2 else data[1]
            
            if window_name and hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class:
                    # 切换到该窗口类
                    self.project_manager.window_class_manager.set_current_window_class(window_name)
                    # 选中控件
                    window_class.control_manager.set_selection([control_id])
                    # 复制控件
                    mapping = window_class.control_manager.copy_selected_controls()
                    if mapping:
                        self._refresh_project_preview()
                        self.project_manager.set_modified(True)
                        self.design_canvas.update()
            else:
                # 向后兼容：使用当前control_manager
                self.control_manager.set_selection([control_id])
                mapping = self.control_manager.copy_selected_controls()
                if mapping:
                    self._refresh_project_preview()
                    self.project_manager.set_modified(True)
                    self.design_canvas.update()
    
    def _jump_to_event_function(self, control_id: str, event_id: str, window_name: str = None):
        """跳转到事件函数"""
        try:
            # 确保事件函数文件标签页已打开
            self._update_event_function_tab()
            
            # 获取窗口名称
            if not window_name:
                window_name = self.project_manager.get_window_name()
            
            # 获取事件函数名
            function_name = EventManager.get_event_function_name(control_id, event_id)
            
            # 查找对应的编辑器
            tab_name = f"{window_name}_function.py"
            if tab_name in self._code_editors:
                editor = self._code_editors[tab_name]
                
                # 查找函数定义
                code_text = editor.toPlainText()
                lines = code_text.splitlines()
                
                for i, line in enumerate(lines):
                    if f"def {function_name}(" in line:
                        # 跳转到该行
                        if isinstance(editor, MonacoEditorWidget):
                            # MonacoEditorWidget: 使用setCursorPosition方法
                            editor.setCursorPosition(i + 1, 1)  # Monaco行号从1开始
                        else:
                            # QPlainTextEdit: 使用document方法
                            block = editor.document().findBlockByNumber(i)
                            cursor = editor.textCursor()
                            cursor.setPosition(block.position())
                            editor.setTextCursor(cursor)
                            editor.centerCursor()
                        break
        except Exception:
            pass
    
    def _jump_to_code_location(self, file_name: str, target_name: str, target_line: int = None, 
                               target_type: str = "function", class_name: str = None, 
                               data_type_name: str = None):
        """
        跳转到代码中的指定位置
        
        Args:
            file_name: 文件名
            target_name: 目标名称（函数名、类名、成员名等）
            target_line: 目标行号（如果已知）
            target_type: 目标类型（"function", "class", "member", "data_structure", "ds_member", "data_type", "data_type_member"）
            class_name: 类名（如果是类成员）
            data_type_name: 数据类型名（如果是数据类型成员）
        """
        try:
            # 切换到对应的标签页
            if file_name in self._code_editors:
                # 找到对应的标签页索引
                tab_index = -1
                for i in range(self.code_tab_widget.count()):
                    if self.code_tab_widget.tabText(i) == file_name:
                        tab_index = i
                        break
                
                if tab_index >= 0:
                    self.code_tab_widget.setCurrentIndex(tab_index)
                    editor = self._code_editors[file_name]
                    
                    # 如果已知行号，直接跳转
                    if target_line:
                        if isinstance(editor, MonacoEditorWidget):
                            editor.setCursorPosition(target_line, 1)
                        else:
                            block = editor.document().findBlockByNumber(target_line - 1)
                            cursor = editor.textCursor()
                            cursor.setPosition(block.position())
                            editor.setTextCursor(cursor)
                            editor.centerCursor()
                        return
                    
                    # 否则搜索目标
                    code_text = editor.toPlainText()
                    lines = code_text.splitlines()
                    
                    search_patterns = []
                    if target_type == "function":
                        search_patterns = [f"def {target_name}("]
                    elif target_type == "class":
                        search_patterns = [f"class {target_name}"]
                    elif target_type == "member":
                        # 类成员：可能是方法或属性
                        search_patterns = [f"def {target_name}(", f"{target_name} =", f"self.{target_name} ="]
                    elif target_type == "data_structure" or target_type == "data_type":
                        search_patterns = [f"class {target_name}"]
                    elif target_type == "ds_member" or target_type == "data_type_member":
                        search_patterns = [f"{target_name} =", f"def {target_name}("]
                    
                    for i, line in enumerate(lines):
                        for pattern in search_patterns:
                            if pattern in line:
                                # 检查是否在正确的类中（如果是类成员）
                                if class_name and target_type in ["member", "ds_member"]:
                                    # 向上查找类定义
                                    found_in_class = False
                                    for j in range(i, -1, -1):
                                        if f"class {class_name}" in lines[j]:
                                            found_in_class = True
                                            break
                                        elif j < i and "class " in lines[j]:
                                            # 遇到了其他类，说明不在目标类中
                                            break
                                    if not found_in_class:
                                        continue
                                
                                # 跳转到该行
                                if isinstance(editor, MonacoEditorWidget):
                                    editor.setCursorPosition(i + 1, 1)
                                else:
                                    block = editor.document().findBlockByNumber(i)
                                    cursor = editor.textCursor()
                                    cursor.setPosition(block.position())
                                    editor.setTextCursor(cursor)
                                    editor.centerCursor()
                                return
        except Exception as e:
            print(f"跳转到代码位置失败: {e}")
            import traceback
            traceback.print_exc()
    
    def _jump_to_function_code_location(self, window_name: str, function_file_name: str = None, line: int = 1):
        """
        跳转到功能代码的指定位置
        
        Args:
            window_name: 窗口名称
            function_file_name: 功能代码文件名（可选）
            line: 行号
        """
        try:
            # 确保功能代码标签页已打开
            if hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class:
                    self._switch_to_window_class(window_name)
                    self._update_event_function_tab()
            
            # 查找功能代码标签页
            if function_file_name:
                tab_name = function_file_name
            else:
                tab_name = f"{window_name}_function.py"
            
            # 查找标签页
            for i in range(self.code_tab_widget.count()):
                if self.code_tab_widget.tabText(i) == tab_name:
                    self.code_tab_widget.setCurrentIndex(i)
                    editor = self.code_tab_widget.widget(i)
                    
                    # 跳转到指定行
                    if isinstance(editor, MonacoEditorWidget):
                        editor.setCursorPosition(line, 1)
                    else:
                        block = editor.document().findBlockByNumber(line - 1)
                        cursor = editor.textCursor()
                        cursor.setPosition(block.position())
                        editor.setTextCursor(cursor)
                        editor.centerCursor()
                    return
        except Exception as e:
            print(f"跳转到功能代码位置失败: {e}")
            import traceback
            traceback.print_exc()
    
    def _jump_to_control_in_canvas_code(self, window_name: str, control_id: str):
        """
        跳转到画布代码中的控件定义
        
        Args:
            window_name: 窗口名称
            control_id: 控件ID
        """
        try:
            # 获取控件对象
            control = None
            if window_name and hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class:
                    control = window_class.control_manager.get_control(control_id)
            else:
                control = self.control_manager.get_control(control_id)
            
            if not control:
                return
            
            # 切换到画布代码标签页
            canvas_tab_name = f"{window_name}_canvas.py" if window_name else "canvas.py"
            
            # 查找对应的标签页
            tab_index = -1
            for i in range(self.code_tab_widget.count()):
                if self.code_tab_widget.tabText(i) == canvas_tab_name:
                    tab_index = i
                    break
            
            if tab_index >= 0:
                self.code_tab_widget.setCurrentIndex(tab_index)
                editor = self._code_editors.get(canvas_tab_name)
                
                if editor:
                    # 搜索控件定义（通常是 self.控件名 = 控件类型(...)）
                    code_text = editor.toPlainText()
                    lines = code_text.splitlines()
                    
                    # 搜索模式：self.控件名 = 或 控件名 =
                    control_name = control.name
                    search_patterns = [
                        f"self.{control_name} =",
                        f"{control_name} =",
                        f"self.{control_id} =",
                        f"{control_id} ="
                    ]
                    
                    for i, line in enumerate(lines):
                        for pattern in search_patterns:
                            if pattern in line:
                                # 跳转到该行
                                if isinstance(editor, MonacoEditorWidget):
                                    editor.setCursorPosition(i + 1, 1)
                                else:
                                    block = editor.document().findBlockByNumber(i)
                                    cursor = editor.textCursor()
                                    cursor.setPosition(block.position())
                                    editor.setTextCursor(cursor)
                                    editor.centerCursor()
                                return
        except Exception as e:
            print(f"跳转到画布代码失败: {e}")
            import traceback
            traceback.print_exc()
    
    def _create_output_panel(self):
        """创建底部输出面板（终端输出和软件输出）"""
        # 创建输出面板容器
        self.output_panel = QWidget()
        self.output_panel.setVisible(False)  # 默认隐藏
        output_layout = QVBoxLayout(self.output_panel)
        output_layout.setContentsMargins(0, 0, 0, 0)
        output_layout.setSpacing(0)
        
        # 创建选择夹
        self.output_tab_widget = QTabWidget()
        self.output_tab_widget.setTabsClosable(False)
        
        # 创建终端输出标签页
        self.terminal_output = QPlainTextEdit()
        self.terminal_output.setReadOnly(True)
        self.terminal_output.setFont(QFont("Consolas", 9))
        self.terminal_output.setStyleSheet("""
            QPlainTextEdit {
                background-color: #1e1e1e;
                color: #d4d4d4;
                border: none;
            }
        """)
        self.output_tab_widget.addTab(self.terminal_output, "终端输出")
        
        # 创建软件输出标签页
        self.software_output = QPlainTextEdit()
        self.software_output.setReadOnly(True)
        self.software_output.setFont(QFont("Consolas", 9))
        self.software_output.setStyleSheet("""
            QPlainTextEdit {
                background-color: #1e1e1e;
                color: #d4d4d4;
                border: none;
            }
        """)
        self.output_tab_widget.addTab(self.software_output, "软件输出")
        
        # 创建提示标签页
        self.tip_output = QPlainTextEdit()
        self.tip_output.setReadOnly(True)
        self.tip_output.setFont(QFont("Consolas", 9))
        self.tip_output.setStyleSheet("""
            QPlainTextEdit {
                background-color: #1e1e1e;
                color: #d4d4d4;
                border: none;
            }
        """)
        self.output_tab_widget.addTab(self.tip_output, "提示")
        
        output_layout.addWidget(self.output_tab_widget)
        
        # 设置输出面板的初始高度
        self.output_panel.setMaximumHeight(300)
        self.output_panel.setMinimumHeight(100)
        
        # 将输出面板添加到主布局
        main_layout = self.centralWidget().layout()
        main_layout.addWidget(self.output_panel)
        
        # 注意：终端输出按钮已在左侧按钮栏中创建，不需要在这里创建
        
        # 保存原始print函数
        self._original_print = print
        self._print_buffer = []
    
    def _redirect_print_output(self):
        """重定向print输出到软件输出窗口"""
        class PrintRedirector:
            def __init__(self, output_widget):
                self.output_widget = output_widget
            
            def write(self, text):
                if text and text.strip():  # 忽略空行
                    # 使用QTimer延迟更新，避免阻塞
                    from PySide6.QtCore import QTimer
                    def append_text():
                        self.output_widget.appendPlainText(text.rstrip())
                        # 自动滚动到底部
                        cursor = self.output_widget.textCursor()
                        cursor.movePosition(QTextCursor.End)
                        self.output_widget.setTextCursor(cursor)
                    QTimer.singleShot(0, append_text)
                return len(text)
            
            def flush(self):
                pass
        
        # 重定向sys.stdout
        import sys
        self._stdout_redirector = PrintRedirector(self.software_output)
        sys.stdout = self._stdout_redirector
    
    def _toggle_output_panel(self):
        """切换输出面板的显示/隐藏"""
        is_visible = self.output_panel.isVisible()
        self.output_panel.setVisible(not is_visible)
        self.btn_toggle_output.setChecked(not is_visible)
    
    def _append_terminal_output(self, text: str):
        """添加终端输出（线程安全）"""
        self.terminal_output.appendPlainText(text)
        # 自动滚动到底部
        cursor = self.terminal_output.textCursor()
        cursor.movePosition(QTextCursor.End)
        self.terminal_output.setTextCursor(cursor)
    
    def _append_tip_output(self, text: str):
        """添加提示输出"""
        self.tip_output.clear()
        self.tip_output.setPlainText(text)
        # 自动滚动到顶部
        cursor = self.tip_output.textCursor()
        cursor.movePosition(QTextCursor.Start)
        self.tip_output.setTextCursor(cursor)
        # 切换到提示标签页
        if hasattr(self, 'output_tab_widget') and self.output_panel.isVisible():
            for i in range(self.output_tab_widget.count()):
                if self.output_tab_widget.tabText(i) == "提示":
                    self.output_tab_widget.setCurrentIndex(i)
                    break
    
    def _on_function_clicked(self, library_name: str, function_name: str, docstring: str):
        """
        处理函数库中函数被点击的事件
        
        Args:
            library_name: 函数库名称
            function_name: 函数名称
            docstring: 函数注释
        """
        # 显示函数注释到提示标签页
        tip_text = f"函数: {function_name}\n"
        tip_text += f"所属库: {library_name}\n"
        tip_text += "=" * 60 + "\n"
        tip_text += "注释:\n"
        tip_text += docstring if docstring and docstring != "无" else "无"
        
        self._append_tip_output(tip_text)
    
    def _on_process_finished(self, returncode: int):
        """进程结束时的处理"""
        self._append_terminal_output("=" * 60)
        self._append_terminal_output(f"[进程结束] 返回码: {returncode}")
        if returncode != 0:
            self._append_terminal_output(f"[警告] 进程异常退出，返回码: {returncode}")
    
    def _on_event_binding_changed(self, control_id: str, event_id: str, is_bound: bool):
        """
        事件绑定变化处理函数（画布代码绑定函数的响应处理）
        当用户在属性编辑器中绑定或取消绑定事件时，此函数会被调用
        
        此函数负责：
        1. 更新左侧代码视图，重新生成包含事件绑定代码的完整代码
        2. 如果绑定事件，生成或更新事件函数文件（{窗口名称}_function.py）
        3. 将事件函数文件内容保存到项目管理器
        4. 更新事件函数文件标签页，显示事件处理函数代码
        5. 刷新项目预览窗口
        
        Args:
            control_id: 控件ID（如"button1"、"label1"等）
            event_id: 事件ID（如"clicked"、"textChanged"等）
            is_bound: 是否绑定（True表示绑定，False表示取消绑定）
        """
        try:
            # ========== 获取控件对象 ==========
            # 从控件管理器中获取对应的控件对象
            control = self.control_manager.get_control(control_id)
            # 如果控件不存在，直接返回（不执行任何操作）
            if not control:
                return
            
            # ========== 更新代码视图 ==========
            # 刷新左侧代码编辑器，重新生成包含最新事件绑定代码的完整代码
            # 这会调用代码生成器的_generate_signal_connections方法生成信号连接代码
            self._refresh_code_view_left()
            
            # ========== 如果绑定事件，生成或更新事件函数文件 ==========
            # 只有当事件被绑定（is_bound为True）时才生成事件函数文件
            # 取消绑定时不需要生成事件函数文件（但会更新代码视图中的信号连接代码）
            if is_bound:
                # ========== 创建事件函数生成器 ==========
                # 导入事件函数生成器模块
                from .event_function_generator_pyside import EventFunctionGenerator
                # 创建事件函数生成器实例，传入控件管理器以便读取所有控件的绑定事件
                event_gen = EventFunctionGenerator(self.control_manager)
                
                # ========== 获取窗口名称 ==========
                # 优先从窗口类管理器获取当前窗口类名称
                window_name = None
                if hasattr(self.project_manager, 'window_class_manager'):
                    window_name = self.project_manager.window_class_manager.current_window_class_name
                if not window_name:
                    window_name = self.project_manager.get_window_name() if hasattr(self.project_manager, 'get_window_name') else "GeneratedWindow"
                
                # 设置事件函数生成器的窗口名称
                event_gen.set_window_name(window_name)
                
                # ========== 生成事件函数文件内容 ==========
                # 调用事件函数生成器生成完整的事件函数文件内容
                # 生成的内容包括所有已绑定控件的事件处理函数定义（带示例代码）
                function_content = event_gen.generate_function_file_content()
                
                # ========== 保存到窗口类（优先） ==========
                if hasattr(self.project_manager, 'window_class_manager'):
                    current_window = self.project_manager.window_class_manager.get_window_class(window_name)
                    if current_window:
                        current_window.event_function_file = function_content
                
                # ========== 保存到项目管理器（向后兼容） ==========
                # 将生成的事件函数文件内容保存到项目管理器
                # 项目管理器会在保存项目时将此文件内容一并保存
                if hasattr(self.project_manager, 'set_event_function_file'):
                    self.project_manager.set_event_function_file(function_content)
                # 确保项目管理器中的窗口名称是最新的
                if hasattr(self.project_manager, 'set_window_name'):
                    self.project_manager.set_window_name(window_name)
                
                # ========== 更新状态栏 ==========
                # 在状态栏显示事件绑定成功的信息
                self.status_label.setText(f"已绑定事件: {EventManager.get_event_display_name(event_id, control.control_type)}")
                
                # ========== 更新事件函数文件标签 ==========
                # 更新或创建事件函数文件的标签页，显示事件处理函数代码
                # 用户可以在标签页中编辑事件处理函数的实现
                self._update_event_function_tab()
                
                # ========== 刷新项目预览 ==========
                # 刷新项目预览窗口，使预览窗口反映最新的事件绑定状态
                self._refresh_project_preview()
        except Exception as e:
            # ========== 处理异常 ==========
            # 如果事件绑定过程中发生任何异常，打印详细的错误堆栈信息（用于调试）
            import traceback
            traceback.print_exc()
            # 在状态栏显示错误信息，提示用户事件绑定失败
            self.status_label.setText(f"事件绑定失败: {str(e)}")
    
    def closeEvent(self, event):
        """窗口关闭事件，保存窗口状态和设置"""
        from PySide6.QtCore import QSettings
        settings = QSettings("VisualProgramming", "Settings")
        
        # 检查是否需要保存窗口状态
        remember_window_state = settings.value("basic/remember_window_state", True, type=bool)
        if remember_window_state:
            # 保存窗口几何信息（位置和大小）
            geometry = self.saveGeometry()
            if geometry:
                settings.setValue("window/geometry", geometry.toBase64().data().decode())
            
            # 保存窗口状态（最大化、最小化等）
            window_state = self.saveState()
            if window_state:
                settings.setValue("window/state", window_state.toBase64().data().decode())
            
            # 保存主分割器状态
            if hasattr(self, 'main_splitter'):
                try:
                    splitter_state = self.main_splitter.saveState()
                    if splitter_state:
                        settings.setValue("window/main_splitter_state", splitter_state.toBase64().data().decode())
                except Exception:
                    pass
            
            # 保存中间分割器状态
            if hasattr(self, 'center_splitter'):
                try:
                    splitter_state = self.center_splitter.saveState()
                    if splitter_state:
                        settings.setValue("window/center_splitter_state", splitter_state.toBase64().data().decode())
                except Exception:
                    pass
            
            # 保存左侧面板宽度
            if hasattr(self, '_left_tab_widget_saved_width'):
                settings.setValue("window/left_panel_width", self._left_tab_widget_saved_width)
        
        # 同步设置到磁盘
        settings.sync()
        
        # 检查是否需要确认关闭
        confirm_on_close = settings.value("basic/confirm_on_close", False, type=bool)
        if confirm_on_close:
            from PySide6.QtWidgets import QMessageBox
            reply = QMessageBox.question(
                self,
                "确认关闭",
                "确定要退出程序吗？",
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if reply == QMessageBox.No:
                event.ignore()
                return
        
        # 调用父类的关闭事件处理
        super().closeEvent(event)
    
    # ========== 文件操作方法 ==========
    
    def _open_file(self, file_path: str):
        """打开文件（文本文件用代码编辑器，其他用系统默认程序）"""
        try:
            if not os.path.exists(file_path):
                QMessageBox.warning(self, "错误", f"文件不存在: {file_path}")
                return
            
            if is_text_file(file_path):
                # 文本文件，使用代码编辑器打开
                self._open_file_in_editor(file_path)
            else:
                # 非文本文件，使用系统默认程序打开
                open_with_system_app(file_path)
        except Exception as e:
            QMessageBox.critical(self, "错误", f"打开文件失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _open_file_in_editor(self, file_path: str):
        """在代码编辑器中打开文件"""
        try:
            # 检查文件是否已经在编辑器中打开
            file_name = os.path.basename(file_path)
            for i in range(self.code_tab_widget.count()):
                if self.code_tab_widget.tabText(i) == file_name:
                    self.code_tab_widget.setCurrentIndex(i)
                    return
            
            # 读取文件内容
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # 创建新的编辑器标签页
            editor = MonacoEditorWidget()
            editor.setReadOnly(False)
            editor.setLineWrapMode(QPlainTextEdit.NoWrap)
            editor.setPlainText(content)
            
            # 连接运行代码信号
            if hasattr(editor, 'run_code_requested'):
                editor.run_code_requested.connect(self._run_code_from_editor)
            
            # 应用编辑器设置
            self._apply_editor_settings_to_editor(editor)
            
            # 连接文本改变信号，保存文件
            def on_text_changed():
                try:
                    current_content = editor.toPlainText()
                    with open(file_path, 'w', encoding='utf-8') as f:
                        f.write(current_content)
                except Exception as e:
                    print(f"保存文件失败: {e}")
            
            editor.textChanged.connect(on_text_changed)
            
            # 添加到标签页
            self.code_tab_widget.addTab(editor, file_name)
            self._code_editors[file_name] = editor
            self.code_tab_widget.setCurrentIndex(self.code_tab_widget.count() - 1)
        except Exception as e:
            QMessageBox.critical(self, "错误", f"打开文件失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _open_file_with_system(self, file_path: str):
        """使用系统默认程序打开文件"""
        try:
            if not os.path.exists(file_path):
                QMessageBox.warning(self, "错误", f"文件不存在: {file_path}")
                return
            
            if open_with_system_app(file_path):
                self.statusBar().showMessage(f"已使用系统程序打开: {file_path}")
            else:
                QMessageBox.warning(self, "错误", "无法使用系统程序打开文件")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"打开文件失败: {str(e)}")
    
    def _new_file_in_folder(self, folder_path: str):
        """在文件夹中新建文件"""
        try:
            from PySide6.QtWidgets import QInputDialog
            
            # 获取文件名和中文名称
            file_name, ok = QInputDialog.getText(
                self,
                "新建文件",
                "请输入文件名（包含扩展名）:",
                text="new_file.txt"
            )
            if not ok or not file_name.strip():
                return
            
            file_name = file_name.strip()
            file_path = os.path.join(folder_path, file_name)
            
            # 检查文件是否已存在
            if os.path.exists(file_path):
                reply = QMessageBox.question(
                    self,
                    "文件已存在",
                    f"文件 '{file_name}' 已存在，是否覆盖？",
                    QMessageBox.Yes | QMessageBox.No
                )
                if reply == QMessageBox.No:
                    return
            
            # 获取中文名称（可选）
            chinese_name, ok = QInputDialog.getText(
                self,
                "文件中文名称",
                "请输入文件的中文名称（可选，直接确定则跳过）:",
                text=""
            )
            if ok and chinese_name.strip():
                self._file_chinese_names[file_path] = chinese_name.strip()
            
            # 创建文件
            if create_file(file_path, ""):
                self.statusBar().showMessage(f"已创建文件: {file_name}")
                self._refresh_project_preview()
            else:
                QMessageBox.warning(self, "错误", "创建文件失败")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"新建文件失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _new_folder_in_folder(self, folder_path: str):
        """在文件夹中新建文件夹"""
        try:
            from PySide6.QtWidgets import QInputDialog
            
            # 获取文件夹名称和中文名称
            folder_name, ok = QInputDialog.getText(
                self,
                "新建文件夹",
                "请输入文件夹名称:",
                text="new_folder"
            )
            if not ok or not folder_name.strip():
                return
            
            folder_name = folder_name.strip()
            new_folder_path = os.path.join(folder_path, folder_name)
            
            # 检查文件夹是否已存在
            if os.path.exists(new_folder_path):
                QMessageBox.warning(self, "错误", f"文件夹 '{folder_name}' 已存在")
                return
            
            # 获取中文名称（可选）
            chinese_name, ok = QInputDialog.getText(
                self,
                "文件夹中文名称",
                "请输入文件夹的中文名称（可选，直接确定则跳过）:",
                text=""
            )
            if ok and chinese_name.strip():
                self._file_chinese_names[new_folder_path] = chinese_name.strip()
            
            # 创建文件夹
            if create_directory(new_folder_path):
                self.statusBar().showMessage(f"已创建文件夹: {folder_name}")
                self._refresh_project_preview()
            else:
                QMessageBox.warning(self, "错误", "创建文件夹失败")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"新建文件夹失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _copy_file_path(self, path: str):
        """复制文件/文件夹的相对路径"""
        try:
            if hasattr(self, 'project_dir') and self.project_dir:
                rel_path = get_relative_path(path, self.project_dir)
                QApplication.clipboard().setText(rel_path)
                self.statusBar().showMessage(f"已复制相对路径: {rel_path}")
            else:
                QApplication.clipboard().setText(path)
                self.statusBar().showMessage(f"已复制路径: {path}")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"复制路径失败: {str(e)}")
    
    def _copy_file(self, file_path: str):
        """复制文件/文件夹到剪贴板"""
        try:
            self._clipboard_file_path = file_path
            self._clipboard_is_cut = False
            self.statusBar().showMessage(f"已复制: {os.path.basename(file_path)}")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"复制失败: {str(e)}")
    
    def _cut_file(self, file_path: str):
        """剪切文件/文件夹到剪贴板"""
        try:
            self._clipboard_file_path = file_path
            self._clipboard_is_cut = True
            self.statusBar().showMessage(f"已剪切: {os.path.basename(file_path)}")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"剪切失败: {str(e)}")
    
    def _paste_file(self, target_folder: str):
        """粘贴文件/文件夹"""
        try:
            if not self._clipboard_file_path:
                QMessageBox.warning(self, "警告", "剪贴板中没有文件")
                return
            
            if not os.path.exists(self._clipboard_file_path):
                QMessageBox.warning(self, "错误", "源文件不存在")
                self._clipboard_file_path = None
                return
            
            source_name = os.path.basename(self._clipboard_file_path)
            target_path = os.path.join(target_folder, source_name)
            
            # 检查目标是否已存在
            if os.path.exists(target_path):
                reply = QMessageBox.question(
                    self,
                    "文件已存在",
                    f"目标位置已存在 '{source_name}'，是否覆盖？",
                    QMessageBox.Yes | QMessageBox.No
                )
                if reply == QMessageBox.No:
                    return
            
            # 执行复制或移动
            if self._clipboard_is_cut:
                # 剪切（移动）
                if move_file_or_directory(self._clipboard_file_path, target_path):
                    # 更新中文名称
                    if self._clipboard_file_path in self._file_chinese_names:
                        chinese_name = self._file_chinese_names.pop(self._clipboard_file_path)
                        self._file_chinese_names[target_path] = chinese_name
                    self.statusBar().showMessage(f"已移动: {source_name}")
                    self._clipboard_file_path = None
                else:
                    QMessageBox.warning(self, "错误", "移动失败")
            else:
                # 复制
                if copy_file_or_directory(self._clipboard_file_path, target_path):
                    self.statusBar().showMessage(f"已复制: {source_name}")
                else:
                    QMessageBox.warning(self, "错误", "复制失败")
            
            self._refresh_project_preview()
        except Exception as e:
            QMessageBox.critical(self, "错误", f"粘贴失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _rename_file(self, file_path: str):
        """重命名文件/文件夹"""
        try:
            from PySide6.QtWidgets import QInputDialog
            
            old_name = os.path.basename(file_path)
            parent_dir = os.path.dirname(file_path)
            
            # 获取新名称
            new_name, ok = QInputDialog.getText(
                self,
                "重命名",
                f"请输入新名称:",
                text=old_name
            )
            if not ok or not new_name.strip() or new_name.strip() == old_name:
                return
            
            new_name = new_name.strip()
            new_path = os.path.join(parent_dir, new_name)
            
            # 检查新名称是否已存在
            if os.path.exists(new_path):
                QMessageBox.warning(self, "错误", f"'{new_name}' 已存在")
                return
            
            # 重命名
            if rename_file_or_directory(file_path, new_path):
                # 更新中文名称
                if file_path in self._file_chinese_names:
                    chinese_name = self._file_chinese_names.pop(file_path)
                    self._file_chinese_names[new_path] = chinese_name
                
                self.statusBar().showMessage(f"已重命名: {old_name} -> {new_name}")
                self._refresh_project_preview()
            else:
                QMessageBox.warning(self, "错误", "重命名失败")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"重命名失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _delete_file(self, file_path: str):
        """删除文件/文件夹"""
        try:
            file_name = os.path.basename(file_path)
            is_dir = os.path.isdir(file_path)
            
            reply = QMessageBox.question(
                self,
                "确认删除",
                f"确定要删除{'文件夹' if is_dir else '文件'} '{file_name}' 吗？\n此操作不可撤销！",
                QMessageBox.Yes | QMessageBox.No
            )
            
            if reply == QMessageBox.Yes:
                if delete_file_or_directory(file_path):
                    # 清除中文名称
                    if file_path in self._file_chinese_names:
                        del self._file_chinese_names[file_path]
                    
                    self.statusBar().showMessage(f"已删除: {file_name}")
                    self._refresh_project_preview()
                else:
                    QMessageBox.warning(self, "错误", "删除失败")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"删除失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _export_file(self, file_path: str):
        """导出文件"""
        try:
            file_name = os.path.basename(file_path)
            target_path, _ = QFileDialog.getSaveFileName(
                self,
                "导出文件",
                file_name,
                "所有文件 (*.*)"
            )
            
            if target_path:
                if copy_file_or_directory(file_path, target_path):
                    self.statusBar().showMessage(f"已导出: {file_name}")
                else:
                    QMessageBox.warning(self, "错误", "导出失败")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"导出失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _import_file(self, target_folder: str):
        """导入文件"""
        try:
            source_path, _ = QFileDialog.getOpenFileName(
                self,
                "导入文件",
                "",
                "所有文件 (*.*)"
            )
            
            if source_path:
                file_name = os.path.basename(source_path)
                target_path = os.path.join(target_folder, file_name)
                
                # 检查目标是否已存在
                if os.path.exists(target_path):
                    reply = QMessageBox.question(
                        self,
                        "文件已存在",
                        f"目标位置已存在 '{file_name}'，是否覆盖？",
                        QMessageBox.Yes | QMessageBox.No
                    )
                    if reply == QMessageBox.No:
                        return
                
                if copy_file_or_directory(source_path, target_path):
                    self.statusBar().showMessage(f"已导入: {file_name}")
                    self._refresh_project_preview()
                else:
                    QMessageBox.warning(self, "错误", "导入失败")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"导入失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _open_canvas_code(self, window_name: str):
        """打开画布代码"""
        try:
            if hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class:
                    self._switch_to_window_class(window_name)
                    canvas_tab_name = f"{window_name}_canvas.py"
                    # 查找或创建标签页
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == canvas_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            return
                    # 如果不存在，创建它
                    self._update_canvas_code_tab(window_name, window_class)
        except Exception as e:
            QMessageBox.critical(self, "错误", f"打开画布代码失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _copy_canvas_code(self, window_name: str):
        """复制画布代码"""
        try:
            if hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class and window_class.canvas_code:
                    QApplication.clipboard().setText(window_class.canvas_code)
                    self.statusBar().showMessage(f"已复制画布代码: {window_name}")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"复制画布代码失败: {str(e)}")
    
    def _open_functional_code(self, window_name: str):
        """打开功能代码"""
        try:
            if hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class:
                    self._switch_to_window_class(window_name)
                    # 更新功能代码标签页
                    self._update_event_function_tab()
                    # 查找标签页
                    function_tab_name = f"{window_name}_function.py"
                    for i in range(self.code_tab_widget.count()):
                        if self.code_tab_widget.tabText(i) == function_tab_name:
                            self.code_tab_widget.setCurrentIndex(i)
                            return
        except Exception as e:
            QMessageBox.critical(self, "错误", f"打开功能代码失败: {str(e)}")
            import traceback
            traceback.print_exc()
    
    def _copy_functional_code(self, window_name: str):
        """复制功能代码"""
        try:
            if hasattr(self.project_manager, 'window_class_manager'):
                window_class = self.project_manager.window_class_manager.get_window_class(window_name)
                if window_class and window_class.event_function_file:
                    QApplication.clipboard().setText(window_class.event_function_file)
                    self.statusBar().showMessage(f"已复制功能代码: {window_name}")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"复制功能代码失败: {str(e)}")
    
    def _expand_all_components(self, item: QTreeWidgetItem):
        """展开所有组件"""
        try:
            item.setExpanded(True)
            for i in range(item.childCount()):
                child = item.child(i)
                child.setExpanded(True)
                # 递归展开子节点
                self._expand_item_recursive(child)
        except Exception as e:
            print(f"展开组件失败: {e}")
    
    def _collapse_all_components(self, item: QTreeWidgetItem):
        """折叠所有组件"""
        try:
            item.setExpanded(False)
            for i in range(item.childCount()):
                child = item.child(i)
                child.setExpanded(False)
                # 递归折叠子节点
                self._collapse_item_recursive(child)
        except Exception as e:
            print(f"折叠组件失败: {e}")
    
    def _expand_item_recursive(self, item: QTreeWidgetItem):
        """递归展开节点"""
        try:
            item.setExpanded(True)
            for i in range(item.childCount()):
                self._expand_item_recursive(item.child(i))
        except Exception:
            pass
    
    def _collapse_item_recursive(self, item: QTreeWidgetItem):
        """递归折叠节点"""
        try:
            item.setExpanded(False)
            for i in range(item.childCount()):
                self._collapse_item_recursive(item.child(i))
        except Exception:
            pass
    
    def _open_function_file(self, file_name: str):
        """打开功能文件"""
        try:
            # 查找项目目录中的文件
            if hasattr(self, 'project_dir') and self.project_dir:
                file_path = os.path.join(self.project_dir, file_name)
                if os.path.exists(file_path):
                    self._open_file(file_path)
                else:
                    QMessageBox.warning(self, "错误", f"文件不存在: {file_path}")
        except Exception as e:
            QMessageBox.critical(self, "错误", f"打开文件失败: {str(e)}")
    
    def _show_component_info(self, component_name: str):
        """显示组件信息"""
        try:
            QMessageBox.information(self, "组件信息", f"组件名称: {component_name}\n\n此功能待完善")
        except Exception as e:
            print(f"显示组件信息失败: {e}")