#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
无标题栏窗口基类
提供自定义标题栏、最大化、最小化、关闭按钮等功能
"""

from PySide6.QtWidgets import (
    QWidget, QHBoxLayout, QVBoxLayout, QLabel, QPushButton, QFrame
)
from PySide6.QtCore import Qt, QPoint, QSize, Signal, QEventLoop
from PySide6.QtGui import QIcon, QMouseEvent, QPainter, QColor, QFont


class FramelessWindow(QWidget):
    """无标题栏窗口基类"""
    
    # 窗口状态改变信号
    window_state_changed = Signal(bool)  # 是否最大化
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self._is_maximized = False
        self._drag_position = QPoint()
        self._is_dragging = False
        
        # 设置窗口标志
        self.setWindowFlags(Qt.FramelessWindowHint | Qt.Window)
        
        # 创建自定义标题栏
        self._init_title_bar()
        
        # 创建主布局
        self._main_layout = QVBoxLayout(self)
        self._main_layout.setContentsMargins(0, 0, 0, 0)
        self._main_layout.setSpacing(0)
        
        # 添加标题栏
        self._main_layout.addWidget(self._title_bar)
        
        # 创建内容区域
        self._content_widget = QWidget()
        self._content_layout = QVBoxLayout(self._content_widget)
        self._content_layout.setContentsMargins(0, 0, 0, 0)
        self._main_layout.addWidget(self._content_widget, 1)
    
    def _init_title_bar(self):
        """初始化标题栏"""
        self._title_bar = QFrame()
        self._title_bar.setObjectName("TitleBar")
        self._title_bar.setFixedHeight(40)
        self._title_bar.setMouseTracking(True)
        
        title_layout = QHBoxLayout(self._title_bar)
        title_layout.setContentsMargins(10, 0, 5, 0)
        title_layout.setSpacing(5)
        
        # 标题标签
        self._title_label = QLabel("窗口标题")
        self._title_label.setObjectName("TitleLabel")
        title_layout.addWidget(self._title_label)
        
        title_layout.addStretch()
        
        # 最小化按钮（使用图标或文字）
        self._minimize_btn = QPushButton("−")
        self._minimize_btn.setObjectName("MinimizeButton")
        self._minimize_btn.setFixedSize(30, 30)
        self._minimize_btn.setToolTip("最小化")
        self._minimize_btn.clicked.connect(self.showMinimized)
        title_layout.addWidget(self._minimize_btn)
        
        # 最大化/还原按钮（使用图标或文字）
        self._maximize_btn = QPushButton("□")
        self._maximize_btn.setObjectName("MaximizeButton")
        self._maximize_btn.setFixedSize(30, 30)
        self._maximize_btn.setToolTip("最大化")
        self._maximize_btn.clicked.connect(self._toggle_maximize)
        title_layout.addWidget(self._maximize_btn)
        
        # 关闭按钮（使用图标或文字）
        self._close_btn = QPushButton("×")
        self._close_btn.setObjectName("CloseButton")
        self._close_btn.setFixedSize(30, 30)
        self._close_btn.setToolTip("关闭")
        self._close_btn.clicked.connect(self.close)
        title_layout.addWidget(self._close_btn)
    
    def setWindowTitle(self, title: str):
        """设置窗口标题"""
        super().setWindowTitle(title)
        if hasattr(self, '_title_label'):
            self._title_label.setText(title)
    
    def _toggle_maximize(self):
        """切换最大化状态"""
        if self._is_maximized:
            self.showNormal()
        else:
            self.showMaximized()
    
    def showMaximized(self):
        """显示最大化"""
        super().showMaximized()
        self._is_maximized = True
        if hasattr(self, '_maximize_btn'):
            self._maximize_btn.setText("❐")
            self._maximize_btn.setToolTip("还原")
        self.window_state_changed.emit(True)
    
    def showNormal(self):
        """显示正常大小"""
        super().showNormal()
        self._is_maximized = False
        if hasattr(self, '_maximize_btn'):
            self._maximize_btn.setText("□")
            self._maximize_btn.setToolTip("最大化")
        self.window_state_changed.emit(False)
    
    def mousePressEvent(self, event: QMouseEvent):
        """鼠标按下事件"""
        if event.button() == Qt.LeftButton:
            if event.y() < 40:  # 标题栏区域
                self._is_dragging = True
                self._drag_position = event.globalPosition().toPoint() - self.frameGeometry().topLeft()
                event.accept()
        super().mousePressEvent(event)
    
    def mouseMoveEvent(self, event: QMouseEvent):
        """鼠标移动事件"""
        if self._is_dragging and event.buttons() == Qt.LeftButton:
            if not self._is_maximized:
                self.move(event.globalPosition().toPoint() - self._drag_position)
            event.accept()
        super().mouseMoveEvent(event)
    
    def mouseReleaseEvent(self, event: QMouseEvent):
        """鼠标释放事件"""
        self._is_dragging = False
        super().mouseReleaseEvent(event)
    
    def contentWidget(self) -> QWidget:
        """获取内容区域"""
        return self._content_widget
    
    def contentLayout(self) -> QVBoxLayout:
        """获取内容布局"""
        return self._content_layout
    
    def exec(self):
        """执行对话框（模态显示）"""
        self.setWindowModality(Qt.ApplicationModal)
        self.show()
        self.raise_()
        self.activateWindow()
        loop = QEventLoop()
        self.destroyed.connect(loop.quit)
        self.finished = False
        self._loop = loop
        result = loop.exec()
        return result if hasattr(self, 'result') else (1 if self.finished else 0)
    
    def exec_(self):
        """执行对话框（模态显示，兼容QDialog）"""
        return self.exec()
    
    def accept(self):
        """接受对话框（兼容QDialog）"""
        self.finished = True
        self.result = 1
        if hasattr(self, '_loop'):
            self._loop.quit()
        self.close()
    
    def reject(self):
        """拒绝对话框（兼容QDialog）"""
        self.finished = False
        self.result = 0
        if hasattr(self, '_loop'):
            self._loop.quit()
        self.close()

