#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
函数库组件
显示函数库列表，使用树形框展示函数、类、数据类型等
"""

import os
from pathlib import Path
from PySide6.QtWidgets import (
    QWidget, QVBoxLayout, QTreeWidget, QTreeWidgetItem, QLabel, QMenu
)
from PySide6.QtCore import Qt, Signal
from PySide6.QtGui import QIcon

# 导入函数库管理器
import sys
sys.path.insert(0, str(Path(__file__).parent.parent))
from lib.fnpy_library_manager import fnpyLibraryManager, LibraryInfo


class fnpyLibraryWidget(QWidget):
    """
    函数库组件
    使用树形框显示函数库的结构
    """
    
    # 定义信号：函数被点击时发出
    function_clicked = Signal(str, str, str)  # library_name, function_name, docstring
    
    def __init__(self, parent=None):
        """
        初始化函数库组件
        
        Args:
            parent: 父窗口部件
        """
        super().__init__(parent)
        
        # 创建函数库管理器
        self.library_manager = fnpyLibraryManager()
        
        # 初始化UI
        self._init_ui()
        
        # 加载函数库
        self._load_libraries()
    
    def _init_ui(self):
        """初始化用户界面"""
        # 创建垂直布局
        layout = QVBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        
        # 创建标题
        title_label = QLabel("函数库")
        title_label.setStyleSheet("font-weight: bold; font-size: 14px; padding: 5px;")
        layout.addWidget(title_label)
        
        # 创建树形控件
        self.tree_widget = QTreeWidget()
        self.tree_widget.setHeaderLabel("函数库结构")
        self.tree_widget.setColumnCount(1)
        layout.addWidget(self.tree_widget)
        
        # 连接点击事件
        self.tree_widget.itemClicked.connect(self._on_item_clicked)
        
        # 启用右键菜单
        self.tree_widget.setContextMenuPolicy(Qt.CustomContextMenu)
        self.tree_widget.customContextMenuRequested.connect(self._show_context_menu)
    
    def _load_libraries(self):
        """加载并显示函数库"""
        self.tree_widget.clear()
        
        # 扫描所有函数库
        library_names = self.library_manager.scan_libraries()
        
        for library_name in library_names:
            # 加载函数库信息
            library_info = self.library_manager.load_library(library_name)
            if not library_info:
                continue
            
            # 创建函数库根节点（显示中文名（英文名））
            library_item = QTreeWidgetItem(self.tree_widget)
            display_name = f"{library_info.chinese_name} ({library_info.name})"
            library_item.setText(0, display_name)
            library_item.setData(0, Qt.UserRole, ("library", library_name))
            library_item.setExpanded(False)
            
            # 设置图标
            icon_path = self._get_library_icon_path()
            if icon_path and os.path.exists(icon_path):
                library_item.setIcon(0, QIcon(icon_path))
            
            # 添加函数节点
            if library_info.functions:
                functions_item = QTreeWidgetItem(library_item)
                functions_item.setText(0, f"函数 ({len(library_info.functions)})")
                functions_item.setData(0, Qt.UserRole, ("functions_group", library_name))
                functions_item.setExpanded(False)
                
                for func in library_info.functions:
                    func_item = QTreeWidgetItem(functions_item)
                    # 显示中文名（英文名）
                    func_display = f"{func.chinese_name} ({func.name})"
                    func_item.setText(0, func_display)
                    func_item.setData(0, Qt.UserRole, ("function", library_name, func.name, func.line, func.docstring or ""))
            
            # 添加类节点
            if library_info.classes:
                classes_item = QTreeWidgetItem(library_item)
                classes_item.setText(0, f"类 ({len(library_info.classes)})")
                classes_item.setData(0, Qt.UserRole, ("classes_group", library_name))
                classes_item.setExpanded(False)
                
                for cls in library_info.classes:
                    class_item = QTreeWidgetItem(classes_item)
                    # 显示中文名（英文名）
                    class_display = f"{cls.chinese_name} ({cls.name})"
                    class_item.setText(0, class_display)
                    class_item.setData(0, Qt.UserRole, ("class", library_name, cls.name, cls.line))
                    class_item.setExpanded(False)
                    
                    # 添加类的成员
                    if cls.members:
                        members_item = QTreeWidgetItem(class_item)
                        members_item.setText(0, f"成员 ({len(cls.members)})")
                        members_item.setData(0, Qt.UserRole, ("members_group", library_name, cls.name))
                        members_item.setExpanded(False)
                        
                        for member in cls.members:
                            member_item = QTreeWidgetItem(members_item)
                            member_item.setText(0, f"{member['name']} ({member['type']})")
                            member_item.setData(0, Qt.UserRole, ("member", library_name, cls.name, member['name'], member['line']))
                    
                    # 添加数据结构
                    if cls.data_structures:
                        ds_item = QTreeWidgetItem(class_item)
                        ds_item.setText(0, f"数据结构 ({len(cls.data_structures)})")
                        ds_item.setData(0, Qt.UserRole, ("data_structures_group", library_name, cls.name))
                        ds_item.setExpanded(False)
                        
                        for ds in cls.data_structures:
                            ds_name_item = QTreeWidgetItem(ds_item)
                            ds_name_item.setText(0, ds['name'])
                            ds_name_item.setData(0, Qt.UserRole, ("data_structure", library_name, cls.name, ds['name'], ds['line']))
                            ds_name_item.setExpanded(False)
                            
                            # 添加数据结构的成员
                            if 'members' in ds and ds['members']:
                                ds_members_item = QTreeWidgetItem(ds_name_item)
                                ds_members_item.setText(0, "成员")
                                ds_members_item.setData(0, Qt.UserRole, ("ds_members_group", library_name, cls.name, ds['name']))
                                ds_members_item.setExpanded(False)
                                
                                for member in ds['members']:
                                    member_item = QTreeWidgetItem(ds_members_item)
                                    member_item.setText(0, f"{member['name']} ({member['type']})")
                                    member_item.setData(0, Qt.UserRole, ("ds_member", library_name, cls.name, ds['name'], member['name'], member['line']))
            
            # 添加数据类型节点
            if library_info.data_types:
                data_types_item = QTreeWidgetItem(library_item)
                data_types_item.setText(0, f"数据类型 ({len(library_info.data_types)})")
                data_types_item.setData(0, Qt.UserRole, ("data_types_group", library_name))
                data_types_item.setExpanded(False)
                
                for dt in library_info.data_types:
                    dt_item = QTreeWidgetItem(data_types_item)
                    # 显示中文名（英文名）
                    dt_display = f"{dt.chinese_name} ({dt.name})"
                    dt_item.setText(0, dt_display)
                    dt_item.setData(0, Qt.UserRole, ("data_type", library_name, dt.name, dt.line))
                    dt_item.setExpanded(False)
                    
                    # 添加数据类型的成员
                    if dt.members:
                        dt_members_item = QTreeWidgetItem(dt_item)
                        dt_members_item.setText(0, "成员")
                        dt_members_item.setData(0, Qt.UserRole, ("data_type_members_group", library_name, dt.name))
                        dt_members_item.setExpanded(False)
                        
                        for member in dt.members:
                            member_item = QTreeWidgetItem(dt_members_item)
                            member_item.setText(0, f"{member['name']} ({member['type']})")
                            member_item.setData(0, Qt.UserRole, ("data_type_member", library_name, dt.name, member['name'], member['line']))
            
            # 添加常量节点
            if library_info.constants:
                constants_item = QTreeWidgetItem(library_item)
                constants_item.setText(0, f"常量 ({len(library_info.constants)})")
                constants_item.setData(0, Qt.UserRole, ("constants_group", library_name))
                constants_item.setExpanded(False)
                
                for const in library_info.constants:
                    const_item = QTreeWidgetItem(constants_item)
                    # 显示中文名（英文名）
                    const_display = f"{const.chinese_name} ({const.name})"
                    const_item.setText(0, const_display)
                    const_item.setData(0, Qt.UserRole, ("constant", library_name, const.name, const.line))
    
    def _get_library_icon_path(self) -> str:
        """获取函数库图标路径"""
        try:
            from .theme_manager import ThemeManager
            theme_manager = ThemeManager()
            theme_name = theme_manager.get_current_theme()
            icon_path = Path(__file__).parent.parent / "ui" / "icon" / theme_name / "fnpydata.png"
            return str(icon_path) if icon_path.exists() else ""
        except Exception:
            return ""
    
    def _on_item_clicked(self, item: QTreeWidgetItem, column: int):
        """处理树形项点击事件"""
        data = item.data(0, Qt.UserRole)
        if not data:
            # 如果没有数据，检查是否有子节点，如果有则展开/折叠
            if item.childCount() > 0:
                item.setExpanded(not item.isExpanded())
            return
        
        item_type = data[0]
        has_children = item.childCount() > 0
        
        # 如果是函数，发送信号显示注释
        if item_type == "function":
            library_name = data[1] if len(data) > 1 else ""
            function_name = data[2] if len(data) > 2 else ""
            docstring = data[4] if len(data) > 4 else ""
            # 发送信号
            self.function_clicked.emit(library_name, function_name, docstring or "无")
        
        # 如果有子节点，展开/折叠
        if has_children:
            item.setExpanded(not item.isExpanded())
    
    def _show_context_menu(self, pos):
        """显示右键菜单"""
        item = self.tree_widget.itemAt(pos)
        if not item:
            return
        
        data = item.data(0, Qt.UserRole)
        if not data:
            return
        
        item_type = data[0]
        
        # 只有函数库根节点才显示"查看详细"
        if item_type == "library":
            library_name = data[1] if len(data) > 1 else ""
            
            menu = QMenu(self)
            view_detail_action = menu.addAction("查看详细")
            
            action = menu.exec(self.tree_widget.mapToGlobal(pos))
            if action == view_detail_action:
                self._show_library_details(library_name)
    
    def _show_library_details(self, library_name: str):
        """显示函数库详细信息"""
        try:
            library_info = self.library_manager.load_library(library_name)
            if not library_info:
                return
            
            from PySide6.QtWidgets import QDialog, QVBoxLayout, QLabel, QTextEdit, QPushButton, QHBoxLayout, QScrollArea, QWidget
            from PySide6.QtCore import Qt
            
            dialog = QDialog(self)
            dialog.setWindowTitle(f"函数库详细信息 - {library_info.chinese_name}")
            dialog.setMinimumSize(600, 500)
            
            layout = QVBoxLayout(dialog)
            
            # 创建滚动区域
            scroll = QScrollArea()
            scroll.setWidgetResizable(True)
            content = QWidget()
            content_layout = QVBoxLayout(content)
            
            # 显示详细信息
            details = [
                ("支持库名称", library_info.chinese_name),
                ("英文名称", library_info.name),
                ("版本", library_info.version),
                ("创建时间", library_info.create_time or "未设置"),
                ("作者", library_info.author or "未设置"),
                ("作者联系方式", library_info.author_contact or "未设置"),
                ("Python版本要求", library_info.python_version or "未设置"),
            ]
            
            for label, value in details:
                row = QHBoxLayout()
                label_widget = QLabel(f"{label}:")
                label_widget.setMinimumWidth(120)
                label_widget.setStyleSheet("font-weight: bold;")
                value_widget = QLabel(value)
                row.addWidget(label_widget)
                row.addWidget(value_widget)
                row.addStretch()
                content_layout.addLayout(row)
            
            # 所需Python库
            if library_info.required_packages:
                content_layout.addWidget(QLabel("所需Python库:"))
                packages_text = "\n".join([f"  {pkg}: {ver}" for pkg, ver in library_info.required_packages.items()])
                packages_label = QLabel(packages_text)
                packages_label.setWordWrap(True)
                content_layout.addWidget(packages_label)
            
            content_layout.addStretch()
            
            scroll.setWidget(content)
            layout.addWidget(scroll)
            
            # 确定按钮
            btn_layout = QHBoxLayout()
            btn_layout.addStretch()
            ok_btn = QPushButton("确定")
            ok_btn.clicked.connect(dialog.accept)
            btn_layout.addWidget(ok_btn)
            layout.addLayout(btn_layout)
            
            dialog.exec()
            
        except Exception as e:
            print(f"显示函数库详细信息失败: {e}")
            import traceback
            traceback.print_exc()
    
    def refresh(self):
        """刷新函数库列表"""
        self._load_libraries()
