#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
文件管理器模块
提供文件操作功能：新建、删除、重命名、复制、剪切、粘贴等
"""

import os
import shutil
from pathlib import Path
from typing import Optional, List, Tuple
from PySide6.QtWidgets import QMessageBox


# 文本文件扩展名列表（支持在代码编辑器中打开）
TEXT_FILE_EXTENSIONS = {
    # 编程语言
    'py', 'pyw', 'pyx', 'pyi',  # Python
    'cpp', 'cxx', 'cc', 'c++', 'hpp', 'hxx', 'hh', 'h++',  # C++
    'c', 'h',  # C
    'java',  # Java
    'cs',  # C#
    'js', 'jsx', 'mjs', 'ts', 'tsx',  # JavaScript/TypeScript
    'php', 'phtml',  # PHP
    'rb', 'rbw',  # Ruby
    'go',  # Go
    'rs',  # Rust
    'swift',  # Swift
    'kt', 'kts',  # Kotlin
    'scala',  # Scala
    'clj', 'cljs', 'cljc',  # Clojure
    'hs', 'lhs',  # Haskell
    'ml', 'mli',  # OCaml
    'erl', 'hrl',  # Erlang
    'ex', 'exs',  # Elixir
    'lua',  # Lua
    'pl', 'pm',  # Perl
    'r', 'R',  # R
    'm', 'mm',  # Objective-C
    'dart',  # Dart
    'vb', 'vbs',  # VB.NET
    'pas', 'pp',  # Pascal
    'f', 'for', 'f90', 'f95', 'f03', 'f08',  # Fortran
    'asm', 's',  # Assembly
    'sh', 'bash', 'zsh', 'fish',  # Shell scripts
    'ps1', 'psm1', 'psd1',  # PowerShell
    'bat', 'cmd',  # Batch
    'vbs', 'wsf',  # VBScript
    
    # 标记语言
    'html', 'htm', 'xhtml',  # HTML
    'xml', 'xsl', 'xslt', 'xsd',  # XML
    'svg',  # SVG
    'md', 'markdown', 'mdown', 'mkd',  # Markdown
    'rst',  # reStructuredText
    'tex', 'latex',  # LaTeX
    'yaml', 'yml',  # YAML
    'toml',  # TOML
    'json', 'json5',  # JSON
    'ini', 'cfg', 'conf', 'config',  # Config files
    
    # 样式表
    'css', 'scss', 'sass', 'less', 'styl',  # CSS
    'xaml',  # XAML
    
    # 数据格式
    'csv', 'tsv',  # CSV
    'sql',  # SQL
    'diff', 'patch',  # Diff/Patch
    
    # 文档
    'txt', 'text',  # Text
    'log',  # Log
    'rtf',  # RTF
    
    # 配置文件
    'gitignore', 'gitattributes', 'gitconfig',  # Git
    'dockerfile',  # Docker
    'makefile', 'mk',  # Makefile
    'cmake',  # CMake
    'gradle',  # Gradle
    'properties',  # Properties
    'env',  # Environment
    
    # 其他文本格式
    'proto',  # Protocol Buffers
    'graphql', 'gql',  # GraphQL
    'gcode',  # G-code
    'nfo',  # NFO
    'reg',  # Registry
    'url',  # URL
    'vscode',  # VS Code settings
}


def is_text_file(file_path: str) -> bool:
    """
    判断文件是否为文本文件
    
    Args:
        file_path: 文件路径
        
    Returns:
        bool: 如果是文本文件返回True，否则返回False
    """
    if not os.path.isfile(file_path):
        return False
    
    ext = Path(file_path).suffix.lstrip('.').lower()
    return ext in TEXT_FILE_EXTENSIONS


def get_file_display_name(file_path: str, chinese_name: Optional[str] = None) -> str:
    """
    获取文件的显示名称（中文名（文件名））
    
    Args:
        file_path: 文件路径
        chinese_name: 中文名称（可选）
        
    Returns:
        str: 显示名称
    """
    file_name = os.path.basename(file_path)
    if chinese_name:
        return f"{chinese_name} ({file_name})"
    return file_name


def create_file(file_path: str, content: str = "") -> bool:
    """
    创建文件
    
    Args:
        file_path: 文件路径
        content: 文件内容（默认为空）
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        # 确保目录存在
        os.makedirs(os.path.dirname(file_path), exist_ok=True)
        
        # 创建文件
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(content)
        return True
    except Exception as e:
        print(f"创建文件失败: {e}")
        return False


def create_directory(dir_path: str) -> bool:
    """
    创建目录
    
    Args:
        dir_path: 目录路径
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        os.makedirs(dir_path, exist_ok=True)
        return True
    except Exception as e:
        print(f"创建目录失败: {e}")
        return False


def delete_file_or_directory(path: str) -> bool:
    """
    删除文件或目录
    
    Args:
        path: 文件或目录路径
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        if os.path.isfile(path):
            os.remove(path)
        elif os.path.isdir(path):
            shutil.rmtree(path)
        else:
            return False
        return True
    except Exception as e:
        print(f"删除失败: {e}")
        return False


def rename_file_or_directory(old_path: str, new_path: str) -> bool:
    """
    重命名文件或目录
    
    Args:
        old_path: 原路径
        new_path: 新路径
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        os.rename(old_path, new_path)
        return True
    except Exception as e:
        print(f"重命名失败: {e}")
        return False


def copy_file_or_directory(src: str, dst: str) -> bool:
    """
    复制文件或目录
    
    Args:
        src: 源路径
        dst: 目标路径
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        if os.path.isfile(src):
            shutil.copy2(src, dst)
        elif os.path.isdir(src):
            shutil.copytree(src, dst, dirs_exist_ok=True)
        else:
            return False
        return True
    except Exception as e:
        print(f"复制失败: {e}")
        return False


def move_file_or_directory(src: str, dst: str) -> bool:
    """
    移动文件或目录（剪切）
    
    Args:
        src: 源路径
        dst: 目标路径
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        shutil.move(src, dst)
        return True
    except Exception as e:
        print(f"移动失败: {e}")
        return False


def get_relative_path(file_path: str, base_path: str) -> str:
    """
    获取相对路径
    
    Args:
        file_path: 文件路径
        base_path: 基础路径
        
    Returns:
        str: 相对路径
    """
    try:
        return os.path.relpath(file_path, base_path)
    except Exception:
        return file_path


def open_with_system_app(file_path: str) -> bool:
    """
    使用系统默认程序打开文件
    
    Args:
        file_path: 文件路径
        
    Returns:
        bool: 成功返回True，失败返回False
    """
    try:
        import platform
        if platform.system() == 'Windows':
            os.startfile(file_path)
        elif platform.system() == 'Darwin':  # macOS
            os.system(f'open "{file_path}"')
        else:  # Linux
            os.system(f'xdg-open "{file_path}"')
        return True
    except Exception as e:
        print(f"使用系统程序打开失败: {e}")
        return False

