#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
事件管理器模块
定义所有控件支持的事件类型和事件处理逻辑
"""

from typing import Dict, List, Tuple, Optional
from .control_pyside import Control


class EventManager:
    """
    事件管理器类
    管理所有控件支持的事件类型，提供事件绑定、取消绑定等功能
    """
    
    # ========== 通用事件（所有控件都支持） ==========
    COMMON_EVENTS = {
        "mousePressEvent": {
            "name": "鼠标按下",
            "signal": None,  # 需要重写事件处理
            "description": "当鼠标按下时触发",
            "function_name_template": "_on_{control_id}_mouse_press"
        },
        "mouseReleaseEvent": {
            "name": "鼠标释放",
            "signal": None,
            "description": "当鼠标释放时触发",
            "function_name_template": "_on_{control_id}_mouse_release"
        },
        "mouseMoveEvent": {
            "name": "鼠标移动",
            "signal": None,
            "description": "当鼠标移动时触发",
            "function_name_template": "_on_{control_id}_mouse_move"
        },
        "mouseDoubleClickEvent": {
            "name": "鼠标双击",
            "signal": None,
            "description": "当鼠标双击时触发",
            "function_name_template": "_on_{control_id}_mouse_double_click"
        },
        "wheelEvent": {
            "name": "鼠标滚轮",
            "signal": None,
            "description": "当鼠标滚轮滚动时触发",
            "function_name_template": "_on_{control_id}_wheel"
        },
        "enterEvent": {
            "name": "鼠标进入",
            "signal": None,
            "description": "当鼠标进入控件区域时触发",
            "function_name_template": "_on_{control_id}_enter"
        },
        "leaveEvent": {
            "name": "鼠标离开",
            "signal": None,
            "description": "当鼠标离开控件区域时触发",
            "function_name_template": "_on_{control_id}_leave"
        },
        "focusInEvent": {
            "name": "获得焦点",
            "signal": None,
            "description": "当控件获得焦点时触发",
            "function_name_template": "_on_{control_id}_focus_in"
        },
        "focusOutEvent": {
            "name": "失去焦点",
            "signal": None,
            "description": "当控件失去焦点时触发",
            "function_name_template": "_on_{control_id}_focus_out"
        },
        "keyPressEvent": {
            "name": "按键按下",
            "signal": None,
            "description": "当按键按下时触发",
            "function_name_template": "_on_{control_id}_key_press"
        },
        "keyReleaseEvent": {
            "name": "按键释放",
            "signal": None,
            "description": "当按键释放时触发",
            "function_name_template": "_on_{control_id}_key_release"
        },
        "paintEvent": {
            "name": "绘制事件",
            "signal": None,
            "description": "当控件需要重绘时触发",
            "function_name_template": "_on_{control_id}_paint"
        },
        "resizeEvent": {
            "name": "大小改变",
            "signal": None,
            "description": "当控件大小改变时触发",
            "function_name_template": "_on_{control_id}_resize"
        },
        "moveEvent": {
            "name": "位置改变",
            "signal": None,
            "description": "当控件位置改变时触发",
            "function_name_template": "_on_{control_id}_move"
        }
    }
    
    # ========== 控件特定事件 ==========
    CONTROL_SPECIFIC_EVENTS = {
        "Button": {
            "clicked": {
                "name": "被点击",
                "signal": "clicked",
                "description": "当按钮被点击时触发",
                "function_name_template": "_on_{control_id}_clicked"
            },
            "pressed": {
                "name": "被按下",
                "signal": "pressed",
                "description": "当按钮被按下时触发",
                "function_name_template": "_on_{control_id}_pressed"
            },
            "released": {
                "name": "被释放",
                "signal": "released",
                "description": "当按钮被释放时触发",
                "function_name_template": "_on_{control_id}_released"
            },
            "toggled": {
                "name": "切换状态",
                "signal": "toggled",
                "description": "当按钮切换状态时触发（仅当按钮可切换时）",
                "function_name_template": "_on_{control_id}_toggled"
            }
        },
        "CheckBox": {
            "stateChanged": {
                "name": "状态改变",
                "signal": "stateChanged",
                "description": "当复选框状态改变时触发",
                "function_name_template": "_on_{control_id}_state_changed"
            },
            "toggled": {
                "name": "切换",
                "signal": "toggled",
                "description": "当复选框切换时触发",
                "function_name_template": "_on_{control_id}_toggled"
            },
            "clicked": {
                "name": "被点击",
                "signal": "clicked",
                "description": "当复选框被点击时触发",
                "function_name_template": "_on_{control_id}_clicked"
            }
        },
        "RadioButton": {
            "toggled": {
                "name": "切换",
                "signal": "toggled",
                "description": "当单选按钮切换时触发",
                "function_name_template": "_on_{control_id}_toggled"
            },
            "clicked": {
                "name": "被点击",
                "signal": "clicked",
                "description": "当单选按钮被点击时触发",
                "function_name_template": "_on_{control_id}_clicked"
            }
        },
        "LineEdit": {
            "textChanged": {
                "name": "文本改变",
                "signal": "textChanged",
                "description": "当文本内容改变时触发",
                "function_name_template": "_on_{control_id}_text_changed"
            },
            "textEdited": {
                "name": "文本编辑",
                "signal": "textEdited",
                "description": "当文本被编辑时触发",
                "function_name_template": "_on_{control_id}_text_edited"
            },
            "returnPressed": {
                "name": "回车按下",
                "signal": "returnPressed",
                "description": "当按下回车键时触发",
                "function_name_template": "_on_{control_id}_return_pressed"
            },
            "editingFinished": {
                "name": "编辑完成",
                "signal": "editingFinished",
                "description": "当编辑完成时触发",
                "function_name_template": "_on_{control_id}_editing_finished"
            },
            "selectionChanged": {
                "name": "选择改变",
                "signal": "selectionChanged",
                "description": "当文本选择改变时触发",
                "function_name_template": "_on_{control_id}_selection_changed"
            }
        },
        "TextEdit": {
            "textChanged": {
                "name": "文本改变",
                "signal": "textChanged",
                "description": "当文本内容改变时触发",
                "function_name_template": "_on_{control_id}_text_changed"
            },
            "selectionChanged": {
                "name": "选择改变",
                "signal": "selectionChanged",
                "description": "当文本选择改变时触发",
                "function_name_template": "_on_{control_id}_selection_changed"
            },
            "cursorPositionChanged": {
                "name": "光标位置改变",
                "signal": "cursorPositionChanged",
                "description": "当光标位置改变时触发",
                "function_name_template": "_on_{control_id}_cursor_position_changed"
            }
        },
        "ComboBox": {
            "currentTextChanged": {
                "name": "当前文本改变",
                "signal": "currentTextChanged",
                "description": "当当前选中文本改变时触发",
                "function_name_template": "_on_{control_id}_current_text_changed"
            },
            "currentIndexChanged": {
                "name": "当前索引改变",
                "signal": "currentIndexChanged",
                "description": "当当前选中索引改变时触发",
                "function_name_template": "_on_{control_id}_current_index_changed"
            },
            "activated": {
                "name": "激活",
                "signal": "activated",
                "description": "当下拉框项被激活时触发",
                "function_name_template": "_on_{control_id}_activated"
            },
            "highlighted": {
                "name": "高亮",
                "signal": "highlighted",
                "description": "当下拉框项被高亮时触发",
                "function_name_template": "_on_{control_id}_highlighted"
            }
        },
        "Slider": {
            "valueChanged": {
                "name": "值改变",
                "signal": "valueChanged",
                "description": "当滑动条的值改变时触发",
                "function_name_template": "_on_{control_id}_value_changed"
            },
            "sliderPressed": {
                "name": "滑块按下",
                "signal": "sliderPressed",
                "description": "当滑块被按下时触发",
                "function_name_template": "_on_{control_id}_slider_pressed"
            },
            "sliderReleased": {
                "name": "滑块释放",
                "signal": "sliderReleased",
                "description": "当滑块被释放时触发",
                "function_name_template": "_on_{control_id}_slider_released"
            },
            "sliderMoved": {
                "name": "滑块移动",
                "signal": "sliderMoved",
                "description": "当滑块移动时触发",
                "function_name_template": "_on_{control_id}_slider_moved"
            }
        },
        "SpinBox": {
            "valueChanged": {
                "name": "值改变",
                "signal": "valueChanged",
                "description": "当数值改变时触发",
                "function_name_template": "_on_{control_id}_value_changed"
            },
            "editingFinished": {
                "name": "编辑完成",
                "signal": "editingFinished",
                "description": "当编辑完成时触发",
                "function_name_template": "_on_{control_id}_editing_finished"
            }
        },
        "DoubleSpinBox": {
            "valueChanged": {
                "name": "值改变",
                "signal": "valueChanged",
                "description": "当数值改变时触发",
                "function_name_template": "_on_{control_id}_value_changed"
            },
            "editingFinished": {
                "name": "编辑完成",
                "signal": "editingFinished",
                "description": "当编辑完成时触发",
                "function_name_template": "_on_{control_id}_editing_finished"
            }
        },
        "ListWidget": {
            "itemClicked": {
                "name": "项被点击",
                "signal": "itemClicked",
                "description": "当列表项被点击时触发",
                "function_name_template": "_on_{control_id}_item_clicked"
            },
            "itemDoubleClicked": {
                "name": "项被双击",
                "signal": "itemDoubleClicked",
                "description": "当列表项被双击时触发",
                "function_name_template": "_on_{control_id}_item_double_clicked"
            },
            "itemSelectionChanged": {
                "name": "选择改变",
                "signal": "itemSelectionChanged",
                "description": "当选择改变时触发",
                "function_name_template": "_on_{control_id}_item_selection_changed"
            },
            "currentItemChanged": {
                "name": "当前项改变",
                "signal": "currentItemChanged",
                "description": "当当前项改变时触发",
                "function_name_template": "_on_{control_id}_current_item_changed"
            },
            "itemChanged": {
                "name": "项改变",
                "signal": "itemChanged",
                "description": "当列表项改变时触发",
                "function_name_template": "_on_{control_id}_item_changed"
            }
        },
        "TreeWidget": {
            "itemClicked": {
                "name": "项被点击",
                "signal": "itemClicked",
                "description": "当树项被点击时触发",
                "function_name_template": "_on_{control_id}_item_clicked"
            },
            "itemDoubleClicked": {
                "name": "项被双击",
                "signal": "itemDoubleClicked",
                "description": "当树项被双击时触发",
                "function_name_template": "_on_{control_id}_item_double_clicked"
            },
            "itemSelectionChanged": {
                "name": "选择改变",
                "signal": "itemSelectionChanged",
                "description": "当选择改变时触发",
                "function_name_template": "_on_{control_id}_item_selection_changed"
            },
            "itemChanged": {
                "name": "项改变",
                "signal": "itemChanged",
                "description": "当树项改变时触发",
                "function_name_template": "_on_{control_id}_item_changed"
            },
            "itemExpanded": {
                "name": "项展开",
                "signal": "itemExpanded",
                "description": "当树项展开时触发",
                "function_name_template": "_on_{control_id}_item_expanded"
            },
            "itemCollapsed": {
                "name": "项折叠",
                "signal": "itemCollapsed",
                "description": "当树项折叠时触发",
                "function_name_template": "_on_{control_id}_item_collapsed"
            }
        },
        "TableWidget": {
            "cellClicked": {
                "name": "单元格被点击",
                "signal": "cellClicked",
                "description": "当表格单元格被点击时触发",
                "function_name_template": "_on_{control_id}_cell_clicked"
            },
            "cellDoubleClicked": {
                "name": "单元格被双击",
                "signal": "cellDoubleClicked",
                "description": "当表格单元格被双击时触发",
                "function_name_template": "_on_{control_id}_cell_double_clicked"
            },
            "cellChanged": {
                "name": "单元格改变",
                "signal": "cellChanged",
                "description": "当表格单元格内容改变时触发",
                "function_name_template": "_on_{control_id}_cell_changed"
            },
            "itemSelectionChanged": {
                "name": "选择改变",
                "signal": "itemSelectionChanged",
                "description": "当选择改变时触发",
                "function_name_template": "_on_{control_id}_item_selection_changed"
            },
            "itemChanged": {
                "name": "项改变",
                "signal": "itemChanged",
                "description": "当表格项改变时触发",
                "function_name_template": "_on_{control_id}_item_changed"
            }
        },
        "TabWidget": {
            "currentChanged": {
                "name": "当前页改变",
                "signal": "currentChanged",
                "description": "当当前标签页改变时触发",
                "function_name_template": "_on_{control_id}_current_changed"
            },
            "tabBarClicked": {
                "name": "标签栏被点击",
                "signal": "tabBarClicked",
                "description": "当标签栏被点击时触发",
                "function_name_template": "_on_{control_id}_tab_bar_clicked"
            },
            "tabCloseRequested": {
                "name": "标签关闭请求",
                "signal": "tabCloseRequested",
                "description": "当标签关闭请求时触发",
                "function_name_template": "_on_{control_id}_tab_close_requested"
            }
        },
        "ProgressBar": {
            "valueChanged": {
                "name": "值改变",
                "signal": "valueChanged",
                "description": "当进度条的值改变时触发",
                "function_name_template": "_on_{control_id}_value_changed"
            }
        },
        "DateEdit": {
            "dateChanged": {
                "name": "日期改变",
                "signal": "dateChanged",
                "description": "当日期改变时触发",
                "function_name_template": "_on_{control_id}_date_changed"
            },
            "editingFinished": {
                "name": "编辑完成",
                "signal": "editingFinished",
                "description": "当编辑完成时触发",
                "function_name_template": "_on_{control_id}_editing_finished"
            }
        },
        "TimeEdit": {
            "timeChanged": {
                "name": "时间改变",
                "signal": "timeChanged",
                "description": "当时间改变时触发",
                "function_name_template": "_on_{control_id}_time_changed"
            },
            "editingFinished": {
                "name": "编辑完成",
                "signal": "editingFinished",
                "description": "当编辑完成时触发",
                "function_name_template": "_on_{control_id}_editing_finished"
            }
        },
        "DateTimeEdit": {
            "dateTimeChanged": {
                "name": "日期时间改变",
                "signal": "dateTimeChanged",
                "description": "当日期时间改变时触发",
                "function_name_template": "_on_{control_id}_date_time_changed"
            },
            "editingFinished": {
                "name": "编辑完成",
                "signal": "editingFinished",
                "description": "当编辑完成时触发",
                "function_name_template": "_on_{control_id}_editing_finished"
            }
        }
    }
    
    @classmethod
    def get_events_for_control(cls, control_type: str) -> List[Tuple[str, Dict]]:
        """
        获取指定控件类型支持的所有事件
        
        Args:
            control_type: 控件类型（如"Button"、"Label"等）
            
        Returns:
            List[Tuple[str, Dict]]: 事件列表，每个元素为(事件ID, 事件信息字典)
        """
        events = []
        
        # 添加通用事件
        for event_id, event_info in cls.COMMON_EVENTS.items():
            events.append((event_id, event_info))
        
        # 添加控件特定事件
        if control_type in cls.CONTROL_SPECIFIC_EVENTS:
            for event_id, event_info in cls.CONTROL_SPECIFIC_EVENTS[control_type].items():
                events.append((event_id, event_info))
        
        return events
    
    @classmethod
    def get_event_function_name(cls, control_id: str, event_id: str) -> str:
        """
        获取事件处理函数的名称
        
        Args:
            control_id: 控件ID
            event_id: 事件ID
            
        Returns:
            str: 事件处理函数名称
        """
        # 查找事件信息
        event_info = None
        
        # 先查找通用事件
        if event_id in cls.COMMON_EVENTS:
            event_info = cls.COMMON_EVENTS[event_id]
        else:
            # 查找控件特定事件
            for control_type, events in cls.CONTROL_SPECIFIC_EVENTS.items():
                if event_id in events:
                    event_info = events[event_id]
                    break
        
        if event_info and "function_name_template" in event_info:
            template = event_info["function_name_template"]
            # 将control_id转换为有效的变量名
            import re
            var_name = re.sub(r'[^a-zA-Z0-9_]', '_', control_id)
            if var_name and var_name[0].isdigit():
                var_name = f"control_{var_name}"
            if not var_name:
                var_name = "control"
            return template.format(control_id=var_name)
        
        # 默认函数名
        import re
        var_name = re.sub(r'[^a-zA-Z0-9_]', '_', control_id)
        if var_name and var_name[0].isdigit():
            var_name = f"control_{var_name}"
        if not var_name:
            var_name = "control"
        return f"_on_{var_name}_{event_id}"
    
    @classmethod
    def get_event_display_name(cls, event_id: str, control_type: str = None) -> str:
        """
        获取事件的显示名称
        
        Args:
            event_id: 事件ID
            control_type: 控件类型（可选）
            
        Returns:
            str: 事件显示名称
        """
        # 先查找通用事件
        if event_id in cls.COMMON_EVENTS:
            return cls.COMMON_EVENTS[event_id].get("name", event_id)
        
        # 查找控件特定事件
        if control_type and control_type in cls.CONTROL_SPECIFIC_EVENTS:
            if event_id in cls.CONTROL_SPECIFIC_EVENTS[control_type]:
                return cls.CONTROL_SPECIFIC_EVENTS[control_type][event_id].get("name", event_id)
        
        return event_id

