#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
事件函数文件生成器模块
用于生成和管理事件处理函数文件（窗口名称_function.py）
"""

from typing import Dict, List, Optional
from .control_pyside import Control, ControlManager
from .event_manager_pyside import EventManager
from .event_manager_pyside import EventManager


class EventFunctionGenerator:
    """
    事件函数文件生成器类
    负责生成和管理事件处理函数文件
    """
    
    def __init__(self, control_manager: ControlManager):
        """
        初始化事件函数生成器
        
        Args:
            control_manager: 控件管理器实例
        """
        self.control_manager = control_manager
        self.window_name = "GeneratedWindow"
    
    def set_window_name(self, name: str):
        """设置窗口名称"""
        self.window_name = name or "GeneratedWindow"
    
    def generate_function_file_content(self) -> str:
        """
        生成事件函数文件内容
        
        Returns:
            str: 事件函数文件的完整内容
        """
        lines = [
            f"# -*- coding: utf-8 -*-",
            f"\"\"\"",
            f"事件处理函数模块",
            f"窗口: {self.window_name}",
            f"此文件包含所有控件的事件处理函数实现",
            f"\"\"\"",
            "",
            "# 导入必要的模块",
            "from PySide6.QtCore import Qt",
            "from PySide6.QtGui import QMouseEvent, QKeyEvent, QWheelEvent",
            "",
        ]
        
        # 获取所有控件
        controls = list(self.control_manager.controls.values())
        
        # 为每个控件生成事件处理函数
        for control in controls:
            bound_events = getattr(control, 'events', {})
            if not isinstance(bound_events, dict) or not bound_events:
                continue
            
            control_id = self._to_valid_variable_name(control.id)
            control_type = control.control_type
            
            # 为每个绑定的事件生成函数
            for event_id, is_bound in bound_events.items():
                if not is_bound:
                    continue
                
                function_name = EventManager.get_event_function_name(control.id, event_id)
                
                # 获取事件信息
                event_info = None
                if event_id in EventManager.COMMON_EVENTS:
                    event_info = EventManager.COMMON_EVENTS[event_id]
                elif control_type in EventManager.CONTROL_SPECIFIC_EVENTS:
                    if event_id in EventManager.CONTROL_SPECIFIC_EVENTS[control_type]:
                        event_info = EventManager.CONTROL_SPECIFIC_EVENTS[control_type][event_id]
                
                if event_info:
                    signal = event_info.get("signal")
                    description = event_info.get("description", "")
                    event_name = event_info.get("name", event_id)
                    
                    lines.append("")
                    
                    # 根据信号类型生成不同的函数签名
                    if signal:
                        # 有信号的事件，根据信号类型确定参数
                        if signal in ["clicked", "pressed", "released", "toggled"]:
                            # 无参数信号
                            lines.append(f"def {function_name}(window, control):")
                        elif signal in ["textChanged", "textEdited", "currentTextChanged"]:
                            # 文本信号，传递文本参数
                            lines.append(f"def {function_name}(window, control, text):")
                        elif signal in ["valueChanged", "stateChanged"]:
                            # 值变化信号，传递值参数
                            lines.append(f"def {function_name}(window, control, value):")
                        elif signal in ["currentIndexChanged"]:
                            # 索引变化信号
                            lines.append(f"def {function_name}(window, control, index):")
                        elif signal in ["itemClicked", "itemDoubleClicked", "cellClicked", "cellDoubleClicked"]:
                            # 项点击信号
                            lines.append(f"def {function_name}(window, control, item):")
                        elif signal in ["dateChanged", "timeChanged", "dateTimeChanged"]:
                            # 日期时间信号
                            lines.append(f"def {function_name}(window, control, date):")
                        else:
                            # 默认情况
                            lines.append(f"def {function_name}(window, control, *args, **kwargs):")
                    else:
                        # 需要重写的事件处理函数，传递event参数
                        lines.append(f"def {function_name}(window, control, event):")
                    
                    lines.append(f"    \"\"\"")
                    lines.append(f"    {event_name}事件处理函数")
                    lines.append(f"    ")
                    lines.append(f"    参数:")
                    lines.append(f"        window: 窗口实例")
                    lines.append(f"        control: 控件实例")
                    if signal:
                        if signal in ["textChanged", "textEdited", "currentTextChanged"]:
                            lines.append(f"        text: 文本内容")
                        elif signal in ["valueChanged", "stateChanged"]:
                            lines.append(f"        value: 值")
                        elif signal in ["currentIndexChanged"]:
                            lines.append(f"        index: 索引")
                        elif signal in ["itemClicked", "itemDoubleClicked", "cellClicked", "cellDoubleClicked"]:
                            lines.append(f"        item: 项对象")
                        elif signal in ["dateChanged", "timeChanged", "dateTimeChanged"]:
                            lines.append(f"        date: 日期时间对象")
                    else:
                        lines.append(f"        event: 事件对象")
                    lines.append(f"    ")
                    lines.append(f"    控件: {control.name} ({control_type})")
                    lines.append(f"    事件: {event_name}")
                    lines.append(f"    描述: {description}")
                    lines.append(f"    \"\"\"")
                    # 注释：此处不再添加TODO，因为后续会生成实际的示例代码
                    
                    # 生成示例代码
                    if signal:
                        if signal in ["textChanged", "textEdited", "currentTextChanged"]:
                            lines.append(f"    print(f\"{event_name}事件触发: {{control.objectName()}}, 文本: {{text}}\")")
                        elif signal in ["valueChanged", "stateChanged"]:
                            lines.append(f"    print(f\"{event_name}事件触发: {{control.objectName()}}, 值: {{value}}\")")
                        elif signal in ["currentIndexChanged"]:
                            lines.append(f"    print(f\"{event_name}事件触发: {{control.objectName()}}, 索引: {{index}}\")")
                        elif signal in ["itemClicked", "itemDoubleClicked", "cellClicked", "cellDoubleClicked"]:
                            lines.append(f"    print(f\"{event_name}事件触发: {{control.objectName()}}, 项: {{item}}\")")
                        elif signal in ["dateChanged", "timeChanged", "dateTimeChanged"]:
                            lines.append(f"    print(f\"{event_name}事件触发: {{control.objectName()}}, 日期: {{date}}\")")
                        else:
                            lines.append(f"    print(f\"{event_name}事件触发: {{control.objectName()}}\")")
                    else:
                        # 需要重写的事件处理函数
                        if event_id == "mousePressEvent":
                            lines.append(f"    pos = event.pos()")
                            lines.append(f"    print(f\"鼠标按下位置: {{pos.x()}}, {{pos.y()}}\")")
                        elif event_id == "mouseReleaseEvent":
                            lines.append(f"    pos = event.pos()")
                            lines.append(f"    print(f\"鼠标释放位置: {{pos.x()}}, {{pos.y()}}\")")
                        elif event_id == "mouseMoveEvent":
                            lines.append(f"    pos = event.pos()")
                            lines.append(f"    print(f\"鼠标移动位置: {{pos.x()}}, {{pos.y()}}\")")
                        elif event_id == "mouseDoubleClickEvent":
                            lines.append(f"    pos = event.pos()")
                            lines.append(f"    print(f\"鼠标双击位置: {{pos.x()}}, {{pos.y()}}\")")
                        elif event_id == "wheelEvent":
                            lines.append(f"    from PySide6.QtCore import QPoint")
                            lines.append(f"    angle = event.angleDelta().y()")
                            lines.append(f"    print(f\"滚轮角度: {{angle}}\")")
                        elif event_id == "keyPressEvent":
                            lines.append(f"    from PySide6.QtCore import Qt")
                            lines.append(f"    key = event.key()")
                            lines.append(f"    print(f\"按键: {{key}}\")")
                        elif event_id == "keyReleaseEvent":
                            lines.append(f"    from PySide6.QtCore import Qt")
                            lines.append(f"    key = event.key()")
                            lines.append(f"    print(f\"释放按键: {{key}}\")")
                        elif event_id == "focusInEvent":
                            lines.append(f"    print('控件获得焦点')")
                        elif event_id == "focusOutEvent":
                            lines.append(f"    print('控件失去焦点')")
                        elif event_id == "enterEvent":
                            lines.append(f"    print('鼠标进入控件')")
                        elif event_id == "leaveEvent":
                            lines.append(f"    print('鼠标离开控件')")
                        elif event_id == "resizeEvent":
                            lines.append(f"    size = event.size()")
                            lines.append(f"    print(f'控件大小改变: {{size.width()}}x{{size.height()}}')")
                        elif event_id == "moveEvent":
                            lines.append(f"    pos = event.pos()")
                            lines.append(f"    print(f'控件位置改变: {{pos.x()}}, {{pos.y()}}')")
                        elif event_id == "paintEvent":
                            lines.append(f"    print('控件需要重绘')")
        
        return "\n".join(lines)
    
    def _to_valid_variable_name(self, control_id: str) -> str:
        """将控件ID转换为有效的Python变量名"""
        import re
        var_name = re.sub(r'[^a-zA-Z0-9_]', '_', control_id)
        if var_name and var_name[0].isdigit():
            var_name = f"control_{var_name}"
        if not var_name:
            var_name = "control"
        return var_name
    
    def get_function_file_name(self, existing_names: set = None) -> str:
        """
        获取事件函数文件名（自动处理重名）
        
        Args:
            existing_names: 已存在的文件名集合，用于避免重名
            
        Returns:
            唯一的文件名
        """
        base_name = f"{self.window_name}_function.py"
        
        if existing_names is None:
            return base_name
        
        if base_name not in existing_names:
            return base_name
        
        # 如果重名，自动添加序号
        counter = 1
        while True:
            new_name = f"{self.window_name}_function_{counter}.py"
            if new_name not in existing_names:
                return new_name
            counter += 1
    
    def update_function_in_file(self, control_id: str, event_id: str, function_code: str) -> str:
        """
        更新事件函数文件中的特定函数
        
        Args:
            control_id: 控件ID
            event_id: 事件ID
            function_code: 新的函数代码
            
        Returns:
            str: 更新后的完整文件内容
        """
        # 获取当前文件内容（如果存在）
        # 这里需要从项目管理器获取，暂时返回新生成的内容
        current_content = self.generate_function_file_content()
        
        # 查找并替换函数
        function_name = EventManager.get_event_function_name(control_id, event_id)
        lines = current_content.splitlines()
        
        new_lines = []
        in_function = False
        function_start = -1
        function_indent = 0
        
        for i, line in enumerate(lines):
            # 检查是否是目标函数
            if line.strip().startswith(f"def {function_name}("):
                in_function = True
                function_start = i
                # 计算缩进
                function_indent = len(line) - len(line.lstrip())
                new_lines.append(line)
                continue
            
            if in_function:
                # 检查函数是否结束（遇到下一个def或空行后遇到def）
                current_indent = len(line) - len(line.lstrip()) if line.strip() else function_indent + 1
                if line.strip() and current_indent <= function_indent and not line.strip().startswith('#'):
                    # 函数结束，插入新代码
                    new_lines.extend(function_code.splitlines())
                    in_function = False
                    new_lines.append(line)
                elif not line.strip() and i > function_start + 5:
                    # 空行且已经过了函数定义，可能是函数结束
                    new_lines.append(line)
                else:
                    # 跳过旧代码
                    continue
            else:
                new_lines.append(line)
        
        # 如果函数未找到，添加到文件末尾
        if in_function or function_start == -1:
            new_lines.extend(function_code.splitlines())
        
        return "\n".join(new_lines)

