#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
设计画布模块
实现可视化设计画布，支持控件的拖拽、选择、移动和调整大小
"""

"""
设计画布模块
实现可视化设计画布，支持控件的拖拽、选择、移动和调整大小
这是可视化编程工具的核心组件，用户在此画布上进行界面设计
"""

# 导入类型提示
from typing import Optional, Tuple, List
# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QWidget,  # 基础窗口部件
    QVBoxLayout,  # 垂直布局
    QHBoxLayout,  # 水平布局
    QLabel,  # 标签
    QFrame,  # 框架
    QPushButton,  # 按钮
    QScrollArea,  # 滚动区域
    QApplication,  # 应用程序类
    QSizePolicy,  # 大小策略
    QMenu,  # 菜单
    QDialog,  # 对话框
    QFormLayout,  # 表单布局
    QTextEdit,  # 文本编辑器
    QDialogButtonBox,  # 对话框按钮盒
    QInputDialog  # 输入对话框
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import QAction  # 动作类
# 导入菜单编辑器
from module.menu_editor_pyside import MenusManagerDialog  # 菜单管理器对话框
from module.window_class_dialog import WindowClassDialog  # 窗口类管理对话框
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QRect,  # 矩形
    QPoint,  # 点坐标
    QSize,  # 大小
    Signal,  # 信号定义
    QMimeData  # MIME数据，用于拖放操作
)
# 导入PySide6图形绘制相关类
from PySide6.QtGui import (
    QPainter,  # 绘图器
    QPen,  # 画笔
    QBrush,  # 画刷
    QColor,  # 颜色
    QDrag,  # 拖放操作
    QFont,  # 字体
    QPixmap  # 像素图
)

# 导入自定义模块
from .control_pyside import Control, ControlManager  # 控件类和控件管理器
from .control_library_pyside import ControlLibrary  # 控件库


class DesignCanvas(QWidget):
    """
    设计画布类
    提供可视化设计界面，支持控件的拖拽、选择、移动、调整大小等操作
    这是用户进行界面设计的主要工作区域
    """
    
    # ========== 信号定义 ==========
    # 控件被选中时发出信号，参数为控件ID
    control_selected = Signal(str)
    # 控件被移动时发出信号，参数为控件ID、X偏移量、Y偏移量
    control_moved = Signal(str, int, int)
    # 控件被调整大小时发出信号，参数为控件ID、调整手柄、X偏移量、Y偏移量
    control_resized = Signal(str, str, int, int)
    # 画布大小被调整时发出信号，参数为新宽度、新高度
    canvas_resized = Signal(int, int)
    # 控件被添加时发出信号，参数为控件ID
    control_added = Signal(str)
    # 控件被删除时发出信号，参数为控件ID
    control_deleted = Signal(str)
    
    def __init__(self, parent=None):
        """
        初始化设计画布
        创建画布的所有必要属性和状态，设置画布的基本行为
        
        Args:
            parent: 父窗口部件
        """
        # ========== 调用父类构造函数 ==========
        # 调用QWidget的构造函数，初始化基础窗口部件
        super().__init__(parent)
        
        # ========== 画布基本设置 ==========
        # 设置画布的最小尺寸：宽度500像素，高度300像素
        self.setMinimumSize(500, 300)
        # 设置画布的大小策略：水平和垂直方向都可以扩展（占据可用空间）
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        # 设置画布接受拖放操作（允许从控件面板拖拽控件到画布）
        self.setAcceptDrops(True)
        # 设置画布的焦点策略：强焦点（可以通过点击或Tab键获取焦点）
        self.setFocusPolicy(Qt.StrongFocus)
        
        # 画布尺寸属性
        self.canvas_width = 500
        self.canvas_height = 300
        
        # 缩放相关属性
        self.scale_factor = 1.0  # 缩放因子，1.0表示原始大小
        self.min_scale = 0.5  # 最小缩放比例
        self.max_scale = 3.0  # 最大缩放比例
        self.scale_step = 0.1  # 缩放步长
        
        # 画布选中状态
        self._canvas_selected = False
        
        # 控件管理器
        self.control_manager = ControlManager()
        
        # 画布状态
        self.grid_size = 10  # 网格大小
        self.show_grid = True  # 是否显示网格
        self.snap_to_grid = True  # 是否启用网格对齐
        self.bg_color = QColor(255, 255, 255)  # 画布背景颜色
        
        # 窗口属性
        self.window_name = "启动窗口"  # 窗口名称
        self.window_comment = "在程序启动后自动调入本窗口"  # 窗口备注
        self.window_left = 50  # 窗口左边位置
        self.window_top = 50  # 窗口顶边位置
        self.window_width = 380  # 窗口宽度
        self.window_height = 250  # 窗口高度
        self.window_visible = True  # 窗口是否可视
        self.window_enabled = True  # 窗口是否启用
        self.window_cursor = "默认型"  # 鼠标指针样式
        self.window_title = "启动窗口"  # 窗口标题
        self.window_border = "普通固定边框"  # 窗口边框样式
        self.window_bg_color = "默认底色"  # 窗口底色
        self.window_bg_image = ""  # 窗口底图
        self.window_bg_image_mode = "图片平铺"  # 底图方式
        self.window_bg_music = ""  # 背景音乐
        self.window_music_play_count = "循环播放"  # 播放次数
        self.window_show_control_button = True  # 控制按钮
        self.window_show_maximize_button = False  # 最大化按钮
        self.window_show_minimize_button = True  # 最小化按钮
        self.window_position = "居中"  # 窗口位置
        self.window_movable = True  # 可否移动
        self.window_icon = ""  # 窗口图标
        self.window_menus = []  # 菜单列表
        self.window_enter_move_focus = False  # 回车下移焦点
        self.window_esc_close = True  # Esc键关闭
        self.window_f1_help = False  # F1键打开帮助
        self.window_help_file = ""  # 帮助文件名
        self.window_help_flag = 0  # 帮助标志值
        self.window_show_in_taskbar = True  # 在任务条中显示
        self.window_free_move = False  # 随意移动
        self.window_shape = "矩形"  # 外形
        self.window_always_on_top = False  # 总在最前
        self.window_keep_title_active = False  # 保持标题条激活
        self.window_class_name = ""  # 窗口类名
        
        # ========== 拖拽和调整大小状态 ==========
        # 是否正在拖拽控件（True表示正在拖拽，False表示未拖拽）
        self.dragging = False
        # 是否正在调整控件大小（True表示正在调整大小，False表示未调整）
        self.resizing = False
        # 拖拽开始时的鼠标位置（用于计算拖拽距离）
        self.drag_start_pos = QPoint()
        # 拖拽开始时的控件矩形（用于计算控件移动）
        self.drag_start_rect = QRect()
        # 当前调整大小的手柄位置（如"top_left"、"bottom_right"等，None表示未调整大小）
        self.resize_handle = None
        
        # 窗口外观设置
        self.window_title = "可视化设计页面"
        self.window_icon = "👤"
        self.show_window_frame = True  # 是否显示窗口框架
        self.window_menus = []
        self.window_dropdown_menus = []
        self.active_page = "页面1"
        self.active_index = 0
        self._menu_item_rects = []
        self._dropdown_rects = []
        self.title_bar_height = 28
        
        # ========== 选择和多选状态 ==========
        # 控件管理器中当前选中的控件ID（None表示未选中控件）
        self.control_manager.selected_control_id = None
        # 是否抑制画布大小改变信号（True表示不发送信号，用于内部调整）
        self._suppress_canvas_resized = False
        # 是否正在进行框选操作（True表示正在框选，False表示未框选）
        self._rubber_selecting = False
        # 框选开始时的鼠标位置（用于绘制框选矩形）
        self._rubber_start = QPoint()
        # 框选矩形的范围（用于确定选中的控件）
        self._rubber_rect = QRect()
        # 框选是否为累加模式（True表示添加到已有选择，False表示替换选择）
        self._rubber_additive = False
        # 多选拖拽的控件ID列表（用于同时移动多个控件）
        self._multi_drag_ids = []
        # 多选拖拽开始时各控件的初始位置（字典：控件ID -> 初始矩形）
        self._multi_drag_start = {}
    
    def set_control_manager(self, control_manager: ControlManager):
        """设置控件管理器"""
        self.control_manager = control_manager
    
    def get_control_manager(self) -> ControlManager:
        """获取控件管理器"""
        return self.control_manager
    
    def set_window_title(self, title: str):
        """设置窗口标题"""
        self.window_title = title
        self.update()
    
    def set_window_property(self, property_name: str, value):
        """通用的窗口属性设置方法
        
        Args:
            property_name: 属性名称
            value: 属性值
        """
        if hasattr(self, property_name):
            setattr(self, property_name, value)
            if property_name == 'window_bg_color':
                m = {
                    '默认底色': None,
                    '黑色': QColor(0, 0, 0),
                    '深灰': QColor(80, 80, 80),
                    '灰色': QColor(160, 160, 160),
                    '浅灰': QColor(220, 220, 220),
                    '白色': QColor(255, 255, 255),
                    '红色': QColor(220, 20, 60),
                    '绿色': QColor(34, 139, 34),
                    '蓝色': QColor(30, 144, 255)
                }
                c = m.get(value)
                if c is not None:
                    self.bg_color = c
            # 对于某些属性，需要更新界面
            if property_name in ['window_title', 'window_icon', 'window_bg_color', 
                                'window_bg_image', 'window_bg_image_mode']:
                self.update()
    
    def set_show_window_frame(self, show: bool):
        """设置是否显示窗口框架"""
        self.show_window_frame = show
        self.update()
    
    def set_canvas_size(self, width: int, height: int):
        """设置画布大小"""
        width = max(width, 200)
        height = max(height, 200)
        
        # 限制画布最大宽度，确保不会超过控件库
        max_width = 1000
        if width > max_width:
            width = max_width
            
        self.canvas_width = width
        self.canvas_height = height
        self.setMinimumSize(width, height)
        self.resize(width, height)
        self.update()
        if not self._suppress_canvas_resized:
            try:
                self.canvas_resized.emit(width, height)
            except Exception:
                pass

    def begin_state_restore(self):
        self._suppress_canvas_resized = True
    def end_state_restore(self):
        self._suppress_canvas_resized = False
    
    def get_canvas_size(self) -> Tuple[int, int]:
        """获取画布大小"""
        return self.canvas_width, self.canvas_height
    
    def get_canvas_width(self) -> int:
        """获取画布宽度"""
        return self.canvas_width
    
    def get_canvas_height(self) -> int:
        """获取画布高度"""
        return self.canvas_height
    
    def set_canvas_width(self, width: int):
        """设置画布宽度"""
        self.set_canvas_size(width, self.canvas_height)
    
    def set_canvas_height(self, height: int):
        """设置画布高度"""
        self.set_canvas_size(self.canvas_width, height)
    
    def set_canvas_bg_color(self, color: QColor):
        """设置画布背景颜色"""
        self.bg_color = color
        self.update()
    
    def set_show_grid(self, show: bool):
        """设置是否显示网格"""
        self.show_grid = show
        self.update()
    
    def set_grid_size(self, size: int):
        """设置网格大小"""
        self.grid_size = size
        self.update()
    
    def set_snap_to_grid(self, snap: bool):
        """设置是否启用网格对齐"""
        self.snap_to_grid = snap
    
    # 窗口属性设置方法
    def set_window_name(self, name: str):
        """设置窗口名称"""
        self.window_name = name
        self.update()
    
    def set_window_comment(self, comment: str):
        """设置窗口备注"""
        self.window_comment = comment
    
    def set_window_left(self, left: int):
        """设置窗口左边位置"""
        self.window_left = left
    
    def set_window_top(self, top: int):
        """设置窗口顶边位置"""
        self.window_top = top
    
    def set_window_width(self, width: int):
        """设置窗口宽度"""
        self.window_width = width
    
    def set_window_height(self, height: int):
        """设置窗口高度"""
        self.window_height = height
    
    def set_window_visible(self, visible: bool):
        """设置窗口是否可视"""
        self.window_visible = visible
    
    def set_window_enabled(self, enabled: bool):
        """设置窗口是否启用"""
        self.window_enabled = enabled
    
    def set_window_cursor(self, cursor: str):
        """设置鼠标指针样式"""
        self.window_cursor = cursor
    
    def set_window_border(self, border: str):
        """设置窗口边框样式"""
        self.window_border = border
    
    def set_window_bg_color(self, color: str):
        """设置窗口底色"""
        self.window_bg_color = color
    
    def set_window_bg_image(self, image: str):
        """设置窗口底图"""
        self.window_bg_image = image
    
    def set_window_bg_image_mode(self, mode: str):
        """设置底图方式"""
        self.window_bg_image_mode = mode
    
    def set_window_bg_music(self, music: str):
        """设置背景音乐"""
        self.window_bg_music = music
    
    def set_window_music_play_count(self, count: str):
        """设置播放次数"""
        self.window_music_play_count = count
    
    def set_window_show_control_button(self, show: bool):
        """设置控制按钮"""
        self.window_show_control_button = show
    
    def set_window_show_maximize_button(self, show: bool):
        """设置最大化按钮"""
        self.window_show_maximize_button = show
    
    def set_window_show_minimize_button(self, show: bool):
        """设置最小化按钮"""
        self.window_show_minimize_button = show
    
    def set_window_position(self, position: str):
        """设置窗口位置"""
        self.window_position = position
    
    def set_window_movable(self, movable: bool):
        """设置可否移动"""
        self.window_movable = movable
    
    def set_window_icon(self, icon: str):
        """设置窗口图标"""
        self.window_icon = icon
    
    def set_window_enter_move_focus(self, move: bool):
        """设置回车下移焦点"""
        self.window_enter_move_focus = move
    
    def set_window_esc_close(self, close: bool):
        """设置Esc键关闭"""
        self.window_esc_close = close
    
    def set_window_f1_help(self, help: bool):
        """设置F1键打开帮助"""
        self.window_f1_help = help
    
    def set_window_help_file(self, file: str):
        """设置帮助文件名"""
        self.window_help_file = file
    
    def set_window_help_flag(self, flag: int):
        """设置帮助标志值"""
        self.window_help_flag = flag
    
    def set_window_show_in_taskbar(self, show: bool):
        """设置在任务条中显示"""
        self.window_show_in_taskbar = show
    
    def set_window_free_move(self, free: bool):
        """设置随意移动"""
        self.window_free_move = free
    
    def set_window_shape(self, shape: str):
        """设置外形"""
        self.window_shape = shape
    
    def set_window_always_on_top(self, top: bool):
        """设置总在最前"""
        self.window_always_on_top = top
    
    def set_window_keep_title_active(self, active: bool):
        """设置保持标题条激活"""
        self.window_keep_title_active = active
    
    def set_window_class_name(self, name: str):
        """设置窗口类名"""
        self.window_class_name = name
    
    def add_control(self, control_type_or_control, x: int = None, y: int = None) -> Optional[Control]:
        """
        添加控件到画布
        
        Args:
            control_type_or_control: 控件类型字符串或Control对象
            x: X坐标，如果为None则使用默认位置
            y: Y坐标，如果为None则使用默认位置
            
        Returns:
            新创建的控件，如果创建失败则返回None
        """
        # ========== 处理传入Control对象的情况 ==========
        # 检查传入的参数是否为Control对象（而不是控件类型字符串）
        if isinstance(control_type_or_control, Control):
            # 如果传入的是Control对象，直接使用该对象
            control = control_type_or_control
            # 检查是否指定了新位置（x和y都不为None）
            if x is not None and y is not None:
                # 如果指定了新位置，更新控件的位置坐标
                control.x = x
                control.y = y
            
            # ========== 将控件添加到控件管理器 ==========
            # 将控件添加到控件管理器的控件字典中（使用控件ID作为键）
            self.control_manager.controls[control.id] = control
            # 发出控件添加信号，通知其他组件有新控件被添加
            self.control_added.emit(control.id)
            # 更新画布显示，使新控件可见
            self.update()
            # 返回创建的控件对象
            return control
        
        # ========== 处理位置参数 ==========
        # 检查是否未指定位置（x或y为None）
        if x is None or y is None:
            # 如果没有指定位置，使用画布中心位置作为默认位置
            # 计算画布中心X坐标：画布宽度减去控件宽度(100)后除以2
            x = (self.width() - 100) // 2
            # 计算画布中心Y坐标：画布高度减去控件高度(30)后除以2
            y = (self.height() - 30) // 2
        
        # ========== 网格对齐处理 ==========
        # 检查是否启用了网格对齐功能
        if self.snap_to_grid:
            # 如果启用网格对齐，将坐标对齐到网格
            # X坐标对齐：先除以网格大小并四舍五入，再乘以网格大小
            x = round(x / self.grid_size) * self.grid_size
            # Y坐标对齐：先除以网格大小并四舍五入，再乘以网格大小
            y = round(y / self.grid_size) * self.grid_size
        
        # ========== 创建新控件 ==========
        # 获取控件类型（此时应该是字符串）
        control_type = control_type_or_control
        # 通过控件管理器创建新控件，传入控件类型和位置
        control = self.control_manager.add_control(control_type, x, y)
        
        # ========== 处理控件创建成功后的操作 ==========
        # 检查控件是否创建成功（不为None）
        if control:
            # ========== 页面属性赋值 ==========
            # 检查控件是否已有page属性（所属页面）
            if not control.properties.get('page'):
                # 如果控件没有page属性，执行以下操作
                # 检查窗口菜单列表是否存在，且当前激活索引在有效范围内
                if self.window_menus and 0 <= getattr(self, 'active_index', 0) < len(self.window_menus):
                    # 如果菜单列表存在且索引有效，使用菜单列表中对应索引的页面名称
                    control.properties['page'] = self.window_menus[self.active_index]
                else:
                    # 如果菜单列表不存在或索引无效，使用默认的活动页面名称
                    control.properties['page'] = getattr(self, 'active_page', '页面1')
            
            # ========== 发出信号并更新画布 ==========
            # 发出控件添加信号，通知其他组件有新控件被添加
            self.control_added.emit(control.id)
            # 更新画布显示，使新控件可见
            self.update()
        
        # 返回创建的控件对象（可能为None如果创建失败）
        return control
    
    def delete_selected_control(self) -> bool:
        """
        删除选中的控件（支持批量删除）
        可以删除单个或多个选中的控件
        
        Returns:
            bool: 是否成功删除了至少一个控件（True表示成功，False表示没有控件被删除）
        """
        # ========== 获取选中的控件ID列表 ==========
        # 从控件管理器中获取所有选中控件的ID列表
        selected_ids = self.control_manager.get_selected_ids()
        # 检查是否有选中的控件（列表不为空）
        if selected_ids:
            # ========== 批量删除所有选中的控件 ==========
            # 初始化删除计数器，记录成功删除的控件数量
            deleted_count = 0
            # 遍历选中控件ID列表
            # 使用切片[:]创建列表副本，避免在遍历时修改原列表导致的问题
            for control_id in selected_ids[:]:
                # 通过控件管理器删除指定ID的控件
                success = self.control_manager.delete_control(control_id)
                # 检查删除是否成功
                if success:
                    # 如果删除成功，执行以下操作
                    # 增加删除计数器
                    deleted_count += 1
                    # 发出控件删除信号，通知其他组件有控件被删除
                    self.control_deleted.emit(control_id)
            
            # ========== 处理删除后的状态更新 ==========
            # 检查是否有控件被成功删除（计数器大于0）
            if deleted_count > 0:
                # 如果有控件被删除，执行以下操作
                # 清空控件管理器的选中状态（因为控件已被删除，不能再被选中）
                self.control_manager.clear_selection()
                # 更新画布显示，移除已删除的控件
                self.update()
                # 返回True表示删除成功
                return True
        
        # 如果没有选中的控件，或删除失败，返回False
        return False
    
    def select_control(self, control_id: Optional[str]):
        """
        选中控件或画布
        设置当前选中的控件或画布，并更新显示
        
        Args:
            control_id: 控件ID，如果为"canvas"则表示选中画布，None表示取消选中
        """
        # ========== 检查是否为选中画布 ==========
        # 检查控件ID是否为"canvas"（选中画布本身）
        if control_id == "canvas":
            # ========== 选中画布 ==========
            # 在控件管理器中取消控件选择（传递None）
            self.control_manager.select_control(None)
            # 发送画布选中信号，通知其他组件画布被选中
            self.control_selected.emit("canvas")
            # 设置画布选中状态标志为True
            self._canvas_selected = True
        else:
            # ========== 选中控件 ==========
            # 在控件管理器中选中指定的控件
            self.control_manager.select_control(control_id)
            # 设置画布选中状态标志为False（控件被选中）
            self._canvas_selected = False
            # 检查控件ID是否存在
            if control_id:
                # 如果控件ID存在，发送控件选中信号，通知其他组件控件被选中
                self.control_selected.emit(control_id)
        
        # ========== 更新画布显示 ==========
        # 更新画布显示，刷新选中状态（显示选中控件的边框和调整句柄）
        self.update()
    
    def get_selected_control(self) -> Optional[Control]:
        """
        获取选中的控件对象
        
        Returns:
            Optional[Control]: 当前选中的控件对象，如果未选中控件则返回None
        """
        # ========== 从控件管理器获取选中控件 ==========
        # 调用控件管理器的get_selected_control方法，获取当前选中的控件对象
        return self.control_manager.get_selected_control()
    
    def clear_all_controls(self):
        """
        清除画布上的所有控件
        删除所有已添加的控件，清空画布
        """
        # ========== 清空所有控件 ==========
        # 调用控件管理器的clear_all方法，删除所有控件
        self.control_manager.clear_all()
        # ========== 更新画布显示 ==========
        # 更新画布显示，刷新清空后的画布（只显示背景和网格）
        self.update()

    def clear_controls(self):
        """
        兼容旧方法名称，清除所有控件
        此方法是为了向后兼容，实际调用clear_all_controls方法
        """
        # ========== 调用清除所有控件的方法 ==========
        # 调用clear_all_controls方法，执行清除所有控件的操作
        self.clear_all_controls()
    
    def paintEvent(self, event):
        """
        绘制事件处理函数
        当画布需要重绘时调用，绘制画布的所有视觉元素
        
        Args:
            event: 绘制事件对象，包含需要重绘的区域信息
        """
        # ========== 创建绘图器 ==========
        # 创建QPainter对象，用于在画布上进行绘制操作
        painter = None
        try:
            painter = QPainter(self)
            # 检查绘图器是否成功初始化
            if not painter.isActive():
                return
            
            # 设置渲染提示：启用抗锯齿（使绘制的内容更平滑）
            painter.setRenderHint(QPainter.Antialiasing)
            
            # ========== 应用缩放变换 ==========
            # 应用缩放变换到绘图器，使所有绘制内容按照缩放因子缩放
            # 例如：scale_factor=1.5时，所有绘制内容放大1.5倍
            # 确保缩放因子有效（大于0）
            if hasattr(self, 'scale_factor') and self.scale_factor > 0:
                painter.scale(self.scale_factor, self.scale_factor)
            
            # ========== 绘制背景 ==========
            # 绘制画布的背景（纯色或图片背景）
            try:
                self._draw_background(painter)
            except Exception:
                pass  # 忽略绘制背景时的异常，继续绘制其他内容
            
            # ========== 绘制网格 ==========
            # 检查是否启用网格显示
            if hasattr(self, 'show_grid') and self.show_grid:
                try:
                    # 如果启用网格显示，绘制网格线（用于辅助对齐控件）
                    self._draw_grid(painter)
                except Exception:
                    pass  # 忽略绘制网格时的异常
            
            # ========== 绘制窗口框架 ==========
            # 检查是否显示窗口框架
            if hasattr(self, 'show_window_frame') and self.show_window_frame:
                try:
                    # 如果显示窗口框架，绘制窗口的标题栏和边框等框架元素
                    self._draw_window_frame(painter)
                except Exception:
                    pass  # 忽略绘制窗口框架时的异常
            
            # ========== 绘制所有控件 ==========
            # 绘制画布上的所有控件（遍历所有控件并绘制）
            try:
                self._draw_controls(painter)
            except Exception:
                pass  # 忽略绘制控件时的异常
            
            # ========== 绘制选中状态 ==========
            # 绘制选中控件的边框和调整句柄（用于标识选中的控件并允许调整大小）
            try:
                self._draw_selection(painter)
            except Exception:
                pass  # 忽略绘制选中状态时的异常
        
        except KeyboardInterrupt:
            # 处理键盘中断（调试时可能发生）
            # 不重新抛出，避免影响调试器
            pass
        except Exception:
            # 捕获所有其他异常，避免绘制错误导致程序崩溃
            # 在调试模式下，这些异常会被调试器捕获
            pass
        finally:
            # ========== 确保绘图器正确结束 ==========
            # 无论是否发生异常，都要确保绘图器正确结束
            if painter is not None and painter.isActive():
                try:
                    painter.end()
                except Exception:
                    pass
    
    def _draw_background(self, painter: QPainter):
        """绘制背景"""
        # 计算实际画布大小（考虑缩放）
        canvas_width = int(self.canvas_width / self.scale_factor)
        canvas_height = int(self.canvas_height / self.scale_factor)
        
        # 绘制背景
        painter.fillRect(0, 0, canvas_width, canvas_height, self.bg_color)
        
        # 绘制黑色边框
        painter.setPen(QPen(QColor(0, 0, 0), 2, Qt.SolidLine))
        painter.drawRect(0, 0, canvas_width, canvas_height)
        
        # ========== 绘制底图 ==========
        # 检查是否设置了窗口背景图片路径（不为空字符串）
        if self.window_bg_image:
            # 如果设置了背景图片，尝试加载和绘制
            try:
                # 从文件路径加载图片到QPixmap对象
                pix = QPixmap(self.window_bg_image)
                # 检查图片是否成功加载（不为空）
                if not pix.isNull():
                    # 如果图片加载成功，执行以下操作
                    # 获取背景图片显示模式
                    mode = self.window_bg_image_mode
                    
                    # ========== 根据显示模式绘制图片 ==========
                    # 检查显示模式是否为"图片平铺"
                    if mode == "图片平铺":
                        # 如果是平铺模式，使用drawTiledPixmap方法平铺绘制图片
                        # 参数：目标矩形区域、源图片对象
                        painter.drawTiledPixmap(QRect(0, 0, canvas_width, canvas_height), pix)
                    
                    # 检查显示模式是否为"图片居中"
                    elif mode == "图片居中":
                        # 如果是居中模式，计算居中位置
                        # 计算X坐标：画布宽度减去图片宽度后除以2（水平居中）
                        x = (canvas_width - pix.width()) // 2
                        # 计算Y坐标：画布高度减去图片高度后除以2（垂直居中）
                        y = (canvas_height - pix.height()) // 2
                        # 在计算出的位置绘制图片（保持原始大小）
                        painter.drawPixmap(x, y, pix)
                    
                    # 检查显示模式是否为"图片拉伸"
                    elif mode == "图片拉伸":
                        # 如果是拉伸模式，将图片拉伸到整个画布大小
                        # 参数：目标矩形区域（整个画布）、源图片对象
                        painter.drawPixmap(QRect(0, 0, canvas_width, canvas_height), pix)
                    
                    # 检查显示模式是否为"图片缩放"
                    elif mode == "图片缩放":
                        # 如果是缩放模式，按比例缩放图片以适应画布（保持宽高比）
                        # scaled方法参数：目标宽度、目标高度、保持宽高比、平滑变换
                        scaled = pix.scaled(canvas_width, canvas_height, Qt.KeepAspectRatio, Qt.SmoothTransformation)
                        # 计算缩放后图片的居中位置
                        # X坐标：画布宽度减去缩放后图片宽度后除以2
                        x = (canvas_width - scaled.width()) // 2
                        # Y坐标：画布高度减去缩放后图片高度后除以2
                        y = (canvas_height - scaled.height()) // 2
                        # 在计算出的位置绘制缩放后的图片
                        painter.drawPixmap(x, y, scaled)
            except Exception:
                # 如果图片加载或绘制过程中发生任何异常（文件不存在、格式错误等），忽略异常
                pass
    
    def _draw_grid(self, painter: QPainter):
        """
        绘制网格
        在画布上绘制网格线，用于辅助控件对齐和定位
        
        Args:
            painter: QPainter对象，用于绘制
        """
        # ========== 设置网格线样式 ==========
        # 设置画笔颜色为浅灰色（RGB: 220, 220, 220），线宽1像素，线型为虚线
        painter.setPen(QPen(QColor(220, 220, 220), 1, Qt.DotLine))
        
        # ========== 计算实际画布大小 ==========
        # 计算考虑缩放后的实际画布宽度（画布宽度除以缩放因子并取整）
        canvas_width = int(self.canvas_width / self.scale_factor)
        # 计算考虑缩放后的实际画布高度（画布高度除以缩放因子并取整）
        canvas_height = int(self.canvas_height / self.scale_factor)
        
        # ========== 绘制垂直线 ==========
        # 从X=0开始，每隔grid_size像素绘制一条垂直线，直到画布右边缘
        # range(0, canvas_width, self.grid_size)：生成X坐标序列，步长为网格大小
        for x in range(0, canvas_width, self.grid_size):
            # 绘制从画布顶部到底部的垂直线
            # 参数：起点X坐标x、起点Y坐标0、终点X坐标x、终点Y坐标canvas_height
            painter.drawLine(x, 0, x, canvas_height)
        
        # ========== 绘制水平线 ==========
        # 从Y=0开始，每隔grid_size像素绘制一条水平线，直到画布底部
        # range(0, canvas_height, self.grid_size)：生成Y坐标序列，步长为网格大小
        for y in range(0, canvas_height, self.grid_size):
            # 绘制从画布左边缘到右边缘的水平线
            # 参数：起点X坐标0、起点Y坐标y、终点X坐标canvas_width、终点Y坐标y
            painter.drawLine(0, y, canvas_width, y)
    
    def _draw_window_frame(self, painter: QPainter):
        canvas_width = int(self.canvas_width / self.scale_factor)
        frame_height = self.title_bar_height
        title_h = self.title_bar_height
        painter.fillRect(QRect(0, 0, canvas_width, title_h), QColor(235, 235, 235))
        painter.setPen(QPen(QColor(180, 180, 180), 1))
        painter.drawLine(0, title_h, canvas_width, title_h)
        painter.setPen(QPen(QColor(50,50,50),1))
        painter.setFont(QFont("Arial", 11))
        x = 8
        icon_val = getattr(self, 'window_icon', '')
        if isinstance(icon_val, str) and icon_val:
            pix = QPixmap(icon_val)
            if not pix.isNull():
                ih = min(20, title_h - 8)
                iw = ih
                painter.drawPixmap(x, (title_h - ih)//2, pix.scaled(iw, ih, Qt.KeepAspectRatio, Qt.SmoothTransformation))
                x += iw + 6
            else:
                painter.drawText(x, (title_h//2)+4, str(icon_val))
                x += 18
        title_text = getattr(self, 'window_title', '') or ''
        painter.drawText(x, (title_h//2)+4, title_text)
        dropdown_list = getattr(self, 'window_dropdown_menus', [])
        tabs_y = frame_height
        if dropdown_list:
            dropdown_y = frame_height
            dropdown_h = 24
            painter.fillRect(QRect(0, dropdown_y, canvas_width, dropdown_h), QColor(240, 240, 240))
            painter.setPen(QPen(QColor(60, 60, 60), 1))
            painter.drawLine(0, dropdown_y+dropdown_h, canvas_width, dropdown_y+dropdown_h)
            painter.setPen(QPen(QColor(50,50,50),1))
            painter.setFont(QFont("Arial", 10))
            x = 8
            self._dropdown_rects.clear()
            for m in dropdown_list[:20]:
                title = (m.get('title') if isinstance(m, dict) else str(m)) or ''
                w = max(60, len(title)*10)
                rect = QRect(x, dropdown_y, w, dropdown_h)
                painter.drawText(x+8, dropdown_y+16, title)
                self._dropdown_rects.append((m, rect))
                x += w + 8
            tabs_y = dropdown_y + dropdown_h

        tabs_h = 24
        self._menu_item_rects = getattr(self, '_menu_item_rects', [])
        self._menu_item_rects.clear()
        if getattr(self, 'window_menus', None):
            painter.fillRect(QRect(0, tabs_y, canvas_width, tabs_h), QColor(245, 245, 245))
            painter.setPen(QPen(QColor(60, 60, 60), 1))
            painter.drawLine(0, tabs_y+tabs_h, canvas_width, tabs_y+tabs_h)
            painter.setPen(QPen(QColor(50,50,50),1))
            painter.setFont(QFont("Arial", 10))
            x = 8
            for idx, item in enumerate(self.window_menus[:20]):
                w = max(60, len(item)*10)
                rect = QRect(x, tabs_y, w, tabs_h)
                if idx == getattr(self, 'active_index', 0):
                    painter.fillRect(rect, QColor(220, 220, 220))
                painter.drawText(x+8, tabs_y+16, item)
                self._menu_item_rects.append((idx, item, rect))
                x += w + 8
    
    def _draw_controls(self, painter: QPainter):
        """绘制当前页面控件"""
        sorted_controls = sorted(
            self.control_manager.controls.values(),
            key=lambda c: c.z_order
        )
        current_page = getattr(self, 'active_page', '页面1')
        for control in sorted_controls:
            if control.properties.get('page', '页面1') != current_page:
                continue
            self._draw_control(painter, control)
    
    def _draw_control(self, painter: QPainter, control: Control):
        """绘制单个控件"""
        # 应用缩放变换
        painter.save()
        painter.scale(self.scale_factor, self.scale_factor)
        
        # 获取控件信息
        control_info = ControlLibrary.get_control_info(control.control_type)
        
        # 绘制控件背景
        rect = control.get_rect()
        
        # 根据控件类型设置不同的颜色
        if control.control_type == "Label":
            painter.fillRect(rect, QColor(230, 230, 250))
        elif control.control_type == "Button":
            painter.fillRect(rect, QColor(200, 220, 240))
        elif control.control_type == "LineEdit":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
        elif control.control_type == "TextEdit":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
        elif control.control_type == "Frame":
            painter.fillRect(rect, QColor(245, 245, 245))
            painter.setPen(QPen(QColor(180, 180, 180), 1))
            painter.drawRect(rect)
        elif control.control_type == "GroupBox":
            painter.fillRect(rect, QColor(245, 245, 245))
            painter.setPen(QPen(QColor(180, 180, 180), 1))
            painter.drawRect(rect)
            # 绘制标题
            painter.setPen(QPen(QColor(50, 50, 50), 1))
            painter.setFont(QFont("Arial", 10))
            title = control.properties.get("title", "GroupBox")
            painter.drawText(rect.x() + 10, rect.y() + 15, title)
        elif control.control_type == "TabWidget":
            painter.fillRect(rect, QColor(245, 245, 245))
            painter.setPen(QPen(QColor(180, 180, 180), 1))
            painter.drawRect(rect)
            # 绘制标签
            tab_height = 25
            tab_rect = QRect(rect.x(), rect.y(), 80, tab_height)
            painter.fillRect(tab_rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(180, 180, 180), 1))
            painter.drawRect(tab_rect)
            painter.setPen(QPen(QColor(50, 50, 50), 1))
            painter.setFont(QFont("Arial", 10))
            tabs = control.properties.get("tabs", ["标签1"]) or ["标签1"]
            painter.drawText(tab_rect.x() + 10, tab_rect.y() + 17, tabs[0])
        elif control.control_type == "StackedWidget":
            painter.fillRect(rect, QColor(245, 245, 245))
            painter.setPen(QPen(QColor(180, 180, 180), 1))
            painter.drawRect(rect)
            painter.setPen(QPen(QColor(80, 80, 80), 1))
            painter.setFont(QFont("Arial", 10))
            pages = control.properties.get("pages", ["页面1","页面2"]) or ["页面1"]
            idx = control.properties.get("current_index", 0)
            idx = max(0, min(idx, len(pages)-1))
            painter.drawText(rect.x() + 8, rect.y() + 20, f"当前页: {pages[idx]}")
        elif control.control_type == "CheckBox":
            painter.fillRect(rect, QColor(255, 255, 255))
            check_box_rect = QRect(rect.x(), rect.center().y() - 8, 16, 16)
            painter.setPen(QPen(QColor(60, 60, 60), 2))
            painter.drawRect(check_box_rect)
            if control.properties.get("checked", False):
                painter.setPen(QPen(QColor(0, 120, 215), 2))
                painter.drawLine(check_box_rect.x()+3, check_box_rect.y()+8, check_box_rect.x()+7, check_box_rect.y()+12)
                painter.drawLine(check_box_rect.x()+7, check_box_rect.y()+12, check_box_rect.x()+13, check_box_rect.y()+4)
            text = control.properties.get("text", "CheckBox")
            painter.drawText(rect.x() + 25, rect.center().y() + 5, text)
        elif control.control_type == "RadioButton":
            painter.fillRect(rect, QColor(255, 255, 255))
            radio_rect = QRect(rect.x(), rect.center().y() - 8, 16, 16)
            painter.setBrush(QBrush(QColor(255, 255, 255)))
            painter.setPen(QPen(QColor(60, 60, 60), 2))
            painter.drawEllipse(radio_rect)
            if control.properties.get("checked", False):
                painter.setBrush(QBrush(QColor(0, 120, 215)))
                painter.drawEllipse(radio_rect.center(), 5, 5)
            text = control.properties.get("text", "RadioButton")
            painter.drawText(rect.x() + 25, rect.center().y() + 5, text)
        elif control.control_type == "ComboBox":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            # 绘制下拉箭头
            arrow_rect = QRect(rect.right() - 20, rect.y(), 20, rect.height())
            painter.fillRect(arrow_rect, QColor(240, 240, 240))
            painter.setPen(QPen(QColor(100, 100, 100), 1))
            painter.drawLine(arrow_rect.left() + 5, arrow_rect.center().y() - 3,
                            arrow_rect.center().x(), arrow_rect.center().y() + 3)
            painter.drawLine(arrow_rect.center().x(), arrow_rect.center().y() + 3,
                            arrow_rect.right() - 5, arrow_rect.center().y() - 3)
        elif control.control_type == "Slider":
            painter.fillRect(rect, QColor(255, 255, 255))
            # 绘制滑轨
            track_rect = QRect(rect.x() + 10, rect.center().y() - 2, rect.width() - 20, 4)
            painter.fillRect(track_rect, QColor(200, 200, 200))
            # 绘制滑块
            value = control.properties.get("value", 50)
            min_val = control.properties.get("minimum", 0)
            max_val = control.properties.get("maximum", 100)
            slider_pos = rect.x() + 10 + (rect.width() - 20) * (value - min_val) / (max_val - min_val)
            slider_rect = QRect(slider_pos - 6, rect.center().y() - 8, 12, 16)
            painter.fillRect(slider_rect, QColor(100, 150, 200))
        elif control.control_type == "ProgressBar":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            # 绘制进度
            value = control.properties.get("value", 30)
            min_val = control.properties.get("minimum", 0)
            max_val = control.properties.get("maximum", 100)
            progress_width = rect.width() * (value - min_val) / (max_val - min_val)
            progress_rect = QRect(rect.x(), rect.y(), progress_width, rect.height())
            painter.fillRect(progress_rect, QColor(100, 150, 200))
        elif control.control_type == "ListWidget":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            # 绘制列表项
            items = control.properties.get("items", ["项目1", "项目2", "项目3"])
            painter.setPen(QPen(QColor(50, 50, 50), 1))
            painter.setFont(QFont("Arial", 10))
            for i, item in enumerate(items[:5]):  # 最多显示5项
                if i * 20 < rect.height() - 10:
                    painter.drawText(rect.x() + 5, rect.y() + 20 + i * 20, item)
        elif control.control_type == "ImageBox":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(180, 180, 180), 1))
            painter.drawRect(rect)
            path = control.properties.get('image_path', '')
            if path:
                pix = QPixmap(path)
                if not pix.isNull():
                    scaled = pix.scaled(rect.size(), Qt.IgnoreAspectRatio if control.properties.get('scaled', True) else Qt.KeepAspectRatio, Qt.SmoothTransformation)
                    painter.drawPixmap(rect.x(), rect.y(), scaled)
            else:
                painter.setPen(QPen(QColor(150, 150, 150), 1))
                painter.setFont(QFont("Arial", 10))
                painter.drawText(rect, Qt.AlignCenter, "图片")
        elif control.control_type == "TreeWidget":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            # 绘制表头
            headers = control.properties.get("header_labels", ["名称", "类型"])
            header_height = 20
            col_width = max(60, rect.width() // max(1, len(headers)))
            painter.setPen(QPen(QColor(80, 80, 80), 1))
            painter.setFont(QFont("Arial", 9))
            for i, h in enumerate(headers):
                painter.drawText(rect.x() + 5 + i * col_width, rect.y() + header_height - 4, h)
            # 简单绘制几行占位
            painter.setPen(QPen(QColor(120, 120, 120), 1))
            for r in range(3):
                painter.drawText(rect.x() + 5, rect.y() + header_height + 18 * (r + 1), f"节点{r+1}")
        elif control.control_type == "TableWidget":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            headers = control.properties.get("headers", ["列1","列2","列3"]) or ["列1"]
            col_width = max(40, rect.width() // max(1, len(headers)))
            painter.setPen(QPen(QColor(80, 80, 80), 1))
            painter.setFont(QFont("Arial", 9))
            for i, h in enumerate(headers):
                painter.drawText(rect.x() + 5 + i * col_width, rect.y() + 18, h)
            painter.setPen(QPen(QColor(220, 220, 220), 1))
            for r in range(1, 4):
                painter.drawLine(rect.x(), rect.y() + 18 + r * 20, rect.right(), rect.y() + 18 + r * 20)
        elif control.control_type == "DateEdit":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            painter.setPen(QPen(QColor(80, 80, 80), 1))
            painter.setFont(QFont("Arial", 10))
            painter.drawText(rect.x() + 5, rect.center().y() + 5, control.properties.get("display_format", "yyyy-MM-dd"))
        elif control.control_type == "TimeEdit":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            painter.setPen(QPen(QColor(80, 80, 80), 1))
            painter.setFont(QFont("Arial", 10))
            painter.drawText(rect.x() + 5, rect.center().y() + 5, control.properties.get("display_format", "HH:mm:ss"))
        elif control.control_type == "DateTimeEdit":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            painter.setPen(QPen(QColor(80, 80, 80), 1))
            painter.setFont(QFont("Arial", 10))
            painter.drawText(rect.x() + 5, rect.center().y() + 5, control.properties.get("display_format", "yyyy-MM-dd HH:mm:ss"))
        elif control.control_type == "SpinBox":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            # 绘制上下箭头
            arrow_width = 20
            arrow_rect = QRect(rect.right() - arrow_width, rect.y(), arrow_width, rect.height())
            painter.fillRect(arrow_rect, QColor(240, 240, 240))
            painter.setPen(QPen(QColor(100, 100, 100), 1))
            # 上箭头
            painter.drawLine(arrow_rect.center().x(), arrow_rect.top() + 5,
                            arrow_rect.center().x() - 4, arrow_rect.top() + 9)
            painter.drawLine(arrow_rect.center().x(), arrow_rect.top() + 5,
                            arrow_rect.center().x() + 4, arrow_rect.top() + 9)
            # 下箭头
            painter.drawLine(arrow_rect.center().x(), arrow_rect.bottom() - 5,
                            arrow_rect.center().x() - 4, arrow_rect.bottom() - 9)
            painter.drawLine(arrow_rect.center().x(), arrow_rect.bottom() - 5,
                            arrow_rect.center().x() + 4, arrow_rect.bottom() - 9)
        elif control.control_type == "DoubleSpinBox":
            painter.fillRect(rect, QColor(255, 255, 255))
            painter.setPen(QPen(QColor(200, 200, 200), 1))
            painter.drawRect(rect)
            # 绘制上下箭头
            arrow_width = 20
            arrow_rect = QRect(rect.right() - arrow_width, rect.y(), arrow_width, rect.height())
            painter.fillRect(arrow_rect, QColor(240, 240, 240))
            painter.setPen(QPen(QColor(100, 100, 100), 1))
            # 上箭头
            painter.drawLine(arrow_rect.center().x(), arrow_rect.top() + 5,
                            arrow_rect.center().x() - 4, arrow_rect.top() + 9)
            painter.drawLine(arrow_rect.center().x(), arrow_rect.top() + 5,
                            arrow_rect.center().x() + 4, arrow_rect.top() + 9)
            # 下箭头
            painter.drawLine(arrow_rect.center().x(), arrow_rect.bottom() - 5,
                            arrow_rect.center().x() - 4, arrow_rect.bottom() - 9)
            painter.drawLine(arrow_rect.center().x(), arrow_rect.bottom() - 5,
                            arrow_rect.center().x() + 4, arrow_rect.bottom() - 9)
        
        # 绘制控件文本
        if control.control_type in ["Label", "Button"]:
            painter.setPen(QPen(QColor(50, 50, 50), 1))
            painter.setFont(QFont("Arial", 10))
            
            if control.control_type == "Label":
                text = control.properties.get("text", "Label")
                alignment = control.properties.get("alignment", "center")
                if alignment == "center":
                    painter.drawText(rect, Qt.AlignCenter, text)
                else:
                    painter.drawText(rect.x() + 5, rect.center().y() + 5, text)
            elif control.control_type == "Button":
                text = control.properties.get("text", "Button")
                painter.drawText(rect, Qt.AlignCenter, text)
        elif control.control_type == "LineEdit":
            text = control.properties.get("text", "")
            if not text:
                text = control.properties.get("placeholder_text", "")
                painter.setPen(QPen(QColor(180, 180, 180), 1))
            else:
                painter.setPen(QPen(QColor(50, 50, 50), 1))
            painter.setFont(QFont("Arial", 10))
            painter.drawText(rect.x() + 5, rect.center().y() + 5, text)
        elif control.control_type == "TextEdit":
            text = control.properties.get("plain_text", "")
            if not text:
                text = control.properties.get("placeholder_text", "")
                painter.setPen(QPen(QColor(180, 180, 180), 1))
            else:
                painter.setPen(QPen(QColor(50, 50, 50), 1))
            painter.setFont(QFont("Arial", 10))
            # 只显示第一行文本
            lines = text.split('\n')
            if lines:
                painter.drawText(rect.x() + 5, rect.y() + 20, lines[0])
        
        # 恢复变换
        painter.restore()
    
    def _draw_selection(self, painter: QPainter):
        if self._rubber_selecting:
            painter.save()
            painter.scale(self.scale_factor, self.scale_factor)
            r = QRect(self._rubber_rect)
            r = QRect(min(r.left(), r.right()), min(r.top(), r.bottom()), abs(r.width()), abs(r.height()))
            painter.setPen(QPen(QColor(0, 150, 0), 2, Qt.DashLine))
            painter.drawRect(r)
            painter.restore()
        try:
            ids = list(self.control_manager.get_selected_ids())
        except Exception:
            ids = list(getattr(self.control_manager, 'selected_control_ids', []))
        if not ids and hasattr(self, '_canvas_selected') and self._canvas_selected:
            canvas_width = int(self.canvas_width / self.scale_factor)
            canvas_height = int(self.canvas_height / self.scale_factor)
            painter.setPen(QPen(QColor(0, 120, 215), 2, Qt.DashLine))
            painter.drawRect(0, 0, canvas_width, canvas_height)
            handle_size = 8
            painter.setBrush(QBrush(QColor(255, 255, 255)))
            painter.setPen(QPen(QColor(0, 120, 215), 1))
            handle_rect = QRect(canvas_width - handle_size//2, canvas_height - handle_size//2, handle_size, handle_size)
            painter.drawRect(handle_rect)
            return
        if len(ids) > 1:
            painter.save()
            painter.scale(self.scale_factor, self.scale_factor)
            for cid in ids:
                c = self.control_manager.get_control(cid)
                if not c:
                    continue
                rect = c.get_rect()
                painter.setPen(QPen(QColor(0, 120, 215), 2, Qt.DashLine))
                painter.drawRect(rect)
            painter.restore()
            return
        if len(ids) == 1:
            c = self.control_manager.get_control(ids[0])
            if not c:
                return
            painter.save()
            painter.scale(self.scale_factor, self.scale_factor)
            rect = c.get_rect()
            painter.setPen(QPen(QColor(0, 120, 215), 2, Qt.DashLine))
            painter.drawRect(rect)
            handle_size = 8
            painter.setBrush(QBrush(QColor(255, 255, 255)))
            painter.setPen(QPen(QColor(0, 120, 215), 1))
            h = QRect(rect.x() - handle_size//2, rect.y() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.center().x() - handle_size//2, rect.y() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.right() - handle_size//2, rect.y() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.x() - handle_size//2, rect.center().y() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.right() - handle_size//2, rect.center().y() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.x() - handle_size//2, rect.bottom() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.center().x() - handle_size//2, rect.bottom() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            h = QRect(rect.right() - handle_size//2, rect.bottom() - handle_size//2, handle_size, handle_size)
            painter.drawRect(h)
            painter.restore()
    
    def mousePressEvent(self, event):
        """鼠标按下事件"""
        if event.button() == Qt.RightButton:
            # 右键命中下拉菜单标题，提供人性化编辑入口
            if getattr(self, '_dropdown_rects', None):
                for m, rect in self._dropdown_rects:
                    if rect.contains(event.pos()):
                        cm = QMenu(self)
                        act_rename = cm.addAction("重命名菜单")
                        act_add_action = cm.addAction("添加动作项")
                        act_add_submenu = cm.addAction("添加子菜单")
                        act_delete = cm.addAction("删除菜单")
                        chosen = cm.exec(self.mapToGlobal(event.pos()))
                        if chosen == act_rename:
                            text, ok = QInputDialog.getText(self, "重命名菜单", "标题:", text=m.get('title',''))
                            if ok:
                                m['title'] = text.strip() or m.get('title','')
                                self.update()
                        elif chosen == act_add_action:
                            label, ok = QInputDialog.getText(self, "添加动作项", "名称:")
                            if ok and label.strip():
                                m.setdefault('items', []).append({"label": label.strip(), "level": 0, "shortcut": "(无)", "checked": False, "allow": True, "visible": True})
                                self.update()
                        elif chosen == act_add_submenu:
                            label, ok = QInputDialog.getText(self, "添加子菜单", "子菜单标题:")
                            if ok and label.strip():
                                m.setdefault('items', []).append({"label": label.strip(), "level": 1, "shortcut": "(无)", "checked": False, "allow": True, "visible": True})
                                self.update()
                        elif chosen == act_delete:
                            try:
                                self.window_dropdown_menus.remove(m)
                                self.update()
                            except ValueError:
                                pass
                        return
            return
        
        if event.button() == Qt.LeftButton:
            # 点击下拉菜单标题，弹出菜单
            if getattr(self, '_dropdown_rects', None):
                for m, rect in self._dropdown_rects:
                    if rect.contains(event.pos()):
                        menu = self._build_dropdown_menu(m)
                        if menu:
                            menu.exec(self.mapToGlobal(rect.bottomLeft()))
                        return
            # 菜单点击切换页面
            if getattr(self, '_menu_item_rects', None):
                for idx, name, rect in self._menu_item_rects:
                    if rect.contains(event.pos()):
                        self.active_index = idx
                        self.active_page = name
                        self.update()
                        return
            # 转换为画布坐标（考虑缩放）
            pos = event.pos()
            canvas_pos = QPoint(int(pos.x() / self.scale_factor), int(pos.y() / self.scale_factor))
            
            
            
            # 检查是否点击了控件
            # 页面内命中检测
            control = None
            sorted_controls = sorted(self.control_manager.controls.values(), key=lambda c: c.z_order, reverse=True)
            for c in sorted_controls:
                if c.properties.get('page', '页面1') != getattr(self, 'active_page', '页面1'):
                    continue
                if c.contains_point(canvas_pos.x(), canvas_pos.y()):
                    control = c
                    break
            if control:
                if event.modifiers() & Qt.ControlModifier:
                    if control.id in self.control_manager.get_selected_ids():
                        self.control_manager.remove_from_selection(control.id)
                    else:
                        self.control_manager.add_to_selection(control.id)
                    try:
                        selected_ids = self.control_manager.get_selected_ids()
                        if len(selected_ids) > 1:
                            self.control_selected.emit('multiple_selected')
                        elif len(selected_ids) == 1:
                            self.control_selected.emit(selected_ids[0])
                        else:
                            self.control_selected.emit('canvas')
                    except Exception:
                        pass
                    self.update()
                    return
                else:
                    current = self.control_manager.get_selected_control()
                    if current and current.id == control.id:
                        handle = current.get_resize_handle(canvas_pos.x(), canvas_pos.y())
                        if handle:
                            self.resizing = True
                            self.resize_handle = handle
                            self.drag_start_pos = canvas_pos
                            self.drag_start_rect = current.get_rect()
                            return
                    selected_ids = self.control_manager.get_selected_ids()
                    if (control.id in selected_ids) and (len(selected_ids) > 1):
                        self.dragging = True
                        self.drag_start_pos = canvas_pos
                        self._multi_drag_ids = selected_ids.copy()
                        self._multi_drag_start = {}
                        for cid in selected_ids:
                            cobj = self.control_manager.controls.get(cid)
                            if cobj:
                                self._multi_drag_start[cid] = cobj.get_rect()
                        return
                    self.control_manager.clear_selection()
                    self.select_control(control.id)
                    self.update()
                    self.dragging = True
                    self.drag_start_pos = canvas_pos
                    self.drag_start_rect = control.get_rect()
            else:
                # 检查是否点击了画布的调整句柄
                canvas_width = int(self.canvas_width / self.scale_factor)
                canvas_height = int(self.canvas_height / self.scale_factor)
                canvas_rect = QRect(0, 0, canvas_width, canvas_height)
                handle = self._get_canvas_resize_handle(canvas_pos, canvas_rect)
                
                # 只有当点击了画布右下角调整句柄时才设置拖拽和调整大小状态
                if handle == "bottom_right":
                    self.dragging = True
                    self.resizing = True
                    self.resize_handle = handle
                    self.drag_start_pos = canvas_pos
                    self.drag_start_rect = QRect(0, 0, canvas_width, canvas_height)
                else:
                    self._rubber_selecting = True
                    self._rubber_additive = bool(event.modifiers() & Qt.ControlModifier)
                    self._rubber_start = canvas_pos
                    self._rubber_rect = QRect(canvas_pos, canvas_pos)
                    self._canvas_selected = False
                    self.update()
    
    def mouseMoveEvent(self, event):
        """
        鼠标移动事件处理函数
        当鼠标在画布上移动时调用，处理框选、拖拽、调整大小等操作
        
        Args:
            event: 鼠标事件对象，包含当前鼠标位置信息
        """
        # ========== 获取鼠标位置 ==========
        # 获取鼠标在窗口中的当前位置（窗口坐标）
        pos = event.position().toPoint()
        # ========== 转换为画布坐标 ==========
        # 将窗口坐标转换为画布坐标（考虑缩放因子）
        # 如果画布被缩放，需要除以缩放因子得到真实坐标
        canvas_pos = QPoint(int(pos.x() / self.scale_factor), int(pos.y() / self.scale_factor))
        
        # ========== 更新鼠标光标 ==========
        # 根据鼠标位置更新光标样式（如调整大小光标、拖拽光标等）
        self._update_cursor(canvas_pos)
        
        # ========== 处理框选操作 ==========
        # 检查是否正在进行框选操作
        if self._rubber_selecting:
            # ========== 更新框选矩形 ==========
            # 更新框选矩形范围（从开始位置到当前位置）
            self._rubber_rect = QRect(self._rubber_start, canvas_pos)
            # ========== 更新画布显示 ==========
            # 更新画布显示，绘制框选矩形
            self.update()
            # 框选操作处理完成，直接返回（不继续处理其他操作）
            return
        # ========== 处理拖拽操作 ==========
        # 检查是否正在拖拽（可能是拖拽控件或调整画布大小）
        if self.dragging:
            # ========== 检查是否选中了画布 ==========
            # 检查是否选中了画布（可能是在调整画布大小）
            if self._canvas_selected:
                # ========== 调整画布大小 ==========
                # 计算鼠标移动的距离（X和Y方向的偏移量）
                dx = canvas_pos.x() - self.drag_start_pos.x()
                dy = canvas_pos.y() - self.drag_start_pos.y()
                
                # ========== 网格对齐 ==========
                # 检查是否启用网格吸附
                if self.snap_to_grid:
                    # 如果启用网格吸附，将移动距离对齐到网格
                    # 将移动距离除以网格大小并四舍五入，再乘以网格大小
                    dx = round(dx / self.grid_size) * self.grid_size
                    dy = round(dy / self.grid_size) * self.grid_size
                
                # ========== 计算新画布大小 ==========
                # 调整画布大小（只允许从右下角调整）
                # 新宽度 = 起始宽度 + 移动距离，最小宽度为200像素
                new_width = max(200, self.drag_start_rect.width() + dx)
                # 新高度 = 起始高度 + 移动距离，最小高度为200像素
                new_height = max(200, self.drag_start_rect.height() + dy)
                
                # ========== 设置画布新大小 ==========
                # 设置画布的新大小（考虑缩放因子，转换为实际像素大小）
                self.set_canvas_size(int(new_width * self.scale_factor), int(new_height * self.scale_factor))
                # 画布大小调整完成，直接返回
                return
            
            # ========== 拖拽控件 ==========
            # 如果未选中画布，则可能是拖拽控件
            # 获取当前选中的所有控件ID列表
            selected_ids = self.control_manager.get_selected_ids()
            # ========== 处理多选拖拽 ==========
            # 检查是否选中了多个控件且正在进行多选拖拽
            if len(selected_ids) > 1 and self._multi_drag_ids:
                # ========== 计算多选拖拽的移动距离 ==========
                # 计算鼠标从拖拽开始位置移动的距离（X和Y方向）
                dx0 = canvas_pos.x() - self.drag_start_pos.x()
                dy0 = canvas_pos.y() - self.drag_start_pos.y()
                
                # ========== 计算顶部边界 ==========
                # 计算画布顶部边界位置（标题栏和菜单栏的高度）
                top_y = (
                    self.title_bar_height +  # 标题栏高度
                    (24 if (getattr(self, 'window_dropdown_menus', []) or []) else 0) +  # 下拉菜单高度
                    (24 if (getattr(self, 'window_menus', []) or []) else 0)  # 菜单栏高度
                )
                
                # ========== 准备多选拖拽操作 ==========
                # 获取第一个控件的ID（用于发送移动信号）
                first_id = selected_ids[0]
                # 初始化第一个控件的移动偏移量（初始为0）
                first_dx = 0
                first_dy = 0
                
                # ========== 遍历所有选中的控件并移动 ==========
                # 遍历所有选中的控件ID，同时移动多个控件
                for cid in selected_ids:
                    # ========== 获取控件的初始位置 ==========
                    # 从多选拖拽初始位置字典中获取该控件的初始矩形
                    rect = self._multi_drag_start.get(cid)
                    # 从控件管理器中获取控件对象
                    cobj = self.control_manager.controls.get(cid)
                    
                    # ========== 检查控件和初始位置是否存在 ==========
                    # 检查控件的初始矩形和控件对象是否都存在
                    if not rect or not cobj:
                        # 如果缺少初始位置或控件对象，跳过该控件
                        continue
                    
                    # ========== 计算新位置 ==========
                    # 检查是否启用网格吸附
                    if self.snap_to_grid:
                        # ========== 启用网格吸附时 ==========
                        # 计算新位置（初始位置 + 移动距离）
                        new_x = rect.x() + dx0
                        new_y = rect.y() + dy0
                        # 将新位置对齐到网格（四舍五入到最近的网格点）
                        new_x = round(new_x / self.grid_size) * self.grid_size
                        new_y = round(new_y / self.grid_size) * self.grid_size
                    else:
                        # ========== 未启用网格吸附时 ==========
                        # 直接计算新位置（初始位置 + 移动距离）
                        new_x = rect.x() + dx0
                        new_y = rect.y() + dy0
                    
                    # ========== 限制X坐标范围 ==========
                    # 限制新X坐标在画布范围内（不能小于0，不能超出画布右边界）
                    new_x = max(0, min(new_x, int(self.canvas_width / self.scale_factor) - cobj.width))
                    # ========== 限制Y坐标范围 ==========
                    # 限制新Y坐标在画布范围内（不能小于顶部边界，不能超出画布底边界）
                    new_y = max(top_y, min(new_y, int(self.canvas_height / self.scale_factor) - cobj.height))
                    
                    # ========== 计算实际移动距离 ==========
                    # 计算控件实际移动的X方向距离（当前X坐标到新X坐标的距离）
                    ddx = new_x - cobj.x
                    # 计算控件实际移动的Y方向距离（当前Y坐标到新Y坐标的距离）
                    ddy = new_y - cobj.y
                    
                    # ========== 保存第一个控件的移动距离 ==========
                    # 检查当前控件是否为第一个控件
                    if cid == first_id:
                        # 如果是第一个控件，保存其移动距离（用于发送信号）
                        first_dx = ddx
                        first_dy = ddy
                    
                    # ========== 执行移动操作 ==========
                    # 调用控件管理器的移动方法，实际移动控件到新位置
                    self.control_manager.move_control(cid, ddx, ddy)
                
                # ========== 发送移动信号 ==========
                # 发送控件移动信号，通知其他组件控件已移动（使用第一个控件的移动距离）
                self.control_moved.emit(first_id, first_dx, first_dy)
                # ========== 更新画布显示 ==========
                # 更新画布显示，刷新控件的新位置
                self.update()
            else:
                # ========== 处理单选拖拽 ==========
                # 如果未进行多选拖拽，处理单个控件的拖拽
                # 获取当前选中的控件对象
                selected_control = self.control_manager.get_selected_control()
                # ========== 检查是否有选中的控件 ==========
                # 检查是否成功获取到选中的控件对象
                if selected_control:
                    # ========== 计算移动距离 ==========
                    # 计算鼠标从拖拽开始位置移动的距离（X和Y方向）
                    dx = canvas_pos.x() - self.drag_start_pos.x()
                    dy = canvas_pos.y() - self.drag_start_pos.y()
                    
                    # ========== 计算新位置 ==========
                    # 检查是否启用网格吸附
                    if self.snap_to_grid:
                        # ========== 启用网格吸附时 ==========
                        # 计算新位置（起始位置 + 移动距离）
                        new_x = self.drag_start_rect.x() + dx
                        new_y = self.drag_start_rect.y() + dy
                        # 将新位置对齐到网格（四舍五入到最近的网格点）
                        new_x = round(new_x / self.grid_size) * self.grid_size
                        new_y = round(new_y / self.grid_size) * self.grid_size
                    else:
                        # ========== 未启用网格吸附时 ==========
                        # 直接计算新位置（起始位置 + 移动距离）
                        new_x = self.drag_start_rect.x() + dx
                        new_y = self.drag_start_rect.y() + dy
                    
                    # ========== 计算顶部边界 ==========
                    # 计算画布顶部边界位置（标题栏和菜单栏的高度）
                    top_y = (
                        self.title_bar_height +  # 标题栏高度
                        (24 if (getattr(self, 'window_dropdown_menus', []) or []) else 0) +  # 下拉菜单高度
                        (24 if (getattr(self, 'window_menus', []) or []) else 0)  # 菜单栏高度
                    )
                    
                    # ========== 限制X坐标范围 ==========
                    # 限制新X坐标在画布范围内（不能小于0，不能超出画布右边界）
                    new_x = max(0, min(new_x, int(self.canvas_width / self.scale_factor) - selected_control.width))
                    # ========== 限制Y坐标范围 ==========
                    # 限制新Y坐标在画布范围内（不能小于顶部边界，不能超出画布底边界）
                    new_y = max(top_y, min(new_y, int(self.canvas_height / self.scale_factor) - selected_control.height))
                    
                    # ========== 计算实际移动距离 ==========
                    # 计算控件实际移动的X方向距离（当前X坐标到新X坐标的距离）
                    dx = new_x - selected_control.x
                    # 计算控件实际移动的Y方向距离（当前Y坐标到新Y坐标的距离）
                    dy = new_y - selected_control.y
                    
                    # ========== 执行移动操作 ==========
                    # 调用控件管理器的移动方法，实际移动控件到新位置
                    self.control_manager.move_control(selected_control.id, dx, dy)
                    # ========== 发送移动信号 ==========
                    # 发送控件移动信号，通知其他组件控件已移动
                    self.control_moved.emit(selected_control.id, dx, dy)
                    # ========== 更新画布显示 ==========
                    # 更新画布显示，刷新控件的新位置
                    self.update()
        
        # ========== 处理调整大小操作 ==========
        # 检查是否正在调整大小（可能是调整画布大小或控件大小）
        elif self.resizing:
            # ========== 检查是否选中了画布 ==========
            # 检查是否选中了画布（可能是在调整画布大小）
            if self._canvas_selected:
                # ========== 调整画布大小 ==========
                # 计算鼠标从调整开始位置移动的距离（X和Y方向）
                dx = canvas_pos.x() - self.drag_start_pos.x()
                dy = canvas_pos.y() - self.drag_start_pos.y()
                
                # ========== 网格对齐 ==========
                # 检查是否启用网格吸附
                if self.snap_to_grid:
                    # 如果启用网格吸附，将调整距离对齐到网格
                    dx = round(dx / self.grid_size) * self.grid_size
                    dy = round(dy / self.grid_size) * self.grid_size
                
                # ========== 计算新画布大小 ==========
                # 调整画布大小（只允许从右下角调整）
                # 新宽度 = 起始宽度 + 调整距离，最小宽度为200像素
                new_width = max(200, self.drag_start_rect.width() + dx)
                # 新高度 = 起始高度 + 调整距离，最小高度为200像素
                new_height = max(200, self.drag_start_rect.height() + dy)
                
                # ========== 设置画布新大小 ==========
                # 设置画布的新大小（考虑缩放因子，转换为实际像素大小）
                self.set_canvas_size(int(new_width * self.scale_factor), int(new_height * self.scale_factor))
                # 画布大小调整完成，直接返回
                return
            
            # ========== 调整控件大小 ==========
            # 如果未选中画布，则可能是调整控件大小
            # 获取当前选中的控件对象
            selected_control = self.control_manager.get_selected_control()
            # ========== 检查是否有选中的控件 ==========
            # 检查是否成功获取到选中的控件对象
            if selected_control:
                # ========== 计算调整距离 ==========
                # 计算当前鼠标位置相对于起始位置的画布坐标差值
                # 这个差值表示用户想要调整的尺寸变化量
                dx = canvas_pos.x() - self.drag_start_pos.x()
                dy = canvas_pos.y() - self.drag_start_pos.y()
                
                # ========== 网格对齐 ==========
                # 检查是否启用网格吸附
                if self.snap_to_grid:
                    # 如果启用网格吸附，将调整距离对齐到网格
                    dx = round(dx / self.grid_size) * self.grid_size
                    dy = round(dy / self.grid_size) * self.grid_size
                
                # ========== 重置控件到起始位置和大小 ==========
                # 将控件的位置和大小重置为调整开始时的状态
                # 这样可以确保调整操作是基于起始状态计算的
                selected_control.x = self.drag_start_rect.x()
                selected_control.y = self.drag_start_rect.y()
                selected_control.width = self.drag_start_rect.width()
                selected_control.height = self.drag_start_rect.height()
                
                # ========== 应用新的尺寸和位置 ==========
                # 根据调整句柄（如top_left、bottom_right等）应用新的尺寸和位置
                # resize_control方法会根据句柄类型调整控件的相应边或角
                self.control_manager.resize_control(selected_control.id, self.resize_handle, dx, dy)
                
                # ========== 计算顶部边界 ==========
                # 计算画布顶部边界位置（标题栏和菜单栏的高度）
                top_y = (
                    self.title_bar_height +  # 标题栏高度
                    (24 if (getattr(self, 'window_dropdown_menus', []) or []) else 0) +  # 下拉菜单高度
                    (24 if (getattr(self, 'window_menus', []) or []) else 0)  # 菜单栏高度
                )
                
                # ========== 限制控件位置和大小在画布范围内 ==========
                # 检查控件的Y坐标是否小于顶部边界
                if selected_control.y < top_y:
                    # 如果控件上边界超出顶部，计算超出量
                    delta = top_y - selected_control.y
                    # 将控件上边界移动到顶部边界
                    selected_control.y = top_y
                    # 减少控件高度，避免超出顶部边界（最小高度为1像素）
                    selected_control.height = max(1, selected_control.height - delta)
                
                # ========== 限制控件高度在画布范围内 ==========
                # 检查控件的底边界是否超出画布底边界
                if selected_control.y + selected_control.height > int(self.canvas_height / self.scale_factor):
                    # 如果控件底边界超出画布，调整控件高度使其在画布内
                    # 控件高度 = 画布高度 - 控件Y坐标（最小高度为1像素）
                    selected_control.height = max(1, int(self.canvas_height / self.scale_factor) - selected_control.y)
                
                # ========== 发送调整大小信号 ==========
                # 发送控件大小调整信号，通知其他组件控件大小已改变
                self.control_resized.emit(selected_control.id, self.resize_handle, dx, dy)
                # ========== 更新画布显示 ==========
                # 更新画布显示，刷新控件的新大小
                self.update()
    
    def mouseReleaseEvent(self, event):
        """
        鼠标释放事件处理函数
        当用户释放鼠标按钮时调用，完成拖拽、框选等操作
        
        Args:
            event: 鼠标事件对象，包含释放的按钮信息
        """
        # ========== 检查是否为左键释放 ==========
        # 检查释放的鼠标按钮是否为左键
        if event.button() == Qt.LeftButton:
            # ========== 处理框选操作完成 ==========
            # 检查是否正在进行框选操作
            if self._rubber_selecting:
                # ========== 规范化框选矩形 ==========
                # 创建框选矩形的副本
                rect = QRect(self._rubber_rect)
                # 规范化矩形（确保左上角在右下角之前，宽度和高度为正数）
                # 这样可以正确处理从右下向左上或从左下向右上拖拽的情况
                rect = QRect(
                    min(rect.left(), rect.right()),  # 左边界：取左右边界的最小值
                    min(rect.top(), rect.bottom()),  # 上边界：取上下边界的最小值
                    abs(rect.width()),  # 宽度：取宽度的绝对值
                    abs(rect.height())  # 高度：取高度的绝对值
                )
                
                # ========== 查找框选矩形内的控件 ==========
                # 初始化选中ID列表，用于存储框选范围内的控件ID
                selected_ids = []
                # ========== 遍历所有控件并检查是否在框选范围内 ==========
                # 遍历控件管理器中的所有控件对象
                for c in self.control_manager.controls.values():
                    # ========== 检查控件是否属于当前活动页面 ==========
                    # 检查控件所属的页面是否与当前活动页面一致
                    if c.properties.get('page', '页面1') != getattr(self, 'active_page', '页面1'):
                        # 如果控件不属于当前活动页面，跳过该控件
                        continue
                    
                    # ========== 检查控件是否与框选矩形相交 ==========
                    # 获取控件的矩形区域
                    r = c.get_rect()
                    # 检查控件的矩形区域是否与框选矩形相交
                    if rect.intersects(r):
                        # 如果相交，将控件ID添加到选中列表中
                        selected_ids.append(c.id)
                
                # ========== 处理累加选择模式 ==========
                # 检查是否为累加选择模式（按住Ctrl键框选）
                if self._rubber_additive:
                    # ========== 累加模式：添加到已有选择 ==========
                    # 获取当前已有的选中控件ID集合
                    base = set(self.control_manager.get_selected_ids())
                    # 将框选范围内的控件ID添加到已有选择中
                    base.update(selected_ids)
                    # 将集合转换回列表，设置新的选中状态
                    self.control_manager.set_selection(list(base))
                else:
                    # ========== 替换模式：替换已有选择 ==========
                    # 如果不是累加模式，直接替换为框选范围内的控件
                    self.control_manager.set_selection(selected_ids)
                
                # ========== 发送选中信号 ==========
                # 获取所有选中的控件ID（包括累加后的完整列表）
                selected_ids_all = self.control_manager.get_selected_ids()
                
                # 尝试发送选中信号，根据选中数量发送不同的信号
                try:
                    if len(selected_ids_all) > 1:
                        # 如果选中了多个控件，发送多选信号
                        self.control_selected.emit('multiple_selected')
                    elif len(selected_ids_all) == 0:
                        # 如果没有选中任何控件，选中画布
                        self.select_control('canvas')
                    elif len(selected_ids_all) == 1:
                        # 如果只选中了一个控件，发送单个控件选中信号
                        self.control_selected.emit(selected_ids_all[0])
                except Exception:
                    # 如果发送信号失败，忽略异常（不中断程序运行）
                    pass
                
                # ========== 结束框选操作 ==========
                # 设置框选状态为False（框选操作完成）
                self._rubber_selecting = False
                # 更新画布显示，移除框选矩形
                self.update()
            
            # ========== 重置拖拽和调整大小状态 ==========
            # 设置拖拽状态为False（拖拽操作完成）
            self.dragging = False
            # 设置调整大小状态为False（调整大小操作完成）
            self.resizing = False
            # 清空调整句柄（不再调整大小）
            self.resize_handle = None
            # 清空多选拖拽的控件ID列表
            self._multi_drag_ids = []
            # 清空多选拖拽的初始位置字典
            self._multi_drag_start = {}
            
            # ========== 发送画布大小变化信号 ==========
            # 检查画布是否被选中（可能在调整画布大小）
            if self._canvas_selected:
                # 计算画布的实际大小（考虑缩放因子）
                canvas_width = int(self.canvas_width / self.scale_factor)
                canvas_height = int(self.canvas_height / self.scale_factor)
                # 发送画布大小变化信号，通知其他组件画布大小已改变
                self.canvas_resized.emit(canvas_width, canvas_height)
            
            # ========== 更新画布显示 ==========
            # 更新画布显示，刷新所有变化
            self.update()
    
    def _find_control_at(self, pos):
        """查找指定位置的控件"""
        return self.control_manager.get_control_at(pos.x(), pos.y())
    
    def _get_canvas_resize_handle(self, pos, canvas_rect):
        """获取画布调整大小的句柄"""
        handle_size = 8
        margin = handle_size // 2
        
        # 右下角句柄
        bottom_right = QRect(
            canvas_rect.right() - margin,
            canvas_rect.bottom() - margin,
            handle_size,
            handle_size
        )
        
        if bottom_right.contains(pos):
            return "bottom_right"
        
        return None
    
    def _get_resize_handle(self, pos):
        """获取调整大小的句柄"""
        selected_control = self.control_manager.get_selected_control()
        if not selected_control:
            return None
        
        return selected_control.get_resize_handle(pos.x(), pos.y())
    
    def _update_cursor(self, pos):
        """根据鼠标位置更新光标样式"""
        # 转换为画布坐标（考虑缩放）
        canvas_pos = QPoint(int(pos.x() / self.scale_factor), int(pos.y() / self.scale_factor))
        
        # 检查是否选中了画布
        if self._canvas_selected:
            # 检查是否在画布右下角调整句柄上
            canvas_width = int(self.canvas_width / self.scale_factor)
            canvas_height = int(self.canvas_height / self.scale_factor)
            canvas_rect = QRect(0, 0, canvas_width, canvas_height)
            handle = self._get_canvas_resize_handle(canvas_pos, canvas_rect)
            if handle == "bottom_right":
                self.setCursor(Qt.SizeFDiagCursor)
                return
        
        selected_control = self.control_manager.get_selected_control()
        if selected_control:
            handle = self._get_resize_handle(canvas_pos)
            if handle:
                # 根据调整句柄设置光标
                if handle in ["top_left", "bottom_right"]:
                    self.setCursor(Qt.SizeFDiagCursor)
                elif handle in ["top_right", "bottom_left"]:
                    self.setCursor(Qt.SizeBDiagCursor)
                elif handle in ["top", "bottom"]:
                    self.setCursor(Qt.SizeVerCursor)
                elif handle in ["left", "right"]:
                    self.setCursor(Qt.SizeHorCursor)
                return
        
        # 检查是否悬停在控件上
        control = self._find_control_at(canvas_pos)
        if control:
            self.setCursor(Qt.PointingHandCursor)
        else:
            self.setCursor(Qt.ArrowCursor)
    
    def dragEnterEvent(self, event):
        """
        拖拽进入事件处理函数
        当用户拖拽控件到画布上方时调用，用于判断是否接受拖放操作
        
        Args:
            event: 拖拽事件对象，包含拖拽数据信息
        """
        # ========== 检查拖拽数据格式 ==========
        # 检查拖拽事件中的MIME数据是否包含文本数据（控件类型字符串）
        if event.mimeData().hasText():
            # 如果包含文本数据，接受建议的拖放操作（允许在画布上放下控件）
            event.acceptProposedAction()
    
    def dropEvent(self, event):
        """
        拖拽放下事件处理函数
        当用户将控件拖放到画布上时调用，在指定位置创建新控件
        
        Args:
            event: 拖拽事件对象，包含放下位置和拖拽数据信息
        """
        # ========== 检查拖拽数据格式 ==========
        # 检查拖拽事件中的MIME数据是否包含文本数据（控件类型字符串）
        if event.mimeData().hasText():
            # ========== 获取控件类型 ==========
            # 从MIME数据中获取控件类型字符串（如"Label"、"Button"等）
            control_type = event.mimeData().text()
            
            # ========== 获取放下位置 ==========
            # 获取鼠标在画布上的放下位置（窗口坐标）
            pos = event.pos()
            
            # ========== 转换为画布坐标 ==========
            # 将窗口坐标转换为画布坐标（考虑缩放因子）
            # 如果画布被缩放，需要除以缩放因子得到真实坐标
            canvas_pos = QPoint(int(pos.x() / self.scale_factor), int(pos.y() / self.scale_factor))
            
            # ========== 计算顶部边界 ==========
            # 计算画布顶部边界位置（标题栏和菜单栏的高度）
            # 标题栏高度：self.title_bar_height
            # 下拉菜单高度：如果有下拉菜单则为24像素，否则为0
            # 菜单栏高度：如果有菜单栏则为24像素，否则为0
            top_y = (
                self.title_bar_height +  # 标题栏高度
                (24 if (getattr(self, 'window_dropdown_menus', []) or []) else 0) +  # 下拉菜单高度
                (24 if (getattr(self, 'window_menus', []) or []) else 0)  # 菜单栏高度
            )
            
            # ========== 确定目标Y坐标 ==========
            # 确定控件的目标Y坐标（确保不在标题栏和菜单栏上方）
            # 使用max函数确保目标Y坐标不小于顶部边界
            target_y = max(top_y, canvas_pos.y())
            
            # ========== 添加控件 ==========
            # 在画布上的指定位置添加新控件
            # 参数：控件类型、X坐标、Y坐标
            control = self.add_control(control_type, canvas_pos.x(), target_y)
            
            # ========== 选中新添加的控件 ==========
            # 检查控件是否创建成功
            if control:
                # 如果控件创建成功，自动选中该控件（便于用户立即编辑属性）
                self.select_control(control.id)
            
            # ========== 接受拖放操作 ==========
            # 接受建议的拖放操作，完成拖放流程
            event.acceptProposedAction()
    
    def keyPressEvent(self, event):
        """
        键盘按下事件处理函数
        处理键盘快捷键操作，如撤销、删除、方向键移动等
        
        Args:
            event: 键盘事件对象，包含按下的键和修饰键信息
        """
        # ========== 获取按键信息 ==========
        # 获取按下的键代码
        key = event.key()
        
        # ========== 处理Ctrl+Z撤销操作 ==========
        # 检查是否为Z键且同时按下了Ctrl键（撤销操作）
        if key == Qt.Key_Z and (event.modifiers() & Qt.ControlModifier):
            # ========== 查找主窗口 ==========
            # 从画布的父窗口向上查找主窗口（包含撤销功能）
            main = self.parent()
            # ========== 向上遍历父窗口 ==========
            # 循环向上查找父窗口，直到找到包含撤销功能的主窗口
            while main:
                # ========== 检查是否有撤销方法 ==========
                # 检查当前窗口是否有_undo方法（撤销功能）
                if hasattr(main, '_undo'):
                    # 如果找到了撤销方法，尝试调用它
                    try:
                        # 调用主窗口的撤销方法
                        main._undo()
                    except Exception:
                        # 如果调用失败，忽略异常（不中断程序运行）
                        pass
                    # 找到主窗口后，退出循环
                    break
                # ========== 继续向上查找 ==========
                # 如果当前窗口不是主窗口，继续向上查找父窗口
                main = main.parent()
            # 处理完撤销操作后，直接返回（不继续处理其他按键）
            return
        
        # ========== 处理Delete键删除操作 ==========
        # 检查是否为Delete键（删除选中控件）
        if key == Qt.Key_Delete:
            # 如果按下Delete键，删除当前选中的控件
            self.delete_selected_control()
        
        # ========== 处理方向键移动操作 ==========
        # 检查是否为方向键（左、右、上、下）
        elif key in [Qt.Key_Left, Qt.Key_Right, Qt.Key_Up, Qt.Key_Down]:
            # ========== 获取选中的控件 ==========
            # 获取当前选中的所有控件ID列表
            ids = self.control_manager.get_selected_ids()
            
            # ========== 检查是否有选中的控件 ==========
            # 检查是否有选中的控件
            if ids:
                # ========== 初始化移动偏移量 ==========
                # 初始化X和Y方向的移动偏移量（初始为0）
                dx, dy = 0, 0
                
                # ========== 计算基础移动步长 ==========
                # 计算基础移动步长：如果启用网格吸附，步长为网格大小；否则为1像素
                base_step = self.grid_size if self.snap_to_grid else 1
                # 设置当前移动步长为基础步长
                step = base_step
                
                # ========== 检查是否按住Shift键 ==========
                # 检查是否同时按下了Shift键（快速移动）
                if event.modifiers() & Qt.ShiftModifier:
                    # 如果按住Shift键，移动步长增加10倍（快速移动）
                    step = base_step * 10
                
                # ========== 根据方向键确定移动方向 ==========
                # 根据按下的方向键确定移动方向和偏移量
                if key == Qt.Key_Left:
                    # 如果按下左方向键，向左移动（X减少）
                    dx = -step
                elif key == Qt.Key_Right:
                    # 如果按下右方向键，向右移动（X增加）
                    dx = step
                elif key == Qt.Key_Up:
                    # 如果按下上方向键，向上移动（Y减少）
                    dy = -step
                elif key == Qt.Key_Down:
                    # 如果按下下方向键，向下移动（Y增加）
                    dy = step
                # ========== 计算顶部边界 ==========
                # 计算画布顶部边界位置（标题栏和菜单栏的高度）
                top_y = (
                    self.title_bar_height +  # 标题栏高度
                    (24 if (getattr(self, 'window_dropdown_menus', []) or []) else 0) +  # 下拉菜单高度
                    (24 if (getattr(self, 'window_menus', []) or []) else 0)  # 菜单栏高度
                )
                
                # ========== 准备移动操作 ==========
                # 获取第一个控件的ID（用于发送移动信号）
                first_id = ids[0]
                # 初始化第一个控件的移动偏移量（初始为0）
                first_dx = 0
                first_dy = 0
                
                # ========== 遍历所有选中的控件并移动 ==========
                # 遍历所有选中的控件ID，逐个移动控件
                for cid in ids:
                    # ========== 获取控件对象 ==========
                    # 从控件管理器中获取控件对象
                    cobj = self.control_manager.controls.get(cid)
                    
                    # ========== 检查控件是否存在 ==========
                    # 检查控件对象是否存在（不为None）
                    if not cobj:
                        # 如果控件不存在，跳过该控件，继续处理下一个
                        continue
                    
                    # ========== 计算新位置 ==========
                    # 计算控件移动后的新X坐标（当前X坐标加上移动偏移量）
                    new_x = cobj.x + dx
                    # 计算控件移动后的新Y坐标（当前Y坐标加上移动偏移量）
                    new_y = cobj.y + dy
                    
                    # ========== 限制X坐标范围 ==========
                    # 限制新X坐标在画布范围内（不能小于0，不能超出画布右边界）
                    # 右边界 = 画布宽度/缩放因子 - 控件宽度
                    new_x = max(0, min(new_x, int(self.canvas_width / self.scale_factor) - cobj.width))
                    
                    # ========== 限制Y坐标范围 ==========
                    # 限制新Y坐标在画布范围内（不能小于顶部边界，不能超出画布底边界）
                    # 底边界 = 画布高度/缩放因子 - 控件高度
                    new_y = max(top_y, min(new_y, int(self.canvas_height / self.scale_factor) - cobj.height))
                    
                    # ========== 计算实际移动距离 ==========
                    # 计算控件实际移动的X方向距离（考虑边界限制后的实际移动距离）
                    ddx = new_x - cobj.x
                    # 计算控件实际移动的Y方向距离（考虑边界限制后的实际移动距离）
                    ddy = new_y - cobj.y
                    
                    # ========== 保存第一个控件的移动距离 ==========
                    # 检查当前控件是否为第一个控件
                    if cid == first_id:
                        # 如果是第一个控件，保存其移动距离（用于发送信号）
                        first_dx = ddx
                        first_dy = ddy
                    
                    # ========== 执行移动操作 ==========
                    # 调用控件管理器的移动方法，实际移动控件到新位置
                    self.control_manager.move_control(cid, ddx, ddy)
                
                # ========== 发送移动信号 ==========
                # 发送控件移动信号，通知其他组件控件已移动（使用第一个控件的移动距离）
                self.control_moved.emit(first_id, first_dx, first_dy)
                
                # ========== 更新画布显示 ==========
                # 更新画布显示，刷新控件的新位置
                self.update()
        elif key == Qt.Key_A and (event.modifiers() & Qt.ControlModifier):
            ids = []
            for c in self.control_manager.controls.values():
                if c.properties.get('page', '页面1') == getattr(self, 'active_page', '页面1'):
                    ids.append(c.id)
            self.control_manager.set_selection(ids)
            last_ids = self.control_manager.get_selected_ids()
            last = last_ids[-1] if last_ids else None
            try:
                self.control_selected.emit(last or 'canvas')
            except Exception:
                pass
            self.update()
        
        # Ctrl+C 复制选中控件
        elif key == Qt.Key_C and (event.modifiers() & Qt.ControlModifier):
            mapping = self.control_manager.copy_selected_controls()
            if mapping:
                # 复制成功，可以在这里添加一些用户反馈
                pass
        
        # Ctrl+V 粘贴控件
        elif key == Qt.Key_V and (event.modifiers() & Qt.ControlModifier):
            new_ids = self.control_manager.paste_controls()
            if new_ids:
                # 粘贴成功，选中新控件
                for new_id in new_ids:
                    self.control_added.emit(new_id)
                self.update()
    
    def toggle_grid(self):
        """切换网格显示"""
        self.show_grid = not self.show_grid
        self.update()
    
    def toggle_snap_to_grid(self):
        """切换网格对齐"""
        self.snap_to_grid = not self.snap_to_grid

    def contextMenuEvent(self, event):
        menu = QMenu(self)
        
        # 获取当前选中的控件
        selected_controls = self.control_manager.get_selected_ids() if hasattr(self.control_manager, 'get_selected_ids') else []
        has_selection = len(selected_controls) > 0
        
        # 编辑操作
        if has_selection:
            # 复制
            act_copy = menu.addAction("复制")
            act_copy.setShortcut("Ctrl+C")
            act_copy.triggered.connect(self._copy_selected_controls)
            
            # 粘贴
            act_paste = menu.addAction("粘贴")
            act_paste.setShortcut("Ctrl+V")
            act_paste.triggered.connect(self._paste_controls)
            
            # 删除
            act_delete = menu.addAction("删除")
            act_delete.setShortcut("Delete")
            act_delete.triggered.connect(self._delete_selected_controls)
            
            menu.addSeparator()
            
            # 撤销
            act_undo = menu.addAction("撤销")
            act_undo.setShortcut("Ctrl+Z")
            act_undo.triggered.connect(self._undo)
            
            # 重做
            act_redo = menu.addAction("重做")
            act_redo.setShortcut("Ctrl+Y")
            act_redo.triggered.connect(self._redo)
            
            menu.addSeparator()
        
        # 画布设置
        act_grid = menu.addAction("显示网格")
        act_grid.setCheckable(True)
        act_grid.setChecked(self.show_grid)
        act_snap = menu.addAction("网格对齐")
        act_snap.setCheckable(True)
        act_snap.setChecked(self.snap_to_grid)
        
        menu.addSeparator()
        
        # 工具
        act_page_editor = menu.addAction("分页编辑器")
        act_menu_editor = menu.addAction("菜单编辑器")
        act_window_manager = menu.addAction("窗口类管理")
        act_window_manager.setShortcut("Ctrl+W")
        
        chosen = menu.exec(event.globalPos())
        if chosen == act_grid:
            self.set_show_grid(act_grid.isChecked())
        elif chosen == act_snap:
            self.set_snap_to_grid(act_snap.isChecked())
        elif chosen == act_page_editor:
            dlg = self._open_page_editor()
            if dlg:
                self.window_menus = dlg
                if self.window_menus:
                    self.active_index = 0
                    self.active_page = self.window_menus[0]
                    for c in self.control_manager.controls.values():
                        pg = c.properties.get('page', '页面1')
                        if (pg == '页面1') or (pg not in self.window_menus):
                            c.properties['page'] = self.window_menus[0]
                self.update()
        elif chosen == act_menu_editor:
            mgr = MenusManagerDialog(self, getattr(self, 'window_dropdown_menus', []))
            if mgr.exec():
                self.window_dropdown_menus = mgr.get_menus()
                self.update()
        elif chosen == act_window_manager:
            # 通过主窗口调用窗口类管理
            main = self.parent()
            while main and not hasattr(main, 'window_class_manager'):
                main = main.parent()
            if main and hasattr(main, '_show_window_class_manager'):
                main._show_window_class_manager()
            # 通过主窗口调用窗口类管理
            main = self.parent()
            while main and not hasattr(main, 'window_class_manager'):
                main = main.parent()
            if main and hasattr(main, '_show_window_class_manager'):
                main._show_window_class_manager()
    
    def _copy_selected_controls(self):
        """复制选中的控件"""
        main = self.parent()
        while main and not hasattr(main, '_copy_selected_control'):
            main = main.parent()
        if main and hasattr(main, '_copy_selected_control'):
            main._copy_selected_control()
    
    def _paste_controls(self):
        """粘贴控件"""
        main = self.parent()
        while main and not hasattr(main, '_paste_control'):
            main = main.parent()
        if main and hasattr(main, '_paste_control'):
            main._paste_control()
    
    def _delete_selected_controls(self):
        """删除选中的控件"""
        main = self.parent()
        while main and not hasattr(main, '_delete_selected'):
            main = main.parent()
        if main and hasattr(main, '_delete_selected'):
            main._delete_selected()
    
    def _undo_from_menu(self):
        """从菜单撤销"""
        main = self.parent()
        while main and not hasattr(main, '_undo'):
            main = main.parent()
        if main and hasattr(main, '_undo'):
            main._undo()
    
    def _redo_from_menu(self):
        """从菜单重做"""
        main = self.parent()
        while main and not hasattr(main, '_redo'):
            main = main.parent()
        if main and hasattr(main, '_redo'):
            main._redo()
            # 窗口类管理
            if hasattr(self, 'main_window') and hasattr(self.main_window, 'project_manager'):
                if hasattr(self.main_window.project_manager, 'window_class_manager'):
                    from module.window_class_dialog import WindowClassDialog
                    dlg = WindowClassDialog(self, self.main_window.project_manager.window_class_manager)
                    if dlg.exec():
                        # 刷新预览
                        if hasattr(self.main_window, '_refresh_project_preview'):
                            self.main_window._refresh_project_preview()
                        self.main_window.project_manager.set_modified(True)

    def _open_page_editor(self):
        d = QDialog(self)
        d.setWindowTitle("分页编辑器")
        form = QFormLayout(d)
        edit = QTextEdit()
        edit.setPlainText("\n".join(self.window_menus) if self.window_menus else "")
        form.addRow("菜单项(每行一个):", edit)
        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        form.addRow(buttons)
        def accept():
            d.accept()
        def reject():
            d.reject()
        buttons.accepted.connect(accept)
        buttons.rejected.connect(reject)
        if d.exec():
            text = edit.toPlainText().splitlines()
            return [t.strip() for t in text if t.strip()]
        return None

    def _open_dropdown_menu_editor(self):
        pass
    
    def set_grid_size(self, size: int):
        """设置网格大小"""
        self.grid_size = max(5, min(50, size))
        self.update()    
    def set_scale(self, scale_factor: float):
        """设置画布缩放比例
        
        Args:
            scale_factor: 缩放因子，1.0表示原始大小
        """
        # 限制缩放范围
        scale_factor = max(self.min_scale, min(self.max_scale, scale_factor))
        
        if scale_factor != self.scale_factor:
            self.scale_factor = scale_factor
            
            # 根据缩放因子调整画布大小
            scaled_width = int(self.canvas_width * scale_factor)
            scaled_height = int(self.canvas_height * scale_factor)
            
            # 更新画布显示大小
            self.setMinimumSize(scaled_width, scaled_height)
            self.resize(scaled_width, scaled_height)
            
            # 更新显示
            self.update()
    
    def zoom_in(self):
        """放大画布"""
        new_scale = self.scale_factor + self.scale_step
        self.set_scale(new_scale)
    
    def zoom_out(self):
        """缩小画布"""
        new_scale = self.scale_factor - self.scale_step
        self.set_scale(new_scale)
    
    def zoom_reset(self):
        """重置画布缩放"""
        self.set_scale(1.0)
    
    def get_scale(self) -> float:
        """获取当前缩放比例"""
        return self.scale_factor
    def _build_dropdown_menu(self, menu_def):
        if not isinstance(menu_def, dict):
            return None
        menu = QMenu(self)
        items = menu_def.get('items', [])
        stack = [menu]
        current_level = 0
        for it in items:
            label = it.get('label', '')
            level = int(it.get('level', 0))
            if label.strip() in ['-', '—']:
                stack[-1].addSeparator()
                continue
            while level < current_level and len(stack) > 1:
                stack.pop()
                current_level -= 1
            while level > current_level:
                sub = QMenu(stack[-1])
                sub.setTitle(label or '子菜单')
                stack[-1].addMenu(sub)
                stack.append(sub)
                current_level += 1
                label = ''
            if label:
                act = QAction(label, self)
                sc = it.get('shortcut', '(无)')
                if sc and sc != '(无)':
                    act.setShortcut(sc)
                if it.get('checked', False):
                    act.setCheckable(True)
                    act.setChecked(True)
                act.setEnabled(it.get('allow', True))
                act.setVisible(it.get('visible', True))
                stack[-1].addAction(act)
        return menu
