#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
选择控件库模块
包含所有选择类控件的定义和实现
提供复选框、单选按钮、下拉框、列表、树形控件等选择控件
"""

# 导入类型提示
from typing import Dict, Any, Type, List
# 导入PySide6选择控件组件
from PySide6.QtWidgets import (
    QCheckBox,  # 复选框（多选）
    QRadioButton,  # 单选按钮（单选）
    QComboBox,  # 下拉选择框
    QListWidget,  # 列表控件
    QTreeWidget,  # 树形控件
    QTreeWidgetItem,  # 树形控件项
    QTableWidget,  # 表格控件
    QTableWidgetItem,  # 表格项
    QWidget,  # 基础窗口部件
    QButtonGroup,  # 按钮组（用于单选按钮分组）
    QListView,  # 列表视图
    QTreeView,  # 树形视图
    QTableView  # 表格视图
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QStandardItemModel,  # 标准项模型（用于视图控件）
    QStandardItem,  # 标准项
    QFont  # 字体（重复导入，保留一个即可）
)
# 导入PySide6核心功能
from PySide6.QtCore import Qt  # Qt命名空间和常量

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class CheckBoxControl(BaseControl):
    """复选框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "CheckBox"
        self._category = "选择"
        self._description = "用于选择或取消选择选项的控件"
        self._size = (120, 30)
        self._properties = {
            "text": "复选框",
            "checked": False,
            "tristate": False,
            "checkState": Qt.CheckState.Unchecked,
            "autoExclusive": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QCheckBox
    
    def create_instance(self, parent=None) -> QWidget:
        check_box = QCheckBox(self.get_property("text", "复选框"), parent)
        check_box.setChecked(self.get_property("checked", False))
        check_box.setTristate(self.get_property("tristate", False))
        check_box.setCheckState(self.get_property("checkState", Qt.CheckState.Unchecked))
        check_box.setAutoExclusive(self.get_property("autoExclusive", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            check_box.resize(width, height)
        
        return check_box


class RadioButtonControl(BaseControl):
    """单选按钮控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "RadioButton"
        self._category = "选择"
        self._description = "用于从多个选项中选择一个的控件"
        self._size = (120, 30)
        self._properties = {
            "text": "单选按钮",
            "checked": False,
            "autoExclusive": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QRadioButton
    
    def create_instance(self, parent=None) -> QWidget:
        radio_button = QRadioButton(self.get_property("text", "单选按钮"), parent)
        radio_button.setChecked(self.get_property("checked", False))
        radio_button.setAutoExclusive(self.get_property("autoExclusive", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            radio_button.resize(width, height)
        
        return radio_button


class ComboBoxControl(BaseControl):
    """下拉框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ComboBox"
        self._category = "选择"
        self._description = "用于从下拉列表中选择一个选项的控件"
        self._size = (120, 30)
        self._properties = {
            "items": ["选项1", "选项2", "选项3"],
            "currentIndex": 0,
            "editable": False,
            "insertPolicy": QComboBox.InsertPolicy.InsertAtBottom,
            "maxVisibleItems": 10,
            "duplicatesEnabled": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QComboBox
    
    def create_instance(self, parent=None) -> QWidget:
        combo_box = QComboBox(parent)
        
        # 添加项目
        items = self.get_property("items", ["选项1", "选项2", "选项3"])
        for item in items:
            combo_box.addItem(item)
        
        combo_box.setCurrentIndex(self.get_property("currentIndex", 0))
        combo_box.setEditable(self.get_property("editable", False))
        combo_box.setInsertPolicy(self.get_property("insertPolicy", QComboBox.InsertPolicy.InsertAtBottom))
        combo_box.setMaxVisibleItems(self.get_property("maxVisibleItems", 10))
        combo_box.setDuplicatesEnabled(self.get_property("duplicatesEnabled", False))
        combo_box.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            combo_box.resize(width, height)
        
        return combo_box


class ListWidgetControl(BaseControl):
    """列表控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ListWidget"
        self._category = "选择"
        self._description = "用于显示和选择项目列表的控件"
        self._size = (150, 150)
        self._properties = {
            "items": ["项目1", "项目2", "项目3"],
            "selectionMode": QListWidget.SelectionMode.SingleSelection,
            "selectionBehavior": QListWidget.SelectionBehavior.SelectRows,
            "alternatingRowColors": False,
            "sortingEnabled": False,
            "dragDropMode": QListWidget.DragDropMode.NoDragDrop,
            "editTriggers": QListWidget.EditTrigger.NoEditTriggers
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QListWidget
    
    def create_instance(self, parent=None) -> QWidget:
        list_widget = QListWidget(parent)
        
        # 添加项目
        items = self.get_property("items", ["项目1", "项目2", "项目3"])
        for item in items:
            list_widget.addItem(item)
        
        list_widget.setSelectionMode(self.get_property("selectionMode", QListWidget.SelectionMode.SingleSelection))
        list_widget.setSelectionBehavior(self.get_property("selectionBehavior", QListWidget.SelectionBehavior.SelectRows))
        list_widget.setAlternatingRowColors(self.get_property("alternatingRowColors", False))
        list_widget.setSortingEnabled(self.get_property("sortingEnabled", False))
        list_widget.setDragDropMode(self.get_property("dragDropMode", QListWidget.DragDropMode.NoDragDrop))
        list_widget.setEditTriggers(self.get_property("editTriggers", QListWidget.EditTrigger.NoEditTriggers))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            list_widget.resize(width, height)
        
        return list_widget


class TreeWidgetControl(BaseControl):
    """树形控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TreeWidget"
        self._category = "选择"
        self._description = "用于显示和选择树形结构数据的控件"
        self._size = (200, 200)
        self._properties = {
            "headerLabels": ["列1", "列2"],
            "selectionMode": QTreeWidget.SelectionMode.SingleSelection,
            "selectionBehavior": QTreeWidget.SelectionBehavior.SelectRows,
            "alternatingRowColors": False,
            "sortingEnabled": False,
            "rootIsDecorated": True,
            "items": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTreeWidget
    
    def create_instance(self, parent=None) -> QWidget:
        tree_widget = QTreeWidget(parent)
        
        # 设置表头
        header_labels = self.get_property("headerLabels", ["列1", "列2"])
        tree_widget.setHeaderLabels(header_labels)
        
        # 添加示例数据
        items = self.get_property("items", [])
        if not items:  # 如果没有提供数据，添加一些示例数据
            root_item1 = QTreeWidgetItem(["根节点1", "值1"])
            root_item2 = QTreeWidgetItem(["根节点2", "值2"])
            
            child_item1 = QTreeWidgetItem(["子节点1", "子值1"])
            child_item2 = QTreeWidgetItem(["子节点2", "子值2"])
            
            root_item1.addChild(child_item1)
            root_item1.addChild(child_item2)
            
            tree_widget.addTopLevelItem(root_item1)
            tree_widget.addTopLevelItem(root_item2)
        else:
            # 添加自定义数据
            for item_data in items:
                item = QTreeWidgetItem(item_data)
                tree_widget.addTopLevelItem(item)
        
        tree_widget.setSelectionMode(self.get_property("selectionMode", QTreeWidget.SelectionMode.SingleSelection))
        tree_widget.setSelectionBehavior(self.get_property("selectionBehavior", QTreeWidget.SelectionBehavior.SelectRows))
        tree_widget.setAlternatingRowColors(self.get_property("alternatingRowColors", False))
        tree_widget.setSortingEnabled(self.get_property("sortingEnabled", False))
        tree_widget.setRootIsDecorated(self.get_property("rootIsDecorated", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            tree_widget.resize(width, height)
        
        return tree_widget


class TableWidgetControl(BaseControl):
    """表格控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TableWidget"
        self._category = "选择"
        self._description = "用于显示和选择表格数据的控件"
        self._size = (250, 200)
        self._properties = {
            "rowCount": 5,
            "columnCount": 3,
            "horizontalHeaderLabels": ["列1", "列2", "列3"],
            "verticalHeaderLabels": ["行1", "行2", "行3", "行4", "行5"],
            "selectionMode": QTableWidget.SelectionMode.SingleSelection,
            "selectionBehavior": QTableWidget.SelectionBehavior.SelectItems,
            "alternatingRowColors": False,
            "sortingEnabled": False,
            "data": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTableWidget
    
    def create_instance(self, parent=None) -> QWidget:
        table_widget = QTableWidget(parent)
        
        # 设置行列数
        row_count = self.get_property("rowCount", 5)
        column_count = self.get_property("columnCount", 3)
        table_widget.setRowCount(row_count)
        table_widget.setColumnCount(column_count)
        
        # 设置表头
        h_header_labels = self.get_property("horizontalHeaderLabels", ["列1", "列2", "列3"])
        v_header_labels = self.get_property("verticalHeaderLabels", ["行1", "行2", "行3", "行4", "行5"])
        
        table_widget.setHorizontalHeaderLabels(h_header_labels)
        table_widget.setVerticalHeaderLabels(v_header_labels)
        
        # 添加示例数据
        data = self.get_property("data", [])
        if not data:  # 如果没有提供数据，添加一些示例数据
            for row in range(row_count):
                for col in range(column_count):
                    item = QTableWidgetItem(f"数据({row},{col})")
                    table_widget.setItem(row, col, item)
        else:
            # 添加自定义数据
            for row, row_data in enumerate(data):
                for col, cell_data in enumerate(row_data):
                    if row < row_count and col < column_count:
                        item = QTableWidgetItem(str(cell_data))
                        table_widget.setItem(row, col, item)
        
        table_widget.setSelectionMode(self.get_property("selectionMode", QTableWidget.SelectionMode.SingleSelection))
        table_widget.setSelectionBehavior(self.get_property("selectionBehavior", QTableWidget.SelectionBehavior.SelectItems))
        table_widget.setAlternatingRowColors(self.get_property("alternatingRowColors", False))
        table_widget.setSortingEnabled(self.get_property("sortingEnabled", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            table_widget.resize(width, height)
        
        return table_widget


class EditableComboBoxControl(BaseControl):
    """可编辑下拉框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "EditableComboBox"
        self._category = "选择"
        self._description = "可以编辑文本的下拉框控件"
        self._size = (120, 30)
        self._properties = {
            "items": ["选项1", "选项2", "选项3"],
            "currentIndex": 0,
            "editable": True,
            "insertPolicy": QComboBox.InsertPolicy.InsertAtBottom,
            "maxVisibleItems": 10,
            "duplicatesEnabled": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QComboBox
    
    def create_instance(self, parent=None) -> QWidget:
        combo_box = QComboBox(parent)
        
        # 添加项目
        items = self.get_property("items", ["选项1", "选项2", "选项3"])
        for item in items:
            combo_box.addItem(item)
        
        combo_box.setCurrentIndex(self.get_property("currentIndex", 0))
        combo_box.setEditable(self.get_property("editable", True))
        combo_box.setInsertPolicy(self.get_property("insertPolicy", QComboBox.InsertPolicy.InsertAtBottom))
        combo_box.setMaxVisibleItems(self.get_property("maxVisibleItems", 10))
        combo_box.setDuplicatesEnabled(self.get_property("duplicatesEnabled", False))
        combo_box.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            combo_box.resize(width, height)
        
        return combo_box


class MultiSelectListControl(BaseControl):
    """多选列表控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "MultiSelectList"
        self._category = "选择"
        self._description = "支持多选的列表控件"
        self._size = (150, 150)
        self._properties = {
            "items": ["项目1", "项目2", "项目3", "项目4", "项目5"],
            "selectionMode": QListWidget.SelectionMode.MultiSelection,
            "selectionBehavior": QListWidget.SelectionBehavior.SelectRows,
            "alternatingRowColors": True,
            "sortingEnabled": False,
            "dragDropMode": QListWidget.DragDropMode.NoDragDrop,
            "editTriggers": QListWidget.EditTrigger.NoEditTriggers
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QListWidget
    
    def create_instance(self, parent=None) -> QWidget:
        list_widget = QListWidget(parent)
        
        # 添加项目
        items = self.get_property("items", ["项目1", "项目2", "项目3", "项目4", "项目5"])
        for item in items:
            list_widget.addItem(item)
        
        list_widget.setSelectionMode(self.get_property("selectionMode", QListWidget.SelectionMode.MultiSelection))
        list_widget.setSelectionBehavior(self.get_property("selectionBehavior", QListWidget.SelectionBehavior.SelectRows))
        list_widget.setAlternatingRowColors(self.get_property("alternatingRowColors", True))
        list_widget.setSortingEnabled(self.get_property("sortingEnabled", False))
        list_widget.setDragDropMode(self.get_property("dragDropMode", QListWidget.DragDropMode.NoDragDrop))
        list_widget.setEditTriggers(self.get_property("editTriggers", QListWidget.EditTrigger.NoEditTriggers))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            list_widget.resize(width, height)
        
        return list_widget


class ListViewControl(BaseControl):
    def __init__(self):
        super().__init__()
        self._name = "ListView"
        self._category = "选择"
        self._description = "基于模型的列表视图"
        self._size = (180, 140)
        self._properties = {
            "items": ["项目1", "项目2", "项目3"],
            "selectionMode": QListView.SelectionMode.SingleSelection
        }

    def get_control_class(self) -> Type[QWidget]:
        return QListView

    def create_instance(self, parent=None) -> QWidget:
        view = QListView(parent)
        model = QStandardItemModel()
        for text in self.get_property("items", []):
            it = QStandardItem(str(text))
            model.appendRow(it)
        view.setModel(model)
        view.setSelectionMode(self.get_property("selectionMode", QListView.SelectionMode.SingleSelection))
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            view.resize(width, height)
        return view


class TreeViewControl(BaseControl):
    def __init__(self):
        super().__init__()
        self._name = "TreeView"
        self._category = "选择"
        self._description = "基于模型的树视图"
        self._size = (200, 160)
        self._properties = {
            "headers": ["名称", "类型"],
            "items": [["根节点1", "类型A"], ["根节点2", "类型B"]]
        }

    def get_control_class(self) -> Type[QWidget]:
        return QTreeView

    def create_instance(self, parent=None) -> QWidget:
        view = QTreeView(parent)
        model = QStandardItemModel()
        headers = self.get_property("headers", [])
        model.setHorizontalHeaderLabels(headers)
        for row in self.get_property("items", []):
            items = [QStandardItem(str(x)) for x in row]
            model.appendRow(items)
        view.setModel(model)
        view.expandAll()
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            view.resize(width, height)
        return view


class TableViewControl(BaseControl):
    def __init__(self):
        super().__init__()
        self._name = "TableView"
        self._category = "选择"
        self._description = "基于模型的表视图"
        self._size = (240, 160)
        self._properties = {
            "rowCount": 3,
            "columnCount": 3,
            "headers": ["列1", "列2", "列3"],
            "data": []
        }

    def get_control_class(self) -> Type[QWidget]:
        return QTableView

    def create_instance(self, parent=None) -> QWidget:
        view = QTableView(parent)
        model = QStandardItemModel()
        model.setColumnCount(self.get_property("columnCount", 3))
        model.setRowCount(self.get_property("rowCount", 3))
        headers = self.get_property("headers", [])
        model.setHorizontalHeaderLabels(headers)
        data = self.get_property("data", [])
        if data:
            for r, row in enumerate(data):
                for c, val in enumerate(row):
                    model.setItem(r, c, QStandardItem(str(val)))
        else:
            for r in range(model.rowCount()):
                for c in range(model.columnCount()):
                    model.setItem(r, c, QStandardItem(f"数据({r},{c})"))
        view.setModel(model)
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            view.resize(width, height)
        return view


# 注册所有选择控件
def register_selection_controls():
    """注册所有选择控件到控件注册表"""
    control_registry.register_control(CheckBoxControl())
    control_registry.register_control(RadioButtonControl())
    control_registry.register_control(ComboBoxControl())
    control_registry.register_control(ListWidgetControl())
    control_registry.register_control(TreeWidgetControl())
    control_registry.register_control(TableWidgetControl())
    control_registry.register_control(EditableComboBoxControl())
    control_registry.register_control(MultiSelectListControl())
    control_registry.register_control(ListViewControl())
    control_registry.register_control(TreeViewControl())
    control_registry.register_control(TableViewControl())


# 自动注册
register_selection_controls()