#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
布局控件库模块
包含所有布局类控件的定义和实现
提供水平布局、垂直布局、网格布局、表单布局等布局控件
"""

# 导入类型提示
from typing import Dict, Any, Type, List
# 导入PySide6布局控件组件
from PySide6.QtWidgets import (
    QWidget,  # 基础窗口部件
    QHBoxLayout,  # 水平布局管理器
    QVBoxLayout,  # 垂直布局管理器
    QGridLayout,  # 网格布局管理器
    QFormLayout,  # 表单布局管理器
    QSplitter,  # 分割器（可调整大小的分隔条）
    QTabWidget,  # 选项卡控件（标签页）
    QStackedWidget,  # 堆叠控件（页面堆叠）
    QToolBox,  # 工具箱控件
    QDockWidget,  # 停靠窗口
    QScrollArea,  # 滚动区域
    QFrame,  # 框架
    QMdiArea,  # MDI区域（多文档界面）
    QMdiSubWindow  # MDI子窗口
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QSize  # 大小对象
)

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class HorizontalLayoutControl(BaseControl):
    """水平布局控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "HorizontalLayout"
        self._category = "布局"
        self._description = "水平排列控件的布局"
        self._size = (300, 100)
        self._properties = {
            "spacing": 6,
            "margin": 9,
            "widgets": [],
            "stretch": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QWidget
    
    def create_instance(self, parent=None) -> QWidget:
        widget = QWidget(parent)
        layout = QHBoxLayout()
        
        layout.setSpacing(self.get_property("spacing", 6))
        layout.setContentsMargins(
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9)
        )
        
        # 添加示例控件
        widgets = self.get_property("widgets", [])
        if not widgets:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QLabel, QPushButton, QLineEdit
            
            label = QLabel("标签:")
            line_edit = QLineEdit("文本框")
            button = QPushButton("按钮")
            
            layout.addWidget(label)
            layout.addWidget(line_edit)
            layout.addWidget(button)
        else:
            # 添加自定义控件
            for w in widgets:
                layout.addWidget(w)
        
        # 设置拉伸因子
        stretch = self.get_property("stretch", [])
        for i, factor in enumerate(stretch):
            if i < layout.count():
                layout.setStretch(i, factor)
        
        widget.setLayout(layout)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            widget.resize(width, height)
        
        return widget


class VerticalLayoutControl(BaseControl):
    """垂直布局控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "VerticalLayout"
        self._category = "布局"
        self._description = "垂直排列控件的布局"
        self._size = (200, 150)
        self._properties = {
            "spacing": 6,
            "margin": 9,
            "widgets": [],
            "stretch": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QWidget
    
    def create_instance(self, parent=None) -> QWidget:
        widget = QWidget(parent)
        layout = QVBoxLayout()
        
        layout.setSpacing(self.get_property("spacing", 6))
        layout.setContentsMargins(
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9)
        )
        
        # 添加示例控件
        widgets = self.get_property("widgets", [])
        if not widgets:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QLabel, QPushButton, QLineEdit, QCheckBox
            
            label = QLabel("标签:")
            line_edit = QLineEdit("文本框")
            check_box = QCheckBox("复选框")
            button = QPushButton("按钮")
            
            layout.addWidget(label)
            layout.addWidget(line_edit)
            layout.addWidget(check_box)
            layout.addWidget(button)
        else:
            # 添加自定义控件
            for w in widgets:
                layout.addWidget(w)
        
        # 设置拉伸因子
        stretch = self.get_property("stretch", [])
        for i, factor in enumerate(stretch):
            if i < layout.count():
                layout.setStretch(i, factor)
        
        widget.setLayout(layout)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            widget.resize(width, height)
        
        return widget


class GridLayoutControl(BaseControl):
    """网格布局控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "GridLayout"
        self._category = "布局"
        self._description = "网格排列控件的布局"
        self._size = (300, 200)
        self._properties = {
            "rowCount": 3,
            "columnCount": 3,
            "spacing": 6,
            "margin": 9,
            "widgets": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QWidget
    
    def create_instance(self, parent=None) -> QWidget:
        widget = QWidget(parent)
        layout = QGridLayout()
        
        layout.setSpacing(self.get_property("spacing", 6))
        layout.setContentsMargins(
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9)
        )
        
        # 添加示例控件
        widgets = self.get_property("widgets", [])
        if not widgets:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QLabel, QPushButton, QLineEdit
            
            # 创建3x3网格的示例控件
            for row in range(3):
                for col in range(3):
                    if row == 0 and col == 0:
                        widget = QLabel("标签")
                    elif row == 0 and col == 1:
                        widget = QLineEdit("文本框")
                    elif row == 0 and col == 2:
                        widget = QPushButton("按钮")
                    elif row == 1 and col == 0:
                        widget = QLabel("更多:")
                    elif row == 1 and col == 1:
                        widget = QLineEdit("更多文本")
                    elif row == 1 and col == 2:
                        widget = QPushButton("更多按钮")
                    elif row == 2 and col == 0:
                        widget = QLabel("底部:")
                    elif row == 2 and col == 1:
                        widget = QLineEdit("底部文本")
                    else:
                        widget = QPushButton("底部按钮")
                    
                    layout.addWidget(widget, row, col)
        else:
            # 添加自定义控件
            for item in widgets:
                widget = item.get("widget")
                row = item.get("row", 0)
                col = item.get("col", 0)
                row_span = item.get("rowSpan", 1)
                col_span = item.get("colSpan", 1)
                
                layout.addWidget(widget, row, col, row_span, col_span)
        
        widget.setLayout(layout)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            widget.resize(width, height)
        
        return widget


class FormLayoutControl(BaseControl):
    """表单布局控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "FormLayout"
        self._category = "布局"
        self._description = "表单样式的布局，通常用于标签和输入框配对"
        self._size = (300, 150)
        self._properties = {
            "spacing": 6,
            "margin": 9,
            "rows": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QWidget
    
    def create_instance(self, parent=None) -> QWidget:
        widget = QWidget(parent)
        layout = QFormLayout()
        
        layout.setSpacing(self.get_property("spacing", 6))
        layout.setContentsMargins(
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9),
            self.get_property("margin", 9)
        )
        
        # 添加示例控件
        rows = self.get_property("rows", [])
        if not rows:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QLabel, QLineEdit, QComboBox, QSpinBox
            
            # 添加表单行
            layout.addRow("姓名:", QLineEdit())
            layout.addRow("年龄:", QSpinBox())
            layout.addRow("性别:", QComboBox())
            layout.addRow("地址:", QLineEdit())
        else:
            # 添加自定义控件
            for row in rows:
                label = row.get("label")
                field = row.get("field")
                layout.addRow(label, field)
        
        widget.setLayout(layout)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            widget.resize(width, height)
        
        return widget


class HorizontalSplitterControl(BaseControl):
    """水平分割器控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "HorizontalSplitter"
        self._category = "布局"
        self._description = "水平分割控件的容器"
        self._size = (400, 200)
        self._properties = {
            "orientation": Qt.Orientation.Horizontal,
            "childrenCollapsible": True,
            "opaqueResize": False,
            "handleWidth": 6,
            "widgets": [],
            "sizes": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QSplitter
    
    def create_instance(self, parent=None) -> QWidget:
        splitter = QSplitter(self.get_property("orientation", Qt.Orientation.Horizontal), parent)
        splitter.setChildrenCollapsible(self.get_property("childrenCollapsible", True))
        splitter.setOpaqueResize(self.get_property("opaqueResize", False))
        splitter.setHandleWidth(self.get_property("handleWidth", 6))
        
        # 添加示例控件
        widgets = self.get_property("widgets", [])
        if not widgets:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QTextEdit, QListWidget, QTableWidget
            
            text_edit = QTextEdit()
            text_edit.setPlainText("这是文本编辑区域")
            
            list_widget = QListWidget()
            list_widget.addItems(["项目1", "项目2", "项目3"])
            
            table_widget = QTableWidget(3, 2)
            table_widget.setHorizontalHeaderLabels(["列1", "列2"])
            
            splitter.addWidget(text_edit)
            splitter.addWidget(list_widget)
            splitter.addWidget(table_widget)
        else:
            # 添加自定义控件
            for w in widgets:
                splitter.addWidget(w)
        
        # 设置大小
        sizes = self.get_property("sizes", [])
        if sizes:
            splitter.setSizes(sizes)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            splitter.resize(width, height)
        
        return splitter


class VerticalSplitterControl(BaseControl):
    """垂直分割器控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "VerticalSplitter"
        self._category = "布局"
        self._description = "垂直分割控件的容器"
        self._size = (300, 300)
        self._properties = {
            "orientation": Qt.Orientation.Vertical,
            "childrenCollapsible": True,
            "opaqueResize": False,
            "handleWidth": 6,
            "widgets": [],
            "sizes": []
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QSplitter
    
    def create_instance(self, parent=None) -> QWidget:
        splitter = QSplitter(self.get_property("orientation", Qt.Orientation.Vertical), parent)
        splitter.setChildrenCollapsible(self.get_property("childrenCollapsible", True))
        splitter.setOpaqueResize(self.get_property("opaqueResize", False))
        splitter.setHandleWidth(self.get_property("handleWidth", 6))
        
        # 添加示例控件
        widgets = self.get_property("widgets", [])
        if not widgets:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QTextEdit, QListWidget, QTableWidget
            
            text_edit = QTextEdit()
            text_edit.setPlainText("这是文本编辑区域")
            
            list_widget = QListWidget()
            list_widget.addItems(["项目1", "项目2", "项目3"])
            
            table_widget = QTableWidget(3, 2)
            table_widget.setHorizontalHeaderLabels(["列1", "列2"])
            
            splitter.addWidget(text_edit)
            splitter.addWidget(list_widget)
            splitter.addWidget(table_widget)
        else:
            # 添加自定义控件
            for w in widgets:
                splitter.addWidget(w)
        
        # 设置大小
        sizes = self.get_property("sizes", [])
        if sizes:
            splitter.setSizes(sizes)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            splitter.resize(width, height)
        
        return splitter


class TabWidgetControl(BaseControl):
    """选项卡控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TabWidget"
        self._category = "布局"
        self._description = "带选项卡的容器控件"
        self._size = (300, 200)
        self._properties = {
            "tabPosition": QTabWidget.TabPosition.North,
            "tabShape": QTabWidget.TabShape.Rounded,
            "tabs": [],
            "currentIndex": 0,
            "movable": False,
            "closable": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTabWidget
    
    def create_instance(self, parent=None) -> QWidget:
        tab_widget = QTabWidget(parent)
        tab_widget.setTabPosition(self.get_property("tabPosition", QTabWidget.TabPosition.North))
        tab_widget.setTabShape(self.get_property("tabShape", QTabWidget.TabShape.Rounded))
        tab_widget.setCurrentIndex(self.get_property("currentIndex", 0))
        tab_widget.setMovable(self.get_property("movable", False))
        tab_widget.setTabsClosable(self.get_property("closable", False))
        
        # 添加示例选项卡
        tabs = self.get_property("tabs", [])
        if not tabs:  # 如果没有提供选项卡，添加一些示例选项卡
            from PySide6.QtWidgets import QLabel, QTextEdit, QListWidget
            
            # 第一个选项卡
            tab1 = QWidget()
            layout1 = QVBoxLayout()
            label1 = QLabel("这是第一个选项卡的内容")
            layout1.addWidget(label1)
            tab1.setLayout(layout1)
            tab_widget.addTab(tab1, "选项卡1")
            
            # 第二个选项卡
            tab2 = QWidget()
            layout2 = QVBoxLayout()
            text_edit = QTextEdit()
            text_edit.setPlainText("这是第二个选项卡的文本编辑器")
            layout2.addWidget(text_edit)
            tab2.setLayout(layout2)
            tab_widget.addTab(tab2, "选项卡2")
            
            # 第三个选项卡
            tab3 = QWidget()
            layout3 = QVBoxLayout()
            list_widget = QListWidget()
            list_widget.addItems(["项目1", "项目2", "项目3", "项目4"])
            layout3.addWidget(list_widget)
            tab3.setLayout(layout3)
            tab_widget.addTab(tab3, "选项卡3")
        else:
            # 添加自定义选项卡
            for tab in tabs:
                widget = tab.get("widget")
                title = tab.get("title", "选项卡")
                tab_widget.addTab(widget, title)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            tab_widget.resize(width, height)
        
        return tab_widget


class StackedWidgetControl(BaseControl):
    """堆叠窗口控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "StackedWidget"
        self._category = "布局"
        self._description = "堆叠显示多个控件的容器"
        self._size = (300, 200)
        self._properties = {
            "widgets": [],
            "currentIndex": 0
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QStackedWidget
    
    def create_instance(self, parent=None) -> QWidget:
        stacked_widget = QStackedWidget(parent)
        stacked_widget.setCurrentIndex(self.get_property("currentIndex", 0))
        
        # 添加示例控件
        widgets = self.get_property("widgets", [])
        if not widgets:  # 如果没有提供控件，添加一些示例控件
            from PySide6.QtWidgets import QLabel, QTextEdit, QListWidget
            
            # 第一个页面
            page1 = QWidget()
            layout1 = QVBoxLayout()
            label1 = QLabel("这是第一个页面")
            layout1.addWidget(label1)
            page1.setLayout(layout1)
            stacked_widget.addWidget(page1)
            
            # 第二个页面
            page2 = QWidget()
            layout2 = QVBoxLayout()
            text_edit = QTextEdit()
            text_edit.setPlainText("这是第二个页面的文本编辑器")
            layout2.addWidget(text_edit)
            page2.setLayout(layout2)
            stacked_widget.addWidget(page2)
            
            # 第三个页面
            page3 = QWidget()
            layout3 = QVBoxLayout()
            list_widget = QListWidget()
            list_widget.addItems(["项目1", "项目2", "项目3"])
            layout3.addWidget(list_widget)
            page3.setLayout(layout3)
            stacked_widget.addWidget(page3)
        else:
            # 添加自定义控件
            for w in widgets:
                stacked_widget.addWidget(w)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            stacked_widget.resize(width, height)
        
        return stacked_widget


# 注册所有布局控件
def register_layout_controls():
    """注册所有布局控件到控件注册表"""
    control_registry.register_control(HorizontalLayoutControl())
    control_registry.register_control(VerticalLayoutControl())
    control_registry.register_control(GridLayoutControl())
    control_registry.register_control(FormLayoutControl())
    control_registry.register_control(HorizontalSplitterControl())
    control_registry.register_control(VerticalSplitterControl())
    control_registry.register_control(TabWidgetControl())
    control_registry.register_control(StackedWidgetControl())


# 自动注册
register_layout_controls()