#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
输入控件库模块
包含所有输入类控件的定义和实现
提供文本输入、数字输入、日期时间输入等各种输入控件
"""

# 导入类型提示
from typing import Dict, Any, Type
# 导入PySide6输入控件组件
from PySide6.QtWidgets import (
    QTextEdit,  # 富文本编辑器（多行文本输入）
    QPlainTextEdit,  # 纯文本编辑器（多行文本输入）
    QSpinBox,  # 整数旋转框（整数输入）
    QDoubleSpinBox,  # 浮点数旋转框（浮点数输入）
    QSlider,  # 滑动条（数值输入）
    QDial,  # 旋钮（数值输入）
    QDateEdit,  # 日期编辑框（日期输入）
    QTimeEdit,  # 时间编辑框（时间输入）
    QDateTimeEdit,  # 日期时间编辑框（日期时间输入）
    QWidget  # 基础窗口部件
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QDate,  # 日期对象
    QTime,  # 时间对象
    QDateTime  # 日期时间对象
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QFont,  # 字体
    QTextOption  # 文本选项（用于文本换行等）
)

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class TextEditControl(BaseControl):
    """多行文本编辑控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TextEdit"
        self._category = "输入"
        self._description = "用于输入多行文本的编辑器"
        self._size = (200, 100)
        self._properties = {
            "plainText": "",
            "placeholderText": "请输入多行文本...",
            "readOnly": False,
            "acceptRichText": True,
            "html": "",
            "lineWrapMode": QTextEdit.LineWrapMode.WidgetWidth,
            "lineWrapColumnOrWidth": 80,
            "wordWrapMode": QTextOption.WrapMode.WordWrap
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTextEdit
    
    def create_instance(self, parent=None) -> QWidget:
        text_edit = QTextEdit(parent)
        text_edit.setPlainText(self.get_property("plainText", ""))
        text_edit.setPlaceholderText(self.get_property("placeholderText", "请输入多行文本..."))
        text_edit.setReadOnly(self.get_property("readOnly", False))
        text_edit.setAcceptRichText(self.get_property("acceptRichText", True))
        text_edit.setLineWrapMode(self.get_property("lineWrapMode", QTextEdit.LineWrapMode.WidgetWidth))
        text_edit.setLineWrapColumnOrWidth(self.get_property("lineWrapColumnOrWidth", 80))
        text_edit.setWordWrapMode(self.get_property("wordWrapMode", QTextOption.WrapMode.WordWrap))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            text_edit.resize(width, height)
        
        return text_edit


class SpinBoxControl(BaseControl):
    """整数输入框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "SpinBox"
        self._category = "输入"
        self._description = "用于输入整数的控件"
        self._size = (100, 30)
        self._properties = {
            "minimum": -2147483648,
            "maximum": 2147483647,
            "value": 0,
            "singleStep": 1,
            "prefix": "",
            "suffix": "",
            "displayIntegerBase": 10,
            "readOnly": False,
            "wrapping": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QSpinBox
    
    def create_instance(self, parent=None) -> QWidget:
        spin_box = QSpinBox(parent)
        spin_box.setMinimum(self.get_property("minimum", -2147483648))
        spin_box.setMaximum(self.get_property("maximum", 2147483647))
        spin_box.setValue(self.get_property("value", 0))
        spin_box.setSingleStep(self.get_property("singleStep", 1))
        spin_box.setPrefix(self.get_property("prefix", ""))
        spin_box.setSuffix(self.get_property("suffix", ""))
        spin_box.setDisplayIntegerBase(self.get_property("displayIntegerBase", 10))
        spin_box.setReadOnly(self.get_property("readOnly", False))
        spin_box.setWrapping(self.get_property("wrapping", False))
        spin_box.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            spin_box.resize(width, height)
        
        return spin_box


class DoubleSpinBoxControl(BaseControl):
    """浮点数输入框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "DoubleSpinBox"
        self._category = "输入"
        self._description = "用于输入浮点数的控件"
        self._size = (100, 30)
        self._properties = {
            "minimum": -2147483648.0,
            "maximum": 2147483647.0,
            "value": 0.0,
            "singleStep": 1.0,
            "decimals": 2,
            "prefix": "",
            "suffix": "",
            "readOnly": False,
            "wrapping": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QDoubleSpinBox
    
    def create_instance(self, parent=None) -> QWidget:
        spin_box = QDoubleSpinBox(parent)
        spin_box.setMinimum(self.get_property("minimum", -2147483648.0))
        spin_box.setMaximum(self.get_property("maximum", 2147483647.0))
        spin_box.setValue(self.get_property("value", 0.0))
        spin_box.setSingleStep(self.get_property("singleStep", 1.0))
        spin_box.setDecimals(self.get_property("decimals", 2))
        spin_box.setPrefix(self.get_property("prefix", ""))
        spin_box.setSuffix(self.get_property("suffix", ""))
        spin_box.setReadOnly(self.get_property("readOnly", False))
        spin_box.setWrapping(self.get_property("wrapping", False))
        spin_box.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            spin_box.resize(width, height)
        
        return spin_box


class SliderControl(BaseControl):
    """滑块控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "Slider"
        self._category = "输入"
        self._description = "用于通过滑动选择数值的控件"
        self._size = (150, 30)
        self._properties = {
            "minimum": 0,
            "maximum": 100,
            "value": 50,
            "singleStep": 1,
            "pageStep": 10,
            "orientation": Qt.Orientation.Horizontal,
            "tickPosition": QSlider.TickPosition.NoTicks,
            "tickInterval": 10,
            "tracking": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QSlider
    
    def create_instance(self, parent=None) -> QWidget:
        slider = QSlider(parent)
        slider.setMinimum(self.get_property("minimum", 0))
        slider.setMaximum(self.get_property("maximum", 100))
        slider.setValue(self.get_property("value", 50))
        slider.setSingleStep(self.get_property("singleStep", 1))
        slider.setPageStep(self.get_property("pageStep", 10))
        slider.setOrientation(self.get_property("orientation", Qt.Orientation.Horizontal))
        slider.setTickPosition(self.get_property("tickPosition", QSlider.TickPosition.NoTicks))
        slider.setTickInterval(self.get_property("tickInterval", 10))
        slider.setTracking(self.get_property("tracking", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            slider.resize(width, height)
        
        return slider


class DateEditControl(BaseControl):
    """日期选择控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "DateEdit"
        self._category = "输入"
        self._description = "用于选择日期的控件"
        self._size = (120, 30)
        self._properties = {
            "date": QDate.currentDate(),
            "minimumDate": QDate(1752, 9, 14),
            "maximumDate": QDate(7999, 12, 31),
            "displayFormat": "yyyy-MM-dd",
            "calendarPopup": True,
            "readOnly": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QDateEdit
    
    def create_instance(self, parent=None) -> QWidget:
        date_edit = QDateEdit(parent)
        date_edit.setDate(self.get_property("date", QDate.currentDate()))
        date_edit.setMinimumDate(self.get_property("minimumDate", QDate(1752, 9, 14)))
        date_edit.setMaximumDate(self.get_property("maximumDate", QDate(7999, 12, 31)))
        date_edit.setDisplayFormat(self.get_property("displayFormat", "yyyy-MM-dd"))
        date_edit.setCalendarPopup(self.get_property("calendarPopup", True))
        date_edit.setReadOnly(self.get_property("readOnly", False))
        date_edit.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            date_edit.resize(width, height)
        
        return date_edit


class TimeEditControl(BaseControl):
    """时间选择控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "TimeEdit"
        self._category = "输入"
        self._description = "用于选择时间的控件"
        self._size = (120, 30)
        self._properties = {
            "time": QTime.currentTime(),
            "minimumTime": QTime(0, 0, 0),
            "maximumTime": QTime(23, 59, 59),
            "displayFormat": "HH:mm:ss",
            "readOnly": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QTimeEdit
    
    def create_instance(self, parent=None) -> QWidget:
        time_edit = QTimeEdit(parent)
        time_edit.setTime(self.get_property("time", QTime.currentTime()))
        time_edit.setMinimumTime(self.get_property("minimumTime", QTime(0, 0, 0)))
        time_edit.setMaximumTime(self.get_property("maximumTime", QTime(23, 59, 59)))
        time_edit.setDisplayFormat(self.get_property("displayFormat", "HH:mm:ss"))
        time_edit.setReadOnly(self.get_property("readOnly", False))
        time_edit.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            time_edit.resize(width, height)
        
        return time_edit


class DateTimeEditControl(BaseControl):
    """日期时间选择控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "DateTimeEdit"
        self._category = "输入"
        self._description = "用于选择日期和时间的控件"
        self._size = (180, 30)
        self._properties = {
            "dateTime": QDateTime.currentDateTime(),
            "minimumDateTime": QDateTime(QDate(1752, 9, 14), QTime(0, 0, 0)),
            "maximumDateTime": QDateTime(QDate(7999, 12, 31), QTime(23, 59, 59)),
            "displayFormat": "yyyy-MM-dd HH:mm:ss",
            "calendarPopup": True,
            "readOnly": False,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QDateTimeEdit
    
    def create_instance(self, parent=None) -> QWidget:
        datetime_edit = QDateTimeEdit(parent)
        datetime_edit.setDateTime(self.get_property("dateTime", QDateTime.currentDateTime()))
        datetime_edit.setMinimumDateTime(self.get_property("minimumDateTime", 
                                                      QDateTime(QDate(1752, 9, 14), QTime(0, 0, 0))))
        datetime_edit.setMaximumDateTime(self.get_property("maximumDateTime", 
                                                      QDateTime(QDate(7999, 12, 31), QTime(23, 59, 59))))
        datetime_edit.setDisplayFormat(self.get_property("displayFormat", "yyyy-MM-dd HH:mm:ss"))
        datetime_edit.setCalendarPopup(self.get_property("calendarPopup", True))
        datetime_edit.setReadOnly(self.get_property("readOnly", False))
        datetime_edit.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            datetime_edit.resize(width, height)
        
        return datetime_edit


class PlainTextEditControl(BaseControl):
    """纯文本编辑控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "PlainTextEdit"
        self._category = "输入"
        self._description = "用于输入纯文本的编辑器，不支持富文本"
        self._size = (200, 100)
        self._properties = {
            "plainText": "",
            "placeholderText": "请输入纯文本...",
            "readOnly": False,
            "lineWrapMode": QTextEdit.LineWrapMode.WidgetWidth,
            "lineWrapColumnOrWidth": 80,
            "wordWrapMode": QTextEdit.WordWrapMode.WordWrap,
            "overwriteMode": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPlainTextEdit
    
    def create_instance(self, parent=None) -> QWidget:
        text_edit = QPlainTextEdit(parent)
        text_edit.setPlainText(self.get_property("plainText", ""))
        text_edit.setPlaceholderText(self.get_property("placeholderText", "请输入纯文本..."))
        text_edit.setReadOnly(self.get_property("readOnly", False))
        text_edit.setOverwriteMode(self.get_property("overwriteMode", False))
        try:
            text_edit.setLineWrapMode(self.get_property("lineWrapMode", QPlainTextEdit.LineWrapMode.WidgetWidth))
        except Exception:
            pass
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            text_edit.resize(width, height)
        
        return text_edit


class DialControl(BaseControl):
    def __init__(self):
        super().__init__()
        self._name = "Dial"
        self._category = "输入"
        self._description = "圆形旋钮"
        self._size = (80, 80)
        self._properties = {
            "minimum": 0,
            "maximum": 100,
            "value": 0,
            "notchesVisible": False,
            "wrapping": False
        }

    def get_control_class(self) -> Type[QWidget]:
        return QDial

    def create_instance(self, parent=None) -> QWidget:
        dial = QDial(parent)
        dial.setMinimum(self.get_property("minimum", 0))
        dial.setMaximum(self.get_property("maximum", 100))
        dial.setValue(self.get_property("value", 0))
        dial.setNotchesVisible(self.get_property("notchesVisible", False))
        dial.setWrapping(self.get_property("wrapping", False))
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            dial.resize(width, height)
        return dial


class VerticalSliderControl(BaseControl):
    """垂直滑块控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "VerticalSlider"
        self._category = "输入"
        self._description = "垂直方向的滑块控件"
        self._size = (30, 150)
        self._properties = {
            "minimum": 0,
            "maximum": 100,
            "value": 50,
            "singleStep": 1,
            "pageStep": 10,
            "orientation": Qt.Orientation.Vertical,
            "tickPosition": QSlider.TickPosition.TicksBothSides,
            "tickInterval": 10,
            "tracking": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QSlider
    
    def create_instance(self, parent=None) -> QWidget:
        slider = QSlider(parent)
        slider.setMinimum(self.get_property("minimum", 0))
        slider.setMaximum(self.get_property("maximum", 100))
        slider.setValue(self.get_property("value", 50))
        slider.setSingleStep(self.get_property("singleStep", 1))
        slider.setPageStep(self.get_property("pageStep", 10))
        slider.setOrientation(self.get_property("orientation", Qt.Orientation.Vertical))
        slider.setTickPosition(self.get_property("tickPosition", QSlider.TickPosition.TicksBothSides))
        slider.setTickInterval(self.get_property("tickInterval", 10))
        slider.setTracking(self.get_property("tracking", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            slider.resize(width, height)
        
        return slider


# 注册所有输入控件
def register_input_controls():
    """注册所有输入控件到控件注册表"""
    control_registry.register_control(TextEditControl())
    control_registry.register_control(SpinBoxControl())
    control_registry.register_control(DoubleSpinBoxControl())
    control_registry.register_control(SliderControl())
    control_registry.register_control(DateEditControl())
    control_registry.register_control(TimeEditControl())
    control_registry.register_control(DateTimeEditControl())
    control_registry.register_control(PlainTextEditControl())
    control_registry.register_control(VerticalSliderControl())
    control_registry.register_control(DialControl())


# 自动注册
register_input_controls()