#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
显示控件库模块
包含所有显示类控件的定义和实现
提供进度条、LCD数字显示、日历、图像显示等显示控件
"""

# 导入类型提示
from typing import Dict, Any, Type, List
# 导入PySide6显示控件组件
from PySide6.QtWidgets import (
    QLabel,  # 标签（用于显示文本或图像）
    QProgressBar,  # 进度条（显示进度）
    QLCDNumber,  # LCD数字显示（数字显示屏）
    QCalendarWidget,  # 日历控件（显示和选择日期）
    QGraphicsView,  # 图形视图（显示图形场景）
    QGraphicsScene,  # 图形场景（图形项的容器）
    QGraphicsItem,  # 图形项基类
    QGraphicsRectItem,  # 矩形图形项
    QGraphicsEllipseItem,  # 椭圆图形项
    QGraphicsLineItem,  # 线条图形项
    QGraphicsTextItem,  # 文本图形项
    QWidget,  # 基础窗口部件
    QScrollArea,  # 滚动区域
    QGroupBox  # 分组框
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    QRectF,  # 浮点矩形
    QPointF,  # 浮点点坐标
    QSizeF  # 浮点大小
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QPixmap,  # 像素图
    QImage,  # 图像
    QFont,  # 字体
    QColor,  # 颜色
    QPen,  # 画笔
    QBrush,  # 画刷
    QPainter  # 绘图器
)

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class ImageLabelControl(BaseControl):
    """图像显示标签控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ImageLabel"
        self._category = "显示"
        self._description = "用于显示图像的标签控件"
        self._size = (200, 150)
        self._properties = {
            "text": "图像",
            "pixmap": None,
            "scaledContents": True,
            "alignment": Qt.AlignmentFlag.AlignCenter,
            "imagePath": "",
            "backgroundColor": "#FFFFFF"
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QLabel
    
    def create_instance(self, parent=None) -> QWidget:
        label = QLabel(self.get_property("text", "图像"), parent)
        label.setScaledContents(self.get_property("scaledContents", True))
        label.setAlignment(self.get_property("alignment", Qt.AlignmentFlag.AlignCenter))
        
        # 设置背景色
        bg_color = self.get_property("backgroundColor", "#FFFFFF")
        label.setStyleSheet(f"background-color: {bg_color}; border: 1px solid #CCCCCC;")
        
        # 设置图像
        image_path = self.get_property("imagePath", "")
        if image_path:
            try:
                pixmap = QPixmap(image_path)
                if not pixmap.isNull():
                    label.setPixmap(pixmap)
            except Exception:
                pass
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            label.resize(width, height)
        
        return label


class ProgressBarControl(BaseControl):
    """进度条控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ProgressBar"
        self._category = "显示"
        self._description = "用于显示进度的控件"
        self._size = (200, 20)
        self._properties = {
            "minimum": 0,
            "maximum": 100,
            "value": 50,
            "orientation": Qt.Orientation.Horizontal,
            "textVisible": True,
            "format": "%p%",
            "invertedAppearance": False,
            "textDirection": QProgressBar.Direction.TopToBottom
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QProgressBar
    
    def create_instance(self, parent=None) -> QWidget:
        progress_bar = QProgressBar(parent)
        progress_bar.setMinimum(self.get_property("minimum", 0))
        progress_bar.setMaximum(self.get_property("maximum", 100))
        progress_bar.setValue(self.get_property("value", 50))
        progress_bar.setOrientation(self.get_property("orientation", Qt.Orientation.Horizontal))
        progress_bar.setTextVisible(self.get_property("textVisible", True))
        progress_bar.setFormat(self.get_property("format", "%p%"))
        progress_bar.setInvertedAppearance(self.get_property("invertedAppearance", False))
        progress_bar.setTextDirection(self.get_property("textDirection", QProgressBar.Direction.TopToBottom))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            progress_bar.resize(width, height)
        
        return progress_bar


class LCDNumberControl(BaseControl):
    """LCD数字显示控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "LCDNumber"
        self._category = "显示"
        self._description = "以LCD风格显示数字的控件"
        self._size = (100, 50)
        self._properties = {
            "value": 0,
            "digitCount": 5,
            "mode": QLCDNumber.Mode.Dec,
            "segmentStyle": QLCDNumber.SegmentStyle.Filled,
            "smallDecimalPoint": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QLCDNumber
    
    def create_instance(self, parent=None) -> QWidget:
        lcd_number = QLCDNumber(parent)
        lcd_number.display(self.get_property("value", 0))
        lcd_number.setDigitCount(self.get_property("digitCount", 5))
        lcd_number.setMode(self.get_property("mode", QLCDNumber.Mode.Dec))
        lcd_number.setSegmentStyle(self.get_property("segmentStyle", QLCDNumber.SegmentStyle.Filled))
        lcd_number.setSmallDecimalPoint(self.get_property("smallDecimalPoint", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            lcd_number.resize(width, height)
        
        return lcd_number


class CalendarControl(BaseControl):
    """日历控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "Calendar"
        self._category = "显示"
        self._description = "用于显示和选择日期的日历控件"
        self._size = (300, 250)
        self._properties = {
            "minimumDate": None,
            "maximumDate": None,
            "selectedDate": None,
            "firstDayOfWeek": Qt.DayOfWeek.Monday,
            "gridVisible": True,
            "horizontalHeaderFormat": QCalendarWidget.HorizontalHeaderFormat.SingleLetterDayNames,
            "verticalHeaderFormat": QCalendarWidget.VerticalHeaderFormat.ISOWeekNumbers,
            "navigationBarVisible": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QCalendarWidget
    
    def create_instance(self, parent=None) -> QWidget:
        calendar = QCalendarWidget(parent)
        
        min_date = self.get_property("minimumDate")
        if min_date:
            calendar.setMinimumDate(min_date)
        
        max_date = self.get_property("maximumDate")
        if max_date:
            calendar.setMaximumDate(max_date)
        
        selected_date = self.get_property("selectedDate")
        if selected_date:
            calendar.setSelectedDate(selected_date)
        
        calendar.setFirstDayOfWeek(self.get_property("firstDayOfWeek", Qt.DayOfWeek.Monday))
        calendar.setGridVisible(self.get_property("gridVisible", True))
        calendar.setHorizontalHeaderFormat(self.get_property("horizontalHeaderFormat", QCalendarWidget.HorizontalHeaderFormat.SingleLetterDayNames))
        calendar.setVerticalHeaderFormat(self.get_property("verticalHeaderFormat", QCalendarWidget.VerticalHeaderFormat.ISOWeekNumbers))
        calendar.setNavigationBarVisible(self.get_property("navigationBarVisible", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            calendar.resize(width, height)
        
        return calendar


class GraphicsViewControl(BaseControl):
    """图形视图控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "GraphicsView"
        self._category = "显示"
        self._description = "用于显示2D图形的视图控件"
        self._size = (300, 200)
        self._properties = {
            "sceneRect": QRectF(0, 0, 300, 200),
            "items": [],
            "backgroundBrush": QBrush(Qt.GlobalColor.white),
            "foregroundBrush": QBrush(),
            "cacheMode": QGraphicsView.CacheMode.CacheNone,
            "dragMode": QGraphicsView.DragMode.NoDrag,
            "optimizationFlags": QGraphicsView.OptimizationFlag.DontSavePainterState,
            "renderHints": QPainter.RenderHint.Antialiasing | QPainter.RenderHint.TextAntialiasing
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QGraphicsView
    
    def create_instance(self, parent=None) -> QWidget:
        scene_rect = self.get_property("sceneRect", QRectF(0, 0, 300, 200))
        scene = QGraphicsScene(scene_rect, parent)
        
        # 添加示例图形
        items = self.get_property("items", [])
        if not items:  # 如果没有提供图形，添加一些示例图形
            # 添加矩形
            rect_item = QGraphicsRectItem(20, 20, 80, 60)
            rect_item.setBrush(QBrush(Qt.GlobalColor.blue))
            rect_item.setPen(QPen(Qt.GlobalColor.black, 2))
            scene.addItem(rect_item)
            
            # 添加圆形
            ellipse_item = QGraphicsEllipseItem(120, 20, 80, 80)
            ellipse_item.setBrush(QBrush(Qt.GlobalColor.red))
            ellipse_item.setPen(QPen(Qt.GlobalColor.black, 2))
            scene.addItem(ellipse_item)
            
            # 添加线条
            line_item = QGraphicsLineItem(20, 120, 200, 120)
            line_item.setPen(QPen(Qt.GlobalColor.green, 3))
            scene.addItem(line_item)
            
            # 添加文本
            text_item = QGraphicsTextItem("示例文本")
            text_item.setPos(20, 140)
            text_item.setDefaultTextColor(Qt.GlobalColor.black)
            scene.addItem(text_item)
        else:
            # 添加自定义图形
            for item_data in items:
                item_type = item_data.get("type")
                if item_type == "rect":
                    rect = QGraphicsRectItem(item_data.get("rect"))
                    if "brush" in item_data:
                        rect.setBrush(item_data["brush"])
                    if "pen" in item_data:
                        rect.setPen(item_data["pen"])
                    scene.addItem(rect)
                elif item_type == "ellipse":
                    ellipse = QGraphicsEllipseItem(item_data.get("rect"))
                    if "brush" in item_data:
                        ellipse.setBrush(item_data["brush"])
                    if "pen" in item_data:
                        ellipse.setPen(item_data["pen"])
                    scene.addItem(ellipse)
                elif item_type == "line":
                    line = QGraphicsLineItem(item_data.get("line"))
                    if "pen" in item_data:
                        line.setPen(item_data["pen"])
                    scene.addItem(line)
                elif item_type == "text":
                    text = QGraphicsTextItem(item_data.get("text"))
                    text.setPos(item_data.get("pos", QPointF(0, 0)))
                    if "color" in item_data:
                        text.setDefaultTextColor(item_data["color"])
                    if "font" in item_data:
                        text.setFont(item_data["font"])
                    scene.addItem(text)
        
        view = QGraphicsView(scene, parent)
        view.setBackgroundBrush(self.get_property("backgroundBrush", QBrush(Qt.GlobalColor.white)))
        view.setForegroundBrush(self.get_property("foregroundBrush", QBrush()))
        view.setCacheMode(self.get_property("cacheMode", QGraphicsView.CacheMode.CacheNone))
        view.setDragMode(self.get_property("dragMode", QGraphicsView.DragMode.NoDrag))
        view.setOptimizationFlags(self.get_property("optimizationFlags", QGraphicsView.OptimizationFlag.DontSavePainterState))
        view.setRenderHints(self.get_property("renderHints", QPainter.RenderHint.Antialiasing | QPainter.RenderHint.TextAntialiasing))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            view.resize(width, height)
        
        return view


class ScrollAreaControl(BaseControl):
    """滚动区域控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ScrollArea"
        self._category = "显示"
        self._description = "提供滚动条的区域控件"
        self._size = (300, 200)
        self._properties = {
            "widgetResizable": True,
            "horizontalScrollBarPolicy": Qt.ScrollBarPolicy.AsNeeded,
            "verticalScrollBarPolicy": Qt.ScrollBarPolicy.AsNeeded,
            "widget": None,
            "content": "这是一个滚动区域的内容，可以包含其他控件。"
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QScrollArea
    
    def create_instance(self, parent=None) -> QWidget:
        scroll_area = QScrollArea(parent)
        scroll_area.setWidgetResizable(self.get_property("widgetResizable", True))
        scroll_area.setHorizontalScrollBarPolicy(self.get_property("horizontalScrollBarPolicy", Qt.ScrollBarPolicy.AsNeeded))
        scroll_area.setVerticalScrollBarPolicy(self.get_property("verticalScrollBarPolicy", Qt.ScrollBarPolicy.AsNeeded))
        
        # 添加内容
        widget = self.get_property("widget")
        if widget:
            scroll_area.setWidget(widget)
        else:
            # 创建一个默认的内容标签
            content = self.get_property("content", "这是一个滚动区域的内容，可以包含其他控件。")
            label = QLabel(content)
            label.setWordWrap(True)
            label.setMargin(10)
            scroll_area.setWidget(label)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            scroll_area.resize(width, height)
        
        return scroll_area


class GroupBoxControl(BaseControl):
    """分组框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "GroupBox"
        self._category = "显示"
        self._description = "用于分组显示控件的框架"
        self._size = (200, 150)
        self._properties = {
            "title": "分组",
            "flat": False,
            "checkable": False,
            "checked": True,
            "alignment": Qt.AlignmentFlag.AlignLeft,
            "content": "这是分组框的内容区域。"
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QGroupBox
    
    def create_instance(self, parent=None) -> QWidget:
        group_box = QGroupBox(self.get_property("title", "分组"), parent)
        group_box.setFlat(self.get_property("flat", False))
        group_box.setCheckable(self.get_property("checkable", False))
        group_box.setChecked(self.get_property("checked", True))
        group_box.setAlignment(self.get_property("alignment", Qt.AlignmentFlag.AlignLeft))
        
        # 添加内容
        content = self.get_property("content", "这是分组框的内容区域。")
        label = QLabel(content)
        label.setWordWrap(True)
        
        # 创建布局并添加标签
        from PySide6.QtWidgets import QVBoxLayout
        layout = QVBoxLayout()
        layout.addWidget(label)
        group_box.setLayout(layout)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            group_box.resize(width, height)
        
        return group_box


class VerticalProgressBarControl(BaseControl):
    """垂直进度条控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "VerticalProgressBar"
        self._category = "显示"
        self._description = "垂直方向的进度条控件"
        self._size = (20, 200)
        self._properties = {
            "minimum": 0,
            "maximum": 100,
            "value": 50,
            "orientation": Qt.Orientation.Vertical,
            "textVisible": True,
            "format": "%p%",
            "invertedAppearance": False,
            "textDirection": QProgressBar.Direction.TopToBottom
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QProgressBar
    
    def create_instance(self, parent=None) -> QWidget:
        progress_bar = QProgressBar(parent)
        progress_bar.setMinimum(self.get_property("minimum", 0))
        progress_bar.setMaximum(self.get_property("maximum", 100))
        progress_bar.setValue(self.get_property("value", 50))
        progress_bar.setOrientation(self.get_property("orientation", Qt.Orientation.Vertical))
        progress_bar.setTextVisible(self.get_property("textVisible", True))
        progress_bar.setFormat(self.get_property("format", "%p%"))
        progress_bar.setInvertedAppearance(self.get_property("invertedAppearance", False))
        progress_bar.setTextDirection(self.get_property("textDirection", QProgressBar.Direction.TopToBottom))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            progress_bar.resize(width, height)
        
        return progress_bar


# 注册所有显示控件
def register_display_controls():
    """注册所有显示控件到控件注册表"""
    control_registry.register_control(ImageLabelControl())
    control_registry.register_control(ProgressBarControl())
    control_registry.register_control(LCDNumberControl())
    control_registry.register_control(CalendarControl())
    control_registry.register_control(GraphicsViewControl())
    control_registry.register_control(ScrollAreaControl())
    control_registry.register_control(GroupBoxControl())
    control_registry.register_control(VerticalProgressBarControl())


# 自动注册
register_display_controls()