#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
对话框控件库模块
包含所有对话框类控件的定义和实现
提供消息框、输入对话框、颜色对话框、字体对话框、文件对话框等
"""

# 导入类型提示
from typing import Dict, Any, Type, List, Optional, Callable
# 导入PySide6对话框组件
from PySide6.QtWidgets import (
    QMessageBox,  # 消息框（显示提示、警告、错误等信息）
    QInputDialog,  # 输入对话框（获取用户输入）
    QColorDialog,  # 颜色选择对话框
    QFontDialog,  # 字体选择对话框
    QFileDialog,  # 文件选择对话框（打开/保存文件）
    QProgressDialog,  # 进度对话框（显示操作进度）
    QErrorMessage,  # 错误消息对话框
    QWidget,  # 基础窗口部件
    QPushButton,  # 按钮（用于触发对话框）
    QLabel,  # 标签
    QVBoxLayout  # 垂直布局
)
# 导入PySide6核心功能
from PySide6.QtCore import (
    Qt,  # Qt命名空间和常量
    Signal,  # 信号定义
    QObject  # Qt对象基类
)
# 导入PySide6图形界面相关类
from PySide6.QtGui import (
    QColor,  # 颜色
    QFont,  # 字体
    QPixmap  # 像素图
)

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class MessageBoxControl(BaseControl):
    """消息对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "MessageBox"
        self._category = "对话框"
        self._description = "用于显示消息的对话框"
        self._size = (100, 30)
        self._properties = {
            "title": "消息",
            "text": "这是一条消息",
            "icon": QMessageBox.Icon.Information,
            "buttons": QMessageBox.StandardButton.Ok,
            "defaultButton": QMessageBox.StandardButton.Ok
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示消息框", parent)
        
        # 连接点击事件
        def show_message():
            msg_box = QMessageBox(parent)
            msg_box.setWindowTitle(self.get_property("title", "消息"))
            msg_box.setText(self.get_property("text", "这是一条消息"))
            msg_box.setIcon(self.get_property("icon", QMessageBox.Icon.Information))
            msg_box.setStandardButtons(self.get_property("buttons", QMessageBox.StandardButton.Ok))
            msg_box.setDefaultButton(self.get_property("defaultButton", QMessageBox.StandardButton.Ok))
            msg_box.exec()
        
        button.clicked.connect(show_message)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class InputDialogControl(BaseControl):
    """输入对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "InputDialog"
        self._category = "对话框"
        self._description = "用于获取用户输入的对话框"
        self._size = (100, 30)
        self._properties = {
            "title": "输入",
            "label": "请输入内容:",
            "mode": "text",  # text, int, double, item
            "text": "",
            "intValue": 0,
            "doubleValue": 0.0,
            "items": ["选项1", "选项2", "选项3"],
            "editable": True,
            "okButtonText": "确定",
            "cancelButtonText": "取消"
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示输入框", parent)
        
        # 连接点击事件
        def show_input_dialog():
            title = self.get_property("title", "输入")
            label = self.get_property("label", "请输入内容:")
            mode = self.get_property("mode", "text")
            
            if mode == "text":
                text, ok = QInputDialog.getText(
                    parent, title, label,
                    text=self.get_property("text", "")
                )
                if ok:
                    print(f"用户输入的文本: {text}")
            elif mode == "int":
                value, ok = QInputDialog.getInt(
                    parent, title, label,
                    value=self.get_property("intValue", 0)
                )
                if ok:
                    print(f"用户输入的整数: {value}")
            elif mode == "double":
                value, ok = QInputDialog.getDouble(
                    parent, title, label,
                    value=self.get_property("doubleValue", 0.0)
                )
                if ok:
                    print(f"用户输入的浮点数: {value}")
            elif mode == "item":
                item, ok = QInputDialog.getItem(
                    parent, title, label,
                    self.get_property("items", ["选项1", "选项2", "选项3"]),
                    editable=self.get_property("editable", True)
                )
                if ok:
                    print(f"用户选择的项: {item}")
        
        button.clicked.connect(show_input_dialog)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class ColorDialogControl(BaseControl):
    """颜色选择对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ColorDialog"
        self._category = "对话框"
        self._description = "用于选择颜色的对话框"
        self._size = (100, 30)
        self._properties = {
            "title": "选择颜色",
            "color": QColor(0, 0, 255),
            "options": QColorDialog.ColorDialogOption.ShowAlphaChannel
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("选择颜色", parent)
        
        # 连接点击事件
        def show_color_dialog():
            color = QColorDialog.getColor(
                self.get_property("color", QColor(0, 0, 255)),
                parent,
                self.get_property("title", "选择颜色"),
                self.get_property("options", QColorDialog.ColorDialogOption.ShowAlphaChannel)
            )
            
            if color.isValid():
                print(f"用户选择的颜色: RGB({color.red()}, {color.green()}, {color.blue()})")
                # 更新按钮背景色
                button.setStyleSheet(f"background-color: rgb({color.red()}, {color.green()}, {color.blue()});")
        
        button.clicked.connect(show_color_dialog)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class FontDialogControl(BaseControl):
    """字体选择对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "FontDialog"
        self._category = "对话框"
        self._description = "用于选择字体的对话框"
        self._size = (100, 30)
        self._properties = {
            "title": "选择字体",
            "font": QFont("Arial", 12)
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("选择字体", parent)
        
        # 连接点击事件
        def show_font_dialog():
            font, ok = QFontDialog.getFont(
                self.get_property("font", QFont("Arial", 12)),
                parent,
                self.get_property("title", "选择字体")
            )
            
            if ok:
                print(f"用户选择的字体: {font.family()}, {font.pointSize()}pt")
                # 更新按钮字体
                button.setFont(font)
        
        button.clicked.connect(show_font_dialog)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class FileDialogControl(BaseControl):
    """文件选择对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "FileDialog"
        self._category = "对话框"
        self._description = "用于选择文件的对话框"
        self._size = (100, 30)
        self._properties = {
            "title": "选择文件",
            "directory": "",
            "filter": "所有文件 (*.*)",
            "mode": "open",  # open, save, directory
            "options": None
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("选择文件", parent)
        
        # 连接点击事件
        def show_file_dialog():
            title = self.get_property("title", "选择文件")
            directory = self.get_property("directory", "")
            filter = self.get_property("filter", "所有文件 (*.*)")
            mode = self.get_property("mode", "open")
            
            if mode == "open":
                file_path, _ = QFileDialog.getOpenFileName(
                    parent, title, directory, filter
                )
                if file_path:
                    print(f"用户选择的文件: {file_path}")
            elif mode == "save":
                file_path, _ = QFileDialog.getSaveFileName(
                    parent, title, directory, filter
                )
                if file_path:
                    print(f"用户保存的文件: {file_path}")
            elif mode == "directory":
                dir_path = QFileDialog.getExistingDirectory(
                    parent, title, directory
                )
                if dir_path:
                    print(f"用户选择的目录: {dir_path}")
        
        button.clicked.connect(show_file_dialog)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class ProgressDialogControl(BaseControl):
    """进度对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ProgressDialog"
        self._category = "对话框"
        self._description = "用于显示进度的对话框"
        self._size = (100, 30)
        self._properties = {
            "title": "进度",
            "labelText": "正在处理...",
            "minimum": 0,
            "maximum": 100,
            "value": 0,
            "modal": True,
            "cancelButtonText": "取消",
            "autoClose": True,
            "autoReset": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示进度对话框", parent)
        
        # 连接点击事件
        def show_progress_dialog():
            progress = QProgressDialog(
                self.get_property("labelText", "正在处理..."),
                self.get_property("cancelButtonText", "取消"),
                self.get_property("minimum", 0),
                self.get_property("maximum", 100),
                parent
            )
            
            progress.setWindowTitle(self.get_property("title", "进度"))
            progress.setModal(self.get_property("modal", True))
            progress.setAutoClose(self.get_property("autoClose", True))
            progress.setAutoReset(self.get_property("autoReset", True))
            progress.setValue(self.get_property("value", 0))
            
            progress.show()
            
            # 模拟进度更新
            def update_progress():
                value = progress.value()
                if value < progress.maximum():
                    progress.setValue(value + 1)
                else:
                    timer.stop()
            
            from PySide6.QtCore import QTimer
            timer = QTimer()
            timer.timeout.connect(update_progress)
            timer.start(50)  # 每50毫秒更新一次
        
        button.clicked.connect(show_progress_dialog)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class ErrorMessageControl(BaseControl):
    """错误消息控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ErrorMessage"
        self._category = "对话框"
        self._description = "用于显示错误消息的控件"
        self._size = (100, 30)
        self._properties = {
            "title": "错误",
            "message": "发生了一个错误"
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示错误消息", parent)
        
        # 连接点击事件
        def show_error_message():
            error_dialog = QErrorMessage(parent)
            error_dialog.setWindowTitle(self.get_property("title", "错误"))
            error_dialog.showMessage(self.get_property("message", "发生了一个错误"))
            error_dialog.exec()
        
        button.clicked.connect(show_error_message)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class CustomDialogControl(BaseControl):
    """自定义对话框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "CustomDialog"
        self._category = "对话框"
        self._description = "自定义对话框控件"
        self._size = (100, 30)
        self._properties = {
            "title": "自定义对话框",
            "content": "这是自定义对话框的内容",
            "buttons": ["确定", "取消"]
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton("显示自定义对话框", parent)
        
        # 连接点击事件
        def show_custom_dialog():
            from PySide6.QtWidgets import QDialog
            
            dialog = QDialog(parent)
            dialog.setWindowTitle(self.get_property("title", "自定义对话框"))
            dialog.resize(300, 200)
            
            layout = QVBoxLayout()
            
            # 添加内容标签
            content_label = QLabel(self.get_property("content", "这是自定义对话框的内容"))
            layout.addWidget(content_label)
            
            # 添加按钮
            from PySide6.QtWidgets import QDialogButtonBox
            buttons = QDialogButtonBox()
            
            for button_text in self.get_property("buttons", ["确定", "取消"]):
                if button_text == "确定":
                    buttons.addButton(QDialogButtonBox.StandardButton.Ok)
                elif button_text == "取消":
                    buttons.addButton(QDialogButtonBox.StandardButton.Cancel)
                else:
                    buttons.addButton(button_text, QDialogButtonBox.ButtonRole.ActionRole)
            
            layout.addWidget(buttons)
            dialog.setLayout(layout)
            
            # 连接按钮信号
            buttons.accepted.connect(dialog.accept)
            buttons.rejected.connect(dialog.reject)
            
            dialog.exec()
        
        button.clicked.connect(show_custom_dialog)
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


# 注册所有对话框控件
def register_dialog_controls():
    """注册所有对话框控件到控件注册表"""
    control_registry.register_control(MessageBoxControl())
    control_registry.register_control(InputDialogControl())
    control_registry.register_control(ColorDialogControl())
    control_registry.register_control(FontDialogControl())
    control_registry.register_control(FileDialogControl())
    control_registry.register_control(ProgressDialogControl())
    control_registry.register_control(ErrorMessageControl())
    control_registry.register_control(CustomDialogControl())


# 自动注册
register_dialog_controls()