#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
基础控件库模块
包含所有基础UI控件的定义和实现
提供Label、Button等最常用的基础控件
"""

# 导入类型提示
from typing import Dict, Any, Type
# 导入PySide6界面组件
from PySide6.QtWidgets import (
    QLabel,  # 标签控件
    QPushButton,  # 按钮控件
    QLineEdit,  # 单行文本输入框（可能用于某些基础控件）
    QWidget,  # 基础窗口部件
    QFrame,  # 框架
    QToolButton  # 工具按钮
)
# 导入PySide6核心功能
from PySide6.QtCore import Qt  # Qt命名空间和常量
# 导入PySide6图形界面相关类
from PySide6.QtGui import QFont  # 字体

# 导入控件基类和注册表
from ..control_base import BaseControl, control_registry


class LabelControl(BaseControl):
    """
    标签控件类
    用于显示文本或图像的标签控件
    是最常用的基础控件之一
    """
    
    def __init__(self):
        """
        初始化标签控件
        设置控件的默认属性
        """
        # 调用父类构造函数
        super().__init__()
        # 设置控件名称
        self._name = "Label"
        # 设置控件分类
        self._category = "基础"
        # 设置控件描述
        self._description = "用于显示文本或图像的标签控件"
        # 设置默认大小（宽度, 高度）
        self._size = (100, 30)
        # 设置默认属性字典
        self._properties = {
            "text": "标签",  # 标签文本内容
            "alignment": Qt.AlignmentFlag.AlignLeft,  # 文本对齐方式（左对齐）
            "wordWrap": False  # 是否自动换行（默认不换行）
        }
    
    def get_control_class(self) -> Type[QWidget]:
        """
        获取控件对应的PySide6类
        
        Returns:
            Type[QWidget]: QLabel类
        """
        return QLabel
    
    def create_instance(self, parent=None) -> QWidget:
        """
        创建控件实例
        根据属性创建并配置QLabel实例
        
        Args:
            parent: 父窗口部件
            
        Returns:
            QWidget: 创建的QLabel实例
        """
        # 创建QLabel实例，设置文本内容
        label = QLabel(self.get_property("text", "标签"), parent)
        # 设置文本对齐方式
        label.setAlignment(self.get_property("alignment", Qt.AlignmentFlag.AlignLeft))
        # 设置是否自动换行
        label.setWordWrap(self.get_property("wordWrap", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        # 如果宽度和高度都大于0，则设置控件大小
        if width > 0 and height > 0:
            label.resize(width, height)
        
        # 返回创建的标签实例
        return label


class ButtonControl(BaseControl):
    """
    按钮控件类
    用于执行操作的按钮控件
    """
    
    def __init__(self):
        super().__init__()
        self._name = "Button"
        self._category = "基础"
        self._description = "可点击的按钮控件"
        self._size = (100, 30)
        self._properties = {
            "text": "按钮",
            "checkable": False,
            "checked": False,
            "flat": False,
            "default": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton(self.get_property("text", "按钮"), parent)
        button.setCheckable(self.get_property("checkable", False))
        button.setChecked(self.get_property("checked", False))
        button.setFlat(self.get_property("flat", False))
        button.setDefault(self.get_property("default", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class LineEditControl(BaseControl):
    """单行文本输入控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "LineEdit"
        self._category = "基础"
        self._description = "用于输入单行文本的控件"
        self._size = (150, 30)
        self._properties = {
            "text": "",
            "placeholderText": "请输入文本",
            "readOnly": False,
            "echoMode": QLineEdit.EchoMode.Normal,
            "maxLength": 32767,
            "clearButtonEnabled": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QLineEdit
    
    def create_instance(self, parent=None) -> QWidget:
        line_edit = QLineEdit(parent)
        line_edit.setText(self.get_property("text", ""))
        line_edit.setPlaceholderText(self.get_property("placeholderText", "请输入文本"))
        line_edit.setReadOnly(self.get_property("readOnly", False))
        line_edit.setEchoMode(self.get_property("echoMode", QLineEdit.EchoMode.Normal))
        line_edit.setMaxLength(self.get_property("maxLength", 32767))
        line_edit.setClearButtonEnabled(self.get_property("clearButtonEnabled", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            line_edit.resize(width, height)
        
        return line_edit


class FrameControl(BaseControl):
    """框架控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "Frame"
        self._category = "基础"
        self._description = "用于组织和分组其他控件的框架"
        self._size = (200, 150)
        self._properties = {
            "frameShape": QFrame.Shape.Box,
            "frameShadow": QFrame.Shadow.Plain,
            "lineWidth": 1
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QFrame
    
    def create_instance(self, parent=None) -> QWidget:
        frame = QFrame(parent)
        frame.setFrameShape(self.get_property("frameShape", QFrame.Shape.Box))
        frame.setFrameShadow(self.get_property("frameShadow", QFrame.Shadow.Plain))
        frame.setLineWidth(self.get_property("lineWidth", 1))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            frame.resize(width, height)
        
        return frame


class StyledLabelControl(BaseControl):
    """样式标签控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "StyledLabel"
        self._category = "基础"
        self._description = "带有自定义样式的标签控件"
        self._size = (150, 30)
        self._properties = {
            "text": "样式标签",
            "alignment": Qt.AlignmentFlag.AlignCenter,
            "styleSheet": "font-weight: bold; color: #2196F3; font-size: 14px;",
            "wordWrap": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QLabel
    
    def create_instance(self, parent=None) -> QWidget:
        label = QLabel(self.get_property("text", "样式标签"), parent)
        label.setAlignment(self.get_property("alignment", Qt.AlignmentFlag.AlignCenter))
        label.setStyleSheet(self.get_property("styleSheet", "font-weight: bold; color: #2196F3; font-size: 14px;"))
        label.setWordWrap(self.get_property("wordWrap", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            label.resize(width, height)
        
        return label


class IconButtonControl(BaseControl):
    """图标按钮控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "IconButton"
        self._category = "基础"
        self._description = "带有图标的按钮控件"
        self._size = (100, 30)
        self._properties = {
            "text": "图标按钮",
            "iconPath": "",  # 图标路径
            "iconSize": (16, 16),
            "checkable": False,
            "checked": False,
            "flat": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton(self.get_property("text", "图标按钮"), parent)
        
        # 设置图标
        icon_path = self.get_property("iconPath", "")
        if icon_path:
            from PySide6.QtGui import QIcon
            icon = QIcon(icon_path)
            button.setIcon(icon)
            
            icon_size = self.get_property("iconSize", (16, 16))
            button.setIconSize(icon_size)
        
        button.setCheckable(self.get_property("checkable", False))
        button.setChecked(self.get_property("checked", False))
        button.setFlat(self.get_property("flat", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class ToggleButtonControl(BaseControl):
    """切换按钮控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ToggleButton"
        self._category = "基础"
        self._description = "可以切换状态的按钮控件"
        self._size = (100, 30)
        self._properties = {
            "text": "切换按钮",
            "checkable": True,
            "checked": False,
            "autoExclusive": False
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QPushButton
    
    def create_instance(self, parent=None) -> QWidget:
        button = QPushButton(self.get_property("text", "切换按钮"), parent)
        button.setCheckable(self.get_property("checkable", True))
        button.setChecked(self.get_property("checked", False))
        button.setAutoExclusive(self.get_property("autoExclusive", False))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            button.resize(width, height)
        
        return button


class PasswordLineEditControl(BaseControl):
    """密码输入框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "PasswordLineEdit"
        self._category = "基础"
        self._description = "用于输入密码的文本框，显示为星号或圆点"
        self._size = (150, 30)
        self._properties = {
            "text": "",
            "placeholderText": "请输入密码",
            "echoMode": QLineEdit.EchoMode.Password,
            "maxLength": 32767,
            "clearButtonEnabled": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QLineEdit
    
    def create_instance(self, parent=None) -> QWidget:
        line_edit = QLineEdit(parent)
        line_edit.setText(self.get_property("text", ""))
        line_edit.setPlaceholderText(self.get_property("placeholderText", "请输入密码"))
        line_edit.setEchoMode(self.get_property("echoMode", QLineEdit.EchoMode.Password))
        line_edit.setMaxLength(self.get_property("maxLength", 32767))
        line_edit.setClearButtonEnabled(self.get_property("clearButtonEnabled", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            line_edit.resize(width, height)
        
        return line_edit


class ReadOnlyLineEditControl(BaseControl):
    """只读文本框控件"""
    
    def __init__(self):
        super().__init__()
        self._name = "ReadOnlyLineEdit"
        self._category = "基础"
        self._description = "只读的文本框，用于显示不可编辑的文本"
        self._size = (150, 30)
        self._properties = {
            "text": "这是只读文本",
            "readOnly": True,
            "frame": True
        }
    
    def get_control_class(self) -> Type[QWidget]:
        return QLineEdit
    
    def create_instance(self, parent=None) -> QWidget:
        line_edit = QLineEdit(self.get_property("text", "这是只读文本"), parent)
        line_edit.setReadOnly(self.get_property("readOnly", True))
        line_edit.setFrame(self.get_property("frame", True))
        
        # 设置默认大小
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            line_edit.resize(width, height)
        
        return line_edit


class ToolButtonControl(BaseControl):
    def __init__(self):
        super().__init__()
        self._name = "ToolButton"
        self._category = "基础"
        self._description = "工具按钮，支持图标/文本样式"
        self._size = (100, 30)
        self._properties = {
            "text": "工具按钮",
            "iconPath": "",
            "toolButtonStyle": Qt.ToolButtonStyle.ToolButtonTextBesideIcon,
            "autoRaise": False,
            "checkable": False,
            "checked": False
        }

    def get_control_class(self) -> Type[QWidget]:
        return QToolButton

    def create_instance(self, parent=None) -> QWidget:
        btn = QToolButton(parent)
        btn.setText(self.get_property("text", "工具按钮"))
        icon_path = self.get_property("iconPath", "")
        if icon_path:
            from PySide6.QtGui import QIcon
            btn.setIcon(QIcon(icon_path))
        btn.setToolButtonStyle(self.get_property("toolButtonStyle", Qt.ToolButtonStyle.ToolButtonTextBesideIcon))
        btn.setAutoRaise(self.get_property("autoRaise", False))
        btn.setCheckable(self.get_property("checkable", False))
        btn.setChecked(self.get_property("checked", False))
        width, height = self.get_default_size()
        if width > 0 and height > 0:
            btn.resize(width, height)
        return btn


# 注册所有基础控件
def register_basic_controls():
    """注册所有基础控件到控件注册表"""
    control_registry.register_control(LabelControl())
    control_registry.register_control(ButtonControl())
    control_registry.register_control(LineEditControl())
    control_registry.register_control(FrameControl())
    control_registry.register_control(StyledLabelControl())
    control_registry.register_control(IconButtonControl())
    control_registry.register_control(ToggleButtonControl())
    control_registry.register_control(PasswordLineEditControl())
    control_registry.register_control(ReadOnlyLineEditControl())
    control_registry.register_control(ToolButtonControl())


# 自动注册
register_basic_controls()